/*
   mb-applet-blanker - a screen blanking applet

   Copyright 2004 Matt Purvis <matt@free.as.in.beer.geek.nz>

   Based on:
     mb-applet-menu-launcher	(Copyright 2002 Matthew Allum)
     mb-applet-cards		(Copyright 2004 Alexander Chukov)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

GtkWidget *blanker_window;
static Bool PopupIsMapped = False;

static Bool ScreenBlank = False;
static Bool SuspendDisabled = False;
static Bool ScreensaverDisabled = False;

int config_backlight, config_screensaver, config_suspend;
int custom_backlight = 0;
gint backlight_timer;

#include <apm.h>
#define TIME_LEFT  0
#define PERCENTAGE 1
#define AC_POWER   2

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef USE_DNOTIFY
#define _GNU_SOURCE
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/wait.h>

#include <libmb/mb.h>

#define MB_MSG_FONT        "Verdana-14"

#ifdef USE_LIBSN
#define SN_API_NOT_YET_FROZEN 1
#include <libsn/sn.h>
#endif

#ifndef MB_HAVE_PNG
#include <setjmp.h>
#endif

typedef struct _app
{
  MBTrayApp     *tray_app; 
  MBMenu        *mbmenu;
  MBPixbuf      *pb;

  MBPixbufImage *img_tray;
  MBPixbufImage *img_tray_scaled;
  MBPixbufImage *img_tray_active;
  MBPixbufImage *img_tray_active_scaled;

  Atom           mbtheme_atom;
  Atom           mbcommand_atom;

  char          *theme_name;
  Bool           button_is_down;

#ifdef USE_LIBSN
  SnDisplay  *sn_display;
#endif
   
} AppData;

AppData     *app_data;

MBMenuMenu  *root;
Bool         Update_Pending  = False;
jmp_buf      Jbuf;
MBMenuMenu  *active[10];
int          MenuWasActive   = False;

Bool         WantDebianMenus = False;
volatile Bool WantReload     = False;

static void reap_children (int signum);
static void fork_exec (char *cmd);
static void usage ();
static void build_menu (void);
static void catch_sigsegv (int sig);
static void load_icons(void);
void add_suspend_item();
void remove_suspend_item();
static void suspend_toggle_cb(MBMenuItem *item);
static void screensaver_toggle_cb(MBMenuItem *item);
void backlight_timeout_callback ( MBTrayApp *app );

#ifdef USE_LIBSN
static void sn_exec(char* name, char* bin_name, char *desc);
#endif

#define TRAY_IMG "blanker-screen.png"
#define TRAY_IMG_NOSUSPEND "blanker-screen-no_suspend.png"
#define TRAY_IMG_NOSCREENSAVER "blanker-screen-no_screensaver.png"
#define TRAY_IMG_NEITHER "blanker-screen-neither.png"

#define SINGLE_INSTANCE 1
#define STARTUP_NOTIFY  2

#define UP   0
#define DOWN 1

#define MBMAX(x,y) ((x>y)?(x):(y))

#define NEW(OBJ) ((OBJ *)(malloc(sizeof(OBJ))))

#define IS_WHITESPACE(c) ((c) == ' ' || (c) == '\\' \
                          || (c) == '\t' || (c)=='\n' || (c)=='?')

#ifdef DEBUG
#define DBG(txt, args... ) fprintf(stderr, "DEBUG: " txt , ##args )
#else
#define DBG(txt, args... ) /* nothing */
#endif

static void usage()
{
  printf("usage: mbmenu [options ....]\n"
	  "Where options are\n"
	  "  -display <display> Display to connect to\n"
	  "\nAlso set MB_USE_DEB_MENUS env var to enable parsing /usr/lib/menu"
	  "\n"

	  );
   exit(1);
}

static void catch_sigsegv(int sig)
{
   DBG("ouch\n");
   signal(SIGSEGV, SIG_DFL);
   longjmp(Jbuf, 1);

}

static void reap_children(int signum)
{
    pid_t pid;

    do {
	pid = waitpid(-1, NULL, WNOHANG);
    } while (pid > 0);
}

#ifdef USE_LIBSN

static void 
sn_exec(char* name, char* bin_name, char *desc)
{
  SnLauncherContext *context;
  pid_t child_pid = 0;

  context = sn_launcher_context_new (app_data->sn_display, 
				     mb_tray_app_xscreen(app_data->tray_app));
  
  if (name)     sn_launcher_context_set_name (context, name);
  if (desc)     sn_launcher_context_set_description (context, desc);
  if (bin_name) sn_launcher_context_set_binary_name (context, bin_name);
  
  sn_launcher_context_initiate (context, "mbmenu launch", bin_name,
				CurrentTime);

  switch ((child_pid = fork ()))
    {
    case -1:
      fprintf (stderr, "Fork failed\n" );
      break;
    case 0:
      sn_launcher_context_setup_child_process (context);
      mb_exec(bin_name);
      fprintf (stderr, "mbmenu: Failed to exec %s \n", bin_name);
      _exit (1);
      break;
    }
}

#endif

static void fork_exec(char *cmd)
{
    pid_t pid;
    
    switch (pid = fork()) 
      {
      case 0:
	mb_exec(cmd);
	fprintf(stderr, "mbmenu: exec failed, cleaning up child\n");
	exit(1);
      case -1:
	fprintf(stderr, "mbmenu: can't fork\n"); break;
      }
}

static int 
read_apm(int *values)
{
  /* add stat function here ? */
 
  apm_info info;
  apm_read(&info);

  values[TIME_LEFT] = info.battery_time;
  values[PERCENTAGE] = info.battery_percentage;
  values[AC_POWER] = info.ac_line_status;

  return 1;
}

int ac_power()
{
  int apm_vals[3];

  while (!read_apm(apm_vals))
    usleep(50000L);

  if (apm_vals[AC_POWER] == AC_LINE_STATUS_ON) 
    return True;
  else
    return False;

}

static void read_lnp_config()
{
   FILE *config;
   char line[255];
   float screensaver_mins, suspend_mins;

   if ((config = fopen("/home/root/Choices/lightnpower.cfg", "r"))) {
     /* Skip the first three lines */
     fgets(line,sizeof(line),config);
     fgets(line,sizeof(line),config);
     fgets(line,sizeof(line),config);

     /* Skip the AC line if we're on battery power */
     if (!ac_power())
       fgets(line,sizeof(line),config);

     if (fgets(line,sizeof(line),config) == NULL) {
       config_backlight = 0;
       config_screensaver = 0;
       config_suspend = 0;
     } else {
       sscanf(line," %*25c%d%*c %*11c%f%*c %*9c%f", &config_backlight, &screensaver_mins, &suspend_mins);
       fclose(config);
       config_screensaver = screensaver_mins * 60;
       config_suspend = suspend_mins * 60;
     }
   }
}

static void set_dpms()
{
  char cmd[255];
  int a, b;

  // Only read the config file if we are going to use a value from it
  if (!ScreensaverDisabled || !SuspendDisabled) {
    read_lnp_config();

    a = config_screensaver;
    b = config_suspend;
  }

  if (ScreensaverDisabled && SuspendDisabled) {
    a = 0;
    b = 0;
  } else if (ScreensaverDisabled) {
    a = 0;
  } else if (SuspendDisabled) {
    b = 0;
  }

  snprintf(cmd, sizeof(cmd), "/usr/X11R6/bin/xset dpms %d 0 %d", a, b);

  fork_exec(cmd);
}


static
void reload_icons()
{
  load_icons();
  resize_callback (app_data->tray_app, mb_tray_app_width(app_data->tray_app), mb_tray_app_width(app_data->tray_app) );
  mb_tray_app_repaint (app_data->tray_app);
}

static
void suspend_toggle_cb(MBMenuItem *item)
{
  SuspendDisabled = SuspendDisabled ? False : True;

  set_dpms();

  reload_icons();
}

static
void screensaver_toggle_cb(MBMenuItem *item)
{
  ScreensaverDisabled = ScreensaverDisabled ? False : True;

  set_dpms();

  reload_icons();
}

static
void read_backlight()
{
   FILE *bl;
   char line[255];
   int buffer;

   read_lnp_config();

   if ((bl = fopen("/proc/driver/fl/corgi-bl", "r"))) {
     if (fgets(line,sizeof(line),bl) != NULL) {
       sscanf(line,"%i", &buffer);
       fclose(bl);

       switch (buffer) {
       case 287: buffer = 6; break;
       case 267: buffer = 5; break;
       case 261: buffer = 4; break;
       case  11: buffer = 3; break;
       case   5: buffer = 2; break;
       case   1: buffer = 1; break;
       default : buffer = 0;
       }

       if (buffer > 0) {
	 if (buffer != config_backlight) {
	   custom_backlight = buffer;
	 } else {
	   custom_backlight = 0;
	 }
       }

     }
   }
}

static
void disable_backlight()
{
  if (!ScreenBlank)
    read_backlight();

  fork_exec("/sbin/setfl 0");

  ScreenBlank = True;
}

static
void enable_backlight()
{
  char cmd[255];

  read_lnp_config();

  if (custom_backlight > 0) {
    snprintf(cmd, sizeof(cmd), "/sbin/setfl %d", custom_backlight);
  } else {
    snprintf(cmd, sizeof(cmd), "/sbin/setfl %d", config_backlight);
  }

  fork_exec(cmd);

  ScreenBlank = False;
}

void
blanker_cb(void)
{
  disable_backlight();

  // Hide the mouse cursor
  XWarpPointer (GDK_DISPLAY(), None, GDK_ROOT_WINDOW(), 0, 0, 0, 0, 640, 480);

  backlight_timer = gtk_timeout_add (1000, // 1 second
				(GSourceFunc) backlight_timeout_callback,
				app_data->tray_app);

  gtk_widget_show_all (blanker_window);
  gdk_pointer_grab (blanker_window->window, TRUE, GDK_BUTTON_PRESS_MASK, NULL, NULL, CurrentTime);
  PopupIsMapped = True;
}

static void
build_menu(void)
{
  MBMenuMenu *m = app_data->mbmenu->rootmenu;
  MBMenuActivateCB suspend_callback = suspend_toggle_cb;
  MBMenuActivateCB screensaver_callback = screensaver_toggle_cb;
  MBMenuActivateCB blanker_callback = blanker_cb;
  char *png_path = "/usr/share/pixmaps/blanker-screen.png";
  char *tmp_path = NULL, *tmp_path2 = NULL;

  tmp_path = mb_dot_desktop_icon_get_full_path (app_data->theme_name, 
						16, 
						"mbfolder.png" );

  tmp_path2 = mb_dot_desktop_icon_get_full_path (app_data->theme_name, 
						16, 
						"mbnoapp.png" );

  mb_menu_set_default_icons(app_data->mbmenu, tmp_path, tmp_path2);

  if (tmp_path) free(tmp_path);
  if (tmp_path2) free(tmp_path2);

  if (ScreensaverDisabled) {
    mb_menu_add_item_to_menu(app_data->mbmenu, m,
			     "Enable Screensaver",
			     "/usr/share/pixmaps/blanker-screensaver.png",
			     "",
			     screensaver_callback,
			     NULL, MBMENU_NO_SORT);
  } else {
    mb_menu_add_item_to_menu(app_data->mbmenu, m,
			     "Disable Screensaver",
			     "/usr/share/pixmaps/blanker-no_screensaver.png",
			     "",
			     screensaver_callback,
			     NULL, MBMENU_NO_SORT);
  }

  if (SuspendDisabled) {
    mb_menu_add_item_to_menu(app_data->mbmenu, m,
			     "Enable Suspend",
			     "/usr/share/pixmaps/blanker-suspend.png",
			     "",
			     suspend_callback,
			     NULL, MBMENU_NO_SORT);
  } else {
    mb_menu_add_item_to_menu(app_data->mbmenu, m,
			     "Disable Suspend",
			     "/usr/share/pixmaps/blanker-no_suspend.png",
			     "",
			     suspend_callback,
			     NULL, MBMENU_NO_SORT);
  }

  mb_menu_add_seperator_to_menu(app_data->mbmenu, m, MBMENU_NO_SORT);

  mb_menu_add_item_to_menu(app_data->mbmenu, m,
			   "Blank Screen",
			   png_path,
			   "",
			   blanker_callback,
			   NULL, MBMENU_NO_SORT);
}

static void
menu_get_popup_pos (MBTrayApp *app, int *x, int *y)
{
  int abs_x, abs_y, menu_h, menu_w;
  mb_tray_app_get_absolute_coords (app, &abs_x, &abs_y);
  mb_menu_get_root_menu_size(app_data->mbmenu, &menu_w, &menu_h);

  if (mb_tray_app_tray_is_vertical (app))
    {
      /* XXX need to figure out menu size before its mapped 
	     so we can figure out offset for east panel
      */
      *y = abs_y + mb_tray_app_height(app);

      if (abs_x > (DisplayWidth(mb_tray_app_xdisplay(app), mb_tray_app_xscreen(app)) /2))
	*x = abs_x - menu_w - 2;
      else
	*x = abs_x + mb_tray_app_width(app) + 2;
    }
  else
    {
      *x = abs_x;
      if (abs_y > (DisplayHeight(mb_tray_app_xdisplay(app), mb_tray_app_xscreen(app)) /2))
	*y = abs_y - 2;
      else
	*y = abs_y + mb_tray_app_height(app) + menu_h;
    }
}

void
button_callback (MBTrayApp *app, int x, int y, Bool is_released )
{
  int abs_x, abs_y;
  sigset_t block_sigset;
  static Bool next_cancels;

  app_data->button_is_down = True;
  if (is_released) app_data->button_is_down = False;

  sigemptyset(&block_sigset);
  sigaddset(&block_sigset, SIGRTMIN);

  if (is_released && !next_cancels)
    {
      sigaddset(&block_sigset, SIGRTMIN);
      menu_get_popup_pos (app, &abs_x, &abs_y);
      mb_menu_activate (app_data->mbmenu, abs_x, abs_y);

    }
  else
    if (mb_menu_is_active(app_data->mbmenu)) {
      next_cancels = True;
    } else {
      next_cancels = False;
      mb_menu_free(app_data->mbmenu);
      build_menu();
    }

  mb_tray_app_repaint (app);
}

void
paint_callback (MBTrayApp *app, Drawable drw )
{
  MBPixbufImage *img_bg = NULL;
  MBPixbufImage *img_button = NULL;
  
  img_button = ( app_data->button_is_down ? app_data->img_tray_active_scaled : app_data->img_tray_scaled ); 

  img_bg = mb_tray_app_get_background (app, app_data->pb);

  mb_pixbuf_img_copy_composite (app_data->pb, img_bg, 
				img_button, 
				0, 0, 
				mb_pixbuf_img_get_width(app_data->img_tray_scaled), 
				mb_pixbuf_img_get_height(app_data->img_tray_scaled), 
				0, 0 );
  
  mb_pixbuf_img_render_to_drawable (app_data->pb, img_bg, drw, 0, 0);

  mb_pixbuf_img_free( app_data->pb, img_bg );
}

void
resize_callback (MBTrayApp *app, int w, int h)
{
  if (app_data->img_tray_scaled) 
    mb_pixbuf_img_free(app_data->pb, 
		       app_data->img_tray_scaled);

  app_data->img_tray_scaled = mb_pixbuf_img_scale(app_data->pb, 
						  app_data->img_tray, 
						  w, h);
  if (app_data->img_tray_active_scaled) 
    mb_pixbuf_img_free(app_data->pb, 
		       app_data->img_tray_active_scaled);

  app_data->img_tray_active_scaled 
    = mb_pixbuf_img_scale(app_data->pb, 
			  app_data->img_tray_active, 
			  w, h);
}

void 
theme_callback (MBTrayApp *app, char *theme_name)
{
  if (!theme_name) return;
  if (app_data->theme_name) free(app_data->theme_name);
  app_data->theme_name = strdup(theme_name);

  if (app_data->mbmenu != NULL)
    {
      mb_menu_free(app_data->mbmenu);
      build_menu();
      load_icons();
      resize_callback (app, mb_tray_app_width(app), mb_tray_app_width(app) );
      mb_tray_app_repaint (app_data->tray_app);
    }
}

void
xevent_callback (MBTrayApp *app, XEvent *ev)
{
  sigset_t block_sigset;

  mb_menu_handle_xevent (app_data->mbmenu, ev);  

#ifdef USE_DNOTIFY
  if (!mb_menu_is_active(app_data->mbmenu))
    { 				/* Unblock any dnotify signals */
      sigemptyset(&block_sigset);
      sigaddset(&block_sigset, SIGRTMIN);
      sigprocmask(SIG_UNBLOCK, &block_sigset, NULL); 
    }
#endif

#define MB_CMD_SHOW_EXT_MENU 6

  if (ev->type == ClientMessage)
    {
      if (ev->xclient.message_type == app_data->mbcommand_atom
	  && ev->xclient.data.l[0] == MB_CMD_SHOW_EXT_MENU )
	{
	  sigemptyset(&block_sigset);
	  sigaddset(&block_sigset, SIGRTMIN);

	  if (!mb_menu_is_active(app_data->mbmenu)) 
	    {
	      int abs_x, abs_y;
	      sigprocmask(SIG_BLOCK, &block_sigset, NULL); 
	      menu_get_popup_pos (app, &abs_x, &abs_y);
	      mb_menu_activate(app_data->mbmenu, abs_x, abs_y);
	    } else {
	      mb_menu_deactivate(app_data->mbmenu);
	      sigprocmask(SIG_UNBLOCK, &block_sigset, NULL); 
	    }
	}
    }
}

void 
load_icons(void)
{
 char *icon_path = NULL;
 
 if (app_data->img_tray) 
    mb_pixbuf_img_free(app_data->pb, app_data->img_tray);

 if (app_data->img_tray_active) 
    mb_pixbuf_img_free(app_data->pb, app_data->img_tray_active);

 if (ScreensaverDisabled && SuspendDisabled) {
   icon_path = mb_dot_desktop_icon_get_full_path (app_data->theme_name, 
						  16, 
						  TRAY_IMG_NEITHER );
 } else if (ScreensaverDisabled) {
   icon_path = mb_dot_desktop_icon_get_full_path (app_data->theme_name, 
						  16, 
						  TRAY_IMG_NOSCREENSAVER );
 } else if (SuspendDisabled) {
   icon_path = mb_dot_desktop_icon_get_full_path (app_data->theme_name, 
						  16, 
						  TRAY_IMG_NOSUSPEND );
 } else {
   icon_path = mb_dot_desktop_icon_get_full_path (app_data->theme_name, 
						  16, 
						  TRAY_IMG );
 }

 if (icon_path == NULL 
     || !(app_data->img_tray 
	  = mb_pixbuf_img_new_from_file(app_data->pb, icon_path)))
   {
     fprintf(stderr, "mbmenu: failed to load icon\n");
     exit(1);
   }

 free(icon_path);

 int x, y;

 app_data->img_tray_active 
   = mb_pixbuf_img_clone(app_data->pb, app_data->img_tray);

 for (x=0; x<mb_pixbuf_img_get_width(app_data->img_tray_active); x++)
   for (y=0; y<mb_pixbuf_img_get_height(app_data->img_tray_active); y++)
     {
       int aa;
       unsigned char r,g,b,a;
       mb_pixbuf_img_get_pixel(app_data->pb, app_data->img_tray_active,
			       x, y, &r, &g, &b, &a);

       aa = (int)a;
       aa -=  0x80; if (aa < 0) aa = 0;

       mb_pixbuf_img_set_pixel_alpha(app_data->img_tray_active, 
				     x, y, aa);
     }
}

static void
blanker_clicked (GtkWidget *w, GdkEventButton *ev)
{
  gdk_pointer_ungrab (ev->time);

  gtk_widget_hide (blanker_window);

  gtk_timeout_remove(backlight_timer);

  enable_backlight();
}

void
popup_init(MBTrayApp *app)
{
  GtkWidget     *vbox;
  GtkWidget     *hbox;
  GtkWidget     *label;
  GtkWidget     *button_mute, *button_ok;
  GtkAdjustment *adj;
  GtkStyle	*style = gtk_style_new();

  style->bg[GTK_STATE_NORMAL] = style->black;

  blanker_window = gtk_window_new (GTK_WINDOW_POPUP);

  gtk_widget_set_style(blanker_window, style);
  gtk_window_set_default_size (blanker_window, 640, 480);

  g_signal_connect (G_OBJECT (blanker_window), "button-press-event", G_CALLBACK (blanker_clicked), NULL);

  gtk_widget_add_events (blanker_window, GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);

  gtk_widget_realize (blanker_window);
}

GdkFilterReturn
event_filter (GdkXEvent *xev, GdkEvent *gev, gpointer data)
{
  XEvent    *ev  = (XEvent *)xev;
  MBTrayApp *app = (MBTrayApp*)data;

  Display *dpy = ev->xany.display;

  mb_tray_handle_xevent (app, ev); 

  return GDK_FILTER_CONTINUE;
}

void
backlight_timeout_callback ( MBTrayApp *app )
{
  if(ScreenBlank)
    disable_backlight();
}

void
timeout_callback ( MBTrayApp *app )
{
  if(!mb_menu_is_active(app_data->mbmenu))
    set_dpms();
}

int
main( int argc, char *argv[])
{
  MBTrayApp *app = NULL;
  struct sigaction act;

  gtk_init (&argc, &argv);

#ifdef USE_DNOTIFY
  int fd;
#endif

  app_data = NEW(AppData);
  memset(app_data, 0, sizeof(AppData));

  app = mb_tray_app_new_with_display ( "Screen Blanker",
				       resize_callback,
				       paint_callback,
				       &argc,
				       &argv,
				       GDK_DISPLAY ());

  if (app == NULL) usage();
  
  if (argc > 1 && strstr(argv[1], "-h"))
    usage();

  app_data->tray_app = app;

  app_data->mbtheme_atom 
    = XInternAtom(mb_tray_app_xdisplay(app), "_MB_THEME", False);
  app_data->mbcommand_atom 
    = XInternAtom(mb_tray_app_xdisplay(app), "_MB_COMMAND", False);
  
  app_data->mbmenu = mb_menu_new(mb_tray_app_xdisplay(app), 
				 mb_tray_app_xscreen(app));

  mb_menu_set_font(app_data->mbmenu, MB_MSG_FONT);

#ifdef USE_LIBSN
  app_data->sn_display = sn_display_new (mb_tray_app_xdisplay(app), 
					  NULL, NULL); 
#endif

   app_data->pb = mb_pixbuf_new(mb_tray_app_xdisplay(app), 
				mb_tray_app_xscreen(app));

   XSelectInput (mb_tray_app_xdisplay(app), mb_tray_app_xrootwin(app), 
		 PropertyChangeMask|SubstructureNotifyMask);

   mb_tray_app_set_button_callback (app, button_callback );

   mb_tray_app_set_xevent_callback (app, xevent_callback );

   mb_tray_app_set_theme_change_callback (app, theme_callback );

   gtk_timeout_add (30000, // 30 seconds
		    (GSourceFunc) timeout_callback,
		    app);

   load_icons();

   /* Set up signals */

   act.sa_flags = 0;
   sigemptyset(&act.sa_mask);
   act.sa_handler = reap_children;
   sigaction(SIGCHLD, &act, NULL);

   popup_init(app);

   /* Reset dpms and backlight to config values (just because we can) */
   set_dpms();
   enable_backlight();

   /* Disable X screensaver (kludge) */
   fork_exec("/usr/X11R6/bin/xset s 0");

   mb_tray_app_set_icon(app, app_data->pb, app_data->img_tray);

   mb_tray_app_main_init (app);
   gdk_window_add_filter (NULL, event_filter, (gpointer)app );
   gtk_main ();

   return 1;
}
