
/*
 * MIXMOS - an oss mixer
 *
 * Copyright (C) 2003 Tomasz Mka
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gui.h"
#include "prefs.h"
#include "mixer.h"
#include "tables.h"
#include "config.h"
#include "i18n.h"

static GtkWidget	*frames[SOUND_MIXER_NRDEVICES];
static GtkWidget	*vscale_l[SOUND_MIXER_NRDEVICES];
static GtkWidget	*vscale_r[SOUND_MIXER_NRDEVICES];
static GtkWidget	*label_l[SOUND_MIXER_NRDEVICES];
static GtkWidget	*label_r[SOUND_MIXER_NRDEVICES];
static GtkObject	*adj_l[SOUND_MIXER_NRDEVICES];
static GtkObject	*adj_r[SOUND_MIXER_NRDEVICES];
static GtkWidget	*rec[SOUND_MIXER_NRDEVICES];
static GtkWidget	*lock[SOUND_MIXER_NRDEVICES];
static GtkWidget	*mute[SOUND_MIXER_NRDEVICES];
static GtkWidget 	*logo[SOUND_MIXER_NRDEVICES];
static GtkWidget	*entries[SOUND_MIXER_NRDEVICES];
static gint			muted_l[SOUND_MIXER_NRDEVICES];
static gint			muted_r[SOUND_MIXER_NRDEVICES];

static GtkWidget	*main_window, *options_window, *save_win_pos;
static GtkWidget	*info_dialog;
static gint			channels, timer_tag, restart_info, muteall_b_state, lockall_b_state;
static GtkWidget	*quit_button, *close_button, *muteall_button, *lockall_button;
static GtkWidget	*enable_pixmaps, *enable_labels, *enable_selector, *enable_tooltips;
static GtkWidget	*preset_menu, *store_confirm;
static GtkWidget	*p1_entry, *p2_entry, *p3_entry, *p4_entry;
static GtkWidget	*store_button, *restore_button, *vseparator_preset;

static GtkTooltips	*mixmos_tooltips; 
static GtkWidget	*lockall_image, *muteall_image;
static GtkWidget	*fontname_entry;

static GError		*err;
static GdkPixbuf	*pixbuf_icon;

static gchar		tmpbuf[256];

/******************************************************************************/

gchar* gui_get_pixmap_name(gint channel, gint mode)
{
gint i;
gchar *p;

	i = mixer_get_real_channel(channel);
	p = default_pixmaps[0];

	if(mode == NORMAL)	p = default_pixmaps[i];
	if(mode == RECORD)	p = rec_pixmaps[i];
	if(mode == MUTE)	p = mute_pixmaps[i];

	return p;
}

/******************************************************************************/

void gui_sliders_update(void)
{
gint i, kl, kr;
gchar valbuffer[128];

	/* volume sliders */

	for(i=0;i<channels;i++) {

		if(config.show_state[config.current_preset][i]) {

			kl = mixer_get_volume_l(i);
			kr = mixer_get_volume_r(i);

			GTK_ADJUSTMENT (adj_l[i])->value = 100-kl;	
		    g_signal_emit_by_name (G_OBJECT (adj_l[i]), "changed");
			sprintf(valbuffer, "%d", kl);
			gtk_label_set_text(GTK_LABEL(label_l[i]), valbuffer);

			if(mixer_get_channel_stereo(i)) {
				GTK_ADJUSTMENT (adj_r[i])->value = 100-kr;	
			    g_signal_emit_by_name (G_OBJECT (adj_r[i]), "changed");
				sprintf(valbuffer, "%d", kr);
				gtk_label_set_text(GTK_LABEL(label_r[i]), valbuffer);
			}
		}
	}
}

/******************************************************************************/

void gui_recordsrc_update(void)
{
gint i;

	for(i=0;i<channels;i++) {
		if(config.show_state[config.current_preset][i] && mixer_get_channel_recordable(i)) {
   			if (mixer_get_recordsrc(i)) {
				if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (rec[i]))) 
					gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (rec[i]), TRUE);
			}
		}
	}
}

/******************************************************************************/

gint gui_timeout_callback(gpointer data)
{
	mixer_state_read();
	gui_sliders_update();
	gui_recordsrc_update();
	return 1;
}

/******************************************************************************/

void gui_lock_changed(GtkToggleButton *button, gint no)
{
	config.lock_state[config.current_preset][no] = button->active;
}

/******************************************************************************/

void gui_rec_changed(GtkToggleButton *button, gint no)
{

	if(mixer_get_channel_recordable(no)) {

		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (rec[no]))) {
   			if (!mixer_get_recordsrc(no)) {
				mixer_set_recordsrc(no);
	
				if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (mute[no])))
					gtk_image_set_from_file(GTK_IMAGE(logo[no]), gui_get_pixmap_name(no, RECORD));

			} 
		} else {

			if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (mute[no])))
				gtk_image_set_from_file(GTK_IMAGE(logo[no]), gui_get_pixmap_name(no, NORMAL));

		}
	}

}

/******************************************************************************/

void gui_mute_changed(GtkToggleButton *button, gint no)
{

	if(button->active) {
	/* muted */

		muted_l[no] = mixer_get_volume_l(no);
		muted_r[no] = mixer_get_volume_r(no);

		if(mixer_get_channel_stereo(no))
			mixer_set_volume_stereo(no, 0, 0);
		else
			mixer_set_volume_mono(no, 0);

		gtk_image_set_from_file(GTK_IMAGE(logo[no]), gui_get_pixmap_name(no, MUTE));

	} else {
	/* unmuted */

		if(mixer_get_channel_stereo(no))
			mixer_set_volume_stereo(no, muted_l[no], muted_r[no]);
		else
			mixer_set_volume_mono(no, muted_l[no]);

		if(mixer_get_recordsrc(no))
			gtk_image_set_from_file(GTK_IMAGE(logo[no]), gui_get_pixmap_name(no, RECORD));
		else
			gtk_image_set_from_file(GTK_IMAGE(logo[no]), gui_get_pixmap_name(no, NORMAL));

	}

}

/******************************************************************************/

void gui_muteall_changed(GtkWidget *button)
{
gint i;

	muteall_b_state *= -1;

	if(muteall_b_state == 1)
		gtk_image_set_from_file(GTK_IMAGE(muteall_image), PIXMAPS "mute_2.png");
	else
		gtk_image_set_from_file(GTK_IMAGE(muteall_image), PIXMAPS "mute_1.png");

	for(i=0;i<channels;i++) {

		if(config.show_state[config.current_preset][i] && mute[i]!=NULL) {

			if(muteall_b_state == 1)
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (mute[i]), TRUE);
			else
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (mute[i]), FALSE);

		}
	}

}

/******************************************************************************/

void gui_lockall_changed(GtkWidget *button)
{
gint i;

	lockall_b_state *= -1;

	if(lockall_b_state == -1)
		gtk_image_set_from_file(GTK_IMAGE(lockall_image), PIXMAPS "lock_1.png");
	else
		gtk_image_set_from_file(GTK_IMAGE(lockall_image), PIXMAPS "lock_2.png");

	for(i=0;i<channels;i++) {

		if(config.show_state[config.current_preset][i] && lock[i]!=NULL) {

			if(lockall_b_state == -1)
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (lock[i]), TRUE);
			else
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (lock[i]), FALSE);

		}
	}

}

/******************************************************************************/

void gui_sliders_changed( GtkAdjustment *get, gint no)
{
gchar valbuffer[128];
gint k, nl, nr;


	sprintf(valbuffer, "%d", 100-(gint)get->value);

	if (no >= 500) {
		k = no - 500;
		gtk_label_set_text(GTK_LABEL(label_r[k]), valbuffer);
		nl = mixer_get_volume_l(k);
		nr = 100-(gint)get->value;
	} else {
		k = no;
		gtk_label_set_text(GTK_LABEL(label_l[k]), valbuffer);
		nl = 100-(gint)get->value;
		nr = mixer_get_volume_r(k);
	}

	if(mixer_get_channel_stereo(k)) {
		if(config.lock_state[config.current_preset][k]) {
			if(no < 500) nr = nl;
				else nl = nr;
		}
		mixer_set_volume_stereo(k, nl, nr);
    } else {
		mixer_set_volume_mono(k, nl);
	}
}

/******************************************************************************/

void gui_store_volumes_main(void)
{
gint i;

	for(i=0;i<channels;i++) {
		if(config.show_state[config.current_preset][i]) {

			config.a_vol_l[config.current_preset][i] = mixer_get_volume_l(i);
			config.a_vol_r[config.current_preset][i] = mixer_get_volume_r(i);

			if (mixer_get_channel_recordable(i) && mixer_get_recordsrc(i)) 
				config.a_rec[config.current_preset] = i;
		}
	}

}

/******************************************************************************/

void gui_store_volumes( GtkWidget *widget, gpointer data )
{
gint response;
GtkWidget *hbox;
GtkWidget *label;
GtkWidget *stock;

	if(config.store_confirmation) {

		info_dialog = gtk_dialog_new_with_buttons ("Question",
						GTK_WINDOW (main_window),
						GTK_DIALOG_MODAL| GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_STOCK_YES, 
						GTK_RESPONSE_OK,
						GTK_STOCK_NO,
						GTK_RESPONSE_NO,
						NULL);

		hbox = gtk_hbox_new (FALSE, 8);
		gtk_container_set_border_width (GTK_CONTAINER (hbox), 8);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (info_dialog)->vbox), hbox, FALSE, FALSE, 0);

		stock = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION, GTK_ICON_SIZE_DIALOG);
		gtk_box_pack_start (GTK_BOX (hbox), stock, FALSE, FALSE, 0);

		label = gtk_label_new ("Store current volumes ?");
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  		gtk_widget_show_all (hbox);

		response = gtk_dialog_run (GTK_DIALOG (info_dialog));

		if (response == GTK_RESPONSE_OK)
			gui_store_volumes_main();

		gtk_widget_destroy(info_dialog);

	} else 
			gui_store_volumes_main();

}

/******************************************************************************/

void gui_restore_volumes(void)
{
gint i;

	for(i=0;i<channels;i++) {
		if(config.show_state[config.current_preset][i]) {

			gtk_image_set_from_file(GTK_IMAGE(logo[i]), gui_get_pixmap_name(i, NORMAL));

			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (mute[i]), FALSE);

			if(mixer_get_channel_stereo(i)) {
				mixer_set_volume_stereo(i, config.a_vol_l[config.current_preset][i], config.a_vol_r[config.current_preset][i]);
			} else {
				mixer_set_volume_mono(i, config.a_vol_l[config.current_preset][i]);
			}
		}
	}

	mixer_set_recordsrc(config.a_rec[config.current_preset]);
	gtk_image_set_from_file(GTK_IMAGE(logo[config.a_rec[config.current_preset]]), gui_get_pixmap_name(config.a_rec[config.current_preset], RECORD));

}

/******************************************************************************/

void gui_restore_preset(void)
{
gint i;

	gui_restore_volumes();

	for(i=0;i<channels;i++) {
		if(config.show_state[config.current_preset][i]) {

			if (lock[i]!=NULL) {
				if (config.lock_state[config.current_preset][i])
				    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (lock[i]), TRUE);
				else 
				    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (lock[i]), FALSE);
			}

			if (config.mute_state[config.current_preset][i])
			    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (mute[i]), TRUE);
			else
			    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (mute[i]), FALSE);

		}

		if(config.show_state[config.current_preset][i]==1)
			gtk_widget_show(frames[i]);
		else
			gtk_widget_hide(frames[i]);

	}
}

/******************************************************************************/

void gui_close_mixer_window( GtkWidget *widget, gpointer data )
{

	gtk_window_get_size (GTK_WINDOW(main_window),
						&config.window_size_x, &config.window_size_y);

	if (config.save_window_pos)
		gdk_window_get_root_origin (main_window->window,
									&config.window_x, &config.window_y);

	gtk_timeout_remove(timer_tag);
	gtk_main_quit ();
}

/******************************************************************************/

gint gui_main_window_key_press_event (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	switch (event->keyval) {
    
		case GDK_Escape:
			g_signal_emit_by_name (G_OBJECT(quit_button), "clicked");
			break;

		case GDK_o:
			gui_show_options();
			break;

		case GDK_r:
			gui_restore_preset();
			break;
        
		case GDK_1:
			config.current_preset = 0;
			gtk_option_menu_set_history (GTK_OPTION_MENU (preset_menu), 0);
			break;
		case GDK_2:
			config.current_preset = 1;
			gtk_option_menu_set_history (GTK_OPTION_MENU (preset_menu), 1);
			break;
		case GDK_3:
			config.current_preset = 2;
			gtk_option_menu_set_history (GTK_OPTION_MENU (preset_menu), 2);
			break;
		case GDK_4:
			config.current_preset = 3;
			gtk_option_menu_set_history (GTK_OPTION_MENU (preset_menu), 3);
			break;

		default:
			break;
	}

	return FALSE;
}

/******************************************************************************/

void gui_pr_menu_select(GtkWidget *item, GtkUpdateType type)
{
	config.current_preset = type;
}

/******************************************************************************/

void gui_create_mixer_window (void) 
{
gint i, k, l;
gchar 		title[256];
GtkWidget	*frame;
GtkWidget	*hbox1;
GtkWidget	*hbox2;
GtkWidget	*hbox3;
GtkWidget	*hbox4;
GtkWidget	*hbox5;
GtkWidget	*hbox6;
GtkWidget	*hbox7;
GtkWidget	*scrolledwindow;
GtkWidget	*viewport;
GtkWidget	*vbox1;
GtkWidget	*vbox2;
GtkWidget	*vbox4;
GtkWidget	*options_button;
GSList 		*rec_group;
GtkWidget	*pr_item ,*pr_menu;
GtkWidget	*box;
GtkWidget	*vseparator, *imagew;
GtkWidget	*label;
PangoFontDescription *pfd;


	muteall_b_state = -1;
	lockall_b_state = -1;

	pfd = pango_font_description_from_string(config.labels_font);

	mixmos_tooltips = gtk_tooltips_new();

    lockall_image = gtk_image_new_from_file (PIXMAPS "lock_1.png");
    muteall_image = gtk_image_new_from_file (PIXMAPS "mute_1.png");


	main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	g_signal_connect (G_OBJECT (main_window), "key_press_event",
						G_CALLBACK (gui_main_window_key_press_event), NULL);

	gtk_window_set_resizable (GTK_WINDOW (main_window), TRUE);
	gtk_widget_set_size_request (GTK_WIDGET(main_window), 
								config.window_size_x, config.window_size_y);
	if (config.save_window_pos)
		gtk_window_move (GTK_WINDOW (main_window),
									config.window_x, config.window_y);

	g_signal_connect (G_OBJECT (main_window), "delete_event",
						G_CALLBACK (gui_close_mixer_window), NULL);

	sprintf(title, "Mixmos " VERSION " - %s", mixer_get_name());
	gtk_window_set_title (GTK_WINDOW (main_window), title);
	gtk_widget_show (main_window);

	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_container_add (GTK_CONTAINER (main_window), frame);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 2);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (frame), vbox1);

	/*********************************************************************/
	/* SLIDERS */

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox2, TRUE, TRUE, 0);

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow);
	gtk_box_pack_start (GTK_BOX (hbox2), scrolledwindow, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow), 2);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	viewport = gtk_viewport_new (NULL, NULL);
	gtk_widget_show (viewport);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), viewport);

	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox1);
	gtk_container_add (GTK_CONTAINER (viewport), hbox1);

	vbox4 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox4);
	gtk_box_pack_start (GTK_BOX (hbox1), vbox4, TRUE, TRUE, 0);

	hbox5 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox5);
	gtk_box_pack_start (GTK_BOX (vbox4), hbox5, TRUE, TRUE, 0);



	/*********************************************************************/
	/* BUTTONS */

	hbox6 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox6);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox6, FALSE, TRUE, 0);

    muteall_button = gtk_button_new ();
	if(config.enable_tooltips)
		gtk_tooltips_set_tip (mixmos_tooltips, muteall_button, "Mute all", NULL);
	gtk_button_set_relief (GTK_BUTTON (muteall_button), GTK_RELIEF_NONE);

    box = gtk_hbox_new (TRUE, 0);
    gtk_box_pack_start (GTK_BOX (box), muteall_image, FALSE, FALSE, 0);
    gtk_widget_show (muteall_image);
    gtk_widget_show (box);

    gtk_container_add (GTK_CONTAINER (muteall_button), box);
    gtk_widget_show (muteall_button);

	g_signal_connect (G_OBJECT (muteall_button), "clicked",
						G_CALLBACK (gui_muteall_changed), NULL);

	gtk_box_pack_start (GTK_BOX (hbox6), muteall_button, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (muteall_button), 0);

	/*----------------------------------*/

    lockall_button = gtk_button_new ();
	if(config.enable_tooltips)
		gtk_tooltips_set_tip (mixmos_tooltips, lockall_button, "Lock all", NULL);
	gtk_button_set_relief (GTK_BUTTON (lockall_button), GTK_RELIEF_NONE);

    box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (box), lockall_image, FALSE, FALSE, 0);
    gtk_widget_show (lockall_image);
    gtk_widget_show (box);

    gtk_container_add (GTK_CONTAINER (lockall_button), box);
	gtk_widget_show (lockall_button);

	g_signal_connect (G_OBJECT (lockall_button), "clicked",
						G_CALLBACK (gui_lockall_changed), NULL);
	gtk_box_pack_start (GTK_BOX (hbox6), lockall_button, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (lockall_button), 0);

	/*----------------------------------*/

	vseparator = gtk_vseparator_new ();
	gtk_widget_show (vseparator);
  	gtk_box_pack_start (GTK_BOX (hbox6), vseparator, FALSE, TRUE, 4);

	/*----------------------------------*/

	/* preset menu */

  	preset_menu = gtk_option_menu_new ();
  	gtk_widget_show (preset_menu);
  	gtk_box_pack_start (GTK_BOX (hbox6), preset_menu, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (preset_menu), 4);

  	pr_menu = gtk_menu_new ();

    pr_item = gtk_menu_item_new_with_label (config.preset_1_name);
	g_signal_connect (G_OBJECT (pr_item), "activate",
						G_CALLBACK (gui_pr_menu_select), GINT_TO_POINTER(0));

    gtk_widget_show (pr_item);
  	gtk_menu_shell_append (GTK_MENU_SHELL (pr_menu), pr_item);
    pr_item = gtk_menu_item_new_with_label (config.preset_2_name);
	g_signal_connect (G_OBJECT (pr_item), "activate",
						G_CALLBACK (gui_pr_menu_select), GINT_TO_POINTER(1));
    gtk_widget_show (pr_item);
  	gtk_menu_shell_append (GTK_MENU_SHELL (pr_menu), pr_item);
    pr_item = gtk_menu_item_new_with_label (config.preset_3_name);
	g_signal_connect (G_OBJECT (pr_item), "activate",
						G_CALLBACK (gui_pr_menu_select), GINT_TO_POINTER(2));
    gtk_widget_show (pr_item);
  	gtk_menu_shell_append (GTK_MENU_SHELL (pr_menu), pr_item);
    pr_item = gtk_menu_item_new_with_label (config.preset_4_name);
	g_signal_connect (G_OBJECT (pr_item), "activate",
						G_CALLBACK (gui_pr_menu_select), GINT_TO_POINTER(3));
    gtk_widget_show (pr_item);
  	gtk_menu_shell_append (GTK_MENU_SHELL (pr_menu), pr_item);

  	gtk_option_menu_set_menu (GTK_OPTION_MENU (preset_menu), pr_menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (preset_menu), config.current_preset);

	/*----------------------------------*/

    store_button = gtk_button_new ();
	if(config.enable_tooltips)
		gtk_tooltips_set_tip (mixmos_tooltips, store_button, "Store current volumes", NULL);
	gtk_button_set_relief (GTK_BUTTON (store_button), GTK_RELIEF_NONE);

    box = gtk_hbox_new (TRUE, 0);
    imagew = gtk_image_new_from_file (PIXMAPS "store.png");
    gtk_box_pack_start (GTK_BOX (box), imagew, FALSE, FALSE, 0);
    gtk_widget_show (imagew);
    gtk_widget_show (box);

    gtk_container_add (GTK_CONTAINER (store_button), box);
    gtk_widget_show (store_button);

	g_signal_connect (G_OBJECT (store_button), "clicked",
						G_CALLBACK (gui_store_volumes), NULL);

	gtk_box_pack_start (GTK_BOX (hbox6), store_button, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (store_button), 0);

	/*----------------------------------*/

    restore_button = gtk_button_new ();
	if(config.enable_tooltips)
		gtk_tooltips_set_tip (mixmos_tooltips, restore_button, "Restore current volumes", NULL);
	gtk_button_set_relief (GTK_BUTTON (restore_button), GTK_RELIEF_NONE);

    box = gtk_hbox_new (TRUE, 0);
    imagew = gtk_image_new_from_file (PIXMAPS "restore.png");
    gtk_box_pack_start (GTK_BOX (box), imagew, FALSE, FALSE, 0);
    gtk_widget_show (imagew);
    gtk_widget_show (box);

    gtk_container_add (GTK_CONTAINER (restore_button), box);
    gtk_widget_show (restore_button);

	g_signal_connect (G_OBJECT (restore_button), "clicked",
						G_CALLBACK (gui_restore_volumes), NULL);

	gtk_box_pack_start (GTK_BOX (hbox6), restore_button, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (restore_button), 0);

	/*----------------------------------*/

	vseparator_preset = gtk_vseparator_new ();
    gtk_widget_show (vseparator_preset);
  	gtk_box_pack_start (GTK_BOX (hbox6), vseparator_preset, FALSE, TRUE, 4);

	/*----------------------------------*/

    options_button = gtk_button_new ();
	if(config.enable_tooltips)
		gtk_tooltips_set_tip (mixmos_tooltips, options_button, "Preferences", NULL);
	gtk_button_set_relief (GTK_BUTTON (options_button), GTK_RELIEF_NONE);

    box = gtk_hbox_new (TRUE, 0);
    imagew = gtk_image_new_from_file (PIXMAPS "options.png");
    gtk_box_pack_start (GTK_BOX (box), imagew, FALSE, FALSE, 0);
    gtk_widget_show (imagew);
    gtk_widget_show (box);

    gtk_container_add (GTK_CONTAINER (options_button), box);
    gtk_widget_show (options_button);

	g_signal_connect (G_OBJECT (options_button), "clicked",
						G_CALLBACK (gui_show_options), NULL);

	gtk_box_pack_start (GTK_BOX (hbox6), options_button, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (options_button), 0);

	/*----------------------------------*/

	hbox7 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox7);
	gtk_box_pack_start (GTK_BOX (hbox6), hbox7, TRUE, TRUE, 0);

	/*----------------------------------*/

    quit_button = gtk_button_new ();
	if(config.enable_tooltips)
		gtk_tooltips_set_tip (mixmos_tooltips, quit_button, "Quit!", NULL);
	gtk_button_set_relief (GTK_BUTTON (quit_button), GTK_RELIEF_NONE);

    box = gtk_hbox_new (TRUE, 0);
    imagew = gtk_image_new_from_file (PIXMAPS "exit.png");
    gtk_box_pack_start (GTK_BOX (box), imagew, FALSE, FALSE, 0);
    gtk_widget_show (imagew);
    gtk_widget_show (box);

    gtk_container_add (GTK_CONTAINER (quit_button), box);
    gtk_widget_show (quit_button);

	g_signal_connect (G_OBJECT (quit_button), "clicked",
						G_CALLBACK (gui_close_mixer_window), NULL);

	gtk_box_pack_start (GTK_BOX (hbox6), quit_button, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (quit_button), 0);

	/*********************************************************************/


	/*------------------------------------------------------------------------*/
	/*------------------------------------------------------------------------*/
	/*------------------------------------------------------------------------*/

	channels = mixer_get_channels();
	rec_group = NULL;


	/* lock all initial state */

	for(i=k=0, l=0;i<channels;i++)
		if(config.show_state[config.current_preset][i] && mixer_get_channel_stereo(i)) {
			k += config.lock_state[config.current_preset][i];
			l++;
		}

	if(l!=k) {
		lockall_b_state = 1;
		gtk_image_set_from_file(GTK_IMAGE(lockall_image), PIXMAPS "lock_2.png");
	}


	/* initialize channels */

	for(i=0;i<channels;i++) {

		frames[i] = gtk_frame_new (NULL);

		label = gtk_label_new (NULL);
		gtk_widget_show (label);
		gtk_frame_set_label_widget (GTK_FRAME (frames[i]), label);
		sprintf (tmpbuf, _("<i> %s </i>"), config.channels_names[i]);
		gtk_label_set_markup (GTK_LABEL (label), tmpbuf);

		if(config.show_state[config.current_preset][i])
			gtk_widget_show (frames[i]);
		gtk_box_pack_start (GTK_BOX (hbox5), frames[i], FALSE, TRUE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (frames[i]), 2);

		hbox3 = gtk_hbox_new (FALSE, 0);
		gtk_widget_show (hbox3);
		gtk_container_add (GTK_CONTAINER (frames[i]), hbox3);

		if(mixer_get_channel_stereo(i)) {

			vbox2 = gtk_vbox_new (FALSE, 0);
			gtk_widget_show (vbox2);
			gtk_box_pack_start (GTK_BOX (hbox3), vbox2, TRUE, TRUE, 4);

  			logo[i] = gtk_image_new_from_file (gui_get_pixmap_name(i, NORMAL));
			if(config.enable_pixmaps==1)
				gtk_widget_show (logo[i]);
			gtk_box_pack_start (GTK_BOX (vbox2), logo[i], FALSE, FALSE, 0);

			hbox4 = gtk_hbox_new (FALSE, 0);
			gtk_widget_show (hbox4);
			gtk_box_pack_start (GTK_BOX (vbox2), hbox4, TRUE, TRUE, 4);

			vbox1 = gtk_vbox_new (FALSE, 0);
			gtk_widget_show (vbox1);
			gtk_box_pack_start (GTK_BOX (hbox4), vbox1, TRUE, TRUE, 4);

			label_l[i] = gtk_label_new ("100");
			gtk_label_set_justify (GTK_LABEL (label_l[i]), GTK_JUSTIFY_FILL);
			gtk_box_pack_start (GTK_BOX (vbox1), label_l[i], FALSE, FALSE, 0);
			if(config.enable_labels==1)
			    gtk_widget_show (label_l[i]);

			gtk_widget_modify_font (label_l[i], pfd);

		    adj_l[i] = gtk_adjustment_new (0, 0, 100, 1, 10, 0);
			g_signal_connect (G_OBJECT (adj_l[i]), "value_changed",
							G_CALLBACK (gui_sliders_changed), GINT_TO_POINTER (i));
			vscale_l[i] = gtk_vscale_new (GTK_ADJUSTMENT (adj_l[i]));
			gtk_widget_show (vscale_l[i]);
			gtk_scale_set_digits (GTK_SCALE (vscale_l[i]), 0);
		    gtk_scale_set_draw_value (GTK_SCALE (vscale_l[i]), 0);
			gtk_box_pack_start (GTK_BOX (vbox1), vscale_l[i], TRUE, TRUE, 0);

			vbox1 = gtk_vbox_new (FALSE, 0);
			gtk_widget_show (vbox1);
			gtk_box_pack_start (GTK_BOX (hbox4), vbox1, TRUE, TRUE, 3);

			label_r[i] = gtk_label_new ("100");
			gtk_label_set_justify (GTK_LABEL (label_r[i]), GTK_JUSTIFY_FILL);
			gtk_box_pack_start (GTK_BOX (vbox1), label_r[i], FALSE, FALSE, 0);
			if(config.enable_labels==1)
			    gtk_widget_show (label_r[i]);

			gtk_widget_modify_font (label_r[i], pfd);

		    adj_r[i] = gtk_adjustment_new (0, 0, 100, 1, 10, 0);
			g_signal_connect (G_OBJECT (adj_r[i]), "value_changed",
							G_CALLBACK (gui_sliders_changed), GINT_TO_POINTER (i+500));
			vscale_r[i] = gtk_vscale_new (GTK_ADJUSTMENT (adj_r[i]));
			gtk_widget_show (vscale_r[i]);
			gtk_scale_set_digits (GTK_SCALE (vscale_r[i]), 0);
		    gtk_scale_set_draw_value (GTK_SCALE (vscale_r[i]), 0);
			gtk_box_pack_start (GTK_BOX (vbox1), vscale_r[i], TRUE, TRUE, 0);

			rec[i] = NULL;
			lock[i] = NULL;
			mute[i] = NULL;

			if (mixer_get_channel_recordable(i)) {
				rec[i] = gtk_radio_button_new_with_label (rec_group, _("Rec"));
				rec_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(rec[i]));
	    		gtk_box_pack_start (GTK_BOX (vbox2), rec[i], FALSE, TRUE, 0);
				g_signal_connect (G_OBJECT (rec[i]), "toggled",
								G_CALLBACK (gui_rec_changed), GINT_TO_POINTER (i));

				if (mixer_get_recordsrc(i)) {
				    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rec[i]), TRUE);
					gtk_image_set_from_file(GTK_IMAGE(logo[i]), gui_get_pixmap_name(i, RECORD));
				}
				gtk_widget_show (rec[i]);
			}

			if (mixer_get_channel_stereo(i)) {
			    lock[i] = gtk_check_button_new_with_label(_("Lock"));
	    		gtk_box_pack_start (GTK_BOX (vbox2), lock[i], FALSE, TRUE, 0);
				g_signal_connect (G_OBJECT (lock[i]), "toggled",
								G_CALLBACK (gui_lock_changed), GINT_TO_POINTER (i));
				if (config.lock_state[config.current_preset][i]) // || lockall_b_state == 1)
				    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (lock[i]), TRUE);
				gtk_widget_show (lock[i]);
			}

		    mute[i] = gtk_check_button_new_with_label(_("Mute"));
    		gtk_box_pack_start (GTK_BOX (vbox2), mute[i], FALSE, TRUE, 0);
			g_signal_connect (G_OBJECT (mute[i]), "toggled",
							G_CALLBACK (gui_mute_changed), GINT_TO_POINTER (i));
			gtk_widget_show (mute[i]);


		} else {

			vbox2 = gtk_vbox_new (FALSE, 0);
			gtk_widget_show (vbox2);
			gtk_box_pack_start (GTK_BOX (hbox3), vbox2, TRUE, TRUE, 3);

 			logo[i] = gtk_image_new_from_file (gui_get_pixmap_name(i, NORMAL));
			if(config.enable_pixmaps==1)
				gtk_widget_show (logo[i]);
			gtk_box_pack_start (GTK_BOX (vbox2), logo[i], FALSE, FALSE, 0);

			hbox4 = gtk_hbox_new (FALSE, 0);
			gtk_widget_show (hbox4);
			gtk_box_pack_start (GTK_BOX (vbox2), hbox4, TRUE, TRUE, 4);

			vbox1 = gtk_vbox_new (FALSE, 0);
			gtk_widget_show (vbox1);
			gtk_box_pack_start (GTK_BOX (hbox4), vbox1, TRUE, TRUE, 3);

			label_l[i] = gtk_label_new ("100");
			gtk_label_set_justify (GTK_LABEL (label_l[i]), GTK_JUSTIFY_FILL);
			gtk_box_pack_start (GTK_BOX (vbox1), label_l[i], FALSE, FALSE, 0);
			if(config.enable_labels==1)
			    gtk_widget_show (label_l[i]);

			gtk_widget_modify_font (label_l[i], pfd);

			label_r[i] = NULL;
			vscale_r[i] = NULL;
			adj_r[i] = NULL;

		    adj_l[i] = gtk_adjustment_new (0, 0, 100, 1, 10, 0);
			g_signal_connect (G_OBJECT (adj_l[i]), "value_changed",
							G_CALLBACK (gui_sliders_changed), GINT_TO_POINTER (i));
			vscale_l[i] = gtk_vscale_new (GTK_ADJUSTMENT (adj_l[i]));
			gtk_widget_show (vscale_l[i]);
			gtk_scale_set_digits (GTK_SCALE (vscale_l[i]), 0);
		    gtk_scale_set_draw_value (GTK_SCALE (vscale_l[i]), 0);
			gtk_box_pack_start (GTK_BOX (vbox1), vscale_l[i], TRUE, TRUE, 0);

			if (mixer_get_channel_recordable(i)) {
				rec[i] = gtk_radio_button_new_with_label (rec_group, _("Rec"));
				rec_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(rec[i]));

	    		gtk_box_pack_start (GTK_BOX (vbox2), rec[i], FALSE, TRUE, 0);
				g_signal_connect (G_OBJECT (rec[i]), "toggled",
								G_CALLBACK (gui_rec_changed), GINT_TO_POINTER (i));
				if (mixer_get_recordsrc(i)) {
				    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rec[i]), TRUE);
					gtk_image_set_from_file(GTK_IMAGE(logo[i]), gui_get_pixmap_name(i, RECORD));
				}
				gtk_widget_show (rec[i]);
			}

			if (mixer_get_channel_stereo(i)) {
			    lock[i] = gtk_check_button_new_with_label(_("Lock"));
	    		gtk_box_pack_start (GTK_BOX (vbox2), lock[i], FALSE, TRUE, 0);
				g_signal_connect (G_OBJECT (lock[i]), "toggled",
								G_CALLBACK (gui_lock_changed), GINT_TO_POINTER (i));
				if (config.lock_state[config.current_preset][i] || lockall_b_state == 1)
				    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (lock[i]), TRUE);
				gtk_widget_show (lock[i]);
			}

		    mute[i] = gtk_check_button_new_with_label(_("Mute"));
    		gtk_box_pack_start (GTK_BOX (vbox2), mute[i], FALSE, TRUE, 0);
			g_signal_connect (G_OBJECT (mute[i]), "toggled",
							G_CALLBACK (gui_mute_changed), GINT_TO_POINTER (i));
			gtk_widget_show (mute[i]);

		}

		}

	if (!config.enable_preset_selector) {
		gtk_widget_hide(preset_menu);
		gtk_widget_hide(store_button);
		gtk_widget_hide(restore_button);
		gtk_widget_hide(vseparator_preset);
	}

	err = NULL;
    pixbuf_icon = gdk_pixbuf_new_from_file (PIXMAPS "icon.png", &err);
	gtk_window_set_icon (GTK_WINDOW(main_window), pixbuf_icon);

	timer_tag = gtk_timeout_add(400, (GtkFunction)gui_timeout_callback, NULL);
	gui_sliders_update();
	gui_recordsrc_update();

	pango_font_description_free (pfd);
}

/******************************************************************************/

void gui_confirm_show_changed(GtkToggleButton *button)
{

	config.store_confirmation = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (store_confirm));

}

/******************************************************************************/

void gui_tips_show_changed(GtkToggleButton *button)
{

	config.enable_tooltips = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (enable_tooltips));                                

	restart_info = TRUE;

}


/******************************************************************************/

void gui_pix_show_changed(GtkToggleButton *button)
{
gint i;

	config.enable_pixmaps = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (enable_pixmaps));                                

	for(i=0;i<channels;i++) {
		if(config.enable_pixmaps==1) {
			gtk_widget_show(logo[i]);
		} else {
			gtk_widget_hide(logo[i]);
		}
	}
}

/******************************************************************************/

void gui_lbl_show_changed(GtkToggleButton *button)
{
gint i;

	config.enable_labels = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (enable_labels));   

	for(i=0;i<channels;i++) {

		if(config.enable_labels==1) {

			gtk_widget_show(label_l[i]);
			if (mixer_get_channel_stereo(i))
				gtk_widget_show(label_r[i]);

		} else {

			gtk_widget_hide(label_l[i]);
			if (mixer_get_channel_stereo(i))
				gtk_widget_hide(label_r[i]);

		}
	}
}

/******************************************************************************/

void gui_sel_show_changed(GtkToggleButton *button)
{
	config.enable_preset_selector = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (enable_selector));   

	if (!config.enable_preset_selector) {
		gtk_widget_hide(preset_menu);
		gtk_widget_hide(store_button);
		gtk_widget_hide(restore_button);
		gtk_widget_hide(vseparator_preset);
	} else {
		gtk_widget_show(preset_menu);
		gtk_widget_show(store_button);
		gtk_widget_show(restore_button);
		gtk_widget_show(vseparator_preset);
	}
}


/******************************************************************************/

void gui_chn_show_changed(GtkToggleButton *button, gint no)
{
	config.show_state[config.current_preset][no] = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (button));

	if(config.show_state[config.current_preset][no]==1)
		gtk_widget_show(frames[no]);
	else
		gtk_widget_hide(frames[no]);

}

/******************************************************************************/

void gui_options_window_close(void)
{
gint i;

	if(restart_info == TRUE) gui_show_info();

	gdk_window_get_root_origin (options_window->window,
								&config.opt_window_x, &config.opt_window_y);

	config.save_window_pos = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (save_win_pos));                                

	strncpy(config.preset_1_name, gtk_entry_get_text (GTK_ENTRY(p1_entry)), 20);
	strncpy(config.preset_2_name, gtk_entry_get_text (GTK_ENTRY(p2_entry)), 20);
	strncpy(config.preset_3_name, gtk_entry_get_text (GTK_ENTRY(p3_entry)), 20);
	strncpy(config.preset_4_name, gtk_entry_get_text (GTK_ENTRY(p4_entry)), 20);

	for(i=0;i<channels;i++)
		strncpy(config.channels_names[i], gtk_entry_get_text (GTK_ENTRY(entries[i])), 16);

	gtk_widget_destroy (options_window);
}

/******************************************************************************/

gint gui_opt_window_key_press_event (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	switch (event->keyval) {
    
		case GDK_Escape:
			g_signal_emit_by_name (G_OBJECT(close_button), "clicked");
			break;

		default:
			break;
	}

	return FALSE;
}

/******************************************************************************/

int gui_options_entries_activity (GtkWidget *widget, gpointer data)
{
	restart_info = TRUE;

	return FALSE;
}

/******************************************************************************/

void gui_font_select(GtkWidget *widget)
{
gchar *s;
GtkWidget *font_selector;
gint response, i;
PangoFontDescription *pfd;

	font_selector = gtk_font_selection_dialog_new ("Select a font...");
	gtk_window_set_modal(GTK_WINDOW(font_selector), TRUE);
	gtk_font_selection_dialog_set_font_name (GTK_FONT_SELECTION_DIALOG
						 (font_selector),
						 config.labels_font);
	gtk_widget_show (font_selector);
	response = gtk_dialog_run (GTK_DIALOG (font_selector));
	if (response == GTK_RESPONSE_OK)
	{
		s = gtk_font_selection_dialog_get_font_name (GTK_FONT_SELECTION_DIALOG(font_selector));

		strncpy(config.labels_font, s, 128);
		gtk_entry_set_text(GTK_ENTRY(fontname_entry), s);

		pfd = pango_font_description_from_string
			(gtk_font_selection_dialog_get_font_name
			 (GTK_FONT_SELECTION_DIALOG (font_selector)));


		for(i=0;i<channels;i++) {

			if(config.show_state[config.current_preset][i]) {

				gtk_widget_modify_font (label_l[i], pfd);

				if(mixer_get_channel_stereo(i))
					gtk_widget_modify_font (label_r[i], pfd);
			}
		}

		g_free (s);
		pango_font_description_free (pfd);
	}

	gtk_widget_destroy (font_selector);



}

/******************************************************************************/

void gui_show_options(void)
{
gint i;
GtkWidget *hbox1;
GtkWidget *hbox2;
GtkWidget *vbox1;
GtkWidget *vbox2;
GtkWidget *vbox3;
GtkWidget *vbox4;
GtkWidget *notebook;
GtkWidget *scrolledwindow;
GtkWidget *viewport;
GtkWidget *spage1;
GtkWidget *spage2;
GtkWidget *spage3;
GtkWidget *spage4;
//GtkWidget *spage5;
GtkWidget *label;
GtkWidget *hseparator;
GtkWidget *bbox;
GtkWidget *frame;
GtkWidget *chn;
GtkWidget *mix_logo;
GtkWidget *fontsel_button;

	restart_info = FALSE;

	options_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_transient_for(GTK_WINDOW(options_window),GTK_WINDOW(main_window));

	g_signal_connect (G_OBJECT (options_window), "delete_event",
						G_CALLBACK (gui_options_window_close), NULL);
	g_signal_connect (G_OBJECT (options_window), "key_press_event",
						G_CALLBACK (gui_opt_window_key_press_event), NULL);

	gtk_window_set_modal(GTK_WINDOW(options_window), TRUE);
                    
	gtk_widget_set_size_request (GTK_WIDGET(options_window), 320, 380);
	gtk_window_move (GTK_WINDOW (options_window),
								config.opt_window_x, config.opt_window_y);

	gtk_container_set_border_width (GTK_CONTAINER (options_window), 4);
	gtk_window_set_resizable (GTK_WINDOW (main_window), TRUE);
	gtk_window_set_title (GTK_WINDOW (options_window), _("Mixmos options"));
    
	err = NULL;
    pixbuf_icon = gdk_pixbuf_new_from_file (PIXMAPS "icon.png", &err);
	gtk_window_set_icon (GTK_WINDOW(main_window), pixbuf_icon);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (options_window), vbox1);

	notebook = gtk_notebook_new ();
	gtk_widget_show (notebook);
	gtk_box_pack_start (GTK_BOX (vbox1), notebook, TRUE, TRUE, 0);

	/******************************************************************************/
	/* MAIN */

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_show (scrolledwindow);
	gtk_container_add (GTK_CONTAINER (notebook), scrolledwindow);

	viewport = gtk_viewport_new (NULL, NULL);
	gtk_widget_show (viewport);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), viewport);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_container_add (GTK_CONTAINER (viewport), vbox2);


	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 6);

	label = gtk_label_new (NULL);
	gtk_widget_show (label);
	gtk_frame_set_label_widget (GTK_FRAME (frame), label);
	sprintf (tmpbuf, _("<i> Settings </i>"));
	gtk_label_set_markup (GTK_LABEL (label), tmpbuf);

	vbox4 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox4);
	gtk_container_add (GTK_CONTAINER (frame), vbox4);

    enable_selector = gtk_check_button_new_with_label(_("Enable preset selector"));
	gtk_box_pack_start (GTK_BOX (vbox4), enable_selector, FALSE, TRUE, 0);
	g_signal_connect (G_OBJECT (enable_selector), "toggled",
						G_CALLBACK (gui_sel_show_changed), NULL);
	gtk_widget_show (enable_selector);

    store_confirm = gtk_check_button_new_with_label(_("Store confirmation"));
	gtk_box_pack_start (GTK_BOX (vbox4), store_confirm, FALSE, TRUE, 0);
	g_signal_connect (G_OBJECT (store_confirm), "toggled",
						G_CALLBACK (gui_confirm_show_changed), NULL);
	gtk_widget_show (store_confirm);

    enable_tooltips = gtk_check_button_new_with_label(_("Enable tooltips"));
	gtk_box_pack_start (GTK_BOX (vbox4), enable_tooltips, FALSE, TRUE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (enable_tooltips), config.enable_tooltips);
	g_signal_connect (G_OBJECT (enable_tooltips), "toggled",
						G_CALLBACK (gui_tips_show_changed), NULL);
	gtk_widget_show (enable_tooltips);

    enable_pixmaps = gtk_check_button_new_with_label(_("Enable pixmaps"));
	gtk_box_pack_start (GTK_BOX (vbox4), enable_pixmaps, FALSE, TRUE, 0);
	g_signal_connect (G_OBJECT (enable_pixmaps), "toggled",
						G_CALLBACK (gui_pix_show_changed), NULL);
	gtk_widget_show (enable_pixmaps);

    enable_labels = gtk_check_button_new_with_label(_("Enable volume labels"));
	gtk_box_pack_start (GTK_BOX (vbox4), enable_labels, FALSE, TRUE, 0);
	g_signal_connect (G_OBJECT (enable_labels), "toggled",
						G_CALLBACK (gui_lbl_show_changed), NULL);
	gtk_widget_show (enable_labels);

    save_win_pos = gtk_check_button_new_with_label(_("Save window position on exit"));
	gtk_box_pack_start (GTK_BOX (vbox4), save_win_pos, FALSE, TRUE, 0);
	gtk_widget_show (save_win_pos);

	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 6);

	label = gtk_label_new (NULL);
	gtk_widget_show (label);
	gtk_frame_set_label_widget (GTK_FRAME (frame), label);
	sprintf (tmpbuf, _("<i> Labels font </i>"));
	gtk_label_set_markup (GTK_LABEL (label), tmpbuf);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_container_add (GTK_CONTAINER (frame), hbox2);

	fontname_entry = gtk_entry_new ();
	gtk_widget_show (fontname_entry);
	gtk_box_pack_start (GTK_BOX (hbox2), fontname_entry, TRUE, TRUE, 4);
	gtk_editable_set_editable (GTK_EDITABLE (fontname_entry), FALSE);

	fontsel_button = gtk_button_new_from_stock (GTK_STOCK_SELECT_FONT);
	gtk_widget_show (fontsel_button);
	g_signal_connect (G_OBJECT (fontsel_button), "clicked",
						G_CALLBACK (gui_font_select), NULL);
	gtk_box_pack_start (GTK_BOX (hbox2), fontsel_button, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (fontsel_button), 2);

	spage1 = gtk_label_new (_("Main"));
	gtk_widget_show (spage1);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 0), spage1);

	/******************************************************************************/
	/* PRESETS */

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_show (scrolledwindow);
	gtk_container_add (GTK_CONTAINER (notebook), scrolledwindow);

	viewport = gtk_viewport_new (NULL, NULL);
	gtk_widget_show (viewport);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), viewport);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_container_add (GTK_CONTAINER (viewport), vbox2);

	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 6);

	label = gtk_label_new (NULL);
	gtk_widget_show (label);
	gtk_frame_set_label_widget (GTK_FRAME (frame), label);
	sprintf (tmpbuf, _("<i> Presets names </i>"));
	gtk_label_set_markup (GTK_LABEL (label), tmpbuf);

	vbox4 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox4);
	gtk_container_add (GTK_CONTAINER (frame), vbox4);

	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox4), frame, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 2);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox1);
	gtk_container_add (GTK_CONTAINER (frame), hbox1);
	label = gtk_label_new (_("Preset 1 name: "));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 2);
	p1_entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(p1_entry), 20);
	g_signal_connect (G_OBJECT (p1_entry), "key_press_event",
						G_CALLBACK (gui_options_entries_activity), NULL);
	gtk_widget_show (p1_entry);
	gtk_box_pack_start (GTK_BOX (hbox1), p1_entry, FALSE, FALSE, 0);


	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox4), frame, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 2);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox1);
	gtk_container_add (GTK_CONTAINER (frame), hbox1);
	label = gtk_label_new (_("Preset 2 name: "));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 2);
	p2_entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(p2_entry), 20);
	g_signal_connect (G_OBJECT (p2_entry), "key_press_event",
						G_CALLBACK (gui_options_entries_activity), NULL);
	gtk_widget_show (p2_entry);
	gtk_box_pack_start (GTK_BOX (hbox1), p2_entry, FALSE, FALSE, 0);


	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox4), frame, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 2);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox1);
	gtk_container_add (GTK_CONTAINER (frame), hbox1);
	label = gtk_label_new (_("Preset 3 name: "));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 2);
	p3_entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(p3_entry), 20);
	g_signal_connect (G_OBJECT (p3_entry), "key_press_event",
						G_CALLBACK (gui_options_entries_activity), NULL);
	gtk_widget_show (p3_entry);
	gtk_box_pack_start (GTK_BOX (hbox1), p3_entry, FALSE, FALSE, 0);


	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox4), frame, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 2);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox1);
	gtk_container_add (GTK_CONTAINER (frame), hbox1);
	label = gtk_label_new (_("Preset 4 name: "));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 2);
	p4_entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(p4_entry), 20);
	g_signal_connect (G_OBJECT (p4_entry), "key_press_event",
						G_CALLBACK (gui_options_entries_activity), NULL);
	gtk_widget_show (p4_entry);
	gtk_box_pack_start (GTK_BOX (hbox1), p4_entry, FALSE, FALSE, 0);



	spage2 = gtk_label_new (_("Presets"));
	gtk_widget_show (spage2);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 1), spage2);

	/******************************************************************************/
	/* DEVICES */
/*
	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_show (scrolledwindow);
	gtk_container_add (GTK_CONTAINER (notebook), scrolledwindow);

	viewport = gtk_viewport_new (NULL, NULL);
	gtk_widget_show (viewport);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), viewport);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_container_add (GTK_CONTAINER (viewport), vbox2);




	spage3 = gtk_label_new (_("Devices"));
	gtk_widget_show (spage3);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 2), spage3);
*/

	/******************************************************************************/
	/* CHANNELS */

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_show (scrolledwindow);
	gtk_container_add (GTK_CONTAINER (notebook), scrolledwindow);

	viewport = gtk_viewport_new (NULL, NULL);
	gtk_widget_show (viewport);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), viewport);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_container_add (GTK_CONTAINER (viewport), vbox2);

	frame = gtk_frame_new (NULL);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 6);

	label = gtk_label_new (NULL);
	gtk_widget_show (label);
	gtk_frame_set_label_widget (GTK_FRAME (frame), label);
	sprintf (tmpbuf, _("<i> Enable/Disable (current preset) </i>"));
	gtk_label_set_markup (GTK_LABEL (label), tmpbuf);

    
	vbox4 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox4);
	gtk_container_add (GTK_CONTAINER (frame), vbox4);

	for(i=0;i<channels;i++) {

		frame = gtk_frame_new (NULL);
		gtk_widget_show (frame);
		gtk_box_pack_start (GTK_BOX (vbox4), frame, FALSE, FALSE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (frame), 2);
		gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);

		hbox1 = gtk_hbox_new (FALSE, 0);
		gtk_widget_show (hbox1);
		gtk_container_add (GTK_CONTAINER (frame), hbox1);

		entries[i] = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(entries[i]), 16);
		g_signal_connect (G_OBJECT (entries[i]), "key_press_event",
						G_CALLBACK (gui_options_entries_activity), NULL);
		gtk_widget_show (entries[i]);
   		gtk_box_pack_start (GTK_BOX (hbox1), entries[i], FALSE, FALSE, 0);
		gtk_entry_set_text (GTK_ENTRY(entries[i]), config.channels_names[i]);

	    chn = gtk_check_button_new_with_label(mixer_get_channel_name (i));
   		gtk_box_pack_start (GTK_BOX (hbox1), chn, TRUE, TRUE, 0);
		g_signal_connect (G_OBJECT (chn), "toggled",
						G_CALLBACK (gui_chn_show_changed), GINT_TO_POINTER(i));
		gtk_widget_show (chn);

		if(config.show_state[config.current_preset][i])
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (chn), TRUE);
	}

	spage3 = gtk_label_new (_("Channels"));
	gtk_widget_show (spage3);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 2), spage3);

	/******************************************************************************/
	/* CREDITS */

	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox3);
	gtk_container_add (GTK_CONTAINER (notebook), vbox3);

	mix_logo = gtk_image_new_from_file(PIXMAPS "logo.png");
	gtk_widget_show (mix_logo);
	gtk_box_pack_start (GTK_BOX (vbox3), mix_logo, FALSE, TRUE, 8);

	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox1);
	gtk_box_pack_start (GTK_BOX (vbox3), hbox1, FALSE, FALSE, 2);

	label = gtk_label_new ("version " VERSION);
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox3), label, FALSE, FALSE, 2);

	vbox4 = gtk_vbox_new (TRUE, 0);
	gtk_widget_show (vbox4);
	gtk_box_pack_start (GTK_BOX (vbox3), vbox4, FALSE, TRUE, 8);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox3), hbox2, TRUE, TRUE, 0);

	label = gtk_label_new ("");
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox2), label, TRUE, FALSE, 8);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>CODE</b>"));

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox3), hbox2, TRUE, TRUE, 0);

	label = gtk_label_new ("Tomasz Mąka <pasp@ll.pl>");
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox2), label, TRUE, FALSE, 8);

	vbox4 = gtk_vbox_new (TRUE, 0);
	gtk_widget_show (vbox4);
	gtk_box_pack_start (GTK_BOX (vbox3), vbox4, FALSE, TRUE, 8);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox3), hbox2, TRUE, TRUE, 0);

	label = gtk_label_new ("");
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox2), label, TRUE, FALSE, 8);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>GRAPHICS</b>"));

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox3), hbox2, TRUE, TRUE, 0);

	label = gtk_label_new ("Maja Kocoń <ironya@ll.pl>");
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox2), label, TRUE, FALSE, 8);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox3), hbox2, TRUE, TRUE, 0);

	label = gtk_label_new ("Piotr Mąka <sill@clay.ll.pl>");
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox2), label, TRUE, FALSE, 8);

	vbox4 = gtk_vbox_new (TRUE, 0);
	gtk_widget_show (vbox4);
	gtk_box_pack_start (GTK_BOX (vbox3), vbox4, FALSE, TRUE, 8);


	spage4 = gtk_label_new (_("Credits"));
	gtk_widget_show (spage4);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 3), spage4);

	/******************************************************************************/

	hseparator = gtk_hseparator_new ();
	gtk_widget_show (hseparator);
	gtk_box_pack_start (GTK_BOX (vbox1), hseparator, FALSE, TRUE, 3);

	bbox = gtk_hbutton_box_new();
	gtk_widget_show (bbox);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
	gtk_box_set_spacing(GTK_BOX(bbox), 5);
	gtk_box_pack_start (GTK_BOX (vbox1), bbox, FALSE, TRUE, 0);
    
	close_button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
	gtk_widget_show (close_button);
	gtk_box_pack_start (GTK_BOX (bbox), close_button, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (close_button), 4);
	GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
	g_signal_connect (G_OBJECT (close_button), "clicked",
						G_CALLBACK (gui_options_window_close), NULL);
       
	gtk_widget_grab_default (close_button);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (save_win_pos), config.save_window_pos);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (enable_pixmaps), config.enable_pixmaps);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (enable_labels), config.enable_labels);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (enable_selector), config.enable_preset_selector);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (store_confirm), config.store_confirmation);

	gtk_entry_set_text (GTK_ENTRY(p1_entry), config.preset_1_name);
	gtk_entry_set_text (GTK_ENTRY(p2_entry), config.preset_2_name);
	gtk_entry_set_text (GTK_ENTRY(p3_entry), config.preset_3_name);
	gtk_entry_set_text (GTK_ENTRY(p4_entry), config.preset_4_name);

	for(i=0;i<channels;i++)
		gtk_entry_set_text (GTK_ENTRY(entries[i]), config.channels_names[i]);

	gtk_entry_set_text(GTK_ENTRY(fontname_entry), config.labels_font);

	gtk_widget_show (options_window);

}

/******************************************************************************/

void gui_show_info(void)
{

	  info_dialog = gtk_message_dialog_new (GTK_WINDOW (main_window),
					   GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
					   GTK_MESSAGE_INFO,
					   GTK_BUTTONS_OK,
					   _("You will need to restart mixmos for these changes to take effect."));

	  gtk_widget_show (info_dialog);

	  g_signal_connect (G_OBJECT(info_dialog), "response",
			    G_CALLBACK (gtk_widget_destroy), NULL);

}

/******************************************************************************/

