
/*
 * MIXMOS - an oss mixer
 *
 * Copyright (C) 2003 Tomasz Mka
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#include <gtk/gtk.h>

#include "prefs.h"
#include "mixer.h"

struct mixprefs config;

FILE *prefs_filehandle;


/*---------------------------------------------------------------------------*/

gchar* s_getdir_config (void)
{
static gchar cfgdir[MAX_PATH];
struct stat cfg;

	s_strcpy (cfgdir, getenv ("HOME"), MAX_PATH);
	s_strcat (cfgdir, "/.clay", MAX_PATH);

	if(stat (cfgdir, &cfg) < 0)
		mkdir (cfgdir, S_IRUSR | S_IWUSR | S_IXUSR);

	return cfgdir;
}

/*---------------------------------------------------------------------------*/

gchar* s_getfilename_config (void)
{
static gchar filename[MAX_PATH];

	s_strcpy (filename, s_getdir_config(), MAX_PATH);
	s_strcat (filename, slash, MAX_PATH);
	s_strcat (filename, config_filename, MAX_PATH);

	return filename;
}

/*---------------------------------------------------------------------------*/

void* s_prefs_openfile (gchar *filename, gint openmode)
{
	prefs_filehandle = NULL;

	if (openmode == P_READ)
		prefs_filehandle = fopen (filename, "rb");
	else if (openmode == P_WRITE)
		prefs_filehandle = fopen (filename, "wb");
    
	return prefs_filehandle;
}


void s_prefs_closefile (void)
{
	fclose (prefs_filehandle);
}

/*---------------------------------------------------------------------------*/

void s_prefs_put_int (gchar *tagname, gint value)
{
	fprintf (prefs_filehandle, "%s = %d\n", tagname, value);
}


void s_prefs_put_float (gchar *tagname, gfloat value)
{
	fprintf (prefs_filehandle, "%s = %f\n", tagname, value);
}


void s_prefs_put_string (gchar *tagname, gchar *value)
{
	fprintf (prefs_filehandle, "%s = %s\n", tagname, value);
}


void s_prefs_put_lf (void)
{
	fprintf (prefs_filehandle, "\n");
}


void s_prefs_put_comment (gchar *comment)
{
gchar text[MAX_LINE_LEN];

		s_strcpy (text, "# ", MAX_LINE_LEN);
		s_strcat (text, comment, MAX_LINE_LEN);
		fprintf (prefs_filehandle, text);
}

/*---------------------------------------------------------------------------*/

gchar* s_prefs_get_line_with_tag (gchar *tagname)
{
static gchar prfline[MAX_LINE_LEN];
gint i;
gchar c;

	fseek (prefs_filehandle, 0, SEEK_SET);	
    
	while (!feof (prefs_filehandle)) {
		i = 0;

		while (((c = fgetc (prefs_filehandle)) != crlf_char) && c!= EOF && i < MAX_LINE_LEN)
			prfline[i++] = c;
            
		prfline[i] = null_char;
        
		if (prfline[0] != '#')
			if (!strncmp (tagname, prfline, strlen (tagname))) break;
	}
        
	return prfline;
}


gchar* s_prefs_get_value_field (gchar *tagname)
{
static gchar valuestr[MAX_VALUE_LEN];
gchar *valpos, c;
gint i;

	i = 0;

	if ((valpos = strchr (s_prefs_get_line_with_tag (tagname), '='))) {
		while((c = valpos[i+2]) != null_char && i < MAX_VALUE_LEN) valuestr[i++] = c;
	}
    
	valuestr[i] = null_char;
	return valuestr;
}


gint s_prefs_get_int (gchar *tagname)
{
	return (atoi (s_prefs_get_value_field (tagname)));
}


gfloat s_prefs_get_float (gchar *tagname)
{
	return (atof (s_prefs_get_value_field (tagname)));
}


gchar* s_prefs_get_string (gchar *tagname)
{
	return (s_prefs_get_value_field (tagname));
}

/*---------------------------------------------------------------------------*/
/* following functions based on samba sources.                               */
/* safe_strcpy and safe_strcat routines written by Andrew Tridgell           */
/*---------------------------------------------------------------------------*/

gchar* s_strcpy (gchar *dest, const gchar *src, guint maxlength)
{
guint len;

	if (!dest) {
		printf ("ERROR: NULL dest in safe_strcpy\n");
		return NULL;
	}

	if (!src) {
		*dest = 0;
		return dest;
	}

	len = strlen(src);

	if (len > maxlength) {
		printf ("ERROR: string overflow by %d in safe_strcpy [%.50s]\n",
				(int)(len-maxlength), src);
		len = maxlength;
	}
      
	memcpy(dest, src, len);
	dest[len] = 0;
	return dest;
}

/*---------------------------------------------------------------------------*/

gchar* s_strcat (gchar *dest, const gchar *src, guint maxlength)
{
guint src_len, dest_len;

	if (!dest) {
		printf ("ERROR: NULL dest in safe_strcat\n");
		return NULL;
	}

	if (!src) {
		return dest;
	}

	src_len = strlen(src);
	dest_len = strlen(dest);

	if (src_len + dest_len > maxlength) {
		printf ("ERROR: string overflow by %d in safe_strcat [%.50s]\n",
				(int)(src_len + dest_len - maxlength), src);
		src_len = maxlength - dest_len;
	}
      
	memcpy(&dest[dest_len], src, src_len);
	dest[dest_len + src_len] = 0;
	return dest;
}

/*---------------------------------------------------------------------------*/

void read_config (void)
{
guint i, j;
gchar tmpbuffer[128];

	if (s_prefs_openfile (s_getfilename_config (), P_READ)) {
    
		config.window_x = s_prefs_get_int ("window_pos_x");
		config.window_y = s_prefs_get_int ("window_pos_y");
		config.window_size_x = s_prefs_get_int ("window_size_x");
		config.window_size_y = s_prefs_get_int ("window_size_y");

		config.opt_window_x = s_prefs_get_int ("opt_window_pos_x");
		config.opt_window_y = s_prefs_get_int ("opt_window_pos_y");

		config.run_counter = s_prefs_get_int ("run_counter");
		config.pane_size = s_prefs_get_int ("pane_size");

		config.save_window_pos = s_prefs_get_int ("save_window_pos");
		config.enable_tooltips = s_prefs_get_int ("enable_tooltips");
		config.enable_pixmaps = s_prefs_get_int ("enable_pixmaps");
		config.enable_labels = s_prefs_get_int ("enable_labels");
		config.enable_preset_selector = s_prefs_get_int ("enable_preset_selector");

		config.current_preset = s_prefs_get_int ("current_preset");
		config.store_confirmation = s_prefs_get_int ("store_confirmation");

		strncpy(config.preset_1_name, s_prefs_get_string("preset_1_name"), 20);
		strncpy(config.preset_2_name, s_prefs_get_string("preset_2_name"), 20);
		strncpy(config.preset_3_name, s_prefs_get_string("preset_3_name"), 20);
		strncpy(config.preset_4_name, s_prefs_get_string("preset_4_name"), 20);

		strncpy(config.labels_font, s_prefs_get_string("labels_font"), 128);

		for(j=0;j<4;j++) {
			sprintf(tmpbuffer, "a_rec[%d]", j);
			config.a_rec[j] = s_prefs_get_int (tmpbuffer);
		}

		for(i=0;i<mixer_get_channels();i++) {
			sprintf(tmpbuffer, "chn_name[%d]", i);
			strncpy(config.channels_names[i], s_prefs_get_string (tmpbuffer), 16);
		}

		for(j=0;j<4;j++) {
			for(i=0;i<mixer_get_channels();i++) {
				sprintf(tmpbuffer, "show_%d[%d]", j, i);
				config.show_state[j][i] = s_prefs_get_int (tmpbuffer);
				sprintf(tmpbuffer, "lock_%d[%d]", j, i);
				config.lock_state[j][i] = s_prefs_get_int (tmpbuffer);
				sprintf(tmpbuffer, "mute_%d[%d]", j, i);
				config.mute_state[j][i] = s_prefs_get_int (tmpbuffer);
				sprintf(tmpbuffer, "a_vol_l_%d[%d]", j, i);
				config.a_vol_l[j][i] = s_prefs_get_int (tmpbuffer);
				sprintf(tmpbuffer, "a_vol_r_%d[%d]", j, i);
				config.a_vol_r[j][i] = s_prefs_get_int (tmpbuffer);
			}
		}

		s_prefs_closefile ();
        
	} else {

		config.window_x = 50;
		config.window_y = 100;
		config.window_size_x = 545;
		config.window_size_y = 320;

		config.opt_window_x = 150;
		config.opt_window_y = 140;

		config.run_counter = 0;
		config.pane_size = 420;

		config.save_window_pos = 1;
		config.enable_tooltips = 1;
		config.enable_pixmaps = 1;
		config.enable_labels = 1;
		config.enable_preset_selector = 1;

		config.current_preset = 0;
		config.store_confirmation = 1;

		strncpy(config.preset_1_name, "preset 1", 20);
		strncpy(config.preset_2_name, "preset 2", 20);
		strncpy(config.preset_3_name, "preset 3", 20);
		strncpy(config.preset_4_name, "preset 4", 20);

		strncpy(config.labels_font, "Sans 8", 128);

		for(j=0;j<4;j++)
			config.a_rec[j] = 0;

		for(i=0;i<mixer_get_channels();i++)
			strncpy(config.channels_names[i], mixer_get_channel_name (i), 16);

		for(j=0;j<4;j++) {
			for(i=0;i<mixer_get_channels();i++) {
				config.show_state[j][i] = 1;
				config.lock_state[j][i] = 0;
				config.mute_state[j][i] = 0;
				config.a_vol_l[j][i] = 75;
				config.a_vol_r[j][i] = 75;
			}
		}

		write_config ();
	}
    
}

/*------------------------------------------------------------------------*/

void write_config(void)
{
guint i, j;
gchar tmpbuffer[128];

	if (s_prefs_openfile (s_getfilename_config (), P_WRITE)) {
    
		s_prefs_put_int ("window_pos_x", config.window_x);
		s_prefs_put_int ("window_pos_y", config.window_y);
		s_prefs_put_int ("window_size_x", config.window_size_x);
		s_prefs_put_int ("window_size_y", config.window_size_y);

		s_prefs_put_int ("opt_window_pos_x", config.opt_window_x);
		s_prefs_put_int ("opt_window_pos_y", config.opt_window_y);

		s_prefs_put_int ("run_counter", config.run_counter);
		s_prefs_put_int ("pane_size", config.pane_size);

		s_prefs_put_int ("save_window_pos", config.save_window_pos);
		s_prefs_put_int ("enable_tooltips", config.enable_tooltips);
		s_prefs_put_int ("enable_pixmaps", config.enable_pixmaps);
		s_prefs_put_int ("enable_labels", config.enable_labels);
		s_prefs_put_int ("enable_preset_selector", config.enable_preset_selector);
		s_prefs_put_int ("current_preset", config.current_preset);
		s_prefs_put_int ("store_confirmation", config.store_confirmation);

		s_prefs_put_string ("preset_1_name", config.preset_1_name);
		s_prefs_put_string ("preset_2_name", config.preset_2_name);
		s_prefs_put_string ("preset_3_name", config.preset_3_name);
		s_prefs_put_string ("preset_4_name", config.preset_4_name);

		s_prefs_put_string ("labels_font", config.labels_font);

		for(j=0;j<4;j++) {
			sprintf(tmpbuffer, "a_rec[%d]", j);
			s_prefs_put_int (tmpbuffer, config.a_rec[j]);
		}

		for(i=0;i<mixer_get_channels();i++) {
			sprintf(tmpbuffer, "chn_name[%d]", i);
			s_prefs_put_string (tmpbuffer, config.channels_names[i]);
		}

		for(j=0;j<4;j++) {
			for(i=0;i<mixer_get_channels();i++) {
				sprintf(tmpbuffer, "show_%d[%d]", j, i);
				s_prefs_put_int (tmpbuffer, config.show_state[j][i]);
				sprintf(tmpbuffer, "lock_%d[%d]", j, i);
				s_prefs_put_int (tmpbuffer, config.lock_state[j][i]);
				sprintf(tmpbuffer, "mute_%d[%d]", j, i);
				s_prefs_put_int (tmpbuffer, config.mute_state[j][i]);
				sprintf(tmpbuffer, "a_vol_l_%d[%d]", j, i);
				s_prefs_put_int (tmpbuffer, config.a_vol_l[j][i]);
				sprintf(tmpbuffer, "a_vol_r_%d[%d]", j, i);
				s_prefs_put_int (tmpbuffer, config.a_vol_r[j][i]);
			}
		}

	}
        
	s_prefs_closefile ();
}
    
/*------------------------------------------------------------------------*/

