/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Handles command line switches.  Instantiates the Mobile
 * Mesh Router and runs it.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *
 *
 * $Id: MmMain.C,v 1.4 2004/04/19 10:56:42 br1 Exp $
 *
 */

#include <MmRouter.h>
#include <UtMain.h>
#include <UtString.h>
#include <UtReport.h>

static Router* router = 0;

static void Version() {
  char *version = "mmrp - Mobile Mesh Routing Protocol v1.0\n";
  Report::Info(version);
}

static void Usage() {
  char *usage = "Usage: mmrp [-hvz] [-d file] [-f file] \n"\
    " -d file, use flags 'file' to enable debug output \n"\
    " -f file, use config 'file' instead of mmrp.conf \n"\
    " -h     , display this help message \n"\
    " -v     , display version information \n"\
    " -e     , turns encryption on\n"\
    " -z     , run in the foreground instead of running as a daemon \n";
  Report::Error(usage);
}

// isEncrypted defined in UtUDPsocket.h 

int main(int argc, char* argv[]) {
  String ident("mmrp");
  String configFile  = "/etc/mobilemesh/mmrp.conf";
  String debugFile   = "";
  int    flags       = Report::fError | Report::fWarn | Report::fDebug;
  String ifname = "";
  bool isDaemon = true;

  Report::UseOstream(&cerr, flags);

  // presets isEncryption to no
  isEncrypted = false;
  // Parse all command line switches
  bool done = false;
  while(!done) {
    int c = getopt(argc,argv,"d:f:hi:vez");
    switch(c) {
    case 'd':
      debugFile = optarg;
      break;
    case 'f':
      configFile = optarg;
      break;
    case 'h':
      Usage();
      break;
    case 'i':
      ifname = optarg;
      break;
    case 'v':
      Version();
      exit(0);
      break;
    case 'e':
      isEncrypted = true;
      break;
    case 'z':
      isDaemon = false;
      break;
    case EOF:
      done = true;
      break;
    default:
      Usage();
    }
  }
  if(optind < argc) Usage();

  if(isDaemon) {
    Daemonize();           
    Report::UseOstream(0); // Don't send anything to cerr

    // Use syslog to record output
    Report::UseSyslog(ident,flags);
  }

  // Convert any relative file names to absolute paths so we can access them after
  // changing from the current directory
  debugFile  = AbsoluteFilename(debugFile);
  configFile = AbsoluteFilename(configFile);

  // We must run in the directory where the Unix sockets exists because
  // their names are limited to something like 14 characters
  Chdir("/var/run/mobilemesh");
    
  // In case a signal occurs, register for garbage collection
  GarbageCollection<Router>::Register(&router);

  // Build the router and run it
  router = new Router(ifname,configFile,debugFile);
  router->Run();
  
  // Clean up if we get here
  delete(router);
  router = 0;

  return(0);
}












