/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Utilities needed by various applications
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id: UtMain.h,v 1.1.1.1 2003/10/24 10:34:57 br1 Exp $
 *  
 */
#ifndef  __UtMain_h
#define  __UtMain_h

#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <UtString.h>
#include <UtReport.h>

/* A class for ensuring that a dynamically created object is 
 * deleted prior to exiting due to a signal.
 * NOTE: only a single object can be registered...but it can be of any type.
 */
template<class T> 
class GarbageCollection {
  static T** sApplication;
  
  /* We attempt to support a graceful shutdown by catching signals and
   * closing down.
   */
  static void SignalHandler(int signum) {
    String s = String("SignalHandler() - caught signal ") +
      strsignal(signum) + ", closing down.";
    Report::Info(s);
    delete(*sApplication);
    *sApplication = 0;
    exit(0);
  }

  static void Cleanup() {
    delete(*sApplication);
    *sApplication = 0;
  }

public:
  /* Register a pointer to a pointer to an object which will be 
     deleted if a signal is caught
  */
  static void Register(T** app) {
    sApplication = app;

    // Catch these signals so that we may clean up before dieing
    signal(SIGHUP,  SignalHandler);
    signal(SIGINT,  SignalHandler);
    signal(SIGQUIT, SignalHandler);
    signal(SIGILL,  SignalHandler);
    signal(SIGABRT, SignalHandler);
    signal(SIGFPE,  SignalHandler);
    signal(SIGPIPE, SignalHandler);
    signal(SIGALRM, SignalHandler);
    signal(SIGTERM, SignalHandler);
    atexit(Cleanup);
  }
};

template<class T> T** GarbageCollection<T>::sApplication = 0;

void Daemonize() {
  pid_t pid;
  if((pid = fork()) != 0) exit(0); // Parent teminates
  setsid();                        // Become session leader
  if((pid = fork()) != 0) exit(0); // 1st child teminates
  umask(0);
  for(int i=0; i<64; i++) {
    close(i);
  }
}
	
// Convert relative file name to absolute path file name
String AbsoluteFilename(const String& fname) {
  String result = fname;
  const int BUFSIZE = 2048;
  char buf[BUFSIZE];
  if(getcwd(buf,BUFSIZE)) {
    String abspath = String(buf) + "/"; 
    if(fname.length() && (fname[0] != '/')) {
      result = abspath + fname;
    }
  }
  return(result);
}

void Chdir(const String& dir) {
  int result = chdir(dir.str());
  if(result) {
    Report::Error(String("Could not change directory to ") + dir + ": " + strerror(errno));
  }
}

#endif



