/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Classes for handling Unix sockets.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         Mike Butler,    mgb@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *
 * $Id: UtUnixSocket.h,v 1.2 2003/10/24 11:08:14 br1 Exp $
 */

#ifndef __UnixSocket_h
#define __UnixSocket_h

#include <UtReport.h>
#include <UtString.h>

#include <cerrno>
#include <cstdio>
#include <sys/un.h>
#include <sys/socket.h>
#include <strstream>
#include <unistd.h>

/* A facade class to facilitate converting between 'sockaddr' and 'sockaddr_un' structs
 */
class UnixAddr {
  friend ostream &operator<<(ostream &os, const UnixAddr &rhs) {
    os << rhs.cAddr.sun_path;
    return(os);
  }
private:
  sockaddr_un cAddr;
public:
  UnixAddr(const String &path = "") { 
    memset(&cAddr, 0, sizeof(cAddr));
    cAddr.sun_family = AF_LOCAL;
    unsigned int maxPath = sizeof(cAddr.sun_path);
    unsigned int len = ((path.length() > maxPath) ? maxPath : path.length());
    unsigned int term = ((path.length() >= maxPath) ? maxPath - 1 : path.length());
    strncpy(cAddr.sun_path,path.c_str(),len);
    // Make sure we're null terminated
    cAddr.sun_path[term] = 0;
  }
  UnixAddr(const sockaddr& sa)     { cAddr = (*((sockaddr_un*)&sa)); }
  UnixAddr(const sockaddr_un &sa) { cAddr = sa; }

  operator String() const { 
	  std::ostrstream tmp;
    tmp << (*this) << ends;
    char *s = tmp.str();
    String result(s);
    delete [] s;
    return(result);
  }
  String Path() const {
    return(String(cAddr.sun_path));
  }
  operator sockaddr() const { 
    return(*((sockaddr *)&cAddr));
  };
  operator sockaddr_un() const { return(cAddr); }
  const size_t Size() const { return(sizeof(cAddr)); }
};

class UnixDatagram {
  friend ostream &operator<<(ostream &os, const UnixDatagram &rhs) {
    os << "(" << rhs.cAddr << ") \"" << rhs.cData << "\"";
    return(os);
  }
private:
  UnixAddr cAddr;
  String cData;
public:
  UnixDatagram(const String &s, UnixAddr addr) : cAddr(addr), cData(s) {}
  UnixDatagram(const String &s = "") : cData(s) {}
  const String &GetData() const { return(cData); }
  const String &SetData(const String &d) { return(cData = d); }
  const UnixAddr &GetAddr() const { return(cAddr); }
  void  SetAddr(const UnixAddr &addr) { cAddr = addr; }
  size_t Length() const { return(cData.length()); }
  
};

class UnixSocket {
  int cFd;
  UnixAddr cListenAddr;
  
public:
  UnixSocket(const UnixAddr& listen = UnixAddr("")) {
    if ((cFd = socket(AF_LOCAL, SOCK_DGRAM, 0)) < 0) {
      Report::Error("UnixSocket::UnixSocket() - socket() failed.");
    }

    if(listen.Path().length() == 0) {
      //      cListenAddr = UnixAddr(tmpnam(0));
      char buf[128];
      strcpy(buf,"dump-XXXXXX");
      cListenAddr = UnixAddr(mktemp(buf));
    } 
    else {
      cListenAddr = listen;
      unlink(listen.Path().c_str());
    }


    if(cListenAddr.Path().length() == 0) {
      Report::Error("UnixSocket::UnixSocket() - invalid path for listening.");
    }

    sockaddr addr = cListenAddr;
    if(bind(cFd, &addr, sizeof(addr))) {
      close(cFd);
      char buf[1024];
      sprintf(buf,"UnixSocket::UnixSocket() - bind() failed: %s",strerror(errno));
      Report::Error(buf);
    }

  }
  
  ~UnixSocket() { close(cFd); }

  int Fd() { return cFd; }

  UnixDatagram Recv() {
    char buf[2048];
    sockaddr addr = cListenAddr;
    
    int len;
    socklen_t aLen = sizeof(addr);
    len = recvfrom(cFd, buf, sizeof(buf), 0, &addr, &aLen);
    if(len < 0) {
      char buf[1024];
      sprintf(buf,"UnixSocket::Recv() - recvfrom() failed: %s",strerror(errno));
      Report::Error(buf);
    }
    return(UnixDatagram(String(buf, 0, len), addr));
  }

  bool Send(const UnixDatagram &data) {
    int len;
    sockaddr addr = data.GetAddr();
    
    len = sendto(cFd, (void *)data.GetData().c_str(), data.Length(), 
		 0, &addr, sizeof(addr));
    
    if(len <= 0) {
      char buf[1024];
      sprintf(buf,"UnixSocket::Send() - sendto() failed: %s",strerror(errno));
      Report::Warn(buf);
      return(false);
    }
    return(true);
  }

};
  
#if 0
int main(int argc, char* argv[]) {
  if((argc != 2) && (argc != 3)) {
    cerr << "Usage: dbUnix <server path> [client]" << endl;
  }
  
  String serverPath = String(argv[1]);
  UnixAddr serverAddr(serverPath);
  bool isClient = (argc == 3);
  
  String listen = (isClient ? "" : serverPath);

  UnixSocket sock(listen);
  
  // Find the max number of file descriptors for this process
  int fdlimit = sock.Fd() + 1;

  while(1) {
    fd_set fds;
    FD_ZERO(&fds);
    if(isClient) FD_SET(0,&fds);
    FD_SET(sock.Fd(),&fds);

    int n;
    if((n=select(fdlimit, &fds, 0, 0, 0)) == -1) {
      cerr << "Call to select() failed! - " << strerror(errno) << endl;
      return 1;
    }
    
    if(n > 0) {
      if(FD_ISSET(sock.Fd(),&fds)) {
	// Handle receive
	UnixDatagram dg = sock.Recv();
	cerr << "From:  " << String(dg.GetAddr()) << " '" << dg.GetData() << "'\n";
	if(!isClient) {
	  String s = listen + " heard '" + dg.GetData() + "'";
	  UnixDatagram reply(s, dg.GetAddr());
	  sock.Send(reply);
	}
      }
      if(isClient && FD_ISSET(0,&fds)) {
	String s;
	s.Getline(cin);
	cerr << "Sending: '" << s << "'" << endl;
	UnixDatagram dg(s, serverAddr);
	sock.Send(dg);
      }
    }
  }
  return(0);
}

#endif

#endif
