/*	motion.c
 *
 *	Detect changes in a video stream.
 *	Copyright 2000 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"

#ifdef __freebsd__
#include "video_freebsd.h"
#else
#include "video.h"
#endif /* __freebsd__ */

#include "conf.h"
#include "alg.h"
#include "track.h"
#include "event.h"
#include "picture.h"
#include "ffmpeg.h"
#include "rotate.h"

struct context **cnt_list=NULL;

int threads_running=0;
int restart=0;

typedef struct
{
	pthread_mutex_t lock;
	char **list;
	int size, maxsize;
	char finish;
} storethread;

static void context_init (struct context *cnt)
{
	/*
	* We first clear the entire structure to zero, then fill in any
	* values which have non-zero default values.  Note that this
	* assumes that a NULL address pointer has a value of binary 0
	* (this is also assumed at other places within the code, i.e.
	* there are instances of "if (ptr)").  Just for possible future
	* changes to this assumption, any pointers which are intended
	* to be initialised to NULL are listed within a comment.
	*/

	memset(cnt, 0, sizeof(struct context));
	cnt->noise=255;
	cnt->currenttime=NULL;
	cnt->lastrate=25;

	memcpy(&cnt->track, &track_template, sizeof(struct trackoptions));
	cnt->pipe=-1;
	cnt->mpipe=-1;

#if 0                 /* Here are the pointers required to be set */
#ifdef HAVE_MYSQL
	cnt->database=NULL;
#endif /* HAVE_MYSQL */

#ifdef HAVE_PGSQL
	cnt->database_pg=NULL;
#endif /* HAVE_PGSQL */

#ifdef HAVE_FFMPEG
	cnt->ffmpeg_new=NULL;
	cnt->ffmpeg_motion=NULL;
	cnt->ffmpeg_timelapse=NULL;
#endif /* HAVE_FFMPEG */
#endif
}

/*	Our SIGNAL-Handler
 *	We need this to handle alarms and external signals
 */
static void sig_handler(int signo)
{
	int i;

	switch(signo) {
		case SIGHUP: {
			const char msg[]="Received SIGHUP, preparing restart";
			printf("%s\n", msg);
			syslog(LOG_DEBUG, "%s\n", msg);
			restart=1;
			if (cnt_list) {
				i=-1;
				while (cnt_list[++i]) {
					cnt_list[i]->makemovie=1;
					cnt_list[i]->finish=1;
					if(cnt_list[i]->pipe != -1)
						close(cnt_list[i]->pipe);
					if(cnt_list[i]->mpipe != -1)
						close(cnt_list[i]->mpipe);
				}
			}
			signal(SIGHUP, sig_handler);
			return;
		}
		case SIGALRM: {
			/* Somebody (maybe we ourself) wants us to make a snapshot
			 * This feature triggers snapshots on ALL threads that have
			 * snapshot_interval different from 0.
			 */
			if (cnt_list) {
				i=-1;
				while (cnt_list[++i]) {
					if (cnt_list[i]->conf.snapshot_interval) {
						cnt_list[i]->snapshot=1;
					}
				}
			}
			/* Set everything for the next snapshot */
			signal(SIGALRM, sig_handler);
			return;
		}
		case SIGUSR1: {
			/* Ouch! We have been hit from the outside! Someone wants us to
			   make a movie! */
			if (cnt_list) {
				i=-1;
				while (cnt_list[++i])
					cnt_list[i]->makemovie=1;
			}
			signal(SIGUSR1, sig_handler);
			return;
		}
		case SIGINT:
		case SIGQUIT:
		case SIGTERM: {

			/* Somebody wants us to quit! We should better finish the actual
			    movie and end up! */
			signal(SIGINT, sig_handler);
			signal(SIGQUIT, sig_handler);
			signal(SIGTERM, sig_handler);
			if (cnt_list) {
				i=-1;
				while (cnt_list[++i]) {
					cnt_list[i]->makemovie=1;
					cnt_list[i]->finish=1;
					if(cnt_list[i]->pipe != -1)
						close(cnt_list[i]->pipe);
					if(cnt_list[i]->mpipe != -1)
						close(cnt_list[i]->mpipe);
				}
			}
			return;
		}
		default: {
			const char msg[]="Unknown signal: ";
			printf("%s%d\n", msg, signo);
			syslog(LOG_ERR, "%s%d\n", msg, signo);
			if (signo==11)
				exit(0);
		}
	}

	return;
}

/* The following function is a POSIX compliant replacement of the commonly used
 * signal(SIGCHLD, SIG_IGN)
 */
static void sigchild_handler(int signo)
{
#ifdef WNOHANG
	while (waitpid(-1, NULL, WNOHANG) > 0) {};
#endif /* WNOHANG */
	signal(SIGCHLD, sigchild_handler);
	return;
}

static int motion_detected (struct context *cnt, int diffs, int dev, int devpipe, int devmpipe, unsigned char *newimg)
{
	struct config *conf=&cnt->conf;
	struct images *imgs=&cnt->imgs;
	struct coord *location=&cnt->location;

	if (diffs) {
		if (cnt->locate==LOCATE_ON)
			alg_draw_location(location, imgs, imgs->width, imgs->height, newimg, LOCATE_BOTH);
		if (conf->switchfilter) {
			diffs=alg_switchfilter(cnt, diffs, location, newimg);
			cnt->diffs=diffs;
			if (diffs <= cnt->threshold )
				return 0;
		}
		event(EVENT_MOTION, cnt, NULL, NULL, cnt->currenttime);
	}


	cnt->lasttime=cnt->currenttimep;
	/* Take action if this is a new event */
	if (cnt->event_nr!=cnt->prev_event) {
		int i, tmpshots;
		struct tm tmptime;
		cnt->preview_max=0;
		
		event(EVENT_FIRSTMOTION, cnt, newimg, NULL, cnt->currenttime);
		cnt->prev_event=cnt->event_nr;
		cnt->eventtime=cnt->currenttimep;
		
		if (cnt->conf.setup_mode)
			printf("[%d] Motion detected - starting event %d\n", cnt->threadnr, cnt->event_nr);
		
		/* pre_capture frames are written as jpegs and to the ffmpeg film
		 * We store the current cnt->shots temporarily until we are done with
		 * the pre_capture stuff
		 */
		 
		tmpshots = cnt->shots;
		
		for (i=cnt->precap_cur; i < cnt->precap_nr; i++) {
			memcpy(&tmptime, localtime((cnt->imgs.timestamp + i)), sizeof(struct tm));
			cnt->shots = *(cnt->imgs.shotstamp + i);
			event(EVENT_IMAGE_DETECTED, cnt,
			    cnt->imgs.new + (cnt->imgs.size * i), NULL, &tmptime);
		}
		
		if (cnt->precap_cur) {
			memcpy(&tmptime, localtime((cnt->imgs.timestamp+cnt->precap_nr)), sizeof(struct tm));
			cnt->shots = *(cnt->imgs.shotstamp + cnt->precap_nr);
			event(EVENT_IMAGE_DETECTED, cnt,
			      cnt->imgs.new + (cnt->imgs.size * cnt->precap_nr),
			      NULL, &tmptime);
		}

		for (i=0; i < cnt->precap_cur-1; i++) {
			memcpy(&tmptime, localtime((cnt->imgs.timestamp+i)), sizeof(struct tm));
			cnt->shots = *(cnt->imgs.shotstamp + i);
			event(EVENT_IMAGE_DETECTED, cnt,
			      cnt->imgs.new + (cnt->imgs.size * i),
			      NULL, &tmptime);
		}
		/* If output_normal=first always capture first motion frame as preview-shot */
		if (cnt->new_img == NEWIMG_FIRST){
			cnt->preview_shot=1;
			if (cnt->locate==LOCATE_PREVIEW){
				alg_draw_location(location, imgs, imgs->width, imgs->height, newimg, LOCATE_NORMAL);
			}
		}
		cnt->shots = tmpshots;
	}
	
	/* Check for most significant preview-shot when output_normal=best */
	if (cnt->new_img==NEWIMG_BEST && diffs > cnt->preview_max) {
		memcpy(cnt->imgs.preview_buffer, newimg, cnt->imgs.size);
		cnt->preview_max=diffs;
		if (cnt->locate==LOCATE_PREVIEW){
			alg_draw_location(location, imgs, imgs->width, imgs->height, cnt->imgs.preview_buffer, LOCATE_NORMAL);
		}
	}
	
	if (cnt->shots < conf->frame_limit && cnt->currenttimep-cnt->lastshot >= conf->mingap )
	{
		cnt->lastshot=cnt->currenttimep;

		/* Output the latest picture 'image_new' or image_out for motion picture.
		 */
		event(EVENT_IMAGE_DETECTED, cnt, newimg, NULL, cnt->currenttime);
		cnt->preview_shot=0;
	}

	if (cnt->track.type != 0 && diffs != 0)
	{
		cnt->moved=track_move(cnt, dev, devpipe, devmpipe, &cnt->location, imgs, 0);
	}

	return diffs;
}


static void *motion_loop(void *arg)
{
	struct context *cnt=arg;
	int i, j, detecting_motion=0;
	struct tm curtime;
	time_t lastframe=0;
	FILE *picture;
	int postcap=0;
	int frame_buffer_size;
	int smartmask_ratio=0;
	int smartmask_count=20;
	int smartmask_lastrate=0;
	int olddiffs=0;
	
	int passflag=0;
	long int *rolling_average_data;
	long int rolling_average_limit, required_frame_time, frame_delay;
	int rolling_frame=0;
	struct timeval tv1, tv2;
	struct timespec delay_time, remaining_time;
	unsigned long int rolling_average, elapsedtime, splittime, timenow=0, timebefore=0;

#ifdef HAVE_FFMPEG
	/* Next two variables are used for timelapse feature
	 * time_last_frame is set to 1 so that first comming timelapse or second=0
	 * is acted upon.
	 */
	unsigned long int time_last_frame=1, time_current_frame;
#endif /* HAVE_FFMPEG */

	cnt->diffs=0;
	cnt->currenttime=&curtime;
	cnt->smartmask_speed=0;
	
	/* We initialize cnt->event_nr to 1 and cnt->prev_event to 0 (not really needed) so
	 * that certain code below does not run until motion has been detected the first time */
	cnt->event_nr=1;
	cnt->prev_event=0;
	
	printf("Thread %d started\n", cnt->threadnr);
	syslog(LOG_DEBUG, "Thread %d started", cnt->threadnr);

	if (!cnt->conf.filepath)
		cnt->conf.filepath=".";

	/* Work out expected frame rate based on config setting */
	if (cnt->conf.frame_limit)
		required_frame_time=1000000L/cnt->conf.frame_limit;
	else
		required_frame_time=0;
	frame_delay=required_frame_time;
	
	/* Reserve enough space for a 10 second timing history buffer */
	rolling_average_limit=10*cnt->conf.frame_limit;
	rolling_average_data=mymalloc(sizeof(long int)*rolling_average_limit);
	if (!rolling_average_data)
	{
		const char msg[]="Unable to allocate memory [rolling_average_data]";
		printf("[%d] %s\n", cnt->threadnr, msg);
		syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg);
		printf("Thread %d finishing...\n", cnt->threadnr);
		exit(1);
	}
	
	/* Preset history buffer with expected frame rate */
	for (j=0; j< rolling_average_limit; j++)
		rolling_average_data[j]=required_frame_time;

	/* set the device settings */
	cnt->video_dev=vid_start (cnt);
	if (cnt->video_dev==-1) {
		const char msg[]="Capture error ";
		printf("[%d] %s%s\n", cnt->threadnr, msg, strerror(errno));
		syslog(LOG_ERR, "[%d] %s%s", cnt->threadnr, msg, strerror(errno));
		printf("Thread %d finishing...\n", cnt->threadnr);
		exit(1);
	}
	cnt->imgs.new=mymalloc(cnt->imgs.size);
	memset(cnt->imgs.new, 0, cnt->imgs.size);       /* initialize to zero */
	cnt->imgs.ref=mymalloc(cnt->imgs.size);
	cnt->imgs.out=mymalloc(cnt->imgs.size);
	cnt->imgs.smartmask=mymalloc(cnt->imgs.motionsize);
	cnt->imgs.smartmask_final=mymalloc(cnt->imgs.motionsize);
	cnt->imgs.smartmask_buffer=mymalloc(cnt->imgs.motionsize*sizeof(int));
	cnt->imgs.labels=mymalloc(cnt->imgs.motionsize*sizeof(cnt->imgs.labels));
	cnt->imgs.labelsize=mymalloc((cnt->imgs.motionsize/2+1)*sizeof(cnt->imgs.labelsize));
	cnt->imgs.timestamp=mymalloc(sizeof(time_t));
	cnt->imgs.shotstamp=mymalloc(sizeof(int));

	/* Allocate a buffer for temp. usage in some places */
	/* Only despeckle for now... */
	cnt->imgs.common_buffer = malloc(3*cnt->imgs.width);

	/* Now is a good time to init rotation data. Since vid_start has been
 	 * called, we know that we have imgs.width and imgs.height. When capturing
	 * from a V4L device, these are copied from the corresponding conf values
	 * in vid_start. When capturing from a netcam, they get set in netcam_start, 
	 * which is called from vid_start.
	 *
	 * rotate_init will set cap_width and cap_height in cnt->rotate_data.
	 */
	rotate_init(cnt); /* rotate_deinit is called in main */

	/* Allow videodevice to settle in */
	sleep(1);

	/* Capture first image, or we will get an alarm on start */
	if (!(vid_next(cnt, cnt->imgs.new))) {
		const char msg[]="Capture error ";
		printf("[%d] %s%s\n", cnt->threadnr, msg, strerror(errno));
		syslog(LOG_ERR, "[%d] %s%s", cnt->threadnr, msg, strerror(errno));
		printf("Thread %d finishing...\n", cnt->threadnr);
		exit(1);
	}

	/* create a reference frame */
	memcpy(cnt->imgs.ref, cnt->imgs.new, cnt->imgs.size);

#ifndef WITHOUT_V4L
#ifndef __freebsd__
	/* open video loopback devices if enabled */
	if (cnt->conf.vidpipe) {
		if (cnt->conf.setup_mode)
			printf("[%d] Opening video loopback device for normal pictures\n", cnt->threadnr);
		/* vid_startpipe should get the output dimensions */
		cnt->pipe=vid_startpipe(cnt->conf.vidpipe, cnt->imgs.width, cnt->imgs.height, cnt->imgs.type, cnt);
		if (cnt->pipe < 0) {
			const char msg[]="Failed to open video loopback";
			printf("[%d] %s\n", cnt->threadnr, msg);
			syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg);
			printf("Thread %d finishing...\n", cnt->threadnr);
			exit(1);
		}
	}
	if (cnt->conf.motionvidpipe) {
		if (cnt->conf.setup_mode)
			printf("[%d] Opening video loopback device for motion pictures\n", cnt->threadnr);
		/* vid_startpipe should get the output dimensions */
		cnt->mpipe=vid_startpipe(cnt->conf.motionvidpipe, cnt->imgs.width, cnt->imgs.height, cnt->imgs.type, cnt);
		if (cnt->mpipe < 0) {
			const char msg[]="Failed to open video loopback";
			printf("[%d] %s\n", cnt->threadnr, msg);
			syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg);
			printf("Thread %d finishing...\n", cnt->threadnr);
			exit(1);
		}
	}
#endif /* __freebsd__ */ 
#endif /*WITHOUT_V4L*/

#ifdef HAVE_MYSQL
	if(cnt->conf.mysql_db) {
		cnt->database=(MYSQL *) mymalloc(sizeof(MYSQL));
		mysql_init(cnt->database);
		if (!mysql_real_connect(cnt->database, cnt->conf.mysql_host, cnt->conf.mysql_user,
		    cnt->conf.mysql_password, cnt->conf.mysql_db, 0, NULL, 0)) {
			const char msg1[]="Cannot connect to MySQL database ";
			const char msg2[]=" on host ";
			const char msg3[]=" with user ";
			printf("[%d] %s%s%s%s%s%s\n", cnt->threadnr, msg1, cnt->conf.mysql_db, msg2, cnt -> conf.mysql_host, msg3, cnt -> conf.mysql_user);
			printf("[%d] MySQL error was %s\n", cnt->threadnr, mysql_error(cnt->database));
			syslog(LOG_ERR, "[%d] %s%s%s%s%s%s", cnt->threadnr, msg1, cnt->conf.mysql_db, msg2, cnt -> conf.mysql_host, msg3, cnt -> conf.mysql_user);
			syslog(LOG_ERR, "[%d] MySQL error was %s\n", cnt->threadnr, mysql_error(cnt->database));
			printf("Thread %d finishing...\n", cnt->threadnr);
			exit(1);
		}
	}
#endif /* HAVE_MYSQL */
	
#ifdef HAVE_PGSQL
	if (cnt->conf.pgsql_db) {
		char connstring[255];
	
		/* create the connection string. 
		   Quote the values so we can have null values (blank)*/
		snprintf(connstring, 255,
		    "dbname='%s' host='%s' user='%s' password='%s' port='%d'",
		    cnt->conf.pgsql_db, /* dbname */
		    (cnt->conf.pgsql_host ? cnt->conf.pgsql_host : ""), /* host (may be blank) */
		    (cnt->conf.pgsql_user ? cnt->conf.pgsql_user : ""), /* user (may be blank) */
		    (cnt->conf.pgsql_password ? cnt->conf.pgsql_password : ""), /* password (may be blank) */
		    cnt->conf.pgsql_port
		);

		cnt->database_pg = PQconnectdb(connstring);
		if (PQstatus(cnt->database_pg) == CONNECTION_BAD) {
			const char msg1[]="Connection to PostgreSQL database ";
			const char msg2[]=" failed: ";
			printf("[%d] %s'%s'%s%s\n", cnt->threadnr, msg1, cnt->conf.pgsql_db, msg2, PQerrorMessage(cnt->database_pg));
			syslog(LOG_ERR, "[%d] %s'%s'%s%s", cnt->threadnr, msg1, cnt->conf.pgsql_db, msg2, PQerrorMessage(cnt->database_pg));
			printf("Thread %d finishing...\n", cnt->threadnr);
			exit(1);
		}
	}
#endif /* HAVE_PGSQL */

	/* Load the mask file if any */
	if (cnt->conf.mask_file) {
		if ((picture=fopen(cnt->conf.mask_file, "r"))) {
			/* NOTE: The mask is expected to have the output dimensions. I.e., the mask
			 * applies to the already rotated image, not the capture image. Thus, use
			 * width and height from imgs.
			 */
			cnt->imgs.mask=get_pgm(cnt, picture, cnt->imgs.width, cnt->imgs.height);
			fclose(picture);
		} else {
			const char msg[]="Error opening mask file ";
			printf("[%d] %s%s: %m\n", cnt->threadnr, msg, cnt->conf.mask_file);
			syslog(LOG_ERR, "[%d] %s%s: %m", cnt->threadnr, msg, cnt->conf.mask_file);
			/* Try to write an empty mask file to make it easier
			   for the user to edit it */
			put_fixed_mask(cnt, cnt->conf.mask_file);
		}
		if (!cnt->imgs.mask) {
			const char msg[]="Failed to read mask image. Mask feature disabled.";
			printf("[%d] %s\n", cnt->threadnr, msg);
			syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg);
		}
	} else
		cnt->imgs.mask=NULL;

	/* Always initialize smart_mask - someone could turn it on later... */
	memset(cnt->imgs.smartmask, 0, cnt->imgs.motionsize);
	memset(cnt->imgs.smartmask_final, 255, cnt->imgs.motionsize);
	memset(cnt->imgs.smartmask_buffer, 0, cnt->imgs.motionsize*sizeof(int));
	
	/* Set noise level */
	cnt->noise=cnt->conf.noise;

	/* Set threshold value */
	cnt->threshold=cnt->conf.max_changes;

	/* Set signal handlers */
	signal(SIGALRM, sig_handler);
	signal(SIGUSR1, sig_handler);
	signal(SIGHUP, sig_handler);
	signal(SIGTERM, sig_handler);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGINT, sig_handler);

	/* Initialize webcam server if webcam port is specified to not 0 */
	if (cnt->conf.webcam_port) {
		if ( webcam_init(cnt) == -1 ) {
			const char msg[]="Problem enabling stream server: ";
			printf("[%d] %s%s\n", cnt->threadnr, msg, strerror(errno));
			syslog(LOG_ERR, "[%d] %s%s", cnt->threadnr, msg, strerror(errno));
			cnt->finish=1;
			cnt->makemovie=0;
		}
	}

	/* Prevent first few frames from triggering motion... */
	cnt->moved=8;
	

	/* MAIN MOTION LOOP BEGINS HERE */
	/* Should go on forever... unless you bought vaporware :) */

	while (!cnt->finish || cnt->makemovie) {
		unsigned char *newimg=NULL;

		/* since we don't have sanity checks done when options are set,
		 * this sanity check must go in the main loop :(, before pre_captures
		 * are attempted. */
		if (cnt->conf.minimum_motion_frames < 1)
			cnt->conf.minimum_motion_frames = 1;
		if (cnt->conf.pre_capture < 0)
			cnt->conf.pre_capture = 0;

		/* Check if our buffer is still the right size */
		frame_buffer_size = cnt->conf.pre_capture + cnt->conf.minimum_motion_frames - 1;
		if (cnt->precap_nr != frame_buffer_size) {
			/* Only decrease if at last position in new buffer */
			if (frame_buffer_size > cnt->precap_nr || frame_buffer_size == cnt->precap_cur) {
				char *tmp;
				time_t *tmp2;
				int *tmp3;
				int smallest;
				smallest = (cnt->precap_nr < frame_buffer_size) ? cnt->precap_nr : frame_buffer_size;
				tmp=mymalloc(cnt->imgs.size*(1+frame_buffer_size));
				tmp2=mymalloc(sizeof(time_t)*(1+frame_buffer_size));
				tmp3=mymalloc(sizeof(int)*(1+frame_buffer_size));
				memcpy(tmp, cnt->imgs.new, cnt->imgs.size*(1+smallest));
				memcpy(tmp2, cnt->imgs.timestamp, sizeof(time_t)*(1+smallest));
				memcpy(tmp3, cnt->imgs.shotstamp, sizeof(int)*(1+smallest));
				free(cnt->imgs.new);
				free(cnt->imgs.timestamp);
				free(cnt->imgs.shotstamp);
				cnt->imgs.new=tmp;
				cnt->imgs.timestamp=tmp2;
				cnt->imgs.shotstamp=tmp3;
				cnt->precap_nr=frame_buffer_size;
			}
		}

		/* Get time for current frame */
		cnt->currenttimep=time(NULL);
		
		/* localtime returns static data and is not threadsafe
		 * so we copy the time to a safe place
		 */
		memcpy(cnt->currenttime, localtime(&cnt->currenttimep), sizeof(struct tm));
		
		/* If we have started on a new second we reset the shots variable
		 * lastrate is updated to be the number of the last frame. last rate
		 * is used as the ffmpeg framerate when motion is detected.
		 */
		if (lastframe!=cnt->currenttimep) {
			cnt->lastrate=cnt->shots + 1;
			cnt->shots=-1;
			lastframe=cnt->currenttimep;
		}

		/* Increase the shots variable for each frame captured within this second */
		cnt->shots++;

		/* Store time with pre_captured image*/
		*(cnt->imgs.timestamp+cnt->precap_cur)=cnt->currenttimep;

		/* Store shot number with pre_captured image*/
		*(cnt->imgs.shotstamp+cnt->precap_cur)=cnt->shots;

		/* newimg points to the current image and precap_cur incremented pointing to the
		 * position in the buffer for the NEXT image frame, not the current!!!
		 */
		newimg=cnt->imgs.new+(cnt->imgs.size*(cnt->precap_cur++));
		
		/* If we are at the end of the ring buffer go to the start */
		if (cnt->precap_cur > cnt->precap_nr)
			cnt->precap_cur=0;

		/* Fetch next frame from camera */
		if (!vid_next(cnt, newimg))
			break;
		
		/* Get current time and preserver last time for frame interval calc. */
		timebefore=timenow;
		gettimeofday(&tv1, NULL);
		timenow=tv1.tv_usec+1000000L*tv1.tv_sec;
		
		/* The actual motion detection takes place in the following
		 * diffs is the number of pixels detected as changed
		 * Make a differences picture in image_out
		 */
		if (cnt->threshold && !cnt->pause) {
			/* if we've already detected motion and we want to see if there's
			 * still motion, don't bother trying the fast one first. IF there's
			 * motion, the alg_diff will trigger alg_diff_standard
			 * anyway
			 */
			if (detecting_motion || cnt->conf.setup_mode)
				cnt->diffs=alg_diff_standard(cnt, newimg);
			else
				cnt->diffs=alg_diff(cnt, newimg);

			/* Despeckle feature is run now */
			cnt->imgs.total_labels=0;
			cnt->imgs.largest_label=0;
			olddiffs=0;
			if (cnt->conf.despeckle && cnt->diffs > 0) {
				olddiffs = cnt->diffs;
				cnt->diffs = alg_despeckle(cnt, olddiffs);
			}
		} else if (!cnt->conf.setup_mode)
			cnt->diffs=0;

		/* Manipulate smart_mask sensitivity (only every smartmask_ratio seconds) */
		if (cnt->smartmask_speed){
			if (!--smartmask_count){
				tune_smartmask(cnt);
				smartmask_count=smartmask_ratio;
			}
		}
		
		if (cnt->moved) {
			cnt->moved--;
			cnt->diffs=0;
		}
		
		/* Lightswitch feature - has lightintensity changed?
		 * This can happen due to change of light conditions or due to a sudden change of the camera
		 * sensitivity. If alg_lightswitch returns 
		 */
		if (cnt->conf.lightswitch) {
			if (alg_lightswitch(cnt, cnt->diffs)) {
				if (cnt->conf.setup_mode)
					printf("[%d] Lightswitch detected\n", cnt->threadnr);
				if (cnt->moved<5)
					cnt->moved = 5;
				cnt->diffs = 0;
			}
		}
		
		// Note: the make_movie and gap code has been moved further down the loop from here
		
		/* Old image slowly decays, this will make it even harder on
		 * a slow moving object to stay undetected
		 */
		for (i=cnt->imgs.size-1; i>=0; i--) {
			cnt->imgs.ref[i]=(cnt->imgs.ref[i]+newimg[i])/2;
		}
		
		/* Some overlays on top of the motion image */
		if (cnt->smartmask_speed && (cnt->conf.motion_img || cnt->conf.ffmpeg_cap_motion || cnt->conf.setup_mode))
			overlay_smartmask(cnt, cnt->imgs.out);

		if (cnt->imgs.mask && (cnt->conf.motion_img || cnt->conf.ffmpeg_cap_motion || cnt->conf.setup_mode))
			overlay_fixed_mask(cnt, cnt->imgs.out);

		if (cnt->imgs.largest_label && (cnt->conf.motion_img || cnt->conf.ffmpeg_cap_motion || cnt->conf.setup_mode))
			overlay_largest_label(cnt, cnt->imgs.out);
			
		/* If motion is detected (cnt->diffs > cnt->threshold) and before we add text to the pictures
		   we find the center and size coordinates of the motion to be used for text overlays and later
		   for adding the locate rectangel */		   
		if (cnt->diffs > cnt->threshold) {
			cnt->location=alg_locate_center_size(&cnt->imgs, cnt->imgs.width, cnt->imgs.height);
		}

		/* Add changed pixels in upper right corner of the pictures */
		if (cnt->conf.text_changes) {
			char tmp[15];
			sprintf(tmp, "%d", cnt->diffs);
			draw_text(newimg, cnt->imgs.width-10, 10, cnt->imgs.height, cnt->imgs.width, tmp);
		}

		/* Add changed pixels to motion-images (for webcam) in setup_mode
		   and always overlay smartmask (not only when motion is detected) */
		if (cnt->conf.setup_mode) {
			char tmp[PATH_MAX];
			sprintf(tmp, "D:%5d L:%3d N:%3d", cnt->diffs, cnt->imgs.total_labels, cnt->noise);
			draw_text(cnt->imgs.out, cnt->imgs.width-10, cnt->imgs.height-30, cnt->imgs.height, cnt->imgs.width, tmp);
			sprintf(tmp, "THREAD %d SETUP", cnt->threadnr);
			draw_text(cnt->imgs.out, cnt->imgs.width-10, cnt->imgs.height-10, cnt->imgs.height, cnt->imgs.width, tmp);
		}

		/* Add text in lower left corner of the pictures */
		if (cnt->conf.text_left) {
			char tmp[PATH_MAX];
			mystrftime(tmp, sizeof(tmp), cnt->conf.text_left, cnt->currenttime, cnt);
			draw_text(newimg, 10, cnt->imgs.height-10, cnt->imgs.height, cnt->imgs.width, tmp);
		}

		/* Add text in lower right corner of the pictures */
		if (cnt->conf.text_right) {
			char tmp[PATH_MAX];
			mystrftime(tmp, sizeof(tmp), cnt->conf.text_right, cnt->currenttime, cnt);
			draw_text(newimg, cnt->imgs.width-10, cnt->imgs.height-10, cnt->imgs.height, cnt->imgs.width, tmp);
		}

		/* Is output_all enabled?
		 * If so, take appropriate action by calling motion_detected() */
		if (cnt->conf.output_all) {
			detecting_motion=1;
			motion_detected(cnt, 0, cnt->video_dev, cnt->pipe, cnt->mpipe, newimg);
		} else if (cnt->diffs > cnt->threshold) {
			/* Did we detect motion (like the cat just walked in :) )?
			 * If so, ensure the motion is sustained if minimum_motion_frames
			 * is set, and take action by calling motion_detected().
			 * pre_capture is handled by motion_detected(), and we handle
			 * post_capture here. */
			if (!detecting_motion)
				detecting_motion=1;
			
			detecting_motion++;
			
			if (detecting_motion > cnt->conf.minimum_motion_frames) {
				if (motion_detected(cnt, cnt->diffs, cnt->video_dev, cnt->pipe, cnt->mpipe, newimg))
					postcap=cnt->conf.post_capture;
			}
		} else if (postcap) {
			motion_detected(cnt, 0, cnt->video_dev, cnt->pipe, cnt->mpipe, newimg);
			postcap--;
		} else {
			detecting_motion=0;
		}

		/* if noise tuning was selected, do it now. but only when
		 * no frames have been recorded
		 */
		if (cnt->conf.noise_tune && cnt->shots==0)
		{
			if (!detecting_motion)
			{
				alg_noise_tune(cnt, newimg);
			}
		}

		/* if we are not noise tuning lets make sure that remote controlled
		 * changes of noise_level are used.
		 */
		if (!cnt->conf.noise_tune)
			cnt->noise=cnt->conf.noise;

		/* threshold tuning if enabled 
		 * if we are not threshold tuning lets make sure that remote controlled
		 * changes of threshold are used.
		 */
		if (cnt->conf.threshold_tune)
			alg_threshold_tune(cnt, cnt->diffs, detecting_motion);
		else
			cnt->threshold=cnt->conf.max_changes;

		/* If setup_mode enabled output some numbers to console */
		if (cnt->conf.setup_mode){
			char msg[1024];
			char part[100];
			sprintf(msg, "[%d] ", cnt->threadnr);
			if (cnt->conf.despeckle){
				snprintf(part, 99, "Raw changes: %5d - changes after '%s': %5d", olddiffs, cnt->conf.despeckle, cnt->diffs);
				strcat(msg, part);
				if (strchr(cnt->conf.despeckle, 'l')){
					sprintf(part, " - labels: %3d", cnt->imgs.total_labels);
					strcat(msg, part);
				}
			}
			else{
				sprintf(part, "Changes: %5d", cnt->diffs);
				strcat(msg, part);
			}
			if (cnt->conf.noise_tune){
				sprintf(part, " - noise level: %2d", cnt->noise);
				strcat(msg, part);
			}
			if (cnt->conf.threshold_tune){
				sprintf(part, " - threshold: %d", cnt->threshold);
				strcat(msg, part);
			}
			printf("%s\n", msg);
		}
		/* Prevent the motion created by moving camera or sudden light intensity
		 * being detected by creating a fresh reference frame
		 */
		if ((cnt->track.type || cnt->conf.lightswitch) && cnt->moved)
			memcpy(cnt->imgs.ref, newimg, cnt->imgs.size);

		/* Is the mpeg movie to long? Then make movies
		 * First test for max mpegtime
		 */
		if (cnt->conf.maxmpegtime && cnt->event_nr==cnt->prev_event)
			if (cnt->currenttimep - cnt->eventtime >= cnt->conf.maxmpegtime)
				cnt->makemovie=1;

		/* Now test for quiet longer than 'gap' OR make movie as decided in
		 * previous statement.
		 */
		if (((cnt->currenttimep - cnt->lasttime >= cnt->conf.gap) && cnt->conf.gap > 0) || cnt->makemovie) {
			if (cnt->event_nr == cnt->prev_event || cnt->makemovie) {
				
				/* When output_normal=best save best preview_shot here at the end of event */
				if (cnt->new_img==NEWIMG_BEST)
					preview_best(cnt);
					
				event(EVENT_ENDMOTION, cnt, NULL, localtime(&cnt->eventtime), cnt->currenttime);
				
				/* if tracking is enabled we track a little */
				if (cnt->track.type) {
					//fprintf(stderr,"%d %d %d\n", cnt->prev_event, cnt->event_nr, cnt->makemovie);
					cnt->moved=track_center(cnt, cnt->video_dev, 0, 0, 0);
				}
				if (cnt->conf.setup_mode)
					printf("[%d] End of event %d\n", cnt->threadnr, cnt->event_nr);

				cnt->makemovie=0;
				cnt->event_nr++;
			}
		}

		/* Did we get triggered to make a snapshot from control http? Then snap
		 * If snapshot_interval is not zero and time since epoch MOD snapshot_interval = 0 then snap
		 * Note: Negative value means SIGALRM snaps is enabled
		 * httpd-control snaps are always enabled.
		 */
		if ( (cnt->conf.snapshot_interval > 0 && cnt->shots==0 &&
		      mktime(cnt->currenttime)%cnt->conf.snapshot_interval==0
		     ) || cnt->snapshot) {
			event(EVENT_IMAGE_SNAPSHOT, cnt, newimg, NULL, cnt->currenttime);
			cnt->snapshot=0;
		}

#ifdef HAVE_FFMPEG
		
		time_current_frame = mktime(cnt->currenttime);
 		
		if (cnt->conf.timelapse) {

			/* Check to see if we should start a new timelapse file. We start one when 
			 * we are on the first shot, and and the seconds are zero. We must use the seconds
			 * to prevent the timelapse file from getting reset multiple times during the minute.
			 */ 
			if (cnt->currenttime->tm_min == 0 && (time_current_frame % 60 < time_last_frame % 60) && cnt->shots == 0) {
				
				if (strcasecmp(cnt->conf.timelapse_mode,"manual") == 0)
				;/* No action */ 
				
				/* If we are daily, raise timelapseend event at midnight */ 
				else if (strcasecmp(cnt->conf.timelapse_mode,"daily") == 0) {
					if (cnt->currenttime->tm_hour == 0)
						event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}
				
				/* handle the hourly case */ 	
				else if (strcasecmp(cnt->conf.timelapse_mode,"hourly") == 0) {
					event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}
				
				/* If we are weekly-sunday, raise timelapseend event at midnight on sunday */ 
				else if (strcasecmp(cnt->conf.timelapse_mode,"weekly-sunday") == 0) {
					if (cnt->currenttime->tm_wday == 0 && cnt->currenttime->tm_hour == 0)
						event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}
				
				/* If we are weekly-monday, raise timelapseend event at midnight on monday */ 
				else if (strcasecmp(cnt->conf.timelapse_mode,"weekly-monday") == 0) {
					if (cnt->currenttime->tm_wday == 1 && cnt->currenttime->tm_hour == 0)
						event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}
				
				/* If we are monthly, raise timelapseend event at midnight on first day of month */
				else if (strcasecmp(cnt->conf.timelapse_mode,"monthly") == 0) {
					if (cnt->currenttime->tm_mday == 1 && cnt->currenttime->tm_hour == 0)
						event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}
				
				/* If invalid we report in syslog once and continue in manual mode */
				else {
					const char msg[]="Invalid timelapse_mode argument ";
					const char msg1[]="Defaulting to manual timelapse mode";
					syslog(LOG_ERR, "[%d] %s'%s'", cnt->threadnr, msg, cnt->conf.timelapse_mode);
					printf("[%d] %s'%s'\n", cnt->threadnr, msg, cnt->conf.timelapse_mode);
					syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg1);
					printf("[%d] %s\n", cnt->threadnr, msg1);
					conf_cmdparse(&cnt, "ffmpeg_timelapse_mode","manual");
				}
			}
			
			/* If ffmpeg timelapse is enabled and time since epoch MOD ffmpeg_timelaps = 0
			 * add a timelapse frame to the timelapse mpeg.
			 */
			if (cnt->shots == 0 &&
				time_current_frame % cnt->conf.timelapse <= time_last_frame % cnt->conf.timelapse)	
				event(EVENT_TIMELAPSE, cnt, newimg, NULL, cnt->currenttime);
		}
		
		/* if timelapse mpeg is in progress but conf.timelapse is zero then close timelapse file
		 * This is an important feature that allows manual roll-over of timelapse file using xmlrpc
		 * via a cron job
		 */
		else if (cnt->ffmpeg_timelapse)
			event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
			
		time_last_frame = time_current_frame;
		
#endif /* HAVE_FFMPEG */

		/* feed last image and motion image to video device pipes */
		if (cnt->conf.setup_mode)
			event(EVENT_IMAGE, cnt, cnt->imgs.out, &cnt->pipe, cnt->currenttime);
		else
			event(EVENT_IMAGE, cnt, newimg, &cnt->pipe, cnt->currenttime);
		event(EVENT_IMAGEM, cnt, cnt->imgs.out, &cnt->mpipe, cnt->currenttime);
		
		/* Check for some parameter changes but only every second */
		if (cnt->shots==0){
			if (strcasecmp(cnt->conf.output_normal,"on") == 0)
				cnt->new_img=NEWIMG_ON;
			else if (strcasecmp(cnt->conf.output_normal,"first") == 0)
				cnt->new_img=NEWIMG_FIRST;
			else if (strcasecmp(cnt->conf.output_normal,"best") == 0){
				cnt->new_img=NEWIMG_BEST;
				/* alocate buffer here when not yet done */
				if (!cnt->imgs.preview_buffer){
					cnt->imgs.preview_buffer=mymalloc(cnt->imgs.size);
					if (cnt->conf.setup_mode)
						printf("[%d] Preview buffer allocated\n", cnt->threadnr);
				}
			}
			else
				cnt->new_img=NEWIMG_OFF;

			if (strcasecmp(cnt->conf.locate,"on") == 0)
				cnt->locate=LOCATE_ON;
			else if (strcasecmp(cnt->conf.locate,"preview") == 0)
				cnt->locate=LOCATE_PREVIEW;
			else
				cnt->locate=LOCATE_OFF;

			/* Sanity check for smart_mask_speed, silly value disables smart mask */
			if (cnt->conf.smart_mask_speed < 0 || cnt->conf.smart_mask_speed > 10)
				cnt->conf.smart_mask_speed = 0;
			/* Has someone changed smart_mask_speed or framerate? */
			if (cnt->conf.smart_mask_speed != cnt->smartmask_speed || smartmask_lastrate != cnt->lastrate){
				if (cnt->conf.smart_mask_speed==0){
					memset(cnt->imgs.smartmask, 0, cnt->imgs.motionsize);
					memset(cnt->imgs.smartmask_final, 255, cnt->imgs.motionsize);
				}
				smartmask_lastrate=cnt->lastrate;
				cnt->smartmask_speed=cnt->conf.smart_mask_speed;
				/* Decay delay - based on smart_mask_speed (framerate independent)
				   This is always 5*smartmask_speed seconds */
				smartmask_ratio=5*cnt->lastrate*(11-cnt->smartmask_speed);
			}
			
#if defined(HAVE_MYSQL) || defined(HAVE_PGSQL)
			/* Set the sql mask file according to the SQL config options
			 * We update it for every frame in case the config was updated
			 * via remote control.
			 */
			cnt->sql_mask = cnt->conf.sql_log_image * (FTYPE_IMAGE + FTYPE_IMAGE_MOTION) +
			                cnt->conf.sql_log_snapshot * FTYPE_IMAGE_SNAPSHOT +
			                cnt->conf.sql_log_mpeg * (FTYPE_MPEG + FTYPE_MPEG_MOTION) +
			                cnt->conf.sql_log_timelapse * FTYPE_MPEG_TIMELAPSE;
#endif /* defined(HAVE_MYSQL) || defined(HAVE_PGSQL) */	
			
		}

		/* Work out expected frame rate based on config setting which may
		   have changed from http-control */
		if (cnt->conf.frame_limit)
			required_frame_time=1000000L/cnt->conf.frame_limit;
		else
			required_frame_time=0;

		/* Get latest time to calculate time taken to process video data */
		gettimeofday(&tv2, NULL);
		splittime=tv2.tv_usec+1000000L*tv2.tv_sec;
		elapsedtime=splittime-timenow;

		/* Update history buffer but ignore first pass as timebefore
		   variable will be inaccurate
		 */
		if (passflag)
			rolling_average_data[rolling_frame]=timenow-timebefore;
		rolling_frame++;
		passflag=1;
		if (rolling_frame>=rolling_average_limit)
			rolling_frame=0;

		/* Calculate 10 second average and use deviation in delay calculation */
		rolling_average=0L;
		for (j=0; j<rolling_average_limit; j++)
			rolling_average+=rolling_average_data[j];
		rolling_average/=rolling_average_limit;
		frame_delay=required_frame_time-elapsedtime-(rolling_average-required_frame_time);

		if (frame_delay>0) {
			/* Apply delay to meet frame time */
			if (frame_delay>required_frame_time)
				frame_delay=required_frame_time;

			/* sleep using nanosleep. If a signal such as SIG_CHLD interrupts the
			   sleep we just continue sleeping */
			delay_time.tv_sec = 0;
			delay_time.tv_nsec = frame_delay * 1000;

			if (delay_time.tv_nsec > 999999999)
				delay_time.tv_nsec = 999999999;
			
			while ( nanosleep(&delay_time,&remaining_time) == -1 )
			{
				delay_time.tv_sec  = remaining_time.tv_sec;
				delay_time.tv_nsec = remaining_time.tv_nsec;
			}
		}

		/* This will limit the framerate to 1 frame while not detecting
		   motion. Using a different motion flag to allow for multiple frames per second
		 */

		if (cnt->conf.low_cpu && !detecting_motion) {
			/* Recalculate remaining time to delay for a total of 1/low_cpu seconds */
			if (frame_delay+elapsedtime<(1000000L/cnt->conf.low_cpu)) {
				frame_delay=(1000000L/cnt->conf.low_cpu)-frame_delay-elapsedtime;

				/* sleep using nanosleep. If a signal such as SIG_CHLD interrupts the
				   sleep we just continue sleeping */			
				delay_time.tv_sec = 0;
				delay_time.tv_nsec = frame_delay * 1000;
				
				if (delay_time.tv_nsec > 999999999)
					delay_time.tv_nsec = 999999999;
	
				while ( nanosleep(&delay_time,&remaining_time) == -1 )
				{
					delay_time.tv_sec  = remaining_time.tv_sec;
					delay_time.tv_nsec = remaining_time.tv_nsec;
				}
				/* Correct frame times to ensure required_frame_time is maintained */
				gettimeofday(&tv1, NULL);
				timenow=tv1.tv_usec+1000000L*tv1.tv_sec-required_frame_time;
			}
		}
	}

	if (rolling_average_data)
		free(rolling_average_data);	

	printf("Thread %d exiting\n",cnt->threadnr);
	if (!cnt->finish)
		syslog(LOG_ERR, "Somebody stole the video device, lets hope we got his picture");
	event(EVENT_STOP, cnt, NULL, NULL, NULL);
	threads_running--;
	return NULL;
}

int main (int argc, char **argv)
{
	int i, j;
	int webcam_port;
	pthread_attr_t thread_attr;
	pthread_t thread_id;

	/* cnt_list is an array of pointers to the context structures cnt for each thread.
	 * First we reserve room for a pointer to thread 0's context structure
	 * and a NULL pointer which indicates that end of the array of pointers to
	 * thread context structures.
	 */
	cnt_list=mymalloc(sizeof(struct context *)*2);
	
	/* Now we reserve room for thread 0's context structure and let cnt_list[0] point to it */
	cnt_list[0]=mymalloc(sizeof(struct context));
	
	/* Populate context structure with start/default values */
	context_init(cnt_list[0]);
	
	/* cnt_list[1] pointing to zero indicates no more thread context structures - they get added later */
	cnt_list[1]=NULL;

	/* Enable automatic zombie reaping */
	signal(SIGCHLD, sigchild_handler);

	/* Command line arguments are being pointed to from cnt_list[0] and we call conf_load which loads
	 * the config options from motion.conf, thread config files and the command line.
	 */
	cnt_list[0]->conf.argv=argv;
	cnt_list[0]->conf.argc=argc;
	cnt_list=conf_load(cnt_list);

#ifdef HAVE_FFMPEG
	ffmpeg_init();
#endif /* HAVE_FFMPEG */

	if (cnt_list[0]->daemon && cnt_list[0]->conf.setup_mode==0) {
		
		const char msg[]="Motion running as daemon process";
		
		if (fork()) {
			printf("Motion going to daemon mode\n");
			exit(0);
		}
		
		signal(SIGQUIT, sig_handler);
		
		chdir("/");

#ifdef __freebsd__
		setpgrp(0,getpid());
#else
		setpgrp();
#endif /* __freebsd__ */

		if ((i=open("/dev/tty", O_RDWR)) >= 0) {
			ioctl(i, TIOCNOTTY, NULL);
			close(i);
		}
		
		setsid();
		i = open("/dev/null", O_RDONLY);
		
		if(i != -1) {
			dup2(i, STDIN_FILENO);
			close(i);
		}
		
		i = open("/dev/null", O_WRONLY);
		
		if(i != -1) {
			dup2(i, STDOUT_FILENO);
			dup2(i, STDERR_FILENO);
			close(i);
		}
		
		signal(SIGTTOU, SIG_IGN);
		signal(SIGTTIN, SIG_IGN);
		signal(SIGTSTP, SIG_IGN);

		printf("%s\n", msg);
		syslog(LOG_INFO, "%s", msg);
		
		if (cnt_list[0]->conf.low_cpu){
			printf("Capturing %d frames/s when idle", cnt_list[0]->conf.low_cpu);
			syslog(LOG_INFO, "Capturing %d frames/s when idle", cnt_list[0]->conf.low_cpu);
		}
	}
	if(cnt_list[0]->conf.setup_mode)
		printf("Motion running in setup mode.\n");
		
	pthread_attr_init(&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
#ifndef WITHOUT_V4L
	vid_init();
#endif
	do {
		if (restart) {
			i=-1;
			while (cnt_list[++i]){
				if (cnt_list[i]->imgs.out)
					free(cnt_list[i]->imgs.out);
				if (cnt_list[i]->imgs.ref)
					free(cnt_list[i]->imgs.ref);
				if (cnt_list[i]->imgs.new)
					free(cnt_list[i]->imgs.new);
				if (cnt_list[i]->imgs.labels)
					free(cnt_list[i]->imgs.labels);
				if (cnt_list[i]->imgs.labelsize)
					free(cnt_list[i]->imgs.labelsize);
				if (cnt_list[i]->imgs.smartmask)
					free(cnt_list[i]->imgs.smartmask);
				if (cnt_list[i]->imgs.smartmask_final)
					free(cnt_list[i]->imgs.smartmask_final);
				if (cnt_list[i]->imgs.smartmask_buffer)
					free(cnt_list[i]->imgs.smartmask_buffer);
				if (cnt_list[i]->imgs.common_buffer)
					free(cnt_list[i]->imgs.common_buffer);
				if (cnt_list[i]->imgs.timestamp)
					free(cnt_list[i]->imgs.timestamp);
				if (cnt_list[i]->imgs.shotstamp)
					free(cnt_list[i]->imgs.shotstamp);
				if (cnt_list[i]->imgs.preview_buffer)
					free(cnt_list[i]->imgs.preview_buffer);
				rotate_deinit(cnt_list[i]); /* cleanup image rotation data */
				if (cnt_list[i])
					free(cnt_list[i]);
			}
			free(cnt_list);
#ifndef WITHOUT_V4L
			vid_close();
			vid_cleanup();
#endif	
			cnt_list=mymalloc(sizeof(struct context *)*2);
			cnt_list[0]=mymalloc(sizeof(struct context));
			context_init(cnt_list[0]);
			cnt_list[1]=NULL;

			signal(SIGCHLD, sigchild_handler);

			cnt_list[0]->conf.argv=argv;
			cnt_list[0]->conf.argc=argc;
			cnt_list=conf_load(cnt_list);
			restart=0;
#ifndef WITHOUT_V4L
			sleep(5); // maybe some cameras needs less time
			vid_init();
#endif
		}

		
		/* Check the webcam port number for conflicts.
		 * First we check for conflict with the control port.
		 * Second we check for that two threads does not use the same port number
		 * for the webcam. If a duplicate port is found the webcam feature gets disabled (port =0)
		 * for this thread and a warning is written to console and syslog.
		 */
		i = 0;
		while (cnt_list[++i]) {
			webcam_port = cnt_list[i]->conf.webcam_port;
			
			if (cnt_list[0]->conf.setup_mode)
				printf("Webcam port for thread %d is %d\n", i, webcam_port);
				
			if (cnt_list[0]->conf.control_port == webcam_port && webcam_port != 0) {
				cnt_list[i]->conf.webcam_port = 0;
				printf("Webcam port number %d for thread %d conflicts with the control port\n",
				       webcam_port, i);
				printf("Webcam feature for thread %d is disabled.\n", i);
				syslog(LOG_ERR, "Webcam port number %d for thread %d conflicts with the control port\n",
				       webcam_port, i);
				syslog(LOG_ERR, "Webcam feature for thread %d is disabled.", i);
			}
			
			j = i;
			while (cnt_list[++j]) {
				if (cnt_list[j]->conf.webcam_port == webcam_port && webcam_port != 0) {
					cnt_list[j]->conf.webcam_port = 0;
					printf("Webcam port number %d for thread %d conflicts with thread %d\n",
					        webcam_port, j, i);
					printf("Webcam feature for thread %d is disabled.\n", j);
					syslog(LOG_ERR, "Webcam port number %d for thread %d conflicts with thread %d",
					       webcam_port, j, i);
					syslog(LOG_ERR, "Webcam feature for thread %d is disabled.", j);	
				}
			}
		}

		i=-1;
		while (cnt_list[++i]) {
			/* first cnt_list is global if 'thread' option is used */
			if (!i && cnt_list[i+1])
				continue;
			if (cnt_list[0]->conf.setup_mode)
				printf("Thread %d device: %s input: %d\n",
				    cnt_list[i]->threadnr,
				    cnt_list[i]->conf.netcam_url ? cnt_list[i]->conf.netcam_url : cnt_list[i]->conf.video_device,
				    cnt_list[i]->conf.netcam_url ? -1 : cnt_list[i]->conf.input
			);
			threads_running++;
			pthread_create(&thread_id, &thread_attr, &motion_loop, cnt_list[i]);
		}
		
		if (cnt_list[0]->conf.control_port)
			pthread_create(&thread_id, &thread_attr, &motion_web_control, cnt_list);

		if (cnt_list[0]->conf.setup_mode)
			printf("Waiting for threads to finish, pid: %d\n", getpid());
			
		while(threads_running > 0) {
			sleep(1);
		}
		
		if (cnt_list[0]->conf.setup_mode)
			syslog(LOG_DEBUG, "Threads finished\n");
		if (restart)
			sleep(2);

	} while (restart);

	printf("Motion terminating\n");
	syslog(LOG_INFO, "Motion terminating");


	/* Cleanup some memory before we exit */

	i=-1;
	while (cnt_list[++i]){
		if (cnt_list[i]->imgs.out)
			free(cnt_list[i]->imgs.out);
		if (cnt_list[i]->imgs.ref)
			free(cnt_list[i]->imgs.ref);
		if (cnt_list[i]->imgs.new)
			free(cnt_list[i]->imgs.new);
		if (cnt_list[i]->imgs.labels)
			free(cnt_list[i]->imgs.labels);
		if (cnt_list[i]->imgs.labelsize)
			free(cnt_list[i]->imgs.labelsize);
		if (cnt_list[i]->imgs.smartmask)
			free(cnt_list[i]->imgs.smartmask);
		if (cnt_list[i]->imgs.smartmask_final)
			free(cnt_list[i]->imgs.smartmask_final);
		if (cnt_list[i]->imgs.smartmask_buffer)
			free(cnt_list[i]->imgs.smartmask_buffer);
		if (cnt_list[i]->imgs.common_buffer)
			free(cnt_list[i]->imgs.common_buffer);
		if (cnt_list[i]->imgs.timestamp)
			free(cnt_list[i]->imgs.timestamp);
		if (cnt_list[i]->imgs.shotstamp)
			free(cnt_list[i]->imgs.shotstamp);
		if (cnt_list[i]->imgs.preview_buffer)
			free(cnt_list[i]->imgs.preview_buffer); 
		for (j=0; config_params[j].param_name != NULL; j++) {
			if (config_params[j].copy == copy_string) {
				void **val;
				val=(void *)cnt_list[i]+config_params[j].conf_value;
				if (*val) {
					free(*val);
					*val = NULL;
				}
			}
		}

		rotate_deinit(cnt_list[i]); /* cleanup image rotation data */
		if (cnt_list[i])
			free(cnt_list[i]);
	}
	free(cnt_list);
#ifndef WITHOUT_V4L
	/* close devices ? */
	vid_close();
	/* That should clean viddev structs */
	vid_cleanup();
#endif /* WITHOUT_V4L */
	return 0;
}

/* NOTE: Kenneth Lavrsen change printing of size_t types so instead of using conversion
 * specifier %zd I changed it to %llu and casted the size_t variable to unsigned long long
 * The reason for this nonsense is that older versions of gcc like 2.95 uses %Zd and does
 * not understand %zd. So to avoid this mess I used a more generic way. Long Long should
 * have enough bits for 64 bit machines with large memory areas
 */

/* allocate some memory and check if that succeeded or not. if it failed
 * do some errorlogging and bail out
 */
void * mymalloc(size_t nbytes)
{
	void *dummy = malloc(nbytes);
	if (!dummy) {
		printf("Could not allocate %llu bytes of memory!\n", (unsigned long long) nbytes);
		syslog(LOG_EMERG, "Could not allocate %llu bytes of memory!", (unsigned long long) nbytes);
		exit(1);
	}

	return dummy;
}

/* reallocate some memory and check if that succeeded or not. if it failed
 * do some errorlogging and bail out
 */
void * myrealloc(void *ptr, size_t size, char *desc)
{
	void *dummy = NULL;

	if (size == 0)
	{
		free(ptr);
		printf("Warning! Function %s tries to resize memoryblock at %p to %llu bytes!\n", desc, ptr, (unsigned long long)size);
		syslog(LOG_WARNING, "Warning! Function %s tries to resize memoryblock at %p to %llu bytes!", desc, ptr, (unsigned long long)size);
	}
	else
	{
		dummy = realloc(ptr, size);
		if (!dummy) {
			printf("Could not resize memory-block at offset %p to %llu bytes (function %s)!\n", ptr, (unsigned long long)size, desc);
			syslog(LOG_EMERG, "Could not resize memory-block at offset %p to %llu bytes (function %s)!", ptr, (unsigned long long)size, desc);
			exit(1);
		}
	}

	return dummy;
}

/* this function creates a whole path:
 * this/is/an/example/
 * /this/is/an/example/
 * Warning: a path *must* end with a slash!
 * etc.
 */
int create_path(char *path)
{
	char *start;
	mode_t mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;

	if (path[0] == '/')
		start = strchr(path + 1, '/');
	else
		start = strchr(path, '/');

	while(start)
	{
		char *buffer = strdup(path);
		buffer[start-path] = 0x00;

		if (mkdir(buffer, mode) == -1 && errno != EEXIST)
		{
			const char msg[]="Problem creating directory ";
			printf("%s%s\n", msg, buffer);
			syslog(LOG_ERR, "%s%s", msg, buffer);
			free(buffer);
			return -1;
		}

		free(buffer);

		start = strchr(start + 1, '/');
	}

	return 0;
}

/* this function opens a file, if that failed because of an ENOENT error (which is: path
 * does not exist), the path is created and then things are tried again.
 * this is faster then trying to create that path over and over again. if someone
 * removes the path after it was created, myfopen will recreate the path automatically
 */
FILE * myfopen(char *path, char *mode)
{
	/* first, just try to open the file */
	FILE *dummy = fopen(path, mode);

	/* could not open file... */
	if (!dummy)
	{
		/* path did not exist? */
		if (errno == ENOENT)
		{
			//DEBUG CODE  syslog(LOG_DEBUG, "Could not open file %s directly; path did not exist. Creating path & retrying.", path);

			/* create path for file... */
			if (create_path(path) == -1)
			{
				return NULL;
			}

			/* and retry opening the file */
			dummy = fopen(path, mode);
			if (dummy)
				return dummy;
		}

		/* two possibilities
		 * 1: there was an other error while trying to open the file for the first time
		 * 2: could still not open the file after the path was created
		 */
		printf("Error opening file %s with mode %s: %s\n", path, mode, strerror(errno));
		syslog(LOG_ERR, "Error opening file %s with mode %s: %s", path, mode, strerror(errno));

		return NULL;
	}

	return dummy;
}

/* 
   strftime(3) but with %v for eVent and %q for shots
*/
size_t mystrftime(char *s, size_t max, char *userformat,
                  const struct tm *tm, struct context *cnt)
{
	char formatstring[PATH_MAX]="", tempstring[20]="";
	char *pos, *format, *tempstr;
	
	format = formatstring;
	tempstr = tempstring;
	
	pos = userformat;

	for ( ; *pos; ++pos) {

		if (*pos == '%') {
		
			switch (*++pos) {
				case '\0': // end of string
					--pos;
					break;
					
				case 'v': // event
					sprintf(tempstr, "%02d", cnt->event_nr);
					while ((*format = *tempstr++) != '\0')
						++format;
					continue;
					
				case 'q': // shots
					sprintf(tempstr, "%02d", cnt->shots);
					while ((*format = *tempstr++) != '\0')
						++format;
					continue;
				
				case 'D': // diffs
					sprintf(tempstr, "%d", cnt->diffs);
					while ((*format = *tempstr++) != '\0')
						++format;
					continue;
				
				case 'N': // noise
					sprintf(tempstr, "%d", cnt->noise);
					while ((*format = *tempstr++) != '\0')
						++format;
					continue;
				
				case 'i': // motion width
					sprintf(tempstr, "%d", cnt->location.width);
					while ((*format = *tempstr++) != '\0')
						++format;
					continue;
				
				case 'J': // motion height
					sprintf(tempstr, "%d", cnt->location.height);
					while ((*format = *tempstr++) != '\0')
						++format;
					continue;
				
				case 'K': // motion center x
					sprintf(tempstr, "%d", cnt->location.x);
					while ((*format = *tempstr++) != '\0')
						++format;
					continue;
				
				case 'L': // motion center y
					sprintf(tempstr, "%d", cnt->location.y);
					while ((*format = *tempstr++) != '\0')
						++format;
					continue;
					
				
				default: // Any other code is copied with the %-sign
					*format++ = '%';
					*format++ = *pos;
					continue;
			}
		}
		
		/* For any other character than % we just simply copy the character */
		*format++ = *pos;
	}
	
	*format = '\0';
	format = formatstring;
	
	return strftime(s, max, format, tm);
}
