/*
 *	netcam.c
 *
 *	Copyright 2002 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	Additional copyright 2005 Angel Carpintero and Christopher Price
 *	(only main contributors of this file listed).
 *	This software is distributed under the GNU Public License Version 2
 *	See also the file 'COPYING'.
 *
 */

#include "motion.h"

#ifdef __freebsd__
#include "video_freebsd.h"
#else
#include "video.h"
#endif /* __freebsd__ */

#include <netdb.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <regex.h>
#include <jpeglib.h>
#include <setjmp.h>

/* for rotation */
#include "rotate.h"
#include "netcam_wget.h"

/* define NETCAM_DEBUG to enable LOTS of console debugging information
 * define NETCAM_NO_CONTENTLENGTH_DEBUG to test non-contentlength support
 * on netcam that cannot turn off this header
#define NETCAM_DEBUG
#define NETCAM_NO_CONTENTLENGTH_DEBUG
*/

#define REQUEST_AUTH	"GET %s HTTP/1.0\r\n" \
			"User-Agent: Motion-netcam/"VERSION"\r\n" \
			"Connection: close\r\n" \
			"Authorization: Basic %s\r\n\r\n"
#define REQUEST		"GET %s HTTP/1.0\r\n" \
			"User-Agent: Motion-netcam/"VERSION"\r\n" \
			"Connection: close\r\n\r\n"

#define INIT_BUFFER 4096		/* initial buffer sizes, grows dynamically */
#define CONNECT_TIMEOUT 5		/* timeout in seconds for netcam connect */
#define RECONNECT_TIMEOUT 60		/* timeout in seconds in between netcam reconnects */
#define RECONNECT_RETRIES 60		/* maximum retries for a netcam reconnect */
#define COND_WHICH_TIMEOUT 1		/* timeout in seconds for netcam which condition */ 
#define COND_DIRTY_TIMEOUT 1		/* timeout in seconds for netcam dirty buffer condition */ 
#define MAX_HEADER_RETRIES 20

/* netcam initialization/run states */
#define STATE_INIT1 0			/* first header */
#define STATE_INIT2 1			/* first image header */
#define STATE_RUN 2			/* running/processing images */

/* structures for overridden jpeglib functions */
typedef struct {
	struct jpeg_source_mgr pub;
	char *data;
	int length;
	JOCTET *buffer;
	boolean start_of_file;
} my_source_mgr;
typedef my_source_mgr *my_src_ptr;

struct my_error_mgr {
	struct jpeg_error_mgr pub;
	jmp_buf setjmp_buffer;
};
 
void my_init_source(j_decompress_ptr); 
boolean my_fill_input_buffer(j_decompress_ptr); 
void my_skip_input_data(j_decompress_ptr, long);
void my_term_source(j_decompress_ptr);
void jpeg_memory_src(j_decompress_ptr, char *, int);
void my_error_exit(j_common_ptr);

char *netcam_url_match(regmatch_t, const char *);
int netcam_parse_url(struct url_t *, char *);
void netcam_url_free(struct url_t *);

void netcam_cleanup(struct context *);
static void *netcam_loop(void *);
int netcam_read_header(struct context *);
int netcam_connect(struct context *);
int netcam_reconnect(struct context *);
void netcam_disconnect(struct context *);
int netcam_read_image_no_contentlength(struct context *);
int netcam_read_image_contentlength(struct context *);
int netcam_read_image(struct context *);
int netcam_single_read(struct context *);
int netcam_stream_read(struct context *);

int netcam_start(struct context *);
unsigned char *netcam_next(struct context *, char *);

/*
 * finds the matched part of the regular expression 
 * 
 * CALLED BY: netcam_url_parse
 *
 * RETURNS: string matched
 */
char *netcam_url_match(regmatch_t m, const char *input) 
{
	char *match = NULL;
	int len;
    
	if (m.rm_so != -1) {
		len = m.rm_eo - m.rm_so; 
	   
		if ((match = (char *) malloc(len + 1)) != NULL) {
			strncpy(match, input + m.rm_so, len);
			match[len] = '\0';
		}
	}

	return(match);
}

/*
 * parses the url
 * 
 * CALLED BY: netcam_start
 *
 * RETURNS: 1 on success
 * 	    parsed url in parse_url argument
 */
int netcam_url_parse (struct url_t *parse_url, char *text_url)
{
	char *s;
	int i, ret = -1;
	const char *re = "(http)://(((.*):(.*))@)?([^/:]|[-.a-z0-9]+)(:([0-9]+))?(/[^:]*)";
	regex_t pattbuf;		
	regmatch_t matches[10];   

	parse_url->service = NULL;
	parse_url->userpass = NULL;
	parse_url->host = NULL;
	parse_url->port = 80;
	parse_url->path = NULL;

	if (!regcomp(&pattbuf, re, REG_EXTENDED | REG_ICASE)) {
		if (regexec(&pattbuf, text_url, 10, matches, 0) != REG_NOMATCH) {
			ret = 0;
			for (i = 0; i < 10; i++) {
				if ((s = netcam_url_match(matches[i], text_url)) != NULL) {
					switch (i) {
						case 1:
							parse_url->service = s;
							break;
						case 3:
							parse_url->userpass = s;
							break;
						case 6:
							parse_url->host = s; 
							break;
						case 8:
							parse_url->port = atoi(s);
							free(s);
							break;
						case 9:
							parse_url->path = s;
							break;
					}
				}
			}
		}
	}

	regfree(&pattbuf);

	return ret;
}

/*
 * frees data allocated for url structure
 * 
 * CALLED BY: netcam_cleanup 
 */
void netcam_url_free(struct url_t *parse_url)
{
	if (parse_url->service)
		free(parse_url->service);
	if (parse_url->userpass)
		free(parse_url->userpass);
	if (parse_url->host)
		free(parse_url->host);
	if (parse_url->path)
		free(parse_url->path);
}

/* 
 * jpeglib init_source overridden by our own init_source routine.
 * Initialize source before any data is actually read.
 *
 * CALLED BY: jpeg_read_header
 */
void my_init_source(j_decompress_ptr cinfo) 
{
	my_src_ptr src = (my_src_ptr) cinfo->src;
	/* We reset the empty-input-file flag for each image,
	but we don't clear the input buffer.
	This is correct behavior for reading a series of images from one source.
	*/
	src->start_of_file = TRUE;
}

/*
 * jpeglib fill_input_buffer overridden by our own fill_input_buffer routine.
 *
 * Fill the input buffer
 *
 * CALLED BY: internal to jpeglib
 */
boolean my_fill_input_buffer(j_decompress_ptr cinfo)
{
	my_src_ptr src = (my_src_ptr) cinfo->src;
	size_t nbytes;
	
	if (src->start_of_file) {
		nbytes=src->length;
		src->buffer=src->data;
	} else {
	// Insert a fake EOI marker - as per jpeglib recommendation
		src->buffer[0] = (JOCTET) 0xFF;
		src->buffer[1] = (JOCTET) JPEG_EOI; // 0xD9
		nbytes = 2;
	}
	
	src->pub.next_input_byte = src->buffer;
	src->pub.bytes_in_buffer = nbytes;
	src->start_of_file = FALSE;
	
	return TRUE;
}

/*
 * jpeglib skip_input_data overridden by our own skip_input_data routine.
 *
 * skips over potentially large amounts of uninteresting data
 * 
 * CALLED BY: internal to jpeglib
 */
void my_skip_input_data(j_decompress_ptr cinfo, long num_bytes)
{
	my_src_ptr src = (my_src_ptr) cinfo->src;

	if (num_bytes > 0) {
		while (num_bytes > (long) src->pub.bytes_in_buffer) {
			num_bytes -= (long) src->pub.bytes_in_buffer;
			(void) my_fill_input_buffer(cinfo);
		}
	src->pub.next_input_byte += (size_t) num_bytes;
	src->pub.bytes_in_buffer -= (size_t) num_bytes;
	}
}

/*
 * jpeglib term_source overridden by our own term_source routine.
 *
 * terminate source after all data has been read. Often a no-op.
 *
 * NOTE: not called by jpeg_abort or jpeg_destroy; surrounding
 * application must deal with any cleanup that should happen even
 * for abnormal exit condition.
 *
 * CALLED BY: internal to jpeglib
 */
void my_term_source(j_decompress_ptr cinfo) { }

/*
 * jpeglib jpeg_memory_src
 *
 * Prepare for input from memory.
 *
 * NOTE: caller must have already allocated memory, and is responsible
 * for freeing it after finishing decompression
 *
 * CALLED BY: internal to jpeglib
 */
void jpeg_memory_src(j_decompress_ptr cinfo, char *data, int length)
{
	my_src_ptr src;

	if (cinfo->src == NULL) {
		cinfo->src = (struct jpeg_source_mgr *) (*cinfo->mem->alloc_small)
			((j_common_ptr) cinfo, JPOOL_PERMANENT,sizeof(my_source_mgr));
		src = (my_src_ptr) cinfo->src;
	}

	src = (my_src_ptr) cinfo->src;
	src->data = data;
	src->length = length;
	src->pub.init_source = my_init_source;
	src->pub.fill_input_buffer = my_fill_input_buffer;
	src->pub.skip_input_data = my_skip_input_data;
	src->pub.resync_to_restart = jpeg_resync_to_restart;
	src->pub.term_source = my_term_source;
	src->pub.bytes_in_buffer = 0;
	src->pub.next_input_byte = NULL;
}

/*
 * jpeglib error_exit overridden by our own error_exit routine.
 *
 * triggerd when an abnormal error condition occurs
 * 
 * CALLED BY: internal to jpeglib
 */
void my_error_exit(j_common_ptr cinfo)
{
	struct my_error_mgr *myerr = (struct my_error_mgr *) cinfo->err;
	(*cinfo->err->output_message) (cinfo);
	/* Return control to the setjmp point */
	longjmp(myerr->setjmp_buffer, 1);
}

/*
 * destroys memory/pthread data dynamically created by the netcam
 * 
 * CALLED BY: netcam_start, netcam_loop
 */
void netcam_cleanup(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_cleanup()\n", cnt->threadnr);
#endif
	
	if (cnt->conf.setup_mode) {
		const char msg[] = "netcam: freeing data...";
		syslog(LOG_INFO, "[%d] %s", cnt->threadnr, msg);
		printf("[%d] %s\n", cnt->threadnr, msg);
	}

	pthread_mutex_lock(&netcam->mutex);
	pthread_mutex_lock(&netcam->image[0].mutex);
	pthread_mutex_lock(&netcam->image[1].mutex);
	
	pthread_cond_signal(&netcam->dirty_cond);

	netcam_disconnect(cnt);

	if (netcam->image[0].buffer.ptr)
		free(netcam->image[0].buffer.ptr);
	
	if (netcam->image[1].buffer.ptr)
		free(netcam->image[1].buffer.ptr);
	
	if ((cnt->conf.netcam_userpass) && (netcam->userpass))  { 
		free(netcam->userpass);
		netcam->userpass=NULL;
		cnt->conf.netcam_userpass=NULL;
	}	

	if (netcam->boundary)
		free(netcam->boundary);
	
	if (netcam->response)
		free(netcam->response);
	
	netcam_url_free(&netcam->url);

	pthread_mutex_unlock(&netcam->image[0].mutex);
	pthread_mutex_unlock(&netcam->image[1].mutex);
	pthread_mutex_unlock(&netcam->mutex);

	pthread_attr_destroy(&netcam->thread_attr);
	pthread_cond_destroy(&netcam->which_cond);
	pthread_cond_destroy(&netcam->dirty_cond);
	pthread_mutex_destroy(&netcam->image[0].mutex);
	pthread_mutex_destroy(&netcam->image[1].mutex);
	pthread_mutex_destroy(&netcam->mutex);

	if (cnt->netcam) free(cnt->netcam);
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_cleanup()\n", cnt->threadnr);
#endif
}

/*
 * netcam client thread, process incoming http streams from server
 * 
 * CALLED BY: netcam_start
 *
 * RETURNS: NULL
 */
static void *netcam_loop(void *arg)
{
	struct context *cnt = arg;

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_loop()\n", cnt->threadnr);
#endif

	if (cnt->conf.setup_mode) {
		const char msg[] = "netcam: entering loop...";
		syslog(LOG_INFO, "[%d] %s", cnt->threadnr, msg);
		printf("[%d] %s\n", cnt->threadnr, msg);
	}
	threads_running++;
	do {
		if (cnt->netcam->read(cnt) == -1) {
			const char msg[] = "netcam: error reading image";
			syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg);
			printf("[%d] %s : %m\n", cnt->threadnr, msg);
		}
	} while (!cnt->finish);
	
	netcam_cleanup(cnt);
	
	threads_running--;
	
	if (cnt->conf.setup_mode) {
		const char msg[] = "netcam: exiting loop...";
		syslog(LOG_INFO, "[%d] %s", cnt->threadnr, msg);
		printf("[%d] %s\n", cnt->threadnr, msg);
	}

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_loop()\n", cnt->threadnr);
#endif

	return NULL;
}

/*
 * process the http header 
 * 
 * CALLED BY: netcam_start, netcam_single_read, netcam_stream_read 
 *
 * RETURNS: -1 on failure, # of headers processed on success
 */
int netcam_read_header(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;
	char *header, *conttype, *boundary;
	int which, status, hcount = 0, ret = 0;
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_read_header()\n", cnt->threadnr);
	printf("[%d] netcam: state [%d]\n", cnt->threadnr, netcam->state);
#endif

	which = netcam->which;
	
	while (1) {
		status = header_get(netcam->response, &header, HG_NONE);
	
		if ((status == HG_ERROR) || (status == HG_EOF)) {
			const char msg[] = "netcam: error reading header";
			syslog(LOG_ERR, "[%d] %s [%s]", cnt->threadnr, msg, header);
			printf("[%d] %s [%s]\n", cnt->threadnr, msg, header);
			free(header);
			ret = -1;
			break;
		} else {
			if (cnt->conf.setup_mode) {
				const char msg[] = "netcam: header";
				syslog(LOG_DEBUG, "[%d] %s [%s]", cnt->threadnr, msg, header);
				printf("[%d] %s [%s]\n", cnt->threadnr, msg, header);
			}
		}

		if ((netcam->state == STATE_INIT1) || (netcam->state == STATE_INIT2)) {
			if ((header_process(header, "Content-Type", http_process_type, &conttype)) && 
					netcam->state == STATE_INIT1) {
				if (strcmp("multipart/x-mixed-replace", conttype) == 0 ||
				    strcmp("multipart/mixed", conttype) == 0) { 
					if (cnt->conf.setup_mode) {
						const char msg[] = "netcam: process mjpeg stream";
						syslog(LOG_DEBUG, "[%d] %s", cnt->threadnr, msg);
						printf("[%d] %s\n", cnt->threadnr, msg);
					}
					netcam->caps.streaming = 1;
					if ((boundary = strstr(header, "boundary="))) {
						boundary += 9;
						netcam->boundary = strdup(boundary);
						if (cnt->conf.setup_mode) {
							const char msg[] = "netcam: boundary string";
							syslog(LOG_DEBUG, "[%d] %s [%s]", cnt->threadnr, msg, netcam->boundary);
							printf("[%d] %s [%s]\n", cnt->threadnr, msg, netcam->boundary);
						}
					}
					netcam->read = (void *) netcam_stream_read;
				} else if (strcmp("image/jpeg", conttype) == 0) {
					if (cnt->conf.setup_mode) {
						const char msg[] = "netcam: process jpeg image";
						syslog(LOG_DEBUG, "[%d] %s", cnt->threadnr, msg);
						printf("[%d] %s\n", cnt->threadnr, msg);
					}
					netcam->caps.streaming = 0;
					netcam->read = (void *) netcam_single_read;
				} else {
					const char msg[] = "netcam: unknown Content-Type";
					syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg);
					printf("[%d] %s\n", cnt->threadnr, msg);
					free(header);
					ret = -1;
					break;
				}
			}
			if ((header_process(header, "Content-Length", 
				header_extract_number,
				&netcam->image[which].buffer.used))) {
#ifdef NETCAM_NO_CONTENTLENGTH_DEBUG
				netcam->caps.contentlength = 0;
				if (cnt->conf.setup_mode) {
					const char msg[] = "netcam: content-length not supported";
					syslog(LOG_DEBUG, "[%d] %s", cnt->threadnr, msg);
					printf("[%d] %s\n", cnt->threadnr, msg);
				}
#else
				netcam->caps.contentlength = 1;
				if (cnt->conf.setup_mode) {
					const char msg[] = "netcam: content-length supported";
					syslog(LOG_DEBUG, "[%d] %s", cnt->threadnr, msg);
					printf("[%d] %s\n", cnt->threadnr, msg);
				}
#endif

#ifdef NETCAM_DEBUG
				printf("[%d] netcam: init image size %d\n", cnt->threadnr, netcam->image[which].buffer.used);
#endif
			}
		} else if (netcam->caps.contentlength == 1) {
			if ((header_process(header, "Content-Length", 
				header_extract_number,
				&netcam->image[which].buffer.used))) {

#ifdef NETCAM_DEBUG
				printf("[%d] netcam: cont image size %d\n", cnt->threadnr, netcam->image[which].buffer.used);
#endif
			}
		}
		
		if (!*header) {
			free(header);
			break;
		}
		free(header);
		hcount++;
	}

	if (ret != -1) {
		ret = hcount;
	}
		
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_read_header() = %d\n", cnt->threadnr, ret);
#endif

	return ret;
}

/*
 * opens a netcam network connection 
 * 
 * CALLED BY: netcam_start, netcam_single_read, netcam_stream_read
 *
 * RETURNS: -1 on failure, 0 on success
 */
int netcam_connect(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;
	int sock, ret = 0;
	struct sockaddr_in server;
	struct hostent *host_info;
	struct timeval tv;	
	char *request = NULL;
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_connect()\n", cnt->threadnr);
#endif

	netcam_disconnect(cnt);
	
	sock = socket(PF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		const char msg[] = "netcam: socket()";
		syslog(LOG_ERR, "[%d] %s : %m", cnt->threadnr, msg);
		printf("[%d] %s : %m\n", cnt->threadnr, msg);
		ret = -1;
	} else {
		memset(&server, 0, sizeof(server));
		host_info = gethostbyname(netcam->url.host);
		if (host_info == NULL) {
			const char msg[] = "netcam: gethostbyname()";
			syslog(LOG_ERR, "[%d] %s(%s) : %m", cnt->threadnr, msg, netcam->url.host);
			printf("[%d] %s(%s) : %m\n", cnt->threadnr, msg, netcam->url.host);
			ret = -1;
		} else {
			memcpy((char *) &server.sin_addr, host_info->h_addr, host_info->h_length);
	
			server.sin_family = AF_INET;
			server.sin_port = htons(netcam->url.port);
	
			tv.tv_sec = CONNECT_TIMEOUT;
			tv.tv_usec = 0;
			setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
	
			if (connect(sock, (struct sockaddr *) &server, sizeof(server)) < 0) {
				const char msg[] = "netcam: connect()";
				syslog(LOG_ERR, "[%d] %s : %m", cnt->threadnr, msg);
				printf("[%d] %s : %m\n", cnt->threadnr, msg);
				ret = -1;
			} else {
				if (netcam->userpass != NULL) {
					request = (char *) mymalloc(strlen(REQUEST_AUTH) +
					strlen(netcam->url.path) + strlen(netcam->userpass) + 1);
					sprintf(request, REQUEST_AUTH, netcam->url.path, netcam->userpass);
				} else {
					request = (char *) mymalloc(strlen(REQUEST) +
					          strlen(netcam->url.path) + 1);
					sprintf(request, REQUEST, netcam->url.path);
				}
#ifdef NETCAM_DEBUG
				printf("[%d] netcam: request=[%s]\n", cnt->threadnr, request);
#endif
			}
		}
		if (ret == -1) {
			close(sock);
		}
	}

	if (ret == 0) {
		rbuf_initialize(netcam->response, sock);
	
		send(RBUF_FD(netcam->response), request, strlen(request), 0);
	}
	
	if (request) free(request);
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_connect() = %d\n", cnt->threadnr, ret);
#endif

	return ret;
}

/*
 * reopen a netcam network connection 
 * 
 * CALLED BY: netcam_start, netcam_stream_read
 *
 * RETURNS: -1 on failure, 0 on success
 */
int netcam_reconnect(struct context *cnt)
{
	int retries, ret = 0;
	struct timespec delay_time, remaining_time;

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_reconnect()\n", cnt->threadnr);
#endif

	retries = 0;
	do {
		const char msg[] = "netcam: reconnecting";
		syslog(LOG_ERR, "[%d] %s [%d]", cnt->threadnr, msg, retries);
		printf("[%d] %s [%d]\n", cnt->threadnr, msg, retries);
		if (cnt->finish) {
			ret = -1;
			break;
		}
		retries++;
		delay_time.tv_sec = RECONNECT_TIMEOUT;
		delay_time.tv_nsec = 0;
		while (nanosleep(&delay_time, &remaining_time) == -1) {
			delay_time.tv_sec = remaining_time.tv_sec;
			delay_time.tv_nsec = remaining_time.tv_nsec;
		}
	} while (((ret = netcam_connect(cnt)) != 0) && (retries < RECONNECT_RETRIES));
	
	if (ret == 0) {
		const char msg[] = "netcam: reconnection successful";
		syslog(LOG_ERR, "[%d] %s [%d]", cnt->threadnr, msg, retries);
		printf("[%d] %s [%d]\n", cnt->threadnr, msg, retries);
	} else {
		const char msg[] = "netcam: reconnection failed";
		syslog(LOG_ERR, "[%d] %s [%d]", cnt->threadnr, msg, retries);
		printf("[%d] %s [%d]\n", cnt->threadnr, msg, retries);
	}	

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_reconnect() = %d\n", cnt->threadnr, ret);
#endif

	return ret;
}

/*
 * closes a netcam network connection
 *
 * CALLED BY: netcam_connect, netcam_cleanup, netcam_single_read
 */
void netcam_disconnect(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_disconnect()\n", cnt->threadnr);
#endif

	if (rbuf_initialized_p(netcam->response)) {
		if (netcam->caps.streaming) {
			netcam->image[netcam->which].buffer.dirty = 1;
			netcam->image[netcam->which].buffer.used = 0;
		}
		close(RBUF_FD(netcam->response));
		rbuf_discard(netcam->response);
		rbuf_uninitialize(netcam->response);
	}

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_disconnect()\n", cnt->threadnr);
#endif

}

/*
 * reads a single non-content length image from the netcam
 *
 * CALLED BY: netcam_read_image
 *
 * RETURNS: -1 on failure, 0 on success
 * 
 * TODO: consider breaking this function into two parts, it is really parsing 
 *   two different types of images; not to mention it is too long! 
 */
int netcam_read_image_no_contentlength(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;
	int which, offset, ret = 0, cur_used, nbytes;
	char ch, *boundary, *image_pos;

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_read_image_no_contentlength(), which = %d\n", cnt->threadnr, netcam->which);
#endif

	which = netcam->which;

	image_pos = netcam->image[which].buffer.ptr;
	if (netcam->caps.streaming) {		/* streaming mode */
		boundary = netcam->boundary;
		while (1) {
			/* FIXME: this will never fail if a boundary string is never found;
			 * well, that's not true.. eventually we will run out of memory. :)
			 * So, I will probably determine an acceptable amount of allocatable
			 * memory and set that as the maximum limit for this routine. */
			RBUF_READCHAR(netcam->response, &ch);
#if 0
			if (ch == 0) break;
#endif
			if (netcam->image[which].buffer.size <=
					(image_pos - netcam->image[which].buffer.ptr)) {
				offset = image_pos - netcam->image[which].buffer.ptr;
				netcam->image[which].buffer.size += INIT_BUFFER;
				netcam->image[which].buffer.ptr =
					(char *) myrealloc(netcam->image[which].buffer.ptr, 
							   netcam->image[which].buffer.size, "netcam image");
				image_pos = netcam->image[which].buffer.ptr + offset;
#ifdef NETCAM_DEBUG
				printf("[%d] netcam: buffer[%d] resize = %d\n", 
						cnt->threadnr, which, netcam->image[which].buffer.size);
#endif
			}
			*image_pos = ch;
			if (*image_pos++ == *boundary++) {
				if (*boundary == '\0') {
#ifdef NETCAM_DEBUG
					printf("[%d] netcam: found boundary string\n", cnt->threadnr);
#endif
					netcam->image[which].buffer.used = 
						image_pos - netcam->image[which].buffer.ptr;
#ifdef NETCAM_DEBUG
					printf("[%d] netcam: buffer[%d] used = %d\n", 
							cnt->threadnr, which, netcam->image[which].buffer.used);
#endif
					break;
				}
			} else {
				boundary = netcam->boundary;
			}
		}
	} else {				/* single image mode */
		cur_used = rbuf_flush(netcam->response, netcam->image[which].buffer.ptr, INIT_BUFFER);
	
		while (1) {
			if ((netcam->image[which].buffer.size - cur_used) < INIT_BUFFER) {
				netcam->image[which].buffer.size += INIT_BUFFER;
				netcam->image[which].buffer.ptr =
					(char *) myrealloc(netcam->image[which].buffer.ptr,
							   netcam->image[which].buffer.size, "netcam image");
#ifdef NETCAM_DEBUG
				printf("[%d] netcam: buffer[%d] resize = %d\n", 
						cnt->threadnr, which, netcam->image[which].buffer.size);
#endif
			}
			nbytes = recv(RBUF_FD(netcam->response), &netcam->image[which].buffer.ptr[cur_used], INIT_BUFFER, 0);
			if (nbytes < 0) {
				const char msg[] = "netcam: error reading image";
				syslog(LOG_ERR, "[%d] %s : %m", cnt->threadnr, msg);
					printf("[%d] %s : %m\n", cnt->threadnr, msg);
					netcam->image[which].buffer.used = 0;
				ret = -1;
				break;
			}
			cur_used += nbytes;
			if (nbytes == 0) break;
		}

		netcam->image[which].buffer.used = cur_used;
	}

	if (ret == 0) {
		netcam->image[which].buffer.dirty = 0;
	}
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_read_image_no_contentlength() = %d\n", cnt->threadnr, ret);
#endif

	return ret;
}	
/*
 * reads a single content length image from the netcam
 *
 * CALLED BY: netcam_read_image
 *
 * RETURNS: -1 on failure, 0 on success
 */
int netcam_read_image_contentlength(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;
	int which, cur_size, cur_left, ret = 0;
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_read_image_contentlength(), which = %d\n", cnt->threadnr, netcam->which);
#endif

	which = netcam->which;
	if (netcam->image[which].buffer.used > netcam->image[which].buffer.size) {
		netcam->image[which].buffer.size = netcam->image[which].buffer.used;
		netcam->image[which].buffer.ptr =
		     myrealloc(netcam->image[which].buffer.ptr,
				     netcam->image[which].buffer.size,
				     "netcam_read_image");
	}
	
	cur_size = rbuf_flush(netcam->response, netcam->image[which].buffer.ptr,
	                      netcam->image[which].buffer.size);
	
	while (netcam->image[which].buffer.used > cur_size) {
		cur_left = recv(RBUF_FD(netcam->response),
				&netcam->image[which].buffer.ptr[cur_size],
				netcam->image[which].buffer.used - cur_size, 0);
		if (cur_left < 0) {
			const char msg[] = "netcam: error reading image";
			syslog(LOG_ERR, "[%d] %s : %m", cnt->threadnr, msg);
			printf("[%d] %s : %m\n", cnt->threadnr, msg);
			netcam->image[which].buffer.used = 0;
			ret = -1;
			break;
		}
		if (cur_left == 0) break;
		cur_size += cur_left;
	}
	if (ret == 0) {
		netcam->image[which].buffer.dirty = 0;
	}
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_read_image_contentlength() = %d\n", cnt->threadnr, ret);
#endif

	return ret;
}	

/*
 * determine proper image type to read and call appropriate function
 * to read the image
 *
 * CALLED BY: netcam_stream_read, netcam_single_read
 *
 * RETURNS: -1 on failure, 0 on success
 */
int netcam_read_image(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;
	int ret = 0;
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_read_image(), which = %d\n", cnt->threadnr, netcam->which);
#endif

	if (netcam->caps.contentlength) { 
		ret = netcam_read_image_contentlength(cnt);
	} else {
		ret = netcam_read_image_no_contentlength(cnt);
	}
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_read_image() = %d\n", cnt->threadnr, ret);
#endif

	return ret;
}	

/*
 * reads from a non-streaming (single image based) netcam
 *
 * CALLED BY: netcam_next
 *
 * RETURNS: -1 on failure, 0 on success
 */
int netcam_single_read(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;
	int which, hretries = 0, ret = 0;
	struct timespec timeout;
	struct timeval now;

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_single_read()\n", cnt->threadnr);
#endif

	if (netcam_connect(cnt) != 0) {
		ret = -1;
	} else {
		pthread_mutex_lock(&netcam->mutex);
		which = netcam->which = !netcam->which;
		gettimeofday(&now, NULL);
		timeout.tv_sec = now.tv_sec + COND_WHICH_TIMEOUT;
		timeout.tv_nsec = now.tv_usec * 1000;
		pthread_cond_timedwait(&netcam->which_cond, &netcam->mutex, &timeout);
		pthread_mutex_unlock(&netcam->mutex);

		pthread_mutex_lock(&netcam->image[which].mutex);
		do {
			ret = netcam_read_header(cnt);
			hretries++;
		} while ((ret == 0) && (hretries < MAX_HEADER_RETRIES));
		if (ret != -1) {
			ret = netcam_read_image(cnt);
		}
		pthread_cond_signal(&netcam->dirty_cond);
		pthread_mutex_unlock(&netcam->image[which].mutex);
		netcam_disconnect(cnt);
	}

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_single_read() = %d\n", cnt->threadnr, ret);
#endif
	return ret;
}

/*
 * reads from a streaming netcam
 *
 * CALLED BY: netcam_next
 *
 * RETURNS: -1 on failure, 0 on success
 */
int netcam_stream_read(struct context *cnt)
{
	struct netcam_context *netcam = cnt->netcam;
	int which, hretries, ret = 0;

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_stream_read()\n", cnt->threadnr);
#endif

	pthread_mutex_lock(&netcam->mutex);
	which = netcam->which = !netcam->which;
	pthread_mutex_unlock(&netcam->mutex);

	pthread_mutex_lock(&netcam->image[which].mutex);
stream_read_retry:
	hretries = 0;
	do {
		ret = netcam_read_header(cnt);
		hretries++;
	} while ((ret == 0) && (hretries < MAX_HEADER_RETRIES));
	if (ret != -1) {
		ret = netcam_read_image(cnt);
	}
	if (ret < 0) {
		ret = netcam_reconnect(cnt);
		if (ret == 0) {
			netcam_read_header(cnt);
			goto stream_read_retry;
		}
	}
	pthread_cond_signal(&netcam->dirty_cond);
	pthread_mutex_unlock(&netcam->image[which].mutex);

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_stream_read() = %d\n", cnt->threadnr, ret);
#endif

	return ret;
}

/*
 * initialize the netcam, open network connection, begin the initial 
 * processing of data and creates the loop thread
 *
 * called by motion video thread.
 *
 * RETURNS: -1 on failure, 0 on success
 *
 */
int netcam_start(struct context *cnt)
{
	struct netcam_context *netcam;
	struct jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;
	char *userpass;
	int which, hretries, ret;
	
#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_start()\n", cnt->threadnr);
#endif

	if (cnt->conf.setup_mode) {
		const char msg[] = "netcam: starting...";
		syslog(LOG_INFO, "[%d] %s", cnt->threadnr, msg);
		printf("[%d] %s\n", cnt->threadnr, msg);
	}
	
	cnt->netcam = 
		(struct netcam_context *) mymalloc(sizeof(struct netcam_context));
	netcam = cnt->netcam;

	if (netcam_url_parse(&netcam->url, cnt->conf.netcam_url)) {
		const char msg[] = "netcam: invalid url";
		syslog(LOG_INFO, "[%d] %s [%s]", cnt->threadnr, msg, cnt->conf.netcam_url);
		printf("[%d] %s [%s]\n", cnt->threadnr, msg, cnt->conf.netcam_url);
		return -1;
	}
	
	pthread_mutex_init(&netcam->mutex, NULL);
	pthread_mutex_init(&netcam->image[0].mutex, NULL);
	pthread_mutex_init(&netcam->image[1].mutex, NULL);
	pthread_cond_init(&netcam->which_cond, NULL);	
	pthread_cond_init(&netcam->dirty_cond, NULL);

	netcam->state = 0;
	netcam->caps.streaming = 0;
	netcam->caps.contentlength = 0;
	netcam->response = (struct rbuf *) mymalloc(sizeof(struct rbuf));
	rbuf_uninitialize(netcam->response);
	netcam->boundary = NULL;
	netcam->image[0].buffer.dirty = 1;
	netcam->image[0].buffer.size = INIT_BUFFER;
	netcam->image[0].buffer.used = 0;
	netcam->image[0].buffer.ptr = (char *) mymalloc(netcam->image[0].buffer.size);
	netcam->image[1].buffer.dirty = 1;
	netcam->image[1].buffer.size = INIT_BUFFER;
	netcam->image[1].buffer.used = 0;
	netcam->image[1].buffer.ptr = (char *) mymalloc(netcam->image[1].buffer.size);
	which = netcam->which = 0;

	userpass = NULL;
	if (cnt->conf.netcam_userpass) {
		userpass = cnt->conf.netcam_userpass;
	} else if (netcam->url.userpass) {
		userpass = netcam->url.userpass;
	}
	if (userpass) {
		netcam->userpass = (char *) mymalloc(BASE64_LENGTH(strlen(userpass)) + 1);
		/* base64_encode can read 3 bytes after the end of the string, reallocate it */
		userpass = myrealloc(userpass, strlen(userpass) + 3, "userpass");
		base64_encode(userpass, netcam->userpass, strlen(userpass));
	} else {
		netcam->userpass = NULL;
	}

	ret = netcam_connect(cnt);
start_retry:
	netcam->state = STATE_INIT1;
	hretries = 0;
	do {
		ret = netcam_read_header(cnt);
		hretries++;
	} while ((ret == 0) && (hretries < MAX_HEADER_RETRIES));
	if (ret != -1) {
		if (netcam->caps.streaming) {
			netcam->state = STATE_INIT2;
			ret = netcam->read(cnt);
		} else {
			ret = netcam_read_image(cnt); 
			netcam_disconnect(cnt);
		}
	}
	if (ret < 0) {
		ret = netcam_reconnect(cnt);
		if (ret == 0) {
			goto start_retry;
		}
	}

	if (ret == 0) {
		netcam->state = STATE_RUN;

		netcam->read(cnt);
		pthread_mutex_lock(&netcam->mutex);
		pthread_cond_signal(&netcam->which_cond);
		pthread_mutex_unlock(&netcam->mutex);

		cinfo.err = jpeg_std_error(&jerr);
		jpeg_create_decompress(&cinfo);
		jpeg_memory_src(&cinfo,	netcam->image[which].buffer.ptr,
		                netcam->image[which].buffer.used);
		jpeg_read_header(&cinfo, TRUE);

		cnt->imgs.width=cinfo.image_width;
		cnt->imgs.height=cinfo.image_height;
		cnt->imgs.size=cnt->imgs.width*cnt->imgs.height*3/2;
		cnt->imgs.motionsize=cnt->imgs.width*cnt->imgs.height;
		cnt->imgs.type=VIDEO_PALETTE_YUV420P;

		/* Create the netcam loop thread */
		pthread_attr_init(&netcam->thread_attr);
		pthread_attr_setdetachstate(&netcam->thread_attr, PTHREAD_CREATE_DETACHED);
		pthread_create(&netcam->thread, &netcam->thread_attr, &netcam_loop, cnt);
	} else {
		netcam_cleanup(cnt);
	}

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_start() = %d\n", cnt->threadnr, ret);
#endif

	return ret;
}

/*
 * process next image in netcam buffer, called by motion video thread.
 *
 * RETURNS: next image in buffer
 *
 */
unsigned char *netcam_next(struct context *cnt, char *image)
{
	struct jpeg_decompress_struct cinfo;
	struct my_error_mgr jerr;
	unsigned char *pic, *upic, *vpic;
	JSAMPARRAY line;
	int i, line_size, y, which, width, height;
	JSAMPROW row[1];
	struct netcam_context *netcam;
	struct timespec timeout;
	struct timeval now;

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: enter netcam_next(), which = %d\n", cnt->threadnr, cnt->netcam->which);
#endif

	pic = NULL;
	netcam = cnt->netcam;

	pthread_mutex_lock(&netcam->mutex);
	which = netcam->which;
	pthread_mutex_unlock(&netcam->mutex);

	pthread_mutex_lock(&netcam->image[which].mutex);
	while (netcam->image[which].buffer.dirty && !cnt->finish) {
		gettimeofday(&now, NULL);
		timeout.tv_sec = now.tv_sec + COND_DIRTY_TIMEOUT;
		timeout.tv_nsec = now.tv_usec * 1000;
		pthread_cond_timedwait(&netcam->dirty_cond, &netcam->image[which].mutex, &timeout);
	}
	pthread_mutex_lock(&netcam->mutex);
	pthread_cond_signal(&netcam->which_cond);
	pthread_mutex_unlock(&netcam->mutex);

	if (netcam->image[which].buffer.used > 0) {
		cinfo.err = jpeg_std_error(&jerr.pub);
		jerr.pub.error_exit = my_error_exit;

		if (setjmp(jerr.setjmp_buffer)) {
			const char msg[] = "netcam: jpeglib decompression failed";
			syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg);
			printf("[%d] %s\n", cnt->threadnr, msg);
			jpeg_destroy_decompress(&cinfo);
			pic = NULL;
		} else {
			jpeg_create_decompress(&cinfo);
			jpeg_memory_src(&cinfo, netcam->image[which].buffer.ptr, 
					netcam->image[which].buffer.used);
			jpeg_read_header(&cinfo, TRUE);

			cinfo.out_color_space = JCS_YCbCr;

			width = cnt->rotate_data.cap_width;
			height = cnt->rotate_data.cap_height;

			jpeg_start_decompress(&cinfo);
			line_size = width * 3;

			line = (*cinfo.mem->alloc_sarray) ((j_common_ptr) &cinfo,
				JPOOL_IMAGE, cinfo.output_width * cinfo.output_components, 1);
			pic = image;
			upic = pic + width * height;
			vpic = upic + (width * height) / 4;
			row[0] = (unsigned char *) line;
			y = 0;

			while (cinfo.output_scanline < height) {
				jpeg_read_scanlines(&cinfo, row, 1);
				for (i = 0; i < line_size; i += 3) {
					pic[i / 3] = ((unsigned char *) line)[i];
					if (i & 1) {
						upic[(i / 3) / 2] = ((unsigned char *) line)[i + 1];
						vpic[(i / 3) / 2] = ((unsigned char *) line)[i + 2];
					}
				}
				pic += line_size / 3;
				if (y++ & 1) {
					upic += width / 2;
					vpic += width / 2;
				}
			}

			jpeg_finish_decompress(&cinfo);
			jpeg_destroy_decompress(&cinfo);

			if(cnt->rotate_data.degrees > 0) {
				/* rotate as specified */
				rotate_map(image, cnt);
			}
		}
	}
	netcam->image[which].buffer.dirty = 1;
	pthread_mutex_unlock(&netcam->image[which].mutex);

#ifdef NETCAM_DEBUG
	printf("[%d] netcam: exit netcam_next()\n", cnt->threadnr);
#endif

	return pic;
}
