/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <errno.h>
#include <signal.h>

#include <math.h>
#include <ctype.h>
#include <string.h>
#include <dirent.h>
#include <pwd.h>

#if defined(__sun) || defined(_AIX)
#include <strings.h>
#endif

#include <fstream.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/resource.h>
#include <sys/utsname.h>

#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#ifndef __CYGWIN32__
#include <netinet/tcp.h>
#include <sys/un.h>
#endif

#ifdef __APPLE__
typedef int socklen_t;
#endif

#ifdef _AIX
#include <sys/select.h>
#endif

//
// NX include files.
//

#include "NX.h"

#include "Misc.h"
#include "Control.h"
#include "Socket.h"
#include "Statistics.h"
#include "Alerts.h"
#include "Context.h"

#include "ClientProxy.h"
#include "ServerProxy.h"

#include "Message.h"

//
// System specific defines.
//

#if defined(__EMX__ ) || defined(__CYGWIN32__)

struct sockaddr_un
{
  u_short sun_family;
  char sun_path[108];
};

#endif

//
// HP-UX hides this define.
//

#if defined(hpux) && !defined(RLIM_INFINITY)

#define RLIM_INFINITY    0x7fffffff

#endif

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// Enable log output in signal handler.
// This is very dangerous when proxy is
// run as thread.
//

#undef  UNSAFE

//
// Upper limit of pre-allocated
// buffer for string parameters.
//

#define DEFAULT_STRING_LENGTH              256

//
// Maximum length of remote options
// string passed by peer proxy at
// startup.
//

#define DEFAULT_REMOTE_OPTIONS_LENGTH      512

//
// Maximum length of NX display string.
//

#define DEFAULT_DISPLAY_OPTIONS_LENGTH     1024

//
// Default number of cache file names
// to send from client to server side.
//

#define DEFAULT_REMOTE_CACHE_FILES         100

//
// Maximum length of configuration file.
//

#define CONTROL_PARAMETERS_FILE_LENGTH     4096

//
// Lines expected in configuration file.
//

#define CONTROL_PARAMETERS_FILE_ENTRIES    87

//
// Define this to set core limits.
//

#define COREDUMPS

#ifdef COREDUMPS

static int EnableCoreDumps(void);

#endif

//
// Define FLUSH in Misc.h if you want
// immediate flush of log output.
//

ostream *logofs = NULL;

//
// Provide operational parameters.
//

Control *control = NULL;

//
// Collect and print statistics.
//

Statistics *statistics = NULL;

//
// This class makes the hard work.
//

Proxy *proxy = NULL;

//
// Allow faults to be recovered by
// jumping back into the main loop.
//

Context *context = NULL;

//
// Initialization functions.
//

static int SetupProxyConnection();
static int ResetProxyConnection();
static int NegotiateProxyCache();

static int SetupTCPSocket();
static int SetupUnixSocket();
static int SetupSyncSocket();
static int SetupKeybdSocket();
static int SetupSambaSocket();
static int SetupMediaSocket();
static int SetupDisplaySocket(int &xServerAddrFamily, sockaddr *&xServerAddr,
                                  unsigned int &xServerAddrLength);
//
// Convenience functions.
//

static void StartDaemon(unsigned int);
static void KillDaemon(unsigned int);
static void MakeLockFileName(char *, unsigned int);

static int WaitForRemote(int portNum);
static int ConnectToRemote(const char *const remoteHost, int portNum);
static int ConnectToRemoteWithSSH(const char *const remoteHost, int portNum);
static int SendRemoteOptions(int fd);

static int ReadRemoteOptions(int fd, char *buffer, int size, char stop);
static int WriteRemoteOptions(int fd, const char *buffer, int size);

static void PrintVersionInfo();
static void PrintProcessInfo();
static void PrintConnectionInfo();
static void PrintResetInfo();
static void PrintUsage(int argc, const char **argv);

//
// This is not static to avoid a warning.
// Option was replaced with loading of NX
// library version.
//

void PrintCopyrightInfo();

static const char *GetArg(int &argi, int argc, const char **argv);
static int CheckArg(const char *type, const char *name, const char *value);
static int ParseArg(const char *type, const char *name, const char *value);

extern "C"
{
  int ParseCommandLineOptions(int argc, const char **argv);
  int ParseEnvironmentOptions(const char *env, int force);
}

static int ParseFileOptions(const char *file);
static int NegotiateProxyOptions(int fd);
static int ParseRemoteOptions(char *opts);

//
// These functions are used to parse literal
// values provided by the user and set the
// control parameters accordingly.
//

static int ParseLinkOption(const char *opt);
static int ParseLimitOption(const char *opt);
static int ParseCacheOption(const char *opt);
static int ParseShmemOption(const char *opt);
static int ParseImagesOption(const char *opt);
static int ParsePackOption(const char *opt);

//
// Set host and port where NX proxy is supposed
// to be listening in case such parameters are
// given on the command line.
//

static int ParseHostOption(const char *opt);

//
// Translate pack method id in a literal.
//

static int ParsePackMethod(const int method, const int quality);

//
// Read control parameters from a simple
// configuration file. Agent will query
// some of these values at startup, after
// having successfully opened the display.
//

static int ParseControlParameters(const char *link);

//
// Check if local and remote protocol versions
// are compatible and, eventually, downgrade
// local version to the minimum level that is
// known to work.
//

static int SetVersion();

//
// Setup defaults used for log and statistics.
//

static int SetLog();

//
// Setup incoming TCP ports used for sync,
// multimedia, samba and embedded keyboard
// connections according to user's wishes.
//

static int SetPorts();

//
// Set the maximum number of open descriptors.
//

static int SetDescriptors();

//
// Initialize, one after the other, all the
// configuration parameters.
//

static int SetParameters();

//
// Initialize the specific parameters.
//

static int SetSession();
static int SetCache();
static int SetShmem();
static int SetPack();
static int SetImages();
static int SetLimits();

//
// Set up control parameters according to
// expected link speed negotiated between
// proxies.
//

static int SetLink();

static int SetLinkModem();
static int SetLinkIsdn();
static int SetLinkAdsl();
static int SetLinkWan();
static int SetLinkLan();

//
// Adjust compression parameters.
//

static int SetCompression();

static int SetCompressionModem();
static int SetCompressionIsdn();
static int SetCompressionAdsl();
static int SetCompressionWan();
static int SetCompressionLan();

//
// Manage to restore cache from file at startup.
//

static int SendLocalCaches(int fd, const char *path, const char prefix, int entries);
static char *ReceiveRemoteCaches(int fd, int entries);
static char *SelectLastCache(char *list, const char *path);

static char *GetRootPath();
static char *GetCachePath();
static char *GetImagesPath();
static char *GetSessionPath();

static int OpenInputFile(char *name, istream *&stream);
static int OpenOutputFile(char *name, ostream *&stream);
static int ReopenOutputFile(char *name, ostream *&stream, int limit);

//
// Perform operations on managed descriptors
// in main loop.
//

static void handleCheckSessionInLoop();
static void handleCheckLimitsInLoop();

#if defined(INFO) || defined(TEST)
static void handleCheckStateInLoop(int &totalFDs);
static void handleCheckSelectInLoop(int &totalFDs, int &readFDs, int &writeFDs,
                                        fd_set &readSet, fd_set &writeSet,
                                            T_timestamp selectTs);
#endif

static void handleCheckSessionInConnect();

static inline void handleSetReadInLoop(fd_set &readSet, int &readFDs);
static inline void handleSetWriteInLoop(fd_set &writeSet, int &writeFDs);
static inline void handleSetTimeoutInLoop(T_timestamp &selectTs);
static inline void handleSetListenersInLoop(fd_set &writeSet, int &readFDs);
static inline void handleSetScheduleInLoop();

static void handleAlertInLoop();
static void handleStatisticsInLoop();
static void handleResetInLoop();
static void handlePingInLoop(int &diffTs);

static inline void handleWakeupChannelsInLoop();
static inline void handleSplitChannelsInLoop();
static inline void handleMotionChannelsInLoop();
static inline void handleCongestionChannelsInLoop();

static void handleAcceptTcpConnectionInLoop(int &fd);
static void handleAcceptUnixConnectionInLoop(int &fd);
static void handleAcceptSyncConnectionInLoop(int &fd);
static void handleAcceptKeybdConnectionInLoop(int &fd);
static void handleAcceptSambaConnectionInLoop(int &fd);
static void handleAcceptMediaConnectionInLoop(int &fd);

static inline void handleReadProxyInLoop(int &resultFDs, int &totalFDs, fd_set &readSet);
static inline void handleReadChannelsInLoop(int &resultFDs, int &totalFDs, fd_set &readSet);

static inline void handleFlushProxyInLoop(int &resultFDs, fd_set &writeSet);
static inline void handleFlushChannelsInLoop(int &resultFDs, fd_set &writeSet);

static inline void handleWriteProxyEarlyInLoop();
static inline void handleWriteProxyInLoop();

static void handleLogReopenInLoop(T_timestamp &logsTs, T_timestamp &nowTs);

//
// This variable tells whether or not the
// client should initiate the connection.
//

static int clientInitiateConnection = 0;

//
// These two variables are needed for the forking
// We don't fork by default since old versions
// didn't.
//

int useDaemonMode = 0;

static char lockFileName[DEFAULT_STRING_LENGTH] = { 0 };

//
// Default proxy lock file. This goes in user's
// home directory.
//

static const char *LOCK_FILE_NAME = ".nxproxy.pid";

//
// Root of directory structure to be created by proxy.
//

static char rootDir[DEFAULT_STRING_LENGTH] = { 0 };

//
// Root of statistics and log files to be created by proxy.
//

static char sessionDir[DEFAULT_STRING_LENGTH] = { 0 };

//
// Log files for errors and statistics. Error log is
// the place where we print also debug informations.
// Both files are closed, deleted and reopened as
// their size exceed the limit set in control class.
//

static char logFileName[DEFAULT_STRING_LENGTH] = { 0 };
static char statFileName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal representing selected link speed
// parameter. Value is translated in control values
// used by proxies to stay synchronized.
//

static char linkSpeedName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal representing selected
// cache size.
//

static char cacheSizeName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal representing selected
// shared memory segment size.
//

static char shmemSizeName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal of images cache size.
//

static char imagesSizeName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal for bandwidth limit.
//

static char bitrateLimitName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal for image packing method.
//

static char packMethodName[DEFAULT_STRING_LENGTH] = { 0 };

//
// Its corresponding value from NXpack.h.
//

static int packMethod  = -1;
static int packQuality = -1;

//
// String literal for session type. Persistent caches
// are searched in directory whose name matches this
// parameter.
//

static char sessionType[DEFAULT_STRING_LENGTH] = { 0 };

//
// Unique id assigned to session. It is used as
// name of directory where all files are placed.
//

static char sessionId[DEFAULT_STRING_LENGTH] = { 0 };

//
// This is set to last signal received in handler.
//

static int lastSignal = 0;

//
// This is set to code of last requested alert.
// 

static int lastAlert = 0;

//
// Set to last dialog process launched by proxy.
// 

int lastDialog = 0;

//
// Set to watchdog process launched by proxy.
// 

int lastWatchdog = 0;

//
// Set if a cache house-keeper process is run.
// 

int lastKeeper = 0;

//
// Set if shutdown was requested through a signal.
//

static int lastKill = 0;

//
// Other destriptors used locally.
//

ostream *statofs = NULL;
ostream *errofs = NULL;

//
// Save standard error's rdbuf here
// and restore at exit.
//

static streambuf *errsbuf = NULL;

//
// Main loop file descriptors.
//

static int tcpFD      = -1;
static int unixFD     = -1;
static int syncFD     = -1;
static int keybdFD    = -1;
static int sambaFD    = -1;
static int mediaFD    = -1;
static int proxyFD    = -1;
static int internalFD = -1;

int useUnixSocket = 1;

static int useTCPSocket      = 1;
static int useSyncSocket     = 0;
static int useKeybdSocket    = 0;
static int useSambaSocket    = 0;
static int useMediaSocket    = 0;
static int useInternalSocket = 0;

//
// Set by user if he/she wants to modify
// the default TCP_NODELAY option as set
// in control.
//

static int useNoDelay = -1;

//
// Set if user wants to override default
// flush timeout set according to link.
//

static int useFlush = -1;

//
// Set if user wants to hide the RENDER
// extension or wants to short-circuit
// some simple replies at client side.
//

static int useRender = -1;
static int useTaint  = -1;

//
// Don't use SSH encryption by default.
//

static int sshPort = -1;

//
// Name of Unix socket created by the client proxy to
// accept client connections. File must be unlinked
// by cleanup function.
//

static char unixSocketName[DEFAULT_STRING_LENGTH] = { 0 };

//
// Other parameters.
//

static char remoteHost[DEFAULT_STRING_LENGTH]  = { 0 };
static char acceptHost[DEFAULT_STRING_LENGTH]  = { 0 };
static char displayHost[DEFAULT_STRING_LENGTH] = { 0 };
static char authCookie[DEFAULT_STRING_LENGTH]  = { 0 };

static int proxyPort = DEFAULT_NX_PROXY_PORT;
static int xPort     = DEFAULT_NX_X_PORT;

//
// Helper channels are disabled by default.
//

static int syncPort  = -1;
static int keybdPort = -1;
static int sambaPort = -1;
static int mediaPort = -1;

//
// TODO: Must add HTTP support at
// both client and server side in
// stock production code.
//
// static int httpPort  = -1;
//

//
// State variables shared between the init
// function and the main loop.
//

T_timestamp startTs;
T_timestamp logsTs;
T_timestamp selectTs;
T_timestamp nowTs;

fd_set readSet;
fd_set writeSet;

int readFDs;
int writeFDs;
int totalFDs;
int resultFDs;

int diffTs;

//
// Macro is TRUE if our side is client side.
//

#define WE_ARE_CLIENT_PROXY       ((*remoteHost == '\0' && \
                                      clientInitiateConnection == 0) || \
                                  (*remoteHost != '\0' && \
                                      clientInitiateConnection == 1))

//
// Macro is TRUE if we assigned our own proxy mode.
//

#define WE_SET_PROXY_MODE         (control -> ProxyMode != PROXY_UNDEFINED)

//
// Macro is TRUE if our side is the one that
// should connect to remote.
//

#define WE_INITIATE_CONNECTION   ((control -> ProxyMode == PROXY_SERVER && \
                                       clientInitiateConnection == 0) ||   \
                                  (control -> ProxyMode == PROXY_CLIENT && \
                                       clientInitiateConnection == 1))

//
// Here are interfaces declared in NX.h.
//

int NXProxy(int fd, int mode, const char* display)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  #ifdef __CYGWIN32__

  char *info;

  if ((info = getenv("NX_SESSION")) != NULL)
  {
    errofs = new ofstream(info, ios::app);

    cerr.rdbuf(errofs -> rdbuf());
  }

 #endif

  //
  // Check if have already performed a parsing of
  // parameters, as in the case we are running as
  // a stand-alone process. If needed create the
  // parameters repository
  //

  if (control == NULL)
  {
    control = new Control();
  }

  if (mode == NX_MODE_CLIENT)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXProxy: Selected proxy mode is NX_MODE_CLIENT.\n"
            << logofs_flush;
    #endif

    control -> ProxyMode = PROXY_CLIENT;

    if (fd != NX_FD_ANY)
    {
      #ifdef TEST

      *logofs << "NXProxy: Internal descriptor for X client connection is FD#"
              << fd << ".\n" << logofs_flush;

      *logofs << "NXProxy: Disabling listening on further X client connections.\n"
              << logofs_flush;

      #endif

      useTCPSocket      = 0;
      useUnixSocket     = 0;
      useInternalSocket = 1;

      internalFD = fd;
    }
  }
  else if (mode == NX_MODE_SERVER)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXProxy: Selected proxy mode is NX_MODE_SERVER.\n"
            << logofs_flush;
    #endif

    control -> ProxyMode = PROXY_SERVER;

    if (fd != NX_FD_ANY)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXProxy: PANIC! Internal descriptor for X server connections "
              << "not supported yet.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Internal descriptor for X server connections "
           << "not supported yet.\n";

      HandleCleanup();
    }
  }
  #if defined(INFO) || defined(TEST)
  else
  {
    *logofs << "NXProxy: Requested proxy mode is NX_MODE_ANY.\n"
            << logofs_flush;
  }
  #endif

  const char *env;

  if (display != NULL)
  {
    env = display;

    if (strncasecmp(env, "nx/nx,", 6) != 0 &&
            strncasecmp(env, "nx,", 3) != 0 &&
                strncasecmp(env, "nx", 2) != 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXProxy: PANIC! Display options string '" << env
              << "' must start with 'nx' or 'nx/nx' prefix.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Display options string '" << env
           << "' must start with 'nx' or 'nx/nx' prefix.\n";

      HandleCleanup();
    }
  }
  else
  {
    env = getenv("DISPLAY");
  }

  if (ParseEnvironmentOptions(env, 0) < 0)
  {
    cerr << "Error" << ": Parsing of display options failed.\n";

    HandleCleanup();
  }

  //
  // Setup the proxy connection.
  //

  #ifdef TEST
  *logofs << "NXProxy: Going to initialize the NX transport.\n"
          << logofs_flush;
  #endif

  NXTransportInit();

  context = new Context();

  int result = setjmp(context -> getLoopContext());

  if (result == 1)
  {
    NXTransportRestore();
  }

  //
  // Loop forever until the connections
  // to the peer proxy is dropped.
  //

  #ifdef TEST
  *logofs << "NXProxy: Going to run the NX transport loop.\n"
          << logofs_flush;
  #endif

  for (;;)
  {
    NXTransportPrepare();

    NXTransportSelect();

    NXTransportExecute();
  }

  //
  // Loop should never return.
  //

  return 0;
}

int NXCreateProcess(int fd, int mode, const char* display)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "NXCreateProcess: Going to fork with NX pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  int pid = fork();

  if (pid != 0)
  {
    if (pid < 0)
    { 
      #if defined(INFO) || defined(TEST)
      *logofs << "NXCreateProcess: PANIC! Function fork failed with result '"
              << pid << "'. Error is " << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Function fork failed with result '" << pid
           << "'. Error is " << EGET() << " '" << ESTR() << "'.\n";
    }
    #if defined(INFO) || defined(TEST)
    else
    {
      *logofs << "NXCreateProcess: Created NX child process with pid "
              << pid << ".\n" << logofs_flush;
    }
    #endif

    return pid;
  }

  //
  // Use brute force. Unfortunately close-on-exec
  // on parent fd, as set in agent, doesn't look
  // to work for us.
  //

  for (int i = 3; i < 256 && i != fd; i++)
  {
    if (close(i) != 0 && EGET() != EBADF)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXCreateProcess: PANIC! Cannot close FD#" << i
              << ". Error is " << EGET() << " '" << ESTR()
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Cannot close FD#" << i
           << ". Error is " << EGET() << " '" << ESTR()
           << "'.\n";

      HandleCleanup();
    }
  }

  return NXProxy(fd, mode, display);
}

int NXCreateProcessWithInfo(int fd, int mode, const char* display, const char* info)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "NXCreateProcessWithInfo: Name selected for info file is '"
          << info << "'.\n" << logofs_flush;
  #endif

  if (info == NULL || *info == '\0')
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXCreateProcessWithInfo: PANIC! No info file was specified.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": No info file specified for creation of NX transport.\n";

    return -1;
  }

  //
  // Redirect standard error to file.
  //

  if (errofs == NULL)
  {
    errofs = new ofstream(info, ios::app);

    errsbuf = cerr.rdbuf(errofs -> rdbuf());
  }

  int pid = NXCreateProcess(fd, mode, display);

  //
  // Reset errofs so we don't get confused
  // in case of a caller's retry.
  //

  errofs = NULL;

  return pid;
}

void NXExit(int code)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  static int recurse;

  if (++recurse > 1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXExit: Aborting parent due to recursion through exit.\n"
            << logofs_flush;
    #endif

    abort();
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "NXExit: Parent called exit with code '" << code << "'.\n"
          << logofs_flush;
  #endif

  exit(code);
}

//
// Checks the parameters for subsequent
// initialization of the NX transport.
//

int NXTransportCreate(int fd, int mode, const char* display)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  #ifdef __CYGWIN32__

  char *info;

  if ((info = getenv("NX_SESSION")) != NULL)
  {
    errofs = new ofstream(info, ios::app);

    cerr.rdbuf(errofs -> rdbuf());
  }

 #endif

  //
  // Check if have already performed a parsing of
  // parameters, as in the case we are running as
  // a stand-alone process. If needed create the
  // parameters repository
  //

  if (control == NULL)
  {
    control = new Control();
  }

  if (mode == NX_MODE_CLIENT)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXTransportCreate: Selected proxy mode is NX_MODE_CLIENT.\n"
            << logofs_flush;
    #endif

    control -> ProxyMode = PROXY_CLIENT;

    if (fd != NX_FD_ANY)
    {
      #ifdef TEST

      *logofs << "NXTransportCreate: Internal descriptor for X client "
              << "connection is FD#" << fd << ".\n" << logofs_flush;

      *logofs << "NXTransportCreate: Disabling listening on further X "
              << "client connections.\n" << logofs_flush;

      #endif

      useTCPSocket      = 0;
      useUnixSocket     = 0;
      useInternalSocket = 1;

      internalFD = fd;
    }
  }
  else if (mode == NX_MODE_SERVER)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXTransportCreate: Selected proxy mode is NX_MODE_SERVER.\n"
            << logofs_flush;
    #endif

    control -> ProxyMode = PROXY_SERVER;

    if (fd != NX_FD_ANY)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXTransportCreate: PANIC! Internal descriptor for X "
              << "server connections not supported yet.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Internal descriptor for X server "
           << "connections not supported yet.\n";

      HandleCleanup();
    }
  }
  #if defined(INFO) || defined(TEST)
  else
  {
    *logofs << "NXTransportCreate: Requested proxy mode is NX_MODE_ANY.\n"
            << logofs_flush;
  }
  #endif

  const char *env;

  if (display != NULL)
  {
    env = display;

    if (strncasecmp(env, "nx/nx,", 6) != 0 &&
            strncasecmp(env, "nx,", 3) != 0 &&
                strncasecmp(env, "nx", 2) != 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXTransportCreate: PANIC! Display options string '" << env
              << "' must start with 'nx' or 'nx/nx' prefix.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Display options string '" << env
           << "' must start with 'nx' or 'nx/nx' prefix.\n";

      HandleCleanup();
    }
  }
  else
  {
    env = getenv("DISPLAY");
  }

  if (ParseEnvironmentOptions(env, 0) < 0)
  {
    cerr << "Error" << ": Parsing of display options failed.\n";

    HandleCleanup();
  }

  return 1;
}

//
// Connects or waits for connection from
// the remote peer, negotiates the session
// parameters and initializes the proxy.
//

int NXTransportInit()
{
  #ifdef TEST
  *logofs << "NXTransportInit: Going to initialize the NX transport.\n"
          << logofs_flush;
  #endif

  //
  // Disable limits on core dumps.
  //

  #ifdef COREDUMPS

  EnableCoreDumps();

  #endif

  //
  // Set signal handlers.
  //

  EnableSignals();

  //
  // Track how much time we spend in initialization.
  //

  startTs = getTimestamp();

  //
  // If not explicitly specified, determine if local
  // mode is client or server according to parameters
  // provided so far.
  //

  if (!WE_SET_PROXY_MODE)
  {
    #if defined(INFO) || defined (TEST)

    *logofs << "Loop: No proxy mode defined yet with "
            << "remote host '" << remoteHost << "' and port '"
            << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
            << "'.\n" << logofs_flush;

    *logofs << "Loop: Client initiating connection flag is "
            << clientInitiateConnection << ".\n" << logofs_flush;

    *logofs << "Loop: WE_ARE_CLIENT_PROXY macro returns "
            << WE_ARE_CLIENT_PROXY << ".\n" << logofs_flush;

    *logofs << "Loop: WE_INITIATE_CONNECTION macro returns "
            << WE_INITIATE_CONNECTION << ".\n" << logofs_flush;

    #endif

    if (WE_ARE_CLIENT_PROXY)
    {
      control -> ProxyMode = PROXY_CLIENT;
    }
    else
    {
      control -> ProxyMode = PROXY_SERVER;
    }
  }

  //
  // Start a watchdog. If initialization cannot
  // be completed before timeout, then clean up
  // everything and exit.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Starting watchdog process with timeout of "
            << control -> InitTimeout / 1000 << " seconds.\n"
            << logofs_flush;
    #endif

    lastWatchdog = NXWatchdog(control -> InitTimeout / 1000);
  }

  //
  // Open the log files.
  //

  SetLog();

  //
  // Set sync, multimedia and other
  // auxiliary ports.
  //

  SetPorts();

  //
  // Increase the number of maximum open
  // file descriptors for this process.
  //

  SetDescriptors();

  //
  // Print preliminary info.
  //

  PrintProcessInfo();

  //
  // Set local endianess.
  //

  unsigned int test = 1;

  setHostBigEndian(*((unsigned char *) (&test)) == 0);

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Local host is "
          << (hostBigEndian() ? "big endian" : "little endian")
          << ".\n" << logofs_flush;
  #endif

  int xServerAddrFamily;
  sockaddr *xServerAddr;
  unsigned int xServerAddrLength;

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    //
    // If running as client proxy, listen on sockets
    // that mimic an X display to which X clients can
    // connect to (e.g. unix:8 and/or localhost:8).
    //

    if (useTCPSocket)
    {
      SetupTCPSocket();
    }

    if (useUnixSocket)
    {
      SetupUnixSocket();
    }

    if (useSyncSocket)
    {
      SetupSyncSocket();
    }

    if (useKeybdSocket)
    {
      SetupKeybdSocket();
    }

    if (useSambaSocket)
    {
      SetupSambaSocket();
    }

    if (useMediaSocket)
    {
      SetupMediaSocket();
    }
  }
  else
  {
    //
    // Disable any listener.
    //

    useUnixSocket  = 0;
    useTCPSocket   = 0;
    useSyncSocket  = 0;
    useKeybdSocket = 0;
    useSambaSocket = 0;
    useMediaSocket = 0;

    //
    // Cannot determine if channel is an
    // internal connection at X server
    // side.
    //

    useInternalSocket = 0;

    //
    // If we are at server side then get
    // ready to open the local display.
    //

    SetupDisplaySocket(xServerAddrFamily, xServerAddr, xServerAddrLength);

    //
    // Check that user provided for embedded
    // keyboard the same display where appli-
    // cations are going to be run.
    //

    if (keybdPort > 0 && keybdPort != X_TCP_PORT + xPort)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Embedded keyboard port must match the X server port '"
              << X_TCP_PORT + xPort << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Embedded keyboard port must match the X server port '"
           << X_TCP_PORT + xPort << "'.\n";

      HandleCleanup();
    }
  }

  //
  // Daemon mode is never used in NX.
  // It's here just for compatibility
  // with older versions of nxproxy.
  //

  if (useDaemonMode)
  {
    StartDaemon(DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort);
  }

  //
  // If we are client proxy we need to complete
  // our initializazion phase. We, infact, are
  // the side who imposes link speed and cache
  // size to the remote peer.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    SetParameters();
  }

  //
  // Set up low-bandwidth connection between
  // the proxies.
  //

  SetupProxyConnection();

  //
  // Deal with caches.
  //

  NegotiateProxyCache();

  //
  // If we are server proxy we completed our
  // initializazion phase according to values
  // provided by the client side.
  //

  if (control -> ProxyMode == PROXY_SERVER)
  {
    SetParameters();
  }

  //
  // We completed both parsing of user's parameters
  // and handlshaking with remote proxy. Now print
  // a brief summary including the most significant
  // control values.
  //

  PrintConnectionInfo();

  //
  // Start the cache house-keeping process.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Starting house-keeping process with "
            << "cache " << (control -> ClientTotalStorageSizeLimit > 0 ?
               control -> PersistentCacheDiskLimit : 0)
            << " and images " << 0 << ".\n" << logofs_flush;
    #endif

    lastKeeper = NXKeeper((control -> ClientTotalStorageSizeLimit > 0 ?
                               control -> PersistentCacheDiskLimit : 0),
                                   0, control -> RootPath);
  }
  else
  {
    //
    // Actually it would be enough to start
    // the process only if save is enabled.
    //

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Starting house-keeping process with "
            << "cache " << (control -> ClientTotalStorageSizeLimit > 0 ?
               control -> PersistentCacheDiskLimit : 0) << " and images "
            << (control -> ImageCacheEnableLoad > 0 ||
                    control -> ImageCacheEnableSave > 0 ?
                        control -> ImageCacheDiskLimit : 0)
            << ".\n" << logofs_flush;
    #endif

    lastKeeper = NXKeeper((control -> ClientTotalStorageSizeLimit > 0 ?
                               control -> PersistentCacheDiskLimit : 0),
                                   (control -> ImageCacheEnableLoad > 0 ||
                                        control -> ImageCacheEnableSave > 0 ?
                                            control -> ImageCacheDiskLimit : 0),
                                                control -> RootPath);
  }

  //
  // Create the proxy.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    proxy = new ClientProxy(proxyFD);
  }
  else
  {
    proxy = new ServerProxy(proxyFD, displayHost, xServerAddrFamily,
                                xServerAddr, xServerAddrLength,
                                    syncPort, keybdPort, sambaPort,
                                        mediaPort);
  }

  if (proxy == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating the NX transport.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating the NX transport.\n";

    HandleCleanup();
  }

  //
  // Set socket options on proxy link, then propagate link
  // configuration to proxy. This includes translating some
  // control parameters in 'local' and 'remote'. Finally
  // adjust cache parameters according to pack method and
  // session type selected by user.
  //

  if (proxy -> handleSocketConfiguration() < 0 ||
          proxy -> handleLinkConfiguration() < 0 ||
              proxy -> handleCacheConfiguration() < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error configuring the NX transport.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error configuring the NX transport.\n";

    HandleCleanup();
  }

  //
  // Create the statistics repository.
  //

  if (control -> CollectStatistics)
  {
    statistics = new Statistics(proxy);

    if (statistics == NULL)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Error initializing NX statistics.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Error initializing NX statistics.\n";

      HandleCleanup();
    }
  }

  //
  // Are we going to use an internal IPC connection
  // with agent? In this case create the connection
  // using the socket descriptor provided by caller.
  //

  if (control -> ProxyMode == PROXY_CLIENT &&
          useInternalSocket == 1)
  {
    if (proxy -> handleNewXConnection(internalFD) < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Error creating new X connection.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Error creating new X connection.\n";

      HandleCleanup();
    }
  }

  //
  // Cancel the initialization watchdog.
  //

  if (lastWatchdog > 0)
  {
    if (kill(lastWatchdog, SIGTERM) < 0 && EGET() != ESRCH)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Couldn't kill watchdog process with pid '"
              << lastWatchdog << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Couldn't kill watchdog process with pid '"
           << lastWatchdog << "'.\n";

      HandleCleanup();
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Resetting pid of last watchdog process.\n"
            << logofs_flush;
    #endif

    lastWatchdog = 0;
  }

  logsTs = getTimestamp();

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Time spent in initialization is "
          << diffTimestamp(startTs, getTimestamp())
          << " Ms.\n" << logofs_flush;
  #endif

  startTs = getTimestamp();

  #ifdef DEBUG
  *logofs << "Loop: New timestamp is " << startTs.tv_sec
          << " seconds and " << startTs.tv_usec
          << " microseconds.\n" << logofs_flush;
  #endif

  //
  // Ensure any data produced by initialization
  // is actually written to the proxy socket.
  //

  proxy -> handleFlush(flush_if_any);

  //
  // In case the proxy connection is broken we
  // want to jump here and try to setup a new
  // proxy connection.
  //

  cerr << "Info" << ": Starting X protocol compression.\n";

  return 1;
}

//
// Restore the proxy connection in case
// of failure.
//

int NXTransportRestore()
{
  #ifdef TEST
  *logofs << "NXTransportRestore: Going to restore the NX transport.\n"
          << logofs_flush;
  #endif

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Showing the broken connection alert.\n"
          << logofs_flush;
  #endif

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    HandleAlert(RESTART_DEAD_PROXY_CONNECTION_CLIENT_ALERT);
  }
  else
  {
    HandleAlert(RESTART_DEAD_PROXY_CONNECTION_SERVER_ALERT);
  }

  handleAlertInLoop();

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Going to restore proxy connection.\n"
          << logofs_flush;
  #endif

  ResetProxyConnection();

  return 1;
}

//
// Prepare the file sets and the timeout
// for a later execution of the select().
//

int NXTransportPrepare()
{
  #ifdef DEBUG
  *logofs << "NXTransportPrepare: Going to prepare the NX transport.\n"
          << logofs_flush;
  #endif

  //
  // Check if it's time to give up.
  //

  handleCheckSessionInLoop();

  #ifdef DEBUG
  *logofs << "Loop: Back in the loop waiting for data.\n"
          << logofs_flush;
  #endif

  //
  // Check if local proxy is consuming
  // too many resources.
  //

  handleCheckLimitsInLoop();

  //
  // Set up masks for select.
  //

  readFDs  = 0;
  writeFDs = 0;

  FD_ZERO(&readSet);
  FD_ZERO(&writeSet);

  //
  // Set descriptors of listening sockets.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    handleSetListenersInLoop(readSet, readFDs);
  }

  //
  // Set descriptors of both proxy and X
  // connections.
  //

  handleSetReadInLoop(readSet, readFDs);

  //
  // Find out which file descriptors have
  // data to write.
  //

  handleSetWriteInLoop(writeSet, writeFDs);

  //
  // Set value of timeout of select.
  //

  handleSetTimeoutInLoop(selectTs);

  totalFDs = (readFDs > writeFDs ? readFDs : writeFDs);

  #if defined(INFO) || defined(TEST)

  handleCheckSelectInLoop(totalFDs, readFDs, writeFDs,
                              readSet, writeSet, selectTs);
  #endif

  //
  // Register time spent handling messages.
  //

  nowTs = getTimestamp();

  diffTs = diffTimestamp(startTs, nowTs);

  #ifdef TEST
  *logofs << "Loop: Difference from loop timestamp is "
          << diffTs << " Ms.\n"
          << logofs_flush;
  #endif

  //
  // TODO: Should add read time in two parts
  // otherwise limits are checked before the
  // counters are updated with time spent in
  // the last loop.
  //

  control -> addReadTime(diffTs);

  if (control -> CollectStatistics)
  {
    statistics -> addReadTime(diffTs);
  }

  startTs = nowTs;

  #ifdef DEBUG
  *logofs << "Loop: New timestamp is " << startTs.tv_sec
          << " seconds and " << startTs.tv_usec
          << " microseconds.\n" << logofs_flush;
  #endif

  #if defined(INFO) || defined(TEST)
  *logofs << endl << logofs_flush;
  #endif

  return 1;
}

//
// True if descriptor is actually an
// internal NX connection. This is
// really weird.
//

int NXTransportCheck()
{
  return (internalFD != -1);
}

//
// Returns the best suited timeout for
// the next call to select().
//

const struct timeval *NXTransportTimeout()
{
  //
  // TODO: This is the bogus call used to
  // go out of select often enough in X11
  // code to enable the new NX transport.
  // 

  if (selectTs.tv_sec > 0 ||
          selectTs.tv_usec > 10 * 1000)
  {
    selectTs.tv_sec  = 0;
    selectTs.tv_usec = 10 * 1000;
  }

  #ifdef DEBUG
  *logofs << "NXTransportTimeout: Going to return NX timeout of "
          << selectTs.tv_sec << " S and " << (double) selectTs.tv_usec / 1000
          << " Ms at " << strTimestamp() << ".\n" << logofs_flush;
  #endif

  return &selectTs;
}

//
// Call select() to find out if any of
// the handled descriptors has data.
//

int NXTransportSelect()
{
  #ifdef DEBUG
  *logofs << "NXTransportSelect: Going to select the NX descriptors.\n"
          << logofs_flush;
  #endif

  //
  // Wait for selected sockets or timeout.
  // PGID_USE_PID is specific to HP-UX 10.X.
  //

  ESET(0);

  //
  // TODO: Bogus call to select as it should
  // be implemented to enable use of new NX
  // transport functions in nx-X11 code.
  // 
  // if (internalFD != -1)
  // {
  //   selectTs.tv_sec  = 0;
  //   selectTs.tv_usec = 0;
  // }
  //
  // resultFDs = select(totalFDs, &readSet, &writeSet, NULL, &selectTs);
  //

  #if (defined(__hpux) || defined(hpux)) && !defined(PGID_USE_PID)

  resultFDs = select(totalFDs, (int *) &readSet, (int *) &writeSet, NULL, &selectTs);

  #else

  resultFDs = select(totalFDs, &readSet, &writeSet, NULL, &selectTs);

  #endif

  //
  // Get time spent in select. Unfortunately accouting of
  // time statistics was initially done in milliseconds
  // and I never had time to switch to microseconds. This
  // is a real problem on fast machines where each run is
  // unlikely to take more than 500 us. So consider that
  // the results can be inaccurate.
  //

  nowTs = getTimestamp();

  diffTs = diffTimestamp(startTs, nowTs);

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Out of select after " << diffTs
          << " Ms " << "at " << strTimestamp(nowTs)
          << " with result " << resultFDs << ".\n"
          << logofs_flush;
  #endif

  startTs = nowTs;

  #ifdef DEBUG
  *logofs << "Loop: New timestamp is "
          << startTs.tv_sec << " seconds and "
          << startTs.tv_usec << " microseconds.\n"
          << logofs_flush;
  #endif

  control -> addIdleTime(diffTs);

  if (control -> CollectStatistics)
  {
    statistics -> addIdleTime(diffTs);
  }

  //
  // Check the result of select.
  //

  if (resultFDs < 0)
  {
    if (EGET() == EINTR)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Select failed due to EINTR error.\n"
              << logofs_flush;
      #endif

      FD_ZERO(&readSet);
      FD_ZERO(&writeSet);

      ESET(0);

      resultFDs = 0;
    }
    else
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to select failed. Error is "
              << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Call to select failed. Error is "
           << EGET() << " '" << ESTR() << "'.\n";

      //
      // It seems that Solaris can return an EINVAL
      // error even before we actually close the socket
      // instead of just failing at the time we try to
      // read or write to it. This is definitely a
      // Solaris bug.
      //

      #ifdef __sun

      FD_ZERO(&readSet);
      FD_ZERO(&writeSet);

      ESET(0);

      resultFDs = 0;

      #else

      HandleCleanup();

      #endif
    }
  }

  return 1;
}

//
// Perform the required actions on all
// the descriptors having I/O pending.
//

int NXTransportExecute()
{
  #ifdef DEBUG
  *logofs << "NXTransportExecute: Going to execute I/O on NX descriptors.\n"
          << logofs_flush;
  #endif

  //
  // Determine for how many bytes and for how
  // long this loop is going to run.
  //

  handleSetScheduleInLoop();

  //
  // Flush any data on newly writable sockets.
  //

  handleFlushProxyInLoop(resultFDs, writeSet);

  handleFlushChannelsInLoop(resultFDs, writeSet);

  //
  // Check if there are bytes left in proxy
  // read buffer. Increase interactivity by
  // doing this before handling X messages
  // at client side.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    handleReadProxyInLoop(resultFDs, totalFDs, readSet);
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: At mark - 1 - difference from loop timestamp is "
          << diffTimestamp(startTs, getTimestamp())
          << " Ms.\n" << logofs_flush;
  #endif

  //
  // Handle incoming data on X channels.
  //

  handleReadChannelsInLoop(resultFDs, totalFDs, readSet);

  //
  // Restart, if possible, any client that
  // was put to sleep.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    handleWakeupChannelsInLoop();
  }

  //
  // Check if there are pointer motion
  // events to flush. This applies only
  // to X server side.
  //

  if (control -> ProxyMode == PROXY_SERVER)
  {
    handleMotionChannelsInLoop();
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: At mark - 2 - difference from loop timestamp is "
          << diffTimestamp(startTs, getTimestamp())
          << " Ms.\n" << logofs_flush;
  #endif

  //
  // Decoding data coming from remote proxy
  // can be time-consuming at X server side
  // so flush data read from X connections
  // (for example replies) before doing it.
  //

  if (control -> ProxyMode == PROXY_SERVER)
  {
    handleWriteProxyEarlyInLoop();
  }

  //
  // Handle incoming data on the proxy link.
  // At server side we do this after having
  // managed data coming from X connections.
  //

  if (control -> ProxyMode == PROXY_SERVER)
  {
    handleReadProxyInLoop(resultFDs, totalFDs, readSet);
  }

  //
  // Check if any channel is bogged down
  // by a faster remote peer.
  //

  handleCongestionChannelsInLoop();

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: At mark - 3 - difference from loop timestamp is "
          << diffTimestamp(startTs, getTimestamp())
          << " Ms.\n" << logofs_flush;
  #endif

  //
  // Check if there are images that are
  // currently being streamed.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    handleSplitChannelsInLoop();
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: At mark - 4 - difference from loop timestamp is "
          << diffTimestamp(startTs, getTimestamp())
          << " Ms.\n" << logofs_flush;
  #endif

  //
  // Check if connection is still alive.
  //

  handlePingInLoop(diffTs);

  //
  // Check if user sent a signal to produce
  // statistics.
  //

  handleStatisticsInLoop();

  //
  // Check if user sent a signal to restart
  // the proxy link or change configuration.
  //

  handleResetInLoop();

  //
  // Check if we have an alert to show.
  //

  handleAlertInLoop();

  //
  // Finally check if is possible to write
  // any data produced for the proxy link.
  //

  handleWriteProxyInLoop();

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: At mark - 5 - difference from loop timestamp is "
          << diffTimestamp(startTs, getTimestamp())
          << " Ms.\n" << logofs_flush;
  #endif

  //
  // Check coherency of internal state.
  //

  #if defined(INFO) || defined(TEST)

  handleCheckStateInLoop(totalFDs);

  #endif

  //
  // Truncate the logs if needed.
  //

  handleLogReopenInLoop(logsTs, nowTs);

  return 1;
}

static int SetupProxyConnection()
{
  if (WE_INITIATE_CONNECTION)
  {
    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Going to connect to " << remoteHost
            << ":" << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
            << ".\n" << logofs_flush;
    #endif

    if (sshPort == -1)
    {
      proxyFD = ConnectToRemote(remoteHost, DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort);
    }
    else
    {
      proxyFD = ConnectToRemoteWithSSH(remoteHost, DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort);
    }

    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Connected to remote proxy on FD#" 
            << proxyFD << ".\n" << logofs_flush;
    #endif

    cerr << "Info" << ": Connection to remote proxy '" << remoteHost
         << ":" << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
         << "' established.\n" << logofs_flush;
  }
  else
  {
    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Going to wait for connection on port " 
            << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort << ".\n"
            << logofs_flush;
    #endif

    proxyFD = WaitForRemote(DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort);

    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Connected to remote proxy on FD#" 
            << proxyFD << ".\n" << logofs_flush;
    #endif

    cerr << "Info" << ": Connection with remote proxy "
         << "established.\n" << logofs_flush;
  }

  //
  // Set TCP_NODELAY on proxy descriptor
  // to reduce startup time. Option will
  // later be disabled if needed.
  //

  SetNoDelay(proxyFD, 1);

  //
  // Server side proxy must always be the one that
  // sends its version and options first, so, in
  // some way, client side can be the the one that
  // has the last word on the matter.
  //

  if (control -> ProxyMode == PROXY_SERVER)
  {
    //
    // At the moment only client side can listen
    // for connections, thus we are the connecting
    // side and it's our turn to send authorization
    // cookie and any local option to the remote
    // end.
    //

    if (SendRemoteOptions(proxyFD) < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Cannot send options to remote proxy.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Cannot send options to remote proxy.\n";

      HandleCleanup();
    }

    //
    // If remote proxy will like our version,
    // it will respond with its parameters.
    //

    if (NegotiateProxyOptions(proxyFD) < 0)
    {
      HandleCleanup();
    }

    cerr << "Info" << ": Handshaking with remote proxy '" << remoteHost
         << ":" << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
         << "' completed.\n" << logofs_flush;
  }
  else
  {
    //
    // Get options from remote proxy and
    // validate them. 
    //

    if (NegotiateProxyOptions(proxyFD) < 0)
    {
      HandleCleanup();
    }

    //
    // Now we send our version number and any
    // required option. If server side doesn't
    // like our setup it will simply close the
    // connection.
    //

    if (SendRemoteOptions(proxyFD) < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Cannot send options to remote proxy.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Cannot send options to remote proxy.\n";

      HandleCleanup();
    }

    cerr << "Info" << ": Handshaking with remote proxy "
         << "completed.\n" << logofs_flush;
  }

  return 1;
}

static int NegotiateProxyCache()
{
  //
  // Deal with caches.
  //

  if ((control -> PersistentCachePath = GetCachePath()) == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error getting or creating cache path.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error getting or creating cache path.\n";

    HandleCleanup();
  }

  cerr << "Info" << ": Synchronizing local and remote caches.\n" << logofs_flush;

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Path of cache files is '" << control -> PersistentCachePath
          << "'.\n" << logofs_flush;
  #endif

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    SendLocalCaches(proxyFD, control -> PersistentCachePath,
                       'C', DEFAULT_REMOTE_CACHE_FILES);

    //
    // We will read the name of cache plus the stop character.
    //

    char buffer[DEFAULT_STRING_LENGTH];

    //
    // Leave space for a trailing null.
    //

    if (ReadRemoteOptions(proxyFD, buffer, sizeof("cachefile=") + MD5_LENGTH * 2 + 3, ' ') < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Remote NX proxy closed the connection.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Remote NX proxy closed the connection.\n";

      #ifdef PANIC
      *logofs << "Loop: PANIC! Error receiving name of cache selected by peer.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Error receiving name of cache selected by peer.\n";

      HandleCleanup();
    }

    char *cacheName = strstr(buffer, "cachefile=");

    if (cacheName == NULL)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Invalid cache file option '"
              << buffer << "' provided by remote proxy.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Invalid cache file option '"
           << buffer << "' provided by remote proxy.\n";

      HandleCleanup();
    }

    cacheName += strlen("cachefile=");

    if (control -> PersistentCacheName != NULL)
    {
      delete [] control -> PersistentCacheName;
    }

    control -> PersistentCacheName = NULL;

    if (strncasecmp(cacheName, "none", strlen("none")) == 0)
    {
      #if defined(INFO) || defined (TEST)
      *logofs << "Loop: No cache file selected by remote proxy.\n"
              << logofs_flush;
      #endif
    }
    else if (strlen(cacheName) != MD5_LENGTH * 2 + 3 ||
                 *(cacheName + MD5_LENGTH * 2 + 2) != ' ')
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Invalid cache file name '"
              << cacheName << "' provided by remote proxy.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Invalid cache file name '"
           << cacheName << "' provided by remote proxy.\n";

      HandleCleanup();
    }
    else
    {
      //
      // It is "C-" + 32 + "\0".
      //

      control -> PersistentCacheName = new char[MD5_LENGTH * 2 + 3];

      *(cacheName + MD5_LENGTH * 2 + 2) = '\0';

      strcpy(control -> PersistentCacheName, cacheName);

      #if defined(INFO) || defined (TEST)
      *logofs << "Loop: Cache file '" << control -> PersistentCacheName
              << "' selected by remote proxy.\n" << logofs_flush;
      #endif
    }
  }
  else
  {
    char *cacheList = ReceiveRemoteCaches(proxyFD, DEFAULT_REMOTE_CACHE_FILES);

    if (cacheList == NULL)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Error receiving list of caches available from peer.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Error receiving list of caches available from peer.\n";

      HandleCleanup();
    }

    control -> PersistentCacheName = SelectLastCache(cacheList, control -> PersistentCachePath);

    //
    // Get rid of list of caches.
    //

    delete [] cacheList;

    //
    // Send back the selected cache name.
    //

    char buffer[DEFAULT_STRING_LENGTH];

    if (control -> PersistentCacheName != NULL)
    {
      #if defined(INFO) || defined (TEST)
      *logofs << "Loop: Name of selected cache file is '"
              << control -> PersistentCacheName << "'.\n"
              << logofs_flush;
      #endif

      sprintf(buffer, "cachefile=%s%s ",
                  *(control -> PersistentCacheName) == 'C' ? "S-" : "C-",
                      control -> PersistentCacheName + 2);
    }
    else
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: No valid cache file was selected.\n"
              << logofs_flush;
      #endif

      sprintf(buffer, "cachefile=none ");
    }

    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Sending string '" << buffer
            << "' as selected cache file.\n"
            << logofs_flush;
    #endif

    WriteRemoteOptions(proxyFD, buffer, strlen(buffer));
  }

  return 1;
}

static int ResetProxyConnection()
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Send the reset notification to X channels.
  //

  proxy -> handleNotify(notify_begin_reset);

  //
  // Get rid of old persistent cache. In theory
  // path can change to follow a different type
  // of session. Image cache path is determined
  // at startup and will remain the same.
  //

  delete [] control -> PersistentCachePath;

  control -> PersistentCachePath = NULL;

  delete [] control -> PersistentCacheName;

  control -> PersistentCacheName = NULL;

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Closing old proxy connection.\n"
          << logofs_flush;
  #endif

  int oldFD = proxyFD;

  close(proxyFD);

  proxyFD = -1;

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Creating a new proxy connection.\n"
          << logofs_flush;
  #endif

  //
  // Will set proxyFD to the connected socket.
  //

  SetupProxyConnection();

  //
  // Ensure the new socket descriptor has the
  // same number as the old one.
  //

  if (proxyFD != oldFD)
  {
    if (dup2(proxyFD, oldFD) < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to dup2 failed. Error is "
              << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Call to dup2 failed. Error is "
           << EGET() << " '" << ESTR() << "'.\n";

      HandleCleanup();
    }
    else
    {
      close(proxyFD);

      proxyFD = oldFD;
    }
  }

  //
  // Negotiate, if possible, a new cache file.
  //

  NegotiateProxyCache();

  //
  // Inform user about the new cache parameters.
  //

  PrintResetInfo();

  //
  // Reset state from proxy down to channels.
  //

  if (proxy -> handleReset() < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Proxy failure in handleReset().\n"
            << logofs_flush;
    #endif

    HandleCleanup();
  }

  //
  // Set socket options on proxy link.
  //

  proxy -> handleSocketConfiguration();

  //
  // Propagate link configuration. This
  // includes translating some control
  // parameters in 'local' and 'remote'.
  //

  proxy -> handleLinkConfiguration();

  //
  // Further adjust cache parameters according
  // to pack method and session type selected
  // by user.
  //

  proxy -> handleCacheConfiguration();

  //
  // Inform X channels that reset is completed.
  //

  proxy -> handleNotify(notify_end_reset);

  return 1;
}

int SetupTCPSocket()
{
  //
  // Open TCP socket emulating local display.
  //

  tcpFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (tcpFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(tcpFD) < 0)
  {
    HandleCleanup();
  }

  sockaddr_in tcpAddr;
  tcpAddr.sin_family = AF_INET;
  unsigned int xPortTCP = X_TCP_PORT + xPort;
  tcpAddr.sin_port = htons(xPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(tcpFD, (sockaddr *) & tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for TCP port "
            << xPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for TCP port "
         << xPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(tcpFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for TCP port "
            << xPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for TCP port "
         << xPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupUnixSocket()
{
  //
  // Open UNIX domain socket for display.
  //

  unixFD = socket(AF_UNIX, SOCK_STREAM, PF_UNSPEC);

  if (unixFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for UNIX domain"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for UNIX domain"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  sockaddr_un unixAddr;
  unixAddr.sun_family = AF_UNIX;
  struct stat dirStat;

  if ((stat("/tmp/.X11-unix", &dirStat) == -1) && (EGET() == ENOENT))
  {
    mkdir("/tmp/.X11-unix", 0777);
    chmod("/tmp/.X11-unix", 0777);
  }

  sprintf(unixSocketName, "/tmp/.X11-unix/X%d", xPort);
  strcpy(unixAddr.sun_path, unixSocketName);

  if (bind(unixFD, (sockaddr *) & unixAddr,
               strlen(unixSocketName) + 2) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for UNIX domain socket "
            << unixSocketName << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ":  Call to bind failed for UNIX domain socket "
         << unixSocketName << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(unixFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for UNIX domain socket "
            << unixSocketName << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ":  Call to listen failed for UNIX domain socket "
         << unixSocketName << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  //
  // Let any local user to gain access to socket.
  //

  chmod(unixSocketName, 0777);

  return 1;
}

//
// The following is a dumb copy-paste. I
// will write a generic function someday.
//

int SetupSyncSocket()
{
  syncFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (syncFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(syncFD) < 0)
  {
    HandleCleanup();
  }

  sockaddr_in tcpAddr;
  tcpAddr.sin_family = AF_INET;
  unsigned int syncPortTCP = syncPort;
  tcpAddr.sin_port = htons(syncPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(syncFD, (sockaddr *) & tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for sync TCP port "
            << syncPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for sync TCP port "
         << syncPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(syncFD, 4) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for sync TCP port "
            << syncPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for sync TCP port "
         << syncPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupKeybdSocket()
{
  keybdFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (keybdFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(keybdFD) < 0)
  {
    HandleCleanup();
  }

  sockaddr_in tcpAddr;
  tcpAddr.sin_family = AF_INET;
  unsigned int keybdPortTCP = keybdPort;
  tcpAddr.sin_port = htons(keybdPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(keybdFD, (sockaddr *) & tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for embedded keyboard "
            << "TCP port " << keybdPortTCP << ". Error is " << EGET()
            << " '" << ESTR() << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for embedded keyboard "
         << "TCP port " << keybdPortTCP << ". Error is " << EGET()
         << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  if (listen(keybdFD, 4) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for embedded keyboard "
            << "TCP port " << keybdPortTCP << ". Error is " << EGET()
            << " '" << ESTR() << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for embedded keyboard "
         << "TCP port " << keybdPortTCP << ". Error is " << EGET()
         << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupSambaSocket()
{
  sambaFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (sambaFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(sambaFD) < 0)
  {
    HandleCleanup();
  }

  sockaddr_in tcpAddr;
  tcpAddr.sin_family = AF_INET;
  unsigned int sambaPortTCP = sambaPort;
  tcpAddr.sin_port = htons(sambaPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(sambaFD, (sockaddr *) & tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for samba TCP port "
            << sambaPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for samba TCP port "
         << sambaPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(sambaFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for samba TCP port "
            << sambaPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for samba TCP port "
         << sambaPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupMediaSocket()
{
  mediaFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (mediaFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(mediaFD) < 0)
  {
    HandleCleanup();
  }

  sockaddr_in tcpAddr;
  tcpAddr.sin_family = AF_INET;
  unsigned int mediaPortTCP = mediaPort;
  tcpAddr.sin_port = htons(mediaPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(mediaFD, (sockaddr *) & tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for multimedia TCP port "
            << mediaPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for multimedia TCP port "
         << mediaPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(mediaFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for multimedia TCP port "
            << mediaPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for multimedia TCP port "
         << mediaPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupDisplaySocket(int &xServerAddrFamily, sockaddr *&xServerAddr,
                           unsigned int &xServerAddrLength)
{
  xServerAddrFamily = AF_INET;
  xServerAddr = NULL;
  xServerAddrLength = 0;

  char *display;

  if (*displayHost == '\0')
  {
    //
    // Assume DISPLAY as the X server to which
    // we will forward the proxied connections.
    // This means that NX parameters have been
    // passed through other means.
    //

    display = getenv("DISPLAY");

    if (display == NULL || *display == '\0')
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Host X server DISPLAY is not set.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Host X server DISPLAY is not set.\n";

      HandleCleanup();
    }
    else if (strncasecmp(display, "nx/nx,", 6) == 0 ||
                 strncasecmp(display, "nx,", 3) == 0 ||
                     strncasecmp(display, "nx", 2) == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! NX transport on host X server '"
              << display << "' not supported.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": NX transport on host X server '"
           << display << "' not supported.\n";

      cerr << "Error" << ": Please run the local proxy specifying "
           << "the host X server to connect to.\n";

      HandleCleanup();
    }
    else if (strlen(display) >= DEFAULT_STRING_LENGTH)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Host X server DISPLAY cannot exceed "
              << DEFAULT_STRING_LENGTH << " characters.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Host X server DISPLAY cannot exceed "
           << DEFAULT_STRING_LENGTH << " characters.\n";

      HandleCleanup();
    }

    strcpy(displayHost, display);
  }

  display = new char[strlen(displayHost) + 1];

  if (display == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Out of memory handling DISPLAY variable.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Out of memory handling DISPLAY variable.\n";

    HandleCleanup();
  }

  strcpy(display, displayHost);

  char *separator = rindex(display, ':');

  if ((separator == NULL) || !isdigit(*(separator + 1)))
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid display '" << display << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid display '" << display << "'.\n";

    HandleCleanup();
  }

  *separator = '\0';

  xPort = atoi(separator + 1);

  #ifdef TEST
  *logofs << "Loop: Using local X display '" << displayHost
          << "' with host '" << display << "' and port '"
          << xPort << "'.\n" << logofs_flush;
  #endif

  if (separator == display || strcmp(display, "unix") == 0 ||
          strcmp(display, "localhost") == 0)
  {
    //
    // UNIX domain port.
    //

    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Connecting to real X server on UNIX domain socket.\n"
            << logofs_flush;
    #endif

    sockaddr_un *xServerAddrUNIX = new sockaddr_un;

    xServerAddrFamily = AF_UNIX;
    xServerAddrUNIX -> sun_family = AF_UNIX;
    sprintf(unixSocketName, "/tmp/.X11-unix/X%d", xPort);
    struct stat statInfo;

    if (stat(unixSocketName, &statInfo) < 0)
    {
      #if (defined(__hpux) || defined(hpux)) && !defined(PGID_USE_PID)

      sprintf(unixSocketName, "/usr/spool/sockets/X11/%d", xPort);

      if (stat(unixSocketName, &statInfo) < 0)
      {

      #endif

        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't open UNIX connection to X server.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Can't open UNIX connection to X server.\n";

        #ifdef PANIC
        *logofs << "Loop: PANIC! Error " << EGET() << " '" << ESTR()
             << "' opening '" << unixSocketName << "'.\n"
             << logofs_flush;
        #endif

        cerr << "Error" << ": Error " << EGET() << " '" << ESTR()
             << "' opening '" << unixSocketName << "'.\n";

        HandleCleanup();

      #if (defined(__hpux) || defined(hpux)) && !defined(PGID_USE_PID)

      }

      #endif
    }

    strcpy(xServerAddrUNIX -> sun_path, unixSocketName);
    xServerAddr = (sockaddr *) xServerAddrUNIX;
    xServerAddrLength = sizeof(sockaddr_un);
  }
  else
  {
    //
    // TCP port.
    //

    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Connecting to real X server on TCP port.\n"
            << logofs_flush;
    #endif

    xServerAddrFamily = AF_INET;

    int xServerIPAddr = GetHostAddress(display);

    if (xServerIPAddr == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't open TCP connection to X server.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't open TCP connection to X server.\n";

      #ifdef PANIC
      *logofs << "Loop: PANIC! Unknown display host '" << display
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Unknown display host '" << display
           << "'.\n";

      HandleCleanup();
    }

    sockaddr_in *xServerAddrTCP = new sockaddr_in;

    xServerAddrTCP -> sin_family = AF_INET;
    xServerAddrTCP -> sin_port = htons(X_TCP_PORT + xPort);
    xServerAddrTCP -> sin_addr.s_addr = xServerIPAddr;

    xServerAddr = (sockaddr *) xServerAddrTCP;
    xServerAddrLength = sizeof(sockaddr_in);
  }

  delete [] display;

  return 1;
}

void HandleShutdown()
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (proxy -> getShutdown() == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! No shutdown of proxy link "
            << "performed by remote proxy.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Connection with remote peer broken.\n";

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Bytes received so far are " << control -> getBytesIn()
            << ".\n" << logofs_flush;
    #endif

    if (control -> getBytesIn() < 1024)
    {
      cerr << "Error" << ": Please report this problem to "
           << "support personnel.\n";
    }
    else if (control -> EnableRestartOnFailure == 1)
    {
      cerr << "Error" << ": Please check state of your network "
           << "connection.\n";

      //
      // If not in cleanup phase jump back
      // to the main loop.
      //

      if (lastKill == 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Loop: Trying to revert to loop context.\n"
                << logofs_flush;
        #endif

        //
        // TODO: A close() in the reconnection
        // procedure seems to be not enough to
        // ensure that remote proxy identifies
        // the link shutdown.
        //

        shutdown(proxyFD, SHUT_RDWR);

        longjmp(context -> getLoopContext(), 1);
      }
    }
    else
    {
      cerr << "Error" << ": Please check state of your network "
           << "connection and retry.\n";
    }
  }
  else
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Performing clean shutdown of proxy link.\n"
            << logofs_flush;
    #endif

    if (control -> getBytesIn() < 1024 &&
            useInternalSocket == 1)
    {
      cerr << "Info" << ": Your session was closed before reaching "
           << "a usable state.\n";
      cerr << "Info" << ": This can be due to local X server "
           << "refusing access to client.\n";
      cerr << "Info" << ": Please check authorization provided "
           << "by remote X application.\n";
    }

    cerr << "Info" << ": Shutting down the link and exiting.\n";
  }

  HandleCleanup();
}

void HandleChildren()
{
  int pid;
  int status;
  int options;

  options = WNOHANG | WUNTRACED;

  pid = waitpid(-1, &status, options);

  if (pid == 0 || (pid < 0 && EGET() == ECHILD))
  {
    #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
    *logofs << "Loop: No child process exited.\n"
            << logofs_flush;
    #endif
  }
  else if (pid != 0)
  {
    if (WIFSTOPPED(status))
    {
      #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
      *logofs << "Loop: Child process '" << pid << "' was stopped "
              << "with signal " << (WSTOPSIG(status)) << ".\n"
              << logofs_flush;
      #endif
    }
    else
    {
      #if defined(UNSAFE) && (defined(INFO) || defined(TEST))

      if (WIFEXITED(status))
      {
        *logofs << "Loop: Child process '" << pid << "' exited "
                << "with status " << (WEXITSTATUS(status))
                << ".\n" << logofs_flush;
      }
      else if (WIFSIGNALED(status))
      {
        *logofs << "Loop: Child process '" << pid << "' died "
                << "because of signal " << (WTERMSIG(status))
                << ".\n" << logofs_flush;
      }

      #endif

      if (lastDialog == pid)
      {
        #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
        *logofs << "Loop: Resetting pid of last dialog process.\n"
                << logofs_flush;
        #endif

        lastDialog = 0;
      }
      else if (lastWatchdog == pid)
      {
        #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
        *logofs << "Loop: Resetting pid of last watchdog process.\n"
                << logofs_flush;
        #endif

        lastWatchdog = 0;
      }
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to waitpid failed. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to waitpid failed. "
         << "Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }
}

void HandleCleanup()
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Ignore any further signal.
  //

  DisableSignals();

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Going to clean up system resources.\n"
          << logofs_flush;
  #endif

  if (useDaemonMode == 1 && *lockFileName != '\0')
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Going to remove the lock file '"
            << lockFileName << "'.\n" << logofs_flush;
    #endif

    if (unlink(lockFileName) != 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't remove lock file '"
              << lockFileName << "'. Error is " << EGET()
              << " '" << ESTR() << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Can't remove lock file '"
           << lockFileName << "'. Error is " << EGET()
           << " '" << ESTR() << "'.\n";
    }
  }

  //
  // Remove any pending dialog
  // currently showed to user.
  //

  if (lastDialog > 0)
  {
    if (kill(lastDialog, SIGTERM) < 0 && EGET() != ESRCH)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Couldn't kill dialog process with pid '"
              << lastDialog << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Couldn't kill dialog process with pid '"
           << lastDialog << "'.\n";
    }
  }

  //
  // Remove any watchdog.
  //

  if (lastWatchdog > 0)
  {
    if (kill(lastWatchdog, SIGTERM) < 0 && EGET() != ESRCH)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Couldn't kill watchdog process with pid '"
              << lastWatchdog << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Couldn't kill watchdog process with pid '"
           << lastWatchdog << "'.\n";
    }
  }

  //
  // Kill the cache house-keeping process.
  //

  if (lastKeeper > 0)
  {
    if (kill(lastKeeper, SIGTERM) < 0 && EGET() != ESRCH)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Couldn't kill cache house-keeping "
              << "process with pid '" << lastKeeper << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Couldn't kill cache house-keeping "
           << "process with pid '" << lastKeeper << "'.\n";
    }
  }

  //
  // Release local objects.
  //

  if (proxy != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Freeing up proxy.\n"
            << logofs_flush;
    #endif

    delete proxy;

    proxy = NULL;
  }

  if (context != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Freeing up context.\n"
            << logofs_flush;
    #endif

    delete context;

    context = NULL;
  }

  if (tcpFD != -1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Closing TCP listener.\n"
            << logofs_flush;
    #endif

    close(tcpFD);

    tcpFD = -1;
  }

  if (unixFD != -1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Closing UNIX listener.\n"
            << logofs_flush;
    #endif

    close(unixFD);

    unixFD = -1;
  }

  if (useUnixSocket == 1 && *unixSocketName != '\0')
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Going to remove the Unix domain socket '"
            << unixSocketName << "'.\n" << logofs_flush;
    #endif

    unlink(unixSocketName);
  }

  if (syncFD != -1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Closing sync listener.\n"
            << logofs_flush;
    #endif

    close(syncFD);

    syncFD = -1;
  }

  if (keybdFD != -1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Closing embedded keyboard listener.\n"
            << logofs_flush;
    #endif

    close(keybdFD);

    keybdFD = -1;
  }

  if (sambaFD != -1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Closing samba listener.\n"
            << logofs_flush;
    #endif

    close(sambaFD);

    sambaFD = -1;
  }

  if (mediaFD != -1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Closing multimedia listener.\n"
            << logofs_flush;
    #endif

    close(mediaFD);

    mediaFD = -1;
  }

  if (proxyFD != -1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Closing proxy FD.\n"
            << logofs_flush;
    #endif

    close(proxyFD);

    proxyFD = -1;
  }

  if (internalFD != -1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Closing internal FD.\n"
            << logofs_flush;
    #endif

    close(internalFD);

    internalFD = -1;
  }

  //
  // Release global objects.
  //

  if (statistics != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Freeing up statistics.\n"
            << logofs_flush;
    #endif

    delete statistics;

    statistics = NULL;
  }

  if (control != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Freeing up control.\n"
            << logofs_flush;
    #endif

    delete control;

    control = NULL;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Exiting from main process.\n"
          << logofs_flush;
  #endif

  if (logofs != NULL && logofs != &cerr &&
          *logFileName != '\0')
  {
    *logofs << flush;

    delete logofs;

    logofs = NULL;
  }

  if (statofs != NULL && statofs != &cerr &&
          *statFileName != '\0')
  {
    *statofs << flush;

    delete statofs;

    statofs = NULL;
  }

  if (errofs != NULL)
  {
    *errofs << flush;

    if (errofs == &cerr)
    {
      errofs = NULL;
    }
    else if (errsbuf != NULL)
    {
      cerr.rdbuf(errsbuf);

      errsbuf = NULL;

      delete errofs;
    }

    errofs = NULL;
  }

  exit(0);
}

void HandleAbort()
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  *logofs << flush;

  //
  // When running under the supervision of
  // a memory checker, is better to disable
  // production of core dumps.
  //

  if (control -> EnableCoreDumpOnAbort == 1)
  {
    cerr << "Error" << ": Generating a core file to help "
         << "investigations.\n";
  }

  if (lastSignal != SIGHUP)
  {
    cerr << "Error" << ": Please report this problem to "
         << "support personnel.\n";
  }
  else
  {
    lastSignal = 0;
  }

  if (control -> EnableCoreDumpOnAbort == 1)
  {
    cerr << flush;

    signal(SIGABRT, SIG_DFL);

    raise(SIGABRT);
  }
  else if (control -> EnableRestartOnFailure == 1)
  {
    //
    // Check if we have already entered the
    // cleanup phase. If not, jump back to
    // the main loop.
    //

    if (lastKill == 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Trying to revert to loop context.\n"
              << logofs_flush;
      #endif

      //
      // TODO: A close() in the reconnection
      // procedure seems to be not enough to
      // ensure that remote proxy identifies
      // the link shutdown.
      //

      shutdown(proxyFD, SHUT_RDWR);

      longjmp(context -> getLoopContext(), 1);
    }
  }

  exit(1);
}

void HandleAlert(int alert)
{
  if (lastAlert == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Requesting alert dialog with code "
            << alert << ".\n" logofs_flush;
    #endif

    lastAlert = alert;
  }
  #if defined(INFO) || defined(TEST)
  else
  {
    *logofs << "Loop: WARNING! Alert dialog already requested "
            << "with code " << alert << ".\n"
            << logofs_flush;
  }
  #endif

  return;
}

void EnableSignals()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Enabling signals in process with pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  //
  // Install signal handler used
  // to invoke statistics output.
  //

  signal(SIGUSR1, HandleSignal);
  signal(SIGUSR2, HandleSignal);

  //
  // Have a graceful shutdown.
  //

  signal(SIGHUP,  HandleSignal);
  signal(SIGINT,  HandleSignal);
  signal(SIGTERM, HandleSignal);

  signal(SIGPIPE, HandleSignal);

  //
  // Interrupt blocking operation
  // after a timeout.
  //

  signal(SIGALRM, HandleSignal);

  //
  // Get child signals generated
  // by dialogs and watchdogs.
  //

  signal(SIGCHLD, HandleSignal);

  //
  // Be sure we ignore signals we
  // use to send to agent.
  //

  signal(SIGURG,  SIG_IGN);

  #if defined(__APPLE__) || defined(__FreeBSD__)

  signal(SIGIO, SIG_IGN);

  #else

  signal(SIGPOLL, SIG_IGN);

  #endif

  //
  // We could have disabled signals for a
  // while. Check if any of our children
  // has exited.
  //

  HandleChildren();
}

void DisableSignals()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Disabling signals in process with pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  signal(SIGUSR1, SIG_IGN);
  signal(SIGUSR2, SIG_IGN);

  signal(SIGHUP, SIG_IGN);
  signal(SIGINT, SIG_IGN);
  signal(SIGTERM, SIG_IGN);

  signal(SIGPIPE, SIG_IGN);
  signal(SIGALRM, SIG_IGN);

  signal(SIGCHLD, SIG_IGN);
}

void HandleSignal(int signal)
{
  #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
  *logofs << "Loop: Signal handler called with signal '"
          << signal << "' in process with pid '" << getpid()
          << "'.\n" << logofs_flush;
  #endif

  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (signal == SIGUSR1)
  {
    #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
    *logofs << "Loop: Received signal 'SIGUSR1'.\n"
            << logofs_flush;
    #endif

    if (proxy != NULL && lastSignal == 0)
    {
      lastSignal = SIGUSR1;
    }
  }
  else if (signal == SIGUSR2)
  {
    #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
    *logofs << "Loop: Received signal 'SIGUSR2'.\n"
            << logofs_flush;
    #endif

    if (proxy != NULL && lastSignal == 0)
    {
      lastSignal = SIGUSR2;
    }
  }
  else if (signal == SIGHUP)
  {
    #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
    *logofs << "Loop: Received signal 'SIGHUP'.\n"
            << logofs_flush;
    #endif

    if (proxy != NULL)
    {
      lastSignal = SIGHUP;
    }
  }
  else if (signal == SIGPIPE)
  {
    #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
    *logofs << "Loop: Received signal 'SIGPIPE'.\n"
            << logofs_flush;
    #endif

    //
    // Force shutdown of internal NX descriptor.
    // This will cause a failure at the next time
    // descriptor is accessed so channel can be
    // closed cleanly.
    //

    //
    // TODO: Can probably happen that SIGPIPE is
    // delivered to proxy even in case some other
    // descriptor is unexpectedly closed.
    //
    // if (internalFD != -1)
    // {
    //   cerr << "Info" << ": Received signal 'SIGPIPE'. "
    //        << "Closing agent conection.\n";
    //
    //   shutdown(internalFD, SHUT_RDWR);
    // }
    //
  }
  else if (signal == SIGALRM)
  {
    //
    // Nothing to do. Just allow to wake up
    // the process on blocking operations.
    //

    #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
    *logofs << "Loop: Received signal 'SIGALRM'.\n"
            << logofs_flush;
    #endif
  }
  else if (signal == SIGCHLD)
  {
    //
    // Check if any of our children has exited.
    //

    #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
    *logofs << "Loop: Received signal 'SIGCHLD'.\n"
            << logofs_flush;
    #endif

    HandleChildren();

    //
    // Don't save this signal or it will override
    // any previous signal sent by child before
    // exiting.
    //
  }
  else
  {
    if (proxy == NULL || proxyFD < 0)
    {
      //
      // Exit immediately if yet there is no proxy
      // or signal was received while reconnecting.
      //

      #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
      *logofs << "Loop: Giving up immediately due to signal "
              << signal << ".\n" << logofs_flush;
      #endif

      HandleCleanup();
    }
    else
    {
      //
      // Register the signal so we can handle it
      // inside the main loop. We will probably
      // dispose any resource and exit.
      //

      #if defined(UNSAFE) && (defined(INFO) || defined(TEST))
      *logofs << "Loop: Registering end of session request "
              << "due to signal " << signal << ".\n"
              << logofs_flush;
      #endif

      lastSignal = signal;
    }
  }

  //
  // This is needed under Solaris because
  // raising a signal seems to reset the
  // previous handler.
  //

  #ifdef __sun

  EnableSignals();

  #endif
}

//
// Open TCP socket to listen for remote proxy and
// block until remote connects. If successful close
// the listening socket and return FD on which the
// other party is connected.
//

int WaitForRemote(int portNum)
{
  //
  // Get IP address of host to be awaited.
  //

  int acceptIPAddr = 0;

  if (*acceptHost != '\0')
  {
    acceptIPAddr = GetHostAddress(acceptHost);

    if (acceptIPAddr == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Cannot accept connections from unknown host '"
              << acceptHost << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Cannot accept connections from unknown host '"
           << acceptHost << "'.\n";

      HandleCleanup();
    }
  }

  int proxyFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (proxyFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(proxyFD) < 0)
  {
    HandleCleanup();
  }

  sockaddr_in tcpAddr;

  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(portNum);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(proxyFD, (sockaddr *) & tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for TCP port "
            << portNum << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for TCP port "
         << portNum << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(proxyFD, 4) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for TCP port "
            << portNum << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for TCP port "
         << portNum << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  char host[DEFAULT_STRING_LENGTH] = { 0 };

  if (*acceptHost != '\0')
  {
    strcat(host, "'");
    strcat(host, acceptHost);
    strcat(host, "'");
  }
  else
  {
    strcpy(host, "any host");
  }

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Waiting for connection from "
          << host  << " on port '" << portNum
          << "'.\n" << logofs_flush;
  #endif

  cerr << "Info" << ": Waiting for connection from "
       << host << " on port '" << portNum
       << "'.\n";

  //
  // How many times to loop waiting for connections
  // from the selected host? Each loop wait for at
  // most 20 seconds so a default value of 3 gives
  // a timeout of 1 minute.
  //

  int retryAccept = control -> OptionProxyRetryAccept;

  for (;;)
  {
    fd_set readSet;

    FD_ZERO(&readSet);
    FD_SET(proxyFD, &readSet);

    T_timestamp selectTs;

    selectTs.tv_sec  = 20;
    selectTs.tv_usec = 0;

    #if (defined(__hpux) || defined(hpux)) && !defined(PGID_USE_PID)

    int result = select(proxyFD + 1, (int *) &readSet, NULL, NULL, &selectTs);

    #else

    int result = select(proxyFD + 1, &readSet, NULL, NULL, &selectTs);

    #endif

    if (result == -1)
    {
      if (EGET() == EINTR)
      {
        continue;
      }

      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to select failed. Error is "
              << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Call to select failed. Error is "
           << EGET() << " '" << ESTR() << "'.\n";

      HandleCleanup();
    }
    else if (result > 0 && FD_ISSET(proxyFD, &readSet))
    {
      sockaddr_in newAddr;

      size_t addrLen = sizeof(sockaddr_in);

      int newFD = accept(proxyFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

      if (newFD == -1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Call to accept failed. Error is "
                << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Call to accept failed. Error is "
             << EGET() << " '" << ESTR() << "'.\n";

        HandleCleanup();
      }

      char *connectedHost = inet_ntoa(newAddr.sin_addr);
      unsigned int connectedPort = ntohs(newAddr.sin_port);

      if (*acceptHost == '\0' || (int) newAddr.sin_addr.s_addr == acceptIPAddr)
      {
        #if defined(INFO) || defined (TEST)
        *logofs << "Loop: Accepted connection from '" << connectedHost
                << "' on port '" << connectedPort << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Info" << ": Accepted connection from '" << connectedHost
             << "' on port '" << connectedPort << "'.\n";

        //
        // Get rid of listening NX port.
        //

        close(proxyFD);

        return newFD;
      }
      else
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Refusing connection from '" << connectedHost
                << "' on port '" << connectedPort << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Refusing connection from '" << connectedHost
             << "' on port '" << connectedPort << "'.\n";
      }

      //
      // Not the best way to elude a DOS attack...
      //

      sleep(5);

      close(newFD);
    }

    if (--retryAccept == 0)
    {
      if (*acceptHost == '\0')
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Connection with remote host "
                << "could not be established.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Connection with remote host "
             << "could not be established.\n";
      }
      else
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Connection with remote host '"
                << acceptHost << "' could not be established.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Connection with remote host '"
             << acceptHost << "' could not be established.\n";
      }

      close(proxyFD);

      HandleCleanup();
    }
    else
    {
      handleCheckSessionInConnect();
    }
  }
}

//
// Connect to remote proxy.  If successful
// return FD of connection, else return -1.
//

int ConnectToRemote(const char *const remoteHost, int portNum)
{
  int remoteProxyFD;

  int remoteIPAddr = GetHostAddress(remoteHost);

  if (remoteIPAddr == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Unknown remote host '"
            << remoteHost << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Unknown remote host '"
         << remoteHost << "'.\n";

    HandleCleanup();
  }

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Connecting to remote host '" 
          << remoteHost << ":" << portNum << "'.\n"
          << logofs_flush;
  #endif

  cerr << "Info" << ": Connecting to remote host '"
       << remoteHost << ":" << portNum << "'.\n"
       << logofs_flush;

  //
  // How many times we retry to connect to remote
  // host in case of failure?
  //

  int retryConnect = control -> OptionProxyRetryConnect;
  int retryAlert   = control -> OptionProxyRetryConnect - 4;

  sockaddr_in addr;
  addr.sin_family = AF_INET;
  addr.sin_port = htons(portNum);
  addr.sin_addr.s_addr = remoteIPAddr;

  for (;;)
  {
    remoteProxyFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

    if (remoteProxyFD == -1)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to socket failed. Error is "
              << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Call to socket failed. Error is "
           << EGET() << " '" << ESTR() << "'.\n";

      HandleCleanup();
    }
    else if (SetReuseAddress(remoteProxyFD) < 0)
    {
      HandleCleanup();
    }

    //
    // Ensure operation is timed out
    // if there is a network problem.
    //

    alarm(retryConnect * 3 / 2);

    int result = connect(remoteProxyFD, (sockaddr *) &addr, sizeof(sockaddr_in));

    retryConnect -= (((retryConnect * 3 / 2) - alarm(0)) / 3);

    if (result < 0)
    {
      #if defined(INFO) || defined (TEST)
      *logofs << "Loop: Connection to '" << remoteHost
              << ":" << portNum << "' failed with error '"
              << ESTR() << "'. Retrying.\n"
              << logofs_flush;
      #endif

      close(remoteProxyFD);

      if (--retryConnect <= 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Connection to '" << remoteHost
                << ":" << portNum << "' failed. Error is "
                << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Connection to '" << remoteHost
             << ":" << portNum << "' failed. Error is "
             << EGET() << " '" << ESTR() << "'.\n";

        HandleCleanup();
      }
      else if (retryConnect < retryAlert && retryConnect % 4 == 0)
      {
        //
        // Wait some time before showing the
        // dialog and don't show it too often.
        //

        handleCheckSessionInConnect();
      }
    }
    else
    {
      break;
    }

    sleep(3);
  }

  return remoteProxyFD;
}

int ConnectToRemoteWithSSH(const char *const remoteHost, int portNum)
{
  int remoteProxyFD;

  char header[DEFAULT_STRING_LENGTH];

  int localIPAddr = (int) inet_addr("127.0.0.1");

  int remoteIPAddr = GetHostAddress(remoteHost);

  if (remoteIPAddr == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Unknown remote SSL host '"
            << remoteHost << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Unknown remote SSL host '"
         << remoteHost << "'.\n";

    HandleCleanup();
  }

  sockaddr_in addr;
  addr.sin_family = AF_INET;
  addr.sin_port = htons(sshPort);
  addr.sin_addr.s_addr = localIPAddr;

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Connecting to remote host '"
          << remoteHost << ":" << portNum << "' on SSH port '"
          << "localhost:" << sshPort << "'.\n" << logofs_flush;
  #endif

  cerr << "Info" << ": Connecting to remote host '"
       << remoteHost << ":" << portNum << "' on SSH port '"
       << "localhost:" << sshPort << "'.\n";

  //
  // How many times we retry to connect to remote
  // host in case of failure?
  //

  int retryConnect = control -> OptionProxyRetryConnect;
  int retryAlert   = control -> OptionProxyRetryConnect - 4;

  //
  // Connect to the SSH tunnel first and then
  // request a connection to the remote proxy
  // through the SOCKS interface.
  //

  for(;;)
  {
    for (;;)
    {
      remoteProxyFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

      if (remoteProxyFD == -1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Call to socket failed. Error is "
                << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Call to socket failed. Error is "
             << EGET() << " '" << ESTR() << "'.\n";

        HandleCleanup();
      }
      else if (SetReuseAddress(remoteProxyFD) < 0)
      {
        HandleCleanup();
      }

      //
      // Ensure operation is timed out
      // if there is a network problem.
      //

      alarm(retryConnect * 3 / 2);

      int result = connect(remoteProxyFD, (sockaddr *) &addr, sizeof(sockaddr_in));

      retryConnect -= (((retryConnect * 3 / 2) - alarm(0)) / 3);

      if (result < 0)
      {
        #if defined(INFO) || defined (TEST)
        *logofs << "Loop: Connection to '" << remoteHost
                << ":" << portNum << "' over SSH failed with error '"
                << ESTR() << "'. Retrying.\n" << logofs_flush;
        #endif

        close(remoteProxyFD);

        if (--retryConnect <= 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Connection to '" << remoteHost
                  << ":" << portNum << "' over SSH failed. Error is "
                  << EGET() << " '" << ESTR() << "'.\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Connection to '" << remoteHost
               << ":" << portNum << "' over SSH failed. Error is "
               << EGET() << " '" << ESTR() << "'.\n";

          HandleCleanup();
        }
        else if (retryConnect < retryAlert && retryConnect % 4 == 0)
        {
          //
          // Wait some time before showing the
          // dialog and don't show it too often.
          //

          handleCheckSessionInConnect();
        }
      }
      else
      {
        break;
      }

      sleep(3);
    }

    header[0] = 4;
    header[1] = 1;

    PutUINT(portNum, (unsigned char *) header + 2, 1);

    memcpy(header + 4, &addr.sin_addr, sizeof(addr.sin_addr));
    memcpy(header + 4 + sizeof(addr.sin_addr), "NXPROXY\0", 8);

    send(remoteProxyFD, header, 4 + sizeof(addr.sin_addr) + 8, 0);

    if ((recv(remoteProxyFD, header, 8, 0) == 0) || (--retryConnect == 0))
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Connection to '" << remoteHost
              << ":" << portNum << "' over SSH failed."
               << "\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Connection to '" << remoteHost
           << ":" << portNum << "' over SSH failed."
           << "\n";

      close(remoteProxyFD);

      HandleCleanup();
    }
  
    if ((int) header[1] == 90)
    {
      break;
    }

    close(remoteProxyFD);

    sleep(3);
  }

  return remoteProxyFD;
}

//
// Make a string of options for the remote
// proxy and write it to the descriptor.
//

int SendRemoteOptions(int fd)
{
  char options[DEFAULT_REMOTE_OPTIONS_LENGTH];

  //
  // TODO: Due to a stupid bug in version 1.2.2,
  // we can't send our real version to the remote
  // peer or otherwise it would end up adopting
  // protocol step 2. We add the real version
  // after the 1.2.2 identifier. This data will
  // be ignored by a 'real' 1.2.2.
  //
  // sprintf(options, "NXPROXY-%i.%i.%i", control -> LocalVersionMajor,
  //             control -> LocalVersionMinor, control -> LocalVersionPatch);
  //

  sprintf(options, "NXPROXY-1.2.2-%i.%i.%i", control -> LocalVersionMajor,
              control -> LocalVersionMinor, control -> LocalVersionPatch);

  //
  // If you want to send options from proxy
  // initiating connection use something like
  // this:
  //
  // if (WE_INITIATE_CONNECTION)
  // {
  //   sprintf(options + strlen(options), "%s=%s", option, value);
  // }
  //
  // If you want to send options according to
  // local proxy mode use something like this:
  //
  // if (control -> ProxyMode == PROXY_CLIENT)
  // {
  //   sprintf(options + strlen(options), "%s=%s", option, value);
  // }
  //

  //
  // Send the authorization cookie if any. We assume
  // user can choose to not provide any auth cookie
  // and allow any connection to be accepted.
  //

  if (WE_INITIATE_CONNECTION &&
          (authCookie != NULL && *authCookie != '\0'))
  {
    sprintf(options + strlen(options), " cookie=%s,", authCookie);
  }
  else
  {
    sprintf(options + strlen(options), " ");
  }

  //
  // Now link chracteristics and compression
  // options. Delta compression, as well as
  // preferred pack method, are imposed by
  // client proxy.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    //
    // If protocol level is not 3 we don't
    // have support for PNG decoding at the
    // remote side.
    //

    if (control -> isProtoStep3() == 0 &&
            packMethod >= PACK_PNG_JPEG_8_COLORS &&
                packMethod <= PACK_PNG_JPEG_16M_COLORS)
    {
      //
      // Arbitrarily assume that corresponding
      // JPEG method is at this offset.
      //

      packMethod = packMethod - 21;

      ParsePackMethod(packMethod, packQuality);

      #ifdef WARNING
      *logofs << "Loop: WARNING! Method PNG unsupported by remote. "
              << "Setting preferred method '" << packMethodName
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Method PNG unsupported by remote. "
           << "Setting preferred method '" << packMethodName
           << "'.\n";
    }

    sprintf(options + strlen(options), "link=%s,pack=%s,cache=%s,",
                linkSpeedName, packMethodName, cacheSizeName);

    if (*bitrateLimitName != '\0')
    {
      sprintf(options + strlen(options), "limit=%s,",
                  bitrateLimitName);
    }

    //
    // Starting from protocol level 2 client side
    // suggests the flush timeout to be used by the
    // remote proxy. We know which protocol level
    // is the remote as at client side we are the
    // first to get the remote options.
    //

    if (control -> isProtoStep2() == 1)
    {
      if (useFlush != -1)
      {
        sprintf(options + strlen(options), "flush=%d,", useFlush);
      }
      else
      {
        sprintf(options + strlen(options), "flush=%d,",
                    control -> FlushTimeout);
      }
    }

    //
    // Old proxy versions assume that agent
    // doesn't support the render extension.
    //

    if (control -> SessionMode == SESSION_X &&
            control -> isProtoStep4() == 0)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Not using render extension "
              << "due to lack of alpha support.\n"
              << logofs_flush;
      #endif

      control -> PersistentCacheLoadRender = 0;

      control -> AgentHideRender = 1;
    }

    //
    // Starting from protocol level 3 we let user
    // disable render extension or instruct client
    // proxy to short-circuit simple replies. We
    // also pass the session type to be sure the
    // remote side gets the right value.
    //

    if (control -> isProtoStep3() == 1)
    {
      sprintf(options + strlen(options), "render=%d,taint=%d,",
                  (control -> AgentHideRender == 0),
                      control -> AgentTaintReplies);

      if (*sessionType != '\0')
      {
        sprintf(options + strlen(options), "type=%s,", sessionType);
      }
      else
      {
        sprintf(options + strlen(options), "type=default,");
      }
    }

    //
    // Send image cache parameters.
    //

    if (control -> isProtoStep3() == 1)
    {
      sprintf(options + strlen(options), "images=%s,", imagesSizeName);
    }
    else
    {
      control -> ImageCacheEnableLoad = 0;
      control -> ImageCacheEnableSave = 0;
    }

    sprintf(options + strlen(options), "delta=%d,stream=%d,data=%d ",
                control -> LocalDeltaCompression,
                    control -> LocalStreamCompressionLevel,
                        control -> LocalDataCompressionLevel);
  }
  else
  {
    //
    // If no special compression level was selected,
    // server side will use compression levels set
    // by client.
    //

    if (control -> LocalStreamCompressionLevel < 0)
    {
      sprintf(options + strlen(options), "stream=default,");
    }
    else
    {
      sprintf(options + strlen(options), "stream=%d,",
                  control -> LocalStreamCompressionLevel);
    }

    if (control -> LocalDataCompressionLevel < 0)
    {
      sprintf(options + strlen(options), "data=default ");
    }
    else
    {
      sprintf(options + strlen(options), "data=%d ",
                  control -> LocalDataCompressionLevel);
    }
  }

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Sending remote options '"
          << options << "'.\n" << logofs_flush;
  #endif

  return (WriteRemoteOptions(fd, options, strlen(options)));
}

int NegotiateProxyOptions(int fd)
{
  //
  // Read until the first space in string.
  // We expect the remote version number.
  //

  char options[DEFAULT_REMOTE_OPTIONS_LENGTH];

  if (ReadRemoteOptions(fd, options, sizeof(options), ' ') < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Remote NX proxy closed the connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Remote NX proxy closed the connection.\n";

    cerr << "Error" << ": Please check your authorization credentials.\n";

    return -1;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Received remote option string '"
          << options << "' from FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  if (strncmp(options, "NXPROXY-", strlen("NXPROXY-")) != 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Parse error in remote options string '"
            << options << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Parse error in remote options string '"
         << options << ".\n";

    return -1;
  }

  //
  // TODO: Due to the same stupid bug, we must
  // read the remote version in a special way.
  //

  int major = -1;
  int minor = -1;
  int patch = -1;

  sscanf(options, "NXPROXY-%i.%i.%i-%i.%i.%i", &(control -> RemoteVersionMajor),
             &(control -> RemoteVersionMinor), &(control -> RemoteVersionPatch),
                 &major, &minor, &patch);

  if (control -> RemoteVersionMajor == 1 &&
          control -> RemoteVersionMinor == 2 &&
              control -> RemoteVersionPatch == 2 &&
                  major != -1 && minor != -1 && patch != -1)
  {
    control -> RemoteVersionMajor = major;
    control -> RemoteVersionMinor = minor;
    control -> RemoteVersionPatch = patch;

    #ifdef TEST
    *logofs << "Loop: Read trailing remote version '" << major
            << "." << minor << "." << patch << "'.\n"
            << logofs_flush;
    #endif

    #ifdef TEST
    *logofs << "Loop: Assuming remote version '" << control -> RemoteVersionMajor
            << "." << control -> RemoteVersionMinor << "." << control -> RemoteVersionPatch
            << "'.\n" << logofs_flush;
    #endif
  }

  if (SetVersion() < 0)
  {
    if (control -> ProxyMode == PROXY_SERVER)
    {
      HandleAlert(WRONG_PROXY_VERSION_ALERT);
    }

    handleAlertInLoop();

    #ifdef PANIC
    *logofs << "Loop: PANIC! Remote NXPROXY version number "
            << control -> RemoteVersionMajor << "." << control -> RemoteVersionMinor
            << "." << control -> RemoteVersionPatch << " doesn't match local version "
            << control -> LocalVersionMajor << "." << control -> LocalVersionMinor
            << "." << control -> LocalVersionPatch << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Remote NXPROXY version number "
         << control -> RemoteVersionMajor << "." << control -> RemoteVersionMinor
         << "." << control -> RemoteVersionPatch << " doesn't match local version "
         << control -> LocalVersionMajor << "." << control -> LocalVersionMinor
         << "." << control -> LocalVersionPatch << ".\n";

    return -1;
  }
  else if (control -> RemoteVersionMinor != control -> LocalVersionMinor ||
               control -> RemoteVersionPatch != control -> LocalVersionPatch)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Connected to remote NXPROXY version "
            << control -> RemoteVersionMajor << "." << control -> RemoteVersionMinor
            << "." << control -> RemoteVersionPatch << " with local version "
            << control -> LocalVersionMajor << "." << control -> LocalVersionMinor
            << "." << control -> LocalVersionPatch << ".\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Connected to remote NXPROXY version "
         << control -> RemoteVersionMajor << "." << control -> RemoteVersionMinor
         << "." << control -> RemoteVersionPatch << " with local version "
         << control -> LocalVersionMajor << "." << control -> LocalVersionMinor
         << "." << control -> LocalVersionPatch << ".\n" << logofs_flush;

    if (control -> RemoteVersionMinor > control -> LocalVersionMinor ||
            (control -> RemoteVersionMinor == control -> LocalVersionMinor &&
                 control -> RemoteVersionPatch > control -> LocalVersionPatch))
    {
      cerr << "Warning" << ": Consider checking http://www.nomachine.com/ for updates.\n";
    }
  }

  //
  // Get the remote options, delimited by a space character.
  // Note that there will be a further initialization phase
  // at the time proxies negotiate cache file to restore.
  //

  if (ReadRemoteOptions(fd, options, sizeof(options), ' ') < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Remote NX proxy closed the connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Remote NX proxy closed the connection.\n";

    cerr << "Error" << ": Please check your authorization credentials.\n";

    return -1;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Received remote options string '"
          << options << "' from FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  if (ParseRemoteOptions(options) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Couldn't negotiate a valid "
            << "session with remote NX proxy.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Couldn't negotiate a valid "
         << "session with remote NX proxy.\n";

    return -1;
  }

  return 1;
}

int ReadRemoteOptions(int fd, char *buffer, int size, char stop)
{
  int position = 0;

  while (position < (size - 1))
  {
    int result = read(fd, buffer + position, 1);

    if (result <= 0)
    {
      if (result < 0 && EGET() == EINTR)
      {
        continue;
      }

      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Error reading data from FD#"
              << fd << ".\n" << logofs_flush;
      #endif

      return -1;
    }
    else if (*(buffer + position) == stop)
    {
      #ifdef TEST
      *logofs << "Loop: Read stop character from FD#"
              << fd << ".\n" << logofs_flush;
      #endif

      *(buffer + position + 1) = '\0';

      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Remote option string '" << buffer
              << "' read from FD#" << fd << ".\n"
              << logofs_flush;
      #endif

      return position;
    }
    else
    {
      //
      // Make sure string received
      // from far end is printable.
      //

      if (isgraph(*(buffer + position)) == 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Non printable character decimal '"
                << (unsigned int) *(buffer + position)
                << "' received in remote options from FD#"
                << fd << ".\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Non printable character decimal '"
                << (unsigned int) *(buffer + position)
                << "' received in remote options from FD#"
                << fd << ".\n" << logofs_flush;

        *(buffer + position) = ' ';
      }

      position++;
    }
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Stop character missing from FD#"
          << fd << " after " << position 
          << " characters read in string " << buffer
          << ".\n" << logofs_flush;
  #endif

  *(buffer + position) = '\0';

  return -1;
}

int WriteRemoteOptions(int fd, const char *buffer, int size)
{
  int position = 0;

  while (position < size)
  {
    int result = write(fd, buffer + position, size - position);

    if (result <= 0)
    {
      if (result < 0 && EGET() == EINTR)
      {
        continue;
      }

      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Error writing data from FD#"
              << fd << ".\n" << logofs_flush;
      #endif

      return -1;
    }

    position += result;
  }

  return position;
}

static void StartDaemon(unsigned int xPort)
{
  #ifdef __EMX__

  #ifdef PANIC
  *logofs << "Loop: PANIC! Daemon option disabled under OS/2.\n"
          << logofs_flush;
  #endif

  cerr << "Error" << ": Daemon option disabled under OS/2.\n";

  HandleCleanup();

  #else

  switch (fork())
  {
    case -1:
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to fork failed. Error is "
              << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Call to fork failed. Error is "
           << EGET() << " '" << ESTR() << "'.\n";

      HandleCleanup();
    }
    case 0:
    {
      //
      // Continue with the rest.
      //

      break;
    }
    default:
    {
      HandleCleanup();
    }
  }

  pid_t pid;

  if ((pid = setsid()) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to setsid failed. Error is "
            << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to setsid failed. Error is "
         << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  MakeLockFileName(lockFileName, xPort);

  //
  // First we try to open the file to see
  // if nxproxy is already running.
  //

  FILE *pidfile = fopen(lockFileName, "r");

  if (pidfile)
  {
    //
    // The open was successful so we
    // try and read a pid out of it.
    //

    char oldpid[10];

    switch (fread(oldpid, 1, sizeof(oldpid), pidfile))
    {
      case 0:
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Found empty pidfile " << lockFileName
                << ". Overriding.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Found empty pidfile " << lockFileName
             << ". Overriding.\n";

        break;
      }
      case -1:
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Error reading old pidfile " << lockFileName
                << ". Overriding.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Error reading old pidfile " << lockFileName
             << ". Overriding.\n";

        break;
      }
      default:
      {
        //
        // Do a sanity check on data read from pidfile.
        //

        if (!isdigit((int) ((unsigned char) oldpid[0])))
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Invalid data in pidfile " << lockFileName
                  << ". Aborting.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Invalid data in pidfile " << lockFileName
               << ". Aborting.\n";

          fclose(pidfile);

          HandleCleanup();
        }

        long oldpidval = atoi(oldpid);

        int override = 0;

        switch (kill(oldpidval, 0))
        {
          case ESRCH:
          case EPERM:
          {
            //
            // Either the pid doesn't exist or is owned
            // by someone else.  It's probably safe to
            // override.
            //

            #ifdef PANIC
            *logofs << "Loop: PANIC! Stale pidfile " << lockFileName
                    << ". Overriding.\n" << logofs_flush;
            #endif

            cerr << "Error" << ": Stale pidfile " << lockFileName
                 << ". Overriding.\n";

            override = 1;

            break;
          }
          default:
          {
            #ifdef PANIC
            *logofs << "Loop: PANIC! It appears another nxproxy is running at pid "
                    << oldpidval << ".\n" << logofs_flush;
            #endif

            cerr << "Error" << ": It appears another nxproxy is running at pid "
                 << oldpidval << ".\n";

            #ifdef PANIC
            *logofs << "Loop: PANIC! If this isn't correct then delete the file "
                    << lockFileName << ".\n" << logofs_flush;
            #endif

            cerr << "Error" << ": If this isn't correct then delete the file "
                 << lockFileName << ".\n";
          }
        }

        if (override) break;

        fclose(pidfile);

        //
        // Don't delete the lockfile.
        //

        useDaemonMode = 0;

        HandleCleanup();
      }
    }

    fclose(pidfile);
  }

  if (!(pidfile = fopen(lockFileName, "w")))
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't open pid file " << lockFileName
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't open pid file " << lockFileName
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  fprintf(pidfile, "%d", (int) pid);

  fclose(pidfile);

  #endif

  return;
}

void KillDaemon(unsigned int xPort)
{
  char lockFileName[DEFAULT_STRING_LENGTH];

  MakeLockFileName(lockFileName, xPort);

  FILE *pidfile = fopen(lockFileName, "r");

  if (pidfile)
  {
    //
    // The open was successful so we try
    // to read a pid out of it.
    //

    char pid[10];

    switch (fread(pid, 1, sizeof(pid), pidfile))
    {
    case 0:
    case -1:

      #ifdef PANIC
      *logofs << "Loop: PANIC! Error reading pid from " << lockFileName << ".\n"
              << logofs_flush;

      *logofs << "Loop: PANIC! You will have to manually kill the daemon.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Error reading pid from " << lockFileName << ".\n";

      cerr << "Error" << ": You will have to manually kill the daemon.\n";

      break;

    default:

      //
      // Sanity check on data in pidfile.
      //

      fclose(pidfile);

      if (!isdigit((int) ((unsigned char) pid[0])))
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Invalid data in pidfile " << lockFileName
                 << ". Aborting.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Invalid data in pidfile " << lockFileName
             << ". Aborting.\n";

        HandleCleanup();
      }

      long pidval = atoi(pid);

      #ifdef WARNING
      *logofs << "Loop: WARNING! Killing nxproxy at pid " << pidval << ".\n"
              << logofs_flush;
      #endif

      if (kill(pidval, SIGTERM) == -1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't kill nxproxy at pid " << pidval
                 << ". Leaving pidfile intact and aborting.\n"
                 << logofs_flush;
        #endif

        cerr << "Error" << ": Can't kill proxy at pid " << pidval
             << ". Leaving pidfile intact and aborting.\n";

        HandleCleanup();
      }
    }

    HandleCleanup();
  }
  else
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't find any proxy daemon running.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't find any proxy daemon running.\n";

    HandleCleanup();
  }
}

static void MakeLockFileName(char* lockFileName, unsigned int xPort)
{
  char *homedir = getenv("HOME");

  if (!homedir)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! No environment variable HOME.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": No environment variable HOME.\n";

    HandleCleanup();
  }

  strcpy(lockFileName, homedir);
  strcat(lockFileName, "/");
  strcat(lockFileName, LOCK_FILE_NAME);

  struct utsname unameInfo;

  if (uname(&unameInfo) == -1)
  {
    unameInfo.nodename[0] = 0;
  }
  else
  {
    char* dotptr = strchr( unameInfo.nodename, '.' );

    if (dotptr)
    {
      *dotptr = 0;
    }

    strcat(lockFileName, "-");
    strcat(lockFileName, unameInfo.nodename);
  }

  //
  // Get the broken out fields
  // of a line from /etc/passwd.
  //

  struct passwd* passdata;

  if ((passdata = getpwuid(getuid())) != 0)
  {
    strcat(lockFileName, "-");
    strcat(lockFileName, passdata -> pw_name);
  }
  else
  {
    strcat(lockFileName, "");
  }

  sprintf(lockFileName + strlen(lockFileName), "-%u", xPort);
}

//
// Parse the string passed by calling process in
// the environment. This is not necessarily the
// content of DISPLAY variable, but can be the
// parameters passed when creating the process
// or thread.
//

int ParseEnvironmentOptions(const char *env, int force)
{
  //
  // Execute only once unless force is set.
  //

  static int parsed;

  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Be sure we have a parameters repository
  //

  if (control == NULL)
  {
    control = new Control();
  }

  if (force == 0 && parsed == 1)
  {
    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Skipping a further parse of environment "
            << "options string '" << (env != NULL ? env : "")
            << "'.\n" << logofs_flush;
    #endif

    return 1;
  }

  if (env == NULL || *env == '\0')
  {
    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Nothing to do with empty environment "
            << "options string '" <<  (env != NULL ? env : "")
            << "'.\n" << logofs_flush;
    #endif

    return 0;
  }
  else if (force == 1 && ParseHostOption(env) > 0)
  {
    return 1;
  }

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Going to parse environment options string '"
          << env << "'.\n" << logofs_flush;
  #endif

  parsed = 1;

  //
  // Copy the string passed as parameter
  // because we need to modify it.
  //

  char opts[DEFAULT_DISPLAY_OPTIONS_LENGTH];

  #ifdef VALGRIND

  memset(opts, '\0', DEFAULT_DISPLAY_OPTIONS_LENGTH);

  #endif

  if (strlen(env) >= DEFAULT_DISPLAY_OPTIONS_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Environment options string '" << env
            << "' exceeds length of " << DEFAULT_DISPLAY_OPTIONS_LENGTH
            << " characters.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Environment options string '" << env
         << "' exceeds length of " << DEFAULT_DISPLAY_OPTIONS_LENGTH
         << " characters.\n";

    return -1;
  }

  strcpy(opts, env);

  char *nextOpts = opts;

  //
  // Ensure that DISPLAY environment variable
  // (roughly) follows the X convention for
  // transport notation.
  //

  if (strncasecmp(opts, "nx/nx,:", 7) == 0 ||
          strncasecmp(opts, "nx,:", 4) == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Parse error in options string '"
            << opts << "' at 'nx,:'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Parse error in options string '"
         << opts << "' at 'nx,:'.\n";

    return -1;
  }
  else if (strncasecmp(opts, "nx/nx,", 6) == 0)
  {
    nextOpts += 6;
  }
  else if (strncasecmp(opts, "nx,", 3) == 0)
  {
    nextOpts += 3;
  }
  else if (strncasecmp(opts, "nx", 2) == 0)
  {
    nextOpts += 2;
  }
  else if (!force)
  {
    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: WARNING! Ignoring host X server display string '"
            << opts << "'.\n" << logofs_flush;
    #endif

    return 0;
  }

  //
  // Save here and parse later
  // user specified option file.
  //

  char fileOptions[DEFAULT_STRING_LENGTH] = { 0 };

  //
  // The options string is intended to be a series
  // of name/value tuples in the form name=value
  // separated by the ',' character ended by a ':'
  // followed by remote NX proxy port.
  //

  char *name;
  char *value;

  value = rindex(nextOpts, ':');

  if (value != NULL)
  {
    char *check = value + 1;

    if (*check == '\0' || isdigit(*check) == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't identify NX port in string '"
              << value << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Can't identify NX port in string '"
           << value << "'.\n";

      return -1;
    }

    proxyPort = atoi(check);

    //
    // Get rid of the port specification.
    //

    *value = '\0';
  }
  else if (proxyPort == DEFAULT_NX_PROXY_PORT && force == 0)
  {
    //
    // Complain only if user didn't specify
    // the port on the command line.
    //

    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't identify NX port in string '"
            << opts << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't identify NX port in string '"
         << opts << "'.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Loop: Parsing options string '"
          << nextOpts << "'.\n" << logofs_flush;
  #endif

  //
  // Now all the other optional parameters.
  //

  name = strtok(nextOpts, "=");

  while (name)
  {
    value = strtok(NULL, ",");

    if (CheckArg("environment", name, value) < 0)
    {
      return -1;
    }

    if (strcasecmp(name, "options") == 0)
    {
      strncpy(fileOptions, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "display") == 0)
    {
      strncpy(displayHost, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "link") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'link' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring option 'link' with value '"
             << value << "' at X server side.\n";
      }
      else if (ParseLinkOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify 'link' option in string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify 'link' option in string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "limit") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'limit' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring option 'limit' with value '"
             << value << "' at X server side.\n";
      }
      else if (ParseLimitOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify 'limit' option in string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify 'limit' option in string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "session") == 0)
    {
      //
      // Name of session as stored in NX client's
      // configuration. Ignored by NX proxy but
      // checked against blanks.
      //

      char *blank = value;

      while (*blank != '\0' && isspace(*blank++) == 0);

      if (*blank != '\0')
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Option 'session' should not contain white spaces.\n"
                << logofs_flush;
        #endif

        cerr << "Warning" << ": Option 'session' should not contain white spaces.\n";
      }
    }
    else if (strcasecmp(name, "type") == 0)
    {
      //
      // Type of session, for example "windows" or
      // "unix-kde" or "unix-application", etc.
      //
      // TODO: In protocol level 3 this became a client
      // side option. We must be able to handle it when
      // passed at server side for compatibility with
      // proto 2.
      //

      if (strcasecmp(value, "default") == 0)
      {
        *sessionType = '\0';
      }
      else
      {
        strncpy(sessionType, value, DEFAULT_STRING_LENGTH - 1);
      }
    }
    else if (strcasecmp(name, "connect") == 0)
    {
      strncpy(remoteHost, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "retry") == 0)
    {
      if (atoi(value) > 0)
      {
        control -> OptionProxyRetryConnect  = atoi(value);
        control -> OptionServerRetryConnect = atoi(value);
      }
      else
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Invalid value '" << value
                << "' for option '" << name << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Invalid value '" << value
             << "' for option '" << name << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "log") == 0)
    {
      strncpy(logFileName, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "root") == 0)
    {
      strncpy(rootDir, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "id") == 0)
    {
      strncpy(sessionId, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "stat") == 0)
    {
      control -> CollectStatistics = 1;

      strncpy(statFileName, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "cookie") == 0)
    {
      strncpy(authCookie, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "accept") == 0)
    {
      strncpy(acceptHost, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "port") == 0)
    {
      proxyPort = atoi(value);

      if (xPort == DEFAULT_NX_X_PORT)
      {
        xPort = proxyPort;
      }
    }
    else if (strcasecmp(name, "nodelay") == 0)
    {
      useNoDelay = (atoi(value) > 0);
    }
    else if (strcasecmp(name, "flush") == 0)
    {
      useFlush = atoi(value);
    }
    else if (strcasecmp(name, "render") == 0)
    {
      //
      // TODO: In protocol level 3 this became a client
      // side option. We must be able to handle it when
      // passed at server side for compatibility with
      // proto 2.
      //

      useRender = atoi(value);
    }
    else if (strcasecmp(name, "taint") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'taint' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring option 'taint' with value '"
             << value << "' at X server side.\n";
      }
      else
      {
        useTaint = atoi(value);
      }
    }
    else if (strcasecmp(name, "delta") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'delta' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring option 'delta' with value '"
             << value << "' at X server side.\n";
      }
      else
      {
        control -> LocalDeltaCompression = (atoi(value) > 0);
      }
    }
    else if (strcasecmp(name, "data") == 0)
    {
      control -> LocalDataCompressionLevel = atoi(value);

      if (control -> LocalDataCompressionLevel == 0)
      {
        control -> LocalDataCompression = 0;
      }
      else
      {
        control -> LocalDataCompression = 1;
      }
    }
    else if (strcasecmp(name, "stream") == 0)
    {
      control -> LocalStreamCompressionLevel = atoi(value);

      if (control -> LocalStreamCompressionLevel == 0)
      {
        control -> LocalStreamCompression = 0;
      }
      else
      {
        control -> LocalStreamCompression = 1;
      }
    }
    else if (strcasecmp(name, "memory") == 0)
    {
      control -> LocalMemoryLevel = atoi(value);
    }
    else if (strcasecmp(name, "cache") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'cache' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else if (ParseCacheOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify cache size for string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify cache size for string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "images") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'images' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else if (ParseImagesOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify images cache size for string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify images cache size for string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "shmem") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'shmem' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else if (ParseShmemOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify shmem size for string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify shmem size for string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "load") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'load' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else
      {
        control -> PersistentCacheEnableLoad = atoi(value);

        if (control -> PersistentCacheEnableLoad > 0)
        {
          control -> PersistentCacheEnableLoad = 1;
        }
        else
        {
          if (control -> PersistentCacheName != NULL)
          {
            delete [] control -> PersistentCacheName;
          }

          control -> PersistentCacheName = NULL;

          control -> PersistentCacheEnableLoad = 0;
        }
      }
    }
    else if (strcasecmp(name, "save") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'save' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else
      {
        control -> PersistentCacheEnableSave = atoi(value);

        if (control -> PersistentCacheEnableSave > 0)
        {
          control -> PersistentCacheEnableSave = 1;
        }
        else
        {
          if (control -> PersistentCacheName != NULL)
          {
            delete [] control -> PersistentCacheName;
          }

          control -> PersistentCacheName = NULL;

          control -> PersistentCacheEnableSave = 0;
        }
      }
    }
    else if (strcasecmp(name, "sync") == 0)
    {
      syncPort = atoi(value);
    }
    else if (strcasecmp(name, "keybd") == 0)
    {
      keybdPort = atoi(value);
    }
    else if (strcasecmp(name, "samba") == 0)
    {
      sambaPort = atoi(value);
    }
    else if (strcasecmp(name, "media") == 0)
    {
      mediaPort = atoi(value);
    }
    else if (strcasecmp(name, "timeout") == 0)
    {
      int timeout = atoi(value);

      if (timeout == 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Disabling timeout on broken proxy connection.\n"
                << logofs_flush;
        #endif

        control -> ProxyTimeout = 0;
      }
      else if ((timeout * 1000) >= control -> PingTimeout)
      {
        control -> ProxyTimeout = timeout * 1000;
      }
      else
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'timeout' with "
                << "invalid value '" << value << "'.\n"
                << logofs_flush;
        #endif
      }
    }
    else if (strcasecmp(name, "cleanup") == 0)
    {
      int cleanup = atoi(value);

      if (cleanup == 0)
      {
        control -> CleanupTimeout = 0;
      }
      else if (cleanup > 0)
      {
        control -> CleanupTimeout = cleanup * 1000;
      }
      else
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'cleanup' with "
                << "invalid value '" << value << "'.\n"
                << logofs_flush;
        #endif
      }
    }
    else if (strcasecmp(name, "pack") == 0)
    {
      if (control -> ProxyMode == PROXY_SERVER)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'pack' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else if (ParsePackOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify pack method for string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify pack method for string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "core") == 0)
    {
      control -> EnableCoreDumpOnAbort = (atoi(value) > 0);
    }
    else
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Ignoring unknown option '"
              << name << "' with value '" << value << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Ignoring unknown option '"
           << name << "' with value '" << value << "'.\n";
    }

    name = strtok(NULL, "=");

  } // End of while (name) ...

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Completed parsing of string '"
          << env << "'.\n" << logofs_flush;
  #endif

  if (*fileOptions != '\0')
  {
    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Reading options from '" << fileOptions
            << "'.\n" << logofs_flush;
    #endif

    if (ParseFileOptions(fileOptions) < 0)
    {
      return -1;
    }
  }

  //
  // If port where proxy is acting as an X server
  // was not specified assume the same port where
  // proxy is listening for the remote peer.
  //

  if (xPort == DEFAULT_NX_X_PORT)
  {
    xPort = proxyPort;
  }

  return 1;
}

//
// Parse the command line options passed
// by user when running proxy in stand
// alone mode.
//

int ParseCommandLineOptions(int argc, const char **argv)
{
  //
  // Execute only once unless force is set.
  //

  static int parsed;

  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Be sure we have a parameters repository
  //

  if (control == NULL)
  {
    control = new Control();
  }

  if (parsed == 1)
  {
    #if defined(INFO) || defined (TEST)
    *logofs << "Loop: Skipping a further parse of command line options.\n"
            << logofs_flush;
    #endif

    return 1;
  }

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Going to parse command line options.\n"
          << logofs_flush;
  #endif

  parsed = 1;

  //
  // Print out arguments.
  //

  #ifdef TEST

  *logofs << "Loop: argc is " << argc << ".\n" << logofs_flush;

  for (int argi = 0; argi < argc; argi++)
  {
    *logofs << "Loop: argv[" << argi << "] is " << argv[argi]
            << ".\n" << logofs_flush;
  }

  #endif

  //
  // Shall use getopt here.
  //

  for (int argi = 1; argi < argc; argi++)
  {
    const char *nextArg = argv[argi];

    if (*nextArg == '-')
    {
      switch (*(nextArg + 1))
      {
      case 'C':
        {
          //
          // Start proxy in CLIENT mode.
          //

          if (!WE_SET_PROXY_MODE)
          {
            #if defined(INFO) || defined (TEST)
            *logofs << "Loop: Setting local proxy mode to PROXY_CLIENT.\n"
                    << logofs_flush;
            #endif

            control -> ProxyMode = PROXY_CLIENT;
          }
          else if (control -> ProxyMode != PROXY_CLIENT)
          {
            #ifdef PANIC
            *logofs << "Loop: PANIC! Can't redefine local proxy to "
                    << "client mode.\n" << logofs_flush;
            #endif

            cerr << "Error" << ": Can't redefine local proxy to "
                 << "client mode.\n";

            return -1;
          }
        }
        break;
      case 'S':
        {
          //
          // Start proxy in SERVER mode.
          //

          if (!WE_SET_PROXY_MODE)
          {
            #if defined(INFO) || defined (TEST)
            *logofs << "Loop: Setting local proxy mode to PROXY_SERVER.\n"
                    << logofs_flush;
            #endif

            control -> ProxyMode = PROXY_SERVER;
          }
          else if (control -> ProxyMode != PROXY_SERVER)
          {
            #ifdef PANIC
            *logofs << "Loop: PANIC! Can't redefine local proxy to "
                    << "server mode.\n" << logofs_flush;
            #endif

            cerr << "Error" << ": Can't redefine local proxy to "
                 << "server mode.\n";

            return -1;
          }
        }
        break;
      case 'F':
        {
          //
          // Merge the provided parameters with
          // options read from file.
          //

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else if (ParseFileOptions(arg) < 0)
          {
            return -1;
          }
        }
        break;
      case 'L':
        {
          //
          // Type of link between client and server proxy.
          //

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            if (ParseLinkOption(arg) < 0)
            {
              return -1;
            }
          }
        }
        break;
      case 'N':
        {
          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL || atoi(arg) < 0)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            #ifdef TEST

            *logofs << "Loop: Internal descriptor for X client connection is FD#"
                    << atoi(arg) << ".\n" << logofs_flush;

            *logofs << "Loop: Disabling listening on further X client connections.\n"
                    << logofs_flush;

            #endif

            useTCPSocket      = 0;
            useUnixSocket     = 0;
            useInternalSocket = 1;

            internalFD = atoi(arg);
          }
        }
        break;
      case 'T':
        {
          //
          // This option is to force proxy to
          // run as a thread. Ignore it.
          //
        }
        break;
      case 'd':
        {
          //
          // This option forces a different tunneled
          // port than port where proxy is listening
          // for remote peer.
          //

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            xPort = atoi(arg);
          }
        }
        break;
      case 'D':
        {
          const char *arg = GetArg(argi, argc, argv);

          if (!arg)
          {
            #ifdef PANIC
            *logofs << "Loop: PANIC! No value provided for -D parameter.\n"
                    << logofs_flush;
            #endif

            cerr << "Error" << ": No value provided for -D parameter.\n";

            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            if (strlen(arg) >= DEFAULT_STRING_LENGTH)
            {
              #ifdef PANIC
              *logofs << "Loop: PANIC! Parameter -D exceeds length of "
                      << DEFAULT_STRING_LENGTH << " characters.\n"
                      << logofs_flush;
              #endif

              cerr << "Error" << ": Parameter -D exceeds length of "
                   << DEFAULT_STRING_LENGTH << " characters.\n";

              return -1;
            }

            strcpy(displayHost, arg);
          }
        }
        break;
      case 'f':
        {
          useDaemonMode = 1;

          break;
        }
      case 'k':
        {
          //
          // This function doesn't return.
          //

          KillDaemon(DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort);
        }
        break;
      case 'l':
        {
          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            if (strlen(arg) >= DEFAULT_STRING_LENGTH)
            {
              #ifdef PANIC
              *logofs << "Loop: PANIC! Parameter -l exceeds length of "
                      << DEFAULT_STRING_LENGTH << " characters.\n"
                      << logofs_flush;
              #endif

              cerr << "Error" << ": Parameter -l exceeds length of "
                   << DEFAULT_STRING_LENGTH << " characters.\n";

              return -1;
            }

            strcpy(logFileName, arg);
          }
        }
        break;
      case 's':
        {
          //
          // Enable printing of statistics to the
          // log file provided as parameter.
          //

          control -> CollectStatistics = 1;

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            if (strlen(arg) >= DEFAULT_STRING_LENGTH)
            {
              #ifdef PANIC
              *logofs << "Loop: PANIC! Parameter -s exceeds length of "
                      << DEFAULT_STRING_LENGTH << " characters.\n"
                      << logofs_flush;
              #endif

              cerr << "Error" << ": Parameter -s exceeds length of "
                   << DEFAULT_STRING_LENGTH << " characters.\n";

              return -1;
            }

            strcpy(statFileName, arg);
          }
        }
        break;
      case 'p':
        {
          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            proxyPort = atoi(arg);

            if (xPort == DEFAULT_NX_X_PORT)
            {
              xPort = proxyPort;
            }
          }
        }
        break;
      case 'P':
        {
          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            sshPort = atoi(arg);
          }
        }
        break;
      case 't':
        useTCPSocket = 0;
        break;
      case 'u':
        useUnixSocket = 0;
        break;
      case 'v':
        PrintVersionInfo();
        exit(0);
      case 'V':
        {
          //
          // Ignore the version argument.
          //

          GetArg(argi, argc, argv);

          //
          // TODO: This was used to print the extended
          // copyrigth info until we added option to
          // load NX libraries at proxy startup, so we
          // need to find another literal.
          //
          // PrintCopyrightInfo();
          //
        }
        break;
      case 'w':
        clientInitiateConnection = 1;
        break;
      case 'i':
        {
          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            //
            // Use default compression level or
            // the one selected in environment.
            //

            argi--;
          }
          else
          {
            control -> LocalDataCompressionLevel = atoi(arg);
          }

          if (control -> LocalDataCompressionLevel == 0)
          {
            control -> LocalDataCompression = 0;
          }
          else
          {
            control -> LocalDataCompression = 1;
          }
        }
        break;
      case 'I':
        {
          //
          // When data compression is enabled set
          // threshold of minimum data size to be
          // compressed through ZLIB.
          //

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            control -> LocalDataCompressionThreshold = atoi(arg);
          }
        }
        break;
      case 'z':
        {
          //
          // Use ZLIB stream compression.
          //

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            //
            // Use default compression level or
            // the one selected in environment.
            //

            argi--;
          }
          else
          {
            control -> LocalStreamCompressionLevel = atoi(arg);
          }

          if (control -> LocalStreamCompressionLevel == 0)
          {
            control -> LocalStreamCompression = 0;
          }
          else
          {
            control -> LocalStreamCompression = 1;
          }
        }
        break;
      case 'c':
        {
          //
          // Set maximum size in KB of cache. As client
          // side controls cache usage, it will ensure
          // that also server side honor this limit.
          //

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            control -> ClientTotalStorageSizeLimit = (atoi(arg) * 1024);
            control -> ServerTotalStorageSizeLimit = (atoi(arg) * 1024);
          }
        }
        break;
      case 'b':
        {
          //
          // Set block size of chunks of split messages.
          //

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            control -> SplitDataPacketLimit = atoi(arg);
          }
        }
        break;
      case 'B':
        {
          //
          // Set maximum size of objects to be transferred
          // without splitting them in chunks.
          //

          const char *arg = GetArg(argi, argc, argv);

          if (arg == NULL)
          {
            PrintUsage(argc, argv);

            return -1;
          }
          else
          {
            control -> SplitDataThreshold = atoi(arg);
          }
        }
        break;
      default:
        {
          PrintUsage(argc, argv);

          return -1;
        }
      }
    }
    else
    {
      if (nextArg)
      {
        //
        // Merge the provided parameters with
        // the DISPLAY environment variable.
        //

        if (ParseEnvironmentOptions(nextArg, 1) < 0)
        {
          PrintUsage(argc, argv);

          return -1;
        }
      }
    }
  }

  return 1;
}

//
// Read options from file and merge with environment.
//

int ParseFileOptions(const char *file)
{
  char *fileName;

  if (*file != '/' && *file != '.')
  {
    char *filePath = GetSessionPath();

    if (filePath == NULL)
    {
      cerr << "Error" << ": Cannot determine directory for NX option file.\n";

      HandleCleanup();
    }

    fileName = new char[strlen(filePath) + strlen("/") +
                            strlen(file) + 1];

    strcpy(fileName, filePath);

    strcat(fileName, "/");
    strcat(fileName, file);

    delete [] filePath;
  }
  else
  {
    fileName = new char[strlen(file) + 1];

    strcpy(fileName, file);
  }

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Going to read options from file '"
          << fileName << "'.\n" << logofs_flush;
  #endif

  FILE *filePtr = fopen(fileName, "r");

  if (filePtr == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't open options file '" << fileName
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't open options file '" << fileName
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    delete [] fileName;

    return -1;
  }

  char format[DEFAULT_STRING_LENGTH];

  sprintf(format, "%%%ds", DEFAULT_DISPLAY_OPTIONS_LENGTH - 1);

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Reading with format string " << format
          << " " << "from options file '" << fileName
          << "'.\n" << logofs_flush;
  #endif

  char options[DEFAULT_DISPLAY_OPTIONS_LENGTH];

  #ifdef VALGRIND

  memset(options, '\0', DEFAULT_DISPLAY_OPTIONS_LENGTH);

  #endif

  if (fscanf(filePtr, format, options) <= 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't read options from file '" << fileName
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't read options from file '" << fileName
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    delete [] fileName;

    return -1;
  }

  fclose(filePtr);

  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Read options '" << options << "' from file '"
          << fileName << "'.\n" << logofs_flush;
  #endif

  if (ParseEnvironmentOptions(options, 1) < 0)
  {
    delete [] fileName;

    return -1;
  }

  delete [] fileName;

  return 1;
}

//
// Parse the option string passed from the
// remote proxy at startup.
//

int ParseRemoteOptions(char *opts)
{
  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Going to parse remote options string '"
          << opts << "'.\n" << logofs_flush;
  #endif

  char *name;
  char *value;

  //
  // The options string is intended to be a series
  // of name/value tuples in the form name=value
  // separated by the ',' character.
  //

  int hasCookie = 0;
  int hasLink   = 0;
  int hasPack   = 0;
  int hasCache  = 0;
  int hasImages = 0;
  int hasDelta  = 0;
  int hasStream = 0;
  int hasData   = 0;
  int hasLimit  = 0;
  int hasFlush  = 0;
  int hasRender = 0;
  int hasTaint  = 0;
  int hasType   = 0;

  //
  // Get rid of the terminating space.
  //

  if (*(opts + strlen(opts) - 1) == ' ')
  {
    *(opts + strlen(opts) - 1) = '\0';
  }

  name = strtok(opts, "=");

  while (name)
  {
    value = strtok(NULL, ",");

    if (CheckArg("remote", name, value) < 0)
    {
      return -1;
    }

    if (strcasecmp(name, "cookie") == 0)
    {
      if (WE_INITIATE_CONNECTION)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'cookie' with value '"
                << value << "' when initiating connection.\n"
                << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'cookie' with value '"
             << value << "' when initiating connection.\n";
      }
      else if (strncasecmp(authCookie, value, strlen(authCookie)) != 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Authorization cookie '" << value
                << "' doesn't match '" << authCookie << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Authorization cookie '" << value
             << "' doesn't match '" << authCookie << "'.\n";

        return -1;
      }

      hasCookie = 1;
    }
    else if (strcasecmp(name, "link") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'link' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'link' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        if (*linkSpeedName != '\0' && strcasecmp(linkSpeedName, value) != 0)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Overriding option 'link' with new value '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Overriding option 'link' with new value '"
               << value << "'.\n";
        }

        if (ParseLinkOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify remote 'link' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify remote 'link' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasLink = 1;
    }
    else if (strcasecmp(name, "pack") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'pack' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'pack' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        if (*packMethodName != '\0' && strcasecmp(packMethodName, value) != 0)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Overriding option 'pack' with remote value '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Overriding option 'pack' with remote value '"
               << value << "'.\n";
        }

        if (ParsePackOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify remote 'pack' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify remote 'pack' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasPack = 1;
    }
    else if (strcasecmp(name, "cache") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'cache' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'cache' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // Cache size is sent as a hint of how much memory
        // the remote proxy is going to consume. A very low
        // powered thin client could choose to refuse the
        // connection.
        //

        if (ParseCacheOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify remote 'cache' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify remote 'cache' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasCache = 1;
    }
    else if (strcasecmp(name, "images") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'images' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'images' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // Images cache size is sent as a hint.
        // There is no obbligation for the local
        // proxy to use the persistent cache.
        //

        if (ParseImagesOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify remote 'images' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify remote 'images' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasImages = 1;
    }
    else if (strcasecmp(name, "limit") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'limit' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'limit' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        if (*bitrateLimitName != '\0' && strcasecmp(bitrateLimitName, value) != 0)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Overriding option 'limit' with new value '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Overriding option 'limit' with new value '"
               << value << "'.\n";
        }

        if (ParseLimitOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify 'limit' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify 'limit' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasLimit = 1;
    }
    else if (strcasecmp(name, "flush") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'flush' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'flush' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // Follow client side suggestion only
        // if user did not specify a value.
        //

        if (useFlush == -1)
        {
          useFlush = atoi(value);
        }
      }

      hasFlush = 1;
    }
    else if (strcasecmp(name, "render") == 0)
    {
      //
      // Became a client side option starting
      // from protocol level 3.
      //

      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'render' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'render' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // This was not negotiated in protocol
        // level 2. In this case ignore client
        // settings to guarantee compatibility.
        //

        if (useRender == -1)
        {
          useRender = atoi(value);
        }
      }

      hasRender = 1;
    }
    else if (strcasecmp(name, "taint") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'taint' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'taint' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        useTaint = atoi(value);
      }

      hasTaint = 1;
    }
    else if (strcasecmp(name, "type") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'type' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'type' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // This was not negotiated in protocol
        // level 2. Silently override the local
        // settings. This is safe as old proxy
        // versions don't pass it.
        //

        if (strcasecmp(value, "default") == 0)
        {
          *sessionType = '\0';
        }
        else
        {
          strncpy(sessionType, value, DEFAULT_STRING_LENGTH - 1);
        }
      }

      hasType = 1;
    }
    else if (strcasecmp(name, "delta") == 0)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'delta' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'delta' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        control -> RemoteDeltaCompression = (atoi(value) > 0);

        //
        // Follow for delta compression the
        // same settings as the client proxy.
        //

        control -> LocalDeltaCompression  = control -> RemoteDeltaCompression;
      }

      hasDelta = 1;
    }
    else if (strcasecmp(name, "stream") == 0)
    {
      //
      // If remote side didn't choose its own
      // stream compression level then assume
      // local settings.
      //

      if (strcasecmp(value, "default") == 0)
      {
        //
        // This applies only at client side.
        //

        control -> RemoteStreamCompression =
            control -> LocalStreamCompression;

        control -> RemoteStreamCompressionLevel =
            control -> LocalStreamCompressionLevel;
      }
      else
      {
        control -> RemoteStreamCompressionLevel = atoi(value);

        if (control -> RemoteStreamCompressionLevel > 0)
        {
          control -> RemoteStreamCompression = 1;
        }
        else
        {
          control -> RemoteStreamCompression = 0;
        }

        if (control -> LocalStreamCompressionLevel < 0)
        {
          control -> LocalStreamCompressionLevel = atoi(value);

          if (control -> LocalStreamCompressionLevel > 0)
          {
            control -> LocalStreamCompression = 1;
          }
          else
          {
            control -> LocalStreamCompression = 0;
          }
        }
      }

      hasStream = 1;
    }
    else if (strcasecmp(name, "data") == 0)
    {
      //
      // Apply the same to data compression level.
      //

      if (strcasecmp(value, "default") == 0)
      {
        control -> RemoteDataCompression =
            control -> LocalDataCompression;

        control -> RemoteDataCompressionLevel =
            control -> LocalDataCompressionLevel;
      }
      else
      {
        control -> RemoteDataCompressionLevel = atoi(value);

        if (control -> RemoteDataCompressionLevel > 0)
        {
          control -> RemoteDataCompression = 1;
        }
        else
        {
          control -> RemoteDataCompression = 0;
        }

        if (control -> LocalDataCompressionLevel < 0)
        {
          control -> LocalDataCompressionLevel = atoi(value);

          if (control -> LocalDataCompressionLevel > 0)
          {
            control -> LocalDataCompression = 1;
          }
          else
          {
            control -> LocalDataCompression = 0;
          }
        }
      }

      hasData = 1;
    }
    else
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Ignoring unknown remote option '"
              << name << "' with value '" << value << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Ignoring unknown remote option '"
           << name << "' with value '" << value << "'.\n";
    }

    name = strtok(NULL, "=");

  } // End of while (name) ...

  //
  // If we are client side, we need remote 'stream'
  // and 'data' options. If we are server, we need
  // all the above plus 'link' and some others.
  //

  char missing[DEFAULT_STRING_LENGTH];

  *missing = '\0';

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    if (hasStream == 0)
    {
      strcpy(missing, "stream");
    }
    else if (hasData == 0)
    {
      strcpy(missing, "data");
    }
  }
  else
  {
    if (hasLink == 0)
    {
      strcpy(missing, "link");
    }
    else if (hasCache == 0)
    {
      strcpy(missing, "cache");
    }
    else if (hasPack == 0)
    {
      strcpy(missing, "pack");
    }
    else if (hasDelta == 0)
    {
      strcpy(missing, "delta");
    }
    else if (hasStream == 0)
    {
      strcpy(missing, "stream");
    }
    else if (hasData == 0)
    {
      strcpy(missing, "data");
    }

    if (control -> isProtoStep3() == 1)
    {
      //
      // Don't complain if 'flush', 'render'
      // and 'taint' options are not passed.
      //

      if (hasType == 0)
      {
        strcpy(missing, "type");
      }
      else if (hasImages == 0)
      {
        strcpy(missing, "images");
      }
    }
  }

  if (!WE_INITIATE_CONNECTION)
  {
    //
    // Can be that user doesn't have requested to
    // check the authorization cookie provided by
    // the connecting peer.
    //

    if (hasCookie == 0 && *authCookie != '\0')
    {
      strcpy(missing, "cookie");
    }
  }

  if (*missing != '\0')
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Remote peer didn't specify option '"
            << missing << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Remote peer didn't specify option '"
         << missing << "'.\n";

    return -1;
  }

  return 1;
}

#ifdef COREDUMPS

int EnableCoreDumps(void)
{
  rlimit rlim;

  if (getrlimit(RLIMIT_CORE, &rlim))
  {
    #ifdef TEST
    *logofs << "Cannot read RLIMIT_CORE. Error is '"
            << ESTR() << "'.\n" << logofs_flush;
    #endif

    return -1;
  }

  if (rlim.rlim_cur < rlim.rlim_max)
  {
    rlim.rlim_cur = rlim.rlim_max;

    if (setrlimit(RLIMIT_CORE, &rlim))
    {
      #ifdef TEST
      *logofs << "Loop: Cannot read RLIMIT_CORE. Error is '"
              << ESTR() << "'.\n" << logofs_flush;
      #endif

      return -2;
    }
  }

  #ifdef TEST
  *logofs << "Loop: Set RLIMIT_CORE to "<< rlim.rlim_max
          << ".\n" << logofs_flush;
  #endif

  return 0;
}

#endif

int SendLocalCaches(int fd, const char *path, const char prefix, int entries)
{
  if (control -> LocalDeltaCompression == 0 ||
          control -> PersistentCacheEnableLoad == 0)
  {
    WriteRemoteOptions(fd, "cachelist=none ", strlen("cachelist=none "));

    return 0;
  }

  int count = 0;

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Looking for cache files in directory '"
          << path << "'.\n" << logofs_flush;
  #endif

  DIR *cacheDir = opendir(path);

  if (cacheDir != NULL)
  {
    dirent *dirEntry;

    int prologue = 0;

    while (((dirEntry = readdir(cacheDir)) != NULL) && (count < entries))
    {
      if (*dirEntry -> d_name == prefix &&
              strlen(dirEntry -> d_name) == (MD5_LENGTH * 2 + 2))
      {
        if (prologue == 0)
        {
          WriteRemoteOptions(fd, "cachelist=", strlen("cachelist="));

          prologue = 1;
        }
        else
        {
          WriteRemoteOptions(fd, ",", strlen(","));
        }

        #ifdef TEST
        *logofs << "Loop: Writing entry '" << path << "/"
                << dirEntry -> d_name << "' to FD#" << fd
                << ".\n" << logofs_flush;
        #endif

        //
        // Write cache file name to the socket,
        // including leading 'C-' or 'S-'.
        //

        WriteRemoteOptions(fd, dirEntry -> d_name, MD5_LENGTH * 2 + 2);

        count++;
      }
    }

    closedir(cacheDir);
  }

  if (count == 0)
  {
    WriteRemoteOptions(fd, "cachelist=none ", strlen("cachelist=none "));
  }
  else
  {
    WriteRemoteOptions(fd, " ", 1);
  }

  return count;
}

char *ReceiveRemoteCaches(int fd, int entries)
{
  int size = ((MD5_LENGTH * 2 + 2) + strlen(",")) * entries +
                 strlen("cachelist=") + strlen(" ") + 1;

  char *buffer = new char[size];

  if (ReadRemoteOptions(fd, buffer, size - 1, ' ') < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Failed to read cache list from proxy FD#"
             << fd << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Failed to read remote cache list.\n";

    delete [] buffer;

    return NULL;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Read list of caches from remote side as '"
          << buffer << "'.\n" << logofs_flush;
  #endif

  //
  // Prepare the buffer. What we want is a list
  // like "cache1,cache2,cache2" terminated by
  // null.
  //

  *(buffer + strlen(buffer) - 1) = '\0';

  if (strncasecmp(buffer, "cachelist=", strlen("cachelist=")) != 0)
  {
    #ifdef PANIC
    *logofs << "Loop: Wrong format for list of cache files "
            << "read from FD#" << fd << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Wrong format for list of cache files.\n";

    delete [] buffer;

    return NULL;
  }

  return buffer;
}

char *SelectLastCache(char *listBuffer, const char *searchPath)
{
  if (listBuffer == NULL || searchPath == NULL ||
          strncmp(listBuffer, "cachelist=", strlen("cachelist=")) != 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Invalid parameters '" << listBuffer << "' and '"
            << searchPath << "'. Can't select any cache.\n"
            << logofs_flush;
    #endif

    return NULL;
  }

  char *selectedName = new char[MD5_LENGTH * 2 + 3];

  *selectedName = '\0';

  char *localPrefix;
  char *remotePrefix;

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    localPrefix  = "C-";
    remotePrefix = "S-";
  }
  else
  {
    localPrefix  = "S-";
    remotePrefix = "C-";
  }

  //
  // Get rid of prefix.
  //

  listBuffer += strlen("cachelist=");

  char *fileName;

  fileName = strtok(listBuffer, ",");

  //
  // It is "/path/to/file" + "/" + "C-" + 32 + "\0".
  //

  char fullPath[strlen(searchPath) + MD5_LENGTH * 2 + 4];

  time_t selectedTime = 0;

  struct stat fileStat;

  while (fileName)
  {
    if (strncmp(fileName, "none", strlen("none")) == 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: No cache files seem to be available.\n"
              << logofs_flush;
      #endif

      delete [] selectedName;

      return NULL;
    }
    else if (strlen(fileName) != MD5_LENGTH * 2 + 2 ||
                 strncmp(fileName, remotePrefix, 2) != 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Bad cache file name '"
               << fileName << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Bad cache file name '"
           << fileName << "'.\n";

      delete [] selectedName;

      HandleCleanup();
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Parsing remote cache name '"
            << fileName << "'.\n" << logofs_flush;
    #endif

    //
    // Prefix, received as "S-", becomes
    // "C-" and viceversa.
    //

    *fileName = *localPrefix;

    strcpy(fullPath, searchPath);
    strcat(fullPath, "/");
    strcat(fullPath, fileName);

    if (stat(fullPath, &fileStat) == 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Found a matching cache '"
              << fullPath << "'.\n" << logofs_flush;
      #endif

      if (fileStat.st_mtime >= selectedTime)
      {
        strcpy(selectedName, fileName);

        selectedTime = fileStat.st_mtime;
      }
    }
    #if defined(INFO) || defined(TEST)
    else
    {
      *logofs << "Loop: Can't get stats of file '"
              << fullPath << "'.\n" << logofs_flush;
    }
    #endif

    fileName = strtok(NULL, ",");
  }

  if (*selectedName != '\0')
  {
    return selectedName;
  }
  else
  {
    delete [] selectedName;

    return NULL;
  }
}

char *GetRootPath()
{
  if (*rootDir == '\0')
  {
    strcpy(rootDir, getenv("HOME"));

    if (*rootDir == '\0')
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! No environment variable for HOME.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": No environment variable for HOME.\n";

      return NULL;
    }

    strcat(rootDir, "/.nx");

    //
    // Create NX root directory.
    //

    struct stat dirStat;

    if ((stat(rootDir, &dirStat) == -1) && (EGET() == ENOENT))
    {
      if (mkdir(rootDir, 0777) < 0 && (EGET() != EEXIST))
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't create directory '" << rootDir
                << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Can't create directory '" << rootDir
             << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

        return NULL;
      }
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Root of NX directory structure is '"
            << rootDir << "'.\n" << logofs_flush;
    #endif
  }

  char *rootPath = new char[strlen(rootDir) + 1];

  strcpy(rootPath, rootDir);

  return rootPath;
}

char *GetCachePath()
{
  char *rootPath = GetRootPath();

  if (rootPath == NULL)
  {
    return NULL;
  }

  char *cachePath;

  if (*sessionType != '\0')
  {
    cachePath = new char[strlen(rootPath) + strlen("/cache-") +
                             strlen(sessionType) + 1];
  }
  else
  {
    cachePath = new char[strlen(rootPath) + strlen("/cache") + 1];
  }

  strcpy(cachePath, rootPath);

  if (*sessionType != '\0')
  {
    strcat(cachePath, "/cache-");

    strcat(cachePath, sessionType);
  }
  else
  {
    strcat(cachePath, "/cache");
  }

  //
  // Create cache directory if needed.
  //

  struct stat dirStat;

  if ((stat(cachePath, &dirStat) == -1) && (EGET() == ENOENT))
  {
    if (mkdir(cachePath, 0777) < 0 && (EGET() != EEXIST))
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't create directory '" << cachePath
              << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't create directory '" << cachePath
           << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

      delete [] rootPath;
      delete [] cachePath;

      return NULL;
    }
  }

  delete [] rootPath;

  return cachePath;
}

char *GetImagesPath()
{
  char *rootPath = GetRootPath();

  if (rootPath == NULL)
  {
    return NULL;
  }

  char *imagesPath = new char[strlen(rootPath) + strlen("/images") + 1];

  strcpy(imagesPath, rootPath);

  strcat(imagesPath, "/images");

  //
  // Create cache directory if needed.
  //

  struct stat dirStat;

  if ((stat(imagesPath, &dirStat) == -1) && (EGET() == ENOENT))
  {
    if (mkdir(imagesPath, 0777) < 0 && (EGET() != EEXIST))
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't create directory '" << imagesPath
              << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't create directory '" << imagesPath
           << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

      delete [] rootPath;
      delete [] imagesPath;

      return NULL;
    }
  }

  //
  // Create 16 directories in the path to
  // hold the images whose name begins with
  // the corresponding hexadecimal digit.
  //

  char *digitPath = new char[strlen(imagesPath) + 5];

  strcpy(digitPath, imagesPath);

  //
  // Image paths have format "[path][/I-c][\0]",
  // where c is the first digit of the checksum.
  //

  for (char digit = 0; digit < 16; digit++)
  {
    sprintf(digitPath + strlen(imagesPath), "/I-%01X", digit);

    if ((stat(digitPath, &dirStat) == -1) && (EGET() == ENOENT))
    {
      if (mkdir(digitPath, 0777) < 0 && (EGET() != EEXIST))
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't create directory '" << digitPath
                << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Can't create directory '" << digitPath
             << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

        delete [] rootPath;
        delete [] imagesPath;
        delete [] digitPath;

        return NULL;
      }
    }
  }

  delete [] rootPath;
  delete [] digitPath;

  return imagesPath;
}

char *GetSessionPath()
{
  if (*sessionDir == '\0')
  {
    char *rootPath = GetRootPath();

    if (rootPath == NULL)
    {
      return NULL;
    }

    strcpy(sessionDir, rootPath);

    if (control -> ProxyMode == PROXY_CLIENT)
    {
      strcat(sessionDir, "/C-");
    }
    else
    {
      strcat(sessionDir, "/S-");
    }

    if (*sessionId == '\0')
    {
      char port[DEFAULT_STRING_LENGTH];

      sprintf(port, "%d", proxyPort);

      strcpy(sessionId, port);
    }

    strcat(sessionDir, sessionId);

    struct stat dirStat;

    if ((stat(sessionDir, &dirStat) == -1) && (EGET() == ENOENT))
    {
      if (mkdir(sessionDir, 0777) < 0 && (EGET() != EEXIST))
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't create directory '" << sessionDir
                << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Can't create directory '" << sessionDir
             << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

        delete [] rootPath;

        return NULL;
      }
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Root of NX session is '" << sessionDir
            << "'.\n" << logofs_flush;
    #endif

    delete [] rootPath;
  }

  char *sessionPath = new char[strlen(sessionDir) + 1];

  strcpy(sessionPath, sessionDir);

  return sessionPath;
}

//
// Identify requested link characteristics
// and set control parameters accordingly.
//

int ParseLinkOption(const char *opt)
{
  //
  // Reset user's selection and check if
  // the newly requested value is correct.
  //

  strcpy(linkSpeedName, opt);

  //
  // Normalize user input.
  //

  if (strcasecmp(opt, "33k") == 0 ||
      strcasecmp(opt, "56k") == 0)
  {
    strcpy(linkSpeedName, "modem");
  }
  else if (strcasecmp(opt, "64k")  == 0 ||
           strcasecmp(opt, "128k") == 0)
  {
    strcpy(linkSpeedName, "isdn");
  }
  else if (strcasecmp(opt, "256k") == 0 ||
           strcasecmp(opt, "640k") == 0)
  {
    strcpy(linkSpeedName, "adsl");
  }
  else if (strcasecmp(opt, "1m")  == 0 ||
           strcasecmp(opt, "2m")  == 0 ||
           strcasecmp(opt, "34m") == 0)
  {
    strcpy(linkSpeedName, "wan");
  }
  else if (strcasecmp(opt, "10m")   == 0 ||
           strcasecmp(opt, "100m")  == 0 ||
           strcasecmp(opt, "local") == 0)
  {
    strcpy(linkSpeedName, "lan");
  }

  if (strcasecmp(linkSpeedName, "modem") != 0 &&
      strcasecmp(linkSpeedName, "isdn")  != 0 &&
      strcasecmp(linkSpeedName, "adsl")  != 0 &&
      strcasecmp(linkSpeedName, "wan")   != 0 &&
      strcasecmp(linkSpeedName, "lan")   != 0)
  {
    return -1;
  }

  return 1;
}

int ParsePackOption(const char *opt)
{
  #ifdef DEBUG
  *logofs << "Loop: Pack method is " << packMethod
          << " quality is " << packQuality << ".\n"
          << logofs_flush;
  #endif

  #ifdef DEBUG
  *logofs << "Loop: Parsing pack method '" << opt
          << "'.\n" << logofs_flush;
  #endif

  //
  // We need to check 'png-jpeg' before 'png'
  // otherwise we'll always match the latter.
  //

  if (strcasecmp(opt, "0") == 0 ||
          strcasecmp(opt, "nopack") == 0 ||
              strcasecmp(opt, "no-pack") == 0)
  {
    packMethod = NO_PACK;
  }
  else if (strcasecmp(opt, "8") == 0)
  {
    packMethod = PACK_MASKED_8_COLORS;
  }
  else if (strcasecmp(opt, "64") == 0)
  {
    packMethod = PACK_MASKED_64_COLORS;
  }
  else if (strcasecmp(opt, "256") == 0)
  {
    packMethod = PACK_MASKED_256_COLORS;
  }
  else if (strcasecmp(opt, "512") == 0)
  {
    packMethod = PACK_MASKED_512_COLORS;
  }
  else if (strcasecmp(opt, "4k") == 0)
  {
    packMethod = PACK_MASKED_4K_COLORS;
  }
  else if (strcasecmp(opt, "32k") == 0)
  {
    packMethod = PACK_MASKED_32K_COLORS;
  }
  else if (strcasecmp(opt, "64k") == 0)
  {
    packMethod = PACK_MASKED_64K_COLORS;
  }
  else if (strcasecmp(opt, "256k") == 0)
  {
    packMethod = PACK_MASKED_256K_COLORS;
  }
  else if (strcasecmp(opt, "2m") == 0)
  {
    packMethod = PACK_MASKED_2M_COLORS;
  }
  else if (strcasecmp(opt, "16m") == 0)
  {
    packMethod = PACK_MASKED_16M_COLORS;
  }
  else if (strcasecmp(opt, "256-rdp") == 0)
  {
    packMethod = PACK_RDP_PLAIN_256_COLORS;
  }
  else if (strcasecmp(opt, "256-rdp-compressed") == 0)
  {
    packMethod = PACK_RDP_COMPRESSED_256_COLORS;
  }
  else if (strcasecmp(opt, "rfb-hextile") == 0)
  {
    packMethod = PACK_RFB_HEXTILE;
  }
  else if (strcasecmp(opt, "rfb-tight") == 0)
  {
    packMethod = PACK_RFB_TIGHT_PLAIN;
  }
  else if (strcasecmp(opt, "rfb-tight-compressed") == 0)
  {
    packMethod = PACK_RFB_TIGHT_COMPRESSED;
  }
  else if (strcasecmp(opt, "8-tight") == 0)
  {
    packMethod = PACK_TIGHT_8_COLORS;
  }
  else if (strcasecmp(opt, "64-tight") == 0)
  {
    packMethod = PACK_TIGHT_64_COLORS;
  }
  else if (strcasecmp(opt, "256-tight") == 0)
  {
    packMethod = PACK_TIGHT_256_COLORS;
  }
  else if (strcasecmp(opt, "512-tight") == 0)
  {
    packMethod = PACK_TIGHT_512_COLORS;
  }
  else if (strcasecmp(opt, "4k-tight") == 0)
  {
    packMethod = PACK_TIGHT_4K_COLORS;
  }
  else if (strcasecmp(opt, "32k-tight") == 0)
  {
    packMethod = PACK_TIGHT_32K_COLORS;
  }
  else if (strcasecmp(opt, "64k-tight") == 0)
  {
    packMethod = PACK_TIGHT_64K_COLORS;
  }
  else if (strcasecmp(opt, "256k-tight") == 0)
  {
    packMethod = PACK_TIGHT_256K_COLORS;
  }
  else if (strcasecmp(opt, "2m-tight") == 0)
  {
    packMethod = PACK_TIGHT_2M_COLORS;
  }
  else if (strcasecmp(opt, "16m-tight") == 0)
  {
    packMethod = PACK_TIGHT_16M_COLORS;
  }
  else if (strncasecmp(opt, "8-jpeg", strlen("8-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_8_COLORS;
  }
  else if (strncasecmp(opt, "64-jpeg", strlen("64-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_64_COLORS;
  }
  else if (strncasecmp(opt, "256-jpeg", strlen("256-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_256_COLORS;
  }
  else if (strncasecmp(opt, "512-jpeg", strlen("512-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_512_COLORS;
  }
  else if (strncasecmp(opt, "4k-jpeg", strlen("4k-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_4K_COLORS;
  }
  else if (strncasecmp(opt, "32k-jpeg", strlen("32k-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_32K_COLORS;
  }
  else if (strncasecmp(opt, "64k-jpeg", strlen("64k-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_64K_COLORS;
  }
  else if (strncasecmp(opt, "256k-jpeg", strlen("256k-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_256K_COLORS;
  }
  else if (strncasecmp(opt, "2m-jpeg", strlen("2m-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_2M_COLORS;
  }
  else if (strncasecmp(opt, "16m-jpeg", strlen("16m-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_16M_COLORS;
  }
  else if (strncasecmp(opt, "8-png-jpeg", strlen("8-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_8_COLORS;
  }
  else if (strncasecmp(opt, "64-png-jpeg", strlen("64-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_64_COLORS;
  }
  else if (strncasecmp(opt, "256-png-jpeg", strlen("256-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_256_COLORS;
  }
  else if (strncasecmp(opt, "512-png-jpeg", strlen("512-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_512_COLORS;
  }
  else if (strncasecmp(opt, "4k-png-jpeg", strlen("4k-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_4K_COLORS;
  }
  else if (strncasecmp(opt, "32k-png-jpeg", strlen("32k-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_32K_COLORS;
  }
  else if (strncasecmp(opt, "64k-png-jpeg", strlen("64k-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_64K_COLORS;
  }
  else if (strncasecmp(opt, "256k-png-jpeg", strlen("256k-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_256K_COLORS;
  }
  else if (strncasecmp(opt, "2m-png-jpeg", strlen("2m-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_2M_COLORS;
  }
  else if (strncasecmp(opt, "16m-png-jpeg", strlen("16m-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_16M_COLORS;
  }
  else if (strncasecmp(opt, "8-png", strlen("8-png")) == 0)
  {
    packMethod = PACK_PNG_8_COLORS;
  }
  else if (strncasecmp(opt, "64-png", strlen("64-png")) == 0)
  {
    packMethod = PACK_PNG_64_COLORS;
  }
  else if (strncasecmp(opt, "256-png", strlen("256-png")) == 0)
  {
    packMethod = PACK_PNG_256_COLORS;
  }
  else if (strncasecmp(opt, "512-png", strlen("512-png")) == 0)
  {
    packMethod = PACK_PNG_512_COLORS;
  }
  else if (strncasecmp(opt, "4k-png", strlen("4k-png")) == 0)
  {
    packMethod = PACK_PNG_4K_COLORS;
  }
  else if (strncasecmp(opt, "32k-png", strlen("32k-png")) == 0)
  {
    packMethod = PACK_PNG_32K_COLORS;
  }
  else if (strncasecmp(opt, "64k-png", strlen("64k-png")) == 0)
  {
    packMethod = PACK_PNG_64K_COLORS;
  }
  else if (strncasecmp(opt, "256k-png", strlen("256k-png")) == 0)
  {
    packMethod = PACK_PNG_256K_COLORS;
  }
  else if (strncasecmp(opt, "2m-png", strlen("2m-png")) == 0)
  {
    packMethod = PACK_PNG_2M_COLORS;
  }
  else if (strncasecmp(opt, "16m-png", strlen("16m-png")) == 0)
  {
    packMethod = PACK_PNG_16M_COLORS;
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Invalid pack option with value '"
            << opt << "'. Disabling pack.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Invalid pack option with value '"
         << opt << "'. Disabling pack.\n";

    packMethod = NO_PACK;
  }

  if (packMethod == NO_PACK)
  {
    strcpy(packMethodName, "no-pack");
  }
  else
  {
    strcpy(packMethodName, opt);
  }

  if (packMethod >= PACK_JPEG_8_COLORS &&
          packMethod <= PACK_JPEG_16M_COLORS)
  {
    char *dash = rindex(opt, '-') + 1;

    if (strlen(dash) != 1 || *dash < '0' || *dash > '9')
    {
      if (strcmp(dash, "jpeg") != 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Invalid pack quality with value '"
                << dash << "'. Using default.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Invalid pack quality with value '"
             << dash << "'. Using default.\n";
      }
    }
    else
    {
      packQuality = atoi(dash);

      #ifdef DEBUG
      *logofs << "Loop: Using pack quality '" << packQuality
              << "'.\n" << logofs_flush;
      #endif
    }
  }

  if (packMethod >= PACK_PNG_8_COLORS &&
          packMethod <= PACK_PNG_16M_COLORS)
  {
    char *dash = rindex(opt, '-') + 1;

    if (strlen(dash) != 1 || *dash < '0' || *dash > '9')
    {
      if (strcmp(dash, "png") != 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Invalid pack quality with value '"
                << dash << "'. Using default.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Invalid pack quality with value '"
             << dash << "'. Using default.\n";
      }
    }
    else
    {
      packQuality = atoi(dash);

      #ifdef DEBUG
      *logofs << "Loop: Using pack quality '" << packQuality
              << "'.\n" << logofs_flush;
      #endif
    }
  }

  if (packMethod >= PACK_PNG_JPEG_8_COLORS &&
          packMethod <= PACK_PNG_JPEG_16M_COLORS)
  {
    char *dash = rindex(opt, '-') + 1;

    if (strlen(dash) != 1 || *dash < '0' || *dash > '9')
    {
      if (strcmp(dash, "png") != 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Invalid pack quality with value '"
                << dash << "'. Using default.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Invalid pack quality with value '"
             << dash << "'. Using default.\n";
      }
    }
    else
    {
      packQuality = atoi(dash);

      #ifdef DEBUG
      *logofs << "Loop: Using pack quality '" << packQuality
              << "'.\n" << logofs_flush;
      #endif
    }
  }

  return 1;
}

int ParsePackMethod(const int method, const int quality)
{
  switch (method)
  {
    case NO_PACK:
    {
      strcpy(packMethodName, "no-pack");

      break;
    }
    case PACK_MASKED_8_COLORS:
    {
      strcpy(packMethodName, "8");

      break;
    }
    case PACK_MASKED_64_COLORS:
    {
      strcpy(packMethodName, "64");

      break;
    }
    case PACK_MASKED_256_COLORS:
    {
      strcpy(packMethodName, "256");

      break;
    }
    case PACK_MASKED_512_COLORS:
    {
      strcpy(packMethodName, "512");

      break;
    }
    case PACK_MASKED_4K_COLORS:
    {
      strcpy(packMethodName, "4k");

      break;
    }
    case PACK_MASKED_32K_COLORS:
    {
      strcpy(packMethodName, "32k");

      break;
    }
    case PACK_MASKED_64K_COLORS:
    {
      strcpy(packMethodName, "64k");

      break;
    }
    case PACK_MASKED_256K_COLORS:
    {
      strcpy(packMethodName, "256k");

      break;
    }
    case PACK_MASKED_2M_COLORS:
    {
      strcpy(packMethodName, "2m");

      break;
    }
    case PACK_MASKED_16M_COLORS:
    {
      strcpy(packMethodName, "16m");

      break;
    }
    case PACK_RDP_PLAIN_256_COLORS:
    {
      strcpy(packMethodName, "256-rdp");

      break;
    }
    case PACK_RDP_COMPRESSED_256_COLORS:
    {
      strcpy(packMethodName, "256-rdp-compressed");

      break;
    }
    case PACK_RFB_HEXTILE:
    {
      strcpy(packMethodName, "rfb-hextile");

      break;
    }
    case PACK_RFB_TIGHT_PLAIN:
    {
      strcpy(packMethodName, "rfb-tight");

      break;
    }
    case PACK_RFB_TIGHT_COMPRESSED:
    {
      strcpy(packMethodName, "rfb-tight-compressed");

      break;
    }
    case PACK_TIGHT_8_COLORS:
    {
      strcpy(packMethodName, "8-tight");

      break;
    }
    case PACK_TIGHT_64_COLORS:
    {
      strcpy(packMethodName, "64-tight");

      break;
    }
    case PACK_TIGHT_256_COLORS:
    {
      strcpy(packMethodName, "256-tight");

      break;
    }
    case PACK_TIGHT_512_COLORS:
    {
      strcpy(packMethodName, "512-tight");

      break;
    }
    case PACK_TIGHT_4K_COLORS:
    {
      strcpy(packMethodName, "4k-tight");

      break;
    }
    case PACK_TIGHT_32K_COLORS:
    {
      strcpy(packMethodName, "32k-tight");

      break;
    }
    case PACK_TIGHT_64K_COLORS:
    {
      strcpy(packMethodName, "64k-tight");

      break;
    }
    case PACK_TIGHT_256K_COLORS:
    {
      strcpy(packMethodName, "256k-tight");

      break;
    }
    case PACK_TIGHT_2M_COLORS:
    {
      strcpy(packMethodName, "2m-tight");

      break;
    }
    case PACK_TIGHT_16M_COLORS:
    {
      strcpy(packMethodName, "16m-tight");

      break;
    }
    case PACK_JPEG_8_COLORS:
    {
      strcpy(packMethodName, "8-jpeg");

      break;
    }
    case PACK_JPEG_64_COLORS:
    {
      strcpy(packMethodName, "64-jpeg");

      break;
    }
    case PACK_JPEG_256_COLORS:
    {
      strcpy(packMethodName, "256-jpeg");

      break;
    }
    case PACK_JPEG_512_COLORS:
    {
      strcpy(packMethodName, "512-jpeg");

      break;
    }
    case PACK_JPEG_4K_COLORS:
    {
      strcpy(packMethodName, "4k-jpeg");

      break;
    }
    case PACK_JPEG_32K_COLORS:
    {
      strcpy(packMethodName, "32k-jpeg");

      break;
    }
    case PACK_JPEG_64K_COLORS:
    {
      strcpy(packMethodName, "64k-jpeg");

      break;
    }
    case PACK_JPEG_256K_COLORS:
    {
      strcpy(packMethodName, "256k-jpeg");

      break;
    }
    case PACK_JPEG_2M_COLORS:
    {
      strcpy(packMethodName, "2m-jpeg");

      break;
    }
    case PACK_JPEG_16M_COLORS:
    {
      strcpy(packMethodName, "16m-jpeg");

      break;
    }
    case PACK_PNG_8_COLORS:
    {
      strcpy(packMethodName, "8-png");

      break;
    }
    case PACK_PNG_64_COLORS:
    {
      strcpy(packMethodName, "64-png");

      break;
    }
    case PACK_PNG_256_COLORS:
    {
      strcpy(packMethodName, "256-png");

      break;
    }
    case PACK_PNG_512_COLORS:
    {
      strcpy(packMethodName, "512-png");

      break;
    }
    case PACK_PNG_4K_COLORS:
    {
      strcpy(packMethodName, "4k-png");

      break;
    }
    case PACK_PNG_32K_COLORS:
    {
      strcpy(packMethodName, "32k-png");

      break;
    }
    case PACK_PNG_64K_COLORS:
    {
      strcpy(packMethodName, "64k-png");

      break;
    }
    case PACK_PNG_256K_COLORS:
    {
      strcpy(packMethodName, "256k-png");

      break;
    }
    case PACK_PNG_2M_COLORS:
    {
      strcpy(packMethodName, "2m-png");

      break;
    }
    case PACK_PNG_16M_COLORS:
    {
      strcpy(packMethodName, "16m-png");

      break;
    }
    case PACK_PNG_JPEG_8_COLORS:
    {
      strcpy(packMethodName, "8-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_64_COLORS:
    {
      strcpy(packMethodName, "64-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_256_COLORS:
    {
      strcpy(packMethodName, "256-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_512_COLORS:
    {
      strcpy(packMethodName, "512-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_4K_COLORS:
    {
      strcpy(packMethodName, "4k-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_32K_COLORS:
    {
      strcpy(packMethodName, "32k-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_64K_COLORS:
    {
      strcpy(packMethodName, "64k-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_256K_COLORS:
    {
      strcpy(packMethodName, "256k-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_2M_COLORS:
    {
      strcpy(packMethodName, "2m-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_16M_COLORS:
    {
      strcpy(packMethodName, "16m-png-jpeg");

      break;
    }
    default:
    {
      return -1;
    }
  }

  if (quality < 0 || quality > 9)
  {
    return -1;
  }

  if (packMethod >= PACK_JPEG_8_COLORS &&
          packMethod <= PACK_JPEG_16M_COLORS)
  {
    sprintf(packMethodName + strlen(packMethodName),
                "-%d", quality);
  }
  else if (packMethod >= PACK_PNG_8_COLORS &&
               packMethod <= PACK_PNG_16M_COLORS)
  {
    sprintf(packMethodName + strlen(packMethodName),
                "-%d", quality);
  }
  else if (packMethod >= PACK_PNG_JPEG_8_COLORS &&
               packMethod <= PACK_PNG_JPEG_16M_COLORS)
  {
    sprintf(packMethodName + strlen(packMethodName),
                "-%d", quality);
  }

  packMethod  = method;
  packQuality = quality;

  return 1;
}

int SetLog()
{
  //
  // Base NX directory.
  //

  control -> RootPath = GetRootPath();

  //
  // So far we used stderr (or stdout under
  // WIN32). Now use files selected by user.
  //

  if (*statFileName == '\0')
  {
    strcpy(statFileName, "stats");

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Assuming default statistics file '"
            << statFileName << "'.\n" << logofs_flush;
    #endif
  }
  #if defined(INFO) || defined(TEST)
  else
  {
    *logofs << "Loop: Name selected for statistics is '"
            << statFileName << "'.\n" << logofs_flush;
  }
  #endif

  if (OpenOutputFile(statFileName, statofs) < 0)
  {
    HandleCleanup();
  }

  if (*logFileName == '\0')
  {
    strcpy(logFileName, "errors");

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Assuming default log file name '"
            << logFileName << "'.\n" << logofs_flush;
    #endif
  }
  #if defined(INFO) || defined(TEST)
  else
  {
    *logofs << "Loop: Name selected for log file is '"
            << logFileName << "'.\n" << logofs_flush;
  }
  #endif

  if (OpenOutputFile(logFileName, logofs) < 0)
  {
    HandleCleanup();
  }

  //
  // Enable statistics by default.
  //

  control -> CollectStatistics = 1;

  return 1;
}

int SetPorts()
{
  //
  // Three possibilities are given at X client side:
  //
  // Port <= 0: Disable port forwarding.
  // Port == 1: Use default ports.
  // Port >  1: Use the specified port.
  //
  // At X server side, user must specify the ports
  // where to forward the connections. This is both
  // for security reasons and because, when running
  // both proxies on the same host, any connection
  // will be looped back to itself.
  //

  if (syncPort <= 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Disabling sync connections.\n"
            << logofs_flush;
    #endif

    syncPort = 0;

    useSyncSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == PROXY_CLIENT)
    {
      if (syncPort == 1)
      {
        syncPort = DEFAULT_NX_SYNC_PORT_OFFSET + proxyPort;
      }

      useSyncSocket = 1;
    }
    else
    {
      if (syncPort == 1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! No port specified for sync connections.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": No port specified for sync connections.\n";

        HandleCleanup();
      }

      useSyncSocket = 0;
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Using sync port '" << syncPort
            << "'.\n" << logofs_flush;
    #endif
  }

  if (keybdPort <= 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Disabling embedded keyboard connections.\n"
            << logofs_flush;
    #endif

    keybdPort = 0;

    useKeybdSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == PROXY_CLIENT)
    {
      if (keybdPort == 1)
      {
        keybdPort = DEFAULT_NX_KEYBD_PORT_OFFSET + proxyPort;
      }

      useKeybdSocket = 1;
    }
    else
    {
      //
      // Embedded keyboard must always be connected
      // to the display where session is running.
      // We will later check if user did specify a
      // port that matches our X server.
      //

      if (keybdPort == 1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! No port specified for embedded keyboard connections.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": No port specified for embedded keyboard connections.\n";

        HandleCleanup();
      }

      useKeybdSocket = 0;
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Using embedded keyboard port '" << keybdPort
            << "'.\n" << logofs_flush;
    #endif
  }

  if (sambaPort <= 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Disabling SMB connections.\n"
            << logofs_flush;
    #endif

    sambaPort = 0;

    useSambaSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == PROXY_CLIENT)
    {
      if (sambaPort == 1)
      {
        sambaPort = DEFAULT_NX_SAMBA_PORT_OFFSET + proxyPort;
      }

      useSambaSocket = 1;
    }
    else
    {
      if (sambaPort == 1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! No port specified for SMB connections.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": No port specified for SMB connections.\n";

        HandleCleanup();
      }

      useSambaSocket = 0;
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Using SMB port '" << sambaPort
            << "'.\n" << logofs_flush;
    #endif
  }

  if (mediaPort <= 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Disabling multimedia connections.\n"
            << logofs_flush;
    #endif

    mediaPort = 0;

    useMediaSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == PROXY_CLIENT)
    {
      if (mediaPort == 1)
      {
        mediaPort = DEFAULT_NX_MEDIA_PORT_OFFSET + proxyPort;
      }

      useMediaSocket = 1;
    }
    else
    {
      if (mediaPort == 1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! No port specified for multimedia connections.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": No port specified for multimedia connections.\n";

        HandleCleanup();
      }

      useMediaSocket = 0;
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Using multimedia port '" << mediaPort
            << "'.\n" << logofs_flush;
    #endif
  }

  return 1;
}

int SetDescriptors()
{
  unsigned int limit = 0;

  #ifdef RLIMIT_NOFILE

  rlimit limits;

  if (getrlimit(RLIMIT_NOFILE, &limits) == 0)
  {
    if (limits.rlim_max == RLIM_INFINITY)
    {
      limit = 0;
    }
    else
    {
      limit = (unsigned int) limits.rlim_max;
    }
  }

  #endif

  #ifdef _SC_OPEN_MAX

  if (limit == 0)
  {
    limit = sysconf(_SC_OPEN_MAX);
  }

  #endif

  #ifdef FD_SETSIZE

  if (limit > FD_SETSIZE)
  {
    limit = FD_SETSIZE;
  }

  #endif

  #ifdef RLIMIT_NOFILE

  if (limits.rlim_cur < limit)
  {
    limits.rlim_cur = limit;

    setrlimit(RLIMIT_NOFILE, &limits);
  }

  #endif

  if (limit == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Cannot determine number of available "
            << "file descriptors.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot determine number of available "
         << "file descriptors.\n";

    return -1;
  }

  return 1;
}

//
// Initialize all configuration parameters.
//

int SetParameters()
{
  //
  // Find out the type of session.
  //

  SetSession();

  //
  // Initialize the network and compression
  // parameters according to the settings
  // suggested by the user.
  //

  SetLink();

  //
  // Set compression according to link speed.
  //

  SetCompression();

  //
  // Be sure that we have a literal for current
  // cache size. Value will reflect control's
  // default unless we already parsed a 'cache'
  // option. Server side has no control on size
  // of cache but is informed at session nego-
  // tiation about how much memory is going to
  // be used.
  //

  SetCache();

  //
  // Set size of shared memory segments.
  //

  SetShmem();

  //
  // Make adjustments to cache based
  // on pack method.
  //

  SetPack();

  //
  // Set disk-based image cache.
  //

  SetImages();

  //
  // Set CPU and bandwidth limits.
  //

  SetLimits();

  return 1;
}

//
// According to session type literal determine
// the type of X traffic that is going to be
// transported. Literals should be better
// defined in future proxy revisions.
//

int SetSession()
{
  if (strncmp(sessionType, "unix-kde", strlen("unix-kde")) == 0 ||
          strncmp(sessionType, "unix-gnome", strlen("unix-gnome")) == 0 ||
              strncmp(sessionType, "unix-desktop", strlen("unix-desktop")) == 0)
  {
    control -> SessionMode = SESSION_X;
  }
  else if (strncmp(sessionType, "unix-application", strlen("unix-application")) == 0)
  {
    control -> SessionMode = SESSION_APPLICATION;
  }
  else if (strncmp(sessionType, "win", strlen("win")) == 0)
  {
    control -> SessionMode = SESSION_RDP;
  }
  else if (strncmp(sessionType, "vnc", strlen("vnc")) == 0)
  {
    control -> SessionMode = SESSION_RFB;
  }
  else
  {
    //
    // If session is not recognized let's
    // assume user knows what is doing.
    //

    control -> SessionMode = SESSION_X;
  }

  return 1;
}

//
// If a cache size literal was not set
// do it now.
//

int SetCache()
{
  int size = control -> getUpperStorageSizeLimit();

  if (size / 1024 > 0)
  {
    sprintf(cacheSizeName, "%dk", size / 1024);
  }
  else
  { 
    sprintf(cacheSizeName, "%d", size);
  }

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    control -> LocalTotalStorageSizeLimit =
        control -> ClientTotalStorageSizeLimit;

    control -> RemoteTotalStorageSizeLimit =
        control -> ServerTotalStorageSizeLimit;
  }
  else
  {
    control -> LocalTotalStorageSizeLimit =
        control -> ServerTotalStorageSizeLimit;

    control -> RemoteTotalStorageSizeLimit =
        control -> ClientTotalStorageSizeLimit;
  }

  #ifdef DEBUG
  *logofs << "Loop: Storage size limit is "
          << control -> ClientTotalStorageSizeLimit
          << " at client and "
          << control -> ServerTotalStorageSizeLimit
          << " at server.\n"
          << logofs_flush;
  #endif

  #ifdef DEBUG
  *logofs << "Loop: Storage local limit set to "
          << control -> LocalTotalStorageSizeLimit
          << " remote limit set to "
          << control -> RemoteTotalStorageSizeLimit
          << ".\n" << logofs_flush;
  #endif

  //
  // Never reserve for split store more than
  // half the memory available for messages.
  //

  if (control -> SplitTotalStorageSizeLimit > size / 2)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Reducing size of split store to "
            << size / 2 << " bytes.\n"
            << logofs_flush;
    #endif

    control -> SplitTotalStorageSizeLimit = size / 2;
  }

  //
  // Don't load render from persistent
  // cache if extension is hidden or
  // not supported by agent.
  //

  if (control -> AgentHideRender == 1)
  {
    #ifdef TEST
    *logofs << "Loop: Not loading render extension "
            << "from persistent cache.\n"
            << logofs_flush;
    #endif

    control -> PersistentCacheLoadRender = 0;
  }

  return 1;
}

int SetShmem()
{
  //
  // If not set, adjust size of the shared
  // memory segment according to size of
  // the message cache.
  //

  if (*shmemSizeName == '\0')
  {
    int size = control -> getUpperStorageSizeLimit();

    const int mega = 1048576;

    if (size <= 1 * mega)
    {
      size = 0;
    }
    else if (size <= 2 * mega)
    {
      size = 262144;
    }
    else
    {
      size = size / 8;
    }

    if (size > 4194304)
    {
      size = 4194304;
    }

    if (size / 1024 > 0)
    {
      sprintf(shmemSizeName, "%dk", size / 1024);
    }
    else
    {
      sprintf(shmemSizeName, "%d", size);
    }

    control -> AgentShmemClientSize = size;
    control -> AgentShmemServerSize = size;
  }

  //
  // TODO: In current version only X server
  // support is implemented.
  //
  // if (control -> AgentShmemClientSize >= 65536)
  // {
  //   control -> AgentShmemClient = 1;
  // }
  // else
  // {
  //   control -> AgentShmemClient = 0;
  // }
  //

  if (control -> AgentShmemServerSize >= 65536)
  {
    control -> AgentShmemServer = 1;
  }
  else
  {
    control -> AgentShmemServer = 0;
  }

  return 1;
}

//
// Adjust pack method according to session type.
//

int SetPack()
{
  #ifdef TEST
  *logofs << "Loop: Setting pack with initial method "
          << packMethod << " and quality " << packQuality
          << ".\n" << logofs_flush;
  #endif

  #if defined(INFO) || defined(TEST)

  int userMethod = packMethod;

  #endif

  //
  // Check if this is a RDP or RFB session and,
  // in this case, use a pack method optimized
  // for such kind of sessions. X sessions can
  // safely use the control default.
  //

  if (control -> SessionMode == SESSION_RDP)
  {
    //
    // If a pack method was not specified, use
    // compressed bitmaps as they seems to offer
    // better performances even on LAN.
    //

    if (packMethod != PACK_RDP_COMPRESSED_256_COLORS &&
            packMethod != PACK_RDP_PLAIN_256_COLORS &&
                packMethod != NO_PACK)
    {
      packMethod = PACK_RDP_COMPRESSED_256_COLORS;
    }
  }
  else if (control -> SessionMode == SESSION_RFB)
  {
    if (packMethod != PACK_RFB_HEXTILE &&
            packMethod != PACK_RFB_TIGHT_PLAIN &&
                packMethod != PACK_RFB_TIGHT_COMPRESSED &&
                    packMethod != NO_PACK)
    {
      packMethod = PACK_RFB_TIGHT_PLAIN;
    }
  }
  else if (control -> SessionMode == SESSION_APPLICATION)
  {
    //
    // At the moment packed images are not
    // supported for normal X applications
    // as this has not been built yet in
    // Xlib.
    //

    packMethod = NO_PACK;
  }

  //
  // Adjust internal settings according
  // to newly selected pack method.
  //

  ParsePackMethod(packMethod, packQuality);

  #if defined(INFO) || defined(TEST)

  if (userMethod != packMethod)
  {
    *logofs << "Loop: WARNING! Overriding option 'pack'. "
            << "Setting preferred method '"
            << packMethodName << "'.\n" << logofs_flush;
  }

  #endif

  control -> AgentPackMethod  = packMethod;
  control -> AgentPackQuality = packQuality;

  //
  // Don't load messages from persistent
  // cache if packed images are disabled.
  //

  if (control -> AgentPackMethod == NO_PACK)
  {
    #ifdef TEST
    *logofs << "Loop: Not loading packed images "
            << "from persistent cache.\n"
            << logofs_flush;
    #endif

    control -> PersistentCacheLoadPacked = 0;
  }

  //
  // Note that if we are at X client side, we
  // still ignore which protocol version is
  // supported by remote end. If PNG encoding
  // is requested, we'll later need to check
  // if it's actually supported.
  //

  return 1;
}

//
// Set disk-based image cache according to
// user's wishes.
//

int SetImages()
{
  //
  // Be sure we disable cache in case of session
  // application or with link LAN. This is mainly
  // useful to disable the house-keeping process
  // as in these case we don't have split.
  //

  if (control -> SessionMode != SESSION_X ||
          control -> LinkMode == LINK_TYPE_LAN)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Disabling images cache with session "
            << control -> SessionMode << " and link "
            << control -> LinkMode << ".\n"
            << logofs_flush;
    #endif

    sprintf(imagesSizeName, "0");

    control -> ImageCacheEnableLoad = 0;
    control -> ImageCacheEnableSave = 0;

    return 1;
  }

  int size = control -> ImageCacheDiskLimit;

  if (size / 1024 > 0)
  {
    sprintf(imagesSizeName, "%dk", size / 1024);
  }
  else
  {
    sprintf(imagesSizeName, "%d", size);
  }

  if (size > 0)
  {
    control -> ImageCacheEnableLoad = 1;
    control -> ImageCacheEnableSave = 1;

    if (control -> ProxyMode == PROXY_SERVER)
    {
      if ((control -> ImageCachePath = GetImagesPath()) == NULL)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Error getting or creating image cache path.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Error getting or creating image cache path.\n";

        HandleCleanup();
      }

      #if defined(INFO) || defined (TEST)
      *logofs << "Loop: Path of image cache files is '" << control -> ImageCachePath
              << "'.\n" << logofs_flush;
      #endif
    }
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Disabling NX images cache.\n"
            << logofs_flush;
    #endif

    control -> ImageCacheEnableLoad = 0;
    control -> ImageCacheEnableSave = 0;
  }

  return 1;
}

int SetVersion()
{
  //
  // In theory we consider the version numbers to
  // match if the major number matches. Different
  // minor and patch levels should be compatible
  // with each other...
  //

  if (control -> RemoteVersionMajor != 1)
  {
    return -1;
  }

  //
  // ...But in practice we have special cases
  // handled here.
  //

  if (control -> RemoteVersionMinor < 1 ||
          (control -> RemoteVersionMinor == 1 &&
               control -> RemoteVersionPatch < 1))
  {
    return -1;
  }

  if (control -> RemoteVersionMinor >= 2)
  {
    if (control -> RemoteVersionMinor == 2 &&
            control -> RemoteVersionPatch <= 1)
    {
      #ifdef TEST
      *logofs << "Loop: Using NX protocol step 2.\n"
              << logofs_flush;
      #endif

      control -> setProtoStep2();
    }
    else if (control -> RemoteVersionMinor == 2 &&
                 control -> RemoteVersionPatch == 2)
    {
      #ifdef TEST
      *logofs << "Loop: Using NX protocol step 3.\n"
              << logofs_flush;
      #endif

      control -> setProtoStep3();
    }
    else
    {
      #ifdef TEST
      *logofs << "Loop: Using NX protocol step 4.\n"
              << logofs_flush;
      #endif

      control -> setProtoStep4();
    }
  }
  else
  {
    #ifdef TEST
    *logofs << "Loop: Using NX protocol step 1.\n"
            << logofs_flush;
    #endif

    control -> setProtoStep1();
  }

  return 1;
}

//
// Identify requested link characteristics
// and set control parameters accordingly.
//

int SetLink()
{
  #ifdef TEST
  *logofs << "Loop: Setting link with initial value "
          << linkSpeedName << ".\n" << logofs_flush;
  #endif

  if (*linkSpeedName == '\0')
  {
    strcpy(linkSpeedName, "lan");
  }
  
  #ifdef TEST
  *logofs << "Loop: Link speed is " << linkSpeedName
          << ".\n" << logofs_flush;
  #endif

  if (strcasecmp(linkSpeedName, "modem") == 0)
  {
    SetLinkModem();
  }
  else if (strcasecmp(linkSpeedName, "isdn") == 0)
  {
    SetLinkIsdn();
  }
  else if (strcasecmp(linkSpeedName, "adsl") == 0)
  {
    SetLinkAdsl();
  }
  else if (strcasecmp(linkSpeedName, "wan") == 0)
  {
    SetLinkWan();
  }
  else if (strcasecmp(linkSpeedName, "lan") == 0)
  {
    SetLinkLan();
  }
  else
  {
    return -1;
  }

  //
  // In the case of RDP sessions reduce the
  // karma limit and set size of TCP receive
  // buffer of path agent-to-proxy according
  // to the size of scheduled writes.
  //

  if (control -> SessionMode == SESSION_RDP)
  {
    control -> AgentKarmaSize /= 8;

    if (control -> AgentKarmaSize < 1024)
    {
      control -> AgentKarmaSize = 1024;
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Reduced agent karma size to "
            << control -> AgentKarmaSize << ".\n"
            << logofs_flush;
    #endif
  }

  //
  // Override default flush timeout set in control
  // files in case of normal X client connections.
  // Normal X clients use so many replies to make
  // queuing completely useless. Do this at client
  // side as server side will receive our value at
  // session negotiation.
  //

  if (control -> SessionMode == SESSION_APPLICATION)
  {
    if (control -> ProxyMode == PROXY_CLIENT && useFlush == -1)
    {
       //
       // TODO: Should not be required but needs
       // to be better tested before the change
       // can take place.
       //

       control -> FlushTimeout = 0;

      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Overriding flush timeout with new value "
            << control -> FlushTimeout << ".\n"
            << logofs_flush;
      #endif

       control -> AgentFlushImmediate = 1;

      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Overriding immediate flush with new value "
            << control -> AgentFlushImmediate << ".\n"
            << logofs_flush;
      #endif
    }

    //
    // Also ensure we always read from proxy. A window
    // manager seems to deal with the X server in a way
    // that causes a 30-40 seconds deadlock when any X
    // channel finds itself blocked in write for long.
    //

    control -> LimitTimeout = 0;

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Overriding limit timeout with new value "
            << control -> LimitTimeout << ".\n"
            << logofs_flush;
    #endif
  }
  else
  {
    //
    // Don't taint replies if mode
    // is not single X applications.
    //

    control -> AgentTaintReplies = 0;

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Overriding taint of replies with new value "
            << control -> AgentTaintReplies << ".\n"
            << logofs_flush;
    #endif
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Using flush timeout "
          << control -> FlushTimeout << " with immediate flush "
          << control -> AgentFlushImmediate << ".\n"
          << logofs_flush;
  #endif

  //
  // Now that we know the remote proxy version
  // check if we need to disable the render
  // extension because:
  // 
  // 1. The remote proxy assumes that we will
  //    do that.
  //
  // 2. The remote proxy doesn't have support
  //    for the alpha channel.
  //

  if (control -> ProxyMode == PROXY_SERVER)
  {
    if (control -> SessionMode == SESSION_X &&
            control -> isProtoStep4() == 0)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Not using render extension "
              << "due to lack of alpha support.\n"
              << logofs_flush;
      #endif

      control -> PersistentCacheLoadRender = 0;

      control -> AgentHideRender = 1;
    }

    //
    // TODO: There are problems when running agent
    // sessions with RENDER extension on the Solaris
    // client. They look to be caused by differences
    // in the implementation of RENDER on the Sun
    // X server shipped by default. We did not test
    // NX with the XFree86 X server, but it seems
    // that, when using the Sun X server, even plain
    // X clients don't make use of the extension.
    //

    #ifdef __sun

    if (control -> SessionMode == SESSION_X &&
            control -> AgentHideRender == 0)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Not using render extension "
              << "due to incompatible agent libraries.\n"
              << logofs_flush;
      #endif

      control -> AgentHideRender = 1;
    }

    #endif
  }

  return 1;
}

//
// Parameters for MODEM 28.8/33.6/56 Kbps.
//

int SetLinkModem()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Setting parameters for MODEM.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("modem");

  control -> LinkMode = LINK_TYPE_MODEM;

  return 1;
}

//
// Parameters for ISDN 64/128 Kbps.
//

int SetLinkIsdn()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Setting parameters for ISDN.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("isdn");

  control -> LinkMode = LINK_TYPE_ISDN;

  return 1;
}

//
// Parameters for ADSL 256 Kbps.
//

int SetLinkAdsl()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Setting parameters for ADSL.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("adsl");

  control -> LinkMode = LINK_TYPE_ADSL;

  return 1;
}

//
// Parameters for XDSL/FDDI/ATM 1/2/34 Mbps WAN.
//

int SetLinkWan()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Setting parameters for WAN.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("wan");

  control -> LinkMode = LINK_TYPE_WAN;

  return 1;
}

//
// Parameters for LAN 10/100 Mbps.
//

int SetLinkLan()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Setting parameters for LAN.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("lan");

  control -> LinkMode = LINK_TYPE_LAN;

  return 1;
}

//
// Identify requested link characteristics
// and set control parameters accordingly.
//

int SetCompression()
{
  if (strcasecmp(linkSpeedName, "modem") == 0)
  {
    SetCompressionModem();
  }
  else if (strcasecmp(linkSpeedName, "isdn") == 0)
  {
    SetCompressionIsdn();
  }
  else if (strcasecmp(linkSpeedName, "adsl") == 0)
  {
    SetCompressionAdsl();
  }
  else if (strcasecmp(linkSpeedName, "wan") == 0)
  {
    SetCompressionWan();
  }
  else if (strcasecmp(linkSpeedName, "lan") == 0)
  {
    SetCompressionLan();
  }
  else
  {
    return -1;
  }

  if (control -> LocalDeltaCompression < 0)
  {
    control -> LocalDeltaCompression = 1;
  }

  //
  // If we didn't set remote delta compression
  // (as it should always be the case at client
  // side) assume value of local side.
  //

  if (control -> RemoteDeltaCompression < 0)
  {
    control -> RemoteDeltaCompression =
        control -> LocalDeltaCompression;
  }

  //
  // If we didn't set remote compression levels
  // assume values of local side.
  //

  if (control -> RemoteStreamCompression < 0)
  {
    control -> RemoteStreamCompressionLevel =
        control -> LocalStreamCompressionLevel;

    if (control -> RemoteStreamCompressionLevel > 0)
    {
      control -> RemoteStreamCompression = 1;
    }
    else
    {
      control -> RemoteStreamCompression = 0;
    }
  }

  if (control -> RemoteDataCompression < 0)
  {
    control -> RemoteDataCompressionLevel =
        control -> LocalDataCompressionLevel;

    if (control -> RemoteDataCompressionLevel > 0)
    {
      control -> RemoteDataCompression = 1;
    }
    else
    {
      control -> RemoteDataCompression = 0;
    }
  }

  //
  // This is a good idea on embedded platforms.
  // Keep the transport buffers small by forcing
  // write of proxy data to X channels when they
  // are blocked.
  //

  if (control -> LocalMemoryLevel != -1 &&
          control -> LocalMemoryLevel <= 3)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Enabling forced writes of "
            << "data to X connections.\n"
            << logofs_flush;
    #endif

    control -> AgentForceReliable = 1;
  }

  return 1;
}

//
// Compression for MODEM.
//

int SetCompressionModem()
{
  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 1;
    control -> LocalDataCompressionLevel = 6;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 32;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 1;
    control -> LocalStreamCompressionLevel = 9;
  }

  return 1;
}

//
// Compression for ISDN.
//

int SetCompressionIsdn()
{
  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 1;
    control -> LocalDataCompressionLevel = 6;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 32;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 1;
    control -> LocalStreamCompressionLevel = 9;
  }

  return 1;
}

//
// Compression for ADSL.
//

int SetCompressionAdsl()
{
  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 1;
    control -> LocalDataCompressionLevel = 3;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 32;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 1;
    control -> LocalStreamCompressionLevel = 6;
  }

  return 1;
}

//
// Compression for WAN.
//

int SetCompressionWan()
{
  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 1;
    control -> LocalDataCompressionLevel = 1;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 32;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 0;
    control -> LocalStreamCompressionLevel = 0;
  }

  return 1;
}

//
// Compression for LAN.
//

int SetCompressionLan()
{
  //
  // Disable delta compression if not
  // explicitly enabled.
  //

  if (control -> LocalDeltaCompression < 0)
  {
    control -> LocalDeltaCompression = 0;
  }

  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 0;
    control -> LocalDataCompressionLevel = 0;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 0;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 0;
    control -> LocalStreamCompressionLevel = 0;
  }

  //
  // By disabling delta compression we multiply
  // network load by a factor of 10. To increase
  // interactivity we make sure that forwarding
  // of sync messages to the X server is enabled.
  //

  if (control -> LocalDeltaCompression == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Enabling forwarding of sync "
            << "messages to the X server.\n"
            << logofs_flush;
    #endif

    control -> AgentSyncPropagate = 1;
  }

  return 1;
}

int SetLimits()
{
  //
  // Adjust CPU limit. Set limit to 0 if
  // user explicitly disabled limits.
  //

  if (control -> LocalProcessorLimit == -1)
  {
    if (control -> LocalBitrateLimit == 0)
    {
      control -> LocalProcessorLimit = 0;
    }
    else
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        control -> LocalProcessorLimit =
            control -> ClientProcessorLimit;
      }
      else
      {
        control -> LocalProcessorLimit =
            control -> ServerProcessorLimit;
      }
    }
  }

  //
  // Check bitrate limit.
  //

  if (control -> LocalBitrateLimit == -1)
  {
    if (control -> ProxyMode == PROXY_CLIENT)
    {
      control -> LocalBitrateLimit =
          control -> ClientBitrateLimit;
    }
    else
    {
      control -> LocalBitrateLimit =
          control -> ServerBitrateLimit;
    }
  }
  else if (control -> LocalBitrateLimit > 0)
  {
    //
    // If bandwidth limit was explicitly set by
    // user, make CPU limit more restrictive at
    // X client side.
    //

    if (control -> ProxyMode == PROXY_CLIENT)
    {
      control -> LocalProcessorLimit *= 2;

      if (control -> LocalProcessorLimit == 0)
      {
        control -> LocalProcessorLimit = 1;
      }
    }
  }

  //
  // Finally make CPU limits less restrictive
  // in case of RFB or RDP sessions.
  //

  if (control -> LocalProcessorLimit > 1 &&
          (control -> SessionMode == SESSION_RDP ||
              control -> SessionMode == SESSION_RFB))
  {
    control -> LocalProcessorLimit /= 2;
  }    

  return 1;
}

//
// Compression for connection to localhost.
//

int ParseControlParameters(const char *link)
{
  //
  // Open file and get configuration parameters.
  //

  char file[DEFAULT_STRING_LENGTH];

  strcpy(file, "control");

  strcat(file, "-");

  strcat(file, link);

  istream *stream = NULL;

  if (OpenInputFile(file, stream) < 0)
  {
    delete stream;

    HandleCleanup();
  }

  char buffer[CONTROL_PARAMETERS_FILE_LENGTH];

  int parameters[CONTROL_PARAMETERS_FILE_ENTRIES];

  int count = 0;

  //
  // A quite rudimentary way to parse a configuration
  // file. Control files, however, are just intended
  // for developers. Reading them from disk should be
  // disabled on production machines.
  //

  while (stream -> fail() == 0 &&
             stream -> getline(buffer, DEFAULT_STRING_LENGTH))
  {
    #ifdef TEST
    *logofs << "Loop: Parsing line '" << buffer << "'.\n"
            << logofs_flush;
    #endif

    if (count < CONTROL_PARAMETERS_FILE_ENTRIES)
    {
      if (strlen(buffer) > 0)
      {
        parameters[count] = atoi(buffer + 4);
      }
      else
      {
        parameters[count] = 0;
      }
    }

    count++;
  }

  if (count > 0 && count != CONTROL_PARAMETERS_FILE_ENTRIES)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Bad control file '" << file
            << "'. Read " << count << " values instead of "
            << CONTROL_PARAMETERS_FILE_ENTRIES << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Bad control file '" << file
         << "'. Read " << count << " values instead of "
         << CONTROL_PARAMETERS_FILE_ENTRIES << ".\n";

    delete stream;

    HandleCleanup();
  }

  delete stream;

  //
  // If we didn't locate the file then
  // get buffer from control class.
  //

  if (count == 0)
  {
    *file = '\0';

    //
    // LOCAL basic configuration is
    // used for both 'wan' and 'lan'.
    //

    if (!strcmp(link, "modem"))
    {
      strcpy(buffer, ControlModem);
    }
    else if (!strcmp(link, "isdn"))
    {
      strcpy(buffer, ControlIsdn);
    }
    else if (!strcmp(link, "adsl"))
    {
      strcpy(buffer, ControlAdsl);
    }
    else if (!strcmp(link, "wan"))
    {
      strcpy(buffer, ControlLocal);
    }
    else if (!strcmp(link, "lan"))
    {
      strcpy(buffer, ControlLocal);
    }

    char *line = strtok(buffer, "\n");

    while (line != NULL)
    {
      #ifdef TEST
      *logofs << "Loop: Parsing line '" << line << "'.\n"
              << logofs_flush;
      #endif

      if (strlen(line) > 0)
      {
        parameters[count] = atoi(line + 4);
      }
      else
      {
        parameters[count] = 0;
      }

      line = strtok(NULL, "\n");

      count++;
    }

    if (count != CONTROL_PARAMETERS_FILE_ENTRIES)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Bad control parameters file. "
              << "Read " << count << " values instead of "
              << CONTROL_PARAMETERS_FILE_ENTRIES << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Bad control parameters file. "
           << "Read " << count << " values instead of "
           << CONTROL_PARAMETERS_FILE_ENTRIES << ".\n";

      HandleCleanup();
    }
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Reading control parameters from '"
            << file << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Reading control parameters from '"
         << file << "'.\n";
  }

  //
  // Scheduling parameters.
  //

  control -> ClientBytesInARowLimit    = parameters[7];
  control -> ClientTimeInARowLimit     = parameters[8];
  control -> ClientReadsInARowLimit    = parameters[9];
  control -> ClientMessagesInARowLimit = parameters[10];
  control -> ServerBytesInARowLimit    = parameters[11];
  control -> ServerTimeInARowLimit     = parameters[12];
  control -> ServerReadsInARowLimit    = parameters[13];
  control -> ServerMessagesInARowLimit = parameters[14];

  //
  // Split options.
  //

  control -> SplitTimeout               = parameters[18];
  control -> SplitDataThreshold         = parameters[19];
  control -> SplitDataPacketLimit       = parameters[20];
  control -> SplitTotalStorageSizeLimit = parameters[21];

  //
  // Timeout parameters.
  //

  control -> PendingTimeout = parameters[25];
  control -> WakeupTimeout  = parameters[26];

  //
  // Set flush timeout according to user's wish.
  // If parameter is zero or one use the default
  // for timeout (thus enabling message queuing)
  // but later enable immediate flushes.
  //

  if (useFlush != -1)
  {
    if (useFlush > 1)
    {
      control -> FlushTimeout = useFlush;
    }
    else
    {
      control -> FlushTimeout = parameters[27];
    }
  }
  else
  {
    control -> FlushTimeout = parameters[27];
  }

  control -> MotionTimeout     = parameters[28];
  control -> CongestionTimeout = parameters[29];
  control -> LimitTimeout      = parameters[30];
  control -> PingTimeout       = parameters[31];
  control -> StartupTimeout    = parameters[32];

  //
  // Transport parameters.
  //

  control -> TransportWriteThreshold     = parameters[36];
  control -> TransportXBufferLimit       = parameters[37];
  control -> TransportProxyBufferLimit   = parameters[38];
  control -> TransportGenericBufferLimit = parameters[39];

  //
  // Read buffers parameters.
  //

  control -> ClientInitialReadSize  = parameters[43];
  control -> ClientMaximumReadSize  = parameters[44];
  control -> ServerInitialReadSize  = parameters[45];
  control -> ServerMaximumReadSize  = parameters[46];
  control -> ProxyInitialReadSize   = parameters[47];
  control -> ProxyMaximumReadSize   = parameters[48];
  control -> GenericInitialReadSize = parameters[49];
  control -> GenericMaximumReadSize = parameters[50];

  //
  // Socket options.
  //

  control -> OptionProxySendBuffer    = parameters[54];
  control -> OptionProxyReceiveBuffer = parameters[55];

  //
  // Set TCP_NODELAY according to user's wish.
  //

  if (useNoDelay != -1)
  {
    control -> OptionProxyClientNoDelay = useNoDelay;
    control -> OptionProxyServerNoDelay = useNoDelay;
  }
  else
  {
    control -> OptionProxyClientNoDelay = parameters[56];
    control -> OptionProxyServerNoDelay = parameters[57];
  }

  //
  // Agent control parameters.
  //

  control -> AgentKarmaSize = parameters[61];
  control -> AgentSplitSize = parameters[62];

  if (packMethod == -1 || packQuality == -1)
  {
    if (packMethod == -1)
    {
      packMethod = parameters[63];
    }

    if (packQuality == -1)
    {
      packQuality = parameters[64];
    }

    if (ParsePackMethod(packMethod, packQuality) < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Unrecognized pack method id "
              << packMethod << " with quality " << packQuality
              << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Unrecognized pack method id "
           << packMethod << " with quality " << packQuality
           << ".\n";

      HandleCleanup();
    }
  }

  control -> AgentFlushPriority = parameters[65];

  if (useFlush != -1)
  {
    if (useFlush == 0)
    {
      control -> AgentFlushImmediate = 1;
    }
    else
    {
      control -> AgentFlushImmediate = 0;
    }
  }
  else
  {
    control -> AgentFlushImmediate = parameters[66];
  }

  control -> AgentSyncPropagate = parameters[67];
  control -> AgentForceReliable = parameters[68];

  //
  // Check if user wants to hide RENDER extension
  // or wants to short-circuit replies at client
  // side. These operations are not agent related,
  // actually, as they also applies to normal X
  // clients.
  //

  if (useRender != -1)
  {
    control -> AgentHideRender = (useRender == 0);
  }
  else
  {
    control -> AgentHideRender = parameters[69];
  }

  if (useTaint != -1)
  {
    control -> AgentTaintReplies = (useTaint == 1);
  }
  else
  {
    control -> AgentTaintReplies = parameters[70];
  }

  //
  // Cleanup of buffers.
  //

  control -> AgentCleanGet    = parameters[74];
  control -> AgentCleanAlloc  = parameters[75];
  control -> AgentCleanFlush  = parameters[76];
  control -> AgentCleanSend   = parameters[77];
  control -> AgentCleanImages = parameters[78];

  //
  // Image parameters.
  //

  control -> AgentImageSplit       = parameters[82];
  control -> AgentImageMask        = parameters[83];
  control -> AgentImageFrame       = parameters[84];
  control -> AgentImageSplitMethod = parameters[85];
  control -> AgentImageMaskMethod  = parameters[86];

  //
  // Apply the new values.
  //

  if (proxy != NULL)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Configuring proxy according to file '"
            << file << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Configuring proxy according to file '"
         << file << "'.\n";

    proxy -> handleLinkConfiguration();
  }

  return 1;
}

//
// These functions are used to parse literal
// values provided by the user and set the
// control parameters accordingly.
//

int ParseCacheOption(const char *opt)
{
  int size = ParseArg("", "cache", opt);

  if (size < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid value '"
            << opt << "' for option 'cache'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid value '"
         << opt << "' for option 'cache'.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Loop: Setting size of cache to "
          << size << " bytes.\n" << logofs_flush;
  #endif

  control -> ClientTotalStorageSizeLimit = size;
  control -> ServerTotalStorageSizeLimit = size;

  strcpy(cacheSizeName, opt);

  if (size == 0)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Disabling NX delta compression.\n"
            << logofs_flush;
    #endif

    control -> LocalDeltaCompression = 0;

    #ifdef WARNING
    *logofs << "Loop: WARNING! Disabling use of NX persistent cache.\n"
            << logofs_flush;
    #endif

    control -> PersistentCacheEnableLoad = 0;
    control -> PersistentCacheEnableSave = 0;
  }

  return 1;
}

int ParseImagesOption(const char *opt)
{
  int size = ParseArg("", "images", opt);

  if (size < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid value '"
            << opt << "' for option 'images'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid value '"
         << opt << "' for option 'images'.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Loop: Setting size of images cache to "
          << size << " bytes.\n" << logofs_flush;
  #endif

  control -> ImageCacheDiskLimit = size;

  strcpy(imagesSizeName, opt);

  return 1;
}

int ParseShmemOption(const char *opt)
{
  int size = ParseArg("", "shmem", opt);

  if (size < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid value '"
            << opt << "' for option 'shmem'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid value '"
         << opt << "' for option 'shmem'.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Loop: Setting size of shared memory "
          << "segment to " << size << " bytes.\n"
          << logofs_flush;
  #endif

  control -> AgentShmemClientSize = size;
  control -> AgentShmemServerSize = size;

  strcpy(shmemSizeName, opt);

  return 1;
}

int ParseLimitOption(const char *opt)
{
  int bitrate = ParseArg("", "limit", opt);

  if (bitrate < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid value '"
            << opt << "' for option 'limit'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid value '"
         << opt << "' for option 'limit'.\n";

    return -1;
  }

  strcpy(bitrateLimitName, opt);

  if (bitrate == 0)
  {
    #ifdef TEST
    *logofs << "Loop: Disabling bitrate limit on proxy link.\n"
            << logofs_flush;
    #endif

    control -> LocalBitrateLimit = 0;
  }
  else
  {
    #ifdef TEST
    *logofs << "Loop: Setting bitrate to " << bitrate
            << " bits per second.\n" << logofs_flush;
    #endif

    //
    // Internal representation is in bytes
    // per second.
    //

    control -> LocalBitrateLimit = bitrate >> 3;
  }

  return 1;
}

int ParseHostOption(const char *opt)
{
  #if defined(INFO) || defined (TEST)
  *logofs << "Loop: Trying to parse options string '" << opt
          << "' as a remote NX host.\n" << logofs_flush;
  #endif

  if (opt == NULL || *opt == '\0')
  {
    #ifdef PANIC
    *logofs << "Loop: No host parameter provided.\n"
            << logofs_flush;
    #endif

    return -1;
  }
  else if (strlen(opt) >= DEFAULT_STRING_LENGTH)
  {
    #ifdef TEST
    *logofs << "Loop: Host parameter exceeds length of "
            << DEFAULT_STRING_LENGTH << " characters.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  //
  // Look for a host name followed
  // by a colon followed by port.
  //

  const char *separator = rindex(opt, ':');

  if (separator != NULL)
  {
    const char *check = separator + 1;

    while (*check != '\0' && *check != ',' &&
               *check != '=' && isdigit(*check) != 0)
    {
      check++;
    }

    int port = atoi(separator + 1);

    if (port < 0 || *check != '\0')
    {
      #ifdef TEST
      *logofs << "Loop: Can't identify remote NX port in string '"
              << separator << "'.\n" << logofs_flush;
      #endif

      return -1;
    }
    else if ((proxyPort != DEFAULT_NX_PROXY_PORT && proxyPort != port) ||
                  (xPort != DEFAULT_NX_X_PORT && xPort != port))
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Overriding remote NX port '"
              << proxyPort << "' with new value '"
              << port << "'.\n" << logofs_flush;
      #endif
    }

    xPort     = port;
    proxyPort = port;
  }
  else if (proxyPort == DEFAULT_NX_PROXY_PORT ||
               xPort == DEFAULT_NX_X_PORT)
  {
    //
    // Complain only if ports were not
    // passed by other means.
    //

    #ifdef TEST
    *logofs << "Loop: Can't identify remote NX port in string '"
            << opt << "'.\n" << logofs_flush;
    #endif

    return -1;
  }
  else
  {
    separator = opt + strlen(opt);
  }

  char host[DEFAULT_STRING_LENGTH] = { 0 };

  strncpy(host, opt, strlen(opt) - strlen(separator));

  *(host + strlen(opt) - strlen(separator)) = '\0';

  const char *check = host;

  while (*check != '\0' && *check != ',' &&
             *check != '=')
  {
    check++;
  }

  if (*check != '\0')
  {
    #ifdef TEST
    *logofs << "Loop: Can't identify remote NX host in string '"
            << host << "'.\n" << logofs_flush;
    #endif

    return -1;
  }
  else if (*remoteHost != '\0' && strcmp(remoteHost, host) != 0)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Overriding remote NX host '"
            << remoteHost << "' with new value '"
            << host << "'.\n" << logofs_flush;
    #endif
  }

  strcpy(remoteHost, host);

  return 1;
}

int OpenInputFile(char *name, istream *&stream)
{
  if (name == NULL || *name == '\0')
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Cannot determine name of NX control file.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot determine name of NX control file.\n";

    return -1;
  }
  else if (*name != '/' && *name != '.')
  {
    char *filePath = GetRootPath();

    if (filePath == NULL)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Cannot determine directory of NX control file.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Cannot determine directory of NX control file.\n";

      return -1;
    }
    else if (strlen(filePath) + strlen("/") +
                 strlen(name) + 1 > DEFAULT_STRING_LENGTH)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Full name of NX file '" << name
              << " would exceed length of " << DEFAULT_STRING_LENGTH
              << " characters.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Full name of NX file '" << name
           << " would exceed length of " << DEFAULT_STRING_LENGTH
           << " characters.\n";

      return -1;
    }

    char *file = new char[strlen(filePath) + strlen("/") +
                              strlen(name) + 1];

    //
    // Transform name in a fully qualified name.
    //

    strcpy(file, filePath);
    strcat(file, "/");
    strcat(file, name);

    strcpy(name, file);

    delete [] filePath;
    delete [] file;
  }

  stream = new ifstream(name, ios::in);

  return 1;
}

int OpenOutputFile(char *name, ostream *&stream)
{
  if (name == NULL || *name == '\0')
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: WARNING! No name provided for output. Using standard error.\n"
            << logofs_flush;
    #endif

    if (!stream)
    {
      stream = &cerr;
    }

    return 1;
  }

  if (!stream || stream == &cerr)
  {
    if (*name != '/' && *name != '.')
    {
      char *filePath = GetSessionPath();

      if (filePath == NULL)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Cannot determine directory of NX session file.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Cannot determine directory of NX session file.\n";

        return -1;
      }

      if (strlen(filePath) + strlen("/") +
              strlen(name) + 1 > DEFAULT_STRING_LENGTH)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Full name of NX file '" << name
                << " would exceed length of " << DEFAULT_STRING_LENGTH
                << " characters.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Full name of NX file '" << name
             << " would exceed length of " << DEFAULT_STRING_LENGTH
             << " characters.\n";

        return -1;
      }

      char *file = new char[strlen(filePath) + strlen("/") +
                                strlen(name) + 1];

      //
      // Transform name in a fully qualified name.
      //

      strcpy(file, filePath);
      strcat(file, "/");
      strcat(file, name);

      strcpy(name, file);

      delete [] filePath;
      delete [] file;
    }

    stream = new ofstream(name, ios::app);
  }
  else
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Bad stream provided for output.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Bad stream provided for output.\n";

    return -1;
  }

  return 1;
}

int ReopenOutputFile(char *name, ostream *&stream, int limit)
{
  if (*name != '\0' && limit >= 0)
  {
    struct stat fileStat;

    if (limit > 0)
    {
      if (stat(name, &fileStat) != 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Can't get stats of file '"
                << name  << "'. Error is " << EGET() 
                << " '" << ESTR() << "'.\n" << logofs_flush;
        #endif

        return 0;
      }
      else if (fileStat.st_size < (long) limit)
      {
        return 0;
      }

      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Deleting file '" << name
              << "' with size " << fileStat.st_size
              << ".\n" << logofs_flush;
      #endif
    }

    *stream << flush;

    delete stream;

    unlink(name);

    stream = new ofstream(name, ios::app);

    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Reopened file '" << name
            << "'.\n" << logofs_flush;
    #endif
  }

  return 1;
}

void PrintProcessInfo()
{
  if (internalFD == -1)
  {
    cerr << "\nNXPROXY - Version " << control -> LocalVersionMajor
         << "." << control -> LocalVersionMinor << "."
         << control -> LocalVersionPatch << "\n\n";

    cerr << "Copyright (C) 2001,2003 NoMachine.\n"
         << "See http://www.nomachine.com/ for more information.\n\n";
  }

  //
  // People get confused by the fact that client
  // mode is running on NX server and viceversa.
  // Let's adopt an user-friendly naming conven-
  // tion here.
  //

  cerr << "Info: Proxy running in "
       << (control -> ProxyMode == PROXY_CLIENT ? "server" : "client")
       << " mode with pid '" << getpid() << "'.\n";

  #if defined(INFO) || defined(TEST)

  if (*logFileName != '\0')
  {
    cerr << "Info" << ": Using log file '" << logFileName << "'.\n";
  }

  if (*statFileName != '\0')
  {
    cerr << "Info" << ": Using stat file '" << statFileName << "'.\n";
  }

  #endif
}

void PrintConnectionInfo()
{
  if (control -> ProxyMode == PROXY_CLIENT)
  {
    if (control -> LocalDeltaCompression == 1)
    {
      cerr << "Info" << ": Using cache parameters "
           << control -> MinimumMessageSizeThreshold
           << "/" << control -> MaximumMessageSizeThreshold
           << "/" << control -> LocalTotalStorageSizeLimit / 1024 << "KB"
           << "/" << control -> RemoteTotalStorageSizeLimit / 1024 << "KB"
           << ".\n";

      if (control -> ImageCacheEnableLoad > 0 ||
              control -> ImageCacheEnableSave > 0)
      {
        cerr << "Info" << ": Using image cache parameters "
             << control -> ImageCacheEnableLoad
             << "/" << control -> ImageCacheEnableSave
             << "/" << control -> ImageCacheDiskLimit / 1024 << "KB"
             << ".\n";
      }

      cerr << "Info" << ": Using split parameters "
           << control -> SplitTimeout
           << "/" << control -> SplitDataThreshold
           << "/" << control -> SplitDataPacketLimit
           << "/" << control -> SplitTotalStorageSizeLimit / 1024 << "KB"
           << ".\n";
    }
    else
    {
      cerr << "Info" << ": Not using NX delta compression.\n";
    }

    cerr << "Info" << ": Using "
         << linkSpeedName << " link parameters "
         << control -> ClientBytesInARowLimit
         << "/" << control -> ClientTimeInARowLimit
         << "/" << control -> ClientReadsInARowLimit
         << "/" << control -> ClientMessagesInARowLimit
         << ".\n";

    cerr << "Info" << ": Using agent parameters "
         << control -> AgentKarmaSize
         << "/" << control -> AgentFlushPriority
         << "/" << control -> AgentSyncPropagate
         << "/" << control -> AgentForceReliable
         << ".\n";
  }
  else
  {
    if (control -> LocalDeltaCompression == 1)
    {
      cerr << "Info" << ": Using cache parameters "
           << control -> MinimumMessageSizeThreshold
           << "/" << control -> MaximumMessageSizeThreshold
           << "/" << control -> ClientTotalStorageSizeLimit / 1024 << "KB"
           << "/" << control -> ServerTotalStorageSizeLimit / 1024 << "KB"
           << ".\n";

      if (control -> ImageCacheEnableLoad > 0 ||
              control -> ImageCacheEnableSave > 0)
      {
        cerr << "Info" << ": Using image cache parameters "
             << control -> ImageCacheEnableLoad
             << "/" << control -> ImageCacheEnableSave
             << "/" << control -> ImageCacheDiskLimit / 1024 << "KB"
             << ".\n";
      }
    }
    else
    {
      cerr << "Info" << ": Not using NX delta compression.\n";
    }

    cerr << "Info" << ": Using "
         << linkSpeedName << " link parameters "
         << control -> ServerBytesInARowLimit
         << "/" << control -> ServerTimeInARowLimit
         << "/" << control -> ServerReadsInARowLimit
         << "/" << control -> ServerMessagesInARowLimit
         << ".\n";
  }

  cerr << "Info" << ": Using pack method '"
       << packMethodName << "' with session '"
       << sessionType << "'.\n";

  if (control -> LocalDeltaCompression == 1)
  {
    if (control -> LocalDataCompression == 1)
    {
      cerr << "Info" << ": Using ZLIB data compression level "
           << control -> LocalDataCompressionLevel << ".\n";

      cerr << "Info" << ": Using ZLIB data threshold set to " 
           << control -> LocalDataCompressionThreshold << ".\n";
    }
    else
    {
      cerr << "Info" << ": Not using ZLIB data compression.\n";
    }
  }

  if (control -> LocalStreamCompression == 1)
  {
    cerr << "Info" << ": Using ZLIB stream compression level "
         << control -> LocalStreamCompressionLevel << ".\n";
  }
  else
  {
    cerr << "Info" << ": Not using ZLIB stream compression.\n";
  }

  if (control -> LocalDeltaCompression == 1)
  {
    if (control -> RemoteDataCompression == 1)
    {
      cerr << "Info" << ": Using remote ZLIB data compression level "
           << control -> RemoteDataCompressionLevel << ".\n";
    }
    else
    {
      cerr << "Info" << ": Not using remote ZLIB data compression.\n";
    }
  }

  if (control -> RemoteStreamCompression == 1)
  {
    cerr << "Info" << ": Using remote ZLIB stream compression level "
         << control -> RemoteStreamCompressionLevel << ".\n";
  }
  else
  {
    cerr << "Info" << ": Not using remote ZLIB stream compression.\n";
  }

  if (control -> LocalBitrateLimit > 0)
  {
    cerr << "Info" << ": Using bandwidth limit of "
         << bitrateLimitName << " bits per second.\n";
  }

  if (control -> LocalProcessorLimit > 0)
  {
    cerr << "Info" << ": Using processor idle time ratio of "
         << control -> LocalProcessorLimit << ".\n";
  }
  else if (control -> ProxyMode == PROXY_CLIENT)
  {
    cerr << "Info" << ": Not using processor load limit.\n";
  }

  if (control -> PersistentCacheName != NULL)
  {
    cerr << "Info" << ": Using cache file '"
         << control -> PersistentCachePath << "/"
         << control -> PersistentCacheName << "'.\n";
  }
  else
  {
    if (control -> PersistentCacheEnableLoad == 0 ||
            control -> LocalDeltaCompression == 0)
    {
      cerr << "Info" << ": Not using persistent cache.\n";
    }
    else
    {
      cerr << "Info" << ": No suitable cache file found.\n";
    }
  }

  if (WE_INITIATE_CONNECTION)
  {
    cerr << "Info" << ": Using remote server '" << remoteHost
         << ":" << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
         << "'.\n" << logofs_flush;
  }

  if (control -> ProxyMode == PROXY_CLIENT &&
          useSyncSocket > 0 && syncPort > 0)
  {
    cerr << "Info" << ": Listening for sync connections "
         << "on port '" << syncPort << "'.\n";
  }
  else if (control -> ProxyMode == PROXY_SERVER &&
               syncPort > 0)
  {
    cerr << "Info" << ": Forwarding sync connections "
         << "to port '" << syncPort << "'.\n";
  }

  if (control -> ProxyMode == PROXY_CLIENT &&
          useKeybdSocket > 0 && keybdPort > 0)
  {
    cerr << "Info" << ": Listening for embedded keyboard connections "
         << "on port '" << keybdPort << "'.\n";
  }
  else if (control -> ProxyMode == PROXY_SERVER &&
               keybdPort > 0)
  {
    cerr << "Info" << ": Forwarding embedded keyboard connections "
         << "to port '" << keybdPort << "'.\n";
  }
  if (control -> ProxyMode == PROXY_CLIENT &&
          useSambaSocket > 0 && sambaPort > 0)
  {
    cerr << "Info" << ": Listening for SMB connections "
         << "on port '" << sambaPort << "'.\n";
  }
  else if (control -> ProxyMode == PROXY_SERVER &&
               sambaPort > 0)
  {
    cerr << "Info" << ": Forwarding SMB connections "
         << "to port '" << sambaPort << "'.\n";
  }

  if (control -> ProxyMode == PROXY_CLIENT &&
          useMediaSocket > 0 && mediaPort > 0)
  {
    cerr << "Info" << ": Listening for multimedia connections "
         << "on port '" << mediaPort << "'.\n";
  }
  else if (control -> ProxyMode == PROXY_SERVER &&
               mediaPort > 0)
  {
    cerr << "Info" << ": Forwarding multimedia connections "
         << "to port '" << mediaPort << "'.\n";
  }
}

void PrintResetInfo()
{
  if (WE_INITIATE_CONNECTION)
  {
    cerr << "Info" << ": Using remote server '" << remoteHost
         << ":" << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
         << "'.\n" << logofs_flush;
  }
  else
  {
    cerr << "Info" << ": Using local port '"
         << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
         << "'.\n" << logofs_flush;
  }

  if (control -> PersistentCacheName != NULL)
  {
    cerr << "Info" << ": Using cache file '"
         << control -> PersistentCachePath << "/"
         << control -> PersistentCacheName << "'.\n";
  }
  else
  {
    if (control -> PersistentCacheEnableLoad == 0 ||
            control -> LocalDeltaCompression == 0)
    {
      cerr << "Info" << ": Not using persistent cache.\n";
    }
    else
    {
      cerr << "Info" << ": No suitable cache file found.\n";
    }
  }
}

void PrintVersionInfo()
{
  cerr << "NXPROXY - " << "Version "
       << control -> LocalVersionMajor << "."
       << control -> LocalVersionMinor << "."
       << control -> LocalVersionPatch;

  cerr << endl;
}

void PrintCopyrightInfo()
{
  cerr << endl;

  PrintVersionInfo();

  cerr << endl;

  cerr << GetCopyrightInfo();

  //
  // Print third party's copyright info.
  //

  GetOtherCopyrightInfo();
}

const char *GetArg(int &argi, int argc, const char **argv)
{
  //
  // Skip "-" and flag character.
  //

  const char *arg = argv[argi] + 2;

  if (*arg == 0)
  {
    if (argi + 1 == argc)
    {
      return NULL;
    }
    else
    {
      argi++;

      return (*argv[argi] == '-' ? NULL : argv[argi]);
    }
  }
  else
  {
    return (*arg == '-' ? NULL : arg);
  }
}

int CheckArg(const char *type, const char *name, const char *value)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Parsing " << type << " option '"
          << name << "' with value '" << value
          << "'.\n" << logofs_flush;
  #endif

  if (value == NULL || strstr(value, "=") != NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error in " << type << " option '"
            << name << "'. No value found.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error in " << type << " option '"
         << name << "'. No value found.\n";

    return -1;
  }
  else if (strstr(name, ",") != NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Parse error at " << type << " option '"
            << name << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Parse error at " << type << " option '"
         << name << "'.\n";

    return -1;
  }
  else if (strlen(value) >= DEFAULT_STRING_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Value '" << value << "' of "
            << type << " option '" << name << "' exceeds length of "
            << DEFAULT_STRING_LENGTH << " characters.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Value '" << value << "' of "
         << type << " option '" << name << "' exceeds length of "
         << DEFAULT_STRING_LENGTH << " characters.\n";

    return -1;
  }

  return 1;
}

int ParseArg(const char *type, const char *name, const char *value)
{
  if (!strcasecmp(value, "0"))
  {
    return 0;
  }

  //
  // Find the base factor.
  //

  double base;

  const char *id = value + strlen(value) - 1;

  if (!strcasecmp(id, "g"))
  {
    base = 1024 * 1024 * 1024;
  }
  else if (!strcasecmp(id, "m"))
  {
    base = 1024 * 1024;
  }
  else if (!strcasecmp(id, "k"))
  {
    base = 1024;
  }
  else if (!strcasecmp(id, "b") || isdigit(*id))
  {
    base = 1;
  }
  else
  {
    return -1;
  }

  char *string = new char[strlen(value)];

  strncpy(string, value, strlen(value) - 1);

  *(string + (strlen(value) - 1)) = '\0';

  #ifdef TEST

  *logofs << "Loop: Parsing integer option '" << name
          << "' from string '" << string << "' with base set to ";

  switch (tolower(*id))
  {
    case 'k':
    case 'm':
    case 'g':
    {
      *logofs << (char) toupper(*id);
    }
    break;
  }

  *logofs << ".\n" << logofs_flush;

  #endif

  double result = atof(string) * base;

  if (result < 0 || result > (((unsigned) -1) >> 1))
  {
    delete [] string;

    return -1;
  }

  delete [] string;

  return (int) result;
}

static void PrintUsage(int argc, const char **argv)
{
  cerr << "Error" << ": NX transport initialization failed.\n";
}

static void handleCheckSessionInLoop()
{
  //
  // The kill flag can be:
  //
  // 0: No shutdown procedure has started yet.
  //
  // 1: Cleanup procedure has started and we
  //    are waiting for all channels to be
  //    closed.
  //
  // 2: All channels have been successfully
  //    closed.
  //
  // NX server should send the cleanup signal
  // only after this message has been printed
  // in the session log.
  //

  if (lastKill == 1 && proxy -> getChannels(CHANNEL_X) == 0)
  {
    if (control -> CleanupTimeout > 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Waiting for cleanup timeout to complete.\n"
              << logofs_flush;
      #endif

      cerr << "Info" << ": Waiting for cleanup timeout to complete.\n";
    }
    else
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Waiting for a further signal to complete.\n"
              << logofs_flush;
      #endif

      cerr << "Info" << ": Waiting for a further signal to complete.\n";
    }

    //
    // Don't print messages in the
    // session log a second time.
    //

    lastKill = 2;
  }

  //
  // Let client proxy find out if the agent's channel
  // is gone. This is the normal procedure in case of
  // internal connections with any of X, RDP and RFB
  // agents. 
  // 

  int cleanup = 0;

  if (control -> ProxyMode == PROXY_CLIENT && useInternalSocket == 1 &&
          proxy -> getType(internalFD) == CHANNEL_NONE && lastKill == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: End of session requested by agent "
            << "termination.\n" << logofs_flush;
    #endif

    cerr << "Info" << ": End of session requested by "
         << "agent termination.\n";
 
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Bytes sent so far are "
            << (unsigned long long) control -> getBytesOut()
            << ".\n" << logofs_flush;
    #endif

    if (control -> getBytesOut() < 1024)
    {
      cerr << "Info" << ": Your session has died before reaching "
           << "a usable state.\n";
      cerr << "Info" << ": This can be due to remote X server "
           << "refusing access to client.\n";
      cerr << "Info" << ": Please check authorization provided "
           << "by your X application.\n";
    }

    cleanup = 1;
  }

  //
  // Check if user requested session cleanup by sending
  // a signal to the local proxy. All the other signals
  // are handled in the main loop so we need to reset
  // the value to get ready for the next iteration.
  //

  int signal = 0;

  if (lastSignal != 0)
  {
    switch (lastSignal)
    {
      case SIGCHLD:
      case SIGUSR1:
      case SIGUSR2:
      {
        break;
      }
      case SIGHUP:
      {
        if (control -> EnableShutdownOnSighup == 1)
        {
          signal = lastSignal;

          cleanup = 1;
        }

        break;
      }
      default:
      {
        signal = lastSignal;

        cleanup = 1;

        break;
      }
    }

    lastSignal = 0;
  }

  if (cleanup == 1)
  {
    //
    // The first time termination signal is received
    // disable all further connections, close down
    // any X channel and wait for a second signal.
    //

    if (lastKill == 0)
    {
      //
      // Don't print a message if cleanup is
      // due to normal termination of agent.
      //

      if (signal != 0)
      {
        char *label;

        switch (signal)
        {
          case SIGTERM:
          {
            label = "SIGTERM";

            break;
          }
          case SIGINT:
          {
            label = "SIGINT";

            break;
          }
          case SIGHUP:
          {
            label = "SIGHUP";

            break;
          }
          default:
          {
            label = "Unknown";

            break;
          }
        }

        #if defined(INFO) || defined(TEST)
        *logofs << "Loop: End of session requested by signal '"
                << signal << "' '" << label << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Info" << ": End of session requested by signal '"
             << signal << "' '" << label << "'.\n";
      }

      //
      // Disable any further connection.
      //

      if (useUnixSocket == 1 && *unixSocketName != '\0')
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Loop: Going to remove the Unix domain socket '"
                << unixSocketName << "'.\n" << logofs_flush;
        #endif

        unlink(unixSocketName);
      }

      useUnixSocket  = 0;
      useTCPSocket   = 0;
      useSyncSocket  = 0;
      useKeybdSocket = 0;
      useSambaSocket = 0;
      useMediaSocket = 0;

      //
      // Close all the remaining X channels and
      // let proxies to save their persistent
      // caches on disk.
      //

      proxy -> handleFinish();

      //
      // Run a watchdog process so we can finally
      // give up at the time signal is delivered.
      //

      if (lastWatchdog == 0)
      {
        if (control -> CleanupTimeout > 0)
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Loop: Starting watchdog process with timeout of "
                  << control -> CleanupTimeout / 1000 << " seconds.\n"
                  << logofs_flush;
          #endif

          lastWatchdog = NXWatchdog(control -> CleanupTimeout / 1000);
        }
      }
      else
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Previous watchdog detected "
                << "in shutdown with pid '" << lastWatchdog
                << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Previous watchdog detected "
             << "in shutdown with pid '" << lastWatchdog
             << "'.\n";

        HandleCleanup();
      }

      lastKill = 1;
    }
    else
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Shutting down the link and exiting.\n"
              << logofs_flush;
      #endif

      cerr << "Info" << ": Shutting down the link and exiting.\n";

      proxy -> handleShutdown();

      HandleCleanup();
    }
  }
}

static void handleCheckLimitsInLoop()
{
  //
  // TODO: Should become a counter
  // in statistics in new versions.
  //

  static long int slept = 0;

  #ifdef TEST
  *logofs << "Loop: Bitrate is " << control -> getBitrateInShortFrame()
          << " B/s and " << control -> getBitrateInLongFrame()
          << " B/s in " << control -> ShortBitrateTimeFrame / 1000
          << "/" << control -> LongBitrateTimeFrame / 1000
          << " seconds timeframes.\n" << logofs_flush;
  #endif

  #ifdef TEST
  *logofs << "Loop: CPU time is " << control -> getIdleTime()
          << " Ms in select and " << control -> getReadTime()
          << " in loop.\n" << logofs_flush;
  #endif

  //
  // TODO: By using new NX transport the
  // way in which is calsulated the loop
  // time fails to account time spent in
  // agent.
  //
  // return;
  //

  //
  // Ensure time spent in select is higher
  // than time spent handling messages.
  //

  if (control -> isStartup() == 0)
  {
    //
    // Very preliminary. It should track CPU time in
    // short and long timeframes and not only total
    // time since beginning of session.
    //

    if (control -> LocalProcessorLimit > 0)
    {
      double ratio = control -> getIdleTime() /
                         control -> getReadTime();

      if (ratio < control -> LocalProcessorLimit)
      {
        double offset = control -> LocalProcessorLimit + 1 - ratio;

        if (offset > 1.1)
        {
          offset = 1.1;
        }

        slept += (unsigned int) (pow(50000, offset) / 1000);

        if (slept > 2000)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Sleeping due to "
                  << "select/loop CPU time ratio of "
                  << ratio << ".\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Sleeping due to "
               << "select/loop CPU time ratio of "
               << ratio << ".\n";

          slept %= 2000;
        }

        T_timestamp idleTs = getTimestamp();

        usleep((unsigned int) pow(50000, offset));

        int diffTs = diffTimestamp(idleTs, getTimestamp());

        control -> addIdleTime(diffTs);

        control -> subReadTime(diffTs);

        if (control -> CollectStatistics)
        {
          statistics -> addIdleTime(diffTs);

          statistics -> subReadTime(diffTs);
        }
      }
    }

    //
    // Very preliminary. Because of buffering, we don't
    // jump out of select often enough to guarantee
    // accuracy.
    //

    if (control -> LocalBitrateLimit > 0)
    {
      int reference = (control -> getBitrateInLongFrame() +
                           control -> getBitrateInShortFrame()) / 2;

      if (reference > control -> LocalBitrateLimit)
      {
        double ratio = ((double) reference) /
                           ((double) control -> LocalBitrateLimit);

        if (ratio > 1.2)
        {
          ratio = 1.2;
        }

        slept += (unsigned int) (pow(50000, ratio) / 1000);

        if (slept > 2000)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Sleeping due to "
                  << "reference bitrate of " << reference
                  << " B/s.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Sleeping due to "
               << "reference bitrate of " << reference
               << " B/s.\n";

          slept %= 2000;
        }

        T_timestamp idleTs = getTimestamp();

        usleep((unsigned int) pow(50000, ratio));

        int diffTs = diffTimestamp(idleTs, getTimestamp());

        control -> addIdleTime(diffTs);

        control -> subReadTime(diffTs);

        if (control -> CollectStatistics)
        {
          statistics -> addIdleTime(diffTs);

          statistics -> subReadTime(diffTs);
        }
      }
    }
  }
}

#if defined(INFO) || defined(TEST)

static void handleCheckStateInLoop(int &totalFDs)
{
  #ifdef DUMP
  *logofs << "Loop: Looping to check correctness of internal state.\n"
          << logofs_flush;
  #endif

  int fdLength;
  int fdPending;

  for (int j = 0; j < totalFDs; j++)
  {
    if (j != proxyFD)
    {
      fdPending = proxy -> getPending(j);

      if (fdPending > 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Buffer for X descriptor FD#"
                << j << " has pending bytes to read.\n"
                << logofs_flush;
        #endif
      }

      fdLength = proxy -> getLength(j);

      if (fdLength > 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Buffer for X descriptor FD#"
                << j << " has " << fdLength << " bytes to write.\n"
                << logofs_flush;
        #endif
      }
    }
  }

  fdPending = proxy -> getPending(proxyFD);

  if (fdPending > 0)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Buffer for proxy descriptor FD#"
            << proxyFD << " has pending bytes to read.\n"
            << logofs_flush;
    #endif
  }

  fdLength = proxy -> getLength(proxyFD);

  if (fdLength > 0 && control -> isTimeToFlush())
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Buffer for proxy descriptor FD#"
            << proxyFD << " has " << fdLength << " bytes to write.\n"
            << logofs_flush;
    #endif
  }
}

static void handleCheckSelectInLoop(int &totalFDs, int &readFDs, int &writeFDs,
                                        fd_set &readSet, fd_set &writeSet,
                                            T_timestamp selectTs)
{
  #ifdef TEST
  *logofs << "Loop: Total descriptors are [" << totalFDs
          << "] with [" << readFDs << "] for read and ["
          << writeFDs << "] for write.\n"
          << logofs_flush;
  #endif

  int i;

  if (totalFDs > 0)
  {
    i = 0;

    #ifdef TEST
    *logofs << "Loop: Selected for read are ";
    #endif

    for (int j = 0; j < totalFDs; j++)
    {
      if (FD_ISSET(j, &readSet))
      {
        #ifdef TEST
        *logofs << "[" << j << "]" << logofs_flush;
        #endif

        i++;
      }
    }

    if (i > 0)
    {
      #ifdef TEST
      *logofs << ".\n" << logofs_flush;
      #endif
    }
    else
    {
      #ifdef TEST
      *logofs << "[none].\n" << logofs_flush;
      #endif
    }

    i = 0;

    #ifdef TEST
    *logofs << "Loop: Selected for write are ";
    #endif

    for (int j = 0; j < totalFDs; j++)
    {
      if (FD_ISSET(j, &writeSet))
      {
        #ifdef TEST
        *logofs << "[" << j << "]" << logofs_flush;
        #endif

        i++;
      }
    }

    if (i > 0)
    {
      #ifdef TEST
      *logofs << ".\n" << logofs_flush;
      #endif
    }
    else
    {
      #ifdef TEST
      *logofs << "[none].\n" << logofs_flush;
      #endif
    }
  }

  #ifdef TEST
  *logofs << "Loop: Select timeout is " << selectTs.tv_sec
          << " S and " << (double) selectTs.tv_usec / 1000
          << " Ms.\n" << logofs_flush;
  #endif
}

#endif

static void handleCheckSessionInConnect()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Check session in connect called.\n"
          << logofs_flush;
  #endif

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    HandleAlert(FAILED_PROXY_CONNECTION_CLIENT_ALERT);
  }
  else if (lastDialog == 0)
  {
    HandleAlert(FAILED_PROXY_CONNECTION_SERVER_ALERT);
  }

  handleAlertInLoop();
}

static void handleStatisticsInLoop()
{
  int mode = NO_STATS;

  if (lastSignal == SIGUSR1)
  {
    //
    // Print overall statistics.
    //

    mode = TOTAL_STATS;
  }
  else if (lastSignal == SIGUSR2)
  {
    //
    // Print partial statistics.
    //

    mode = PARTIAL_STATS;
  }

  if (mode == TOTAL_STATS || mode == PARTIAL_STATS)
  {
    if (control -> CollectStatistics &&
            proxy != NULL && statistics != NULL)
    {
      if (ReopenOutputFile(statFileName, statofs, 0) < 0)
      {
        HandleCleanup();
      }

      proxy -> handleStatistics(mode, statofs);
    }
  }
}

static void handleResetInLoop()
{
  //
  // You should override this if you just want
  // to reload control parameters on the fly.
  //

  if (lastSignal == SIGHUP)
  {
    if (control -> EnableRestartOnSighup == 1)
    {
      if (control -> EnableCoreDumpOnAbort == 0 &&
              control -> EnableRestartOnFailure == 1)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Received signal SIGHUP. "
                << "Aborting proxy connection.\n"
                << logofs_flush;
        #endif

        cerr << "Warning" << ": Received signal SIGHUP. "
             << "Aborting proxy connection.\n";

        HandleAbort();
      }
    }
    else if (control -> EnableReconfigOnSighup == 1)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Received signal SIGHUP. "
              << "Reading new control parameters.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Received signal SIGHUP. "
           << "Reading new control parameters.\n";

      SetParameters();

      lastSignal = 0;
    }
    else if (control -> EnableShutdownOnSighup == 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Ignoring signal SIGHUP.\n"
              << logofs_flush;
      #endif

      lastSignal = 0;
    }
    #if defined(INFO) || defined(TEST)
    else
    {
      *logofs << "Loop: Received signal SIGHUP. Going "
              << "to close the proxy connection.\n"
              << logofs_flush;
    }
    #endif
  }
}

static void handleAlertInLoop()
{
  if (lastAlert != 0)
  {
    if (control -> ProxyMode == PROXY_CLIENT)
    {
      //
      // If we are at X client side and server
      // proxy is not responding, we don't have
      // any possibility to interact with user.
      //

      if (lastAlert != CLOSE_DEAD_PROXY_CONNECTION_CLIENT_ALERT &&
              lastAlert != RESTART_DEAD_PROXY_CONNECTION_CLIENT_ALERT &&
                  lastAlert != FAILED_PROXY_CONNECTION_CLIENT_ALERT)
      {
        //
        // Let the server proxy show the dialog.
        //

        if (proxy != NULL &&
                proxy -> handleAlert(lastAlert) < 0)
        {
          HandleShutdown();
        }
      }
    }
    else
    {
      char caption[DEFAULT_STRING_LENGTH];

      strcpy(caption, ALERT_CAPTION_PREFIX);

      int length = strlen(sessionId);

      //
      // Get rid of the trailing MD5 from session id.
      //

      if (length > (MD5_LENGTH * 2 + 1) &&
              *(sessionId + (length - (MD5_LENGTH * 2 + 1))) == '-')
      {
        strncat(caption, sessionId, length - (MD5_LENGTH * 2 + 1));
      }
      else
      {
        strcat(caption, sessionId);
      }

      //
      // Use the display for which
      // we are acting as a proxy.
      // 

      char *display = displayHost;

      int local = 1;

      char *message;
      char *type;

      switch (lastAlert)
      {
        case CLOSE_DEAD_X_CONNECTION_CLIENT_ALERT:
        {
          message = CLOSE_DEAD_X_CONNECTION_CLIENT_ALERT_STRING;
          type    = CLOSE_DEAD_X_CONNECTION_CLIENT_ALERT_TYPE;

          break;
        }
        case CLOSE_DEAD_X_CONNECTION_SERVER_ALERT:
        {
          message = CLOSE_DEAD_X_CONNECTION_SERVER_ALERT_STRING;
          type    = CLOSE_DEAD_X_CONNECTION_SERVER_ALERT_TYPE;

          break;
        }
        case CLOSE_DEAD_PROXY_CONNECTION_SERVER_ALERT:
        {
          message = CLOSE_DEAD_PROXY_CONNECTION_SERVER_ALERT_STRING;
          type    = CLOSE_DEAD_PROXY_CONNECTION_SERVER_ALERT_TYPE;

          break;
        }
        case RESTART_DEAD_PROXY_CONNECTION_SERVER_ALERT:
        {
          message = RESTART_DEAD_PROXY_CONNECTION_SERVER_ALERT_STRING;
          type    = RESTART_DEAD_PROXY_CONNECTION_SERVER_ALERT_TYPE;

          break;
        }
        case CLOSE_UNRESPONSIVE_X_SERVER_ALERT:
        {
          message = CLOSE_UNRESPONSIVE_X_SERVER_ALERT_STRING;
          type    = CLOSE_UNRESPONSIVE_X_SERVER_ALERT_TYPE;

          break;
        }
        case WRONG_PROXY_VERSION_ALERT:
        {
          message = WRONG_PROXY_VERSION_ALERT_STRING;
          type    = WRONG_PROXY_VERSION_ALERT_TYPE;

          break;
        }
        case FAILED_PROXY_CONNECTION_SERVER_ALERT:
        {
          message = FAILED_PROXY_CONNECTION_SERVER_ALERT_STRING;
          type    = FAILED_PROXY_CONNECTION_SERVER_ALERT_TYPE;

          break;
        }
        case MISSING_PROXY_CACHE_ALERT:
        {
          message = MISSING_PROXY_CACHE_ALERT_STRING;
          type    = MISSING_PROXY_CACHE_ALERT_TYPE;

          break;
        }
        default:
        {
          message = INTERNAL_ERROR_ALERT_STRING;
          type    = INTERNAL_ERROR_ALERT_TYPE;

          break;
        }
      }

      if (lastDialog > 0)
      {
        if (kill(lastDialog, SIGTERM) < 0 && EGET() != ESRCH)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Couldn't kill dialog process with pid '"
                  << lastDialog << "'.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Couldn't kill dialog process with pid '"
               << lastDialog << "'.\n";
        }
      }

      lastDialog = NXDialog(caption, message, type, local, display);
    }
  }

  //
  // Reset state.
  //

  lastAlert = 0;
}

static inline void handleWriteProxyEarlyInLoop()
{
  if (proxy -> handleFlush(flush_if_priority) < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Proxy failure in early handleFlush().\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleWriteProxyInLoop()
{
  if (proxy -> handleFlush(flush_if_needed) < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Proxy failure in handleFlush().\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static void handlePingInLoop(int &diffTs)
{
  #if defined(INFO) || defined(TEST)
  if (diffTs >= (control -> PingTimeout -
                     control -> LatencyTimeout))
  {
    *logofs << "Loop: No communication within nearly "
            << control -> PingTimeout / 1000 << " seconds.\n"
            << logofs_flush;
  }
  #endif

  if (proxy -> getShutdown() > 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: End of session requested by "
            << "remote proxy.\n" << logofs_flush;
    #endif

    cerr << "Info" << ": End of session requested by "
         << "remote proxy.\n";
 
    HandleShutdown();
  }
  else if (proxy -> handlePing() < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Proxy failure in handlePing().\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleWakeupChannelsInLoop()
{
  #ifdef DEBUG
  *logofs << "Loop: Going to check wakeup for proxy FD#"
          << proxyFD << ".\n" << logofs_flush;
  #endif

  if (proxy -> handleWakeup() < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Failure handling wakeup for proxy FD#"
            << proxyFD << ".\n" << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleSplitChannelsInLoop()
{
  #ifdef DEBUG
  *logofs << "Loop: Going to check splits for proxy FD#"
          << proxyFD << ".\n" << logofs_flush;
  #endif

  if (proxy -> handleSplit() < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Failure handling splits for proxy FD#"
            << proxyFD << ".\n" << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleMotionChannelsInLoop()
{
  #ifdef DEBUG
  *logofs << "Loop: Going to flush motion events.\n"
          << logofs_flush;
  #endif

  if (proxy -> handleMotion() < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Failure flushing motion events.\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleCongestionChannelsInLoop()
{
  #ifdef DEBUG
  *logofs << "Loop: Going to check congestion state "
          << "for proxy FD#" << proxyFD << ".\n"
          << logofs_flush;
  #endif

  if (proxy -> handleCongestion() < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Failure handling congestion state "
            << "for proxy FD#" << proxyFD << ".\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static void handleLogReopenInLoop(T_timestamp &logsTs, T_timestamp &nowTs)
{
  if (diffTimestamp(logsTs, nowTs) > control -> FileSizeCheckTimeout)
  {
    #ifdef DEBUG
    *logofs << "Loop: Checking size of log file '"
            << logFileName << "'.\n" << logofs_flush;
    #endif

    if (ReopenOutputFile(logFileName, logofs, control -> LogFileSizeLimit) < 0)
    {
      HandleShutdown();
    }

    //
    // Reset to current timestamp.
    //

    logsTs = nowTs;
  }
}

static inline void handleFlushProxyInLoop(int &resultFDs, fd_set &writeSet)
{
  if (resultFDs > 0 && FD_ISSET(proxyFD, &writeSet))
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Descriptor for proxy FD#"
            << proxyFD << " reported to be writable.\n"
            << logofs_flush;
    #endif

    if (proxy -> handleFlush(flush_if_any) < 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Proxy failure in handleFlush().\n"
              << logofs_flush;
      #endif

      HandleShutdown();
    }
  }
}

static inline void handleFlushChannelsInLoop(int &resultFDs, fd_set &writeSet)
{
  #ifdef DEBUG
  *logofs << "Loop: Going to flush channels for proxy FD#"
          << proxyFD << ".\n" << logofs_flush;
  #endif

  if (resultFDs > 0 && proxy -> handleFlush(writeSet) < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Failure flushing channels for proxy FD#"
            << proxyFD << ".\n" << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static void handleAcceptTcpConnectionInLoop(int &fd)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Going to accept new connection on TCP socket FD#"
          << fd << ".\n" << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(tcpFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed. Error is "
            << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed. Error is "
         << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  if (proxy -> handleNewXConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new X connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new X connection.\n";

    close(newFD);

    HandleCleanup();
  }
}

static void handleAcceptUnixConnectionInLoop(int &fd)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Going to accept new connection on UNIX socket FD#"
          << fd << ".\n" << logofs_flush;
  #endif

  sockaddr_un newAddr;

  size_t addrLen = sizeof(sockaddr_un);

  int newFD = accept(unixFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed. Error is "
            << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed. Error is "
         << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  if (proxy -> handleNewXConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new X connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new X connection.\n";

    close(newFD);

    HandleCleanup();
  }
}

//
// Same copy-paste. Write a generic function.
//

static void handleAcceptSyncConnectionInLoop(int &fd)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Going to accept new sync connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(syncFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for sync. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for sync. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  if (proxy -> handleNewSyncConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new sync connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new sync connection.\n";

    close(newFD);
  }
}

static void handleAcceptKeybdConnectionInLoop(int &fd)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Going to accept new embedded keyboard connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(keybdFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for embedded keyboard. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for embedded keyboard. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  if (proxy -> handleNewKeybdConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new embedded keyboard connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new embedded keyboard connection.\n";

    close(newFD);
  }
}

static void handleAcceptSambaConnectionInLoop(int &fd)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Going to accept new samba connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(sambaFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for samba. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for samba. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  if (proxy -> handleNewSambaConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new samba connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new samba connection.\n";

    close(newFD);
  }
}

static void handleAcceptMediaConnectionInLoop(int &fd)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Loop: Going to accept new multimedia connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(mediaFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for multimedia. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for multimedia. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  if (proxy -> handleNewMediaConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new multimedia connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new multimedia connection.\n";

    close(newFD);
  }
}

static inline void handleReadChannelsInLoop(int &resultFDs, int &totalFDs, fd_set &readSet)
{
  #ifdef DUMP
  *logofs << "Loop: Checking descriptors selected for read.\n"
          << logofs_flush;
  #endif

  if (resultFDs > 0)
  {
    if (tcpFD != -1 && FD_ISSET(tcpFD, &readSet))
    {
      handleAcceptTcpConnectionInLoop(tcpFD);
    }

    if (unixFD != -1 && FD_ISSET(unixFD, &readSet))
    {
      handleAcceptUnixConnectionInLoop(unixFD);
    }

    if (syncFD != -1 && FD_ISSET(syncFD, &readSet))
    {
      handleAcceptSyncConnectionInLoop(syncFD);
    }

    if (keybdFD != -1 && FD_ISSET(keybdFD, &readSet))
    {
      handleAcceptKeybdConnectionInLoop(keybdFD);
    }

    if (sambaFD != -1 && FD_ISSET(sambaFD, &readSet))
    {
      handleAcceptSambaConnectionInLoop(sambaFD);
    }

    if (mediaFD != -1 && FD_ISSET(mediaFD, &readSet))
    {
      handleAcceptMediaConnectionInLoop(mediaFD);
    }
  }

  //
  // We must let proxy check each channel even
  // if no descriptor is set as there might be
  // channels with data pending from previous
  // reads.
  //

  #ifdef DEBUG
  *logofs << "Loop: Going to read from channels for proxy FD#"
          << proxyFD << ".\n" << logofs_flush;
  #endif

  if (proxy -> handleRead(readSet) < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Loop: Failure reading from channels for proxy FD#"
            << proxyFD << ".\n" << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleReadProxyInLoop(int &resultFDs, int &totalFDs, fd_set &readSet)
{
  if ((resultFDs > 0 && FD_ISSET(proxyFD, &readSet)) ||
          proxy -> getPending(proxyFD) > 0)
  {
    #ifdef DEBUG
    *logofs << "Loop: Going to read messages from proxy FD#"
            << proxyFD << ".\n" << logofs_flush;
    #endif

    if (proxy -> handleRead() < 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Loop: Failure reading messages from proxy FD#"
              << proxyFD << ".\n" << logofs_flush;
      #endif

      HandleShutdown();
    }
  }
}

static inline void handleSetReadInLoop(fd_set &readSet, int &readFDs)
{
  proxy -> setReadFDs(&readSet, readFDs);
}

static inline void handleSetWriteInLoop(fd_set &writeSet, int &writeFDs)
{
  proxy -> setWriteFDs(&writeSet, writeFDs);
}

static inline void handleSetTimeoutInLoop(T_timestamp &selectTs)
{
  proxy -> setTimeout(selectTs);
}

static inline void handleSetScheduleInLoop()
{
  proxy -> setSchedule();
}

static void handleSetListenersInLoop(fd_set &readSet, int &readFDs)
{
  //
  // Set descriptors of listening sockets.
  //

  if (useTCPSocket)
  {
    FD_SET(tcpFD, &readSet);

    if (tcpFD >= readFDs)
    {
      readFDs = tcpFD + 1;
    }

    #ifdef TEST
    *logofs << "Loop: Selected listener tcpFD = " << tcpFD
            << " readFDs = " << readFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useUnixSocket)
  {
    FD_SET(unixFD, &readSet);

    if (unixFD >= readFDs)
    {
      readFDs = unixFD + 1;
    }

    #ifdef TEST
    *logofs << "Loop: Selected listener unixFD = " << unixFD
            << " readFDs = " << readFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useSyncSocket)
  {
    FD_SET(syncFD, &readSet);

    if (syncFD >= readFDs)
    {
      readFDs = syncFD + 1;
    }

    #ifdef TEST
    *logofs << "Loop: Selected listener syncFD = " << syncFD
            << " readFDs = " << readFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useKeybdSocket)
  {
    FD_SET(keybdFD, &readSet);

    if (keybdFD >= readFDs)
    {
      readFDs = keybdFD + 1;
    }

    #ifdef TEST
    *logofs << "Loop: Selected listener keybdFD = " << keybdFD
            << " readFDs = " << readFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useSambaSocket)
  {
    FD_SET(sambaFD, &readSet);

    if (sambaFD >= readFDs)
    {
      readFDs = sambaFD + 1;
    }

    #ifdef TEST
    *logofs << "Loop: Selected listener sambaFD = " << sambaFD
            << " readFDs = " << readFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useMediaSocket)
  {
    FD_SET(mediaFD, &readSet);

    if (mediaFD >= readFDs)
    {
      readFDs = mediaFD + 1;
    }

    #ifdef TEST
    *logofs << "Loop: Selected listener mediaFD = " << mediaFD
            << " readFDs = " << readFDs << ".\n"
            << logofs_flush;
    #endif
  }
}
