/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2002 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/


#include "Proxy.h"
#include "Session.h"
#include "Process.h"
#include "Protocol.h"
#include "Utilities.h"

#include "NXParameters.h"
#include "NXStates.h"

#ifdef WIN32
#include "ProcessWin32.h"
#else
#include "ProcessUnix.h"
#endif

#if defined( NX_DEBUG )
#include <iostream>
#endif

#include <sys/types.h>
#include <sys/stat.h>

#include <fstream>
using namespace std;

#ifdef NX_PROXY_DEBUG
#undef NX_PROXY_DEBUG
#endif
//#define NX_PROXY_DEBUG



namespace NX
{


Proxy::Proxy( Session* pSession )
{
  mp_session = pSession;
  mp_process = NULL;
  m_buffer = "";
}

Proxy::~Proxy()
{
  DeleteProcess();
}

void Proxy::ClearBuffer()
{
  m_buffer = "";
#if defined( NX_PROXY_DEBUG )
  cout << "NXProxy: buffer cleared." << endl << flush;
#endif
}

void Proxy::DeleteProcess()
{
  if( mp_process != NULL )
  {
    delete mp_process;
    mp_process = NULL;
  }
}

bool Proxy::InitProcess()
{
  if( mp_process != NULL )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: a process already exists." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Cannot create another nxproxy process." );
    return false;
  }

  try
  {
#ifdef WIN32
    mp_process = new ProcessWin32();
#else
    mp_process = new ProcessUnix();
#endif
  }
  catch( bad_alloc& )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: unable to find memory to create process." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Cannot create new nxproxy process." );
    return false;
  }

  return true;
}

bool Proxy::InitConnection()
{
#if defined( NX_DEBUG )
  cout << "NXProxy: initializing..." << endl << flush;
#endif

#if defined( NX_PROXY_DEBUG )
  cout << "NXProxy: creating new process." << endl << flush;
#endif

  if( !InitProcess() )
    return false;

#if defined( NX_PROXY_DEBUG )
  cout << "NXProxy: new process created." << endl << flush;
#endif

  ParametersList& parameters = mp_session->GetParameters();

  string sTmp;

  mp_process->SetRedirectFlags( Process::NX_STDERR | Process::NX_STDOUT );

#if defined( NX_PROXY_DEBUG )
  cout << "NXProxy: set flags for redirection." << endl << flush;
#endif

  sTmp = parameters.GetString( NX_ProxyPath, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_ProxyPath' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_ProxyPath' is empty." );
    return false;
  }
  else
  {
    mp_process->SetProcessName( "nxproxy" );
    mp_process->SetProcessPath( sTmp );
  }

  sTmp = parameters.GetString( NX_ProxyName, "" );
  if( sTmp.empty() )
  {
#if defined( NX_PROXY_DEBUG )
    cout << "NXProxy: set default parameter 'NX_ProxyName'." << endl << flush;
#endif
    mp_process->SetProcessName( "nxproxy" );
  }
  else
    mp_process->SetProcessName( sTmp );
  
  sTmp = parameters.GetString( NX_ProxyMode, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_ProxyMode' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_ProxyMode' is empty." );
    return false;
  }
  else
  {
    string mode( "-" );
    mode += sTmp;
    mp_process->AddArgument( mode );
  }

  sTmp = parameters.GetString( NX_ProxyLogPath, "" );
  if( sTmp.empty() )
  {
#if defined( NX_PROXY_DEBUG )
    cout << "NXProxy: parameter 'NX_ProxyLogPath' is empty." << endl << flush;
#endif
    if( Protocol::SetStandardSessionLogPath( parameters ) )
    {
      sTmp = parameters.GetString( NX_ProxyLogPath, "" );
#if defined( NX_PROXY_DEBUG )
      cout << "NXProxy: set default parameter 'NX_ProxyLogPath'." << endl << flush;
#endif
    }
    else
    {
#if defined( NX_DEBUG )
      cout << "NXProxy: cannot create standard parameter 'NX_ProxyLogPath'." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Cannot create standard parameter 'NX_ProxyLogPath'." );
      return false;
    }
  }

  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_ProxyLogPath' is empty." << endl << flush;
#endif
    return false;
  }
  else
    mp_process->SetDeviceName( sTmp );

  sTmp = parameters.GetString( NX_EncryptionPort, "" );
  if( !sTmp.empty() && parameters.GetBool( NX_EnableEncryption, false ) )
  {
#if defined( NX_PROXY_DEBUG )
    cout << "NXProxy: add parameter 'NX_ProxyEncryptionPort'." << endl << flush;
#endif
    mp_process->AddArgument( "-P" );
    mp_process->AddArgument( sTmp );
  }

  if( !CreateOptions() )
    return false;

  if( !CreateSessionDirectory() )
    return false;

  if( parameters.GetBool( NX_ProxyUseOptionsFile, false ) )
  {
     if( !CreateOptionsFile() )
       return false;

     sTmp = "options=";
     sTmp += parameters.GetString( NX_ProxyOptionsPath, "" );
     sTmp += ":";
     sTmp += parameters.GetString( NX_ProxyPort, "" );

     mp_process->AddArgument( sTmp );
  }
  else
  {
     sTmp = parameters.GetString( NX_ProxyOptions, "" );
     mp_process->AddArgument( sTmp );
  }

  sTmp = parameters.GetString( NX_ProxyDisplay, "" );
  if( !sTmp.empty() )
  {
#if defined( NX_PROXY_DEBUG )
    cout << "NXProxy: add parameter 'NX_ProxyDisplay'." << endl << flush;
#endif
    mp_process->AddArgument( "-D" );
    mp_process->AddArgument( sTmp );
  }

  ClearBuffer();
  mp_session->SetState( NX_ProxyReady );
  return true;
}

bool Proxy::CreateOptions()
{
#if defined( NX_DEBUG )
  cout << "NXProxy: initializing options." << endl << flush;
#endif

  ParametersList& parameters = mp_session->GetParameters();

  string sTmp;

  sTmp = parameters.GetString( NX_SessionId, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_SessionId' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_SessionId' is empty." );
    return false;
  }
  else
    parameters.SetString( NX_ProxySessionId, sTmp );

  sTmp = parameters.GetString( NX_SessionName, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_SessionName' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_SessionName' is empty." );
    return false;
  }
  else
    parameters.SetString( NX_ProxySessionName, sTmp );

  sTmp = parameters.GetString( NX_ProxyConnect, "" );
  if( sTmp.empty() )
  {
    string host = parameters.GetString( NX_ProxyServer, "" );
    if( host.empty() )
    {
#if defined( NX_PROXY_DEBUG )
      cout << "NXProxy: parameter 'NX_ProxyServer' is empty." << endl << flush;
      cout << "NXProxy: set default parameter 'NX_ProxyServer' = 'NX_HostName'." << endl << flush;
#endif
      host = parameters.GetString( NX_HostName, "" );

      if( host.empty() )
      {
#if defined( NX_DEBUG )
        cout << "NXProxy: parameter 'NX_HostName' is empty." << endl << flush;
#endif
        mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_ProxyServer' is empty." );
        return false;
      }
    }

    string port = parameters.GetString( NX_ProxyPort, "" );
    if( port.empty() )
    {
#if defined( NX_DEBUG )
      cout << "NXProxy: parameter 'NX_ProxyPort' is empty." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_ProxyPort' is empty." );
      return false;
    }

    sTmp += host;
    sTmp += string( ":" );
    sTmp += port;

    parameters.SetString( NX_ProxyConnect, sTmp );

  }

  sTmp = parameters.GetString( NX_ProxyCookie, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_ProxyCookie' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_ProxyCookie' is empty." );
    return false;
  }

  sTmp = parameters.GetString( NX_ProxyRoot, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_ProxyRoot' is empty." << endl << flush;
#endif

    sTmp = parameters.GetString( NX_PersonalDirectory, "" );
    if( sTmp.empty() )
    {
#if defined( NX_DEBUG )
      cout << "NXProxy: parameter 'NX_PersonalDirectory' is empty." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_PersonalDirectory' is empty." );
      return false;
    }
    else
    {
#if defined( NX_DEBUG )
      cout << "NXProxy: set default parameter 'NX_ProxyRoot'." << endl << flush;
#endif
      parameters.SetString( NX_ProxyRoot, sTmp );
    }

  }

  /* Not useful since proxy 1.2.2
  sTmp = parameters.GetString( NX_ProxyCacheType, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_ProxyCacheType' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_ProxyCacheType' is empty." );
    return false;
  }
  */
  
  sTmp = parameters.GetString( NX_ProxyNoDelay, "" );
  if( sTmp.empty() )
  {
    sTmp = parameters.GetString( NX_EnableTcpNoDelay, "" );
    if( !sTmp.empty() )
    {
      if( parameters.GetBool( NX_EnableTcpNoDelay, false ) )
        parameters.SetNumber( NX_ProxyNoDelay, 1 );
      else
        parameters.SetNumber( NX_ProxyNoDelay, 0 );
    }
  }

  if( parameters.GetBool( NX_EnableMedia, false ) )
  {
    sTmp = parameters.GetString( NX_ProxyMediaPort, "" );
    if( sTmp.empty() )
    {
#if defined( NX_DEBUG )
      cout << "NXProxy: parameter 'NX_ProxyMediaPort' is empty." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_ProxyMediaPort' is empty." );
      return false;
    }
  }

  if( parameters.GetBool( NX_EnableSamba, false ) )
  {
    sTmp = parameters.GetString( NX_ProxySamba, "" );
    if( sTmp.empty() )
    {
#if defined( NX_DEBUG )
      cout << "NXProxy: parameter 'NX_ProxySamba' is empty." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_ProxySamba' is empty." );
      return false;
    }
  }

  sTmp = Protocol::GetProxyOptions( parameters );

  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: unable to create options." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Unable to create options for proxy." );
    return false;
  }
  else
    parameters.SetString( NX_ProxyOptions, sTmp );

  return true;
}

bool Proxy::CreateOptionsFile()
{
  ParametersList& parameters = mp_session->GetParameters();
  string sTmp = parameters.GetString( NX_ProxyOptionsPath, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_ProxyOptionsPath' is empty." << endl << flush;
#endif
    if( Protocol::SetStandardProxyOptionsPath( parameters ) )
    {
      sTmp = parameters.GetString( NX_ProxyOptionsPath, "" );
#if defined( NX_DEBUG )
      cout << "NXProxy: set default parameter 'NX_ProxyOptionsPath'." << endl << flush;
#endif
    }
    else
    {
#if defined( NX_DEBUG )
      cout << "NXProxy: cannot create standard parameter 'NX_ProxyOptionsPath'." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Cannot create standard parameter 'NX_ProxyOptionsPath'." );
      return false;
    }
  }

  ofstream ofs( sTmp.c_str(), ios::out );
  if( !ofs )
  {
#if defined( NX_DEBUG )
      cout << "NXProxy: cannot create options file (" << sTmp << ")." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Cannot create options file for proxy." );
      return false;
  }

  sTmp = parameters.GetString( NX_ProxyOptions, "" );
  ofs << sTmp << flush;
  ofs.close();

  return true;
}

bool Proxy::CreateSessionDirectory()
{
  ParametersList& parameters = mp_session->GetParameters();
  string sTmp = parameters.GetString( NX_SessionDirectory, "" );;
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: parameter 'NX_SessionDirectory' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_SessionDirectory' is empty." );
    return false;
  }

  Utilities::ConvertSeparatorsInPath( sTmp );

  if( Utilities::CreateDirectory( sTmp ) )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: cannot create directory: '" << sTmp << "'." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "NXProxy: cannot create Session Directory." );
    return false;
  }

  return true;
}

bool Proxy::StartConnection()
{
  if( mp_process == NULL )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: nxproxy process not found." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Process nxproxy is not correctly initialized." );
    return false;
  }

#if defined( NX_DEBUG )
  cout << "NXProxy: starting..." << endl << flush;
#endif

  string sTmp = mp_session->GetParameters().GetString( NX_ProxyLibraryPath, "" );
  if( !sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: using library path '" << sTmp << "'." << endl << flush;
#endif
    Process::SetEnvironment( "LD_LIBRARY_PATH", sTmp );
  }

  if( !mp_process->StartProcess() )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: nxproxy cannot start." << endl << flush;
#endif
    mp_session->SetException( mp_process->GetException() );
    return false;
  }

#if defined( NX_DEBUG )
  cout << "NXProxy: connecting..." << endl << flush;
#endif

  mp_session->SetState( NX_ProxyConnecting );
  return true;
}

bool Proxy::CheckCurrentState( unsigned int timeout )
{
  string line = "";
  int iSize = mp_process->Read( line, timeout );
  if( iSize == -1 )
  {
    mp_session->SetException( mp_process->GetException() );
    return false;
  }

  m_buffer += line;

  if( Protocol::CheckProxyStatus( line, mp_session, this ) == NX_Exception )
    return false;
  else
    return true;
}

bool Proxy::AdvanceConnection( unsigned int timeout )
{
  if( mp_session->GetState() == NX_ProxyReady )
    return StartConnection();

  if( !CheckCurrentState( timeout )  )
    return false;

#if defined( NX_DEBUG )
  if( mp_session->GetState() == NX_ProxyConnected )
    cout << "NXProxy: session started." << endl << flush;
#endif

  return true;
}

bool Proxy::BreakConnection()
{
#if defined( NX_DEBUG )
  cout << "NXProxy: break connection." << endl << flush;
#endif

  if( mp_process == NULL )
  {
#if defined( NX_DEBUG )
    cout << "NXProxy: process is NULL." << endl << flush;
#endif
    return false;
  }

  if( mp_process->IsRunning() )
    if( mp_process->StopProcess() )
      mp_process->WaitProcessTerminate();

  DeleteProcess();

  if( mp_session->GetState() != NX_Exception )
    mp_session->SetState( NX_ProxyConnectionEnd );

  return true;
}


} /* NX */


