/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2002 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/


#include "XServer.h"
#include "Session.h"
#include "Process.h"

#include "NXParameters.h"

#ifdef WIN32
#include "ProcessWin32.h"
#else
#include "ProcessUnix.h"
#endif

#if defined( NX_DEBUG )
#include <iostream>
#endif

using namespace std;



namespace NX
{


XServer::XServer( Session* pSession )
{
  mp_session = pSession;
  mp_process = NULL;
  m_state = NoInit;
}

XServer::~XServer()
{
  DeleteProcess();
}

void XServer::DeleteProcess()
{
  if( mp_process != NULL )
  {
    delete mp_process;
    mp_process = NULL;
    m_state = NoInit;
  }
}

bool XServer::InitProcess()
{
  if( mp_process != NULL )
  {
#if defined( NX_DEBUG )
    cout << "XServer: a process already exists." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Cannot create another XServer process." );
    return false;
  }

  try
  {
#ifdef WIN32
    mp_process = new ProcessWin32();
#else
    mp_process = new ProcessUnix();
#endif
  }
  catch( bad_alloc& )
  {
#if defined( NX_DEBUG )
    cout << "XServer: unable to find memory to create process." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Cannot create new XServer process." );
    return false;
  }

  return true;
}

bool XServer::SetStandardWindowName( Session* pSession )
{
  string sTmp = "";
  string sName = "";
  ParametersList& parameters = pSession->GetParameters();

  sTmp = parameters.GetString( NX_Username, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "XServer: parameter 'NX_Username' is empty." << endl << flush;
#endif
    pSession->SetException( NX_InvalidParameter, "Parameter 'NX_Username' is empty." );
    return false;
  }
  
  sName += sTmp;
  sName += "@";

  sTmp = parameters.GetString( NX_HostName, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "XServer: parameter 'NX_HostName' is empty." << endl << flush;
#endif
    pSession->SetException( NX_InvalidParameter, "Parameter 'NX_HostName' is empty." );
    return false;
  }

  sName += sTmp;

  parameters.SetString( NX_XServerWindowName, sName );

  return true;
}

bool XServer::InitConnection()
{
  string sTmp = "";
  const ParametersList& parameters = mp_session->GetParameters();

  if( !InitProcess() )
    return false;

  sTmp = parameters.GetString( NX_XServerPath, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "XServer: parameter 'NX_XServerPath' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_XServerPath' is empty." );
    DeleteProcess();
    return false;
  }

  mp_process->SetProcessPath( sTmp );

  sTmp = parameters.GetString( NX_XServerName, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "XServer: parameter 'NX_XServerName' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_XServerName' is empty." );
    DeleteProcess();
    return false;
  }

  mp_process->SetProcessName( sTmp );

  sTmp = parameters.GetString( NX_XServerWindowName, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "XServer: add default parameter 'NX_XServerWindowName'." << endl << flush;
#endif
    if( !SetStandardWindowName( mp_session ) )
    {
#if defined( NX_DEBUG )
      cout << "XServer: cannot create default parameter 'NX_XServerWindowName'." << endl << flush;
#endif
      DeleteProcess();
      return false;
    }
  }

  for( unsigned int i = NX_XServer_FirstCustomOption; i <= NX_XServer_LastCustomOption; i++ )
  {
    sTmp = parameters.GetString( i, "" );
    if( !sTmp.empty() )
    {
#if defined( NX_DEBUG )
      cout << "XServer: adding custom parameter '" << sTmp << "'." << endl << flush;
#endif
      mp_process->AddArgument( sTmp );
    }
  }

  mp_process->AddArgument( "-name" );
  mp_process->AddArgument( parameters.GetString( NX_XServerWindowName, "Undefined" ) );


  sTmp = parameters.GetString( NX_XServerPort, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "XServer: parameter 'NX_XServerPort' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_XServerPort' is empty." );
    DeleteProcess();
    return false;
  }

  mp_process->AddArgument( sTmp );

  m_state = Init;

  return true;
}
 
bool XServer::StartConnection()
{
  if( mp_process->StartProcess() )
  {
    m_state = Running;
    return true;
  }
  
  return false;
}

bool XServer::IsRunning()
{
  return( mp_process->WaitInitProcess() );
}
  
bool XServer::CheckCurrentState()
{
  if( mp_process->IsRunning() )
  {
    m_state = Running;
    return true;
  }
  else
  {
    m_state = NotRunning;
    return false;
  }
  
}

bool XServer::AdvanceConnection()
{
  switch( m_state )
  {
  case NoInit:
#if defined( NX_DEBUG )
    cout << "XServer: not init yet, cannot advance connection." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "XServer is not initialized." );
    return false;

  case Init:
    return StartConnection();
    break;

  case Running:
    return CheckCurrentState();
    break;

  case NotRunning:
#if defined( NX_DEBUG )
    cout << "XServer: not running." << endl << flush;
#endif
    break;

  default:
#if defined( NX_DEBUG )
    cout << "XServer: invalid connection state." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "XServer has invalid connection state." );
    DeleteProcess();
    return false;
  };
  
  return true;
}

bool XServer::BreakConnection()
{
#if defined( NX_DEBUG )
  cout << "XServer: break connection." << endl << flush;
#endif

  if( mp_process == NULL )
  {
#if defined( NX_DEBUG )
    cout << "XServer: process is NULL." << endl << flush;
#endif
    return false;
  }

  if( mp_process->IsRunning() )
    if( mp_process->StopProcess() )
      mp_process->WaitProcessTerminate();

  DeleteProcess();

  return true;
}


} /* NX */

