/****************************************************************************
**
** Copyright (C) 1992-2004 Trolltech AS. All rights reserved.
**
** This file is part of the widgets module of the Qt Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef QTEXTEDIT_H
#define QTEXTEDIT_H

#include <qviewport.h>
#include <qtextdocument.h>

#ifdef QT_COMPAT
#include <qtextcursor.h>
#include <qtextformat.h>
#include <qtextobject.h>
#include <qtextlayout.h>
#endif

class QTextCharFormat;
class QStyleSheet;
class QTextDocument;
class QTextCursor;
class QTextBlockFormat;
class QMenu;
class QTextEditPrivate;

class Q_GUI_EXPORT QTextEdit : public QViewport
{
    Q_OBJECT
    Q_DECLARE_PRIVATE(QTextEdit)
    Q_FLAGS(AutoFormatting)
    Q_ENUMS(WordWrap)
    Q_PROPERTY(AutoFormatting autoFormatting READ autoFormatting WRITE setAutoFormatting)
    Q_PROPERTY(bool tabChangesFocus READ tabChangesFocus WRITE setTabChangesFocus)
    Q_PROPERTY(QString documentTitle READ documentTitle)
    Q_PROPERTY(bool undoRedoEnabled READ isUndoRedoEnabled WRITE setUndoRedoEnabled)
    Q_PROPERTY(WordWrap wordWrap READ wordWrap WRITE setWordWrap)
    Q_PROPERTY(int wrapColumnOrWidth READ wrapColumnOrWidth WRITE setWrapColumnOrWidth)
    Q_PROPERTY(bool readOnly READ isReadOnly WRITE setReadOnly)
public:
    enum WordWrap {
        NoWrap,
        WidgetWidth,
        FixedPixelWidth,
        FixedColumnWidth
    };

    enum AutoFormattingFlag {
        AutoNone = 0,
        AutoBulletList = 0x00000001,
        AutoAll = 0xffffffff
    };

    Q_DECLARE_FLAGS(AutoFormatting, AutoFormattingFlag)

    enum CursorAction {
        MoveBackward,
        MoveForward,
        MoveWordBackward,
        MoveWordForward,
        MoveUp,
        MoveDown,
        MoveLineStart,
        MoveLineEnd,
        MoveHome,
        MoveEnd,
        MovePageUp,
        MovePageDown
#ifdef QT_COMPAT
        ,
        MovePgUp = MovePageUp,
        MovePgDown = MovePageDown
#endif
    };

    QTextEdit(QWidget *parent = 0);
    QTextEdit(const QString &text, QWidget *parent = 0);
    virtual ~QTextEdit();

    void setDocument(QTextDocument *document);
    QTextDocument *document() const;

    void setTextCursor(const QTextCursor &cursor);
    QTextCursor textCursor() const;

    bool isReadOnly() const;
    void setReadOnly(bool ro);

    float fontPointSize() const;
    QString fontFamily() const;
    int fontWeight() const;
    bool fontUnderline() const;
    bool fontItalic() const;
    QColor textColor() const;
    QFont currentFont() const;
    Qt::Alignment alignment() const;

    void mergeCurrentCharFormat(const QTextCharFormat &modifier);

    void setCurrentCharFormat(const QTextCharFormat &format);
    QTextCharFormat currentCharFormat() const;

    AutoFormatting autoFormatting() const;
    void setAutoFormatting(AutoFormatting features);

    bool tabChangesFocus() const;
    void setTabChangesFocus(bool b);

    inline QString documentTitle() const
    { return document()->documentTitle(); }

    inline bool isUndoRedoEnabled() const
    { return document()->isUndoRedoEnabled(); }
    inline void setUndoRedoEnabled(bool enable)
    { document()->setUndoRedoEnabled(enable); }

    WordWrap wordWrap() const;
    void setWordWrap(WordWrap wrap);

    int wrapColumnOrWidth() const;
    void setWrapColumnOrWidth(int w);

    bool find(const QString &exp, QTextDocument::FindFlags options = 0,
              QTextDocument::FindDirection direction = QTextDocument::FindForward);

    inline QString toPlainText() const
    { return document()->toPlainText(); }
    inline QString toHtml() const
    { return document()->toHtml(); }

    void append(const QString &text);

    void ensureCursorVisible();

    virtual int heightForWidth(int width) const;

public slots:
    void setFontPointSize(float s);
    void setFontFamily(const QString &fontFamily);
    void setFontWeight(int w);
    void setFontUnderline(bool b);
    void setFontItalic(bool b);
    void setTextColor(const QColor &c);
    void setCurrentFont(const QFont &f);
    void setAlignment(Qt::Alignment a);

    void setPlainText(const QString &text);
    void setHtml(const QString &text);

    void cut();
    void copy();
    void paste();

    void clear();
    void selectAll();

    void insertPlainText(const QString &text);
    void insertHtml(const QString &text);

    void scrollToAnchor(const QString &name);

    void zoomIn(int range = 1);
    void zoomOut(int range = 1);

signals:
    void currentCharFormatChanged(const QTextCharFormat &format);
    void copyAvailable(bool b);
    void selectionChanged();
    void cursorPositionChanged();

protected:
    virtual void timerEvent(QTimerEvent *ev);
    virtual void keyPressEvent(QKeyEvent *ev);
    virtual void resizeEvent(QResizeEvent *);
    virtual void paintEvent(QPaintEvent *ev);
    virtual void mousePressEvent(QMouseEvent *ev);
    virtual void mouseMoveEvent(QMouseEvent *ev);
    virtual void mouseReleaseEvent(QMouseEvent *ev);
    virtual void mouseDoubleClickEvent(QMouseEvent *ev);
    virtual bool focusNextPrevChild(bool next);
    virtual void contextMenuEvent(QContextMenuEvent *ev);
    virtual void dragEnterEvent(QDragEnterEvent *ev);
    virtual void dragMoveEvent(QDragMoveEvent *ev);
    virtual void dropEvent(QDropEvent *ev);
    virtual void focusInEvent(QFocusEvent *ev);
    virtual void focusOutEvent(QFocusEvent *ev);
    virtual void showEvent(QShowEvent *);
    virtual void changeEvent(QEvent *ev);
    virtual void wheelEvent(QWheelEvent *ev);

    virtual QMenu *createPopupMenu(const QPoint &pos);

    virtual void inputMethodEvent(QInputMethodEvent *);
    QVariant inputMethodQuery(Qt::InputMethodQuery property) const;

    QTextEdit(QTextEditPrivate &dd, QWidget *parent);

#ifdef QT_COMPAT
signals:
    QT_MOC_COMPAT void textChanged();
    QT_MOC_COMPAT void undoAvailable(bool b);
    QT_MOC_COMPAT void redoAvailable(bool b);
    QT_MOC_COMPAT void currentFontChanged(const QFont &f);
    QT_MOC_COMPAT void currentColorChanged(const QColor &c);

public:
    QT_COMPAT_CONSTRUCTOR QTextEdit(QWidget *parent, const char *name);
    inline QT_COMPAT bool find(const QString &exp, bool cs, bool wo)
    {
        QTextDocument::FindFlags flags = 0;
        if (cs)
            flags |= QTextDocument::FindCaseSensitively;
        if (wo)
            flags |= QTextDocument::FindWholeWords;
        return find(exp, flags);
    }

    inline QT_COMPAT void sync() {}

    QT_COMPAT void moveCursor(CursorAction action, QTextCursor::MoveMode mode = QTextCursor::MoveAnchor);

    enum KeyboardAction {
        ActionBackspace,
        ActionDelete,
        ActionReturn,
        ActionKill,
        ActionWordBackspace,
        ActionWordDelete
    };

    QT_COMPAT void doKeyboardAction(KeyboardAction action);

    QT_COMPAT void setText(const QString &text);
    QT_COMPAT QString text() const;
    QT_COMPAT void setTextFormat(Qt::TextFormat);
    QT_COMPAT Qt::TextFormat textFormat() const;

    inline QT_COMPAT void setBold(bool b) { setFontWeight(b ? QFont::Bold : QFont::Normal); }
    inline QT_COMPAT void setUnderline(bool b) { setFontUnderline(b); }
    inline QT_COMPAT void setItalic(bool i) { setFontItalic(i); }
    inline QT_COMPAT void setFamily(const QString &family) { setFontFamily(family); }
    inline QT_COMPAT void setPointSize(int size) { setFontPointSize(size); }

    inline QT_COMPAT bool italic() const { return fontItalic(); }
    inline QT_COMPAT bool bold() const { return fontWeight() >= QFont::Bold; }
    inline QT_COMPAT bool underline() const { return fontUnderline(); }
    inline QT_COMPAT QString family() const { return fontFamily(); }
    inline QT_COMPAT int pointSize() const { return (int)(fontPointSize()+0.5); }

    inline QT_COMPAT bool hasSelectedText() const
    { return textCursor().hasSelection(); }
    inline QT_COMPAT QString selectedText() const
    { return textCursor().selectedText(); }

    inline QT_COMPAT bool isUndoAvailable() const
    { return document()->isUndoAvailable(); }
    inline QT_COMPAT bool isRedoAvailable() const
    { return document()->isRedoAvailable(); }

    inline QT_COMPAT void insert(const QString &text)
    { insertPlainText(text); }

    inline QT_COMPAT bool isModified() const
    { return document()->isModified(); }

    inline QT_COMPAT QColor color() const
    { return textColor(); }

public slots:
    inline QT_MOC_COMPAT void setModified(bool m = true)
    { document()->setModified(m); }
    inline QT_MOC_COMPAT void undo() const
    { document()->undo(); }
    inline QT_MOC_COMPAT void redo() const
    { document()->redo(); }

    inline QT_MOC_COMPAT void setColor(const QColor &c)
    { setTextColor(c); }

#endif

private:
    Q_DISABLE_COPY(QTextEdit)
    Q_PRIVATE_SLOT(d, void update(const QRect &r))
    Q_PRIVATE_SLOT(d, void updateCurrentCharFormatAndSelection())
    Q_PRIVATE_SLOT(d, void adjustScrollbars())
    Q_PRIVATE_SLOT(d, void emitCursorPosChanged(const QTextCursor &))
};

#endif // QTEXTEDIT_H
