
{
open Parser

exception Eof

let check_reserved = function
    | "algorithm" -> ALGORITHM
    | "and" -> AND
    | "annotation" -> ANNOTATION
    | "assert" -> ASSERT
    | "block" -> BLOCK
    | "class" -> CLASS
    | "connect" -> CONNECT
    | "connector" -> CONNECTOR
    | "constant" -> CONSTANT
    | "discrete" -> DISCRETE
    | "else" -> ELSE
    | "elseif" -> ELSEIF
    | "elsewhen" -> ELSEWHEN
    | "encapsulated" -> ENCAPSULATED
    | "end" -> END
    | "equation" -> EQUATION
    | "extends" -> EXTENDS
    | "external" -> EXTERNAL
    | "false" -> FALSE
    | "final" -> FINAL
    | "flow" -> FLOW
    | "for" -> FOR
    | "function" -> FUNCTION
    | "if" -> IF
    | "import" -> IMPORT
    | "in" -> IN
    | "inner" -> INNER
    | "input" -> INPUT
    | "loop" -> LOOP
    | "model" -> MODEL
    | "not" -> NOT
    | "or" -> OR
    | "outer" -> OUTER
    | "output" -> OUTPUT
    | "package" -> PACKAGE
    | "parameter" -> PARAMETER
    | "partial" -> PARTIAL
    | "protected" -> PROTECTED
    | "public" -> PUBLIC
    | "record" -> RECORD
    | "redeclare" -> REDECLARE
    | "replaceable" -> REPLACEABLE
    | "terminate" -> TERMINATE
    | "then" -> THEN
    | "true" -> TRUE
    | "type" -> TYPE
    | "when" -> WHEN
    | "while" -> WHILE
    | "within" -> WITHIN
    | s -> IDENT s

}

let blank = [' ' '\t' '\r']
let digit = ['0'-'9']
let nondigit = ['_' 'A'-'Z' 'a'-'z']
let schar = [^'\"' '\\']
let sescape = "\\\'" | "\\\"" | "\\?" | "\\\\" | "\\a" | "\\b" | "\\f" |
              "\\n" | "\\r" | "\\t" | "\\v"

let unsigned_integer = digit+

let unsigned_number =
  unsigned_integer ('.' unsigned_integer?)?
    (('e' | 'E')('+' | '-')? unsigned_integer)?

rule token = parse

    | blank
                { token lexbuf }

    | ['\n']
                { token lexbuf }


    | "/*" ( [^ '*'] | '*'+ [^ '*' '/'] )* '*'+ '/'
                { let lxm = Lexing.lexeme lexbuf in
                  token lexbuf }

    | "//" [^ '\n']* '\n'
                { let lxm = Lexing.lexeme lexbuf in
                  token lexbuf }

    | unsigned_integer
                { let lxm = Lexing.lexeme lexbuf in
                    UNSIGNED_INTEGER lxm }

    | unsigned_number
                { let lxm = Lexing.lexeme lexbuf in
                    UNSIGNED_NUMBER lxm }

    | nondigit (nondigit | digit)*
                { let lxm = Lexing.lexeme lexbuf in
                    check_reserved lxm }

    | '\"' (schar | sescape)* '\"'
                { let lxm = Lexing.lexeme lexbuf in
                    STRING (String.sub lxm 1 (String.length lxm - 2)) }

    | '('           { LP }
    | ')'           { RP }
    | '['           { LSB }
    | ']'           { RSB }
    | '{'           { LCB }
    | '}'           { RCB }

    | '.'           { DOT }
    | ','           { CM }
    | ';'           { SC }
    | ':'           { CL }

    | '+'           { PLUS }
    | '-'           { MINUS }
    | '*'           { STAR }
    | '/'           { SLASH }
    | '^'           { EXP }

    | '='           { EQ }
    | ":="          { COLEQ }

    | '<'           { LT }
    | '>'           { GT }
    | "<="          { LE }
    | ">="          { GE }
    | "=="          { EE }
    | "<>"          { NE }

    | eof           { EOF }
