/** @file scim_m17n_imengine.cpp
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@tsinghua.org.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_m17n_imengine.cpp,v 1.13 2005/04/04 10:17:00 suzhe Exp $
 */

#define Uses_SCIM_UTILITY
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_LOOKUP_TABLE
#define Uses_SCIM_CONFIG_BASE
#define Uses_STL_MAP

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <scim.h>
#include "scim_m17n_imengine.h"

#define scim_module_init m17n_LTX_scim_module_init
#define scim_module_exit m17n_LTX_scim_module_exit
#define scim_imengine_module_init m17n_LTX_scim_imengine_module_init
#define scim_imengine_module_create_factory m17n_LTX_scim_imengine_module_create_factory

#define SCIM_CONFIG_IMENGINE_M17N_UUID    "/IMEngine/M17N/UUID-"

#define SCIM_PROP_STATUS                  "/IMEngine/M17N/Status"

#ifndef SCIM_M17N_ICON_FILE
    #define SCIM_M17N_ICON_FILE           (SCIM_ICONDIR "/scim-m17n.png")
#endif

static const size_t  __m17n_nr_uuids = 128;
static const char   *__m17n_uuids [] = {
"de94c8e7-6021-4787-a7ed-0d505bf9eea3",
"31cfe859-8015-4067-9b75-1615bd54c72d",
"a3e7a2f8-2f39-4586-b851-80a857b1557d",
"adfbf67c-156d-44bd-95a7-5827286d6500",
"682a7366-3f4b-4198-8c42-aebc9def0c36",
"68243143-85b7-4d88-8aeb-b5232c34004a",
"a5e616db-63eb-4e10-a15d-4811655af4f3",
"98ec723d-97a3-4cef-9c0a-df9aa62b6e2e",
"f42fbcda-d2e9-459b-82b2-396b187dd5c5",
"7a429b16-8527-4e9e-9783-b30b7d17adaf",
"d5a0cf07-3b36-4c68-b02d-7ef2c78d8a75",
"22879d8e-4824-4f70-947c-c0db8037ecd6",
"65b2743c-c1f0-48c1-be50-ce69a7fa1949",
"8969bed5-eb7b-441c-8d5e-1de5cf463328",
"2523be1e-b11c-4921-a1ec-67e483650039",
"e27d65e3-db40-42ef-a81b-57b762ff7cc0",
"c5af66e1-f017-490c-9598-fe68fe70818f",
"12dc8499-4e99-4237-96f6-555a6d6026f0",
"0f730558-23d2-493a-8bc7-0473ff6125c7",
"3a80941c-2524-4847-9ded-bf44a355f4de",
"3ad143b0-480d-48ed-bfb4-9b9b9ba71c0c",
"415184fb-d645-496f-bbde-f4f8c253395d",
"ae27032d-6c31-4018-9d13-f1839aab6545",
"3d102755-fc08-4e24-9b74-218d044f4a29",
"7dd0fa4a-93ce-466b-8dad-aec41a2a549e",
"c7b0e37d-4ed0-41a9-8e55-6a882889b09c",
"f32952ad-4279-4999-8895-ebe6d6c8aa84",
"b7e3d4e6-e3e0-4a5b-accd-7f911cbdcd36",
"2bd9599d-708a-4af3-957c-d2621bc622a4",
"5b5edc0f-510a-49e0-bcdc-cdb2074ae682",
"d4c08360-d8de-4551-9c9a-e28cb4636a0b",
"338ecf26-9e73-4dda-ac73-29109fdfdc5e",
"992b8610-0173-4bdc-bd0b-5e9016219d33",
"29e26490-02c9-4d43-9668-c043c2e5e3e7",
"f1ccfd6b-88ba-48e8-a359-7df7371bcd82",
"1b76eb2c-7982-47d3-af9c-005577e11b12",
"047a41c8-5da0-4aca-a35f-5f23078840f1",
"0fc676c9-943a-43f4-be05-e7e43b13b96a",
"3da743d7-4d5b-4fe4-aa46-597d176fb452",
"775759d7-49f9-4dd0-a9b6-8ae81bcd899c",
"234f1592-3180-480c-acf3-2c68e8a4a302",
"339f0bfe-b947-4f09-b0b0-2682e34872e3",
"c68ad9a6-8404-4575-9aba-67de51fe6d60",
"4d6e74ac-d752-4173-a732-a1f667cb1526",
"b4c8aca2-7cd3-4d3a-a12a-6797fed89851",
"963a15d2-2384-4543-ac43-e3d1e03c0305",
"6fd6a47a-2178-437f-9f9d-1ef70f4bb7d7",
"4b122e93-b6c1-4625-ae4b-1461ee12ade3",
"439ee607-2e5d-44eb-a164-a6a3122bd5b9",
"d15a79a9-1f21-438c-9d25-22dca6a3ad38",
"c19fc0fa-2cfd-449e-82d9-29b07da79b8a",
"df4e3287-581b-4ce8-9e6a-baebebb0b1a3",
"84457221-9757-46d4-87c9-27a39b7542cc",
"9900e41f-9295-45bc-9cb0-440e4198ad46",
"dea96ada-ea99-4e8a-a0cc-d1b6ae56d702",
"fe9856d3-1776-47b8-b393-31e2a84431ad",
"46260747-acd2-4476-864a-8e208e1a48fb",
"16cefe32-1789-4b1d-b3a5-3cc1e15c8128",
"dd95ff09-82da-4a3d-b51d-aaeed259002b",
"d164d5d9-87cb-4ac7-9953-ab28ce37c59f",
"f74c872c-6ebe-4702-8ef7-c5fd271d36cf",
"06155ca5-554b-45a7-a04a-907ea38c2979",
"0a6c1b0c-0c9e-400b-81d0-69b0323fa655",
"c79aed67-6d75-40af-b2d0-93a1cec5c596",
"8f0c5c5b-9121-4962-9475-96767eae586e",
"766656eb-6575-4e16-9d92-c382c1f5f5a2",
"ff41d2ed-cb47-49b0-920e-06abd0ee277c",
"7b49235d-7550-4cef-b1d1-0a62fad4bd2c",
"34cecab5-1f22-431f-a895-e855cbd41bf0",
"2adfb2f1-6242-4730-996a-18447081670c",
"3d08446a-389c-4c98-aa84-13960d5cfd3e",
"92fef7d9-14e1-44df-b5c5-8bd3e4d364af",
"99bf476f-7746-4b9c-9940-d6f4ca1fcef5",
"fc738e82-98fc-47a6-87d0-9de582e9f625",
"8f4b8524-93f6-4792-ac03-f9ca5f0e49ef",
"c3848b56-0e3c-454c-b859-dddb52a22c42",
"f5117076-707e-41d5-aef3-381027b020bb",
"e1dddc6b-c268-4dd5-8629-1d1bdff8efe2",
"329e9e57-780f-4b1e-8c96-e206cae70fc6",
"3ec7a83a-883c-4c6d-9f34-f616d7f66227",
"9b0989dc-047d-4186-93d7-59c4a67792d3",
"c4366de0-64bd-4672-8613-bf89c269e125",
"5f11f45c-6545-4d3b-ad95-dc02a1827e55",
"52ed1916-f753-4eca-89a4-3dd527f6e7dd",
"5963628b-6cb6-4de5-a531-b05529c3b02f",
"ed49a819-086f-4e48-b60a-815862049f87",
"60a4641f-e585-4940-8142-c853c7c81e5a",
"f85db69b-98b3-42bc-a8f7-c9aff51c93a4",
"a8866a89-1a9d-4584-a562-3731bb678dae",
"ab48f3fb-ec6e-4d16-bb89-845122b75f05",
"5f3881f2-7313-48fa-8ed5-1f3714232829",
"29cbbd59-ac94-461b-8fdd-537c28c50847",
"68ddbf8d-ea8a-4163-96ba-e3f711385507",
"513c0ae8-9a17-4c29-8241-75d803055058",
"0a87a377-f361-40c7-b183-b74cf378d642",
"9a418ce6-0ee6-459d-9f2e-69c1d33a854c",
"d5f5fbcc-2496-41f7-aa5f-d934e88c72fc",
"e439ccdc-d2a2-4949-b658-691be51866fc",
"b5e808c0-72f7-4ba7-a4d7-29d8ee6f33b7",
"022b8e62-5721-4144-8c90-0f87a194c3cb",
"8db6baad-72cb-485f-859d-7a85d3e2ecc7",
"1929ea7b-bdf3-4e63-8c81-ebe95e1274a1",
"f25a4891-1a69-4190-8143-ba2e9d6dfa4e",
"92a12eba-988c-47d4-a473-8a83814a142e",
"38c502a8-8de1-47de-a9c1-f1f9a35f4689",
"740218f0-8aca-433a-88c2-3b2e077df8e4",
"98d3e84e-e78e-40e4-bf4c-bc95bfcb37a5",
"b0bd43a3-c537-4934-802b-a4cccd20b7e7",
"61d4b4ef-b1a1-42fd-9e00-da0e148501f0",
"b29c4400-b880-4a03-9a80-a1d48dcb8b30",
"4d00c0b3-4f1e-4b9e-832d-c5904d48e5d9",
"a1e45eb5-fc35-4455-ab8a-71acbc33d95f",
"e9c7013a-7630-4310-a4a1-c2ec67f76c7a",
"04ab7e89-bdba-4f5e-9442-6f653acf04f2",
"dee60281-0dbb-4d1b-a989-6ee89f4c9250",
"c0ee3079-2c0c-4b4b-894f-beb5404162f5",
"fb17e4fc-9901-4c44-9e7f-7b179276066d",
"d502438f-4e0e-4d39-ab9b-883cf15695c2",
"f3537cf5-6e82-4162-9eca-ffd1a9bd5bb9",
"aa91350b-a0b7-4ce2-ac82-1c208dddcfce",
"2d2d2a6f-64a6-49b4-af11-25c49857bbc6",
"8ac094fb-2b1b-4eeb-be91-4c87f1c7e34b",
"7ab49ede-33da-4ce2-9d6c-7ee83a61cc43",
"1cab3d11-1fe8-4bbe-8759-4844939f486c",
"f36a0c58-f71e-4539-a43e-cad371c71d52",
"f6b7b374-be88-4022-ac10-d5c47998a9b4",
"b6fd6dd7-f4fe-4860-b92d-272f8cba616c",
"d1c01bd6-ad12-42b8-9b73-47e51fcff098"
};

struct M17NInfo {
    String        name;
    String        lang;
    String        uuid;
    MInputMethod *im;
};

static std::vector <M17NInfo>                     __m17n_input_methods;
static std::map <MInputContext *, M17NInstance *> __m17n_input_contexts;

static CommonLookupTable                          __lookup_table;

static ConfigPointer __scim_config (0);

static MConverter *__m17n_converter = 0;

static M17NInstance * __find_instance (MInputContext *ic);

static MSymbol __key_to_symbol (const KeyEvent &key);

extern "C" {
    void scim_module_init (void)
    {
        std::vector <WideString> labels;

        for (size_t i = 0; i < 16; ++i)
            labels.push_back (WideString ());

        __lookup_table.set_candidate_labels (labels);
    }

    void scim_module_exit (void)
    {
        for (size_t i = 0; i < __m17n_input_methods.size (); ++i)
            if (__m17n_input_methods [i].im)
                minput_close_im (__m17n_input_methods [i].im);
        M17N_FINI();
    }

    uint32 scim_imengine_module_init (const ConfigPointer &config)
    {
        SCIM_DEBUG_IMENGINE(1) << "Initialize M17N Engine.\n";

        MPlist *imlist, *elm;
        MSymbol utf8 = msymbol("utf8");
        size_t i;
        size_t count = 0;

        __scim_config = config;

        M17N_INIT();

        __m17n_converter = mconv_buffer_converter(utf8, NULL, 0);

        if (!__m17n_converter) return 0;

        imlist = mdatabase_list(msymbol("input-method"), Mnil, Mnil, Mnil);

        for (elm = imlist; elm && mplist_key(elm) != Mnil; elm = mplist_next(elm)) {
            MDatabase *mdb = (MDatabase *) mplist_value(elm);
            MSymbol *tag = mdatabase_tag(mdb);
            if (tag[1] != Mnil) {
                MInputMethod *im = minput_open_im(tag[1], tag[2], NULL);
                if (im) {
                    const char *im_lang = msymbol_name (im->language);
                    const char *im_name = msymbol_name (im->name);

                    if (im_lang && strlen (im_lang) && im_name && strlen (im_name)) {
                        M17NInfo info;

                        SCIM_DEBUG_IMENGINE(1) << im_lang << "-" << im_name << "\n";

                        info.name = String (im_lang) + String ("-") + String (im_name);
                        info.lang = String (im_lang);
                        info.im = im;

                        __m17n_input_methods.push_back (info);

	                    im->driver.callback_list = M17NInstance::register_callbacks(im->driver.callback_list);

                        if (++count >= __m17n_nr_uuids) break;

                    } else {
                        minput_close_im (im);
                    }
                }
            }
        }

        m17n_object_unref(imlist);

        // Get all uuids.
        for (i = 0; i < count; ++i)
            __m17n_input_methods [i].uuid = config->read (String (SCIM_CONFIG_IMENGINE_M17N_UUID) + __m17n_input_methods [i].name,
                                                          String (""));

        // Allocate uuids for the IMs without uuid.
        for (i = 0; i < count; ++i) {
            if (!__m17n_input_methods [i].uuid.length ()) {
                for (size_t j = 0; j < __m17n_nr_uuids; ++j) {
                    size_t k = 0;
                    for (; k < count; ++k)
                        if (String (__m17n_uuids [j]) == __m17n_input_methods [k].uuid) break;

                    if (k == count) {

                        SCIM_DEBUG_IMENGINE(1) << "Set UUID: " << __m17n_uuids [j] << " -> " << __m17n_input_methods [i].name << "\n";

                        __m17n_input_methods [i].uuid = __m17n_uuids [j];
                        config->write (String (SCIM_CONFIG_IMENGINE_M17N_UUID) + __m17n_input_methods [i].name,
                                       String (__m17n_uuids [j]));
                        break;
                    }
                }
            }
        }

        return count;
    }

    IMEngineFactoryPointer scim_imengine_module_create_factory (uint32 engine)
    {
        if (engine >= __m17n_input_methods.size ()) return 0;

        M17NFactory *factory = 0;

        try {
            factory = new M17NFactory (__m17n_input_methods [engine].im,
                                       __m17n_input_methods [engine].name,
                                       __m17n_input_methods [engine].lang,
                                       __m17n_input_methods [engine].uuid);
        } catch (...) {
            delete factory;
            factory = 0;
        }

        return factory;
    }
}

M17NFactory::M17NFactory (MInputMethod *im,
                          const String &name,
                          const String &lang,
                          const String &uuid)
    : m_im (im),
      m_name (name),
      m_uuid (uuid)
{
    SCIM_DEBUG_IMENGINE(1) << "Create M17N Factory :\n";
    SCIM_DEBUG_IMENGINE(1) << "  Name : " << name << "\n";
    SCIM_DEBUG_IMENGINE(1) << "  Lang : " << lang << "\n";
    SCIM_DEBUG_IMENGINE(1) << "  UUID : " << uuid << "\n";

    if (lang.length () >= 2)
        set_languages (lang);
}

M17NFactory::~M17NFactory ()
{
}

WideString
M17NFactory::get_name () const
{
    return utf8_mbstowcs (String ("M17N-") + m_name);
}

WideString
M17NFactory::get_authors () const
{
    return WideString ();
}

WideString
M17NFactory::get_credits () const
{
    return WideString ();
}

WideString
M17NFactory::get_help () const
{
    return WideString ();
}

String
M17NFactory::get_uuid () const
{
    return m_uuid;
}

String
M17NFactory::get_icon_file () const
{
    return String (SCIM_M17N_ICON_FILE);
}

IMEngineInstancePointer
M17NFactory::create_instance (const String &encoding, int id)
{
    return new M17NInstance (this, encoding, id);
}

M17NInstance::M17NInstance (M17NFactory  *factory,
                            const String &encoding,
                            int           id)
    : IMEngineInstanceBase (factory, encoding, id),
      m_ic (0)
{
    SCIM_DEBUG_IMENGINE(1) << "Create M17N Instance.\n";

    if (factory->m_im)
        m_ic = minput_create_ic (factory->m_im, NULL);

    if (m_ic)
        __m17n_input_contexts [m_ic] = this;
}

M17NInstance::~M17NInstance ()
{
    if (m_ic) {
        __m17n_input_contexts.erase (m_ic);
        minput_destroy_ic (m_ic);
    }
}

bool
M17NInstance::process_key_event (const KeyEvent& key)
{
    if (!m_ic) return false;

    if (key.is_key_release ()) return true;

    MSymbol m17n_key = __key_to_symbol (key);

    if (m17n_key == Mnil) return false;

    SCIM_DEBUG_IMENGINE(2) << "process_key_event. " << msymbol_name (m17n_key) << "\n";

    char buf [1024];
    MText *produced;
    int ret;

    ret = minput_filter(m_ic, m17n_key, NULL);

    if (ret) {
        SCIM_DEBUG_IMENGINE(3) << "minput_filter returns 1\n";
        return true;
    }

    produced = mtext();

    ret = minput_lookup(m_ic, m17n_key, NULL, produced);

    if (ret) {
        SCIM_DEBUG_IMENGINE(3) << "minput_lookup returns 1\n";
    }

    mconv_rebind_buffer(__m17n_converter, (unsigned char *) buf, 1024);
    mconv_encode(__m17n_converter, produced);
    buf[__m17n_converter->nbytes] = 0;
    m17n_object_unref(produced);

    if (strlen (buf)) {
        SCIM_DEBUG_IMENGINE(2) << "commit_string: " << buf << "\n";
        commit_string (utf8_mbstowcs (buf));
    }

    return ret == 0;
}

void
M17NInstance::move_preedit_caret (unsigned int pos)
{
}

void
M17NInstance::select_candidate (unsigned int item)
{
}

void
M17NInstance::update_lookup_table_page_size (unsigned int page_size)
{
}

void
M17NInstance::lookup_table_page_up ()
{
}

void
M17NInstance::lookup_table_page_down ()
{
}

void
M17NInstance::reset ()
{
}

void
M17NInstance::focus_in ()
{
    SCIM_DEBUG_IMENGINE(2) << "focus_in.\n";

    hide_preedit_string ();
    hide_aux_string ();
    hide_lookup_table ();

    PropertyList props;
    Property     prop (String (SCIM_PROP_STATUS),
                       String (""));

    prop.hide ();
    props.push_back (prop);

    register_properties (props);

    preedit_draw_cb (m_ic, Minput_preedit_draw);
    status_draw_cb (m_ic, Minput_status_draw);
    candidates_draw_cb (m_ic, Minput_candidates_draw);
}

void
M17NInstance::focus_out ()
{
    SCIM_DEBUG_IMENGINE(2) << "focus_out.\n";
    minput_reset_ic(m_ic);

    char buf [1024];
    MText *produced;
    int ret;
    produced = mtext();
    ret = minput_lookup(m_ic, Mnil, NULL, produced);

    if (ret) {
        SCIM_DEBUG_IMENGINE(3) << "minput_lookup returns 1\n";
    }

    mconv_rebind_buffer(__m17n_converter, (unsigned char *) buf, 1024);
    mconv_encode(__m17n_converter, produced);
    buf[__m17n_converter->nbytes] = 0;
    m17n_object_unref(produced);

    if (strlen (buf)) {
        SCIM_DEBUG_IMENGINE(2) << "commit_string: " << buf << "\n";
        commit_string (utf8_mbstowcs (buf));
    }
}

void
M17NInstance::trigger_property (const String &property)
{
}

static M17NInstance *
__find_instance (MInputContext *ic)
{
    std::map <MInputContext *, M17NInstance *>::iterator it =
        __m17n_input_contexts.find (ic);

    if (it != __m17n_input_contexts.end ())
        return it->second;

    return 0;
}


MPlist *
M17NInstance::register_callbacks(MPlist *callback_list)
{
    if(!callback_list)
      callback_list = mplist();
 
    mplist_add(callback_list, Minput_preedit_start, (void*)preedit_start_cb);
    mplist_add(callback_list, Minput_preedit_draw,  (void*)preedit_draw_cb);
    mplist_add(callback_list, Minput_preedit_done,  (void*)preedit_done_cb);
    mplist_add(callback_list, Minput_status_start,  (void*)status_start_cb);
    mplist_add(callback_list, Minput_status_draw,   (void*)status_draw_cb);
    mplist_add(callback_list, Minput_status_done,   (void*)status_done_cb);
    mplist_add(callback_list, Minput_candidates_start, (void*)candidates_start_cb);
    mplist_add(callback_list, Minput_candidates_draw,  (void*)candidates_draw_cb);
    mplist_add(callback_list, Minput_candidates_done,  (void*)candidates_done_cb);
 
    return callback_list;
}

void
M17NInstance::preedit_start_cb (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr) {
        SCIM_DEBUG_IMENGINE(2) << "preedit_start_cb.\n";
        this_ptr->show_preedit_string ();
    }
}

void
M17NInstance::preedit_draw_cb  (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr && ic->preedit) {
        SCIM_DEBUG_IMENGINE(2) << "preedit_draw_cb.\n";

        char buf[1024];
        mconv_rebind_buffer(__m17n_converter, (unsigned char *)buf, 1024);
        mconv_encode(__m17n_converter, ic->preedit);
        buf[__m17n_converter->nbytes] = 0;

        WideString wstr = utf8_mbstowcs (buf);
        if (wstr.length ()) {
            AttributeList attrs;

            if (ic->candidate_from < ic->candidate_to && ic->candidate_to <= wstr.length ())
                attrs.push_back (Attribute (ic->candidate_from,
                                            ic->candidate_to - ic->candidate_from,
                                            SCIM_ATTR_DECORATE,
                                            SCIM_ATTR_DECORATE_REVERSE));

            this_ptr->show_preedit_string ();
            this_ptr->update_preedit_string (wstr, attrs);
            this_ptr->update_preedit_caret (ic->cursor_pos);
        } else {
            this_ptr->hide_preedit_string ();
        }
    }
}

void
M17NInstance::preedit_done_cb  (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr) {
        SCIM_DEBUG_IMENGINE(2) << "preedit_done_cb.\n";
        this_ptr->hide_preedit_string ();
    }
}

void
M17NInstance::status_start_cb (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr) {
        SCIM_DEBUG_IMENGINE(2) << "status_start_cb.\n";

        Property prop (String (SCIM_PROP_STATUS), String (""));

        this_ptr->update_property (prop);
    }
}

void
M17NInstance::status_draw_cb (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr && ic->status) {
        SCIM_DEBUG_IMENGINE(2) << "status_draw_cb.\n";

        char buf[1024];
        mconv_rebind_buffer(__m17n_converter, (unsigned char *)buf, 1024);
        mconv_encode(__m17n_converter, ic->status);
        buf[__m17n_converter->nbytes] = 0;

        Property prop (String (SCIM_PROP_STATUS), String (buf));

        this_ptr->update_property (prop);
    }
}

void
M17NInstance::status_done_cb (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr) {
        SCIM_DEBUG_IMENGINE(2) << "status_done_cb.\n";

        Property prop (String (SCIM_PROP_STATUS), String (""));
        prop.hide ();

        this_ptr->update_property (prop);
    }
}

void
M17NInstance::candidates_start_cb (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr) {
        SCIM_DEBUG_IMENGINE(2) << "candidates_start_cb.\n";
        this_ptr->show_lookup_table ();
    }
}

void
M17NInstance::candidates_draw_cb (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr) {
        __lookup_table.clear ();

        SCIM_DEBUG_IMENGINE(2) << "candidates_draw_cb.\n";
        SCIM_DEBUG_IMENGINE(3) << "candidate_index = " << ic->candidate_index << "\n";
        SCIM_DEBUG_IMENGINE(3) << "candidate_from  = " << ic->candidate_from << "\n";
        SCIM_DEBUG_IMENGINE(3) << "candidate_to    = " << ic->candidate_to << "\n";
        SCIM_DEBUG_IMENGINE(3) << "candidate_show  = " << ic->candidate_show << "\n";

        if (ic->candidate_list && ic->candidate_show) {
            MPlist *group;
            MText  *mt;
            char buf [1024];
            int i, len, cur;
            WideString wstr;

            i = 0;
            group = ic->candidate_list;
            while (1) {
                if (mplist_key (group) == Mtext)
                    len = mtext_len ((MText *) mplist_value (group));
                else
                    len = mplist_length ((MPlist*) mplist_value (group));

                if (i + len > ic->candidate_index)
                    break;
                i += len;
                group = mplist_next (group);
            }

            cur = ic->candidate_index - i;

            if (mplist_key (group) == Mtext) {
                mt = (MText *) mplist_value (group);
                mconv_rebind_buffer(__m17n_converter, (unsigned char *)buf, 1024);
                mconv_encode(__m17n_converter, mt);
                buf[__m17n_converter->nbytes] = 0;
                wstr = utf8_mbstowcs (buf);

                for (i = 0; i < wstr.length (); ++i)
                    __lookup_table.append_candidate (wstr [i]);

                __lookup_table.set_page_size (wstr.length ());
            } else {
                MPlist *pl;

                len = 0;
                for (pl = (MPlist *) mplist_value (group); mplist_key (pl) != Mnil; pl = mplist_next (pl)) {
                    mt = (MText *) mplist_value (pl);
                    mconv_rebind_buffer(__m17n_converter, (unsigned char *)buf, 1024);
                    mconv_encode(__m17n_converter, mt);
                    buf[__m17n_converter->nbytes] = 0;
                    wstr = utf8_mbstowcs (buf);

                    __lookup_table.append_candidate (wstr);
                    ++ len;
                }
                __lookup_table.set_page_size (len);
            }

            __lookup_table.set_cursor_pos_in_current_page (cur);
            __lookup_table.show_cursor ();

            this_ptr->update_lookup_table (__lookup_table);
            this_ptr->show_lookup_table ();
        } else {
            this_ptr->hide_lookup_table ();
        }
    }
}

void
M17NInstance::candidates_done_cb (MInputContext *ic, MSymbol command)
{
    M17NInstance *this_ptr = __find_instance (ic);

    if (this_ptr) {
        SCIM_DEBUG_IMENGINE(2) << "candidates_done_cb.\n";
        this_ptr->hide_lookup_table ();
    }
}

static MSymbol
__key_to_symbol (const KeyEvent &key)
{
    int mask = 0;
    String keysym;

    if (key.code >= SCIM_KEY_space && key.code <= SCIM_KEY_asciitilde) {
        int c = key.code;

        if ((c == SCIM_KEY_space) &&
            key.is_shift_down ())
            mask |= SCIM_KEY_ShiftMask;

        if (key.is_control_down ()) {
            if (c >= 'a' && c <= 'z')
                c += 'A' - 'a';
            mask |= SCIM_KEY_ControlMask;
        }

        keysym.push_back (c);
    } else if (key.code >= SCIM_KEY_Shift_L && key.code <= SCIM_KEY_Hyper_R) {
        return Mnil;
    } else {
        if (!scim_key_to_string (keysym, KeyEvent (key.code, 0)))
            return Mnil;
        if (key.is_control_down ())
            mask |= SCIM_KEY_ControlMask;
        if (key.is_shift_down ())
            mask |= SCIM_KEY_ShiftMask;
    }

    if (key.is_alt_down ())
        mask |= SCIM_KEY_AltMask;

    if (!keysym.length ()) return Mnil;

    if (mask & SCIM_KEY_ShiftMask)
        keysym = String ("S-") + keysym;
    if (mask & SCIM_KEY_ControlMask)
        keysym = String ("C-") + keysym;
    if (mask & SCIM_KEY_AltMask)
        keysym = String ("M-") + keysym;

    return msymbol (keysym.c_str ());
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
