/*
 * Switch Event Daemon - Monitor an input device for keyboard switch events
 * and run a command when detected.
 *
 * Copyright 2005 Openedhand Ltd.
 *
 * Author: Richard Purdie <rpurdie@openedhand.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include "input.h"

#define BITS_PER_LONG (sizeof(long) * 8)
#define NBITS(x) ((((x)-1)/BITS_PER_LONG)+1)
#define OFF(x)  ((x)%BITS_PER_LONG)
#define BIT(x)  (1UL<<OFF(x))
#define LONG(x) ((x)/BITS_PER_LONG)
#define test_bit(bit, array)	((array[LONG(bit)] >> OFF(bit)) & 1)

void do_switch_event(int *status, char *cmdbuf, int cmdbufoff) 
{
	int i;

	for (i=0; i<NBITS(SW_MAX); i++)
		sprintf(cmdbuf + cmdbufoff + (i*8), "%08x", status[i]);

	system(cmdbuf);

/*
	switch (fork())
    	{
	case 0:
		execlp ("/bin/sh", "sh", "-c", cmdbuf, (char *)NULL);
      		printf ("Error: Exec of '%s' failed.\n", cmdbuf);
		exit(0);
		break;
	case -1:
		printf ("Error: Fork failed.\n");
		break;
	}
 */
}

void
cleanup_children(int s)
{
        kill(-getpid(), 15);  /* kill every one in our process group  */
        exit(0);
}

void 
install_signal_handlers(void)
{
        signal (SIGCHLD, SIG_IGN);  /* kernel can deal with zombies  */
        signal (SIGINT, cleanup_children);
        signal (SIGQUIT, cleanup_children);
        signal (SIGTERM, cleanup_children);
}


int main (int argc, char **argv)
{
	int fd, rd, i;
	struct input_event ev[64];
	unsigned long bit[EV_MAX][NBITS(KEY_MAX)];
	int switch_bits[NBITS(SW_MAX)];
	char *cmd, *cmdbuf;
	int cmdbuflen, cmdbufoff;

	cmd = getenv("SWITCHEVD_EVENT_SCRIPT");

	if (!cmd) {
//		printf ("Set SWITCHEVD_EVENT_SCRIPT to the script to run on switch events\n");
//		exit (1);
	    cmd = "/usr/bin/switchev.sh";
	}

	if (argc < 2) {
		printf ("Usage: switchevd /dev/input/eventX\n");
		printf ("Where X is the input device to monitor\n");
		exit(1);
	}

	cmdbuflen = strlen(cmd) + strlen(argv[1]) + (NBITS(SW_MAX)*8) + 10;
	cmdbuf = malloc(cmdbuflen);
	if (!cmdbuf) {
		perror("switchevd");
		exit(1);
	}

	cmdbufoff = sprintf(cmdbuf, "%s %s ", cmd, argv[1]);

	if ((fd = open(argv[1], O_RDONLY)) < 0) {
		perror("switchevd");
		exit(1);
	}

	/* Find supported Events */
	memset(bit, 0, sizeof(bit));
	ioctl(fd, EVIOCGBIT(0, EV_MAX), bit[0]);
	
	/* Check Switches are present */
	if (!test_bit(EV_SW, bit[0])) {
		printf("Error: This device doesn't have any switches!\n");
		exit(1);
	}

	/* run as daemon */
	{
		pid_t pid;

		pid = fork();
		switch (pid) {
		case 0:
			break;
		case -1:
			perror("fork of switchev daemon failed");
			exit(1);
		default:
			exit(0);
		}
	}

	install_signal_handlers();

	/* Read Current Switch Status */
	memset(switch_bits, 0, sizeof(switch_bits));
	ioctl(fd, EVIOCGSW(SW_MAX), switch_bits);
	do_switch_event(switch_bits, cmdbuf, cmdbufoff);

	while (1) 
	{
		int pressed=0;
		rd = read(fd, ev, sizeof(struct input_event) * 64);

		if (rd < (int) sizeof(struct input_event)) 
		{
			perror("switchevd: Error reading input device!\n");
			exit(1);
		}

		for (i = 0; i < rd / sizeof(struct input_event); i++)
		{

			if (ev[i].type == EV_SW) {
				ioctl(fd, EVIOCGSW(SW_MAX), switch_bits);
				do_switch_event(switch_bits, cmdbuf, cmdbufoff);
			}
		}
	}
}
