/*
    Name:       eval.c
    Purpose:    evaluates nodes generated by parser for wxBasic
    Author:     David Cuny
    Copyright:  (c) 2002 David Cuny <dcuny@lanset.com>
    Licence:    LGPL
*/

/* compare two strings on the stack */
void wStringCompare( int op, int popFlag )
{
    char    *string1, *string2;

    string2 = wPopString();
    if (popFlag) {
        string1 = wPopString();
    } else {
        string1 = wData[wTos].value.string;
    }

    wMemTest( "stringCompare: string1", string1 );
    wMemTest( "stringCompare: string2", string2 );

    switch( op ) {
    case W_EQ:
        wPushNumber( strcmp(string1,string2) == 0 );
        break;

    case W_NE:
        wPushNumber( strcmp(string1,string2) != 0 );
        break;

    case W_GT:
        wPushNumber( strcmp(string1,string2) == 1 );
        break;
        
    case W_LT:
        wPushNumber( strcmp(string1,string2) == -1 );
        break;

    case W_GE:
        wPushNumber( strcmp(string1,string2) > -1 );
        break;

    case W_LE:
        wPushNumber( strcmp(string1,string2) < 1 );
        break;

    default:
        wFatal( W_RUNTIME, "W_OP_CMP: bad operator" );
    }

    wFree( string2 );
    if (popFlag) {
        wFree( string1 );
    }
}

/* compare two numbers on the stack */
void wNumberCompare( int op, int popFlag )
{
    wNumber  n1, n2;

    n2 = wPopNumber();
    if (popFlag) {
        n1 = wPopNumber();
    } else {
        n1 = wData[wTos].value.number;
    }

    switch (op) {

    case W_EQ:
        wPushNumber( abs(n1 - n2) <= W_ALLOWABLE_ERROR );
        break;

    case W_NE:
        wPushNumber( n1 != n2 );
        break;

    case W_GT:
        wPushNumber( n1 > n2 );
        break;
        
    case W_LT:
        wPushNumber( n1 < n2 );
        break;

    case W_GE:
        wPushNumber( n1 >= n2 );
        break;

    case W_LE:
        wPushNumber( n1 <= n2 );
        break;

    default:
        wFatal( W_RUNTIME, "wNumberCompare: bad operator" );
    }
}


/* string range test for case */
void wStringRangeCompare()
{
    char    *stringHigh, *stringLow, *string;

    stringHigh = wPopString();
    stringLow = wPopString();
    string = wData[wTos].value.string;

    wPushNumber(
        strcmp(string, stringLow) > -1 &&
        strcmp(string, stringHigh) < 1 );

    wFree( stringLow );
    wFree( stringHigh );
}

/* numeric range test for case */
void wNumberRangeCompare()
{
    wNumber  high, low, n;

    high = wPopNumber();
    low = wPopNumber();
    n = wData[wTos].value.number;

    wPushNumber( n >= low && n <= high );
}


/* call a user defined function */
void wCallFunction( wNode *node )
{
    int     priorLocalStart, priorTos, numArgs, localCount, i;
    wSymbol  *s;

    /* return value */
    wPushNumber( 0 );

    /* argc */
    wPushNumber( 0 );

    /* start of local variables */
    priorLocalStart = wLocalStart;
    priorTos = wTos;

    /* mark start of created objects */
    wPushCreateStack( 0 );

    /* evaluate parameters */
    wEval( node->left );

    /* number of args */
    numArgs = (wTos-priorTos);
    wData[priorTos].value.number = (wNumber)numArgs;

    /* get routine */
    s = node->value.symbol;
    if (s->forward) {
        wFatal( W_RUNTIME, "%s has not been defined yet", s->name );
    }

    /* allocate locals */
    localCount = s->optArgs + s->locals;
    for ( i = 0; i < (localCount - numArgs); i++ ) {
        wPushNumber( 0 );
    }

    /* set start of locals */
    wLocalStart = priorTos;

    /* run code */
    wEval( s->code );

    /* exit routine flag set? */
    if (wExitRoutineFlag != 0) {
        wExitRoutineFlag = 0;
    }

    /* destroy created objects */
    wClearCreateStack();

    /* clear locals off stack */
    for ( i = localCount; i > 0; i-- ) {
        /* need to free record? */
        if ( i > numArgs && wData[wTos].datatype == W_DATA_ARRAY) {
            wFreeArray( wData[wTos].value.array );
        }
        wPopNumber();
    }

    /* drop argc */
    wDropTos();

    /* check stack */
    if (priorTos > wTos+1) {
        wFatal( W_RUNTIME, "wCallFunction: %s(): stack underflow", s->name );
    } else if (priorTos < wTos+1) {
        wFatal( W_RUNTIME, "wCallFunction: %s(): garbage on stack", s->name );
    }

    /* drop result? */
    if (node->op == W_OP_FUNCTION_CALL) {
        wDropTos();
    }

    /* restore marker */
    wLocalStart = priorLocalStart;
}


void wEval( wNode *node )
{
    int             i, j;
    wNumber         n1, n2, n3, n4;
    char            *string1;
    wSymbol         *s;
    wNode           *n;
    wArray          *a;

/* debugging */
#ifdef __DEBUG_TRACE__
    wSourceCode     *sc;
#endif

    if (node == NULL) {
        return;
    }

    /* test node */
    wMemTest( "wEval: node is corrupt", node );
    wMemTest( "wEval: node->left is corrupt", node->left );
    wMemTest( "wEval: node->right is corrupt", node->right );

    /* tracing */
    if (node->trace != -1) {
        wRunLineId = node->trace;

/* debugging */
#ifdef __DEBUG_TRACE__
        sc = wFindSource( runLineId );
        wConsole("%d: %s\n", sc->lineNum, sc->text );
#endif

    }

/* debugging */
#ifdef __DEBUG_TRACE__
    wConsole( "Op:%s\n", wOpcodeName[node->op] );
#endif

    switch( node->op ) {
    case NULL:
        break;

    case W_OP_ADD:
        wEval( node->left );
        wEval( node->right );

        /* add or concat? */
        if (wGetStackType( wTos ) == W_DATA_STRING) {
            wBasConcat();
        } else {
            n2 = wPopNumber();
            n1 = wPopNumber();
            wPushNumber( n1 + n2 );
        }
        break;

    case W_OP_AND:
        /* short circuit */
        wEval( node->left );
        if (!wPopNumber()) {
            wPushNumber( (wNumber)0 );
        } else {
            wEval( node->right );
            if (wPopNumber()) {
                wPushNumber( (wNumber)1 );
            } else {
                wPushNumber( (wNumber)0 );
            }
        }
        break;
    
    case W_OP_ARGLIST:
        wEval( node->left );
        wEval( node->right );
        break;

    case W_OP_ARRAY_GET:
        wEval( node->left );     /* indices and index */
        wGetArrayElement( node->value.symbol );
        break;

    case W_OP_ARRAY_PTR:
        wPushArray( node->value.symbol );
        break;

    case W_OP_ARRAY_SET:
        wEval( node->right );    /* value to store */
        wEval( node->left );     /* indices and index */
        wSetArrayElement( node->value.symbol );
        break;

    case W_OP_ASSIGN:

        /* value to be assigned. note the *right* node is used */
        wEval( node->right );

        /* variable to assign to */
        wSetVar( node->value.symbol );
        break;

    case W_OP_BUILTIN:
    case W_OP_BUILTIN_CALL:

        /* for argcount */
        n1 = wTos;

        /* mark start of created objects */
        wPushCreateStack( 0 );

        /* the args */
        wEval( node->left );

        /* builtin symbol */
        s = node->value.symbol;

        if (s == NULL) {
            wFatal( W_RUNTIME, "Builtin function pointer is null");
        }

        /* set args */
        wArgCount = (int)(wTos - n1);

        /* call the c function */
        (*s->builtin)();

        /* destroy created objects */
        wClearCreateStack();

        /* drop result? */
        if (node->op == W_OP_BUILTIN_CALL) {
            wDropTos();
        }

        break;

    case W_OP_CASE_SELECT:
        /* top level of a case statement */

        /* value to compare */
        wEval( node->left );

        /* resolve into real data */
        switch (wGetStackType( wTos )) {
        case W_DATA_NUMBER:
            wPushNumber( wPopNumber() );
            break;

        case W_DATA_STRING:
            wPushString( wPopString() );
            break;

        default:
            wFatal( W_RUNTIME, "OpCaseSelect: can't resolve type %s",
                wDatumName[wGetStackType( wTos )] );
        }

        /* get first test */
        if (node == NULL) {
            /* no tests */
            break;
        } else {
            node = node->right;
        }

        /* walk the chain of cases */
        while (1) {
            /* get a test/action pair */
            n = node->left;

            /* perform the tests */
            wEval( n->left );

            /* true? */
            if (wPopNumber()) {
                /* perform action and leave loop */
                wEval( n->right );
                break;
            }

            /* move to next test */
            node = node->right;
            if (node == NULL) {
                break;
            }
        }

        /* drop the test value from the stack */
        wDropTos();
        break;

    case W_OP_CASE_CMP:
    case W_OP_CASE_RANGE:
    case W_OP_CASE_ELSE:
        /* perform chain of tests until true or end */
        while (1) {
            switch (node->op) {
            case W_OP_CASE_CMP:
                /* value to compare */
                wEval(node->left);

                /* what type of test? */
                switch (wGetStackType( wTos-1 )) {
                case W_DATA_NUMBER:
                    wNumberCompare( node->value.iValue, 0 );
                    break;

                case W_DATA_STRING:
                    wStringCompare( node->value.iValue, 0 );
                    break;

                default:
                    wFatal( W_RUNTIME, "W_OP_CASE_CMP: bad data type" );
                }
                break;

            case W_OP_CASE_RANGE:
                /* values to compare */
                n = node->left;
                wEval( n->left );
                wEval( n->right );

                /* what type of comparison? */
                switch (wGetStackType( wTos-2 )) {
                case W_DATA_NUMBER:
                    wNumberRangeCompare();
                    break;

                case W_DATA_STRING:
                    wStringRangeCompare();
                    break;

                default:
                    wFatal( W_RUNTIME, "W_OP_CASE_RANGE: bad data type" );
                    break;
                }
                break;

            case W_OP_CASE_ELSE:
                /* put true on stack */
                wPushNumber( 1 );
                break;

            default:
                wFatal( W_RUNTIME, "opcode %s found in Case test chain",
                    wOpcodeName[node->op] );
            }

            /* was result true? */
            if (wData[wTos].value.number) {
                /* leave true flag on stack and exit */
                break;
            }

            /* move to next test */
            node = node->right;
            if (node == NULL) {
                /* exit with false flag */
                break;
            }

            /* drop test result flag */
            wDropTos();
        }
        break;

    case W_OP_CLASS_METHOD:
    case W_OP_CLASS_METHOD_CALL:

        /* the args */
        n1 = wTos;

        /* mark start of created objects */
        wPushCreateStack( 0 );

        /* the args */
        wEval( node->right );

        wArgCount = (int)(wTos - n1);
        wMe = 0;
        wRunMethod( node->left->value.symbol->klass, node->value.string );

        /* destroy created objects */
        wClearCreateStack();

        /* drop result? */
        if (node->op == W_OP_CLASS_METHOD_CALL) {
            wDropTos();
        }

        break;

    case W_OP_CLOSE:
        if (node->left == NULL) {
            wFileCloseAll();
        } else {
            wEval( node->left );
            i = (int)wPopNumber();
            wFileClose( i );
        }
        break;

    case W_OP_CMP:
        wEval( node->left );
        wEval( node->right );

        switch(wGetStackType(wTos)) {
        case W_DATA_NUMBER:
            wNumberCompare( node->value.iValue, 1 );
            break;

        case W_DATA_STRING:
            wStringCompare( node->value.iValue, 1 );
            break;

        default:
            wFatal( W_RUNTIME, "W_OP_CMP: can't handle %s datatype",
                wDatumName[wData[wTos].datatype] );
        }
        break;

    case W_OP_COMMA:

        /* optimized for linked lists */
        while (1) {
            /* exit flag set? */
            if (wExitForFlag != 0 ||
                wExitRoutineFlag != 0 ||
                wExitWhileFlag != 0 ||
                wContinueFlag ) {
                break;
            }

            if (node->left != NULL ) {
                wEval( node->left );
            }
    
            /* end of list? */
            if (node->right == NULL ) {
                break;
    
            /* linked list? */
            } else if (node->right->op == W_OP_COMMA) {
                node = node->right;
    
            /* not a list */
            } else {
                wEval( node->right );
                break;
            }
        }
        break;

    case W_OP_CONCAT:
        wEval( node->left );
        wEval( node->right );
        wBasConcat();
        break;

    case W_OP_CONST_GET:
        s = node->value.symbol;
        i = s->stackPos;
        /* this better not be indirected! */
        switch (wData[i].datatype) {
        case W_DATA_STRING:
            wPushStringCopy( wData[i].value.string );
            break;

        case W_DATA_NUMBER:
            wPushNumber( wData[i].value.number );
            break;

        default:
            wFatal( W_RUNTIME, "Can't fetch Const type %s",
                wDatumName[wData[i].datatype] );
            break;
        }
        break;

    case W_OP_CONST_Set:
        s = node->value.symbol;
        wEval( node->left );
        switch (wData[wTos].datatype) {
        case W_DATA_STRING:
            wData[s->stackPos].datatype = W_DATA_STRING;
            wData[s->stackPos].value.string = wData[wTos].value.string;
            wData[wTos].datatype = W_DATA_NUMBER;
            wTos--;
            break;

        case W_DATA_NUMBER:
            wData[s->stackPos].datatype = W_DATA_NUMBER;
            wData[s->stackPos].value.number = wData[wTos].value.number;
            wTos--;
            break;

        default:
            wFatal( W_RUNTIME, "Can't set Const to %s",
                wDatumName[wData[wTos].datatype] );
            break;
        }
        break;

    case W_OP_DELETE:
        wEval( node->left );
        wRunDestructor( (int)wPopNumber(), W_OP_DELETE );
        break;

    case W_OP_DESTROY:
        wEval( node->left );
        wRunDestructor( (int)wPopNumber(), W_OP_DESTROY );
        break;

    case W_OP_CONTINUE:
        wContinueFlag = 1;
        break;

    case W_OP_DIV:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        if (n2 == 0.0) {
            wFatal( W_RUNTIME, "Division by zero" );
        }

        wPushNumber( n1 / n2 );
        break;

    case W_OP_END:
        wShutdown(0);
        break;

    case W_OP_ERASE:
        if (node->left == NULL) {
            /* erase entire array */
            wEraseArray( node->value.symbol );

        } else {
            /* evaluate the indexes */
            wEval( node->left );

            /* erase single element from array */
            wEraseArrayElement( node->value.symbol );
        }
        break;

    case W_OP_EXIT_FOR:
        wExitForFlag = 1;
        break;

    case W_OP_EXIT_ROUTINE:
        wExitRoutineFlag = 1;
        break;

    case W_OP_EXIT_WHILE:
        wExitWhileFlag = 1;
        break;

    case W_OP_FLOAT:
        wPushNumber( node->value.fValue );
        break;

    case W_OP_FOR:
        s = node->value.symbol;
        wEval( node->left );
        n2 = wPopNumber();
        n1 = wPopNumber();

        /* initial value */
        wPushNumber( n3 );
        wSetVar( s );

        for( n3=n1; n3 <= n2; n3++ ) {
            /* set loop variable */
            wPushNumber( n3 );
            wSetVar( s );

            /* run code */
            wEval( node->right );

            /* special exit condition? */
            if (wExitForFlag) {
                wExitForFlag = 0;
                break;
            } else if (wContinueFlag) {
                wContinueFlag = 0;
            }

            /* get loop value (in case it changed) */
            wGetVar( s );
            n3 = wPopNumber();

        }
        break;

    case W_OP_FOR_EACH:
        /* variable to assign */
        s = node->value.symbol;

        /* array to read from */
        a = wGetArray( node->left->value.symbol );

        /* iterate through keys */
        i = 0;
        /* put key on stack, or exit */
        while (wGetDynamicKey( a, i )) {

            /* assign to variable */
            wSetVar( s );

            /* run block */
            wEval( node->right );

            /* next */
            i += 1;
        }
        break;


    case W_OP_FOR_STEP:
        s = node->value.symbol;
        wEval( node->left );
        n3 = wPopNumber();       /* step  */
        n2 = wPopNumber();       /* end   */
        n1 = wPopNumber();       /* start */

        /* which direction? */
        if (n3 > 0) {
            n2 += W_ALLOWABLE_ERROR;
            for( n4=n1; n2 >= n4; n4 += n3 ) {
                /* set loop variable */
                wPushNumber( n4 );
                wSetVar( s );
    
                /* run code */
                wEval( node->right );

                /* special exit condition? */
                if (wExitForFlag) {
                    wExitForFlag = 0;
                    break;
                } else if (wContinueFlag) {
                    wContinueFlag = 0;
                }

                /* get loop value (in case it changed) */
                wGetVar( s );
                n4 = wPopNumber();

            }
        } else {
            n2 -= W_ALLOWABLE_ERROR;
            for( n4=n1; n2 <= n4; n4 += n3 ) {
                /* set loop variable */
                wPushNumber( n4 );
                wSetVar( s );
    
                /* run code */
                wEval( node->right );

                /* special exit condition? */
                if (wExitForFlag) {
                    wExitForFlag = 0;
                    break;
                } else if (wContinueFlag) {
                    wContinueFlag = 0;
                }

                /* get loop value (in case it changed) */
                wGetVar( s );
                n4 = wPopNumber();

            }
        }

        break;

    case W_OP_FUNCTION:
    case W_OP_FUNCTION_CALL:
        wCallFunction( node );
        break;

    case W_OP_INIT_ARRAY:
        if (node->left == NULL) {
            wCreateDynamicArray( node->value.symbol );
        } else {
            wEval( node->left );
            wCreateStaticArray( node->value.symbol );
        }
        break;

    case W_OP_INPUT:
        if (node->value.iValue == 1) {
            wEval( node->left );
            i = (int)wPopNumber();

            string1 = wFileLineInput( i );
            wPushString( string1  );

        } else {
            /* command line */
            wFatal( W_RUNTIME, "Input statement is not currently supported" );

        }
        break;

    case W_OP_INT:
        wPushNumber( node->value.iValue );
        break;

    case W_OP_METHOD:
    case W_OP_METHOD_CALL:

        /* the index */
        wEval( node->left );
        n1 = wPopNumber();

        /* method name */
        string1 = node->value.string;

        /* mark start of created objects */
        wPushCreateStack( 0 );

        /* args */
        n2 = wTos;
        wEval( node->right );
        wArgCount = (int)(wTos - n2);

        /* resolve and run method */
        wResolveMethod( (int)n1, string1 );

        /* drop result? */
        if (node->op == W_OP_METHOD_CALL) {
            wDropTos();
        }

        /* destroy created objects */
        wClearCreateStack();

        break;

    case W_OP_MOD:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        wPushNumber( (long)n1 % (long)n2 );
        break;
    
    case W_OP_MUL:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        wPushNumber( n1 * n2 );
        break;

    case W_OP_OPEN:
        /* file name */
        wEval( node->left );
        string1 = wPopString();

        /* mode */
        i = node->value.iValue;

        /* file number */
        wEval( node->right );
        n1 = wPopNumber();

        wFileOpen( string1, i, (int)n1 );
        wFree( string1 );

        break;

    case W_OP_OR:
        /* short circuit */
        wEval( node->left );
        if (wPopNumber()) {
            wPushNumber( (wNumber)1 );
        } else {
            wEval( node->right );
            if (wPopNumber()) {
                wPushNumber( (wNumber)1 );
            } else {
                wPushNumber( (wNumber)0 );
            }
        }
        break;

    case W_OP_OR_BITS:
        wEval( node->left );
        wEval( node->right);
        i = wPopNumber();
        j = wPopNumber();
        wPushNumber( i | j );
        break;

    case W_OP_IDIV:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        /* check for division by zero */
        if (n2 == 0.0) {
            wFatal( W_RUNTIME, "Division by zero" );
        }

        wPushNumber( floor(n1 / n2) );
        break;

    case W_OP_IF:
        n = node->left;

        /* test */
        wEval( n->left );
        n1 = wPopNumber();
        if (n1 != 0) {
            /* true portion */
            wEval( n->right );
        } else {
            /* false portion */
            wEval( node->right );
        }
        break;

    case W_OP_IF_TRUE:
        wFatal( W_RUNTIME, "W_OP_IF_TRUE: internal error" );
        break;

    case W_OP_IN:
        /* evaluate key */
        wEval( node->left );

        /* look for it in array */
        wPushNumber( wInArray( node->value.symbol ) );

        break;

    case W_OP_INV:
        wFatal( W_RUNTIME, "Inv is not implemented yet" );
        break;
    
    case W_OP_NEGATE:
        wEval( node->left );
        n1 = wPopNumber();
        wPushNumber( -n1 );
        break;

    case W_OP_NEW:
    case W_OP_NEW_TMP:

        /* mark start of created objects */
        wPushCreateStack( 0 );

        /* the args */
        n1 = wTos;
        wEval( node->left );
        wArgCount = (int)(wTos - n1);
        wRunMethod( node->value.symbol->klass, "new" );

        /* destroy created objects *before* adding new object to stack */
        wClearCreateStack();

        /* add new object to create stack? */
        if (node->op == W_OP_NEW_TMP) {
            /* track on stack */
            wCopyStackItem( wTos );
            wPushCreateStack( (int)wPopNumber() );
        }

        break;

    case W_OP_NOOP:
        break;

    case W_OP_NOT:
        wEval( node->left );
        n1 = wPopNumber();
        wPushNumber( !n1 );
        break;

    case W_OP_POWER:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        wPushNumber( pow( n1, n2 ) );
        break;

    case W_OP_PRINT:
        if (node->left == NULL) {
            i = 0;
        } else {
            wEval( node->left );
            i = (int)wPopNumber();
        }

        node = node->right;


        /* empty print statement */
        if (node == NULL) {
            if (i==0) {
                wConsole("\n");
            } else {
                wFilePrintString( i, "\n" );
            }
        }

        /* process nodes */
        while(node != NULL) {

            /* data value */
            if (node->left != NULL) {
                wEval( node->left );
                string1 = wPopString();
                if (i==0) {
                    wConsole("%s", string1 );
                } else {
                    wFilePrintString( i, string1 );
                }
                wFree( string1 );
            }

            /* field delimiter */
            switch (node->value.iValue) {
            case W_PRINT_TAB:
                if (i==0) {
                    wConsole("\t");
                } else {
                    wFilePrintString( i, "\t" );
                }
                break;
            case W_PRINT_NEWLINE:
                if (i==0) {
                    wConsole("\n");
                } else {
                    wFilePrintString( i, "\n" );
                }
                break;
            default:
                /* no action */
                break;
            }

            /* link */
            node = node->right;

        }
        break;


    case W_OP_RETURN_VALUE:
        wEval( node->left );
        wSetReturn();
        wExitRoutineFlag = 1;
        break;


    case W_OP_RETURN_SET_VALUE:
        wEval( node->left );
        wSetReturn();
        break;

    case W_OP_SHL:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        wPushNumber( (long)n1 << (long)n2 );
        break;
    
    case W_OP_SHR:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        wPushNumber( (long)n1 >> (long)n2 );
        break;
    
    case W_OP_STRING:
        wPushStringCopy( node->value.string );
        break;

    case W_OP_SUB:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        wPushNumber( n1 - n2 );
        break;
    
    case W_OP_UNDEFINED:
        wFatal( W_RUNTIME, "Opcode is undefined" );
        break;

    case W_OP_VAR:
        /* check type */
        wGetVar( node->value.symbol );
        if (wGetStackType(wTos) == W_DATA_UNDEFINED) {
            wFatal( W_RUNTIME, "the value of %s is undefined",
                node->value.symbol->name );
        }
        break;

    case W_OP_WHILE:
        while (1) {
            /* test */
            wEval( node->left );
            if (!wPopNumber()){
                break;
            }

            /* code */
            wEval( node->right );

            if (wExitWhileFlag != 0) {
                wExitWhileFlag = 0;
                break;

            } else if (wContinueFlag) {
                wContinueFlag = 0;
            }
        }
        break;


    case W_OP_XOR:
        wEval( node->left );
        wEval( node->right );

        n2 = wPopNumber();
        n1 = wPopNumber();

        wPushNumber( (long)n1 ^ (long)n2 );
        break;

    default:
        wFatal( W_RUNTIME, "Unknown Opcode: %d", node->op );

    }
}
