/*****************************************************************************
 * memcpy.c: These are the MMX/MMX2/SSE optimized versions of memcpy
 *****************************************************************************
 * $Id: memcpy.c,v 1.16 2004/11/22 21:06:27 pingus77 Exp $
 *****************************************************************************
 * Copyright (C) 2001 Keuleu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * Original code:
 *
 * Copyright (C) 2001 the xine project
 * 
 * This file is part of xine, a unix video player.
 * 
 * This code was adapted from Linux Kernel sources by Nick Kurshev to
 * the mplayer program. (http://mplayer.sourceforge.net)
 *
 * Miguel Freitas split the #ifdefs into several specialized functions that
 * are benchmarked at runtime by xine. Some original comments from Nick
 * have been preserved documenting some MMX/SSE oddities.
 * Also added kernel memcpy function that seems faster than glibc one.
 *
 * patch for Gcc 3.4/3.5 Uwe Dffert: adapted by Pingus
 * 
 *****************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "cpu_accel.h"
#include "memcpy.h" 
extern void save_memcpy_method(int method);
extern int debug;
void *(* fast_memcpy)(void *to, const void *from, size_t len);
 
/* Original comments from mplayer (file: aclib.c)
 This part of code was taken by me from Linux-2.4.3 and slightly modified
for MMX, MMX2, SSE instruction set. I have done it since linux uses page aligned
blocks but mplayer uses weakly ordered data and original sources can not
speedup them. Only using PREFETCHNTA and MOVNTQ together have effect!

>From IA-32 Intel Architecture Software Developer's Manual Volume 1,

Order Number 245470:
"10.4.6. Cacheability Control, Prefetch, and Memory Ordering Instructions"

Data referenced by a program can be temporal (data will be used again) or
non-temporal (data will be referenced once and not reused in the immediate
future). To make efficient use of the processor's caches, it is generally
desirable to cache temporal data and not cache non-temporal data. Overloading
the processor's caches with non-temporal data is sometimes referred to as
"polluting the caches".
The non-temporal data is written to memory with Write-Combining semantics.

The PREFETCHh instructions permits a program to load data into the processor
at a suggested cache level, so that it is closer to the processors load and
store unit when it is needed. If the data is already present in a level of
the cache hierarchy that is closer to the processor, the PREFETCHh instruction
will not result in any data movement.
But we should you PREFETCHNTA: Non-temporal data fetch data into location
close to the processor, minimizing cache pollution.

The MOVNTQ (store quadword using non-temporal hint) instruction stores
packed integer data from an MMX register to memory, using a non-temporal hint.
The MOVNTPS (store packed single-precision floating-point values using
non-temporal hint) instruction stores packed floating-point data from an
XMM register to memory, using a non-temporal hint.

The SFENCE (Store Fence) instruction controls write ordering by creating a
fence for memory store operations. This instruction guarantees that the results
of every store instruction that precedes the store fence in program order is
globally visible before any store instruction that follows the fence. The
SFENCE instruction provides an efficient way of ensuring ordering between
procedures that produce weakly-ordered data and procedures that consume that
data.

If you have questions please contact with me: Nick Kurshev: nickols_k@mail.ru.
*/

/*  mmx v.1 Note: Since we added alignment of destinition it speedups
    of memory copying on PentMMX, Celeron-1 and P2 upto 12% versus
    standard (non MMX-optimized) version.
    Note: on K6-2+ it speedups memory copying upto 25% and
          on K7 and P3 about 500% (5 times). 
*/

/* Additional notes on gcc assembly and processors: [MF]
prefetch is specific for AMD processors, the intel ones should be
prefetch0, prefetch1, prefetch2 which are not recognized by my gcc.
prefetchnta is supported both on athlon and pentium 3.

therefore i will take off prefetchnta instructions from the mmx1 version
to avoid problems on pentium mmx and k6-2.  

quote of the day:
"Using prefetches efficiently is more of an art than a science"
*/


#ifdef ARCH_X86

/* for small memory blocks (<256 bytes) this version is faster */
#define small_memcpy(to,from,n)\
{\
register unsigned long int dummy;\
__asm__ __volatile__(\
  "rep; movsb"\
  :"=&D"(to), "=&S"(from), "=&c"(dummy)\
  :"0" (to), "1" (from),"2" (n)\
  : "memory");\
}

/* linux kernel __memcpy (from: /include/asm/string.h) */
static inline void * __memcpy(void * to, const void * from, size_t n)
{
int d0, d1, d2;

  if( n < 4 ) {
    small_memcpy(to,from,n);
  }
  else
    __asm__ __volatile__(
    "rep ; movsl\n\t"
    "testb $2,%b4\n\t"
    "je 1f\n\t"
    "movsw\n"
    "1:\ttestb $1,%b4\n\t"
    "je 2f\n\t"
    "movsb\n"
    "2:"
    : "=&c" (d0), "=&D" (d1), "=&S" (d2)
    :"0" (n/4), "q" (n),"1" ((long) to),"2" ((long) from)
    : "memory");
  
  return (to);
}

#define SSE_MMREG_SIZE 16
#define MMX_MMREG_SIZE 8

#define MMX1_MIN_LEN 0x800  /* 2K blocks */
#define MIN_LEN 0x40  /* 64-byte blocks */

/* SSE note: i tried to move 128 bytes a time instead of 64 but it
didn't make any measureable difference. i'm using 64 for the sake of
simplicity. [MF] */
static void * sse_memcpy(void * to, const void * from, size_t len)
{
  void *retval = to;
  size_t i;
    
  /* PREFETCH has effect even for MOVSB instruction ;) */
  __asm__ __volatile__ (
    "   prefetchnta (%0)\n"
    "   prefetchnta 64(%0)\n"
    "   prefetchnta 128(%0)\n"
    "   prefetchnta 192(%0)\n"
    "   prefetchnta 256(%0)\n"
    : : "r" (from) );
    
  if(len >= MIN_LEN)
  {
    register unsigned long int delta;
    /* Align destinition to MMREG_SIZE -boundary */
    delta = ((unsigned long int)to)&(SSE_MMREG_SIZE-1);
    if(delta)
    {
      delta=SSE_MMREG_SIZE-delta;
      len -= delta;
      small_memcpy(to, from, delta);
    }
    i = len >> 6; /* len/64 */
    len&=63;
    if(((unsigned long)from) & 15)
      /* if SRC is misaligned */
      for(; i>0; i--)
      {
        __asm__ __volatile__ (
        "prefetchnta 320(%0)\n"
        "movups (%0), %%xmm0\n"
        "movups 16(%0), %%xmm1\n"
        "movups 32(%0), %%xmm2\n"
        "movups 48(%0), %%xmm3\n"
        "movntps %%xmm0, (%1)\n"
        "movntps %%xmm1, 16(%1)\n"
        "movntps %%xmm2, 32(%1)\n"
        "movntps %%xmm3, 48(%1)\n"
        :: "r" (from), "r" (to) : "memory");
        from +=64;
        to +=64;
      }
    else 
      /*
         Only if SRC is aligned on 16-byte boundary.
         It allows to use movaps instead of movups, which required data
         to be aligned or a general-protection exception (#GP) is generated.
      */
      for(; i>0; i--)
      {
        __asm__ __volatile__ (
        "prefetchnta 320(%0)\n"
        "movaps (%0), %%xmm0\n"
        "movaps 16(%0), %%xmm1\n"
        "movaps 32(%0), %%xmm2\n"
        "movaps 48(%0), %%xmm3\n"
        "movntps %%xmm0, (%1)\n"
        "movntps %%xmm1, 16(%1)\n"
        "movntps %%xmm2, 32(%1)\n"
        "movntps %%xmm3, 48(%1)\n"
        :: "r" (from), "r" (to) : "memory");
        from +=64;
        to +=64;
      }
    /* since movntq is weakly-ordered, a "sfence"
     * is needed to become ordered again. */
    __asm__ __volatile__ ("sfence":::"memory");
    /* enables to use FPU */
    __asm__ __volatile__ ("emms":::"memory");
  }
  /*
   *	Now do the tail of the block
   */
  if(len) __memcpy(to, from, len);
  return retval;
}

static void * mmx_memcpy(void * to, const void * from, size_t len)
{
  void *retval = to;
  size_t i;

  if(len >= MMX1_MIN_LEN)
  {
    register unsigned long int delta;
    /* Align destinition to MMREG_SIZE -boundary */
    delta = ((unsigned long int)to)&(MMX_MMREG_SIZE-1);
    if(delta)
    {
      delta=MMX_MMREG_SIZE-delta;
      len -= delta;
      small_memcpy(to, from, delta);
    }
    i = len >> 6; /* len/64 */
    len&=63;
    for(; i>0; i--)
    {
      __asm__ __volatile__ (
      "movq (%0), %%mm0\n"
      "movq 8(%0), %%mm1\n"
      "movq 16(%0), %%mm2\n"
      "movq 24(%0), %%mm3\n"
      "movq 32(%0), %%mm4\n"
      "movq 40(%0), %%mm5\n"
      "movq 48(%0), %%mm6\n"
      "movq 56(%0), %%mm7\n"
      "movq %%mm0, (%1)\n"
      "movq %%mm1, 8(%1)\n"
      "movq %%mm2, 16(%1)\n"
      "movq %%mm3, 24(%1)\n"
      "movq %%mm4, 32(%1)\n"
      "movq %%mm5, 40(%1)\n"
      "movq %%mm6, 48(%1)\n"
      "movq %%mm7, 56(%1)\n"
      :: "r" (from), "r" (to) : "memory");
      from+=64;
      to+=64;
    }
    __asm__ __volatile__ ("emms":::"memory");
  }
  /*
   *	Now do the tail of the block
   */
  if(len) __memcpy(to, from, len);
  return retval;
}

void * mmx2_memcpy(void * to, const void * from, size_t len)
{
  void *retval = to;
  size_t i;

  /* PREFETCH has effect even for MOVSB instruction ;) */
  __asm__ __volatile__ (
    "   prefetchnta (%0)\n"
    "   prefetchnta 64(%0)\n"
    "   prefetchnta 128(%0)\n"
    "   prefetchnta 192(%0)\n"
    "   prefetchnta 256(%0)\n"
    : : "r" (from) );

  if(len >= MIN_LEN)
  {
    register unsigned long int delta;
    /* Align destinition to MMREG_SIZE -boundary */
    delta = ((unsigned long int)to)&(MMX_MMREG_SIZE-1);
    if(delta)
    {
      delta=MMX_MMREG_SIZE-delta;
      len -= delta;
      small_memcpy(to, from, delta);
    }
    i = len >> 6; /* len/64 */
    len&=63;
    for(; i>0; i--)
    {
      __asm__ __volatile__ (
      "prefetchnta 320(%0)\n"
      "movq (%0), %%mm0\n"
      "movq 8(%0), %%mm1\n"
      "movq 16(%0), %%mm2\n"
      "movq 24(%0), %%mm3\n"
      "movq 32(%0), %%mm4\n"
      "movq 40(%0), %%mm5\n"
      "movq 48(%0), %%mm6\n"
      "movq 56(%0), %%mm7\n"
      "movntq %%mm0, (%1)\n"
      "movntq %%mm1, 8(%1)\n"
      "movntq %%mm2, 16(%1)\n"
      "movntq %%mm3, 24(%1)\n"
      "movntq %%mm4, 32(%1)\n"
      "movntq %%mm5, 40(%1)\n"
      "movntq %%mm6, 48(%1)\n"
      "movntq %%mm7, 56(%1)\n"
      :: "r" (from), "r" (to) : "memory");
      from+=64;
      to+=64;
    }
     /* since movntq is weakly-ordered, a "sfence"
     * is needed to become ordered again. */
    __asm__ __volatile__ ("sfence":::"memory");
    __asm__ __volatile__ ("emms":::"memory");
  }
  /*
   *	Now do the tail of the block
   */
  if(len) __memcpy(to, from, len);
  return retval;
}

static void *linux_kernel_memcpy(void *to, const void *from, size_t len) {
  return __memcpy(to,from,len);
}


// By Jemino & CuC
void *sse_copie(void *dst, const void *src, size_t n, size_t tb, size_t ltb)
{
	__asm__ __volatile__(	"movl %1, %%esi\n"						// esi = src
							"movl %2, %%edi\n"						// edi = dst
							"movl %0, %%ecx\n"
							"cmpl $128, %%ecx\n"
							"jl 8f\n"
							"movl %%edi, %%ecx\n"					// ecx = dst
							"andl $15, %%ecx\n"						// on recupere le reste de dst/16
							"jz 0f\n"								// s'il est nul, dst est aligne et on demarre le traitement par blocs
							"subl $16, %%ecx\n"						// sinon...
							"negl %%ecx\n"							// ecx = nb d'octets avant d'etre aligne
							"subl %%ecx, %0\n"						// n apres alignement
							"rep movsb\n"							// copie des premiers octets
							"0:	movl %0, %%eax\n"					// eax = nb d'octets a copier
							"	movl $0, %%edx\n" 					// edx = 0 (pour la division)
							"	divl %3\n"
							"	cmpl $0, %%eax\n"
							"	je 1f\n"							// si c'est nul, on passe aux blocs de 128 octets
							"	movl %3, %%edx\n" 					// edx = taille d'un bloc
        						"2:	movl %%edx, %%ecx\n"
								"	shrl $7, %%ecx\n"
   								"3:	movl  -64(%%esi,%%edx), %%ebx\n" 	// boucle de prefetch (source)
								"	movl -128(%%esi,%%edx), %%ebx\n"
   								"	subl $128, %%edx\n"
   								"	jnz 3b\n"
   								"4:	movups    (%%esi,%%edx), %%xmm0\n"		// boucle de copie
    							"	movups  16(%%esi,%%edx), %%xmm1\n"
    							"	movups  32(%%esi,%%edx), %%xmm2\n"
    							"	movups  48(%%esi,%%edx), %%xmm3\n"
       							"	movups  64(%%esi,%%edx), %%xmm4\n"
       							"	movups  80(%%esi,%%edx), %%xmm5\n"
       							"	movups  96(%%esi,%%edx), %%xmm6\n"
       							"	movups 112(%%esi,%%edx), %%xmm7\n"
       							"	movntps %%xmm0,    (%%edi,%%edx)\n"
       							"	movntps %%xmm1,  16(%%edi,%%edx)\n"
       							"	movntps %%xmm2,  32(%%edi,%%edx)\n"
       							"	movntps %%xmm3,  48(%%edi,%%edx)\n"
       							"	movntps %%xmm4,  64(%%edi,%%edx)\n"
       							"	movntps %%xmm5,  80(%%edi,%%edx)\n"
       							"	movntps %%xmm6,  96(%%edi,%%edx)\n"
       							"	movntps %%xmm7, 112(%%edi,%%edx)\n"
       							"	addl $128, %%edx\n"
       							"	loop 4b\n"							// si le bloc n'est pas termine , on boucle
       							"addl %%edx, %%esi\n"				// sinon, bloc suivant
       							"addl %%edx, %%edi\n"
       							"decl %%eax\n"
       							"jnz 2b\n"							// s'il reste encore des blocs, on boucle
 							"1: movl %3, %%ecx\n"					// preparation de la copie des blocs de 128 restants
 							"	dec %%ecx\n"						// ecx = masque de bloc (ex: 4096 -> 0x00000FFF)
   							"	andl %0, %%ecx\n"					// ecx = n % TAILLE_BLOC
							"	shrl $7, %%ecx\n"					// on divise par 128
   							"	jz 6f\n"							// si c'est nul, on va finir
   							"	prefetcht0    (%%esi)\n"			// sinon, prefetch "soft"
   							"	prefetcht0  64(%%esi)\n"
   							"	prefetcht0 128(%%esi)\n"
   							"	prefetcht0 192(%%esi)\n"
   							"	prefetcht0 256(%%esi)\n"
							"	movl $0, %%edx\n"
   								"5: prefetcht0 320(%%esi, %%edx)\n"			// boucle de copie
    							"	prefetcht0 384(%%esi, %%edx)\n"
    							"	movups    (%%esi,%%edx), %%xmm0\n"
    							"	movups  16(%%esi,%%edx), %%xmm1\n"
    							"	movups  32(%%esi,%%edx), %%xmm2\n"
    							"	movups  48(%%esi,%%edx), %%xmm3\n"
       							"	movups  64(%%esi,%%edx), %%xmm4\n"
       							"	movups  80(%%esi,%%edx), %%xmm5\n"
       							"	movups  96(%%esi,%%edx), %%xmm6\n"
       							"	movups 112(%%esi,%%edx), %%xmm7\n"
       							"	movntps %%xmm0,    (%%edi,%%edx)\n"
       							"	movntps %%xmm1,  16(%%edi,%%edx)\n"
       							"	movntps %%xmm2,  32(%%edi,%%edx)\n"
       							"	movntps %%xmm3,  48(%%edi,%%edx)\n"
       							"	movntps %%xmm4,  64(%%edi,%%edx)\n"
       							"	movntps %%xmm5,  80(%%edi,%%edx)\n"
       							"	movntps %%xmm6,  96(%%edi,%%edx)\n"
       							"	movntps %%xmm7, 112(%%edi,%%edx)\n"
       							"	addl $128, %%edx\n"
       							"	loop 5b\n"							// s'il reste des blocs de 128 , on boucle
       							"addl %%edx, %%esi\n"
       							"addl %%edx, %%edi\n"
       						"6: sfence\n"
             				"	emms\n"
				            "	movl %0, %%ecx\n"
   							"	andl $0x0000007F, %%ecx\n"			// il reste au plus 127 octets a copier...
 							"8:	rep movsb\n"						// ...ce que l'on fait
       						:
        	     			: "m"(n) , "m"(src) , "m"(dst), "m"(tb)
        	         		: "eax", "ebx", "ecx", "edx", "esi", "edi", "memory"
        	            );

	return dst;
}


void *sse_memcpy_1K(void *dst, const void *src, size_t n)
{
    return sse_copie(dst, src, n, 1024, 10) ;
}


void *sse_memcpy_2K(void *dst, const void *src, size_t n)
{
    return sse_copie(dst, src, n, 2048, 11) ;
}


void *sse_memcpy_4K(void *dst, const void *src, size_t n)
{
    return sse_copie(dst, src, n, 4096, 12) ;
}


void *sse_memcpy_8K(void *dst, const void *src, size_t n)
{
    return sse_copie(dst, src, n, 8192, 13) ;
}


void *sse_memcpy_16K(void *dst, const void *src, size_t n)
{
    return sse_copie(dst, src, n, 16384, 14) ;
}


void *sse_memcpy_32K(void *dst, const void *src, size_t n)
{
    return sse_copie(dst, src, n, 32768, 15) ;
}


void *sse_memcpy_64K(void *dst, const void *src, size_t n)
{
    return sse_copie(dst, src, n, 65536, 16) ;
}


// By Jemino & CuC
void *mmx2_copie(void *dst, const void *src, size_t n, size_t tb, size_t ltb)
{
	__asm__ __volatile__(	"movl %1, %%esi\n"						// esi = src
							"movl %2, %%edi\n"						// edi = dst
							"movl %0, %%ecx\n"
							"cmpl $64, %%ecx\n"
							"jl 8f\n"
							"movl %%edi, %%ecx\n"					// ecx = dst
							"andl $7, %%ecx\n"						// on recupere le reste de dst/8
							"jz 0f\n"								// s'il est nul, dst est aligne et on demarre le traitement par blocs
							"subl $8, %%ecx\n"						// sinon...
							"negl %%ecx\n"							// ecx = nb d'octets avant d'etre aligne
							"subl %%ecx, %0\n"						// n apres alignement
							"rep movsb\n"							// copie des premiers octets
							"0:	movl %0, %%eax\n"					// eax = nb d'octets a copier
							"	movl $0, %%edx\n" 					// edx = 0 (pour la division)
							"	divl %3\n"
							"	cmpl $0, %%eax\n"
							"	je 1f\n"							// si c'est nul, on passe aux blocs de 128 octets
							"	movl %3, %%edx\n" 					// edx = taille d'un bloc
        						"2:	movl %%edx, %%ecx\n"
								"	shrl $6, %%ecx\n"
   								"3:	movl  -64(%%esi,%%edx), %%ebx\n" 	// boucle de prefetch (source)
								"	movl -128(%%esi,%%edx), %%ebx\n"
   								"	subl $128, %%edx\n"
   								"	jnz 3b\n"
   								"4:	movq   (%%esi,%%edx), %%mm0\n"		// boucle de copie
    							"	movq  8(%%esi,%%edx), %%mm1\n"
    							"	movq 16(%%esi,%%edx), %%mm2\n"
    							"	movq 24(%%esi,%%edx), %%mm3\n"
       							"	movq 32(%%esi,%%edx), %%mm4\n"
       							"	movq 40(%%esi,%%edx), %%mm5\n"
       							"	movq 48(%%esi,%%edx), %%mm6\n"
       							"	movq 56(%%esi,%%edx), %%mm7\n"
       							"	movntq %%mm0,   (%%edi,%%edx)\n"
       							"	movntq %%mm1,  8(%%edi,%%edx)\n"
       							"	movntq %%mm2, 16(%%edi,%%edx)\n"
       							"	movntq %%mm3, 24(%%edi,%%edx)\n"
       							"	movntq %%mm4, 32(%%edi,%%edx)\n"
       							"	movntq %%mm5, 40(%%edi,%%edx)\n"
       							"	movntq %%mm6, 48(%%edi,%%edx)\n"
       							"	movntq %%mm7, 56(%%edi,%%edx)\n"
       							"	addl $64, %%edx\n"
       							"	loop 4b\n"							// si le bloc n'est pas termine , on boucle
       							"addl %%edx, %%esi\n"				// sinon, bloc suivant
       							"addl %%edx, %%edi\n"
       							"decl %%eax\n"
       							"jnz 2b\n"							// s'il reste encore des blocs, on boucle
 							"1: movl %3, %%ecx\n"					// preparation de la copie des blocs de 64 restants
 							"	dec %%ecx\n"						// ecx = masque de bloc (ex: 4096 -> 0x00000FFF)
   							"	andl %0, %%ecx\n"					// ecx = n % TAILLE_BLOC
							"	shrl $6, %%ecx\n"					// on divise par 64
   							"	jz 6f\n"							// si c'est nul, on va finir
   							"	prefetcht0    (%%esi)\n"			// sinon, prefetch "soft"
   							"	prefetcht0  64(%%esi)\n"
   							"	prefetcht0 128(%%esi)\n"
   							"	prefetcht0 192(%%esi)\n"
   							"	prefetcht0 256(%%esi)\n"
							"	movl $0, %%edx\n"
   								"5: prefetcht0 320(%%esi, %%edx)\n"		// boucle de copie
    							"	movq   (%%esi,%%edx), %%mm0\n"
    							"	movq  8(%%esi,%%edx), %%mm1\n"
    							"	movq 16(%%esi,%%edx), %%mm2\n"
    							"	movq 24(%%esi,%%edx), %%mm3\n"
       							"	movq 32(%%esi,%%edx), %%mm4\n"
       							"	movq 40(%%esi,%%edx), %%mm5\n"
       							"	movq 48(%%esi,%%edx), %%mm6\n"
       							"	movq 56(%%esi,%%edx), %%mm7\n"
       							"	movntq %%mm0,   (%%edi,%%edx)\n"
       							"	movntq %%mm1,  8(%%edi,%%edx)\n"
       							"	movntq %%mm2, 16(%%edi,%%edx)\n"
       							"	movntq %%mm3, 24(%%edi,%%edx)\n"
       							"	movntq %%mm4, 32(%%edi,%%edx)\n"
       							"	movntq %%mm5, 40(%%edi,%%edx)\n"
       							"	movntq %%mm6, 48(%%edi,%%edx)\n"
       							"	movntq %%mm7, 56(%%edi,%%edx)\n"
       							"	addl $64, %%edx\n"
       							"	loop 5b\n"							// s'il reste des blocs de 64 , on boucle
       							"addl %%edx, %%esi\n"
       							"addl %%edx, %%edi\n"
       						"6: sfence\n"
             				"	emms\n"						
				            "	movl %0, %%ecx\n"
   							"	andl $0x0000003F, %%ecx\n"			// il reste au plus 63 octets a copier...
 							"8:	rep movsb\n"						// ...ce que l'on fait
       						:
        	     			: "m"(n) , "m"(src) , "m"(dst), "m"(tb)
        	         		: "eax", "ebx", "ecx", "edx", "esi", "edi", "memory"
        	            );

	return dst;
}


void *mmx2_memcpy_1K(void *dst, const void *src, size_t n)
{
    return mmx2_copie(dst, src, n, 1024, 10) ;
}


void *mmx2_memcpy_2K(void *dst, const void *src, size_t n)
{
    return mmx2_copie(dst, src, n, 2048, 11) ;
}


void *mmx2_memcpy_4K(void *dst, const void *src, size_t n)
{
    return mmx2_copie(dst, src, n, 4096, 12) ;
}


void *mmx2_memcpy_8K(void *dst, const void *src, size_t n)
{
    return mmx2_copie(dst, src, n, 8192, 13) ;
}


void *mmx2_memcpy_16K(void *dst, const void *src, size_t n)
{
    return mmx2_copie(dst, src, n, 16384, 14) ;
}


void *mmx2_memcpy_32K(void *dst, const void *src, size_t n)
{
    return mmx2_copie(dst, src, n, 32768, 15) ;
}


void *mmx2_memcpy_64K(void *dst, const void *src, size_t n)
{
    return mmx2_copie(dst, src, n, 65536, 16) ;
}



// By Jemino & CuC
void *mmx_copie(void *dst, const void *src, size_t n, size_t tb, size_t ltb)
{
	__asm__ __volatile__(	"movl %1, %%esi\n"						// esi = src
							"movl %2, %%edi\n"						// edi = dst
							"movl %0, %%ecx\n"
							"cmpl $64, %%ecx\n"
							"jl 8f\n"
							"movl %%edi, %%ecx\n"					// ecx = dst
							"andl $7, %%ecx\n"						// on recupere le reste de dst/8
							"jz 0f\n"								// s'il est nul, dst est aligne et on demarre le traitement par blocs
							"subl $8, %%ecx\n"						// sinon...
							"negl %%ecx\n"							// ecx = nb d'octets avant d'etre aligne
							"subl %%ecx, %0\n"						// n apres alignement
							"rep movsb\n"							// copie des premiers octets
							"0:	movl %0, %%eax\n"					// eax = nb d'octets a copier
							"	movl $0, %%edx\n" 					// edx = 0 (pour la division)
							"	divl %3\n"
							"	cmpl $0, %%eax\n"
							"	je 1f\n"							// si c'est nul, on passe aux blocs de 128 octets
							"	movl %3, %%edx\n" 					// edx = taille d'un bloc
        						"2:	movl %%edx, %%ecx\n"
								"	shrl $6, %%ecx\n"
								"3:	movl  -64(%%esi,%%edx), %%ebx\n" 	// boucle de prefetch (source)
								"	movl -128(%%esi,%%edx), %%ebx\n"
								"	movl  -64(%%edi,%%edx), %%ebx\n" 	// boucle de prefetch (destination)
								"	movl -128(%%edi,%%edx), %%ebx\n"
   								"	subl $128, %%edx\n"
   								"	jnz 3b\n"
								"4:	movq   (%%esi,%%edx), %%mm0\n"		// boucle de copie
    							"	movq  8(%%esi,%%edx), %%mm1\n"
    							"	movq 16(%%esi,%%edx), %%mm2\n"
    							"	movq 24(%%esi,%%edx), %%mm3\n"
       							"	movq 32(%%esi,%%edx), %%mm4\n"
       							"	movq 40(%%esi,%%edx), %%mm5\n"
       							"	movq 48(%%esi,%%edx), %%mm6\n"
       							"	movq 56(%%esi,%%edx), %%mm7\n"
       							"	movq %%mm0,   (%%edi,%%edx)\n"
       							"	movq %%mm1,  8(%%edi,%%edx)\n"
       							"	movq %%mm2, 16(%%edi,%%edx)\n"
       							"	movq %%mm3, 24(%%edi,%%edx)\n"
       							"	movq %%mm4, 32(%%edi,%%edx)\n"
       							"	movq %%mm5, 40(%%edi,%%edx)\n"
       							"	movq %%mm6, 48(%%edi,%%edx)\n"
       							"	movq %%mm7, 56(%%edi,%%edx)\n"
       							"	addl $64, %%edx\n"
       							"	loop 4b\n"							// si le bloc n'est pas termine , on boucle
       							"addl %%edx, %%esi\n"				// sinon, bloc suivant
       							"addl %%edx, %%edi\n"
       							"decl %%eax\n"
       							"jnz 2b\n"							// s'il reste encore des blocs, on boucle
 							"1: movl %3, %%ecx\n"					// preparation de la copie des blocs de 64 restants
 							"	dec %%ecx\n"						// ecx = masque de bloc (ex: 4096 -> 0x00000FFF)
   							"	andl %0, %%ecx\n"					// ecx = n % TAILLE_BLOC
							"	shrl $6, %%ecx\n"					// on divise par 64
   							"	jz 6f\n"							// si c'est nul, on va finir
							"	movl $0, %%edx\n"
								"5:	movq   (%%esi,%%edx), %%mm0\n"		// boucle de copie
    							"	movq  8(%%esi,%%edx), %%mm1\n"
    							"	movq 16(%%esi,%%edx), %%mm2\n"
    							"	movq 24(%%esi,%%edx), %%mm3\n"
       							"	movq 32(%%esi,%%edx), %%mm4\n"
       							"	movq 40(%%esi,%%edx), %%mm5\n"
       							"	movq 48(%%esi,%%edx), %%mm6\n"
       							"	movq 56(%%esi,%%edx), %%mm7\n"
       							"	movq %%mm0,   (%%edi,%%edx)\n"
       							"	movq %%mm1,  8(%%edi,%%edx)\n"
       							"	movq %%mm2, 16(%%edi,%%edx)\n"
       							"	movq %%mm3, 24(%%edi,%%edx)\n"
       							"	movq %%mm4, 32(%%edi,%%edx)\n"
       							"	movq %%mm5, 40(%%edi,%%edx)\n"
       							"	movq %%mm6, 48(%%edi,%%edx)\n"
       							"	movq %%mm7, 56(%%edi,%%edx)\n"
       							"	addl $64, %%edx\n"
       							"	loop 5b\n"							// s'il reste des blocs de 64 , on boucle
       							"addl %%edx, %%esi\n"				
       							"addl %%edx, %%edi\n"
       						"6: emms\n"
				            "	movl %0, %%ecx\n"
   							"	andl $0x0000003F, %%ecx\n"			// il reste au plus 63 octets a copier...
 							"8:	rep movsb\n"						// ...ce que l'on fait
       						:
        	     			: "m"(n) , "m"(src) , "m"(dst), "m"(tb)
        	         		: "eax", "ebx", "ecx", "edx", "esi", "edi", "memory"
        	            );

	return dst;
}


void *mmx_memcpy_512(void *dst, const void *src, size_t n)
{
    return mmx_copie(dst, src, n, 512, 9) ;
}


void *mmx_memcpy_1K(void *dst, const void *src, size_t n)
{
    return mmx_copie(dst, src, n, 1024, 10) ;
}


void *mmx_memcpy_2K(void *dst, const void *src, size_t n)
{
    return mmx_copie(dst, src, n, 2048, 11) ;
}


void *mmx_memcpy_4K(void *dst, const void *src, size_t n)
{
    return mmx_copie(dst, src, n, 4096, 12) ;
}


void *mmx_memcpy_8K(void *dst, const void *src, size_t n)
{
    return mmx_copie(dst, src, n, 8192, 13) ;
}


void *mmx_memcpy_16K(void *dst, const void *src, size_t n)
{
    return mmx_copie(dst, src, n, 16384, 14) ;
}


void *mmx_memcpy_32K(void *dst, const void *src, size_t n)
{
    return mmx_copie(dst, src, n, 32768, 15) ;
}


// By Jemino & CuC
void *fpu_copie(void *dst, const void *src, size_t n, size_t tb, size_t ltb)
{
	__asm__ __volatile__(	"movl %1, %%esi\n"						// esi = src
							"movl %2, %%edi\n"						// edi = dst
							"movl %0, %%ecx\n"
							"cmpl $64, %%ecx\n"
							"jl 8f\n"
							"movl %%edi, %%ecx\n"					// ecx = dst
							"andl $7, %%ecx\n"						// on recupere le reste de dst/8
							"jz 0f\n"								// s'il est nul, dst est aligne et on demarre le traitement par blocs
							"subl $8, %%ecx\n"						// sinon...
							"negl %%ecx\n"							// ecx = nb d'octets avant d'etre aligne
							"subl %%ecx, %0\n"						// n apres alignement
							"rep movsb\n"							// copie des premiers octets
							"0:	movl %0, %%eax\n"					// eax = nb d'octets a copier
							"	movl $0, %%edx\n" 					// edx = 0 (pour la division)
							"	divl %3\n"
							"	cmpl $0, %%eax\n"
							"	je 1f\n"							// si c'est nul, on passe aux blocs de 128 octets
							"	movl %3, %%edx\n" 					// edx = taille d'un bloc
        						"2:	movl %%edx, %%ecx\n"
								"	shrl $6, %%ecx\n"
								"3:	movl  -64(%%esi,%%edx), %%ebx\n" 	// boucle de prefetch (source)
								"	movl -128(%%esi,%%edx), %%ebx\n"
								"	movl  -64(%%edi,%%edx), %%ebx\n" 	// boucle de prefetch (destination)
								"	movl -128(%%edi,%%edx), %%ebx\n"
   								"	subl $128, %%edx\n"
   								"	jnz 3b\n"
								"4: fildq   (%%esi,%%edx)\n"		// boucle de copie
    							"	fildq  8(%%esi,%%edx)\n"
    							"	fildq 16(%%esi,%%edx)\n"
    							"	fildq 24(%%esi,%%edx)\n"
       							"	fildq 32(%%esi,%%edx)\n"
       							"	fildq 40(%%esi,%%edx)\n"
       							"	fildq 48(%%esi,%%edx)\n"
       							"	fildq 56(%%esi,%%edx)\n"
       							"	fistpq 56(%%edi,%%edx)\n"
       							"	fistpq 48(%%edi,%%edx)\n"
       							"	fistpq 40(%%edi,%%edx)\n"
       							"	fistpq 32(%%edi,%%edx)\n"
       							"	fistpq 24(%%edi,%%edx)\n"
       							"	fistpq 16(%%edi,%%edx)\n"
       							"	fistpq  8(%%edi,%%edx)\n"
       							"	fistpq   (%%edi,%%edx)\n"
       							"	addl $64, %%edx\n"
       							"	loop 4b\n"							// si le bloc n'est pas termine , on boucle
       							"addl %%edx, %%esi\n"				// sinon, bloc suivant
       							"addl %%edx, %%edi\n"
       							"decl %%eax\n"
       							"jnz 2b\n"							// s'il reste encore des blocs, on boucle
 							"1: movl %3, %%ecx\n"					// preparation de la copie des blocs de 64 restants
 							"	dec %%ecx\n"						// ecx = masque de bloc (ex: 4096 -> 0x00000FFF)
   							"	andl %0, %%ecx\n"					// ecx = n % TAILLE_BLOC
							"	shrl $6, %%ecx\n"					// on divise par 64
   							"	jz 6f\n"							// si c'est nul, on va finir
							"	movl $0, %%edx\n"
								"5: fildq   (%%esi,%%edx)\n"		// boucle de copie
    							"	fildq  8(%%esi,%%edx)\n"
    							"	fildq 16(%%esi,%%edx)\n"
    							"	fildq 24(%%esi,%%edx)\n"
       							"	fildq 32(%%esi,%%edx)\n"
       							"	fildq 40(%%esi,%%edx)\n"
       							"	fildq 48(%%esi,%%edx)\n"
       							"	fildq 56(%%esi,%%edx)\n"
       							"	fistpq 56(%%edi,%%edx)\n"
       							"	fistpq 48(%%edi,%%edx)\n"
       							"	fistpq 40(%%edi,%%edx)\n"
       							"	fistpq 32(%%edi,%%edx)\n"
       							"	fistpq 24(%%edi,%%edx)\n"
       							"	fistpq 16(%%edi,%%edx)\n"
       							"	fistpq  8(%%edi,%%edx)\n"
       							"	fistpq   (%%edi,%%edx)\n"
       							"	addl $64, %%edx\n"
       							"	loop 5b\n"							// s'il reste des blocs de 64 , on boucle
       							"addl %%edx, %%esi\n"				// sinon, bloc suivant
       							"addl %%edx, %%edi\n"
       						"6: movl %0, %%ecx\n"
   							"	andl $0x0000003F, %%ecx\n"			// il reste au plus 63 octets a copier...
 							"8:	rep movsb\n"						// ...ce que l'on fait
       						:
        	     			: "m"(n) , "m"(src) , "m"(dst), "m"(tb)
        	         		: "eax", "ebx", "ecx", "edx", "esi", "edi", "memory"
        	            );

	return dst;
}


void *fpu_memcpy_512(void *dst, const void *src, size_t n)
{
    return fpu_copie(dst, src, n, 512, 9) ;
}


void *fpu_memcpy_1K(void *dst, const void *src, size_t n)
{
    return fpu_copie(dst, src, n, 1024, 10) ;
}


void *fpu_memcpy_2K(void *dst, const void *src, size_t n)
{
    return fpu_copie(dst, src, n, 2048, 11) ;
}


void *fpu_memcpy_4K(void *dst, const void *src, size_t n)
{
    return fpu_copie(dst, src, n, 4096, 12) ;
}


void *fpu_memcpy_8K(void *dst, const void *src, size_t n)
{
    return fpu_copie(dst, src, n, 8192, 13) ;
}


void *fpu_memcpy_16K(void *dst, const void *src, size_t n)
{
    return fpu_copie(dst, src, n, 16384, 14) ;
}


void *fpu_memcpy_32K(void *dst, const void *src, size_t n)
{
    return fpu_copie(dst, src, n, 32768, 15) ;
}



#endif /* ARCH_X86 */

static struct {
  int tag;
  char *name;
  void *(* function)(void *to, const void *from, size_t len);
  unsigned long long time;
  unsigned long long * times;
  uint32_t cpu_require;
} memcpy_method[] = 
{
  { MEMCPY_GLIBC,      "glibc memcpy()",              memcpy,                0, 0, 0                },
#ifdef ARCH_X86
  { MEMCPY_KERNEL,     "linux kernel memcpy()",         linux_kernel_memcpy, 0, 0, 0                },
  { MEMCPY_FPU_512,    "FPU memcpy() 512",              fpu_memcpy_512,      0, 0, 0                },
  { MEMCPY_FPU_1K,     "FPU memcpy() 1K",               fpu_memcpy_1K,       0, 0, 0                },
  { MEMCPY_FPU_2K,     "FPU memcpy() 2K",               fpu_memcpy_2K,       0, 0, 0                },
  { MEMCPY_FPU_4K,     "FPU memcpy() 4K",               fpu_memcpy_4K,       0, 0, 0                },
  { MEMCPY_FPU_8K,     "FPU memcpy() 8K",               fpu_memcpy_8K,       0, 0, 0                },
  { MEMCPY_FPU_16K,    "FPU memcpy() 16K",              fpu_memcpy_16K,      0, 0, 0                },
  { MEMCPY_FPU_32K,    "FPU memcpy() 32K",              fpu_memcpy_32K,      0, 0, 0                },
  { MEMCPY_MMX,        "MMX optimized memcpy()",        mmx_memcpy,          0, 0, MM_MMX           },
  { MEMCPY_MMX_512,    "MMX optimized memcpy() 512",    mmx_memcpy_512,      0, 0, MM_MMX           },
  { MEMCPY_MMX_1K,     "MMX optimized memcpy() 1K",     mmx_memcpy_1K,       0, 0, MM_MMX           },
  { MEMCPY_MMX_2K,     "MMX optimized memcpy() 2K",     mmx_memcpy_2K,       0, 0, MM_MMX           },
  { MEMCPY_MMX_4K,     "MMX optimized memcpy() 4K",     mmx_memcpy_4K,       0, 0, MM_MMX           },
  { MEMCPY_MMX_8K,     "MMX optimized memcpy() 8K",     mmx_memcpy_8K,       0, 0, MM_MMX           },
  { MEMCPY_MMX_16K,    "MMX optimized memcpy() 16K",    mmx_memcpy_16K,      0, 0, MM_MMX           },
  { MEMCPY_MMX_32K,    "MMX optimized memcpy() 32K",    mmx_memcpy_32K,      0, 0, MM_MMX           },
  { MEMCPY_MMXEXT,     "MMXEXT optimized memcpy()",     mmx2_memcpy,         0, 0, MM_MMXEXT        },
  { MEMCPY_MMXEXT_1K,  "MMXEXT optimized memcpy() 1K",  mmx2_memcpy_1K,      0, 0, MM_MMXEXT        },
  { MEMCPY_MMXEXT_2K,  "MMXEXT optimized memcpy() 2K",  mmx2_memcpy_2K,      0, 0, MM_MMXEXT        },
  { MEMCPY_MMXEXT_4K,  "MMXEXT optimized memcpy() 4K",  mmx2_memcpy_4K,      0, 0, MM_MMXEXT        },
  { MEMCPY_MMXEXT_8K,  "MMXEXT optimized memcpy() 8K",  mmx2_memcpy_8K,      0, 0, MM_MMXEXT        },
  { MEMCPY_MMXEXT_16K, "MMXEXT optimized memcpy() 16K", mmx2_memcpy_16K,     0, 0, MM_MMXEXT        },
  { MEMCPY_MMXEXT_32K, "MMXEXT optimized memcpy() 32K", mmx2_memcpy_32K,     0, 0, MM_MMXEXT        },
  { MEMCPY_MMXEXT_64K, "MMXEXT optimized memcpy() 64K", mmx2_memcpy_64K,     0, 0, MM_MMXEXT        },
# ifndef __FreeBSD__
  { MEMCPY_SSE,        "SSE optimized memcpy()",        sse_memcpy,          0, 0, MM_MMXEXT|MM_SSE },
  { MEMCPY_SSE_1K,     "SSE optimized memcpy() 1K",     sse_memcpy_1K,       0, 0, MM_MMXEXT|MM_SSE },
  { MEMCPY_SSE_2K,     "SSE optimized memcpy() 2K",     sse_memcpy_2K,       0, 0, MM_MMXEXT|MM_SSE },
  { MEMCPY_SSE_4K,     "SSE optimized memcpy() 4K",     sse_memcpy_4K,       0, 0, MM_MMXEXT|MM_SSE },
  { MEMCPY_SSE_8K,     "SSE optimized memcpy() 8K",     sse_memcpy_8K,       0, 0, MM_MMXEXT|MM_SSE },
  { MEMCPY_SSE_16K,    "SSE optimized memcpy() 16K",    sse_memcpy_16K,      0, 0, MM_MMXEXT|MM_SSE },
  { MEMCPY_SSE_32K,    "SSE optimized memcpy() 32K",    sse_memcpy_32K,      0, 0, MM_MMXEXT|MM_SSE },
  { MEMCPY_SSE_64K,    "SSE optimized memcpy() 64K",    sse_memcpy_64K,      0, 0, MM_MMXEXT|MM_SSE },
# endif
#endif /* ARCH_X86 */
  { 0,                 NULL,                            NULL,                0, 0, 0 }
};

#ifdef ARCH_X86
static unsigned long long int rdtsc()
{
  unsigned long long int x;
  __asm__ __volatile__ ("rdtsc" : "=A" (x));
  // __asm__ volatile (".byte 0x0f, 0x31" : "=A" (x));     
  return x;
}
#else
static unsigned long long int rdtsc()
{
  /* FIXME: implement an equivalent for using optimized memcpy on other
            architectures */
  return 0;
}
#endif


void swap(unsigned long long * a, int i, int j)
{
    unsigned long long tmp = a[i];
    a[i] = a[j];
    a[j] = tmp;
}

void quicksort(unsigned long long * a, int left, int right) {
    int last = left, i;

    if (left >= right) return;

    for (i = left + 1; i <= right; i++)
        if (a[i] < a[left])
            swap(a,++last,i);
    swap(a,left,last);
    quicksort(a,left,last-1);
    quicksort(a,last+1,right);
}


unsigned long long median_sum(unsigned long long * p, size_t n)
{
    unsigned i ;
    unsigned long long sum = 0 ;
    quicksort(p, 0, n-1) ;  /* Sorting the execution times */
    for (i = n/5; i < 4*n/5; i++) /* Keeping the 60% median execution times */
        sum += p[i] ;
    return sum ;
}



// #define FLUSH_CACHE
void probe_fast_memcpy(int method)
{
unsigned long long t;
char *buf1, *buf2;
int i, j, best = -1;
static int config_flags = -1;
int n_func = sizeof(memcpy_method)/sizeof(memcpy_method[0])-1; /* Maximum number of functions to test */
int N_TEST = 80; /* Number of times to test each function */
int progress = 0 ; /* Progression counter */
int BUFSIZE = 4*768*576; /* Buffer size for the benchmark */
#ifdef FLUSH_CACHE
char *buf3;
int FLUSH_CACHE_SIZE = (int) 1.5*1024*1024 ;
#endif

#ifdef ARCH_X86
  config_flags = mm_accel();
#else
  config_flags = 0;
#endif


  switch (method)
    {
    case MEMCPY_GLIBC:
      fast_memcpy = memcpy;
      if (debug)
        fprintf(stderr, "Using glibc memcpy.\n");
      return;

#ifdef ARCH_X86
    case MEMCPY_KERNEL:
      fast_memcpy = linux_kernel_memcpy;
      if (debug)
        fprintf(stderr, "Using kernel memcpy.\n");
      return;

    case MEMCPY_FPU_512:
      fast_memcpy = fpu_memcpy_512;
      if (debug)
        fprintf(stderr, "Using FPU memcpy 512.\n");
      return;

    case MEMCPY_FPU_1K:
      fast_memcpy = fpu_memcpy_1K;
      if (debug)
        fprintf(stderr, "Using FPU memcpy 1K.\n");
      return;

    case MEMCPY_FPU_2K:
      fast_memcpy = fpu_memcpy_2K;
      if (debug)
        fprintf(stderr, "Using FPU memcpy 2K.\n");
      return;

    case MEMCPY_FPU_4K:
      fast_memcpy = fpu_memcpy_4K;
      if (debug)
        fprintf(stderr, "Using FPU memcpy 4K.\n");
      return;

    case MEMCPY_FPU_8K:
      fast_memcpy = fpu_memcpy_8K;
      if (debug)
        fprintf(stderr, "Using FPU memcpy 8K.\n");
      return;

    case MEMCPY_FPU_16K:
      fast_memcpy = fpu_memcpy_16K;
      if (debug)
        fprintf(stderr, "Using FPU memcpy 16K.\n");
      return;

    case MEMCPY_FPU_32K:
      fast_memcpy = fpu_memcpy_32K;
      if (debug)
        fprintf(stderr, "Using FPU memcpy 32K.\n");
      return;

    case MEMCPY_MMX:
      fast_memcpy = mmx_memcpy;
      if (debug)
        fprintf(stderr, "Using MMX memcpy.\n");
      return;

    case MEMCPY_MMX_512:
      fast_memcpy = mmx_memcpy_512;
      if (debug)
        fprintf(stderr, "Using MMX memcpy 512.\n");
      return;

    case MEMCPY_MMX_1K:
      fast_memcpy = mmx_memcpy_1K;
      if (debug)
        fprintf(stderr, "Using MMX memcpy 1K.\n");
      return;

    case MEMCPY_MMX_2K:
      fast_memcpy = mmx_memcpy_2K;
      if (debug)
        fprintf(stderr, "Using MMX memcpy 2K.\n");
      return;

    case MEMCPY_MMX_4K:
      fast_memcpy = mmx_memcpy_4K;
      if (debug)
        fprintf(stderr, "Using MMX memcpy 4K.\n");
      return;

    case MEMCPY_MMX_8K:
      fast_memcpy = mmx_memcpy_8K;
      if (debug)
        fprintf(stderr, "Using MMX memcpy 8K.\n");
      return;

    case MEMCPY_MMX_16K:
      fast_memcpy = mmx_memcpy_16K;
      if (debug)
        fprintf(stderr, "Using MMX memcpy 16K.\n");
      return;

    case MEMCPY_MMX_32K:
      fast_memcpy = mmx_memcpy_32K;
      if (debug)
        fprintf(stderr, "Using MMX memcpy 32K.\n");
      return;

    case MEMCPY_MMXEXT:
      fast_memcpy = mmx2_memcpy;
      if (debug)
        fprintf(stderr, "Using MMXEXT memcpy.\n");
      return;

    case MEMCPY_MMXEXT_1K:
      fast_memcpy = mmx2_memcpy_1K;
      if (debug)
        fprintf(stderr, "Using MMXEXT memcpy 1K.\n");
      return;

    case MEMCPY_MMXEXT_2K:
      fast_memcpy = mmx2_memcpy_2K;
      if (debug)
        fprintf(stderr, "Using MMXEXT memcpy 2K.\n");
      return;

    case MEMCPY_MMXEXT_4K:
      fast_memcpy = mmx2_memcpy_4K;
      if (debug)
        fprintf(stderr, "Using MMXEXT memcpy 4K.\n");
      return;

    case MEMCPY_MMXEXT_8K:
      fast_memcpy = mmx2_memcpy_8K;
      if (debug)
        fprintf(stderr, "Using MMXEXT memcpy 8K.\n");
      return;

    case MEMCPY_MMXEXT_16K:
      fast_memcpy = mmx2_memcpy_16K;
      if (debug)
        fprintf(stderr, "Using MMXEXT memcpy 16K.\n");
      return;

    case MEMCPY_MMXEXT_32K:
      fast_memcpy = mmx2_memcpy_32K;
      if (debug)
        fprintf(stderr, "Using MMXEXT memcpy 32K.\n");
      return;

    case MEMCPY_MMXEXT_64K:
      fast_memcpy = mmx2_memcpy_64K;
      if (debug)
        fprintf(stderr, "Using MMXEXT memcpy 64K.\n");
      return;

    case MEMCPY_SSE:
      fast_memcpy = sse_memcpy;
      if (debug)
        fprintf(stderr, "Using SSE memcpy.\n");
      return;

    case MEMCPY_SSE_1K:
      fast_memcpy = sse_memcpy_1K;
      if (debug)
        fprintf(stderr, "Using SSE memcpy 1K.\n");
      return;

    case MEMCPY_SSE_2K:
      fast_memcpy = sse_memcpy_2K;
      if (debug)
        fprintf(stderr, "Using SSE memcpy 2K.\n");
      return;

    case MEMCPY_SSE_4K:
      fast_memcpy = sse_memcpy_4K;
      if (debug)
        fprintf(stderr, "Using SSE memcpy 4K.\n");
      return;

    case MEMCPY_SSE_8K:
      fast_memcpy = sse_memcpy_8K;
      if (debug)
        fprintf(stderr, "Using SSE memcpy 8K.\n");
      return;

    case MEMCPY_SSE_16K:
      fast_memcpy = sse_memcpy_16K;
      if (debug)
        fprintf(stderr, "Using SSE memcpy 16K.\n");
      return;

    case MEMCPY_SSE_32K:
      fast_memcpy = sse_memcpy_32K;
      if (debug)
        fprintf(stderr, "Using SSE memcpy 32K.\n");
      return;

    case MEMCPY_SSE_64K:
      fast_memcpy = sse_memcpy_64K;
      if (debug)
        fprintf(stderr, "Using SSE memcpy 64K.\n");
      return;
#endif

    case MEMCPY_PROBE:
    default:
      fast_memcpy = memcpy;
      break;
    }
  
  //fast_memcpy = memcpy;
  /* checking can not be done for non-cpuid processors (like i486...) */
  if(config_flags == 0) {
    save_memcpy_method(MEMCPY_GLIBC);
    return;
  }

  if( (buf1 = malloc(BUFSIZE)) == NULL )
    return;
    
  if( (buf2 = malloc(BUFSIZE)) == NULL ) {
    free(buf1);
  }
    
#ifdef FLUSH_CACHE
  if( (buf3 = malloc(FLUSH_CACHE_SIZE)) == NULL ) {
    free(buf1);
    free(buf2);
    return;
  }
#endif

  printf("Benchmarking memcpy methods (smaller is better):\n");
  /* make sure buffers are present on physical memory */
  memcpy(buf2,buf1,BUFSIZE);

  for (i = 0; i < BUFSIZE; i++) /* Filling the buffer */
     buf1[i] = i ;

  for (i=0; i<n_func; i++) /* Allocating "times" buffers to store N_TEST results */
  {
      memcpy_method[i].times = (unsigned long long *) malloc(N_TEST * sizeof(unsigned long long)) ;
  }
  for (j=0; j<N_TEST; j++)
  {
     for (i=0; i<n_func; i++) /* Testing each function */
     {
        if( (config_flags & memcpy_method[i].cpu_require) == 
             memcpy_method[i].cpu_require )
        {
#ifdef FLUSH_CACHE
          memset(buf3, i, FLUSH_CACHE_SIZE) ; /* Hoping to flush L1 and L2 cache */
#endif
          t = rdtsc();
          memcpy_method[i].function(buf2,buf1,BUFSIZE);
          // memcpy_method[i].function(buf1,buf2,BUFSIZE);
          t = rdtsc() - t;
          memcpy_method[i].times[j] = t ;
        }
        printf("Please wait... %d%%\r", 100*progress++/(n_func*N_TEST)) ;
        fflush(stdout) ;
     }
  }
  printf("Please wait... 100%%\n") ;
  for (i=0; i<n_func; i++) /* Finding the best function */
  {
     if( (config_flags & memcpy_method[i].cpu_require) == 
          memcpy_method[i].cpu_require )
     {
        memcpy_method[i].time = median_sum(memcpy_method[i].times, N_TEST) ;
        if( best == -1 || memcpy_method[i].time < memcpy_method[best].time )
          best = i;
     }
  }
  for (i=0; i<n_func; i++) /* Showing the results */
  {
     if( (config_flags & memcpy_method[i].cpu_require) == 
          memcpy_method[i].cpu_require )
     {
        printf("\t%-30s : %12lld (%3.1f %%)\n",
                  memcpy_method[i].name,
                          memcpy_method[i].time,
                                  memcpy_method[i].time*100./memcpy_method[0].time);
     }
  }
  for (i=0; i<n_func; i++) /* Freeing "times" buffers */
  {
      free(memcpy_method[i].times) ;
  }

  printf("xdtv: using %s\n", memcpy_method[best].name );
  fast_memcpy = memcpy_method[best].function;
  
  // save the result for the next time, so we can spare this
  // quiet long benchmark
  save_memcpy_method(memcpy_method[best].tag);

  free(buf1);
  free(buf2);
#ifdef FLUSH_CACHE
  free(buf3);
#endif
}
