/* Output pipe
 * Ben Lynn
 */
/*
Copyright (C) 2002 Benjamin Lynn (blynn@cs.stanford.edu)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>

//xmms includes
#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include "common.h"
#include "pipe.h"
#include "config.h"
#include "out.h"

#define OUTPIPELINK "outpipe"

static jumptable_t out_jt, report_jt;
static pipe_t outpipe;
static GString *outfile;
static gchar *outpipelink;

static void add_report_command(gchar *s, void (* function)())
{
    jumptable_add(report_jt, s, function);
}

static void add_out_command(gchar *s, void (* function)())
{
    jumptable_add(out_jt, s, function);
}

static void out_open()
{
    pipe_assign(outpipe, outfile->str);

    pipe_open(outpipe);
    if (!outpipe->is_open) {
	xmms_show_message("XMMSPipe Error", "Error opening output pipe", "Ok",
		FALSE, NULL, NULL);
    }

    common_symlink(outpipe->name, outpipelink);
}

static void out_close()
{
    pipe_close(outpipe);
}

static void out_setfile()
{
    if (arg) g_string_assign(outfile, arg);
}

static void out_setprefix()
{
    if (arg) g_string_assign(config->outprefix, arg);
}

static void out_on()
{
    if (!outpipe->is_open) out_open();
}

static void out_off()
{
    if (outpipe->is_open) out_close();
}

static void out_flush()
{
    pipe_readflush(outpipe);
}

static void report_str(char *s)
{
    if (!outpipe->is_open) return;
    pipe_write(outpipe, s);
}

static void report_title()
{
    gchar *msg;
    gint i = xmms_remote_get_playlist_pos(session);

    msg = g_strconcat(//"Title: ",
	    xmms_remote_get_playlist_title(session, i), "\n", NULL);
    report_str(msg);
    g_free(msg);
}

static void report_file()
{
    gchar *msg;
    gint i = xmms_remote_get_playlist_pos(session);

    msg = g_strconcat(//"File: ",
	    xmms_remote_get_playlist_file(session, i), "\n", NULL);
    report_str(msg);
    g_free(msg);
}

static void report_int(int i)
{
    char buf[32];
    sprintf(buf, "%d\n", i);
    report_str(buf);
}

static void report_is_playing()
{
    report_int(xmms_remote_is_playing(session));
}

static void report_is_paused()
{
    report_int(xmms_remote_is_paused(session));
}

static void report_position()
{
    report_int(xmms_remote_get_playlist_pos(session));
}

static void report_volume()
{
    int vl, vr;
    char kludge[32];
    xmms_remote_get_volume(session, &vl, &vr);
    sprintf(kludge, "%d %d\n", vl, vr);
    report_str(kludge);
}

static void report_balance()
{
    report_int(xmms_remote_get_balance(session));
}

static void report_output_time()
{
    report_int(xmms_remote_get_output_time(session));
}

static void report_playlist_time()
{
    gint i;
    if (!arg) {
	i = xmms_remote_get_playlist_pos(session);
	report_int(xmms_remote_get_playlist_time(session, i));
    } else {
	i = atoi(arg) - 1;
	report_int(xmms_remote_get_playlist_time(session, i));
    }
}

static void out_report()
{
    if (!outpipe->is_open) return;
    subcommand(report_jt);
}

static void out_setauto()
{
    setauto(&config->outauto);
}

static void out_command()
{
    subcommand(out_jt);
}

void out_init()
{
    jumptable_init(out_jt);
    jumptable_init(report_jt);
    pipe_init(outpipe);
    outpipelink = g_strconcat(homedir, "/.xmms/", OUTPIPELINK, NULL);
    outfile = g_string_new(config->outprefix->str);
    g_string_append(outfile, filesuffix);

    if (config->outauto) out_open();

    add_command("out", out_command);
    add_out_command("prefix", out_setprefix);
    add_out_command("file", out_setfile);
    add_out_command("start", out_on);
    add_out_command("on", out_on);
    add_out_command("off", out_off);
    add_out_command("report", out_report);
    add_out_command("auto", out_setauto);
    add_out_command("flush", out_flush);

    add_command("print", out_report);
    add_command("report", out_report);
    add_report_command("is_playing", report_is_playing);
    add_report_command("is_paused", report_is_paused);
    add_report_command("volume", report_volume);
    add_report_command("file", report_file);
    add_report_command("title", report_title);
    add_report_command("balance", report_balance);
    add_report_command("playtime", report_playlist_time);
    add_report_command("playlist_time", report_playlist_time);
    add_report_command("play_time", report_playlist_time);
    add_report_command("output_time", report_output_time);
    add_report_command("otime", report_output_time);
    add_report_command("pos", report_position);
    add_report_command("position", report_position);
}

void out_free()
{
    pipe_free(outpipe);
    jumptable_free(report_jt);
    jumptable_free(out_jt);
}
