/* XMMS Pipe Control Plugin
 * Ben Lynn
 */
/*
Copyright (C) 2002 Benjamin Lynn (blynn@cs.stanford.edu)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>

#include <pthread.h>

//xmms includes
#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

//for select
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

//for open
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>

//for errno
#include <errno.h>

#include "version.h"

#include "about.h"
#include "config.h"
#include "common.h"
#include "control.h"
#include "in.h"
#include "out.h"
#include "ack.h"

static pthread_t rm_thread;
static int is_rmthread;

static void init();
static void cleanup();

// XMMS will call the functions specified in the following struct

GeneralPlugin rm_gp =
{
    NULL,
    NULL,
    -1,
    PLUGIN_NAME,
    init,
    xmmspipe_about,
    xmmspipe_config,
    cleanup
};

extern GList *mainwin_wlist;

void parse_command(gchar *s)
{
    gchar *scopy = g_strdup(s);

    argsplit(s);
    common_execute(s);
    ack_send(scopy);

    g_free(scopy);
}

static GString* lastline = NULL;

static void read_from_pipe(gint infile)
    //reads one char at a time from the pipe
    //until '\n' is encountered
    //or no more is available
    //(TODO: fix this: if the input gets
    //fed to us in more than one piece
    //they will be interpreted as separate lines)
{
    gchar buffer[1];
    gint count;
    if (!lastline) lastline = g_string_new("");
    
    g_string_truncate(lastline, 0);
    for(;;) {
	count = read(infile, buffer, 1);

	if (count < 0) {
	    if (errno == EAGAIN) break;

	    perror("read");
	    g_string_assign(lastline, "");
	    return;
	}

	if (1 != count) break;
	if (buffer[0] == '\n') break;
	g_string_append_c(lastline, buffer[0]);
    }
}

static void *mainloop()
{
    gint status;
    fd_set set;
    gint maxfd;

    maxfd = controlpipe[0];
    if (inpipe->is_open && inpipe->readfd > maxfd) {
	maxfd = inpipe->readfd;
    }

    for(;;) {
	FD_ZERO(&set);
	if (inpipe->is_open) FD_SET(inpipe->readfd, &set);
	//assert(is_controlpipe);
	FD_SET(controlpipe[0], &set);
	status = select(maxfd + 1, &set, NULL, NULL, NULL);
	if (status == -1) break;

	if (FD_ISSET(inpipe->readfd, &set)) {
	    read_from_pipe(inpipe->readfd);
	    if (lastline->str[0]) {
		parse_command(lastline->str);
	    }
	}
	if (FD_ISSET(controlpipe[0], &set)) {
	    //TODO:
	    //can't use read_from_pipe because the pipes are blocking
	    //use fcntl() to change them to nonblocking?
	    char s[64];
	    read(controlpipe[0], s, 64);
	    if (!strncmp(s, C_QUIT, strlen(C_QUIT))) break;
	    if (!strncmp(s, C_RELOAD, strlen(C_RELOAD))) {
		in_reopen();
	    }
	}
    }
    return NULL;
}

static void init()
{
    control_init();
    if (!is_controlpipe) {
	xmms_show_message("XMMSPipe Error", "Cannot create control pipe", "Ok",
		FALSE, NULL, NULL);
	return;
    }

    config_load();

    common_init(rm_gp.xmms_session);
    in_init();
    out_init();
    ack_init();

    if (!pthread_create(&rm_thread, NULL, mainloop, NULL)) {
	is_rmthread = 1;
    }
}

static void cleanup()
{
    if (is_controlpipe) control_request(C_QUIT);

    if (is_rmthread) pthread_join(rm_thread, NULL);

    in_free();
    out_free();
    ack_free();
    control_free();
    common_free();
}

GeneralPlugin *get_gplugin_info(void)
{
    return &rm_gp;
}
