/*
 * arch/arm/mach-lh79520/dma-lh79520.c
 * Copyright (C) 2002 Embedix, Inc.
 *
 * Support functions for the Sharp LH79520 internal DMA channels.
 *
 * Based on arch/arm/mach-sa1100/dma-sa1100.c, which is
 * Copyright (C) 2000 Nicolas Pitre
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/slab.h>
#include <linux/errno.h>

#include <asm/system.h>
#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/mach/dma.h>
#include <asm/arch/iocon.h>


#undef DEBUG

#ifdef DEBUG
#define DPRINTK( s, arg... )  printk( "dma<%s>: " s, dma->device_id , ##arg )
#define DUMPREGS(r,d) \
    printk( "regs=0x%x   src=0x%x:%x  dest=0x%x:%x  count=%d  control=0x%x  tcnt=%d    mask=0x%x  status=0x%x\n", \
	    (u32)r, r->srcHi, r->srcLow, r->destHi, r->destLow, r->count, r->control, r->termCnt, d->mask, d->status)
#define DUMPQ(d)	dumpq(d)
#else
#define DPRINTK( x... )
#define DUMPREGS(r,d)
#define DUMPQ(d)
#endif

/*
 * DMA channel registers structure
 */
typedef struct {
	volatile u32	srcLow;		/* Source base addr, low 16 bits */
	volatile u32	srcHi;		/* Source base addr,  hi 16 bits */
	volatile u32	destLow;	/* Dest base addr,   low 16 bits */ 
	volatile u32	destHi;		/* Dest base addr,    hi 16 bits */ 
	volatile u32	count;		/* Maximum Count */ 		
	volatile u32	control;	/* Control */ 
	volatile u32	currSrcHi;	/* Current src addr,  hi 16 bits*/ 
	volatile u32	currSrcLow;	/* Current src addr, low 16 bits*/ 
	volatile u32	currDstHi;	/* Curr dest addr,    hi 16 bits*/ 
	volatile u32	currDstLow;	/* Curr src addr,    low 16 bits*/ 
	volatile u32	termCnt;	/* Terminal Count */ 		
} channelRegs_t;


/*
 * Control Register Bit Field
 */ 
#define DMAC_CTRL_ENABLE		_BIT(0)		/* Enable DMA */ 
#define DMAC_CTRL_SOINC			_BIT(1)		/* Source Reg inc.bit */ 
#define DMAC_CTRL_DEINC			_BIT(2)		/* Dest Reg inc.bit */ 
/* Source Size */ 
#define DMAC_CTRL_SOSIZE_1BYTE		_SBF(3,0)
#define DMAC_CTRL_SOSIZE_2BYTE		_SBF(3,1)
#define DMAC_CTRL_SOSIZE_4BYTE		_SBF(3,2)
/* Destination Size */ 
#define DMAC_CTRL_DESIZE_1BYTE		_SBF(7,0)
#define DMAC_CTRL_DESIZE_2BYTE		_SBF(7,1)
#define DMAC_CTRL_DESIZE_4BYTE		_SBF(7,2)
/* Peripheral Burst Sizes */ 
#define DMAC_CTRL_SOBURST_SINGLE	_SBF(5,0)	/* Single */ 
#define DMAC_CTRL_SOBURST_4INC		_SBF(5,1)	/* 4 incrementing */ 
#define DMAC_CTRL_SOBURST_8INC		_SBF(5,2)	/* 8 incrementing */ 
#define DMAC_CTRL_SOBURST_16INC		_SBF(5,3)	/* 16 incrementing */ 
/* Address Modes */ 
#define DMAC_CTRL_ADDR_MODE_WRAP	_SBF(9,0)
#define DMAC_CTRL_ADDR_MODE_INCR	_SBF(9,1)

#define DMAC_CTRL_MEM2MEM		_BIT(11)	/* Memory to Memory */ 
/* Direction */ 
#define DMAC_CTRL_PERIPH_SOURCE		_SBF(13,0)
#define DMAC_CTRL_PERIPH_DEST		_SBF(13,1)


typedef struct {
	channelRegs_t	stream0;	/* Data Stream 0 */ 
	volatile u32	reserved0[5];
	channelRegs_t	stream1;	/* Data Stream 1 */ 
	volatile u32	reserved1[5];
	channelRegs_t	stream2;	/* Data Stream 2 */ 
	volatile u32	reserved2[5];
	channelRegs_t	stream3;	/* Data Stream 3 */ 
	volatile u32	reserved3;
	volatile u32	mask;
	volatile u32	clear;
	volatile u32	status;
	volatile u32	reserved4;
} dmaRegs_t;

channelRegs_t *streamRegs[] = {
	&((dmaRegs_t *)IO_ADDRESS(DMAC_PHYS))->stream0,
	&((dmaRegs_t *)IO_ADDRESS(DMAC_PHYS))->stream1,
	&((dmaRegs_t *)IO_ADDRESS(DMAC_PHYS))->stream2,
	&((dmaRegs_t *)IO_ADDRESS(DMAC_PHYS))->stream3
};

/*
 * mask   - Mask Register Bit Fields
 * clear  - Clear Register Bit Fields
 * status - Clear Register Bit Fields
 *
 * Writing DMAC_xN to mask register enables corresponding interrupt
 * Writing DMAC_xN to clear register disables corresponding interrupt
 * AND'ing DMAC_xN with status register yields status
 * Note: "ACTIVEx" constants are only applicable to Status Register
 */ 
#define	DMAC_INT0	_BIT(0)		/* Stream 0 Interrupt */ 
#define	DMAC_INT1	_BIT(1)		/* Stream 1 Interrupt */ 
#define	DMAC_INT2	_BIT(2)		/* Stream 2 Interrupt */ 
#define	DMAC_INT3	_BIT(3)		/* Stream 3 Interrupt */ 
#define	DMAC_ERRINT0	_BIT(4)		/* Stream 0 Error Interrupt */
#define	DMAC_ERRINT1	_BIT(5)		/* Stream 1 Error Interrupt */ 
#define	DMAC_ERRINT2	_BIT(6)		/* Stream 2 Error Interrupt */ 
#define	DMAC_ERRINT3	_BIT(7)		/* Stream 3 Error Interrupt */ 
#define	DMAC_ACTIVE0	_BIT(8)		/* Stream 0 Active */ 
#define	DMAC_ACTIVE1	_BIT(9)		/* Stream 1 Active */ 
#define	DMAC_ACTIVE2	_BIT(10)	/* Stream 2 Active */ 
#define	DMAC_ACTIVE3	_BIT(11)	/* Stream 3 Active */ 

/* all DMA error bits */
#define DMAC_ERROR	(DMAC_ERRINT0 | DMAC_ERRINT1 | DMAC_ERRINT2 | DMAC_ERRINT3 )

/* all DMA done bits */
#define DMAC_DONE	(DMAC_INT0 | DMAC_INT1 | DMAC_INT2 | DMAC_INT3)

/* all the bits in the clear register */
#define DMAC_CLEAR_ALL	(DMAC_DONE | DMAC_ERROR)


#include "dma.h"

lh79520_dma_t dma_chan[LH79520_DMA_CHANNELS];

/*
 * Maximum physical DMA buffer size
 */
#define MAX_DMA_SIZE		0x3ffff
#define MAX_DMA_ORDER		18


static inline void dumpq (lh79520_dma_t *dma)
{
    dma_buf_t *p=dma->tail;

    printk( "Q: curr=0x%p  tail=0x%p  head=0x%p  bid: ", dma->curr, dma->tail, dma->head);

    while( p) {
	printk( "(0x%p 0x%p)  ", p, p->id);
	p = p->next;
    }
    printk("\n");
}


/*
 * DMA processing...
 */

static inline int start_lh79520_dma(lh79520_dma_t * dma, dma_addr_t dma_ptr, int size)
{
    dmaRegs_t     *dmaRegs = (dmaRegs_t *)IO_ADDRESS(DMAC_PHYS);
    cpldRegs_t    *cpld    = (cpldRegs_t *)0; //CPLD_BASE;
    channelRegs_t *regs    = dma->regs;
    int status;

    status = dmaRegs->status;

    /* If the DMA channel is active, there's nothing else we can do. */
    if( status & (DMAC_ACTIVE0 << dma->channel)) {
	DPRINTK("start: st %#x busy\n", status);
	return -EBUSY;
    }

    /* If there's an interrupt pending, split now
     * and let it happen.
     */
    if( status & (DMAC_INT0 << dma->channel)) {
	DPRINTK("start: st %#x IRQ pending\n", status);
	return -EAGAIN;
    }

	printk("XXXXXXXXX DMA NOT YET SUPPORTED ON LPD PLATFORM XXXXXXXXX\n");
	return -EBUSY;

    /*
     * if we're goint to the uda1341, we have to tell the CPLD
     * to start to send/receive via DMA.
     */
    switch( dma->channel) {
	case 2:
	    cpld->audio_control |= CPLD_DAC_DMA_ENABLE;
	    dmaRegs->mask  |= DMAC_INT2;
	    break;

	case 3:
	    cpld->audio_control |= (CPLD_DAC_DMA_ENABLE | CPLD_DAC_USE_REQ1 );
	    dmaRegs->mask  |= DMAC_INT3;
	    break;
    }

    /*
     * set the source or destination registers, based on which 
     * direction the data's going.
     */
    if( dma->direction == DMA_IN) {	/* data coming from peripheral */
	regs->destLow  = dma_ptr & 0xffff;
	regs->destHi   = (dma_ptr >> 16 ) & 0xffff;
    } else {				/* data going to peripheral */
	regs->srcLow   = dma_ptr & 0xffff;
	regs->srcHi    = (dma_ptr >> 16 ) & 0xffff;
    }

    regs->count    = size >> 2;		/* DDD assumes 4-byte transfer size */
    regs->control |= DMAC_CTRL_ENABLE;

    DPRINTK("audio_control=0x%x\n", cpld->audio_control);
    DPRINTK("jif=%d start a=%#x sz=%d  st=0x%x  dma=0x%p  dir=%d\n",
	    jiffies, dma_ptr, size, status, dma, dma->direction);
    DUMPREGS(regs,dmaRegs);

#if 0
    {
	u32 *p = phys_to_virt(dma_ptr);
	int i;

	for( i=0; i<8; i++)
	    printk( " %08x %08x %08x %08x\n", *p++, *p++, *p++, *p++);
    }
#endif // 0

    return 0;
}


static int start_dma(lh79520_dma_t *dma, dma_addr_t dma_ptr, int size)
{
	return start_lh79520_dma(dma, dma_ptr, size);
}


/* This must be called with IRQ disabled */
static void process_dma(lh79520_dma_t * dma)
{
    dma_buf_t *buf;
    int chunksize;

    DUMPQ(dma);

	printk("XXXXXXXXX DMA NOT YET SUPPORTED ON LPD PLATFORM XXXXXXXXX\n");
	return -EBUSY;

    for (;;) {
	buf = dma->tail;

	if (!buf || dma->stopped) {
	    /* no more data available */
	    DPRINTK("process: no more buf (dma %s)  buf=0x%p  stopped=%d\n",
		    dma->curr ? "active" : "inactive", buf, dma->stopped);
	    /*
	     * Some devices may require DMA still sending data
	     * at any time for clock reference, etc.
	     * Note: if there is still a data buffer being
	     * processed then the ref count is negative.  This
	     * allows for the DMA termination to be accounted in
	     * the proper order.
	     */
	    if (dma->spin_size && dma->spin_ref >= 0) {
		chunksize = dma->spin_size;
		if (chunksize > MAX_DMA_SIZE)
		    chunksize = (1 << MAX_DMA_ORDER);
		while (start_dma(dma, dma->spin_addr, chunksize) == 0)
		    dma->spin_ref++;
		if (dma->curr != NULL)
		    dma->spin_ref = -dma->spin_ref;
	    }
	    break;
	}

	/*
	 * Let's try to start DMA on the current buffer.
	 * If DMA is busy then we break here.
	 */
	chunksize = buf->size;
	if (chunksize > MAX_DMA_SIZE)
	    chunksize = (1 << MAX_DMA_ORDER);

	DPRINTK("process: bid=%#x s=%d\n", (int) buf->id, buf->size);
	if (start_dma(dma, buf->dma_ptr, chunksize) != 0)
	    break;

	if (!dma->curr) {
	    dma->curr = buf;
	    DPRINTK("process: set curr %#p\n", dma->curr);
	}

	buf->ref++;
	buf->dma_ptr += chunksize;
	buf->size -= chunksize;
	if (buf->size == 0) {
	    /* current buffer is done: move tail to the next one */
	    dma->tail = buf->next;
	    DPRINTK("process: set tail b=%#x\n", (int) dma->tail);
	}
    }

    DUMPQ(dma);
}


/* This must be called with IRQ disabled */
void lh79520_dma_done (lh79520_dma_t *dma)
{
    dma_buf_t *buf = dma->curr;

    if (dma->spin_ref > 0) {
	dma->spin_ref--;
    } else if (buf) {
	buf->ref--;
	if (buf->ref == 0 && buf->size == 0) {
	    /*
	     * Current buffer is done.
	     * Move current reference to the next one and send
	     * the processed buffer to the callback function,
	     * then discard it.
	     */
	    DPRINTK("IRQ: buf done  set curr=%#p\n", buf->next);
	    dma->curr = buf->next;
	    if (dma->curr == NULL)
		dma->spin_ref = -dma->spin_ref;
	    if (dma->head == buf)
		dma->head = NULL;
	    if (dma->callback) {
		int size = buf->dma_ptr - buf->dma_start;
		dma->callback(buf->id, size);
	    }
	    kfree(buf);
	}
    }

    process_dma(dma);
}


static void dma_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
{
    dmaRegs_t *dmaRegs  = (dmaRegs_t *)IO_ADDRESS(DMAC_PHYS);
    lh79520_dma_t *dma = (lh79520_dma_t *) dev_id;
    int status = dmaRegs->status;

    DPRINTK("jif=%d  IRQ: irq=%d  regs=0x%x  bid=%#x st=%#x, dma=0x%p\n",
	    jiffies, irq, (u32)dma->regs, (int) dma->curr->id, status, dma);

    if (status & (DMAC_ERROR)) {
	printk(KERN_ERR "DMA on \"%s\" caused an error\n", dma->device_id);
	dmaRegs->clear = DMAC_ERROR;
    }

    dmaRegs->clear = status & DMAC_DONE;
    
    if (status & DMAC_DONE)
	lh79520_dma_done (dma);
}


/*
 * DMA interface functions
 */

static spinlock_t dma_list_lock;

int lh79520_request_dma (dmach_t * channel, const char *device_id, dma_device_t device)
{
    lh79520_dma_t *dma = NULL;
    cpldRegs_t    *cpld    = (cpldRegs_t *)0; //CPLD_BASE;
    channelRegs_t *regs;
    int i, err;

    /* DMA address (physical) of audio device */
    void *cpldAudioAddr = (void *)0;// &((cpldRegs_t *)CPLD_START)->adc_dac_left;

	printk("XXXXXXXXX DMA NOT YET SUPPORTED ON LPD PLATFORM XXXXXXXXX\n");
	return -EBUSY;

#ifdef DEBUG
    printk( __FUNCTION__ "(channel=0x%x,  device_id=0x%x  device=0x%x)\n",
	    (u32)channel, (u32)device_id, device);
#endif

    *channel = -1;		/* to be sure we catch the freeing of a misregistered channel */

    err = 0;
    spin_lock(&dma_list_lock);

    /*
     * Allocate a channel.  On the lh79520, channels 0 and 1
     * are dedicated to the SSP.  Channels 2 and 3 are general purpose,
     * but channel 3 can only be used for audio if the rework described
     * in the User's Guide has been performed.
     */
    switch( device) {
	case DMA_Audio_Out:
	    dma = &dma_chan[2];
	    regs = dma->regs;

	    dma->direction = DMA_OUT;

	    regs->destLow = (u32)cpldAudioAddr & 0xffff;
	    regs->destHi  = ((u32)cpldAudioAddr >> 16) & 0xffff;
	    regs->control = DMAC_CTRL_SOINC | 
			    DMAC_CTRL_SOSIZE_4BYTE |
			    DMAC_CTRL_DESIZE_4BYTE |
			    DMAC_CTRL_SOBURST_SINGLE |
			    DMAC_CTRL_ADDR_MODE_WRAP |
			    DMAC_CTRL_PERIPH_DEST;
	    break;

	case DMA_Audio_In:
	    dma = &dma_chan[3];
	    regs = dma->regs;

	    dma->direction = DMA_IN;

	    regs->srcLow = (u32)cpldAudioAddr & 0xffff;
	    regs->srcHi  = ((u32)cpldAudioAddr >> 16) & 0xffff;
	    regs->control = DMAC_CTRL_SOINC | 
			    DMAC_CTRL_SOSIZE_4BYTE |
			    DMAC_CTRL_DESIZE_4BYTE |
			    DMAC_CTRL_SOBURST_SINGLE |
			    DMAC_CTRL_ADDR_MODE_WRAP |
			    DMAC_CTRL_PERIPH_SOURCE;
	    break;

	case DMA_SSP_Rx:	/* not supported */
	case DMA_SSP_Tx:	/* not supported */
	default:
	    err = -ENOSR;
	    break;
    }

    if (!err) {
       if (dma)
	   dma->in_use = 1;
       else
	   err = -ENOSR;
    }
    spin_unlock(&dma_list_lock);
    if (err)
	return err;

    err = request_irq(dma->irq, dma_irq_handler, SA_INTERRUPT,
		      device_id, (void *) dma);
    if (err) {
	printk(KERN_ERR
	       "%s: unable to request IRQ %d for DMA channel.  error=0x%x\n",
	       device_id, dma->irq, err);
	return err;
    }

    *channel = dma - dma_chan;
    dma->device_id = device_id;
    dma->device = device;
    dma->callback = NULL;
    dma->spin_size = 0;

    regs = dma->regs;

    DPRINTK( "channel=%d  regs=0x%x\n", *channel, (u32)regs);
    DPRINTK("requested\n");

    return 0;
}


int lh79520_dma_set_callback(dmach_t channel, dma_callback_t cb)
{
    lh79520_dma_t *dma = &dma_chan[channel];

    if ((unsigned)channel >= LH79520_DMA_CHANNELS || !dma->in_use)
	return -EINVAL;

    dma->callback = cb;
    DPRINTK("cb = %p\n", cb);
    return 0;
}


int lh79520_dma_set_spin(dmach_t channel, dma_addr_t addr, int size)
{
    lh79520_dma_t *dma = &dma_chan[channel];
    int flags;

    if ((unsigned)channel >= LH79520_DMA_CHANNELS || !dma->in_use)
	return -EINVAL;

    DPRINTK("set spin %d at %#x\n", size, addr);
    local_irq_save(flags);
    dma->spin_addr = addr;
    dma->spin_size = size;
    if (size)
	process_dma(dma);

    local_irq_restore(flags);
    return 0;
}


int lh79520_dma_queue_buffer(dmach_t channel, void *buf_id,
			    dma_addr_t data, int size)
{
    lh79520_dma_t *dma;
    dma_buf_t *buf;
    int flags;

    dma = &dma_chan[channel];
    if ((unsigned)channel >= LH79520_DMA_CHANNELS || !dma->in_use)
	return -EINVAL;

    buf = kmalloc(sizeof(*buf), GFP_ATOMIC);
    if (!buf)
	return -ENOMEM;

    buf->next = NULL;
    buf->ref = 0;
    buf->dma_ptr = buf->dma_start = data;
    buf->size = size;
    buf->id = buf_id;

    local_irq_save(flags);

    DPRINTK("queueing bid=%#x a=%#x s=%d\n", (int) buf_id, data, size);

    if (dma->head)
	dma->head->next = buf;

    dma->head = buf;
    if (!dma->tail)
	dma->tail = buf;

    process_dma(dma);
    local_irq_restore(flags);

    return 0;
}


int lh79520_dma_get_current(dmach_t channel, void **buf_id, dma_addr_t *addr)
{
    int flags, ret;
    lh79520_dma_t *dma = &dma_chan[channel];
    channelRegs_t *regs;

    if ((unsigned)channel >= LH79520_DMA_CHANNELS || !dma->in_use)
	return -EINVAL;

    regs = dma->regs;
    local_irq_save(flags);
    if (dma->curr && dma->spin_ref <= 0) {
	dma_buf_t *buf = dma->curr;

	/*
	 * If we got here, that's because there is, or recently was, a
	 * buffer being processed. Two possibilities: either we are
	 * in the middle of a buffer, or the DMA controller just
	 * switched to the next toggle but the interrupt hasn't been
	 * serviced yet.  The former case is straight forward.  In
	 * the later case, we'll do like if DMA is just at the end
	 * of the previous toggle since all registers haven't been
	 * reset yet.  This goes around the edge case and since we're
	 * always a little behind anyways it shouldn't make a big
	 * difference.  If DMA has been stopped prior calling this
	 * then the position is always exact.
	 */
	if (buf_id)
		*buf_id = buf->id;

	if( dma->direction == DMA_IN)
	    *addr = (regs->currDstHi << 16 ) | (regs->currDstLow);
	else
	    *addr = (regs->currSrcHi << 16 ) | (regs->currSrcLow);

	/*
	 * Clamp funky pointers sometimes returned by the hardware
	 * on completed DMA transfers
	 */
	if (*addr < buf->dma_start ||
	    *addr > buf->dma_ptr)
		*addr = buf->dma_ptr;
	DPRINTK("curr_pos: b=%#x a=%#x\n", (int)dma->curr->id, *addr);
	ret = 0;
    } else if (dma->tail && dma->stopped) {
	dma_buf_t *buf = dma->tail;
	if (buf_id)
		*buf_id = buf->id;
	*addr = buf->dma_ptr;
	ret = 0;
    } else {
	if (buf_id)
		*buf_id = NULL;
	*addr = 0;
	ret = -ENXIO;
    }
    local_irq_restore(flags);
    return ret;
}


int lh79520_dma_stop(dmach_t channel)
{
    dmaRegs_t     *dmaRegs = (dmaRegs_t *)IO_ADDRESS(DMAC_PHYS);
    lh79520_dma_t *dma = &dma_chan[channel];
    int flags;

    DPRINTK( "lh79520_dma_stop channel=%d\n", channel);

    if (dma->stopped)
	return 0;

    local_irq_save(flags);
    dma->stopped = 1;

    /*
     * Stop DMA and tweak state variables so everything could restart
     * from there when resume/wakeup occurs.
     */
    dma->regs->control &= ~DMAC_CTRL_ENABLE;
    dmaRegs->mask &= ~((DMAC_INT0 << channel) | (DMAC_ERRINT0 << channel));
    
    if (dma->curr) {
	dma_buf_t *buf = dma->curr;
	if (dma->spin_ref <= 0) {
	    dma_addr_t curpos;
	    lh79520_dma_get_current(channel, NULL, &curpos);
	    buf->size += buf->dma_ptr - curpos;
	    buf->dma_ptr = curpos;
	}
	buf->ref = 0;
	dma->tail = buf;
	dma->curr = NULL;
    }
    dma->spin_ref = 0;
    process_dma(dma);
    local_irq_restore(flags);
    return 0;
}


int lh79520_dma_resume(dmach_t channel)
{
    lh79520_dma_t *dma = &dma_chan[channel];

    if ((unsigned)channel >= LH79520_DMA_CHANNELS || !dma->in_use)
	return -EINVAL;

    if (dma->stopped) {
	int flags;
	save_flags_cli(flags);
	dma->stopped = 0;
	dma->spin_ref = 0;
	process_dma(dma);
	restore_flags(flags);
    }
    return 0;
}


int lh79520_dma_flush_all(dmach_t channel)
{
    dmaRegs_t     *dmaRegs = (dmaRegs_t *)IO_ADDRESS(DMAC_PHYS);
    lh79520_dma_t *dma = &dma_chan[channel];
    dma_buf_t *buf, *next_buf;
    int flags;

    DPRINTK("dma_flush_all channel=%d\n", channel);
    DUMPQ(dma);

    if ((unsigned)channel >= LH79520_DMA_CHANNELS || !dma->in_use)
	return -EINVAL;

    local_irq_save(flags);

    /*
     * Disable the channel, and mask off its interrupts
     */
    dma->regs->control &= ~(DMAC_CTRL_ENABLE);
    dmaRegs->mask &= ~((DMAC_INT0 << channel) | (DMAC_ERRINT0 << channel));

    buf = dma->curr;
    if (!buf)
	buf = dma->tail;

    dma->head = dma->tail = dma->curr = NULL;
    dma->stopped = 0;
    dma->spin_ref = 0;
    process_dma(dma);
    local_irq_restore(flags);

    while (buf) {
	next_buf = buf->next;
	kfree(buf);
	buf = next_buf;
    }
    DPRINTK("flushed\n");
    return 0;
}


void lh79520_free_dma(dmach_t channel)
{
    lh79520_dma_t *dma;

    if ((unsigned)channel >= LH79520_DMA_CHANNELS)
	return;

    dma = &dma_chan[channel];
    if (!dma->in_use) {
	printk(KERN_ERR "Trying to free free DMA%d\n", channel);
	return;
    }

    lh79520_dma_set_spin(channel, 0, 0);
    lh79520_dma_flush_all(channel);

    free_irq(dma->irq, (void *) dma);
    dma->in_use = 0;

    DPRINTK("freed\n");
}


EXPORT_SYMBOL(lh79520_request_dma);
EXPORT_SYMBOL(lh79520_dma_set_callback);
EXPORT_SYMBOL(lh79520_dma_set_spin);
EXPORT_SYMBOL(lh79520_dma_queue_buffer);
EXPORT_SYMBOL(lh79520_dma_get_current);
EXPORT_SYMBOL(lh79520_dma_stop);
EXPORT_SYMBOL(lh79520_dma_resume);
EXPORT_SYMBOL(lh79520_dma_flush_all);
EXPORT_SYMBOL(lh79520_free_dma);


#ifdef CONFIG_PM
/* Drivers should call this from their PM callback function */

int lh79520_dma_sleep(dmach_t channel)
{
    dmaRegs_t     *dmaRegs = (dmaRegs_t *)IO_ADDRESS(DMAC_PHYS);
    lh79520_dma_t *dma = &dma_chan[channel];
    int orig_state;

    if ((unsigned)channel >= LH79520_DMA_CHANNELS || !dma->in_use)
	return -EINVAL;

    orig_state = dma->stopped;
    lh79520_dma_stop(channel);
    dma->regs->control &= ~DMAC_CTRL_ENABLE;
    dmaRegs->mask &= ~((DMAC_INT0 << channel) | (DMAC_ERRINT0 << channel));

    dma->stopped = orig_state;
    dma->spin_ref = 0;
    return 0;
}

int lh79520_dma_wakeup(dmach_t channel)
{
    dmaRegs_t     *dmaRegs = (dmaRegs_t *)IO_ADDRESS(DMAC_PHYS);
    lh79520_dma_t *dma = &dma_chan[channel];
    int flags;

    if ((unsigned)channel >= LH79520_DMA_CHANNELS || !dma->in_use)
	return -EINVAL;

    dma->regs->control &= ~DMAC_CTRL_ENABLE;
    dmaRegs->mask &= ~((DMAC_INT0 << channel) | (DMAC_ERRINT0 << channel));

    local_irq_save(flags);
    process_dma(dma);
    local_irq_restore(flags);

    return 0;
}

EXPORT_SYMBOL(lh79520_dma_sleep);
EXPORT_SYMBOL(lh79520_dma_wakeup);

#endif /* CONFIG_PM */


static int __init lh79520_init_dma(void)
{
    int channel;
    dmaRegs_t *dmaRegs  = (dmaRegs_t *)IO_ADDRESS(DMAC_PHYS);
    ioconRegs_t *ioconRegs = (ioconRegs_t *)IO_ADDRESS( IOCON_PHYS);
    channelRegs_t *regs;

#ifdef DEBUG
    printk( __FUNCTION__ "\n");
#endif

    dmaRegs->clear = DMAC_CLEAR_ALL;
    dmaRegs->mask  = 0;

    for (channel = 0; channel < LH79520_DMA_CHANNELS; channel++) {
	dma_chan[channel].regs = regs = streamRegs[channel];

	regs->control = 0;
	regs->count   = 0;
	regs->srcHi   = 0;
	regs->srcLow  = 0;
	regs->destHi  = 0;
	regs->destLow = 0;

	dma_chan[channel].irq = IRQ_DMA;
	dma_chan[channel].channel = channel;

#ifdef DEBUG
	printk( "dma channel %d at 0x%x\n", channel, (u32)regs);
#endif
    }

    /* assign pins to DMA stream 2 */
    // ioconRegs->DMAMux |= (DMAMUX_DCDEOT0 | DMAMUX_DCDREQ0) ;

    /* assign pins to DMA Stream 3 */
    // These two lines interfere with PWM Audio
    // ioconRegs->MiscMux |= MISCMUX_DCDEOT1;
    // ioconRegs->MiscMux &= ~MISCMUX_RCEII5;

    return 0;
}

__initcall(lh79520_init_dma);
