/* vi: set sw=4 ts=4 ai: */

// #define MODULE

#define TS_DATA_LH79X

/**********************************************************************
*  linux/drivers/misc/ads784x.c
*
*  Provide ADS_784x (touchscreen) functionality for LH7x EVB boards
*
*  Copyright (C) 2002  Lineo, Inc.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License (GPL) version 2
*  as published by the Free Software Foundation.
*
**********************************************************************/

/**********************************************************************
* Algorithm:
*
*	The driver sleeps when there is no pen on the screen.  When a pen_down
*	interrupt occurs, the pen_down interrupt handler wakes the polling thread.
*	The polling thread polls the ADS chip SAMPLES_PER_SECOND.  When the polling
*	thread polls the ADS chip and the pen is no longer down, the polling
*	thread goes to sleep and the pen_down interrupt handler is enabled.
*
*	The ADS device sleeps between conversions to save power.
*
*	The driver stores pen coordinates in a queue. An application can access
*	the coordinate queue by opening and reading the associated device file
*	(char major=10 minor=20).  An application can poll the queue to see if
*	it contains coordinates. If it does, the application can read a coordinate
*	structure back. The coordinate queue is flushed when the application
*	opens the device file.
*
**********************************************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
//#include <linux/signal.h>
//#include <linux/sched.h>
#include <linux/delay.h>
//#include <linux/wait.h>
//#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/miscdevice.h>
#include <linux/poll.h>

#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <asm/arch/platform.h>

//#undef DEBUG
//#undef VERBOSE
#define DEBUG
#define VERBOSE
//#define PRINTK printk
#define PRINTK(x, ...)
#define PRINTK2(x, ...)

#define DRVNAME "ads_784x"
#include <linux/verbosedebug.h>

#include <linux/version.h>
#ifdef MODULE
char kernel_version[] = UTS_RELEASE;
#endif /* MODULE */

#include "ssp.h"

#if defined(CONFIG_ADS7846) && defined(CONFIG_PROC_FS)
#define PROC_TEMPERATURE
#define PROC_BATTERY
#endif

#ifdef PROC_BATTERY
static struct proc_dir_entry *ads_784x_proc_battery;
#endif
#ifdef PROC_TEMPERATURE
static struct proc_dir_entry *ads_784x_proc_temperature;
#endif

/*********************************************************************
* A couple of macros we use here...
*********************************************************************/
#define jiffies_from_ms(a) ((((a) * HZ)/1000)+1)

#ifndef _BIT
#define _BIT(n) (1 << (n))
#endif
#ifndef _SBF
#define _SBF(f,v) ((v) << (f))
#endif
#ifndef _BITMASK
#define _BITMASK(field_width) ( _BIT(field_width) - 1)
#endif

/**********************************************************************
* Define ADS 784x Control byte
**********************************************************************/

#define CTRL_S				_BIT(7)			/* Start bit (REQUIRED) */

// #define CTRL_ADDR(n)		_SBF(4,(n&0x7))
#define CTRL_X				_SBF(4,0x5)		/* Read X value */
#define CTRL_Y				_SBF(4,0x1)		/* Read Y value */
#define CTRL_P1				_SBF(4,0x3)		/* Read P1 value */
#define CTRL_P2				_SBF(4,0x4)		/* Read P2 value */
#define CTRL_BATTERY		_SBF(4,0x2)		/* Read BATTERY value */
#define CTRL_TEMP0			_SBF(4,0x0)		/* Read Temperature-0 value */
#define CTRL_TEMP1			_SBF(4,0x7)		/* Read Temperature-1 value */

// #define CTRL_MODE(n)		_SBF(3,(n&0x1))
#define CTRL_12BIT			_SBF(3,0x0)		/* 12-Bit conversions  */
#define CTRL_8BIT			_SBF(3,0x1)		/* 8-Bit conversions */

// #define CTRL_SER_DFR(n)	_SBF(2,(n&0x1))
#define CTRL_SER			_SBF(2,0x1)		/* Single-Ended reference */
#define CTRL_DFR			_SBF(2,0x0)		/* Differential reference */

// #define CTRL_PD(n)		(n&0x3)
#define CTRL_PD_ZERO		(0x0)		/* PenIRQ enabled - PowerDown after */
#define CTRL_PD_ONE			(0x1)		/* PenIRQ disabled - PowerDown after */
#define CTRL_PD_TWO			(0x2)		/* PenIRQ disabled - RESERVED */
#define CTRL_PD_THREE		(0x3)		/* PenIRQ disabled - Powered ON */

/**********************************************************************
* Define Simplified ADS 784x Control Codes
**********************************************************************/

//#define ADS_784x_PD      0   /* Power Down Settings */
//#define ADS_784x_PD_INIT 0   /* Init setting reference and ADC Off */

#define ADS_784x_INIT \
	( CTRL_S | CTRL_BATTERY | CTRL_12BIT | CTRL_SER | CTRL_PD_ZERO)
#define ADS_784x_READ_X \
	( CTRL_S | CTRL_X | CTRL_12BIT | CTRL_DFR | CTRL_PD_ZERO)
#define ADS_784x_READ_Y \
	( CTRL_S | CTRL_Y | CTRL_12BIT | CTRL_DFR | CTRL_PD_ZERO)
#define ADS_784x_READ_P1 \
	( CTRL_S | CTRL_P1 | CTRL_12BIT | CTRL_DFR | CTRL_PD_ZERO)
#define ADS_784x_READ_P2 \
	( CTRL_S | CTRL_P2 | CTRL_12BIT | CTRL_DFR | CTRL_PD_ZERO)
#define ADS_784x_READ_BATTERY \
	( CTRL_S | CTRL_BATTERY | CTRL_12BIT | CTRL_SER | CTRL_PD_ZERO)
#define ADS_784x_READ_TEMP0 \
	( CTRL_S | CTRL_TEMP0 | CTRL_12BIT | CTRL_SER | CTRL_PD_ZERO)
#define ADS_784x_READ_TEMP1 \
	( CTRL_S | CTRL_TEMP1 | CTRL_12BIT | CTRL_SER | CTRL_PD_ZERO)

/**********************************************************************
* Define our touchscreen data type
*
* This structure is nonsense - millisecs is not very useful
* since the field size is too small.  Also, we SHOULD NOT
* be exposing jiffies to user space directly.
**********************************************************************/
typedef struct tsData_t tsData_t;

#ifdef TS_DATA_COLLIE
#define TS_DATA_STRUCT
#define PROVIDE_TS_MILLISECS
struct tsData_t {
	long x;
	long y;
	long pressure;
	long long millisecs;
};
#endif

#ifdef TS_DATA_IPAQ
#define TS_DATA_STRUCT
struct tsData_t {
	unsigned short pressure;
	unsigned short x;
	unsigned short y;
	unsigned short pad;
};
#endif 

#ifdef TS_DATA_LH79X
#define TS_DATA_STRUCT
struct tsData_t {
	uint16_t pressure;
	uint16_t y;		/* Will be read as X */
	uint16_t x;		/* Will be read as Y */
	uint16_t pad;
};
#endif

#ifndef TS_DATA_STRUCT
#define TS_DATA_DEFAULT
#define PROVIDE_TS_TIMESTAMP
struct tsData_t {
	uint16_t pressure;
	uint16_t x;
	uint16_t y;
	uint16_t pad;
	struct timeval stamp;
};
#endif

#define MAX_TS_DATA 16

#define X_DELTA_MAX 20
#define Y_DELTA_MAX 30

/* Define the readings we are to take per second (default to 50) */
#define SAMPLES_PER_SECOND	50

/*
* Define the pen down debounce we are to take.
* (default of 20 == 1/20 of a second)
*/
#define DEBOUNCE_FRACTION_OF_A_SECOND	50

/**********************************************************************
* Define our ADS context structure
**********************************************************************/
typedef struct adsContext_t adsContext_t;
struct adsContext_t {

	void          *sspContext;
	void          (*write) (void *sspContext, unsigned char data);
	unsigned char (*read) (void *sspContext);
	int           (*lock)(void *sspContext, int device);
	int           (*unlock)(void *sspContext, int device);

	struct fasync_struct *fasync;
	struct completion complete;
	struct task_struct *rtask;

	wait_queue_head_t read_wait;
	wait_queue_head_t irq_wait;

	int tsDataHead;
	int tsDataTail;
	tsData_t tsData[MAX_TS_DATA];
};

static adsContext_t adsContext_l;
wait_queue_head_t *irq_wait_ptr;

/**********************************************************************
* Macro: ads_784x_tsData_pending
**********************************************************************/
#define ads_784x_tsData_pending(adsContext) \
	((adsContext)->tsDataHead != (adsContext)->tsDataTail)

/**********************************************************************
* Macro: ads_784x_tsData_get
**********************************************************************/
#define ads_784x_tsData_get(adsContext) \
	((adsContext)->tsData + (adsContext)->tsDataTail)

/**********************************************************************
* Macro: ads_784x_tsData_pull
**********************************************************************/
#define ads_784x_tsData_pull(adsContext) \
	((adsContext)->tsDataTail = \
		((adsContext)->tsDataTail + 1) & (MAX_TS_DATA - 1))

/**********************************************************************
* Macro: ads_784x_tsData_flush
**********************************************************************/
#define ads_784x_tsData_flush(adsContext) \
	((adsContext)->tsDataTail = (adsContext)->tsDataHead)

/**********************************************************************
* Macro:
**********************************************************************/
static int
is_pen_down(adsContext_t *adsContext)
{
	int pen_down;
	volatile u_int v;
	u_int tmp;
	void *sspContext = adsContext->sspContext;
	spinlock_t lock;
	long flags;

	spin_lock_irqsave(&lock, flags);

	/* setting penirq re-enables the cpld's monitoring
	 * function, so if the line from the touch chip into the
	 * cpld stays low, we'll get another interrupt as soon
	 * as the touch mask bit is cleared */

	/* change penirq to an input, and then arm */
	REG16(CPLD_CE_REG_INTMASK) |= CPLD_CE_INTMASK_PENIRQ;

	barrier();

	/* can't call lpd_clock_sleep_usec() 'cause ints are off (0x1000) */
	/* force an un-cached bus access for a more constant delay */
	for (v = 0; v < 0x200; v++) {
		tmp = REG16(CPLD_CE_REG_INTMASK);
		barrier();
	}

	REG16(CPLD_CE_REG_INTMASK) &= ~CPLD_CE_INTMASK_PENIRQ;

	spin_unlock_irqrestore(&lock, flags);

	pen_down = !(REG16(CPLD_CE_REG_INTMASK) & CPLD_CE_INTMASK_INT_TOUCH);

	return pen_down;
}

/**********************************************************************
* Macro: pen_down_irq_enable
**********************************************************************/
static int
pen_down_irq_enable(void)
{
	int lastState = !(REG16(CPLD_CE_REG_INTMASK) & CPLD_CE_INTMASK_MSK_TOUCH);
	spinlock_t lock;
	long flags;

	PRINTK2("pen_down_irq_enable\n");

	/* change penirq to an input, and then arm */
	
	spin_lock_irqsave(&lock, flags);
	REG16(CPLD_CE_REG_INTMASK) |= CPLD_CE_INTMASK_PENIRQ;
	spin_unlock_irqrestore(&lock, flags);

	/* wait for pullup time constant (25ms) */
	schedule_timeout(jiffies_from_ms(25));

	/* allow the pin to get an interrupt again */
	spin_lock_irqsave(&lock, flags);
	REG16(CPLD_CE_REG_INTMASK) &= ~CPLD_CE_INTMASK_PENIRQ;
	spin_unlock_irqrestore(&lock, flags);

	enable_irq(IRQ_TOUCH);
	return lastState;
}

/**********************************************************************
* Macro: lpdcpld_spi_ts_pen_down_irq_disable
**********************************************************************/
static int
pen_down_irq_disable(void)
{
	int lastState = !(REG16(CPLD_CE_REG_INTMASK) & CPLD_CE_INTMASK_MSK_TOUCH);

	PRINTK2("pen_down_irq_disable\n");
	disable_irq(IRQ_TOUCH);
	return lastState;
}

/**********************************************************************
* Function: lpdcpld_spi_irq_handler
*
* This interrupt handler only directs traffic for the interrupts
* by forwarding on the call to the appropriate interrupt handler.
**********************************************************************/
static void
lpdcpld_ts_irq_handler(int irq, void *_adsContext, struct pt_regs *regs)
{
	if (irq_wait_ptr) {
		if ((REG16(CPLD_CE_REG_INTMASK) & CPLD_CE_INTMASK_INT_TOUCH)) {
			printk("doh! spurious touch interrupt\n");
			return;
		}
		PRINTK2("*!lpdcpld_ts_irq_handler!*\n");
		disable_irq(IRQ_TOUCH);
		wake_up(irq_wait_ptr);
	} else {
		printk("lpdcpld_ts_irq_handler( NO ACTION )\n");
	}
}

/**********************************************************************
* Function: ads_784x_tsData_add
**********************************************************************/
static inline void ads_784x_tsData_add(
	adsContext_t *adsContext,
	unsigned int pressure,
	unsigned int x,
	unsigned int y)
{
	int next_head;

	PRINTK2("ENTER: ads_784x_tsData_add()\n");
	next_head = (adsContext->tsDataHead + 1) & (MAX_TS_DATA - 1);
	if (next_head != adsContext->tsDataTail) {
		adsContext->tsData[adsContext->tsDataHead].pressure = pressure;
		adsContext->tsData[adsContext->tsDataHead].x = x;
		adsContext->tsData[adsContext->tsDataHead].y = y;
#ifdef PROVIDE_TS_TIMESTAMP
		get_fast_time(&adsContext->tsData[adsContext->tsDataHead].stamp);
#endif
		adsContext->tsDataHead = next_head;
		if (adsContext->fasync)
			kill_fasync(&adsContext->fasync, SIGIO, POLL_IN);
		wake_up_interruptible(&adsContext->read_wait);
	}
	PRINTK2("LEAVE: ads_784x_tsData_add()\n");
	return;
}

/**********************************************************************
* Function: ads_784x_Read_tsData
**********************************************************************/
static int ads_784x_Read_tsData(adsContext_t *adsContext)
{
	void *sspContext = adsContext->sspContext;
	int x=0, x_1=0, x_2=0;
	int y=0, y_1=0, y_2=0;
	int pen_down = 0;
	int p = 0;
	int c = 0;
#ifdef CONFIG_ADS7846
	int p1=0, p1_1=0, p1_2=0;
	int p2=0, p2_1=0, p2_2=0;
#endif

	PRINTK2("ENTER: ads_784x_Read_tsData()\n");

	/*
	* Filtering is policy.  Policy belongs in user space.  We
	* therefore leave it to user space to do any filtering
	* they please.
	*
	* We do however, read twice and check against maximum x & y deltas
	* to reduce the amount of garbage returned.
	*
	* Note: There will be one reading with p=0 given when the pen is raised.
	*/

	do {
		adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
		/*
		* Read the pressure, X position, and Y position.
		* Interleave them for better performance
		*/
#ifdef CONFIG_ADS7846
		adsContext->write(sspContext, ADS_784x_READ_P1);
		adsContext->write(sspContext, ADS_784x_READ_P2);
		p1_1 = adsContext->read(sspContext);
		p2_1 = adsContext->read(sspContext);
#endif
		adsContext->write(sspContext, ADS_784x_READ_X);
		x_1 = 0x7f & adsContext->read(sspContext);
		x_1 <<= 8;
		x_1 |= adsContext->read(sspContext);
		x_1 >>= 3;

		adsContext->write(sspContext, ADS_784x_READ_Y);
		y_1 = 0x7f & adsContext->read(sspContext);
		y_1 <<= 8;
		y_1 |= adsContext->read(sspContext);
		y_1 >>= 3;

		adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);

		/*
		* ... AGAIN ...
		* Read the pressure, X position, and Y position.
		* Interleave them for better performance
		*/
		adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
#ifdef CONFIG_ADS7846
		adsContext->write(sspContext, ADS_784x_READ_P1);
		adsContext->write(sspContext, ADS_784x_READ_P2);
		p1_2 = adsContext->read(sspContext);
		p2_2 = adsContext->read(sspContext);
#endif
		adsContext->write(sspContext, ADS_784x_READ_X);
		x_2 = 0x7f & adsContext->read(sspContext);
		x_2 <<= 8;
		x_2 |= adsContext->read(sspContext);
		x_2 >>= 3;

		adsContext->write(sspContext, ADS_784x_READ_Y);
		y_2 = 0x7f & adsContext->read(sspContext);
		y_2 <<= 8;
		y_2 |= adsContext->read(sspContext);
		y_2 >>= 3;

		adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);
		c++;
	} while ((x_1 > (x_2+X_DELTA_MAX)) || (x_2 > (x_1+X_DELTA_MAX))
		||   (y_1 > (y_2+Y_DELTA_MAX)) || (y_2 > (y_1+Y_DELTA_MAX)));


	pen_down = is_pen_down(adsContext);

#ifdef CONFIG_ADS7846
	p1 = (p1_1 + p1_2) / 2;
	p2 = (p2_1 + p2_2) / 2;
	p = p2 - p1;
#else
	p = pen_down;
#endif
	x = (x_1 + x_2) / 2;
	y = (y_1 + y_2) / 2;

	ads_784x_tsData_add(adsContext, p, x, y);

	PRINTK("LEAVE: ads_784x_Read_tsData(p=0x%03x, x=0x%03x, y=0x%03x, c=%d)\n",
			p, x, y, c);
//	printk("LEAVE: ads_784x_Read_tsData(p=0x%03x, x=0x%03x, y=0x%03x, c=%d)\n",
//		   p, x, y, c);

	return pen_down;
}

/**********************************************************************
* Function: ads_784x_thread
*
* This is a RT kernel thread that handles the ADC accesses
* (mainly so we can use semaphores to serialise accesses to the ADC).
**********************************************************************/
static int ads_784x_thread(void *_adsContext)
{
	adsContext_t *adsContext = _adsContext;
	void *sspContext = adsContext->sspContext;
	struct task_struct *tsk = current;
	DECLARE_WAITQUEUE(wait, tsk);
	int pen_down = 0;

	PRINTK("ENTER: ads_784x_thread()\n");
	adsContext->rtask = tsk;

	daemonize();
	tsk->tty = NULL;
	strcpy(tsk->comm, "ktsd");

	/* only want to receive SIGKILL */
	spin_lock_irq(&tsk->sigmask_lock);
	siginitsetinv(&tsk->blocked, sigmask(SIGKILL));
	recalc_sigpending(tsk);
	spin_unlock_irq(&tsk->sigmask_lock);

	add_wait_queue(&adsContext->irq_wait, &wait);

	complete(&adsContext->complete);

	pen_down = is_pen_down(adsContext);

	for (;;) {
		signed long interval;

		/*
		 * Set to interrupt mode and wait a settling time.
		 */
		set_task_state(tsk, TASK_INTERRUPTIBLE);
		if (sspContext == NULL) {
			interval = HZ;	/* Check for change once per second */
		} else if (pen_down) {
			/* If pen is down then periodically read pen position */
			pen_down_irq_disable();
			interval = HZ/SAMPLES_PER_SECOND;
		} else {
			/* If pen is not down then sleep until pen down interrupt */
			pen_down_irq_enable();
			interval = MAX_SCHEDULE_TIMEOUT;
		}
		schedule_timeout(interval);
		if (signal_pending(tsk)) {
			break;
		}
		if (pen_down == 0) {
			/*
			 * On pen down there is some bounce.
			 * Wait a debounce period and read it again.
			 */
			schedule_timeout(HZ/DEBOUNCE_FRACTION_OF_A_SECOND);
			if (signal_pending(tsk)) {
				break;
			}
			/*
			 * If the pen is not down after the debounce period,
			 * ignore the pen down signal.
			 */
			pen_down = is_pen_down(adsContext);
			if (pen_down == 0)
				continue;
		}
		pen_down_irq_disable();

		/*
		 * We got an IRQ, which works us up.  Process the touchscreen.
		 */
		pen_down = ads_784x_Read_tsData(adsContext);
	}

	remove_wait_queue(&adsContext->irq_wait, &wait);
	adsContext->rtask = NULL;
	PRINTK("LEAVE: ads_784x_thread()\n");

	return(0);
}

/**********************************************************************
* Function: ads_784x_read
*
* ***********************************
* *** User space driver interface ***
* ***********************************
**********************************************************************/
static ssize_t ads_784x_read(struct file *filp, char *buffer, size_t _count, loff_t *ppos)
{
	DECLARE_WAITQUEUE(wait, current);
	adsContext_t *adsContext = filp->private_data;
	char *ptr = buffer;
	int err = 0;
	int count = (int)_count;	/* Force a sigened value to be used */

	PRINTK2("ENTER: ads_784x_read()\n");
	add_wait_queue(&adsContext->read_wait, &wait);
	while (count >= sizeof(tsData_t)) {
		err = -ERESTARTSYS;
		if (signal_pending(current))
			break;

		if (ads_784x_tsData_pending(adsContext)) {
			tsData_t *tsData = ads_784x_tsData_get(adsContext);

			err = copy_to_user(ptr, tsData, sizeof(tsData_t));
			ads_784x_tsData_pull(adsContext);

			if (err)
				break;

			ptr += sizeof(tsData_t);
			count -= sizeof(tsData_t);
			continue;
		}

		set_current_state(TASK_INTERRUPTIBLE);
		err = -EAGAIN;
		if (filp->f_flags & O_NONBLOCK)
			break;
		schedule();
	}
 	current->state = TASK_RUNNING;
	remove_wait_queue(&adsContext->read_wait, &wait);
	PRINTK2("LEAVE: ads_784x_read()\n");

	return(ptr == buffer ? err : ptr - buffer);
}

/**********************************************************************
* Function: ads_784x_poll
*
* ***********************************
* *** User space driver interface ***
* ***********************************
**********************************************************************/
static unsigned int ads_784x_poll(struct file *filp, poll_table *wait)
{
	adsContext_t *adsContext = filp->private_data;
	int ret = 0;

	PRINTK2("ENTER: ads_784x_poll(%04x)\n", REG16(CPLD_CE_REG_INTMASK));
	poll_wait(filp, &adsContext->read_wait, wait);
	if (ads_784x_tsData_pending(adsContext))
		ret = POLLIN | POLLRDNORM;

	return(ret);
}

/**********************************************************************
* Function: ads_784x_open
*
* ***********************************
* *** User space driver interface ***
* ***********************************
**********************************************************************/
static int ads_784x_open(struct inode *inode, struct file *filp)
{
	adsContext_t *adsContext = &adsContext_l;
	int ret = 0;

	PRINTK2("ENTER: ads_784x_open()\n");

	filp->private_data = adsContext;

	/* Flush the ts data queue here */
	ads_784x_tsData_flush(adsContext);

	PRINTK2("LEAVE: ads_784x_open()\n");
	return(ret);
}

/**********************************************************************
* Function: ads_784x_fasync
*
* ***********************************
* *** User space driver interface ***
* ***********************************
**********************************************************************/
static int ads_784x_fasync(int fd, struct file *filp, int on)
{
	int sts;
	adsContext_t *adsContext = filp->private_data;

	PRINTK("ENTER: ads_784x_fasync()\n");
	sts = fasync_helper(fd, filp, on, &adsContext->fasync);
	PRINTK("LEAVE: ads_784x_fasync()\n");

	return(sts);
}

/**********************************************************************
* Function: ads_784x_release
*
* ***********************************
* *** User space driver interface ***
* ***********************************
*
* Release touchscreen resources.  Disable IRQs.
**********************************************************************/
static int ads_784x_release(struct inode *inode, struct file *filp)
{
	// adsContext_t *adsContext = filp->private_data;

	PRINTK("ENTER: ads_784x_release()\n");
	lock_kernel();
	ads_784x_fasync(-1, filp, 0);
	unlock_kernel();
	PRINTK("LEAVE: ads_784x_release()\n");

	return(0);
}

/**********************************************************************
* Define (fill in) the user space file operations for this driver
* and initialize the ADS touchscreen driver as a "miscdevice":
* 		Character device
* 		Major(10) --- Non-serial mice, misc features
* 		Minor(20) --- /dev/touchscreen/ads_784x
**********************************************************************/
static struct file_operations ads_784x_fops = {
	owner:		THIS_MODULE,
	read:		ads_784x_read,
	poll:		ads_784x_poll,
	open:		ads_784x_open,
	fasync:		ads_784x_fasync,
	release:	ads_784x_release,
};

static struct miscdevice ads_784x_dev = {
	minor: 20,
	name: "touchscreen/ads_784x",
	fops: &ads_784x_fops,
};

#ifdef PROC_BATTERY
/**********************************************************************
* ads_784x_proc_battery_read
**********************************************************************/
static int ads_784x_proc_battery_read(char *buf, char **start, off_t offset,
		int len, int *eof, void *unused)
{
	adsContext_t *adsContext = &adsContext_l;
	ads784x_sspContext_t *ads784x_sspContext = adsContext->ads784x_sspContext;
	int milliVolts;
	int volts;
	int hundredthsVolts;
	int sample;
	int size;
	int count;
	int last_irq_state;

	PRINTK("ENTER: ads_784x_proc_battery_read(len=%d)\n", len);
	/*
	* Reading the ADS784X takes the part out of pen down interrupt mode
	* causing spurious pen down interrupts. So we must disable 
	* pen down interrupts while reading battery voltage.
	*/
	last_irq_state = pen_down_irq_disable();

	/* Do a dummy read to turn on the internal reference voltage */
	adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
	adsContext->write(ads784x_sspContext, ADS_784x_READ_BATTERY);
	sample = adsContext->read(ads784x_sspContext);
	adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);
	udelay(100);	/* wait until the reference voltage settle */

	sample = 0;
	for (count = 0; count < 3; count++) {
		adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
		adsContext->write(ads784x_sspContext, ADS_784x_READ_BATTERY);
		sample += adsContext->read(ads784x_sspContext);
		adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);
	}       
	sample /= count;
	milliVolts = (sample * 1000 * 10) / 4096;
	volts = milliVolts / 1000;
	hundredthsVolts = (milliVolts - (volts * 1000)) / 10;
	size = sprintf(buf, "battery: %i.%02iV\n", volts, hundredthsVolts);

	/* Do a dummy read to turn off the internal reference voltage */
	adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
	adsContext->write(ads784x_sspContext, ADS_784x_READ_X);
	sample = adsContext->read(ads784x_sspContext);
	adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);

	/* Restore the interrupt enable state */
	if (last_irq_state) {
		udelay(100);	/* Wait until the pen down interrupt settles */
		pen_down_irq_enable();
	}
	PRINTK("LEAVE: ads_784x_proc_battery_read(len=%d)\n", len);

	return(size);
}
#endif

#ifdef PROC_TEMPERATURE
/**********************************************************************
* ads_784x_proc_temperature_read
**********************************************************************/
static int ads_784x_proc_temperature_read(char *buf, char **start,
		off_t offset, int len, int *eof, void *unused)
{
	adsContext_t *adsContext = &adsContext_l;
	ads784x_sspContext_t *ads784x_sspContext = adsContext->ads784x_sspContext;
	int size;
	int count;
	int C10, F10;
	int sample1;
	int sample91;
	int last_irq_state;

	PRINTK("ENTER: ads_784x_proc_temperature_read(len=%d)\n", len);
	/*
	* Reading the ADS784X takes the part out of pen down interrupt mode
	* causing spurious pen down interrupts. So we must disable 
	* pen down interrupts while reading temperature.
	*/
	last_irq_state = pen_down_irq_disable();

	/* do a dummy read to turn on the internal reference voltage */
	adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
	adsContext->write(ads784x_sspContext, ADS_784x_READ_TEMP0);
	sample1 = adsContext->read(ads784x_sspContext);
	adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);
	udelay(100);	/* wait until the reference voltage settle */

	sample1  = 0;
	sample91 = 0;
	/* read the temperature values */
	for (count = 0; count < 3; count++) {
		adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
		adsContext->write(ads784x_sspContext, ADS_784x_READ_TEMP0);
		adsContext->write(ads784x_sspContext, ADS_784x_READ_TEMP1);
		sample1  += adsContext->read(ads784x_sspContext);
		sample91 += adsContext->read(ads784x_sspContext);
		adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);
	}
	/* average values */
	sample1 /= count;
	sample91 /= count;

	C10 = (((sample91 - sample1) * (25 * 2573)) / 4095) - 2730;
	F10 = (C10*9)/5 + 320;

	size = sprintf(buf, "Temperature: %iC, %iF\n", (C10+5)/10, (F10+5)/10);

	/* do a dummy read to turn off the internal reference voltage */
	adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
	adsContext->write(ads784x_sspContext, ADS_784x_READ_X);
	sample1 = adsContext->read(ads784x_sspContext);
	adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);

	/* restore the interrupt enable state */
	if (last_irq_state) {
		udelay(100);	/* wait until the pen down interrupt settles */
		pen_down_irq_enable();
	}
	PRINTK("LEAVE: ads_784x_proc_temperature_read(len=%d)\n", len);

	return(size);
}
#endif

/**********************************************************************
* Function: ads_784x_make_ssp_association
*
* Purpose:
*   Make the association between the eeprom driver and the ssp driver
**********************************************************************/
static int ads_784x__make_ssp_association(adsContext_t *adsContext)
{
    int sts = 0;
    void *vp;

/* NOTE: -EOPNOTSUPP == Operation not supported on transport endpoint */
#define ASSOCIATION_ERROR	-EOPNOTSUPP

	PRINTK("ENTER: ads_784x_make_ssp_association()\n");

	vp = ssp_request_pointer(SSP_DEV_TOUCHSCREEN, "sspContext");
	if ( ! vp )
		sts = ASSOCIATION_ERROR;
	adsContext->sspContext = vp;

	vp = ssp_request_pointer(SSP_DEV_TOUCHSCREEN, "write");
	if ( ! vp )
		sts = ASSOCIATION_ERROR;
	adsContext->write = vp;

	vp = ssp_request_pointer(SSP_DEV_TOUCHSCREEN, "read");
	if ( ! vp )
		sts = ASSOCIATION_ERROR;
	adsContext->read = vp;

	vp = ssp_request_pointer(SSP_DEV_TOUCHSCREEN, "lock");
	if ( ! vp )
		sts = ASSOCIATION_ERROR;
	adsContext->lock = vp;

	vp = ssp_request_pointer(SSP_DEV_TOUCHSCREEN, "unlock");
	if ( ! vp )
		sts = ASSOCIATION_ERROR;
	adsContext->unlock = vp;

	/* Note: The following need reset to NULL when we are finished */

	irq_wait_ptr = &adsContext->irq_wait;

	PRINTK("LEAVE: ads_784x_make_ssp_association(%d)\n", sts);

	return(sts);
}

/**********************************************************************
* Function: ads_784x_init
*
* Purpose:
*	Register & Initialize the module
**********************************************************************/
static int __init ads_784x_init(void)
{
	adsContext_t *adsContext = &adsContext_l;
	int sts = -ENODEV;
	int result;
	void *sspContext;

	PRINTK("ENTER: ads_784x_init()\n");

	init_waitqueue_head(&adsContext->read_wait);
	init_waitqueue_head(&adsContext->irq_wait);

	irq_wait_ptr = NULL;

	/* Retrieve the service information from the SSP driver */
	sts = ads_784x__make_ssp_association(adsContext);
	if (sts != 0) {
		return sts;
	}
	sspContext = adsContext->sspContext;

	/*
	 * Request the touch irq and enable it.
	 */
	result = request_irq(IRQ_TOUCH, lpdcpld_ts_irq_handler,
						 SA_SAMPLE_RANDOM, DRVNAME, adsContext);
	if (result < 0) {
		printk("%s: cannot get requested IRQ(2)\n", DRVNAME);
		return result;
	}

	pen_down_irq_enable();

	/* Start the ADS polling thread */
	lock_kernel();
	if (adsContext->rtask == NULL) {
		init_completion(&adsContext->complete);

		sts = kernel_thread(ads_784x_thread, adsContext,
							CLONE_FS | CLONE_FILES);
		if (sts >= 0) {
			sts = 0;
			/* Only do this if the tread started correctly */
			wait_for_completion(&adsContext->complete);
		}
	}
	unlock_kernel();

	adsContext->lock(sspContext, SSP_DEV_TOUCHSCREEN);
//	adsContext->write(sspContext, ADS_784x_INIT);
	adsContext->write(sspContext, ADS_784x_READ_X);
	adsContext->read(sspContext);	/* dummy read */
	adsContext->read(sspContext);	/* dummy read */
	adsContext->unlock(sspContext, SSP_DEV_TOUCHSCREEN);

	sts = misc_register(&ads_784x_dev);

#ifdef PROC_BATTERY
	ads_784x_proc_battery = create_proc_entry("battery", 0, 0);
	if (ads_784x_proc_battery) {
		ads_784x_proc_battery->read_proc = ads_784x_proc_battery_read;
	} else {
		printk(KERN_ERR "%s: unable to register /proc/battery\n", DRVNAME);
	}
#endif
#ifdef PROC_TEMPERATURE
	ads_784x_proc_temperature = create_proc_entry("temperature", 0, 0);
	if (ads_784x_proc_temperature) {
		ads_784x_proc_temperature->read_proc = ads_784x_proc_temperature_read;
	} else {
		printk(KERN_ERR "%s: unable to register /proc/temperature\n", DRVNAME);
	}
#endif

	PRINTK("LEAVE: ads_784x_init()\n");
	return(sts);
}

/**********************************************************************
* Function: ads_784x_exit
*
* Purpose:
*	Un-Register & Cleanup the module
**********************************************************************/
static void ads_784x_exit(void)
{
	adsContext_t *adsContext = &adsContext_l;

	PRINTK("ENTER: ads_784x_exit()\n");

	if (adsContext->rtask) {
		send_sig(SIGKILL, adsContext->rtask, 1);
		schedule();
	}

	PRINTK("ads_784x_exit(): misc_deregister()\n");
	misc_deregister(&ads_784x_dev);

	/*
	 * Disable & Return the irq
	 */
	lock_kernel();

	pen_down_irq_disable();
	free_irq(IRQ_TOUCH, adsContext);

	unlock_kernel();

	/* Clear the pointer the interrupt handler might use */
	irq_wait_ptr = NULL;

#ifdef PROC_BATTERY
	remove_proc_entry("battery", NULL);
#endif
#ifdef PROC_TEMPERATURE
	remove_proc_entry("temperature", NULL);
#endif

	PRINTK("LEAVE: ads_784x_exit()\n");
}

module_init(ads_784x_init);
module_exit(ads_784x_exit);

MODULE_AUTHOR("Jim Gleason");
MODULE_DESCRIPTION("ADS 784x Driver for Sharp LH7x EVB");
MODULE_LICENSE("Copyright (c) 2002 Lineo, Inc.");

