/*------------------------------------------------------------------------
 . smc91111_cfg.h - local configurations for the LAN91C111 Ethernet Driver
 .
 . Copyright (C) 2002 Accelent Systems, Inc.
 .
 . This program is free software; you can redistribute it and/or modify
 . it under the terms of the GNU General Public License as published by
 . the Free Software Foundation; either version 2 of the License, or
 . (at your option) any later version.
 .
 . This program is distributed in the hope that it will be useful,
 . but WITHOUT ANY WARRANTY; without even the implied warranty of
 . MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 . GNU General Public License for more details.
 .
 . You should have received a copy of the GNU General Public License
 . along with this program; if not, write to the Free Software
 . Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 .
 . Author: Jeff Sutherland <jeffs@accelent.com>
 .
 . Purpose: This file contains machine/architecture specific configuration
 .	    twiddles, hacks, and kludges for the smc91111 driver so that all
 .	    that crud can be removed from the driver in an effort to make it
 .	    as generic (and clean) as possible.
 .
 .
 ---------------------------------------------------------------------------*/

#ifndef _SMC91111_CFG_H_
#define _SMC91111_CFG_H_

/* Support for the Accelent PXA IDP and friends:
 * Architecture: ARM
 * Processor:	 Intel PXA
 * Machine:	 Accelent PXA IDP
 */

#ifdef CONFIG_ARCH_PXA

/* Local i/o macros for the PXA IDP (32 bit bus i/f)
*  Hack alert: 8 bit reads are broken on the PXA.  Don't use u8 types for
*  reading any fifo registers.  u8 reads will succeed on registers that can
*  tolerate multiple reads without changing their contents.
*/

/* Platform specific defines: */

#ifdef CONFIG_ARCH_PXA_IDP
#define NETWORK_INTERFACE0_PHYS_DATA_ADDR (IDP_ETH_PHYS + 0x300 + DATA_REG)
#define NETWORK_INTERFACE0_INTERRUPT			ETHERNET_IRQ
#define NETWORK_INTERFACE0_USE_32_BIT			1
#define NETWORK_INTERFACE0_REGISTER_BASE_ADDRESS	(IDP_ETH_BASE+ 0x300)

#define NETWORK_INTERFACE1_PHYS_DATA_ADDR (IDP_ETH_PHYS + 0x200 + DATA_REG)
#define NETWORK_INTERFACE1_INTERRUPT			0
#define NETWORK_INTERFACE1_USE_32_BIT			0
#define NETWORK_INTERFACE1_REGISTER_BASE_ADDRESS	0

/* The idp doesn't do anything fancy with the contents of the eeprom.
 * The 'interface' param allows specification of more than one mac addr in
 * a given eeprom part, or as an index into an array of addresses somewhere
 * else in memory I suppose... 
 */
static inline u16
SMC_inw(u_int base, u_int reg);

static inline void
SMC_outw(u16 val, u_int base, u_int reg);

static inline void smc_init_mac_addr( struct net_device *dev, unsigned int ioaddr, int interface)
{
	int i;
	
	SMC_outw(1, ioaddr, BANK_SELECT);
	for (i = 0; i < 6; i += 2) {
		u16 address;

		address = SMC_inw(ioaddr, ADDR0_REG + i);
		dev->dev_addr[i + 1] = address >> 8;
		dev->dev_addr[i] = address & 0xFF;
	}
}

static char fallback_mac[MAX_ADDR_LEN] = { 0, 0, 0, 0, 0, 0 };

/*---------------------------------------------------------------------*/

#elif defined CONFIG_ARCH_PXA_CERF

#define NETWORK_INTERFACE0_PHYS_DATA_ADDR		(CERF_ETH_PHYS + 0x300 + DATA_REG)
#define NETWORK_INTERFACE0_INTERRUPT			CERF_ETH_IRQ
#define NETWORK_INTERFACE0_USE_32_BIT			1
#define NETWORK_INTERFACE0_REGISTER_BASE_ADDRESS	(CERF_ETH_BASE + 0x300)

#define NETWORK_INTERFACE1_PHYS_DATA_ADDR		(CERF_ETH2_PHYS + 0x300 + DATA_REG)
#define NETWORK_INTERFACE1_INTERRUPT			CERF_ETH2_IRQ
#define NETWORK_INTERFACE1_USE_32_BIT			0
#define NETWORK_INTERFACE1_REGISTER_BASE_ADDRESS	(CERF_ETH2_BASE + 0x300)

#include <linux/config_eeprom.h>

static inline void smc_init_mac_addr( struct net_device *dev, unsigned int ioaddr, int interface)
{
	char tagname[20];
	u8 data[10];
	u16 datalen=10;

	sprintf( tagname, "MACAD%c", '0'+interface);
	if( config_eeprom_get_data( tagname, strlen(tagname), data, &datalen))
	{
		if( datalen == 6)
		{
			int i;
			for( i=0; i<6; i++) 
			{
				dev->dev_addr[i] = data[i];
			}
		}
		else
		{
			printk("%s: Bad config eeprom tag data\n", dev->name);
		}
	}
#if 0
	else	
	{
		printk("%s: config eeprom tag [%s] not found\n", dev->name, tagname);
	}
#endif
}

static char fallback_mac[MAX_ADDR_LEN] = { 0xde, 0xad, 0xbe, 0xef, 0x00, 0x00 };

#endif /* CONFIG_ARCH_PXA_CERF */

#elif defined(CONFIG_LPD_79520_10)

/* Platform specific defines: */

#define NETWORK_INTERFACE0_PHYS_DATA_ADDR         (0)	/* no dma */
#define NETWORK_INTERFACE0_INTERRUPT              IRQ_ETHERNET
#define NETWORK_INTERFACE0_USE_32_BIT             (0)
#define NETWORK_INTERFACE0_REGISTER_BASE_ADDRESS  (SMC91c111_BASE)

#define NETWORK_INTERFACE1_PHYS_DATA_ADDR         (DATA_REG)
#define NETWORK_INTERFACE1_INTERRUPT              (0)
#define NETWORK_INTERFACE1_USE_32_BIT             (0)
#define NETWORK_INTERFACE1_REGISTER_BASE_ADDRESS  (0)

/* The idp doesn't do anything fancy with the contents of the eeprom.
 * The 'interface' param allows specification of more than one mac addr in
 * a given eeprom part, or as an index into an array of addresses somewhere
 * else in memory I suppose... 
 */
static inline u16
SMC_inw(u_int base, u_int reg);

static inline void
SMC_outw(u16 val, u_int base, u_int reg);

static inline void smc_init_mac_addr( struct net_device *dev, unsigned int ioaddr, int interface)
{
	int i;
	
	SMC_outw(1, ioaddr, BANK_SELECT);
	for (i = 0; i < 6; i += 2) {
		u16 address;

		address = SMC_inw(ioaddr, ADDR0_REG + i);
		dev->dev_addr[i + 1] = address >> 8;
		dev->dev_addr[i] = address & 0xFF;
	}
}

static char fallback_mac[MAX_ADDR_LEN] = { 0x00, 0x08, 0xee, 0x00, 0x00, 0x77 };

#endif  /* CONFIG_LPD_79520_10 */

#define NO_AUTOPROBE 1
//#undef NO_AUTOPROBE

/* Because of bank switching, the LAN91xxx uses only 16 I/O ports (on an
 * isa based system, that is.) */

#ifdef SMC_IO_EXTENT
#undef SMC_IO_EXTENT
#endif

#define SMC_IO_EXTENT	(16 << 2)

// Use power-down feature of the chip
#define SMC_POWER_DOWN	1

#define  MAX_NETWORK_INTERFACE_COUNT 2
static struct
{
	unsigned long port;
	unsigned long dma_addr;
	int use_32bit;
	int irq;
} smc_config[MAX_NETWORK_INTERFACE_COUNT] = {
	{
		port:		NETWORK_INTERFACE0_REGISTER_BASE_ADDRESS,
		dma_addr:	NETWORK_INTERFACE0_PHYS_DATA_ADDR,
		use_32bit:	NETWORK_INTERFACE0_USE_32_BIT,
		irq:		NETWORK_INTERFACE0_INTERRUPT,
	},
	{
		port:		NETWORK_INTERFACE1_REGISTER_BASE_ADDRESS,
		dma_addr:	NETWORK_INTERFACE1_PHYS_DATA_ADDR,
		use_32bit:	NETWORK_INTERFACE1_USE_32_BIT,
		irq:		NETWORK_INTERFACE1_INTERRUPT,
	}
};

/* no support for pcmcia card drivers just yet...*/
/*
#if NETWORK_DRIVER_PCMCIA_MODE == TRUE
static unsigned long smc_attmem_list[] = {
	NETWORK_INTERFACE0_ATTRIBUTE_BASE_ADDRESS,
	
#if NETWORK_INTERFACE_COUNT > 1
	NETWORK_INTERFACE1_ATTRIBUTE_BASE_ADDRESS,
#endif
	0
};
#endif
*/

/*
 * Wait time for TX packet memory to be free.  This probably shouldn't be
 * tuned that much, as waiting for this means nothing else happens
 * in the system. Originally this was set to 16 but I'm not sure you want to
 * sit and spin here waiting to tx packets.  Let interrupts do their job.
 */

#define MEMORY_WAIT_TIME 4

/* If you want to use autoprobing for your hardware, you may make use of
 * this table. The LAN91C111 can be at any of the following port addresses.
 * To change for a slightly different card, you can add it to the array.
 * Keep in mind that the array must end in zero.
 */

#if 0
static unsigned int smc_portlist[] __initdata =
    { 0x200, 0x220, 0x240, 0x260, 0x280, 0x2A0, 0x2C0, 0x2E0,
	0x300, 0x320, 0x340, 0x360, 0x380, 0x3A0, 0x3C0, 0x3E0, 0
};
#endif

#endif /*  _SMC91111_CFG_H_ */
