/*
 * Glue audio driver for the Sharp LH79x board & Philips UDA1341 codec.
 *
 * Copyright (c) 2000 Nicolas Pitre <nico@cam.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 *
 * This is the machine specific part of the Assabet/UDA1341 support.
 * This driver makes use of the UDA1341 and the sa1100-audio modules.
 *
 * History:
 *
 * 2000-05-21	Nicolas Pitre	Initial release.
 *
 * 2001-06-03	Nicolas Pitre	Made this file a separate module, based on
 * 				the former sa1100-uda1341.c driver.
 *
 * 2001-07-17	Nicolas Pitre	Supports 44100Hz and 22050Hz samplerate now.
 *
 * 2001-08-03	Russell King	Fix left/right channel swap.
 *				Attempt to reduce power consumption when idle.
 *
 * 2001-09-23	Russell King	Remove old L3 bus driver.
 *
 * Please note that fiddling too much with MDREFR results in oopses, so we don't
 * touch MDREFR unnecessarily, which means we don't touch it on close.
 *
 * 2002-05-20	Jim Gleason	Borrowed from assabet-uda1341.c and modified
 *                              to fit Sharp LH79520 board and Philips UDA1341
 *                              codec.  Note this was done using the code from
 *                              Sharps LH79520_evb_audio_driver.c file with
 *                              permission from Sharp (see comment below).
 */

/**********************************************************************
 * Workfile:   LH79520_evb_audio_driver.c
 * Revision:   1.3
 * Author:   SuryanG
 * Date:   Feb 01 2002 14:44:16
 *
 * Project: LH79520 EVB
 *
 * Description:
 *  This file contains a DMA driver for the audio on the Sharp LH79520 EVB
 *
 * Notes:
 *  This driver assumes EVB CPLD code version 1.4
 *  See the LH79520 EVB CPLD documentation about using DREQ1 and DEOT1
 *
 * Revision History:
 *
 *    Rev 1.3   Feb 01 2002 14:44:16   SuryanG
 * Added driver support functions for interrupt driven audio.
 *
 *    Rev 1.2   Jan 29 2002 19:42:40   SuryanG
 * Updated file names in the include files list
 *
 *    Rev 1.1   Jan 23 2002 11:29:56   SuryanG
 * Removed CPLD defines because they were duplicated from
 * LH79520_evb.h.
 *
 *    Rev 1.0   Dec 28 2001 13:31:00   KovitzP
 * Initial revision.
 *
 * COPYRIGHT (C) 2001 SHARP MICROELECTRONICS OF THE AMERICAS, INC.
 *              CAMAS, WA
 *********************************************************************/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/errno.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/cpufreq.h>
#include <linux/l3/l3.h>
#include <linux/l3/uda1341.h>

#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/dma.h>

#include <asm/arch/dma.h>
#include <asm/arch/cpld.h>

#include "lh7x-audio.h"

/*
 * Debugging?
 */
#undef DEBUG


#ifdef DEBUG
#define DPRINTK( x... )  printk( ##x )
#else
#define DPRINTK( x... )
#endif


#define AUDIO_RATE_DEFAULT	44100

/*
 * Mixer (UDA1341) interface
 */

static struct l3_client uda1341;

static int
mixer_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
	/*
	 * We only accept mixer (type 'M') ioctls.
	 */
	if (_IOC_TYPE(cmd) != 'M')
		return -EINVAL;

	return l3_command(&uda1341, cmd, (void *)arg);
}

static struct file_operations lh79520_mixer_fops = {
	ioctl:		mixer_ioctl,
	owner:		THIS_MODULE
};

static int fs_list[] = 
{
   8000,
   11025,
   12000,
   16000,
   22050,
   24000,
   32000,
   44100,
   48000
};

#define N_FS (sizeof(fs_list) / sizeof( int))

/*
 * Audio interface
 */
static long audio_samplerate = AUDIO_RATE_DEFAULT;

static void lh79520_set_samplerate(long val)
{
    int i;
    cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;
    struct uda1341_cfg cfg;

    DPRINTK( __FUNCTION__ "(%d)\n", (int)val);

    for( i=0; i < N_FS; i++)
	if( fs_list[i] == val)
	    break;

    if( i < N_FS) {
	cpld->audio_control &= ~CPLD_FS_BITS;
	cpld->audio_control |= _SBF(CPLD_FS_BIT_FIELD, i);
	audio_samplerate = fs_list[i];
	DPRINTK( __FUNCTION__ " set fs to %d (index=%d)\n", fs_list[i], i);
    }

    cfg.fs = 256;
    cfg.format = FMT_I2S;
    l3_command(&uda1341, L3_UDA1341_CONFIGURE, &cfg);
}

/*
 * Initialise the Assabet audio driver.
 *
 * Note that we have to be careful with the order that we do things here;
 * there is a D-type flip flop which is clocked from the SFRM line which
 * indicates whether the same is for the left or right channel to the
 * UDA1341.
 *
 * When you disable the SSP (by clearing SSCR0_SSE) it appears that the
 * SFRM signal can float high.  When you re-enable the SSP, you clock the
 * flip flop once, and end up swapping the left and right channels.
 *
 * The ASSABET_BCR_CODEC_RST line will force this flip flop into a known
 * state, but this line resets other devices as well!
 *
 * In addition to the above, it appears that powering down the UDA1341 on
 * early Assabets leaves the UDA_WS actively driving a logic '1' into the
 * chip, wasting power!  (you can tell this by D11 being half-on).  We
 * attempt to correct this by kicking the flip flop on init/open/close.
 * We should probably do this on PM resume as well.
 *
 * (Note the ordering of ASSABET_BCR_AUDIO_ON, SFRM and ASSABET_BCR_CODEC_RST
 * is important).
 */
static void lh79520_audio_init(void *dummy)
{
    unsigned long flags;
    cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

    DPRINTK( __FUNCTION__ "\n");

    local_irq_save(flags);
    /* setup h/w */
    cpld->audio_control &= ~(CPLD_ALL_DAC_BITS | CPLD_ALL_ADC_BITS);	/* clear all audio bits */

    local_irq_restore(flags);

    /* Wait for the UDA1341 to wake up */
    mdelay(1);

    l3_open(&uda1341);

    lh79520_set_samplerate(audio_samplerate);
}

/*
 * Shutdown the Assabet audio driver.
 *
 * We have to be careful about the SFRM line here for the same reasons
 * described in the initialisation comments above.  This basically means
 * that we must hand the SSP pins back to the GPIO module before disabling
 * the SSP.
 *
 * In addition, to reduce power drain, we toggle the SFRM line once so
 * that the UDA_WS line is at logic 0.
 *
 * We can't clear ASSABET_BCR_CODEC_RST without knowing if the UCB1300 or
 * ADV7171 driver is still active.  If it is, then we still need to play
 * games, so we might as well leave ASSABET_BCR_CODEC_RST set.
 */
static void lh79520_audio_shutdown(void *dummy)
{
    cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;
    DPRINTK( __FUNCTION__ "  audio_control=0x%x\n", cpld->audio_control);

    cpld->audio_control &= ~(CPLD_ALL_DAC_BITS | CPLD_ALL_ADC_BITS);	/* clear all audio bits */
    lh79520_set_samplerate(AUDIO_RATE_DEFAULT);
    DPRINTK( __FUNCTION__ "  audio_control=0x%x\n", cpld->audio_control);

    l3_close(&uda1341);
}

static int lh79520_audio_ioctl( struct inode *inode, struct file *file,
				uint cmd, ulong arg)
{
    long val;
    int ret = 0;

    /*
     * These are platform dependent ioctls which are not handled by the
     * generic sa1100-audio module.
     */
    switch (cmd) {
    case SNDCTL_DSP_STEREO:
	ret = get_user(val, (int *) arg);
	if (ret)
		return ret;
	/* the UDA1341 is stereo only */
	ret = (val == 0) ? -EINVAL : 1;
	return put_user(ret, (int *) arg);

    case SNDCTL_DSP_CHANNELS:
    case SOUND_PCM_READ_CHANNELS:
	/* the UDA1341 is stereo only */
	return put_user(2, (long *) arg);

    case SNDCTL_DSP_SPEED:
	ret = get_user(val, (long *) arg);
	if (ret) break;
	lh79520_set_samplerate(val);
	/* fall through */

    case SOUND_PCM_READ_RATE:
	return put_user(audio_samplerate, (long *) arg);

    case SNDCTL_DSP_SETFMT:
    case SNDCTL_DSP_GETFMTS:
	/* we can do signed 16-bit only */
	return put_user(AFMT_S16_LE, (long *) arg);

    default:
	/* Maybe this is meant for the mixer (As per OSS Docs) */
	return mixer_ioctl(inode, file, cmd, arg);
    }

    return ret;
}

static audio_stream_t output_stream, input_stream;

static audio_state_t audio_state = {
	output_stream:	&output_stream,
	output_dma:	DMA_Audio_Out,
	output_id:	"LH79520 UDA1341 out",
	input_stream:	&input_stream,
	input_dma:	DMA_Audio_In,
	input_id:	"LH79520 UDA1341 in",
	need_tx_for_rx:	0,
	hw_init:	lh79520_audio_init,
	hw_shutdown:	lh79520_audio_shutdown,
	client_ioctl:	lh79520_audio_ioctl,
	sem:		__MUTEX_INITIALIZER(audio_state.sem),
};

static int lh79520_audio_open(struct inode *inode, struct file *file)
{
    return lh7x_audio_attach(inode, file, &audio_state);
}

/*
 * Missing fields of this structure will be patched with the call
 * to sa1100_audio_attach().
 */
static struct file_operations lh79520_audio_fops = {
	open:		lh79520_audio_open,
	owner:		THIS_MODULE
};


static int audio_dev_id, mixer_dev_id;

static int __init lh79520_uda1341_init(void)
{
    int ret;

    DPRINTK( __FUNCTION__ "\n");

    ret = l3_attach_client(&uda1341, "l3-lh7x", "uda1341");
    if (ret)
	return ret;

    /* register devices */
    audio_dev_id = register_sound_dsp(&lh79520_audio_fops, -1);
    mixer_dev_id = register_sound_mixer(&lh79520_mixer_fops, -1);

    printk(KERN_INFO "LH79520 UDA1341 audio driver initialized\n");

    return 0;
}

static void __exit lh79520_uda1341_exit(void)
{
    DPRINTK( __FUNCTION__ "\n");

    unregister_sound_dsp(audio_dev_id);
    unregister_sound_mixer(mixer_dev_id);
    l3_detach_client(&uda1341);
}

module_init(lh79520_uda1341_init);
module_exit(lh79520_uda1341_exit);

MODULE_AUTHOR("Nicolas Pitre");
MODULE_DESCRIPTION("Glue audio driver for the Sharp LH7X EVB & Philips UDA1341 codec.");

EXPORT_NO_SYMBOLS;
