/* vi: set sw=4 ts=4 ai: */

/**********************************************************************
*  linux/include/asm-arm/arch-lh79520/ssp_lh7x.h
*
*  Provide SSP (Synchronous Serial Port) types & definitions
*  for LH7x EVB boards
*
*  Copyright (C) 2002  Lineo, Inc.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License (GPL) version 2
*  as published by the Free Software Foundation.
*
**********************************************************************/

#ifndef _SSP_LH7X_h
#define _SSP_LH7X_h

#define SSP_BASE	SSP_PHYS

/*********************************************************************
* Synchronous Serial Port Registers
*********************************************************************/ 
#define SSPCR0		(SSP_BASE+0x00)	/* Control reg. 0 */
#define SSPCR1		(SSP_BASE+0x04)	/* Control reg. 1 */
#define SSPDR		(SSP_BASE+0x08)	/* Receive FIFO (Read)*/
									/* Transmit FIFO data reg. (Write)*/
#define SSPSR		(SSP_BASE+0x0C)	/* Status Reg. */
#define SSPCPSR		(SSP_BASE+0x10)	/* Clock prescale reg. */
#define SSPIIR		(SSP_BASE+0x14)	/* Interrupt identification reg. (Read) */
#define SSPICR		SSPIIR			/* Interrupt clear reg. (Write) */
/*
* RESERVED:
*	0x18 - 0x3C
*	0x40 - 0x90		(For test purposes)
*	0x94 - 0xFF
*/

/*********************************************************************
* Synchronous Serial Port Register Structure
*********************************************************************/ 
typedef struct {
	volatile unsigned int	cr0;
	volatile unsigned int	cr1;
	volatile unsigned int	dr;
	volatile unsigned int	sr;
	volatile unsigned int	cpsr;
	union {
		volatile unsigned int	iir;
		volatile unsigned int	icr;
	} u;
	volatile unsigned int	reservedssp[58];
} sspRegs_t;

/*
* To use the structure, declare the following in your source
*	static sspRegs_t *ssp = (sspRegs_t *)SSP_BASE;
*/

/*********************************************************************
* A couple of macros we use here...
*********************************************************************/ 
#ifndef _BIT
#define _BIT(n) (1 << (n))
#endif
#ifndef _SBF
#define _SBF(f,v) ((v) << (f))
#endif
#ifndef _BITMASK
#define _BITMASK(field_width) ( _BIT(field_width) - 1)
#endif

/*********************************************************************
* Synchronous Serial Port Bit Fields
*********************************************************************/ 

/*********************************************************************
* Control 0 Register Bit Fields
*********************************************************************/ 
/* Valid range for argument to SSP_CR0_DSS(n) is [4-16] */ 
#define	SSP_CR0_DSS(n)	_SBF(0,(n)-1)	/* Data Size Select */ 
#define	SSP_CR0_FRF_MOT	_SBF(4,0)	/* Motorola SPI frame */ 
#define	SSP_CR0_FRF_TI	_SBF(4,1)	/* TI synchronous serial frame */ 
#define	SSP_CR0_FRF_NS	_SBF(4,2)	/* National Microwire frame */ 
#define SSP_CR0_SPO		_BIT(6)		/* SPI Polarity */ 
#define SSP_CR0_SPH		_BIT(7)		/* SPI Polarity */ 
#define SSP_CR0_SCR(n)	_SBF(8,(n))	/* Serial Clock Rate */ 

/*********************************************************************
* Control 1 Register Bit Fields
*********************************************************************/ 
#define SSP_CR1_RIE		_BIT(0)		/* RX FIFO interrupt enable */ 
#define SSP_CR1_TIE		_BIT(1)		/* TX FIFO interrupt enable */
#define SSP_CR1_RORIE	_BIT(2)		/* RX FIFO overrun int. enable */ 
#define SSP_CR1_LBM		_BIT(3)		/* Loop back mode */ 
#define SSP_CR1_SSE		_BIT(4)		/* Synchronous serial port enable */ 

/*********************************************************************
* Status Register Bit Fields
*********************************************************************/ 
#define SSP_SR_TFE		_BIT(0)		/* TX FIFO Empty */ 
#define SSP_SR_TNF		_BIT(1)		/* TX FIFO not full */
#define SSP_SR_RNE		_BIT(2)		/* RX FIFO not empty */ 
#define SSP_SR_RFF		_BIT(3)		/* RX FIFO full */ 
#define SSP_SR_BSY		_BIT(4)		/* Busy flag */ 

/*********************************************************************
* Clock Prescale Divisor Register Bit Fields
*********************************************************************/ 
#define	SSP_CPSR_CPDVSR(n) _SBF(0,(n)&0xFE) /* Clock prescale divisor */

/*********************************************************************
* Interrupt Identification / Interrupt Clear Register Bit Fields
* Note: ARM Reference conflicts on the definition of these bits
* and the usage of the registers. Verify before using these
* definitions.
*********************************************************************/ 
#define SSP_IIR_RIS		_BIT(0)		/* TX FIFO Empty */ 
#define SSP_IIR_TIS		_BIT(1)		/* TX FIFO not full */
#define SSP_IIR_RORIS	_BIT(2)		/* RX FIFO overrun int. status */ 

/*********************************************************************
* The TouchScreen communication BPS (bits per second)
*********************************************************************/ 
#define LH7x_TS_BPS 100000

/*********************************************************************
* Misc.
*********************************************************************/ 
#define SSP_MAX_TIMEOUT 0xffff
#define RCPC_SSP_PRESCALE_MAX 256
#define SSP_PRESCALE_MAX 254
#define SSP_PRESCALE_MIN 2
#define SSP_DIVIDER_MAX 256

/* Define values to associate with the device we are conencted to */
#define SSP_INVALID_DEVICE	0xFF
#define SSP_EEPROM			0x00
#define SSP_TOUCHSCREEN		0x01

/*********************************************************************
* Context Structure Definition
*********************************************************************/ 
typedef struct sspContext_t sspContext_t;
struct sspContext_t {
	wait_queue_head_t  *irq_wait_ptr;
	int irq_state;
	int ssp_dev_sel;		// cpld->ssp_dev_sel is not readable !!!
	int ts_txTimeout;
	int ts_rxTimeout;
	int ee_txTimeout;
	int ee_rxTimeout;
	int haveIrq;
	spinlock_t sspLock;
};

#endif /* _SSP_LH7X_h */

