/*
 * drivers/mtd/maps/vmu-flash.c
 * 
 * (C)opyright 2001 Paul Mundt <lethal@chaoticdreams.org>
 +
 +
 + Messed about with and fragments copyright 2002, Adrian McMenamin <adrian@mcmen.demon.co.uk>
 +
 * 
 * Flash mapping handler for the Sega Dreamcast VMU.
 *
 * Released under the terms of the GNU GPL v2.0
 *
 */
#include <linux/init.h>
#include <linux/types.h>
#include <linux/maple.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <asm/io.h>


static int VMU_BLOCK_SIZE;

#define _DEBUG_

#ifdef _DEBUG_
#define DEBGM(fmt, args...)  (printk(KERN_INFO fmt, ##args))
#else
#define DEBGM(fmt, args...) ((void) 0)
#endif



/* MTD Information */
static struct mtd_info *vmu_flash_mtd = NULL;


static struct mapleq *lastmq;
static struct maple_driver dc_flashmap_driver;

static int flash_queried = 0;

static char *block_buffer = NULL;

/* Memory card details */
typedef struct memcard_s {
	long partitions;
	long blocklen;
	long writecnt;
	long readcnt;
	long removable;
	long numblocks;
	struct mtd_info *mtd;
} memcard_t;




/* VMU Block */
typedef struct block_s {
	unsigned int num;	/* Block Number */
	unsigned int ofs;	/* Block Offset */
} block_t;

typedef struct vmu_cache_s {
	char *buffer;		/* Cached memory block */
	unsigned int block;	/* Which block was cached */
	unsigned long jiffies_atc;	/* When was it cached */
	int valid;		/* Cache currently valid? */
} vmu_cache_t;


static vmu_cache_t *vmu_cache = NULL;

wait_queue_head_t wq_mq;
DECLARE_WAIT_QUEUE_HEAD(wq_mq);

/**********Read and write routines*************/
static int maple_vmu_read_block(unsigned int num, u_char * buf,
				struct mtd_info *mtd)
{

	/* async maple call
	   assemble maple call
	   wait for return */


	/* Sanity check */
	if (!mtd) {
		DEBGM
		    ("VMU FLASH: Attempting to read data without having set up mtd.\n");
		return -1;
	}

	struct maple_driver_data *mdd =
	    (struct maple_driver_data *) (mtd->priv);
	memcard_t *memcard = (memcard_t *) (mdd->private_data);
	struct mapleq *mqu = (struct mapleq *) &(mdd->mq);
	mqu->command = MAPLE_COMMAND_BREAD;
	mqu->length = 2;

	((unsigned long *) (mqu->recvbuf))[0] =
	    cpu_to_be32(MAPLE_FUNC_MEMCARD);


	/* With thanks to KOS */
	((unsigned long *) (mqu->recvbuf))[1] = num << 24;
	mqu->sendbuf = mqu->recvbuf;
	if (maple_add_packet(mqu) != 0) {
		DEBGM("VMU FLASH: Could not add packet\n");
		return -1;
	}

	lastmq = NULL;

	do {
		interruptible_sleep_on_timeout(&wq_mq, 1);
	} while (lastmq == NULL);

	/* Now check if we've got a proper return */

	if (block_buffer) {
		memcpy(buf, block_buffer, memcard->blocklen);
		kfree(block_buffer);
		block_buffer = NULL;
		return 0;
	}

	DEBGM("VMU FLASH: Read has failed - return is 0x%X\n",
	      lastmq->recvbuf[0]);
	DEBGM("ERROR code is 0x%X\n", lastmq->recvbuf[1]);

	return -1;

}

static int maple_vmu_write_block(unsigned int num, u_char * buf,
				 struct mtd_info *mtd)
{
	/* Sanity check */
	if (!mtd) {
		printk(KERN_WARNING
		       "VMU FLASH: Attempting to write data without having set up mtd.\n");
		return -1;
	}
	struct maple_driver_data *d =
	    (struct maple_driver_data *) (mtd->priv);
	memcard_t *memcard = (memcard_t *) (d->private_data);


	vmu_cache->valid = 0;	/* Writing so any cache may be invalid */

	struct mapleq *mqu = (struct mapleq *) &(d->mq);

	/* Writing is phased */
	int x;
	for (x = 0; x < memcard->writecnt; x++) {
		mqu->command = MAPLE_COMMAND_BWRITE;
		mqu->length =
		    (memcard->blocklen / (memcard->writecnt * 4)) + 2;

		((unsigned long *) (mqu->recvbuf))[0] =
		    cpu_to_be32(MAPLE_FUNC_MEMCARD);

		((unsigned long *) (mqu->recvbuf))[1] = num << 24 | x << 8;
		memcpy(mqu->recvbuf + 8,
		       buf + (memcard->blocklen / memcard->writecnt) * x,
		       memcard->blocklen / memcard->writecnt);

		mqu->sendbuf = mqu->recvbuf;
		if (maple_add_packet(mqu) != 0) {
			printk(KERN_WARNING
			       "VMU FLASH: Could not add packet\n");
			return -1;
		}

		lastmq = NULL;
		/* Wait */
		do {
			interruptible_sleep_on_timeout(&wq_mq, 1);
		} while (lastmq == NULL);
	}
	return 0;

}

/*************************************/


/**
 * __ofs_to_block - Offset to Block Conversion
 * 
 * @src_ofs: 	Source offset
 *
 * Checks a specified offset's validity, finds its
 * corresponding block, and calculates the remaining
 * offset within the block.
 *
 */
static block_t *__ofs_to_block(unsigned long src_ofs, struct mtd_info *mtd)
{
	block_t *block;

	struct maple_driver_data *d =
	    (struct maple_driver_data *) (mtd->priv);
	memcard_t *card = (memcard_t *) (d->private_data);

	block = kmalloc(sizeof(struct block_s), GFP_KERNEL);

	/* Validate we've got allocated memory */
	if (!block) {
		DEBGM("Can't allocate block\n");
		return NULL;
	}

	/* Zero the block */
	memset(block, 0, sizeof(struct block_s));

	/* Make sure we don't overstep our boundaries */
	if (src_ofs >= card->numblocks * card->blocklen) {
		DEBGM("Source offset exceeds total offset\n");
		kfree(block);
		return NULL;
	}

	/* Find the block number */
	block->num = (unsigned int) (src_ofs / card->blocklen);

	/* Validate we've got a valid block */
	if (block->num > card->numblocks) {
		DEBGM("Block number exceeds number of blocks\n");
		kfree(block);
		return NULL;
	}

	/* Calculate remaining offset in block */
	block->ofs = (unsigned int) (src_ofs % card->blocklen);

	return block;
}

/**
 * 
 * vmu_flash_read8 - Read a Byte
 *
 * @map:	Flash map information structure
 * @ofs:	Offset.
 *
 * Reads a byte from a VMU at the specified offset.
 *
 */

static __u8 vmu_flash_read8(unsigned long ofs, long *retval,
			    struct mtd_info *mtd)
{
	block_t *block;
	*retval = 0;
	struct maple_driver_data *d =
	    (struct maple_driver_data *) (mtd->priv);
	memcard_t *card = (memcard_t *) (d->private_data);
	u_char *buf = kmalloc(card->blocklen, GFP_KERNEL);
	/* Find the block */
	block = __ofs_to_block(ofs, mtd);

	/* Validate it was found */
	if (!block) {
		DEBGM("Got an invalid block\n");
		*retval = 1;
		return -1;
	}

	/* Is this cached? */
	if (vmu_cache->valid) {
		if (vmu_cache->block == block->num) {
			if ((vmu_cache->jiffies_atc + 10) >= jiffies) {	/* short lifespan */
				memcpy(buf, vmu_cache->buffer,
				       card->blocklen);
				u_char *buf2 = buf;
				buf2 += block->ofs;

				kfree(block);
				__u8 ret = (__u8) * buf2;
				kfree(buf);

				return ret;
			} else
				vmu_cache->valid = 0;	/* Force a read faster */
		}
	}



	/* Read the block */
	if (maple_vmu_read_block(block->num, buf, mtd) == -1) {
		DEBGM("Can't read block: %d\n", block->num);
		kfree(block);
		*retval = 2;
		return -1;
	}
	/* Handle the remaining offset */
	u_char *buf2 = buf;
	buf2 += block->ofs;

	kfree(block);
	__u8 ret = (__u8) * buf2;
	kfree(buf);

	return ret;
}

/**
 * vmu_flash_write8 - Write a Byte
 *
 * @map:	Flash map information structure
 * @d:		Byte to write
 * @ofs:	Offset to write byte at
 *
 * Writes a byte to a VMU at the specified offset.
 *
 */
static void vmu_flash_write8(__u8 d, unsigned long ofs,
			     struct mtd_info *mtd)
{
	block_t *block;
	struct maple_driver_data *dx =
	    (struct maple_driver_data *) (mtd->priv);
	memcard_t *card = (memcard_t *) (dx->private_data);
	u_char *buf = kmalloc(card->blocklen, GFP_KERNEL);

	/* Find the block */
	block = __ofs_to_block(ofs, mtd);

	/* Validate it was found */
	if (!block) {
		DEBGM("Got an invalid block\n");
		return;
	}


	/* Read the block */
	if (maple_vmu_read_block(block->num, buf, mtd)) {
		DEBGM("Can't read block: %d\n", block->num);
		kfree(block);
		return;
	}

	/* Copy the byte to the block */
	(__u8) (*(buf + block->ofs)) = d;

	/* Write the block */
	if (maple_vmu_write_block(block->num, buf, mtd)) {
		printk(KERN_WARNING "Can't write block: %d\n", block->num);
		kfree(block);
		return;
	}

	kfree(block);
	kfree(buf);
}

/***********************************************/
/* Read and Write routines                     */

static int vmu_flash_read(struct mtd_info *mtd, loff_t from, size_t len,
			  size_t * retlen, u_char * buf)
{
	/* printk("Reading from %llx with length %llx\n", from, len); */

	struct maple_driver_data *d =
	    (struct maple_driver_data *) (mtd->priv);
	memcard_t *card = (memcard_t *) (d->private_data);

	if (len < 1)
		return -1;
	if (len > (card->numblocks * card->blocklen))
		len = card->numblocks * card->blocklen;
	int start_here = (int) (from & 0xffffffff);
	long retval = 0;
	int index = 0;
	do {
		u8 cx = vmu_flash_read8(start_here + index, &retval, mtd);
		if (retval) {
			*retlen = index;
			return -1;
		}
		memset(buf + index, cx, 1);
		index++;
	} while (len > index);
	*retlen = index;
	return 0;

}

static int vmu_flash_write(struct mtd_info *mtd, loff_t from, size_t len,
			   size_t * retlen, const u_char * buf)
{
	if (len < 1)
		return -1;

	if (len == 1) {		/* Simple one char write */
		vmu_flash_write8(buf[0], from, mtd);
		*retlen = 1;
		return 0;
	}
	struct maple_driver_data *d =
	    (struct maple_driver_data *) (mtd->priv);
	memcard_t *card = (memcard_t *) (d->private_data);


	/* Cached write */
	int index = 0;

	block_t *block;
	/* Find the block */
	block = __ofs_to_block(from, mtd);

	/* Validate it was found */
	if (!block) {
		printk(KERN_WARNING "Got an invalid block\n");
		*retlen = 0;
		return -1;
	}

	do {

		u_char *buffer = kmalloc(card->blocklen, GFP_KERNEL);

		/* Read the block */
		if (maple_vmu_read_block(block->num, buffer, mtd)) {
			DEBGM("Can't read block: %d\n", block->num);
			kfree(block);
			*retlen = 0;
			return -1;
		}

		do {
			buffer[block->ofs] = buf[index];
			(block->ofs)++;
			index++;
			if (index >= len)
				break;
		} while (block->ofs < card->blocklen);
		maple_vmu_write_block(block->num, buffer, mtd);
		kfree(buffer);
		(block->num)++;
		block->ofs = 0;
	} while (len > index);
	*retlen = index;
	kfree(block);
	return 0;

}

static int vmu_flash_erase(struct mtd_info *mtd, struct erase_info *erase)
{
	int z;
	erase->state = MTD_ERASING;
	vmu_flash_write(mtd, erase->addr, erase->len, &z, "\0");
	erase->state = MTD_ERASE_DONE;
	if (erase->callback)
		(erase->callback) (erase);

	return 0;
}

static void vmu_flash_sync(struct mtd_info *mtd)
{
	/* Do nothing */
}

/**
 * vmu_flash_map_init - Initialization
 *
 * Registers and initializes the module.
 *
 */
static int __init vmu_flash_map_init(void)
{
	printk(KERN_NOTICE "VMU flash driver available\n");
	maple_register_driver(&dc_flashmap_driver);
	return 0;
}


/**
 * vmu_flash_map_exit - Cleanup
 *
 * Deregisters and cleans up after the module.
 *
 */
static void __exit vmu_flash_map_exit(void)
{
	printk(KERN_NOTICE "Unregestering VMU flash mapping\n");
	maple_unregister_driver(&dc_flashmap_driver);
	/* See if there's anything to unregister */
	if (!vmu_flash_mtd) {
		DEBGM("Nothing to unregister\n");
		return;
	}

	/* Unregister from MTD */
	del_mtd_device(vmu_flash_mtd);
	map_destroy(vmu_flash_mtd);
}

/***********Maple functions***************/

void dc_flashmap_callback(struct maple_driver_data *data)
{
	struct mapleq *mq = &data->mq;
	if (flash_queried == 0) {
		if (mq->recvbuf[0] == MAPLE_RESPONSE_DEVINFO) {
			lastmq = mq;
			wake_up_interruptible(&wq_mq);	/* Wake sleeping code */
			return;
		} else if (mq->recvbuf[0] == MAPLE_RESPONSE_DATATRF) {
			lastmq = mq;
			wake_up_interruptible(&wq_mq);	/* Wake sleeping code */
			flash_queried = 1;
			return;
		}

	}

	if (mq->recvbuf[0] == MAPLE_RESPONSE_DATATRF) {


		memcard_t *card = (memcard_t *) (data->private_data);
		/*      int res = mq->recvbuf[0];

		   printk
		   ("Maple reply (%d, %d) cmd=%d => %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d\n",
		   mq->port, mq->unit, mq->command, res, mq->recvbuf[1],
		   mq->recvbuf[2], mq->recvbuf[3], mq->recvbuf[4],
		   mq->recvbuf[5], mq->recvbuf[6], mq->recvbuf[7],
		   mq->recvbuf[8], mq->recvbuf[9], mq->recvbuf[10],
		   mq->recvbuf[11]);
		 */
		block_buffer = kmalloc(VMU_BLOCK_SIZE, GFP_KERNEL);
		/* Copy over */
		memcpy(block_buffer, mq->recvbuf + 12, card->blocklen);

		lastmq = mq;
		wake_up_interruptible(&wq_mq);	/* Wake sleeping code */
		memcpy(vmu_cache->buffer, block_buffer, card->blocklen);
		vmu_cache->block = mq->recvbuf[11] & 0xff;
		vmu_cache->jiffies_atc = jiffies;	/* Mark the creation time */
		vmu_cache->valid = 1;	/* Block is now valid */
		return;
	}
	lastmq = mq;
}

static int dc_flashmap_connect(struct maple_driver_data *d)
{
	DEBGM
	    ("Registering VMU Flash mapping and loading VMU Flash driver\n");

	/* What are the characteristics of the flash we have? */


	struct mapleq *mqu = (struct mapleq *) &(d->mq);
	mqu->command = MAPLE_COMMAND_DEVINFO;
	mqu->length = 0;

	lastmq = NULL;
	mqu->sendbuf = mqu->recvbuf;
	if (maple_add_packet(mqu) != 0) {
		DEBGM("VMU FLASH: Could not add packet to query device\n");
		return -1;
	}

	do {
		interruptible_sleep_on_timeout(&wq_mq, 1);
	} while (lastmq == NULL);

	/* Read off the flash data */
	/* How many devices are here? */
	int bit_cnt = 0;
	unsigned long test_flash_data =
	    be32_to_cpu(((unsigned long *) (lastmq->recvbuf))[1]);
	int controller_here = test_flash_data & 0x01;
	int x;
	for (x = 0; x < 32; x++) {
		if (test_flash_data & 0x01)
			bit_cnt++;
		test_flash_data = test_flash_data >> 1;
	}

	/* Only device with a lower index is a controller */
	unsigned long basic_flash_data =
	    be32_to_cpu(((unsigned long *) (lastmq->recvbuf))[1 + bit_cnt -
							      controller_here]);

	/* Create a permanent record 
	 * With thanks to Marcus Comstedt */

	memcard_t *card = kmalloc(sizeof(memcard_t), GFP_KERNEL);
	card->partitions = ((basic_flash_data >> 24) & 0xff) + 1;
	DEBGM("    Flash device has %ld partitions\n", card->partitions);
	card->blocklen = (((basic_flash_data >> 16) & 0xff) + 1) << 5;
	DEBGM("    Flash device has block length of %ld bytes\n",
	      card->blocklen);
	card->writecnt = (basic_flash_data >> 12) & 0xf;
	DEBGM("    Flash device has write count of %ld\n", card->writecnt);
	card->readcnt = (basic_flash_data >> 8) & 0xf;
	DEBGM("    Flash device has read count of %ld\n", card->readcnt);
	card->removable = (basic_flash_data >> 7) & 1;
	DEBGM("    Flash device removable status is %ld\n",
	      card->removable);

	VMU_BLOCK_SIZE = card->blocklen;
	d->private_data = card;

	/* Now query flash partition 0 to find definitve block size */
	mqu = (struct mapleq *) &(d->mq);
	mqu->command = MAPLE_COMMAND_GETMINFO;
	mqu->length = 2;
	((unsigned long *) (mqu->recvbuf))[0] =
	    cpu_to_be32(MAPLE_FUNC_MEMCARD);
	((unsigned long *) (mqu->recvbuf))[1] = 0;	/* Assuming only have one partition */

	lastmq = NULL;
	mqu->sendbuf = mqu->recvbuf;
	if (maple_add_packet(mqu) != 0) {
		DEBGM("VMU FLASH: Could not add packet to query device\n");
		return -1;
	}


	do {
		interruptible_sleep_on_timeout(&wq_mq, 1);
	} while (lastmq == NULL);

	int user_blocks = ((unsigned short *) (lastmq->recvbuf))[12];	/* User accessible blocks */
	int root_block = ((unsigned short *) (lastmq->recvbuf))[6];	/* Root block location */
	DEBGM("    Flash device has %d blocks available to user\n",
	      user_blocks);
	DEBGM("    Flash root block is at block %d\n", root_block);
	card->numblocks = root_block + 1;


	/* Register the flash with mtd subsystem */

	/* Populate a mtd_info */
	vmu_flash_mtd = kmalloc(512, GFP_KERNEL);
	vmu_flash_mtd->name = "Dreamcast VMU Flash";

	/* This is 'other' */
	vmu_flash_mtd->type = MTD_OTHER;
	vmu_flash_mtd->flags =
	    MTD_CLEAR_BITS | MTD_SET_BITS | MTD_ERASEABLE |
	    MTD_WRITEB_WRITEABLE;
	vmu_flash_mtd->size = card->numblocks * card->blocklen;
	vmu_flash_mtd->erasesize = card->blocklen;

	/* Mandatory functions */
	vmu_flash_mtd->write = vmu_flash_write;
	vmu_flash_mtd->read = vmu_flash_read;
	vmu_flash_mtd->erase = vmu_flash_erase;
	vmu_flash_mtd->sync = vmu_flash_sync;
	/* Use private data to point to d */
	vmu_flash_mtd->priv = d;

	vmu_flash_mtd->module = THIS_MODULE;

	/* Register with MTD */
	add_mtd_device(vmu_flash_mtd);
	/* Lock the device in */
	vmu_flash_mtd = get_mtd_device(vmu_flash_mtd, -1);
	if (!vmu_flash_mtd) {
		DEBGM("VMU Flash driver initialisation failed\n");
		return -1;
	}

	card->mtd = vmu_flash_mtd;
	/* Create the cache */
	vmu_cache = kmalloc(64, GFP_KERNEL);
	vmu_cache->buffer = kmalloc(card->blocklen, GFP_KERNEL);
	vmu_cache->valid = 0;

	return 0;
}

static void dc_flashmap_disconnect(struct maple_driver_data *d)
{
	printk(KERN_NOTICE "Unregistering VMU flash mapping\n");

	/* Reead off the relevant mtd */
	memcard_t *card = (memcard_t *) (d->private_data);

	/* See if there's anything to unregister */
	if (!(card->mtd)) {
		DEBGM("Nothing to unregister\n");
		return;
	}


	if (card->mtd) {
		/* Unregister from MTD */
		put_mtd_device(card->mtd);
		del_mtd_device(card->mtd);
		kfree(card->mtd);
		card->mtd = NULL;
	}

	/* Clear the cache */
	kfree(vmu_cache->buffer);
	vmu_cache->valid = 0;
	kfree(vmu_cache);
	kfree((struct memcard_t *) d->private_data);
	flash_queried = 0;


}

/***********Maple device*****************/
static struct maple_driver dc_flashmap_driver = {
	function:MAPLE_FUNC_MEMCARD,
	name:"VMU Flash Memory",
	connect:dc_flashmap_connect,
	disconnect:dc_flashmap_disconnect,
	reply:dc_flashmap_callback,


};


module_init(vmu_flash_map_init);
module_exit(vmu_flash_map_exit);
