/*
 *	Home Agent specific ICMP routines
 *
 *	Authors:
 *	Antti Tuominen	<ajtuomin@tml.hut.fi>
 *	Jaakko Laine	<medved@iki.fi>
 *
 *      $Id: mipv6_icmp_ha.c,v 1.2 2003/06/17 18:53:13 jamey Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 */

#include <linux/autoconf.h>
#include <linux/sched.h>
#include <net/ipv6.h>
#include <net/addrconf.h>
#include <net/ip6_route.h>
#include <net/mipv6.h>

#include "halist.h"
#include "mdetect.h"
#include "debug.h"
#include "util.h"
#include "mipv6_icmp.h"
#include "prefix.h"

/**
 * mipv6_icmpv6_send_dhaad_rep - Reply to DHAAD Request
 * @ifindex: index of interface request was received from
 * @id: request's identification number
 * @daddr: requester's IPv6 address
 *
 * When Home Agent receives Dynamic Home Agent Address Discovery
 * request, it replies with a list of home agents available on the
 * home link.
 */
void mipv6_icmpv6_send_dhaad_rep(int ifindex, __u16 id, struct in6_addr *daddr)
{
	__u8 *data, *addrs;
	struct in6_addr home, *ha_addrs = NULL;
	int addr_count, max_addrs, size = 0;

	if (daddr == NULL)
		return;

	if (mipv6_ha_get_addr(ifindex, &home) < 0) {
		DEBUG(DBG_INFO, "Not Home Agent in this interface");
		return;
	}

	/* We send all available HA addresses, not exceeding a maximum
	 * number we can fit in a packet with minimum IPv6 MTU (to
	 * avoid fragmentation).
	 */
	max_addrs = 76;
	addr_count = mipv6_ha_get_pref_list(ifindex, &ha_addrs, max_addrs);

	if (addr_count < 0) return;

	if (addr_count != 0 && ha_addrs == NULL) {
		DEBUG(DBG_ERROR, "addr_count = %d but return no addresses", 
		      addr_count);
		return;
	}
	/* We allocate space for the icmp data with 8 reserved bytes
	 * in the beginning (there is actually 10 but first 2 are part
	 * of the icmp6hdr).
	 */
	size = 8 + addr_count * sizeof(struct in6_addr);
	data = kmalloc(size, GFP_ATOMIC);
	if (data == NULL) {
		DEBUG(DBG_ERROR, "Couldn't allocate memory");
		kfree(ha_addrs);
		return;
	}

	memset(data, 0, size);
	if (addr_count > 0) {
		int off = 0;
		if (ipv6_addr_cmp(ha_addrs, &home) == 0) {
			size -= sizeof(struct in6_addr);
			off = 1;
		}
		if (addr_count > off) {
			addrs = (data + 8); /* skip reserved and copy addresses*/
			memcpy(addrs, ha_addrs + off, 
			       (addr_count - off) * sizeof(struct in6_addr));
		}
		kfree(ha_addrs);
	}

	mipv6_icmpv6_send(daddr, &home, MIPV6_DHAAD_REPLY, 0, &id, &data, size);
	kfree(data);
}

/** 
 * mipv6_icmpv6_dhaad_req - Home Agent Address Discovery Request ICMP handler
 * @skb: buffer containing ICMP information message
 *
 * Special Mobile IPv6 ICMP message.  Handles Dynamic Home Agent
 * Address Discovery Request messages.
 **/
int mipv6_icmpv6_rcv_dhaad_req(struct sk_buff *skb)
{
	struct icmp6hdr *phdr = (struct icmp6hdr *) skb->h.raw;
	struct in6_addr *saddr = &skb->nh.ipv6h->saddr;
	struct in6_addr *daddr = &skb->nh.ipv6h->daddr;
	__u16 identifier;
	int ifindex = ((struct inet6_skb_parm *)skb->cb)->iif;
	struct rt6_info *rt;
	struct in6_addr ha_anycast;

	DEBUG_FUNC();

	/* Invalid packet checks. */
	if (phdr->icmp6_code != 0)
		return 0;

	identifier = ntohs(phdr->icmp6_identifier);


	/*
	 * Use the network prefix only but not the
	 * entire * address from the anycast to find
	 * out the route entry * corresponding to a
	 * proper device entry but not a 'lo' * entry.
	 */
	ipv6_addr_set(&ha_anycast, daddr->s6_addr32[0], 
		      daddr->s6_addr32[1], 0, 0);

	/* 
	 * Make sure we have the right ifindex (if the
	 * req came through another interface. 
	 */

	rt = rt6_lookup(&ha_anycast, NULL, 0, 0);
	if (rt) {
		ifindex = rt->rt6i_dev->ifindex;
		dst_release(&rt->u.dst);
	} else {
		/*
		 * Can't send a reply since the right
		 * interface * is not found 
		 */
		DEBUG(DBG_WARNING, "No route entry found for"
		      "a DHAAD request");
		return 0;
	}
	/*
	 * send reply with list
	 */
	mipv6_icmpv6_send_dhaad_rep(ifindex, identifier, saddr);
	return 1;
}

/**
 * mipv6_icmpv6_handle_pfx_sol - handle prefix solicitations
 * @skb: sk_buff including the icmp6 message
 */
int mipv6_icmpv6_rcv_pfx_sol(struct sk_buff *skb)
{
	struct icmp6hdr *hdr = (struct icmp6hdr *) skb->h.raw;
	struct in6_addr *saddr = &skb->nh.ipv6h->saddr;
	struct in6_addr *daddr = &skb->nh.ipv6h->daddr;
	struct prefix_info *plist;
	struct inet6_ifaddr *ifp;
	__u16 identifier = ntohs(hdr->icmp6_identifier);
	int count;

	DEBUG_FUNC();

	if (!(ifp = ipv6_get_ifaddr(daddr, NULL)))
		return -1;

	if ((count = ipv6_get_prefix_entries(&plist, ifp->idev->dev->ifindex, 0)) > 0) {
		mipv6_icmpv6_send(saddr, daddr, MIPV6_PREFIX_ADV, 0, &identifier,
				  plist, count * sizeof(struct prefix_info));
		kfree(plist);
	}

	in6_ifa_put(ifp);
	mipv6_pfx_cancel_send(saddr, -1);

	return 0;
}
