
/*
 *	FILE    	Itsy.h
 *
 *	Version 	2.0
 *	Author  	Marc A. Viredaz
 *	        	Compaq Western Research Laboratory, Palo Alto, CA
 *	        	(DEC Western Research Laboratory, Palo Alto, CA)
 *	Date    	January 2000 (September 1997)
 *	System  	Itsy versions 1.0/1.1, 1.5, 1.6, 1.7, and 2.0.
 *	        	(StrongARM SA-1100)
 *	Language	C or ARM Assembly
 *	Purpose 	Definition of constants related to the Itsy
 *	        	mother-board versions 1.0/1.1, 1.5, 1.6, 1.7, and 2.0.
 *	        	This system is based on the StrongARM SA-1100
 *	        	microprocessor (Advanced RISC Machine (ARM)
 *	        	architecture version 4).
 *	Copyright	(c) Compaq Computer Corporation, 1997, 2000.
 *
 *	        	Language-specific definitions are selected by the
 *	        	macro "LANGUAGE", which should be defined as either
 *	        	"C" (default) or "Assembly".
 */


#ifndef ITSY
#define ITSY

#ifndef LANGUAGE
#define LANGUAGE	C
#endif /* !defined (LANGUAGE) */

#include "./cmptblty.h"
#include "./bitfield.h"
#include "./SA-1100.h"

#define C       	0
#define Assembly	1


/*
 * General definitions
 */

#define CrystlFreq	3686400 	/* Crystal Freq. [Hz]              */
#define RTCrystlFreq	32768   	/* Real-Time Crystal Freq. [Hz]    */


/*
 * Memory
 *
 * Registers
 *    SM16ID0   	Static Memory 16-bit device IDentification register 0
 *              	(read, 16-bit).
 *    SM16VER0  	Static Memory 16-bit device VERsion register 0
 *              	(read, 16-bit).
 *    SM32ID0   	Static Memory 32-bit device IDentification and
 *              	version register 0 (read, 32-bit).
 *    SM16GPN0  	Static Memory 16-bit device GPIO Number register 0
 *              	(read, 16-bit).
 *    SM32GPN0  	Static Memory 32-bit device GPIO Number register 0
 *              	(read, 32-bit).
 *    SM16C0 [12]	Static Memory 16-bit device Control table 0
 *              	(read, 12 x 16-bit).
 *              	[The SM16C0 table has 11 entries for 11 frequencies
 *              	in version 0 and 12 entries for 12 frequencies in
 *              	versions 1, 2 and higher.]
 *    SM32C0 [6]	Static Memory 32-bit device Control table 0
 *              	(read, 6 x 32-bit).
 *              	[The SM32C0 table has 6 entries for 11 frequencies
 *              	in version 0 and 6 entries for 12 frequencies in
 *              	versions 1, 2 and higher.]
 *    SM16EC0 [23]	Static Memory 16-bit device Extended Control table 0
 *              	(read, 23 x 16-bit).
 *              	[The SM16EC0 table is only implemented in versions 2
 *              	and higher.]
 *    SM32EC0 [12]	Static Memory 32-bit device Extended Control table 0
 *              	(read, 12 x 32-bit).
 *              	[The SM32EC0 table is only implemented in versions 2
 *              	and higher.]
 *
 *    SM16ID1   	Static Memory 16-bit device IDentification register 1
 *              	(read, 16-bit).
 *    SM16VER1  	Static Memory 16-bit device VERsion register 1
 *              	(read, 16-bit).
 *    SM32ID1   	Static Memory 32-bit device IDentification and
 *              	version register 1 (read, 32-bit).
 *    SM16GPN1  	Static Memory 16-bit device GPIO Number register 1
 *              	(read, 16-bit).
 *    SM32GPN1  	Static Memory 32-bit device GPIO Number register 1
 *              	(read, 32-bit).
 *    SM16C1 [12]	Static Memory 16-bit device Control table 1
 *              	(read, 12 x 16-bit).
 *              	[The SM16C1 table has 11 entries for 11 frequencies
 *              	in version 0 and 12 entries for 12 frequencies in
 *              	versions 1, 2 and higher.]
 *    SM32C1 [6]	Static Memory 32-bit device Control table 1
 *              	(read, 6 x 32-bit).
 *              	[The SM32C1 table has 6 entries for 11 frequencies
 *              	in version 0 and 6 entries for 12 frequencies in
 *              	versions 1, 2 and higher.]
 *    SM16EC1 [23]	Static Memory 16-bit device Extended Control table 1
 *              	(read, 23 x 16-bit).
 *              	[The SM16EC1 table is only implemented in versions 2
 *              	and higher.]
 *    SM32EC1 [12]	Static Memory 32-bit device Extended Control table 1
 *              	(read, 12 x 32-bit).
 *              	[The SM32EC1 table is only implemented in versions 2
 *              	and higher.]
 *
 *    SM16ID2   	Static Memory 16-bit device IDentification register 2
 *              	(read, 16-bit).
 *    SM16VER2  	Static Memory 16-bit device VERsion register 2
 *              	(read, 16-bit).
 *    SM32ID2   	Static Memory 32-bit device IDentification and
 *              	version register 2 (read, 32-bit).
 *    SM16GPN2  	Static Memory 16-bit device GPIO Number register 2
 *              	(read, 16-bit).
 *    SM32GPN2  	Static Memory 32-bit device GPIO Number register 2
 *              	(read, 32-bit).
 *    SM16C2 [12]	Static Memory 16-bit device Control table 2
 *              	(read, 12 x 16-bit).
 *              	[The SM16C2 table has 11 entries for 11 frequencies
 *              	in version 0 and 12 entries for 12 frequencies in
 *              	versions 1, 2 and higher.]
 *    SM32C2 [6]	Static Memory 32-bit device Control table 2
 *              	(read, 6 x 32-bit).
 *              	[The SM32C2 table has 6 entries for 11 frequencies
 *              	in version 0 and 6 entries for 12 frequencies in
 *              	versions 1, 2 and higher.]
 *    SM16EC2 [23]	Static Memory 16-bit device Extended Control table 2
 *              	(read, 23 x 16-bit).
 *              	[The SM16EC2 table is only implemented in versions 2
 *              	and higher.]
 *    SM32EC2 [12]	Static Memory 32-bit device Extended Control table 2
 *              	(read, 12 x 32-bit).
 *              	[The SM32EC2 table is only implemented in versions 2
 *              	and higher.]
 *
 *    SM16ID3   	Static Memory 16-bit device IDentification register 3
 *              	(read, 16-bit).
 *    SM16VER3  	Static Memory 16-bit device VERsion register 3
 *              	(read, 16-bit).
 *    SM32ID3   	Static Memory 32-bit device IDentification and
 *              	version register 3 (read, 32-bit).
 *    SM16GPN3  	Static Memory 16-bit device GPIO Number register 3
 *              	(read, 16-bit).
 *    SM32GPN3  	Static Memory 32-bit device GPIO Number register 3
 *              	(read, 32-bit).
 *    SM16C3 [12]	Static Memory 16-bit device Control table 3
 *              	(read, 12 x 16-bit).
 *              	[The SM16C3 table has 11 entries for 11 frequencies
 *              	in version 0 and 12 entries for 12 frequencies in
 *              	versions 1, 2 and higher.]
 *    SM32C3 [6]	Static Memory 32-bit device Control table 3
 *              	(read, 6 x 32-bit).
 *              	[The SM32C3 table has 6 entries for 11 frequencies
 *              	in version 0 and 6 entries for 12 frequencies in
 *              	versions 1, 2 and higher.]
 *    SM16EC3 [23]	Static Memory 16-bit device Extended Control table 3
 *              	(read, 23 x 16-bit).
 *              	[The SM16EC3 table is only implemented in versions 2
 *              	and higher.]
 *    SM32EC3 [12]	Static Memory 32-bit device Extended Control table 3
 *              	(read, 12 x 32-bit).
 *              	[The SM32EC3 table is only implemented in versions 2
 *              	and higher.]
 *
 * Clocks
 *    fxtl, Txtl	Frequency, period of the system crystal (3.6864 MHz).
 *    fcpu, Tcpu	Frequency, period of the CPU core clock (CCLK).
 */

#define DRAMSz  	0x01000000	/* DRAM bank Size [byte]           */
#define DRAM0Sz 	DRAMSz  	/* DRAM bank 0 Size [byte]         */
#define DRAM1Sz 	DRAMSz  	/* DRAM bank 1 Size [byte]         */
                	        	/* [Itsy v2.x only]                */

#define _BootMem	_StMemBnk0	/* Boot Memory                     */
#define _FlMem  	_StMemBnk1	/* Flash Memory                    */
#define _DRAM   	_DRAMBnk0	/* DRAM                            */
#define _DRAM0  	_DRAMBnk0	/* DRAM 0                          */
#define _DRAM1  	_DRAMBnk1	/* DRAM 1 [Itsy v2.x only]         */

#if LANGUAGE == C
#define BootMem 	StMemBnk0	/* Boot Memory                     */
#define FlMem   	StMemBnk1	/* Flash Memory                    */
#define DRAM    	DRAMBnk0	/* DRAM                            */
#define DRAM0   	DRAMBnk0	/* DRAM 0                          */
#define DRAM1   	DRAMBnk1	/* DRAM 1 [Itsy v2.x only]         */
#endif /* LANGUAGE == C */

#define _SM16ID(Nb)	        	/* Static Memory 16-bit device     */ \
                	        	/* IDentification reg. [0..3]      */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 2)
#define _SM16VER(Nb)	        	/* Static Memory 16-bit device     */ \
                	        	/* VERsion reg. [0..3]             */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 4)
#define _SM32ID(Nb)	        	/* Static Memory 32-bit device     */ \
                	        	/* IDentification & version        */ \
                	        	/* reg. [0..3]                     */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 4)
#define _SM16GPN(Nb)	        	/* Static Memory 16-bit device     */ \
                	        	/* GPIO Number reg. [0..3]         */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 6)
#define _SM32GPN(Nb)	        	/* Static Memory 32-bit device     */ \
                	        	/* GPIO Number reg. [0..3]         */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 8)
#define _SM16C(Nb, ClkFreqNb)   	/* Static Memory 16-bit device     */ \
                	        	/* Control table [0..3][0..11]     */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 32 + 2*(ClkFreqNb))
#define _SM32C(Nb, ClkFreqNb)   	/* Static Memory 32-bit device     */ \
                	        	/* Control table [0..3][0..11]     */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 32 + 4*((ClkFreqNb)/2))
#define _SM16EC(Nb, ClkFreqNb)  	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table          */ \
                	        	/* [0..3][0..22]                   */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 256 + 2*(ClkFreqNb))
#define _SM32EC(Nb, ClkFreqNb)  	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table          */ \
                	        	/* [0..3][0..22]                   */ \
                	(_StMemBnk (Nb) + StMemBnkSp - 256 + 4*((ClkFreqNb)/2))

#define _SM16ID0	_SM16ID (0)	/* Static Memory 16-bit device     */
                	        	/* IDentification reg. 0           */
#define _SM16VER0	_SM16VER (0)	/* Static Memory 16-bit device     */
                	        	/* VERsion reg. 0                  */
#define _SM32ID0	_SM32ID (0)	/* Static Memory 32-bit device     */
                	        	/* IDentification & version reg. 0 */
#define _SM16GPN0	_SM16GPN (0)	/* Static Memory 16-bit device     */
                	        	/* GPIO Number reg. 0              */
#define _SM32GPN0	_SM32GPN (0)	/* Static Memory 32-bit device     */
                	        	/* GPIO Number reg. 0              */
#define _SM16C0(ClkFreqNb)      	/* Static Memory 16-bit device     */ \
                	        	/* Control table 0 [0..11]         */ \
                	_SM16C (0, (ClkFreqNb))
#define _SM32C0(ClkFreqNb)      	/* Static Memory 32-bit device     */ \
                	        	/* Control table 0 [0..11]         */ \
                	_SM32C (0, (ClkFreqNb))
#define _SM16EC0(ClkFreqNb)     	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table 0        */ \
                	        	/* [0..22]                         */ \
                	_SM16EC (0, (ClkFreqNb))
#define _SM32EC0(ClkFreqNb)     	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table 0        */ \
                	        	/* [0..22]                         */ \
                	_SM32EC (0, (ClkFreqNb))


#define _SM16ID1	_SM16ID (1)	/* Static Memory 16-bit device     */
                	        	/* IDentification reg. 1           */
#define _SM16VER1	_SM16VER (1)	/* Static Memory 16-bit device     */
                	        	/* VERsion reg. 1                  */
#define _SM32ID1	_SM32ID (1)	/* Static Memory 32-bit device     */
                	        	/* IDentification & version reg. 1 */
#define _SM16GPN1	_SM16GPN (1)	/* Static Memory 16-bit device     */
                	        	/* GPIO Number reg. 1              */
#define _SM32GPN1	_SM32GPN (1)	/* Static Memory 32-bit device     */
                	        	/* GPIO Number reg. 1              */
#define _SM16C1(ClkFreqNb)      	/* Static Memory 16-bit device     */ \
                	        	/* Control table 1 [0..11]         */ \
                	_SM16C (1, (ClkFreqNb))
#define _SM32C1(ClkFreqNb)      	/* Static Memory 32-bit device     */ \
                	        	/* Control table 1 [0..11]         */ \
                	_SM32C (1, (ClkFreqNb))
#define _SM16EC1(ClkFreqNb)     	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table 1        */ \
                	        	/* [0..22]                         */ \
                	_SM16EC (1, (ClkFreqNb))
#define _SM32EC1(ClkFreqNb)     	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table 1        */ \
                	        	/* [0..22]                         */ \
                	_SM32EC (1, (ClkFreqNb))

#define _SM16ID2	_SM16ID (2)	/* Static Memory 16-bit device     */
                	        	/* IDentification reg. 2           */
#define _SM16VER2	_SM16VER (2)	/* Static Memory 16-bit device     */
                	        	/* VERsion reg. 2                  */
#define _SM32ID2	_SM32ID (2)	/* Static Memory 32-bit device     */
                	        	/* IDentification & version reg. 2 */
#define _SM16GPN2	_SM16GPN (2)	/* Static Memory 16-bit device     */
                	        	/* GPIO Number reg. 2              */
#define _SM32GPN2	_SM32GPN (2)	/* Static Memory 32-bit device     */
                	        	/* GPIO Number reg. 2              */
#define _SM16C2(ClkFreqNb)      	/* Static Memory 16-bit device     */ \
                	        	/* Control table 2 [0..11]         */ \
                	_SM16C (2, (ClkFreqNb))
#define _SM32C2(ClkFreqNb)      	/* Static Memory 32-bit device     */ \
                	        	/* Control table 2 [0..11]         */ \
                	_SM32C (2, (ClkFreqNb))
#define _SM16EC2(ClkFreqNb)     	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table 2        */ \
                	        	/* [0..22]                         */ \
                	_SM16EC (2, (ClkFreqNb))
#define _SM32EC2(ClkFreqNb)     	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table 2        */ \
                	        	/* [0..22]                         */ \
                	_SM32EC (2, (ClkFreqNb))

#define _SM16ID3	_SM16ID (3)	/* Static Memory 16-bit device     */
                	        	/* IDentification reg. 3           */
#define _SM16VER3	_SM16VER (3)	/* Static Memory 16-bit device     */
                	        	/* VERsion reg. 3                  */
#define _SM32ID3	_SM32ID (3)	/* Static Memory 32-bit device     */
                	        	/* IDentification & version reg. 3 */
#define _SM16GPN3	_SM16GPN (3)	/* Static Memory 16-bit device     */
                	        	/* GPIO Number reg. 3              */
#define _SM32GPN3	_SM32GPN (3)	/* Static Memory 32-bit device     */
                	        	/* GPIO Number reg. 3              */
#define _SM16C3(ClkFreqNb)      	/* Static Memory 16-bit device     */ \
                	        	/* Control table 3 [0..11]         */ \
                	_SM16C (3, (ClkFreqNb))
#define _SM32C3(ClkFreqNb)      	/* Static Memory 32-bit device     */ \
                	        	/* Control table 3 [0..11]         */ \
                	_SM32C (3, (ClkFreqNb))
#define _SM16EC3(ClkFreqNb)     	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table 3        */ \
                	        	/* [0..22]                         */ \
                	_SM16EC (3, (ClkFreqNb))
#define _SM32EC3(ClkFreqNb)     	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table 3        */ \
                	        	/* [0..22]                         */ \
                	_SM32EC (3, (ClkFreqNb))

#if LANGUAGE == C

typedef Word16  	SM16CType [12] ;
typedef Word32  	SM32CType [6] ;
typedef Word16  	SM16ECType [23] ;
typedef Word32  	SM32ECType [12] ;

#define SM16ID0 	        	/* Static Memory 16-bit device     */ \
                	        	/* IDentification reg. 0           */ \
                	(*((Word16 *) VirtAdd (_SM16ID0)))
#define SM16VER0	        	/* Static Memory 16-bit device     */ \
                	        	/* VERsion reg. 0                  */ \
                	(*((Word16 *) VirtAdd (_SM16VER0)))
#define SM32ID0 	        	/* Static Memory 32-bit device     */ \
                	        	/* IDentification & version reg. 0 */ \
                	(*((Word32 *) VirtAdd (_SM32ID0)))
#define SM16GPN0	        	/* Static Memory 16-bit device     */ \
                	        	/* GPIO Number reg. 0              */ \
                	(*((Word16 *) VirtAdd (_SM16GPN0)))
#define SM32GPN0	        	/* Static Memory 32-bit device     */ \
                	        	/* GPIO Number reg. 0              */ \
                	(*((Word32 *) VirtAdd (_SM32GPN0)))
#define SM16C0  	        	/* Static Memory 16-bit device     */ \
                	        	/* Control table 0 [0..11]         */ \
                	(*((SM16CType *) VirtAdd (_SM16C0 (0))))
#define SM32C0  	        	/* Static Memory 32-bit device     */ \
                	        	/* Control table 0 [0..5]          */ \
                	(*((SM32CType *) VirtAdd (_SM32C0 (0))))
#define SM16EC0 	        	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table 0        */ \
                	        	/* [0..22]                         */ \
                	(*((SM16ECType *) VirtAdd (_SM16EC0 (0))))
#define SM32EC0 	        	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table 0        */ \
                	        	/* [0..11]                         */ \
                	(*((SM32ECType *) VirtAdd (_SM32EC0 (0))))

#define SM16ID1 	        	/* Static Memory 16-bit device     */ \
                	        	/* IDentification reg. 1           */ \
                	(*((Word16 *) VirtAdd (_SM16ID1)))
#define SM16VER1	        	/* Static Memory 16-bit device     */ \
                	        	/* VERsion reg. 1                  */ \
                	(*((Word16 *) VirtAdd (_SM16VER1)))
#define SM32ID1 	        	/* Static Memory 32-bit device     */ \
                	        	/* IDentification & version reg. 1 */ \
                	(*((Word32 *) VirtAdd (_SM32ID1)))
#define SM16GPN1	        	/* Static Memory 16-bit device     */ \
                	        	/* GPIO Number reg. 1              */ \
                	(*((Word16 *) VirtAdd (_SM16GPN1)))
#define SM32GPN1	        	/* Static Memory 32-bit device     */ \
                	        	/* GPIO Number reg. 1              */ \
                	(*((Word32 *) VirtAdd (_SM32GPN1)))
#define SM16C1  	        	/* Static Memory 16-bit device     */ \
                	        	/* Control table 1 [0..11]         */ \
                	(*((SM16CType *) VirtAdd (_SM16C1 (0))))
#define SM32C1  	        	/* Static Memory 32-bit device     */ \
                	        	/* Control table 1 [0..5]          */ \
                	(*((SM32CType *) VirtAdd (_SM32C1 (0))))
#define SM16EC1 	        	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table 1        */ \
                	        	/* [0..22]                         */ \
                	(*((SM16ECType *) VirtAdd (_SM16EC1 (0))))
#define SM32EC1 	        	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table 1        */ \
                	        	/* [0..11]                         */ \
                	(*((SM32ECType *) VirtAdd (_SM32EC1 (0))))

#define SM16ID2 	        	/* Static Memory 16-bit device     */ \
                	        	/* IDentification reg. 2           */ \
                	(*((Word16 *) VirtAdd (_SM16ID2)))
#define SM16VER2	        	/* Static Memory 16-bit device     */ \
                	        	/* VERsion reg. 2                  */ \
                	(*((Word16 *) VirtAdd (_SM16VER2)))
#define SM32ID2 	        	/* Static Memory 32-bit device     */ \
                	        	/* IDentification & version reg. 2 */ \
                	(*((Word32 *) VirtAdd (_SM32ID2)))
#define SM16GPN2	        	/* Static Memory 16-bit device     */ \
                	        	/* GPIO Number reg. 2              */ \
                	(*((Word16 *) VirtAdd (_SM16GPN2)))
#define SM32GPN2	        	/* Static Memory 32-bit device     */ \
                	        	/* GPIO Number reg. 2              */ \
                	(*((Word32 *) VirtAdd (_SM32GPN2)))
#define SM16C2  	        	/* Static Memory 16-bit device     */ \
                	        	/* Control table 2 [0..11]         */ \
                	(*((SM16CType *) VirtAdd (_SM16C2 (0))))
#define SM32C2  	        	/* Static Memory 32-bit device     */ \
                	        	/* Control table 2 [0..5]          */ \
                	(*((SM32CType *) VirtAdd (_SM32C2 (0))))
#define SM16EC2 	        	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table 2        */ \
                	        	/* [0..22]                         */ \
                	(*((SM16ECType *) VirtAdd (_SM16EC2 (0))))
#define SM32EC2 	        	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table 2        */ \
                	        	/* [0..11]                         */ \
                	(*((SM32ECType *) VirtAdd (_SM32EC2 (0))))

#define SM16ID3 	        	/* Static Memory 16-bit device     */ \
                	        	/* IDentification reg. 3           */ \
                	(*((Word16 *) VirtAdd (_SM16ID3)))
#define SM16VER3	        	/* Static Memory 16-bit device     */ \
                	        	/* VERsion reg. 3                  */ \
                	(*((Word16 *) VirtAdd (_SM16VER3)))
#define SM32ID3 	        	/* Static Memory 32-bit device     */ \
                	        	/* IDentification & version reg. 3 */ \
                	(*((Word32 *) VirtAdd (_SM32ID3)))
#define SM16GPN3	        	/* Static Memory 16-bit device     */ \
                	        	/* GPIO Number reg. 3              */ \
                	(*((Word16 *) VirtAdd (_SM16GPN3)))
#define SM32GPN3	        	/* Static Memory 32-bit device     */ \
                	        	/* GPIO Number reg. 3              */ \
                	(*((Word32 *) VirtAdd (_SM32GPN3)))
#define SM16C3  	        	/* Static Memory 16-bit device     */ \
                	        	/* Control table 3 [0..11]         */ \
                	(*((SM16CType *) VirtAdd (_SM16C3 (0))))
#define SM32C3  	        	/* Static Memory 32-bit device     */ \
                	        	/* Control table 3 [0..5]          */ \
                	(*((SM32CType *) VirtAdd (_SM32C3 (0))))
#define SM16EC3 	        	/* Static Memory 16-bit device     */ \
                	        	/* Extended Control table 3        */ \
                	        	/* [0..22]                         */ \
                	(*((SM16ECType *) VirtAdd (_SM16EC3 (0))))
#define SM32EC3 	        	/* Static Memory 32-bit device     */ \
                	        	/* Extended Control table 3        */ \
                	        	/* [0..11]                         */ \
                	(*((SM32ECType *) VirtAdd (_SM32EC3 (0))))

#endif /* LANGUAGE == C */

#define SMID_CID	Fld (8, 0)	/* Class IDentification reg.       */
#define SMID_nNVM	        	/*  not Non-Volatile Memory        */ \
                	FInsrt (0x80, SMID_CID)
#define SMID_NVMemDev	(SMID_nNVM*0)	/*   Non-Volatile Memory Device    */
#define SMID_StMemDev	(SMID_nNVM*1)	/*   Static Memory Device          */
#define SMID_NonVolMem	        	/*  Non-Volatile Memory            */ \
                	((FInsrt (0x00, SMID_CID) BAnd ~SMID_nNVM) + \
                	 SMID_NVMemDev)
#define SMID_CstmDev	        	/*  Custom Device                  */ \
                	((FInsrt (0x80, SMID_CID) BAnd ~SMID_nNVM) + \
                	 SMID_StMemDev)
#define SMID_NoDev	        	/*  No Device                      */ \
                	((FInsrt (0xFF, SMID_CID) BAnd ~SMID_nNVM) + \
                	 SMID_StMemDev)
                	        	/* non-volatile memory device:     */
#define SMID_R  	0x0100  	/*  Read-only static memory        */
#define SMID_FlMem	(SMID_R*0)	/*   Flash Memory                  */
#define SMID_ROM	(SMID_R*1)	/*   Read-Only Memory              */
#define SMID_D  	0x0200  	/*  Daughter-card static memory    */
#define SMID_MBFlMem	(SMID_D*0)	/*   Mother-Board Flash Memory     */
#define SMID_DCStMem	(SMID_D*1)	/*   Daughter-Card Static Memory   */
#define SMID_W  	0x0400  	/*  static memory Width            */
#define SMID_32BitStMem	(SMID_W*0)	/*   32-Bit Static Memory          */
#define SMID_16BitStMem	(SMID_W*1)	/*   16-Bit Static Memory          */
#define SMID_SIZE	Fld (5, 11)	/*  base-2 log. of static memory   */
                	        	/*  SIZE [byte]                    */

#define SM16VER_Version	Fld (16, 0)	/* Version                         */

#define SM32ID_Version	Fld (16, 16)	/* Version                         */

#define SMGPN_NoGPIO	0xFF    	/* No GPIO                         */
#define SMGPN_EN0	Fld (8, 0)	/* static memory ENable 0 ([15:0]) */
#define SMGPN_RYnBY0	Fld (8, 8)	/* static memory ReadY not BusY 0  */
                	        	/* ([15:0])                        */
#define SMGPN_EN1	Fld (8, 16)	/* static memory ENable 1          */
                	        	/* ([31:16])                       */
#define SMGPN_RYnBY1	Fld (8, 24)	/* static memory ReadY not BusY 1  */
                	        	/* ([31:16])                       */

#define SM32C_C(ClkFreqNb)      	/* static memory 32-bit device     */ \
                	        	/* Control table [0..11]           */ \
                	Fld (16, (((ClkFreqNb) + 1) Modulo 2)*16)

#define SM32EC_C(ClkFreqNb)     	/* static memory 32-bit device     */ \
                	        	/* Extended Control table [0..22]  */ \
                	Fld (16, (((ClkFreqNb) + 1) Modulo 2)*16)


/*
 * Power Manager (PM) control registers
 *
 * Registers
 *    PWER      	Power Manager (PM) Wake-up Enable Register
 *              	(read/write).
 *    PGSR      	Power Manager (PM) General-Purpose Input/Output (GPIO)
 *              	Sleep state Register (read/write, see GPIO pins).
 */

                	        	/* Itsy v1.x:                      */
#define PWER_MainPB	GPIO_nMAINPB	/*  Main Push-Button               */
#define PWER_Pwr	GPIO_PWROK	/*  Power OK/bad                   */
#define PWER_FlMem0	GPIO_FL0RYnBY	/*  Flash Memory 0 ([15:0])        */
                	        	/*  ready/busy                     */
#define PWER_FlMem1	GPIO_FL1RYnBY	/*  Flash Memory 1 ([31:16])       */
                	        	/*  ready/busy                     */
#define PWER_CODECInt	GPIO_CODECINT	/*  CODEC Interrupt                */
#define PWER_UARTRc	GPIO_UARTVALID	/*  ser. port 3: UART Receive      */
                	        	/*  signal valid/invalid           */
#define PWER_UARTDSR	GPIO_UARTDSR	/*  ser. port 3: UART Data Set     */
                	        	/*  Ready                          */

                	        	/* Itsy v2.x:                      */
#define PWER_2MainPB	GPIO_2nMAINPB	/*  Main Push-Button               */
#define PWER_2CODECInt	GPIO_2CODECINT	/*  CODEC Interrupt                */
#define PWER_2ChgOnOff	GPIO_2nCHGOFF	/*  Charger On/Off                 */
#define PWER_2FlMem	GPIO_2FLRYnBY	/*  Flash Memory ready/busy        */


/*
 * General-Purpose Input/Output (GPIO) control registers
 *
 * Registers
 *    GPLR      	General-Purpose Input/Output (GPIO) Pin Level
 *              	Register (read).
 *    GPDR      	General-Purpose Input/Output (GPIO) Pin Direction
 *              	Register (read/write).
 *    GPSR      	General-Purpose Input/Output (GPIO) Pin output Set
 *              	Register (write).
 *    GPCR      	General-Purpose Input/Output (GPIO) Pin output Clear
 *              	Register (write).
 *    GRER      	General-Purpose Input/Output (GPIO) Rising-Edge
 *              	detect Register (read/write).
 *    GFER      	General-Purpose Input/Output (GPIO) Falling-Edge
 *              	detect Register (read/write).
 *    GEDR      	General-Purpose Input/Output (GPIO) Edge Detect
 *              	status Register (read/write).
 *    GAFR      	General-Purpose Input/Output (GPIO) Alternate
 *              	Function Register (read/write).
 */

                	        	/* Itsy v1.x:                      */
#define GPIO_nMAINPB	GPIO_GPIO (0)	/*  not MAIN Push-Button (I)       */
#define GPIO_MainPBOn	        	/*   Main Push-Button On           */ \
                	(GPIO_nMAINPB*0)
#define GPIO_MainPBOff	        	/*   Main Push-Button Off          */ \
                	(GPIO_nMAINPB*1)
#define GPIO_PWROK	GPIO_GPIO (2)	/*  PoWeR OK (I)                   */
#define GPIO_PwrBad	(GPIO_PWROK*0)	/*   Power Bad (Vcc < ~ 3.0 V)     */
#define GPIO_PwrOK	(GPIO_PWROK*1)	/*   Power OK (Vcc > ~ 3.0 V)      */
#define GPIO_nFLFOFF	GPIO_GPIO (3)	/*  not FLash memory Force OFF (O) */
#define GPIO_FlMemDis	        	/*   Flash Memory Disable (reset,  */ \
                	        	/*   low-power mode)               */ \
                	(GPIO_nFLFOFF*0)
#define GPIO_FlMemEn	        	/*   Flash Memory Enable (when     */ \
                	        	/*   PWR_EN asserted)              */ \
                	(GPIO_nFLFOFF*1)
#define GPIO_FL0RYnBY	GPIO_GPIO (4)	/*  FLash memory 0 ([15:0]) ReadY  */
                	        	/*  not BusY (I)                   */
#define GPIO_FlMem0Bsy	        	/*   Flash Memory 0 ([15:0]) Busy  */ \
                	(GPIO_FL0RYnBY*0)
#define GPIO_FlMem0Rdy	        	/*   Flash Memory 0 ([15:0]) Ready */ \
                	(GPIO_FL0RYnBY*1)
#define GPIO_FL1RYnBY	GPIO_GPIO (5)	/*  FLash memory 1 ([31:16]) ReadY */
                	        	/*  not BusY (I)                   */
#define GPIO_FlMem1Bsy	        	/*   Flash Memory 1 ([31:16]) Busy */ \
                	(GPIO_FL1RYnBY*0)
#define GPIO_FlMem1Rdy	        	/*   Flash Memory 1 ([31:16])      */ \
                	        	/*   Ready                         */ \
                	(GPIO_FL1RYnBY*1)
#define GPIO_CODECINT	GPIO_GPIO (6)	/*  CODEC INTerrupt (I)            */
#define GPIO_CODECNoInt	        	/*   CODEC No Interrupt            */ \
                	(GPIO_CODECINT*0)
#define GPIO_CODECInt	        	/*   CODEC Interrupt               */ \
                	(GPIO_CODECINT*1)
#define GPIO_UARTVALID	GPIO_GPIO (7)	/*  ser. port 3: UART receive      */
                	        	/*  signal VALID (I)               */
#define GPIO_UARTRcInv	        	/*   ser. port 3: UART Receive     */ \
                	        	/*   signal Invalid                */ \
                	(GPIO_UARTVALID*0)
#define GPIO_UARTRcVal	        	/*   ser. port 3: UART Receive     */ \
                	        	/*   signal Valid                  */ \
                	(GPIO_UARTVALID*1)
#define GPIO_UARTDSR	GPIO_GPIO (8)	/*  ser. port 3: UART Data Set     */
                	        	/*  Ready (I)                      */
#define GPIO_UARTDTR	GPIO_GPIO (9)	/*  ser. port 3: UART Data         */
                	        	/*  Terminal Ready (O)             */
#define GPIO_nLCDLED	GPIO_GPIO (14)	/*  not LCD back-light LED (O)     */
                	        	/*  [Itsy v1.6 & v1.7 only]        */
#define GPIO_LCDLEDEn	        	/*   LCD back-light LED Enable     */ \
                	        	/*   [Itsy v1.6 & v1.7 only]       */ \
                	(GPIO_nLCDLED*0)
#define GPIO_LCDLEDDis	        	/*   LCD back-light LED Disable    */ \
                	        	/*   [Itsy v1.6 & v1.7 only]       */ \
                	(GPIO_nLCDLED*1)
#define GPIO_DCEN	GPIO_GPIO (19)	/*  Daughter-Card ENable (O)       */
#define GPIO_DgtrCrdDis	        	/*   Daughter-Card Disable         */ \
                	(GPIO_DCEN*0)
#define GPIO_DgtrCrdEn	        	/*   Daughter-Card Enable          */ \
                	(GPIO_DCEN*1)
#define GPIO_LCDEN	GPIO_GPIO (20)	/*  LCD ENable (O)                 */
#define GPIO_LCDDis	(GPIO_LCDEN*0)	/*   LCD Disable                   */
#define GPIO_LCDEn	(GPIO_LCDEN*1)	/*   LCD Enable                    */
#define GPIO_AUXLCDEN	GPIO_GPIO (21)	/*  AUXiliary LCD controller       */
                	        	/*  ENable (O)                     */
#define GPIO_AuxLCDDis	        	/*   Auxiliary LCD controller      */ \
                	        	/*   Disable                       */ \
                	(GPIO_AUXLCDEN*0)
#define GPIO_AuxLCDEn	        	/*   Auxiliary LCD controller      */ \
                	        	/*   Enable                        */ \
                	(GPIO_AUXLCDEN*1)
#define GPIO_nIRDAEN	GPIO_GPIO (22)	/*  ser. port 2: not IRDA          */
                	        	/*  transceiver ENable (O)         */
#define GPIO_IrDAEn	        	/*   ser. port 2: IrDA transceiver */ \
                	        	/*   Enable                        */ \
                	(GPIO_nIRDAEN*0)
#define GPIO_IrDADis	        	/*   ser. port 2: IrDA transceiver */ \
                	        	/*   Disable                       */ \
                	(GPIO_nIRDAEN*1)
#define GPIO_nUARTFOFF	GPIO_GPIO (23)	/*  ser. port 3: not UART Force    */
                	        	/*  OFF (O)                        */
#define GPIO_UARTDis	        	/*   ser. port 3: UART Disable     */ \
                	(GPIO_nUARTFOFF*0)
#define GPIO_UARTEn	        	/*   ser. port 3: UART Enable      */ \
                	(GPIO_nUARTFOFF*1)
#define GPIO_UARTFON	GPIO_GPIO (24)	/*  ser. port 3: UART Force ON (O) */
#define GPIO_UARTAuto	        	/*   ser. port 3: UART             */ \
                	        	/*   AutoShutdown mode (when       */ \
                	        	/*   enabled)                      */ \
                	(GPIO_UARTFON*0)
#define GPIO_UARTOn	        	/*   ser. port 3: UART On (when    */ \
                	        	/*   enabled)                      */ \
                	(GPIO_UARTFON*1)

                	        	/* Itsy v2.x:                      */
#define GPIO_2nMAINPB	GPIO_nMAINPB	/*  not MAIN Push-Button (I)       */
#define GPIO_2MainPBOn	        	/*   Main Push-Button On           */ \
                	(GPIO_2nMAINPB*0)
#define GPIO_2MainPBOff	        	/*   Main Push-Button Off          */ \
                	(GPIO_2nMAINPB*1)
#define GPIO_2CODECINT	GPIO_CODECINT	/*  CODEC INTerrupt (I)            */
#define GPIO_2CODECNoInt        	/*   CODEC No Interrupt            */ \
                	(GPIO_2CODECINT*0)
#define GPIO_2CODECInt	        	/*   CODEC Interrupt               */ \
                	(GPIO_2CODECINT*1)
#define GPIO_2RNSX	GPIO_GPIO (7)	/*  Rock 'N' Scroll X-axis (I)     */
#define GPIO_2RNSY	GPIO_GPIO (8)	/*  Rock 'N' Scroll Y-axis (I)     */
#define GPIO_2BATTMON	GPIO_GPIO (9)	/*  BATTery MONitor (I/O)          */
#define GPIO_2UARTTXD	GPIO_UART_TXD	/*  ser. port 1: UART Transmit     */
                	        	/*  Data (O)                       */
#define GPIO_2UARTRXD	GPIO_UART_RXD	/*  ser. port 1: UART Receive      */
                	        	/*  Data (I)                       */
#define GPIO_2nCHGOFF	GPIO_GPIO (16)	/*  not battery CHarGer OFF (I/O)  */
#define GPIO_2ChgOff	        	/*   battery Charger Off           */ \
                	(GPIO_2nCHGOFF*0)
#define GPIO_2ChgOn	        	/*   battery Charger On            */ \
                	(GPIO_2nCHGOFF*1)
#define GPIO_2FSTCHG	GPIO_GPIO (17)	/*  FaST battery CHarGe (O)        */
#define GPIO_2SlowChg	        	/*   Slow battery Charge (100 mA)  */ \
                	(GPIO_2FSTCHG*0)
#define GPIO_2FastChg	        	/*   Fast battery Charge (500 mA)  */ \
                	(GPIO_2FSTCHG*1)
#define GPIO_2nDRAMEN	GPIO_GPIO (18)	/*  not DRAM ENable (O)            */
#define GPIO_2DRAMEn	        	/*   DRAM Enable                   */ \
                	(GPIO_2nDRAMEN*0)
#define GPIO_2DRAMDis	        	/*   DRAM Disable                  */ \
                	(GPIO_2nDRAMEN*1)
#define GPIO_2nLCDLED	GPIO_GPIO (19)	/*  not LCD back-light LED (O)     */
#define GPIO_2LCDLEDEn	        	/*   LCD back-light LED Enable     */ \
                	(GPIO_2nLCDLED*0)
#define GPIO_2LCDLEDDis	        	/*   LCD back-light LED Disable    */ \
                	(GPIO_2nLCDLED*1)
#define GPIO_2LCDEN	GPIO_LCDEN	/*  LCD ENable (O)                 */
#define GPIO_2LCDDis	(GPIO_2LCDEN*0)	/*   LCD Disable                   */
#define GPIO_2LCDEn	(GPIO_2LCDEN*1)	/*   LCD Enable                    */
#define GPIO_2AUXLCDEN	GPIO_AUXLCDEN	/*  AUXiliary LCD controller       */
                	        	/*  ENable (O)                     */
#define GPIO_2AuxLCDDis	        	/*   Auxiliary LCD controller      */ \
                	        	/*   Disable                       */ \
                	(GPIO_2AUXLCDEN*0)
#define GPIO_2AuxLCDEn	        	/*   Auxiliary LCD controller      */ \
                	        	/*   Enable                        */ \
                	(GPIO_2AUXLCDEN*1)
#define GPIO_2nIREN	GPIO_nIRDAEN	/*  ser. port 2: not Infra-Red     */
                	        	/*  transceiver ENable (O)         */
#define GPIO_2IREn	        	/*   ser. port 2: Infra-Red        */ \
                	        	/*   transceiver Enable            */ \
                	(GPIO_2nIREN*0)
#define GPIO_2IRDis	        	/*   ser. port 2: Infra-Red        */ \
                	        	/*   transceiver Disable           */ \
                	(GPIO_2nIREN*1)
#define GPIO_2nUARTFOFF	GPIO_nUARTFOFF	/*  ser. port 1 (GPIO [14:15]):    */
                	        	/*  not UART Force OFF (O)         */
#define GPIO_2UARTDis	        	/*   ser. port 1 (GPIO [14:15]):   */ \
                	        	/*   UART Disable                  */ \
                	(GPIO_2nUARTFOFF*0)
#define GPIO_2UARTAuto	        	/*   ser. port 1 (GPIO [14:15]):   */ \
                	        	/*   UART AutoShutdown Plus mode   */ \
                	(GPIO_2nUARTFOFF*1)
#define GPIO_2nUSBRDY	GPIO_GPIO (24)	/*  not USB ReaDY (O)              */
#define GPIO_2USBRdy	        	/*  USB Ready                      */ \
                	(GPIO_2nUSBRDY*0)
#define GPIO_2USBNotRdy	        	/*  USB Not Ready                  */ \
                	(GPIO_2nUSBRDY*1)
#define GPIO_2PWR15HI	GPIO_GPIO (25)	/*  core PoWeR 1.5 V HIgh (O)      */
#define GPIO_2Pwr1_23	        	/*   core Power = 1.23 V           */ \
                	(GPIO_2PWR15HI*0)
#define GPIO_2Pwr1_50	        	/*   core Power = 1.50 V           */ \
                	(GPIO_2PWR15HI*1)
#define GPIO_2FLRYnBY	GPIO_GPIO (26)	/*  FLash memory ReadY not         */
                	        	/*  BusY (I)                       */
#define GPIO_2FlMemBsy	        	/*   Flash Memory Busy             */ \
                	(GPIO_2FLRYnBY*0)
#define GPIO_2FlMemRdy	        	/*   Flash Memory Ready            */ \
                	(GPIO_2FLRYnBY*1)


/*
 * Interrupt Controller (IC) control registers
 *
 * Registers
 *    ICIP      	Interrupt Controller (IC) Interrupt ReQuest (IRQ)
 *              	Pending register (read).
 *    ICMR      	Interrupt Controller (IC) Mask Register (read/write).
 *    ICLR      	Interrupt Controller (IC) Level Register (read/write).
 *    ICFP      	Interrupt Controller (IC) Fast Interrupt reQuest
 *              	(FIQ) Pending register (read).
 *    ICPR      	Interrupt Controller (IC) Pending Register (read).
 */

                	        	/* Itsy v1.x:                      */
#define IC_MainPB	IC_GPIO (0)	/*  Main Push-Button               */
#define IC_Pwr  	IC_GPIO (2)	/*  Power OK/bad                   */
#define IC_FlMem0	IC_GPIO (4)	/*  Flash Memory 0 ([15:0])        */
                	        	/*  ready/busy                     */
#define IC_FlMem1	IC_GPIO (5)	/*  Flash Memory 1 ([31:16])       */
                	        	/*  ready/busy                     */
#define IC_CODECInt	IC_GPIO (6)	/*  CODEC Interrupt                */
#define IC_UARTRc	IC_GPIO (7)	/*  ser. port 3: UART Receive      */
                	        	/*  signal valid/invalid           */
#define IC_UARTDSR	IC_GPIO (8)	/*  ser. port 3: UART Data Set     */
                	        	/*  Ready                          */

                	        	/* Itsy v2.x:                      */
#define IC_2MainPB	IC_MainPB	/*  Main Push-Button               */
#define IC_2CODECInt	IC_CODECInt	/*  CODEC Interrupt                */
#define IC_2RnSX	IC_GPIO (7)	/*  Rock 'n' Scroll X-axis         */
#define IC_2RnSY	IC_GPIO (8)	/*  Rock 'n' Scroll Y-axis         */
#define IC_2BattMon	IC_GPIO (9)	/*  Battery Monitor                */
#define IC_2ChgOnOff	IC_GPIO11_27	/*  Charger On/Off                 */
#define IC_2FlMem	IC_GPIO11_27	/*  Flash Memory ready/busy        */


/*
 * Liquid Crystal Display (LCD) control registers
 *
 * Registers
 *    LCCR1     	Liquid Crystal Display (LCD) Control Register 1
 *              	(read/write).
 *    LCCR2     	Liquid Crystal Display (LCD) Control Register 2
 *              	(read/write).
 */

#define LCD_Hght	200     	/* LCD Height                      */
#define LCD_Wdth	320     	/* LCD Width                       */


#undef C
#undef Assembly

#endif /* !defined (ITSY) */
