/*
 * linux/drivers/video/skeletonfb.c -- Skeleton for a frame buffer device
 *
 *  Created 28 Dec 1997 by Geert Uytterhoeven
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/vmalloc.h>
#include <asm/io.h>

#include <asm/arch/dsc25def.h>

#include <video/fbcon.h>
#include <video/fbcon-cfb8.h>

#define OSD_BK_COLOR		0x00
#define OSD_TRANSPARENT_COLOR	0x00
#define OSD_MAX_WIDTH		1008	// size max

#define VENC_SCAN		1	// scan mode (0:interlace, 1:non-interlace)
#define VENC_CHROMA		0	// chroma low-pass filter (0:1.5MHz off, 1:3MHz off)
#define VENC_COMPOSIT_SYNC	1	// SYNC selection (0:Vd, 1:composit sync)

#define OSD_CLUT_RAM		1	// RAM-look-up table
#define OSD_FIELD_MODE		0	// 0:field 1:frame
#define OSD_FRAME_MODE		1	// 0:field 1:frame

#define OSD_BITMAP_1BIT_WIDTH	0x00	// 1 bit width for bitmap OSD window
#define OSD_BITMAP_2BIT_WIDTH	0x01	// 2 bits width for bitmap OSD window
#define OSD_BITMAP_4BIT_WIDTH	0x02	// 4 bits width for bitmap OSD window
#define OSD_BITMAP_8BIT_WIDTH	0x03	// 8 bits width for bitmap OSD window

#define OSD_VIDEO0_DISPLAY_ON   0x0001  // Video window 0 display ON 
#define OSD_VIDEO1_DISPLAY_ON   0x0100  // Video window 1 display ON 
#define OSD_WIN0_DISPLAY_ON     0x0001  // OSD window 0 display ON 
#define OSD_WIN1_DISPLAY_ON     0x0001  // OSD window 1 display ON 

struct dsc25fb_info {
    /*
     *  Choose _one_ of the two alternatives:
     *
     *    1. Use the generic frame buffer operations (fbgen_*).
     */
    struct fb_info_gen gen;

    /* Here starts the frame buffer device dependent part */
    /* You can use this to store e.g. the board number if you support */
    /* multiple boards */
};


struct dsc25fb_par {
	int	base_xpos;
	int	base_ypos;
	int	vidwin_xpos;
	int	vidwin_ypos;
	int 	vidwin_width;
	int	vidwin_height;
	int	osdwin_xpos;
	int	osdwin_ypos;
	int	osdwin_width;
	int	osdwin_height;

	int	osd_ntsc_pal;
	int	osd_comprgb;
	int	osd_scan;
	int	osd_chroma;
	int	osd_interp;
	int	osd_vblank;
	int	osd_hblank;
	int	osd_sysclk;
	int	osd_csync;
	int	osd_frame;
	int	osd_rgbblank;
	int	osd_hdpol;
	int	osd_setup;
	int	osd_zoom_x;
	int	osd_zoom_y;

	int	osd_active;
	int	osd_blend;
	int	osd_mix;
	int	osd_videoactive;
	int	osd_pnum;
	int	osd_length;
};


    /*
     *  If your driver supports multiple boards, you should make these arrays,
     *  or allocate them dynamically (using kmalloc()).
     */

static struct dsc25fb_info fb_info;
static struct dsc25fb_par current_par;
static int current_par_valid = 0;
static struct display disp;

static struct fb_var_screeninfo default_var = {
	xres:		320,
	yres:		240,
	xres_virtual:	320,
	yres_virtual:	240,
	xoffset:	0,
	yoffset:	0,
	bits_per_pixel:	8,
	grayscale:	0,
	red:		{ 0,8,0 },
	green:		{ 0,8,0 },
	blue:		{ 0,8,0 },
	nonstd:		0,
	activate:	FB_ACTIVATE_NOW,
	height:		-1,
	width:		-1,
	accel_flags:	0,
	pixclock:	10000, /* dummy value */
	left_margin:	27,
	upper_margin:	27,
	sync:		FB_SYNC_BROADCAST,
	vmode:		FB_VMODE_NONINTERLACED,
};

static struct fb_ops dsc25fb_ops;

int dsc25fb_init(void);
int dsc25fb_setup(char*);

static unsigned int clut_r[256];
static unsigned int clut_g[256];
static unsigned int clut_b[256];

static unsigned int saved_clut_r[256];
static unsigned int saved_clut_g[256];
static unsigned int saved_clut_b[256];

/* ------------------- chipset specific functions -------------------------- */


static void dsc25_detect(void)
{
	/* we do nothing here */
}

static int dsc25_encode_fix(struct fb_fix_screeninfo *fix, struct dsc25fb_par *par,
			  const struct fb_info *info)
{
    /*
     *  This function should fill in the 'fix' structure based on the values
     *  in the `par' structure.
     */

    //printk("dsc25_encode_fix\n");

    strcpy(fix->id, "DM310_OSD");
    fix->smem_start	= OSDMEM_OFFSET_ARM;
    fix->smem_len	= OSDMEM_SIZE;
    fix->type		= FB_TYPE_PACKED_PIXELS;
    fix->type_aux	= 0;
    fix->visual		= FB_VISUAL_PSEUDOCOLOR;
    fix->xpanstep	= 0;
    fix->ypanstep	= 0;
    fix->ywrapstep	= 0;
    fix->line_length	= par->osdwin_width;
    fix->mmio_start	= 0;
    fix->mmio_len	= 0;
    fix->accel		= FB_ACCEL_NONE;

    return 0;
}

static int dsc25_decode_var(struct fb_var_screeninfo *var, struct dsc25fb_par *par,
			  const struct fb_info *info)
{
    /*
     *  Get the video params out of 'var'. If a value doesn't fit, round it up,
     *  if it's too big, return -EINVAL.
     *
     *  Suggestion: Round up in the following order: bits_per_pixel, xres,
     *  yres, xres_virtual, yres_virtual, xoffset, yoffset, grayscale,
     *  bitfields, horizontal timing, vertical timing.
     */

    //printk("dsc25_decode_var\n");

    par->osd_ntsc_pal	= 1;    /* video mode, 0:NTSC, 1:PAL */
    par->osd_comprgb	= 0;	/* 0:composite video, 1:analog rgb, 2:digital rgb, 3:reserved */
    par->osd_rgbblank	= 1;    /* no output on RGB lines */

    /* base coordinates of all windows */
    par->base_xpos	 = 128;
    par->base_ypos	 = 22;
    
    /* coordinates of the movie window (relative to base coordinates) */
    par->vidwin_xpos	 = 0;
    par->vidwin_ypos	 = 0;
    par->vidwin_width	 = 704;
    par->vidwin_height	 = 576;

    /* coordinates of the OSD window (relative to base coordinates) */
    par->osdwin_xpos	= var->left_margin;
    par->osdwin_ypos	= var->upper_margin;
    par->osdwin_width	= var->xres;
    par->osdwin_height	= var->yres;

    if (var->vmode == FB_VMODE_INTERLACED)
    {
    	par->osd_scan	= 0;
	par->osd_frame	= OSD_FRAME_MODE;
    }
    else
    {
    	par->osd_scan	= 1;
	par->osd_frame	= OSD_FIELD_MODE;
    }
    
    par->osd_chroma	= 0;	/* chroma low-pass filter, 0:1.5MHz off, 1:3MHz off */
    par->osd_interp	= 0;	/* interpolation mode, 0:disable, 1:interpolate Y signal */
    par->osd_vblank	= 0;	/* internal vertical blanking, 0:enable, 1:disable */
    par->osd_hblank	= 0;	/* internal horizontal blanking, 0:enable, 1:disable */
    par->osd_sysclk	= 0;	/* system clock select, 0:27MHz, 1:28.6MHz */
    par->osd_csync	= 1;	/* SYNC selection, 0:Vd, 1:composite sync */

    par->osd_zoom_x	= 1;
    par->osd_zoom_y	= 0;
    par->osd_hdpol	= 0;
    par->osd_setup	= 0;

    par->osd_active	= 1;
    par->osd_blend	= 4;
    par->osd_mix	= 0;
    par->osd_videoactive= 0;
    par->osd_pnum	= OSD_MAX_WIDTH;
    par->osd_length	= var->xres;

    return 0;
}

static int dsc25_encode_var(struct fb_var_screeninfo *var, struct dsc25fb_par *par,
			  const struct fb_info *info)
{
    /*
     *  Fill the 'var' structure based on the values in 'par' and maybe other
     *  values read out of the hardware.
     */

    int real_xres;

    //printk("dsc25_encode_var\n");

    *var = default_var;
    real_xres = ((var->xres + 31) / 32) * 32;
    var->xres = var->xres_virtual = real_xres;
    //printk("par->osdwin_width: %d\n", par->osdwin_width);
//    par->osdwin_width = real_xres;

    return 0;
}

static void dsc25_get_par(struct dsc25fb_par *par, const struct fb_info *info)
{
    /*
     *  Fill the hardware's 'par' structure.
     */

    //printk("dsc25_get_par: valid=%i\n", current_par_valid);

    if (current_par_valid)
	*par = current_par;
}

static void dsc25_set_par(struct dsc25fb_par *par, const struct fb_info *info)
{
    /*
     *  Set the hardware according to 'par'.
     */

    u16 regval;
    u16 blend;
        
    //printk("dsc25_set_par\n");

    current_par = *par;
    current_par_valid = 1;

    VENC_VID01=	( par->osd_ntsc_pal	<< 15) |
		( par->osd_scan		<< 14) | 
		( par->osd_chroma	<< 13) |
		( 0			<< 12) | // Display 100% color bar 
		( par->osd_setup	<< 10) | 
		( 0	 		<<  6) | // Enable Red/Y signal
		( 1	 		<<  5) | // Enable Green/Comp signal
		( 0	 		<<  4) | // Enable Blue/Chroma signal
		( par->osd_rgbblank	<<  3) | // Enable Composite signal
		( 1			<<  2) | // Enable Video signal
		( 1 			     ) ; // Active video encoder
        
    VENC_VID02=	( par->osd_comprgb	<< 14) |
		( 1			<< 13) | // Input level not controlled
		( par->osd_csync 	<< 11) |
		( 1	 		<< 10) ; // Enable sync signal
    
    // automatic correction factor 
    OSD_MODE=	( 0			<< 11) | // 6/5 video window vertical expansion (PAL)
		( 0	  		<< 12) | // expand filter
		( 0	 		<< 10) | // 9/8 video window horizontal expansion
		( OSD_CLUT_RAM		<<  8) |
		( OSD_BK_COLOR		     ) ;
    
    OSD_VIDEOWIN_MODE = 	
		(par->osd_frame			<<  1) |
		(par->osd_videoactive 		     ) ;
    
    // Set 1 line offset to main video / 32  
    OSD_VIDEOWIN0_OFF = (par->osd_pnum*2)/32;

    // Set OSD standard position windows
    OSD_BASEP_X = par->base_xpos;
    OSD_BASEP_Y = par->base_ypos;
    
    // set XP,YP,XL,YL of VID window
    OSD_VIDEOWIN0_XP = par->vidwin_xpos;
    OSD_VIDEOWIN0_YP = par->vidwin_ypos;
    OSD_VIDEOWIN0_XL = par->vidwin_width;
    OSD_VIDEOWIN0_YL = par->vidwin_height;
    
    // set XP,YP,XL,YL of OSD window
    OSD_OSDWIN0_XP = par->osdwin_xpos;
    OSD_OSDWIN0_YP = par->osdwin_ypos;
    OSD_OSDWIN0_XL = par->osdwin_width*2;
    OSD_OSDWIN0_YL = par->osdwin_height;
    
    /* set frame buffer address for OSD window */        
    // Set address and 1 line offset to VIDMEM_OFFSET / 32  
    OSD_OSDWIN0_OFF = par->osd_length >> 5;
    OSD_OSDWIN_ADUP = ((OSDMEM_OFFSET_DSP >> 21) & 0x003f) << 8;
    OSD_OSDWIN0_ADL = (OSDMEM_OFFSET_DSP >> 5) & 0xffff;
    //printk("ADUP, ADL: %04x, %04x\n", OSD_OSDWIN_ADUP, OSD_OSDWIN0_ADL);
    
    // Show window 1 in full color (byte set)           
    OSD_BITMAP01_0 |= OSD_TRANSPARENT_COLOR;

    blend = par->osd_blend * 2;
    if (blend > 7) 
    	blend = 7;

    OSD_OSDWIN0_MODE =
    		( OSD_CLUT_RAM		<< 12 ) | 	// RAM-look-up table
		( par->osd_zoom_x	<< 10 ) |
		( par->osd_zoom_y	<<  8 ) |
		( OSD_BITMAP_8BIT_WIDTH	<<  6 ) |	// 8 bit bitmap width
		( blend     		<<  3 ) |	// blending 0 - 4
		( par->osd_mix		<<  2 ) | 	// 1 = transparent
		( par->osd_frame	<<  1 ) |	// 0 = field mode
		( par->osd_active             ) ;	// 1 = active
    //printk("MODE: %04x\n", OSD_OSDWIN0_MODE);
}

static int dsc25_getcolreg(unsigned regno, unsigned *red, unsigned *green,
			 unsigned *blue, unsigned *transp,
			 const struct fb_info *info)
{
    if (regno > 255)
	return 1;

    *red = clut_r[regno];
    *green = clut_g[regno];
    *blue = clut_b[regno];
    *transp = 0;

    return 0;
}

static int dsc25_setcolreg(unsigned regno, unsigned red, unsigned green,
			 unsigned blue, unsigned transp,
			 const struct fb_info *info)
{
    int Y, Cb, Cr;
    
    if (regno > 255)
	return 1;
    
    clut_r[regno] = red;
    clut_g[regno] = green;
    clut_b[regno] = blue;

    red >>= 8;
    green >>= 8;
    blue >>= 8;
    
    Y  = ((int)red *  257 + (int)green *  504 + (int)blue *   98) / 1000 + 0;
    Cb = ((int)red * -148 + (int)green * -291 + (int)blue *  439) / 1000 + 128;
    Cr = ((int)red *  439 + (int)green * -368 + (int)blue *  -71) / 1000 + 128;

    //while (OSD_CTRL_MISC & 0x0008);
    
    OSD_CLUTRAM_YCB = (Y << 8) | Cb;
    OSD_CLUTRAM_CR  = (Cr << 8) | regno;
    
    return 0;
}

static int dsc25_pan_display(struct fb_var_screeninfo *var,
			   struct dsc25fb_par *par, const struct fb_info *info)
{
    /*
     *  Pan (or wrap, depending on the `vmode' field) the display using the
     *  `xoffset' and `yoffset' fields of the `var' structure.
     *  If the values don't fit, return -EINVAL.
     */

    /* ... */
    return 0;
}

static int dsc25_blank(int blank_mode, const struct fb_info *info)
{
    /*
     *  Blank the screen if blank_mode != 0, else unblank. If blank == NULL
     *  then the caller blanks by setting the CLUT (Color Look Up Table) to all
     *  black. Return 0 if blanking succeeded, != 0 if un-/blanking failed due
     *  to e.g. a video mode which doesn't support it. Implements VESA suspend
     *  and powerdown modes on hardware that supports disabling hsync/vsync:
     *    blank_mode == 2: suspend vsync
     *    blank_mode == 3: suspend hsync
     *    blank_mode == 4: powerdown
     */

    /* ... */
    return 0;
}

static void dsc25_set_disp(const void *par, struct display *disp,
			 struct fb_info_gen *info)
{
    /*
     *  Fill in a pointer with the virtual address of the mapped frame buffer.
     *  Fill in a pointer to appropriate low level text console operations (and
     *  optionally a pointer to help data) for the video mode `par' of your
     *  video hardware. These can be generic software routines, or hardware
     *  accelerated routines specifically tailored for your hardware.
     *  If you don't have any appropriate operations, you must fill in a
     *  pointer to dummy operations, and there will be no text output.
     */

    //printk("dsc25_set_disp\n");
    disp->screen_base 	= ioremap(OSDMEM_OFFSET_ARM, OSDMEM_SIZE);
    disp->dispsw 	= &fbcon_cfb8;
}    


/* ------------ Interfaces to hardware functions ------------ */


struct fbgen_hwswitch dsc25_switch = {
    dsc25_detect, dsc25_encode_fix, dsc25_decode_var, dsc25_encode_var, dsc25_get_par,
    dsc25_set_par, dsc25_getcolreg, dsc25_setcolreg, dsc25_pan_display, dsc25_blank,
    dsc25_set_disp
};



/* ------------ Hardware Independent Functions ------------ */


    /*
     *  Initialization
     */

int __init dsc25fb_init(void)
{
    unsigned short *p;
    
    //printk("dm310fb: init\n");

    fb_info.gen.fbhw = &dsc25_switch;
    strcpy(fb_info.gen.info.modename, "PAL");
    fb_info.gen.info.changevar = NULL;
    fb_info.gen.info.node = -1;
    fb_info.gen.info.fbops = &dsc25fb_ops;
    fb_info.gen.info.disp = &disp;
    fb_info.gen.info.switch_con = &fbgen_switch;
    fb_info.gen.info.updatevar = &fbgen_update_var;
    fb_info.gen.info.blank = &fbgen_blank;
    fb_info.gen.info.flags = FBINFO_FLAG_DEFAULT;
    fb_info.gen.parsize = sizeof (struct dsc25fb_par);
    /* This should give a reasonable default video mode */
    fbgen_get_var(&disp.var, -1, &fb_info.gen.info);
    fbgen_do_set_var(&disp.var, 1, &fb_info.gen);
    fbgen_set_disp(-1, &fb_info.gen);
    fbgen_install_cmap(0, &fb_info.gen);
    if (register_framebuffer(&fb_info.gen.info) < 0)
	return -EINVAL;
    printk(KERN_INFO "fb%d: %s frame buffer device\n", GET_FB_IDX(fb_info.gen.info.node),
	   fb_info.gen.info.modename);

    /* uncomment this if your driver cannot be unloaded */
    /* MOD_INC_USE_COUNT; */
    
    return 0;
}


    /*
     *  Cleanup
     */

void dsc25fb_cleanup(void)
{
    /*
     *  If your driver supports multiple boards, you should unregister and
     *  clean up all instances.
     */

    unregister_framebuffer(&fb_info.gen.info);
    /* ... */
}


    /*
     *  Setup
     */

int __init dsc25fb_setup(char *options)
{
    /* Parse user speficied options (`video=dm310fb:') */
    return 0;
}


/* ------------------------------------------------------------------------- */


    /*
     *  Frame buffer operations
     */

static int dsc25fb_open(const struct fb_info *info, int user)
{
    int i;
    
    for (i=0; i<256; i++) {
    	saved_clut_r[i] = clut_r[i];
    	saved_clut_g[i] = clut_g[i];
    	saved_clut_b[i] = clut_b[i];
    }
    
    return 0;
}

static int dsc25fb_release(const struct fb_info *info, int user)
{
    int i;
    
    for (i=0; i<256; i++) {
    	dsc25_setcolreg(i, saved_clut_r[i], saved_clut_g[i], saved_clut_b[i],
				0, info);
    }
    
    return 0;
}

static int dsc25fb_mmap(struct fb_info* info, struct file* file, struct vm_area_struct* vma)
{
    return 0;    
}


    /*
     *  In most cases the `generic' routines (fbgen_*) should be satisfactory.
     *  However, you're free to fill in your own replacements.
     */

static struct fb_ops dsc25fb_ops = {
	owner:		THIS_MODULE,
	fb_open:	dsc25fb_open,    	/* only if you need it to do something */
	fb_release:	dsc25fb_release, 	/* only if you need it to do something */
/*	fb_mmap:	dsc25fb_mmap, */
	fb_get_fix:	fbgen_get_fix,
	fb_get_var:	fbgen_get_var,
	fb_set_var:	fbgen_set_var,
	fb_get_cmap:	fbgen_get_cmap,
	fb_set_cmap:	fbgen_set_cmap,
	fb_pan_display:	fbgen_pan_display,
/*	fb_ioctl:	dsc25fb_ioctl,   optional */
};


/* ------------------------------------------------------------------------- */


    /*
     *  Modularization
     */

#ifdef MODULE
MODULE_LICENSE("GPL");
int init_module(void)
{
    return dsc25fb_init();
}

void cleanup_module(void)
{
    dsc25fb_cleanup();
}
#endif /* MODULE */
