/*
  winspew - a EWMH testing tool.

  Copyright 2003 Matthew Allum

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
*/


#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#define MAX_WINS 32

#define WIN_OPTS_NO_TITLE           (1<<1)
#define WIN_OPTS_STATE_FULLSCREEN   (1<<2)
#define WIN_OPTS_STATE_MODAL        (1<<3)
#define WIN_OPTS_WM_PROTO_ACCEPT    (1<<5)
#define WIN_OPTS_WM_PROTO_HELP      (1<<6)
/* XXX possibles */
#define WIN_OPTS_UTF8_TITLE         (1<<7)
#define WIN_OPTS_NO_DECOR           (1<<8)
#define WIN_OPTS_WM_PROTO_CUSTOM    (1<<9)

enum { 				/* Window Atoms */

  WINDOW_TYPE_TOOLBAR = 1,
  WINDOW_TYPE_DOCK,
  WINDOW_TYPE_DIALOG,
  WINDOW_TYPE_SPLASH,
  WINDOW_TYPE_DESKTOP,
  WINDOW_TYPE,    

  WINDOW_STATE,
  WINDOW_STATE_FULLSCREEN,
  WINDOW_STATE_MODAL,
  
  _NET_CLOSE_WINDOW,
  
  _NET_WM_NAME,
  UTF8_STRING,

  _NET_SHOW_DESKTOP,
  _NET_WM_ICON,

  _MOTIF_WM_HINTS,

  WM_PROTOCOLS,
  WM_DELETE_WINDOW,
  _NET_WM_CONTEXT_HELP,
  _NET_WM_CONTEXT_ACCEPT,
  _NET_WM_CONTEXT_CUSTOM,

  WINDOW_TYPE_MESSAGE,
  WINDOW_TYPE_MESSAGE_TIMEOUT,
  
  _NET_SUPPORTED,

  ATOM_COUNT
};



static void paint(Window win, int width, int height);

static Display* dpy;	
static Window win_top_level[MAX_WINS], win_child[MAX_WINS], 
  win_current_top_level, win_root;
int win_top_level_idx = 0, win_child_idx = 0;
static int screen;
static Atom atoms[ATOM_COUNT];
static Bool WinIsFullscreen = False;
static Bool WantPattern     = True;

static struct { 
  char *key; int  flag; char *desc;
} Options_lookup[] = {
  { "no_title",       WIN_OPTS_NO_TITLE,
    "Request no titlebar / border only\n"},
  { "no_decor",       WIN_OPTS_NO_DECOR,
    "Request no window decorations\n"},
  { "fullscreen",     WIN_OPTS_STATE_FULLSCREEN ,
    "Request Fullscreen\n"},
  { "modal",          WIN_OPTS_STATE_MODAL,
    "Request dialog be modal\n"},    
  { "help_button",    WIN_OPTS_WM_PROTO_HELP,
  "Request help button in window title bar\n"},
  { "accept_button",  WIN_OPTS_WM_PROTO_ACCEPT,
  "Request 'OK' button in window title bar\n"},  
  { "custom_button",  WIN_OPTS_WM_PROTO_CUSTOM,
  "Request 'custom' button in window title bar\n"},  

  { NULL, 0, NULL }
};

static struct { 
  char *key; unsigned char *data; int len;
} UTF8_Names_lookup[] = {
  { "zh_CN.utf8", "\344\275\240\345\245\275\344\270\255\345\233\275!", 13 },
  { "ar_AE.utf8", "Arabic ( \357\273\262\357\272\221\357\272\256\357\273\213 )", 23 },
  { "mixed.utf8", "Hello from Arabic ( \357\273\262\357\272\221\357\272\256\357\273\213 ) and now from china : \344\275\240\345\245\275\344\270\255\345\233\275! clip this clip this clip this clip this clip this clip this clip this", 139 } ,
  /*   { "utf8-ar", "\357\273\262\357\272\221\357\272\256\357\273\213", 12 }, */

  { NULL, NULL, 0 }
};

void 
wm_check_supported_features(void)
{
  /* 
     XXX This functioniality here could be greatly enhances, though
         Im not sure how useful it would be. 
  */
  Atom type, *atoms_supported = NULL;
  int format;
  long bytes_after;
  long n_items;
  int result;
  unsigned char *atom_str_name;
  int i;
  Bool have_help_support = False, have_accept_support = False;

  result =  XGetWindowProperty (dpy, win_root, atoms[_NET_SUPPORTED],
				0, 1024L,
				False, XA_ATOM,
				&type, &format, &n_items,
				&bytes_after, 
				(unsigned char **)&atoms_supported);

  printf("winspew log: Checking wm features..\n");

  if (result != Success || atoms_supported == NULL)
    {
      if (atoms_supported) XFree (atoms_supported);
      printf("winspew log: *WARNING* looks like wm is not EWMH compliant\n"
	     "             Many features _will_not_ work!");
      return;
    }

  for (i=0; i<n_items; i++)
    {
      atom_str_name = XGetAtomName(dpy, atoms_supported[i]);
      printf("\t %s supported\n", atom_str_name); 
      if (atoms_supported[i] == atoms[_NET_WM_CONTEXT_HELP])
	have_help_support = True;
      if (atoms_supported[i] == atoms[_NET_WM_CONTEXT_ACCEPT])
	have_accept_support = True;
      if (atom_str_name) XFree(atom_str_name);
    }

  if (!have_help_support)
    printf("winspew log: *WARNING* This wm _does_not_ support help buttons\n"); 

  if (!have_accept_support)
    printf("winspew log: *WARNING* This wm _does_not_ support accept buttons\n"); 

  XFree (atoms_supported);
}

void
create_atoms(void)
{
  
  atoms[WINDOW_TYPE_TOOLBAR] 
    = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_TOOLBAR",False);
  atoms[WINDOW_TYPE_DOCK]
    = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_DOCK",False);
  atoms[WINDOW_TYPE_DIALOG]
    = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_DIALOG",False);
  atoms[WINDOW_TYPE_SPLASH]
    = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_SPLASH",False);
  atoms[WINDOW_TYPE_DESKTOP]
    = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_DESKTOP",False);
  atoms[WINDOW_TYPE]
    = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE", False);


  atoms[WINDOW_STATE]
    = XInternAtom(dpy, "_NET_WM_STATE",False);
  atoms[WINDOW_STATE_FULLSCREEN]
    = XInternAtom(dpy, "_NET_WM_STATE_FULLSCREEN",False);
  atoms[WINDOW_STATE_MODAL]
    = XInternAtom(dpy, "_NET_WM_STATE_MODAL",False);


  atoms[_NET_WM_NAME]
    = XInternAtom(dpy, "_NET_WM_NAME",False);
  atoms[UTF8_STRING]
    = XInternAtom(dpy, "UTF8_STRING",False);

  atoms[_NET_CLOSE_WINDOW]
    = XInternAtom(dpy, "_NET_CLOSE_WINDOW",False);
  atoms[_NET_SHOW_DESKTOP]
    = XInternAtom(dpy, "_NET_SHOWING_DESKTOP",False);
  atoms[_NET_WM_ICON]
    = XInternAtom(dpy, "_NET_WM_ICON", False);


   atoms[_MOTIF_WM_HINTS]
      = XInternAtom(dpy, "_MOTIF_WM_HINTS", False);

   atoms[WM_PROTOCOLS]
      = XInternAtom(dpy, "WM_PROTOCOLS", False);
   atoms[WM_DELETE_WINDOW]
      = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
   atoms[_NET_WM_CONTEXT_HELP]
      = XInternAtom(dpy, "_NET_WM_CONTEXT_HELP", False);
   atoms[_NET_WM_CONTEXT_ACCEPT]
      = XInternAtom(dpy, "_NET_WM_CONTEXT_ACCEPT", False);
   atoms[_NET_WM_CONTEXT_CUSTOM]
      = XInternAtom(dpy, "_NET_WM_CONTEXT_CUSTOM", False);

   atoms[_NET_SUPPORTED]
      = XInternAtom(dpy, "_NET_SUPPORTED", False);

   atoms[WINDOW_TYPE_MESSAGE]
     = XInternAtom(dpy, "_MB_WM_WINDOW_TYPE_MESSAGE", False);

   atoms[WINDOW_TYPE_MESSAGE_TIMEOUT]
     = XInternAtom(dpy, "_MB_WM_WINDOW_TYPE_MESSAGE_TIMEOUT", False);

}


void
usage(char *bin_name, char *error)
{
  int i = 0;
  if (!bin_name) {
    bin_name = "winspew";
  }
  printf("\n %s usage:\n"
	 "%s [ -display <X11 display> ] [ -nopaint] -top <options> [ -dialog <options> ] ...\n\n"
	 "-top, ( -t ) creates a top level window, <options> should consist of;\n\n"
         "   <title>,<window type>[,[geom],[extra opts], [message window timeout]]\n\n"
	 "-dialog, ( -c) creates a dialog for the previous top level, <options> are;\n\n"
	 "   <title>[,[geom],[extra opts],]\n\n"

	 "   'window type' is one of normal, dock, toolbar, dialog, splash, desktop, message\n"

         "   'geom' is a X11 window geometry definition, like 100x100+100+100\n"
	 "   'message window timeout' is the time ( seconds ) a message win is active\n"
	 "   'extra opts' is a ':' seperated list of the following flags;\n\n",
	 bin_name, bin_name);

  while( Options_lookup[i].key != NULL )
    {
      printf("       %s - %s", Options_lookup[i].key, Options_lookup[i].desc);
      i++;
    }  

  printf("\nNote, you may use as many instances of -top and -dialog as you like.\n"); 

  if (error) {
    printf("\nError: %s\n", error);
  }
  exit(1);
}

void
set_motif_no_title_decoration_hint(Window win, Bool border_only)
{

#define PROP_MOTIF_WM_HINTS_ELEMENTS    5
#define MWM_HINTS_DECORATIONS          (1L << 1)
#define MWM_DECOR_BORDER               (1L << 1)

  typedef struct
  {
    unsigned long       flags;
    unsigned long       functions;
    unsigned long       decorations;
    long                inputMode;
    unsigned long       status;
  } PropMotifWmHints;

  PropMotifWmHints *hints;

  hints = malloc(sizeof(PropMotifWmHints));
  memset(hints, 0, sizeof(PropMotifWmHints));

  hints->flags = MWM_HINTS_DECORATIONS;

  if (border_only)
    hints->decorations = MWM_DECOR_BORDER;
  else
    hints->decorations = 0;

  XChangeProperty(dpy, win, atoms[_MOTIF_WM_HINTS], 
		  XA_ATOM, 32, PropModeReplace, 
		  (unsigned char *)hints, PROP_MOTIF_WM_HINTS_ELEMENTS);

  free(hints);

  return;

}


int
parse_options(const char *options_spec)
{

  int i = 0, result_flags = 0;

  printf("parsing %s\n", options_spec);

  while( Options_lookup[i].key != NULL )
    {
      if (strstr(options_spec, Options_lookup[i].key) != NULL)
	{
	  printf("parsed %s\n", Options_lookup[i].key);
	  result_flags |= Options_lookup[i].flag;
	}
      i++;
    }

  return result_flags;
}

void
win_set_utf8_name(Window         win, 
		  unsigned char *data, 
		  int            data_len)
{
  XChangeProperty(dpy, win, atoms[_NET_WM_NAME], 
		  atoms[UTF8_STRING], 8, PropModeReplace, 
		  data, data_len);
}

void
win_set_standard_props(Window win, char *win_title, int x, int y, int h, int w)
{
  XSizeHints size_hints;

  size_hints.flags = PPosition | PSize | PMinSize;
  size_hints.x = x;
  size_hints.y = y;
  size_hints.width      =  h;
  size_hints.height     =  w;
  size_hints.min_width  =  w;
  size_hints.min_height =  h;
  
  XSetStandardProperties(dpy, win, 
			 win_title, win_title, None,
			 NULL, 0, &size_hints);
}

void
win_set_ewmh_type(Window win, Atom atom_type)
{
  printf("winspew log: Setting type\n");
  XChangeProperty(dpy, win, atoms[WINDOW_TYPE], XA_ATOM, 32, 
		  PropModeReplace, (unsigned char *) &atom_type, 1);
}

void
win_set_extened_options(Window win, int option_flags, int message_timeout)
{
  Atom wm_protocols[3];
  int wm_protocols_idx = 1;

  wm_protocols[0] = atoms[WM_DELETE_WINDOW];

  if (option_flags & WIN_OPTS_NO_TITLE)
    {
      printf("winspew log: Setting no decor hint\n"); 
      set_motif_no_title_decoration_hint(win, True);
    }

  if (option_flags & WIN_OPTS_NO_DECOR)
    {
      printf("winspew log: Setting no decor hint\n"); 
      set_motif_no_title_decoration_hint(win, False);
    }

  
  if (option_flags & WIN_OPTS_STATE_FULLSCREEN)
    {
      printf("winspew log: Setting fullscreen hint\n"); 
      XChangeProperty(dpy, win, 
		      atoms[WINDOW_STATE], XA_ATOM, 32, 
		      PropModeReplace, 
		      (unsigned char *) &atoms[WINDOW_STATE_FULLSCREEN], 1);
      WinIsFullscreen = True;
    }

  if (option_flags & WIN_OPTS_STATE_MODAL)
    {
      printf("winspew log: Setting modal hint\n"); 
      XChangeProperty(dpy, win, 
		      atoms[WINDOW_STATE], XA_ATOM, 32, 
		      PropModeReplace, 
		      (unsigned char *) &atoms[WINDOW_STATE_MODAL], 1);
    }

  if (message_timeout)
    {
      int data[1];

      printf("winspew log: Setting message timeout\n"); 

      data[0] = message_timeout;
      XChangeProperty(dpy, win, 
		      atoms[WINDOW_TYPE_MESSAGE_TIMEOUT], XA_CARDINAL, 32, 
		      PropModeReplace, 
		      (unsigned char *) &data[0], 1);
    }
  
  if (option_flags & WIN_OPTS_WM_PROTO_ACCEPT)
    {
      printf("winspew log: Setting accept protocol\n"); 
      wm_protocols[wm_protocols_idx] = atoms[_NET_WM_CONTEXT_ACCEPT];
      wm_protocols_idx++;
    }
  
  if (option_flags & WIN_OPTS_WM_PROTO_HELP)
    {
      printf("winspew log: Setting help protocol\n"); 
      wm_protocols[wm_protocols_idx] = atoms[_NET_WM_CONTEXT_HELP];
      wm_protocols_idx++;
    }

  if (option_flags & WIN_OPTS_WM_PROTO_CUSTOM)
    {
      printf("winspew log: Setting help protocol\n"); 
      wm_protocols[wm_protocols_idx] = atoms[_NET_WM_CONTEXT_CUSTOM];
      wm_protocols_idx++;
    }
  
  if (wm_protocols_idx)
    XSetWMProtocols(dpy, win, wm_protocols, wm_protocols_idx );

}

void
win_realize(Window win)
{
  XSelectInput(dpy, win,
	       KeyPressMask		
	       |KeyReleaseMask		
	       |ButtonPressMask
	       |ButtonReleaseMask	
	       |EnterWindowMask
	       |LeaveWindowMask
	       |PointerMotionMask	
	       |PointerMotionHintMask	
	       |Button1MotionMask	
	       |Button2MotionMask	
	       |Button3MotionMask	
	       |Button4MotionMask	
	       |Button5MotionMask	
	       |ButtonMotionMask	
	       |KeymapStateMask
	       |ExposureMask		
	       |VisibilityChangeMask	
	       |StructureNotifyMask	
	       |SubstructureNotifyMask	
	       |FocusChangeMask	
	       |PropertyChangeMask	
	       |ColormapChangeMask	
	       |OwnerGrabButtonMask	
	       ); 		/* XXX really need all this ? */
     
  XMapWindow(dpy, win);
}

void
win_destroy(Window win) 	
{
  Window win_tmp;
  int i;

  for (i=0; i<win_child_idx; i++) /* destroy any transient dialogs */
    {
      if (win_child[i] != None 
	  && XGetTransientForHint(dpy, win_child[i], &win_tmp)
	  && win_tmp == win)
	{
	  XDestroyWindow(dpy, win_child[i]);
	  win_child[i] = None;
	}
    }

  XDestroyWindow(dpy,win);

  /* Update window arrays */

  for (i=0; i<win_top_level_idx; i++)
    if (win_top_level[i] == win)
      {
	win_top_level[i] = None;
	return;
      }

  for (i=0; i<win_child_idx; i++)
    if (win_child[i] == win)
      win_child[i] = None;

}

void
create_top_level(int argc, char **argv, char *spec)
{
  char *str;
  const char delim[] = ",";
  char *tmp;
  int i;

  char *win_title;
  int win_type = 0, win_x = 0, win_y = 0, win_w = 100, win_h = 50;

  int option_flags    = 0;
  int message_timeout = 0;
  int geom_bitmask    = 0;

  struct {
    char *name;
    int id;
  } win_types[] = {
    {"normal",  0},
    {"dock",    WINDOW_TYPE_DOCK},
    {"dialog",  WINDOW_TYPE_DIALOG},
    {"splash",  WINDOW_TYPE_SPLASH},
    {"toolbar", WINDOW_TYPE_TOOLBAR},
    {"desktop", WINDOW_TYPE_DESKTOP},
    {"message", WINDOW_TYPE_MESSAGE},
    {NULL, 0}
  };

  if ((strchr(spec, delim[0]) != NULL))
  {
     str = strdup(spec);

     if ((win_title = strsep (&str, delim)) == NULL)
       usage(NULL, "top title missing");

     if ((tmp = strsep (&str, delim)) == NULL)
       usage(NULL, "top type missing");
     
     if (atoi(tmp) > 0)
       win_type = atoi(tmp);
     else
       {
	 Bool found = False;
	 i = 0;
	 while(win_types[i].name != NULL)
	   {
	     if (!strcmp(tmp, win_types[i].name))
	       {
		 win_type =  win_types[i].id;
		 found = True;
		 break;
	       }
	     i++;
	   }
       }
     
     /* Rest are optional */

     if ((tmp = strsep (&str, delim)) != NULL)
       {
	 geom_bitmask = XParseGeometry(tmp, &win_x, &win_y, &win_w, &win_h);

	 if (!win_w) win_w = 100;
	 if (!win_h) win_h = 100;

	 /* XXX see code for dialogs
	 if (!(geom_bitmask & XNegative))
	   {
	     win_x = (-1 * DisplayWidth(dpy, screen)) + win_x;
	   }

	 if (!(geom_bitmask & YNegative)) 
	   {
	     win_y = (-1 * DisplayHeight(dpy, screen)) + win_y;
	   }
	 */
       }

     if ((tmp = strsep (&str, delim)) != NULL)
       {
	 option_flags = parse_options(tmp);
       }

     if ((tmp = strsep (&str, delim)) != NULL)
       {
	 message_timeout = atoi(tmp);
       }

     free(str);

     /* Now create actual window */

     printf("winspew log: creating top level x:%i, y:%i, w:%i, h:%i\n",
	    win_x, win_y, win_w, win_h);

     win_top_level[win_top_level_idx] 
       = XCreateSimpleWindow(dpy, win_root, win_x, win_y,
			     win_w, win_h, 0,
			     BlackPixel(dpy, screen),
			     WhitePixel(dpy, screen));

     printf("winspew log: window id is %li\n", 
	    win_top_level[win_top_level_idx]);

     win_current_top_level = win_top_level[win_top_level_idx];

     win_set_standard_props(win_top_level[win_top_level_idx], win_title,
			    win_x, win_y, win_h, win_w);

     if (win_type) 		/* something other than a normal window */
       win_set_ewmh_type(win_top_level[win_top_level_idx], atoms[win_type]);

     win_set_extened_options(win_top_level[win_top_level_idx], 
			     option_flags, message_timeout);

     /* Set utf8 name from lookup */
     i = 0;
     while( UTF8_Names_lookup[i].key != NULL )
       {
	 if (!strcmp(win_title, UTF8_Names_lookup[i].key))
	   {
	     win_set_utf8_name(win_top_level[win_top_level_idx],
			       UTF8_Names_lookup[i].data, 
			       UTF8_Names_lookup[i].len);
	   }
	 i++;
       }

     win_realize(win_top_level[win_top_level_idx]);


      win_top_level_idx++;
     
  } else {
    /* Failed */
    usage("winspew", "top options missing");
  }
}

void 				/* XXX this should be merged into above */
create_child_dialog(int argc, char **argv, char *spec)
{
  char *str;
  const char delim[] = ",";
  char *tmp;

  char *win_title;
  int win_x = 0, win_y = 0, win_w = 100, win_h = 50;
  int option_flags = 0, geom_bitmask = 0;

  if (spec == NULL) usage("winspew", "child options missing");

  if ((strchr(spec, delim[0]) != NULL))
  {
     str = strdup(spec);

     if ((win_title = strsep (&str, delim)) == NULL)
       usage(NULL, "child title missing");

     /* Rest are optional */

     if ((tmp = strsep (&str, delim)) != NULL)
       {
	 geom_bitmask = XParseGeometry(tmp, &win_x, &win_y, &win_w, &win_h);

	 if (!win_w) win_w = 100;
	 if (!win_h) win_h = 100;

	 /* XXX we need to do gravitys for the below to work correctly. 
	 if (!(geom_bitmask & XNegative))
	   {
	     win_x = (-1 * DisplayWidth(dpy, screen)) - win_x - win_w;
	   }

	 if (!(geom_bitmask & YNegative)) 
	   {
	     win_y = (-1 * DisplayHeight(dpy, screen)) - win_y - win_h;
	   }
	 */
       }

     if ((tmp = strsep (&str, delim)) != NULL)
       {
	 option_flags = parse_options(tmp);
       }

     free(str);
  } 
  else win_title = spec;

  /* Now create actual window */

  printf("winspew log: creating dialog x:%i, y:%i, w:%i, h:%i\n",
	 win_x, win_y, win_w, win_h);

  win_child[win_child_idx] 
    = XCreateSimpleWindow(dpy, win_root, win_x, win_y,
			  win_w, win_h, 0,
			  BlackPixel(dpy, screen),
			  WhitePixel(dpy, screen));

  printf("winspew log: window id is %li\n", 
	 win_child[win_child_idx]);

  XSetTransientForHint(dpy, win_child[win_child_idx], 
		       win_current_top_level);

  win_set_standard_props(win_child[win_child_idx], win_title,
			 win_x, win_y, win_h, win_w);

  win_set_extened_options(win_child[win_child_idx], option_flags, 0);
  
  win_realize(win_child[win_child_idx]);

  win_child_idx++;

}

static void  /* Fill the window with a simple pattern  */
paint(Window win, int width, int height)
{
  XWindowAttributes win_attr;
  XGCValues gc_values;
  GC gc;
  int x, y;
  Pixmap pxm_backing;

  XGetWindowAttributes(dpy, win, &win_attr);

  gc_values.graphics_exposures = False;
  gc_values.function           = GXcopy;
  gc_values.background         = WhitePixel(dpy, DefaultScreen(dpy));
  gc = XCreateGC(dpy, win, GCGraphicsExposures|GCFunction|GCForeground, 
		 &gc_values);

  pxm_backing = XCreatePixmap(dpy, win, width, height, 
			      win_attr.depth);

  XSetForeground(dpy, gc, WhitePixel(dpy, DefaultScreen(dpy)));
  XFillRectangle(dpy, pxm_backing, gc, 0, 0, width, height);

  XSetForeground(dpy, gc, BlackPixel(dpy, DefaultScreen(dpy)));

  /* Quick hack for a window pattern, using window ID  */
  if (win % 2)
    {
      for (y = 0; y < height; y += win % 12)
	XDrawLine(dpy, pxm_backing, gc, 0, y, width, y);
    } else {
      for (x = 0; x < width; x += win % 12)
	    XDrawLine(dpy, pxm_backing, gc, x, 0, x, height);
    }
  
  XSetWindowBackgroundPixmap(dpy, win, pxm_backing);
   

  XClearWindow(dpy, win);

  XFreePixmap(dpy, pxm_backing);
  XFreeGC(dpy, gc);
} 

void
toggle_ewmh_fullscreen (Window win_to_toggle)
{

#define _NET_WM_STATE_TOGGLE        2    /* toggle property  */

  XEvent ev;
  Atom atom_win_state, atom_win_state_fullscreen;

  atom_win_state = XInternAtom(dpy, "_NET_WM_STATE",False);
  atom_win_state_fullscreen = XInternAtom(dpy,
                                          "_NET_WM_STATE_FULLSCREEN",False);

  printf("winspew-log: attempting toggle on %li\n", win_to_toggle);

  memset(&ev, 0, sizeof(ev));
  ev.xclient.type = ClientMessage;
  ev.xclient.window = win_to_toggle;
  ev.xclient.message_type = atom_win_state;
  ev.xclient.format = 32;
  ev.xclient.data.l[1] = atom_win_state_fullscreen;
  ev.xclient.data.l[0] = _NET_WM_STATE_TOGGLE;

  XSendEvent(dpy, win_root, False, SubstructureRedirectMask , &ev);

  XSync(dpy, False);
}


int 
main(int argc, char **argv)
{
  int i, j = 1;
  char *dpy_name = NULL;
  XEvent xevent;

  for (i = 1; i < argc; i++) {
    if (!strcmp ("-display", argv[i]) || !strcmp ("-d", argv[i])) {
      if (++i>=argc) usage (argv[0], "display missing");
      dpy_name = argv[i++];
      j++;
      continue;
    }
    if (!strcmp ("--nopattern", argv[i]) || !strcmp ("-np", argv[i])) 
      {
	j++;
	WantPattern = False;
	continue;
      }
  }

  if ((argc - j) == 0) 
    usage(argv[0], "no options");

  if ((dpy = XOpenDisplay(dpy_name)) == NULL) {
    fprintf(stderr, "Cannot connect to X server on display %s.",
	    dpy_name);
    exit(1);
  }
  
  screen   = DefaultScreen(dpy);
  win_root = DefaultRootWindow(dpy);

  create_atoms();

  wm_check_supported_features();

  win_current_top_level = win_root;

  for (i = j; i < argc; i++) {
    if (!strcmp ("-top", argv[i]) || !strcmp ("-t", argv[i])) {
      if (++i>=argc) usage (argv[0], "no -top options");
      create_top_level(argc, argv, argv[i]);
      continue;
    }
    if (!strcmp ("-dialog", argv[i]) || !strcmp ("-c", argv[i])) {
      if (++i>=argc) usage (argv[0], "no -child options");
      create_child_dialog(argc, argv, argv[i]);
      continue;
    }
    usage(argv[0], argv[i]);
  }

  while(1)
    {
      char *tmp;
      fflush(stdout);
      XNextEvent(dpy, &xevent);
      switch (xevent.type) 
	{
	case ButtonPress:
	  printf("winspew log: %li, Button press event\n", 
		 xevent.xbutton.window);
	  if (WinIsFullscreen) 
	    toggle_ewmh_fullscreen (xevent.xbutton.window);
	  break;
	case Expose:
	  printf("winspew log: %li, Expose event\n", 
		 xevent.xexpose.window);
	  /*
	  paint(xevent.xexpose.window, xevent.xexpose.width, 
		xevent.xexpose.height);
	  */
	  break;
	case MapNotify:
	  printf("winspew log: %li, MapNotify event\n", 
		 xevent.xmap.window);
	  break;
	case UnmapNotify:
	  printf("winspew log: %li, UnmapNotify event\n", 
		 xevent.xunmap.window);
	  break;
	case ReparentNotify:
	  printf("winspew log: %li, ReparentNotify\n", 
		 xevent.xreparent.window);
	  break;
	  
	case ConfigureNotify:
	  printf("winspew log: %li, ConfigureNotify event\n", 
		 xevent.xconfigure.window);
	  /* XXX compress configure notifys ? */
	  if (WantPattern)
	    paint(xevent.xconfigure.window, xevent.xconfigure.width, 
		  xevent.xconfigure.height);
	  break;
	case ClientMessage:
	  printf("winspew log: %li, ClientMessage event\n", 
		 xevent.xclient.window);
	  if ((xevent.xclient.message_type == atoms[WM_PROTOCOLS])
	      && (xevent.xclient.data.l[0] == atoms[WM_DELETE_WINDOW])) 
	    {
	      printf("\tis WM_DELETE, deleting...\n");
	      win_destroy(xevent.xclient.window);
	    }
	  break;
	case KeyPress:
	  break;
	case FocusIn:
	  printf("winspew log: %li, FocusIn event\n", 
		 xevent.xfocus.window);
	  break;
	case FocusOut:
	  printf("winspew log: %li, FocusOut event\n", 
		 xevent.xfocus.window);
	  break;
	case PropertyNotify:
	  tmp = XGetAtomName(dpy, xevent.xproperty.atom);
	  printf("winspew log: %li, PropertyNotify event. Atom '%s' ( ID: %li )\n", 
		 xevent.xproperty.window, tmp, xevent.xproperty.atom);
	  if (tmp) XFree(tmp);
	  break;
	default:
	  break;
	}
    }
  
}
