/*
	CONTROLM.c

	Copyright (C) 2005 Paul C. Pratt

	You can redistribute this file and/or modify it under the terms
	of version 2 of the GNU General Public License as published by
	the Free Software Foundation.  You should have received a copy
	of the license along with this file; see the file COPYING.

	This file is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	license for more details.
*/

/*
	CONTROL Mode
*/

#ifndef AllFiles
#include "SYSDEPNS.h"
#include "MYOSGLUE.h"
#include "GLOBGLUE.h"
#include "PROGMAIN.h"
#include "ADDRSPAC.h"
#endif

#if UseControlKeys

#include "CONTROLM.h"

GLOBALVAR int CurControlMode = 0;

LOCALVAR int ControlMessage = 0;

enum {
	kCntrlModeOff,
	kCntrlModeBase,
	kCntrlModeConfirmReset,
	kCntrlModeConfirmInterrupt,
	kCntrlModeConfirmQuit,
	kCntrlModeSpeedControl,

	kNumCntrlModes
};

enum {
	kCntrlMsgBaseStart,
	kCntrlMsgMagnify,
	kCntrlMsgFullScreen,
	kCntrlMsgConfirmResetStart,
	kCntrlMsgHaveReset,
	kCntrlMsgResetCancelled,
	kCntrlMsgConfirmInterruptStart,
	kCntrlMsgHaveInterrupted,
	kCntrlMsgInterruptCancelled,
	kCntrlMsgConfirmQuitStart,
	kCntrlMsgQuitCancelled,
	kCntrlMsgEmCntrl,
	kCntrlMsgSpeedControlStart,
	kCntrlMsgNewSpeed,
	kCntrlMsgNewRunInBack,
	kCntrlMsgAbout,
	kCntrlMsgHelp,

	kNumCntrlMsgs
};

GLOBALPROC DoEnterControlMode(void)
{
	CurControlMode = kCntrlModeBase;
	ControlMessage = kCntrlMsgBaseStart;
}

GLOBALPROC DoLeaveControlMode(void)
{
	CurControlMode = kCntrlModeOff;
}

LOCALPROC SetSpeedValue(ui3b i)
{
	SpeedStopped = falseblnr;
	SpeedLimit = trueblnr;
	SpeedValue = i;
	CurControlMode = kCntrlModeBase;
	ControlMessage = kCntrlMsgNewSpeed;
}

GLOBALPROC DoControlModeKey(int key)
{
	switch (CurControlMode) {
		case kCntrlModeBase:
			switch (key) {
				case MKC_K:
					ControlKeyPressed = ! ControlKeyPressed;
					ControlMessage = kCntrlMsgEmCntrl;
					break;
				case MKC_S:
					CurControlMode = kCntrlModeSpeedControl;
					ControlMessage = kCntrlMsgSpeedControlStart;
					break;
				case MKC_I:
					CurControlMode = kCntrlModeConfirmInterrupt;
					ControlMessage = kCntrlMsgConfirmInterruptStart;
					break;
				case MKC_R:
					if (! AnyDiskInserted()) {
						DoMacReset();
						ControlMessage = kCntrlMsgHaveReset;
					} else {
						CurControlMode = kCntrlModeConfirmReset;
						ControlMessage = kCntrlMsgConfirmResetStart;
					}
					break;
				case MKC_Q:
					if (! AnyDiskInserted()) {
						ForceMacOff = trueblnr;
					} else {
						CurControlMode = kCntrlModeConfirmQuit;
						ControlMessage = kCntrlMsgConfirmQuitStart;
					}
					break;
				case MKC_A:
					ControlMessage = kCntrlMsgAbout;
					break;
				case MKC_H:
					ControlMessage = kCntrlMsgHelp;
					break;
				case MKC_O:
					RequestInsertDisk = trueblnr;
					break;
#if EnableMagnify
				case MKC_M:
					ToggleWantMagnify();
					ControlMessage = kCntrlMsgMagnify;
					break;
#endif
#if EnableFullScreen
				case MKC_F:
					ToggleWantFullScreen();
					ControlMessage = kCntrlMsgFullScreen;
					break;
#endif
			}
			break;
		case kCntrlModeConfirmReset:
			switch (key) {
				case MKC_Y:
					DoMacReset();
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgHaveReset;
					break;
				case MKC_N:
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgResetCancelled;
					break;
			}
			break;
		case kCntrlModeConfirmInterrupt:
			switch (key) {
				case MKC_Y:
					SetInterruptButton(trueblnr);
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgHaveInterrupted;
					break;
				case MKC_N:
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgInterruptCancelled;
					break;
			}
			break;
		case kCntrlModeConfirmQuit:
			switch (key) {
				case MKC_Y:
					ForceMacOff = trueblnr;
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgBaseStart;
						/* shouldn't see this message since quitting */
					break;
				case MKC_N:
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgQuitCancelled;
					break;
			}
			break;
		case kCntrlModeSpeedControl:
			switch (key) {
				case MKC_E:
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgBaseStart;
					break;
				case MKC_B:
					RunInBackground = ! RunInBackground;
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgNewRunInBack;
					break;
				case MKC_S:
					SpeedStopped = trueblnr;
					SpeedLimit = falseblnr;
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgNewSpeed;
					break;
				case MKC_Z:
					SetSpeedValue(0);
					break;
				case MKC_1:
					SetSpeedValue(1);
					break;
				case MKC_2:
					SetSpeedValue(2);
					break;
				case MKC_3:
					SetSpeedValue(3);
					break;
				case MKC_4:
					SetSpeedValue(4);
					break;
				case MKC_5:
					SetSpeedValue(5);
					break;
				case MKC_A:
					SpeedStopped = falseblnr;
					SpeedLimit = falseblnr;
					CurControlMode = kCntrlModeBase;
					ControlMessage = kCntrlMsgNewSpeed;
					break;
			}
			break;
	}
}

enum {
	kCellUpperLeft,
	kCellUpperMiddle,
	kCellUpperRight,
	kCellMiddleLeft,
	kCellSpace,
	kCellMiddleRight,
	kCellLowerLeft,
	kCellLowerMiddle,
	kCellLowerRight,
	kCellGraySep,
	kCellIcon00,
	kCellIcon01,
	kCellIcon02,
	kCellIcon03,
	kCellIcon10,
	kCellIcon11,
	kCellIcon12,
	kCellIcon13,
	kCellIcon20,
	kCellIcon21,
	kCellIcon22,
	kCellIcon23,

	kNumCells
};

LOCALVAR const char CellMap[] =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!&'(),-./:;?";

LOCALVAR const ui3b CellData[] = {
	0xFF, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, /* UpperLeft */
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
	0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* UpperMiddle */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, /* UpperRight */
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, /* MiddleLeft */
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* Space */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, /* MiddleRight */
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, /* LowerLeft */
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* LowerMiddle */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, /* LowerRight */
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, /* kCellGraySep */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* kCellIcon00 */
	0x00, 0x07, 0x08, 0x08, 0x08, 0x09, 0x09, 0x09,
	0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* kCellIcon01 */
	0x00, 0xFF, 0x00, 0x00, 0xFF, 0x00, 0x18, 0x18,
	0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* kCellIcon02 */
	0x00, 0xFF, 0x00, 0x00, 0xFF, 0x00, 0x18, 0x30,
	0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* kCellIcon03 */
	0x00, 0xE0, 0x10, 0x10, 0x10, 0x90, 0x90, 0x90,
	0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, /* kCellIcon10 */
	0x09, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	0x18, 0x18, 0x19, 0x1B, 0x1E, 0x1C, 0x18, 0x10, /* kCellIcon11 */
	0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x60, 0xC0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, /* kCellIcon12 */
	0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00,
	0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, /* kCellIcon13 */
	0x90, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
	0x08, 0x08, 0x08, 0x07, 0x04, 0x04, 0x04, 0x07, /* kCellIcon20 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00, 0xFF, /* kCellIcon21 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00, 0xFF, /* kCellIcon22 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x10, 0x10, 0xE0, 0x20, 0x20, 0x20, 0xE0, /* kCellIcon23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x42, 0x7E, /* 'A' */
	0x42, 0x42, 0x42, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7C, 0x42, 0x42, 0x7C, 0x42, /* 'B' */
	0x42, 0x42, 0x42, 0x7C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x40, 0x40, /* 'C' */
	0x40, 0x40, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7C, 0x42, 0x42, 0x42, 0x42, /* 'D' */
	0x42, 0x42, 0x42, 0x7C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7E, 0x40, 0x40, 0x40, 0x7C, /* 'E' */
	0x40, 0x40, 0x40, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7E, 0x40, 0x40, 0x40, 0x7C, /* 'F' */
	0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x40, 0x40, 0x4E, /* 'G' */
	0x42, 0x42, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x42, 0x42, 0x42, 0x42, 0x7E, /* 'H' */
	0x42, 0x42, 0x42, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, /* 'I' */
	0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x02, /* 'J' */
	0x02, 0x42, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x42, 0x44, 0x48, 0x50, 0x60, /* 'K' */
	0x50, 0x48, 0x44, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, /* 'L' */
	0x40, 0x40, 0x40, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0xC6, 0xAA, 0x92, 0x82, /* 'M' */
	0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x42, 0x42, 0x62, 0x52, 0x4A, /* 'N' */
	0x46, 0x42, 0x42, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x42, 0x42, /* 'O' */
	0x42, 0x42, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7C, 0x42, 0x42, 0x42, 0x7C, /* 'P' */
	0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x42, 0x42, /* 'Q' */
	0x42, 0x42, 0x42, 0x3C, 0x10, 0x0C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7C, 0x42, 0x42, 0x42, 0x7C, /* 'R' */
	0x42, 0x42, 0x42, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x40, 0x40, 0x3C, /* 'S' */
	0x02, 0x02, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7F, 0x08, 0x08, 0x08, 0x08, /* 'T' */
	0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x42, 0x42, 0x42, 0x42, 0x42, /* 'U' */
	0x42, 0x42, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x42, 0x42, 0x42, 0x42, 0x44, /* 'V' */
	0x48, 0x50, 0x60, 0x40, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x42, 0x42, 0x42, 0x42, 0x42, /* 'W' */
	0x42, 0x5A, 0x66, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x42, 0x42, 0x42, 0x24, 0x18, /* 'X' */
	0x24, 0x42, 0x42, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x14, /* 'Y' */
	0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7E, 0x02, 0x04, 0x08, 0x10, /* 'Z' */
	0x20, 0x40, 0x40, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x42, /* 'a' */
	0x42, 0x42, 0x46, 0x3A, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x7C, 0x42, /* 'b' */
	0x42, 0x42, 0x42, 0x7C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x42, /* 'c' */
	0x40, 0x40, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x02, 0x3E, 0x42, /* 'd' */
	0x42, 0x42, 0x42, 0x3E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x42, /* 'e' */
	0x7E, 0x40, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x0E, 0x10, 0x10, 0x3C, 0x10, /* 'f' */
	0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x42, /* 'g' */
	0x42, 0x42, 0x42, 0x3E, 0x02, 0x42, 0x3C, 0x00,
	0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x7C, 0x42, /* 'h' */
	0x42, 0x42, 0x42, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x10, 0x00, 0x10, 0x10, /* 'i' */
	0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x00, 0x08, 0x08, /* 'j' */
	0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x44, 0x48, /* 'k' */
	0x70, 0x48, 0x44, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, /* 'l' */
	0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0x92, /* 'm' */
	0x92, 0x92, 0x92, 0x92, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x42, /* 'n' */
	0x42, 0x42, 0x42, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x42, /* 'o' */
	0x42, 0x42, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x42, /* 'p' */
	0x42, 0x42, 0x42, 0x7C, 0x40, 0x40, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x42, /* 'q' */
	0x42, 0x42, 0x42, 0x3E, 0x02, 0x02, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5C, 0x62, /* 'r' */
	0x42, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x42, /* 's' */
	0x3C, 0x02, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x20, 0x20, 0x78, 0x20, /* 't' */
	0x20, 0x20, 0x20, 0x1C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x42, /* 'u' */
	0x42, 0x42, 0x46, 0x3A, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x44, /* 'v' */
	0x48, 0x50, 0x60, 0x40, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x92, 0x92, /* 'w' */
	0x92, 0x92, 0x92, 0x6C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x24, /* 'x' */
	0x18, 0x18, 0x24, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x42, /* 'y' */
	0x42, 0x42, 0x42, 0x3E, 0x02, 0x42, 0x3C, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0x04, /* 'z' */
	0x08, 0x10, 0x20, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x42, 0x42, /* '0' */
	0x42, 0x42, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x18, 0x08, 0x08, 0x08, /* '1' */
	0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x02, 0x02, 0x04, /* '2' */
	0x08, 0x10, 0x20, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x02, 0x0C, 0x02, /* '3' */
	0x02, 0x02, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x0C, 0x14, 0x24, 0x7E, /* '4' */
	0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7E, 0x40, 0x40, 0x7C, 0x02, /* '5' */
	0x02, 0x02, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x20, 0x40, 0x7C, 0x42, /* '6' */
	0x42, 0x42, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7E, 0x02, 0x02, 0x04, 0x08, /* '7' */
	0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x3C, 0x42, /* '8' */
	0x42, 0x42, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x42, 0x42, /* '9' */
	0x3E, 0x02, 0x04, 0x38, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, /* '!' */
	0x08, 0x00, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x30, 0x48, 0x48, 0x50, 0x20, /* '&' */
	0x50, 0x4A, 0x44, 0x3A, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x00, 0x00, /* ''' */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x08, 0x08, 0x10, 0x10, 0x10, /* '(' */
	0x10, 0x10, 0x08, 0x08, 0x04, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x20, 0x10, 0x10, 0x08, 0x08, 0x08, /* ')' */
	0x08, 0x08, 0x10, 0x10, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* ',' */
	0x00, 0x00, 0x08, 0x08, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* '-' */
	0x3C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* '.' */
	0x00, 0x00, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x04, 0x04, 0x08, 0x08, /* '/' */
	0x10, 0x10, 0x20, 0x20, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, /* ':' */
	0x00, 0x00, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, /* ';' */
	0x00, 0x00, 0x08, 0x08, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x44, 0x04, 0x08, 0x10, /* '?' */
	0x10, 0x00, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00,

	0x00
};

LOCALPROC DrawCell(unsigned int h, unsigned int v, int x)
{
#if 1
	/* safety check */
	if ((h < vMacScreenByteWidth - 2) && (v < (vMacScreenHeight / 16 - 1)))
#endif
	{
		int i;
		ui3p p = ((ui3p)CntrlDisplayBuff) + (h + 1) + (v * 16 + 11) * vMacScreenByteWidth;
		ui3p p0 = ((ui3p)CellData) + 16 * x;

		for (i = 16; --i >= 0; ) {
			*p = *p0;
			p += vMacScreenByteWidth;
			p0 ++;
		}
	}
}

LOCALFUNC int CountNewLines(char *s)
{
	char pc;
	int n = 0;
	char *p = s;

	while ((pc = *p++) != 0) {
		if (pc == '\n') {
			n++;
		}
	}
	return n;
}

#ifndef kEmuName
#if CurEmu == kEmu128K
#define kEmuName "128K"
#elif CurEmu == kEmu512K
#define kEmuName "512K"
#elif CurEmu == kEmu512Ke
#define kEmuName "512Ke"
#elif CurEmu == kEmuPlus1M
#define kEmuName "Plus 1M"
#elif CurEmu == kEmuPlus2M
#define kEmuName "Plus 2M"
#elif CurEmu == kEmuPlus2_5M
#define kEmuName "Plus 2.5M"
#elif CurEmu == kEmuPlus
#define kEmuName "Plus 4M"
#elif CurEmu == kEmuSE1M
#define kEmuName "SE 1M"
#elif CurEmu == kEmuSE2M
#define kEmuName "SE 2M"
#elif CurEmu == kEmuSE2_5M
#define kEmuName "SE 2.5M"
#elif CurEmu == kEmuSE
#define kEmuName "SE 4M"
#elif CurEmu == kEmuClassic1M
#define kEmuName "Classic 1M"
#elif CurEmu == kEmuClassic2M
#define kEmuName "Classic 2M"
#elif CurEmu == kEmuClassic2_5M
#define kEmuName "Classic 2.5M"
#elif CurEmu == kEmuClassic
#define kEmuName "Classic 4M"
#else
#error "kEmuName not defined"
#endif
#endif

LOCALFUNC char * GetSubstitutionStr(char x)
{
	char *s;

	switch (x) {
		case 'w':
			s = kStrHomePage;
			break;
		case 'y':
			s = kStrCopyrightYear;
			break;
		case 'p':
			s = kStrAppName;
			break;
		case 'v':
			s = kAppVariationStr;
			break;
		case 'e':
			s = kEmuName;
			break;
		case 'm':
			s = kMaintainerName;
			break;
		case 'k':
			if (ControlKeyPressed) {
				s = "pressed";
			} else {
				s = "released";
			}
			break;
		case 'g':
			if (WantMagnify) {
				s = "on";
			} else {
				s = "off";
			}
			break;
		case 'f':
			if (WantFullScreen) {
				s = "on";
			} else {
				s = "off";
			}
			break;
		case 'b':
			if (RunInBackground) {
				s = "on";
			} else {
				s = "off";
			}
			break;
		case 's':
			if (SpeedStopped) {
				s = "Stopped";
			} else if (SpeedLimit) {
				switch (SpeedValue) {
					case 1:
						s = "2x";
						break;
					case 2:
						s = "4x";
						break;
					case 3:
						s = "8x";
						break;
					case 4:
						s = "16x";
						break;
					case 5:
						s = "32x";
						break;
					case 0:
					default:
						s = "1x";
						break;
				}
			} else {
				s = "All out";
			}
			break;
		default:
			s = "???";
			break;
	}
	return s;
}

LOCALVAR ui3b Ascii2CellMap[256];

LOCALPROC DrawCellsFromStr(unsigned int h, unsigned int v, char *s)
{
	char pc;
	unsigned int h0 = h;
	char *p = s;

	while ((pc = *p++) != 0) {
		if (pc == '\n') {
			h = h0;
			v++;
		} else if (pc == '^') {
			char *p2 = GetSubstitutionStr(*p++);
			while ((pc = *p2++) != 0) {
				DrawCell(h, v, Ascii2CellMap[(unsigned char)pc]);
				h++;
			}
		} else {
			DrawCell(h, v, Ascii2CellMap[(unsigned char)pc]);
			h++;
		}
	}
}

LOCALFUNC char * ControlMode2Str(void)
{
	char *s;

	switch (CurControlMode) {
		case kCntrlModeConfirmReset:
			s = "Confirm (Type 'Y' to Reset, 'N' to cancel)";
			break;
		case kCntrlModeConfirmInterrupt:
			s = "Confirm (Type 'Y' to Interrupt, 'N' to cancel)";
			break;
		case kCntrlModeConfirmQuit:
			s = "Confirm (Type 'Y' to Quit, 'N' to cancel)";
			break;
		case kCntrlModeSpeedControl:
			s = "Speed Control";
			break;
		case kCntrlModeBase:
		default:
			s = "Control Mode (Type 'H' for help)";
			break;
	}
	return s;
}

LOCALFUNC char * ControlMessage2Str(void)
{
	char *s;

	switch (ControlMessage) {
		case kCntrlMsgAbout:
			s = "\
^v, ^e, Copyright ^y.\n\
\n\
^p contains the work of many people.\n\
This version is maintained by:\n\
^m\n\
\n\
For more information, see:\n\
\n\
^w\n\
\n\
^p is distributed under the terms of the GNU Public\n\
License, version 2.\n\
\n\
^p is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty\n\
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.";
			break;
		case kCntrlMsgHelp:
			s = "\
To leave the Control Mode, release the 'control' key.\n\
Otherwise, type a letter. Available commands are:\n\
\n\
'A' - About (version information)\n\
'O' - Open disk image\n\
'Q' - Quit\n\
'S' - Speed control (^s)\n\
'M' - Magnify toggle (^g)\n\
'F' - Full screen toggle (^f)\n\
'K' - emulated 'control' Key toggle (^k)\n\
'R' - Reset\n\
'I' - Interrupt\n\
'H' - Help (show this message)";
			break;
/*
012345678901234567890123456789012345678901234567890123456789
000000000011111111112222222222333333333344444444445555555555
*/
		case kCntrlMsgSpeedControlStart:
			s = "Current speed: ^s\n\
'S' - Stopped\n\
'Z' - 1x\n\
'1' - 2x\n\
'2' - 4x\n\
'3' - 8x\n\
'4' - 16x\n\
'5' - 32x\n\
'A' - All out\n\
\n\
'B' - run in Background toggle (^b)\n\
\n\
'E' - Exit speed control";
			break;
		case kCntrlMsgNewSpeed:
			s = "Speed: ^s";
			break;
		case kCntrlMsgNewRunInBack:
			s = "Run in background is ^b";
			break;
		case kCntrlMsgMagnify:
			s = "Magnify is ^g";
			break;
		case kCntrlMsgFullScreen:
			s = "Full Screen is ^f";
			break;
		case kCntrlMsgConfirmResetStart:
			s = "\
Are you sure you want to Reset?\n\
\n\
Unsaved changes will be lost, and there is a risk of\n\
corrupting the mounted disk image files.";
			break;
		case kCntrlMsgHaveReset:
			s = "Have reset emulated machine";
			break;
		case kCntrlMsgResetCancelled:
			s = "Reset cancelled";
			break;
		case kCntrlMsgConfirmInterruptStart:
			s = "\
Are you sure you want to Interrupt?\n\
\n\
This will invoke any installed debugger.";
			break;
		case kCntrlMsgHaveInterrupted:
			s = "Have interrupted emulated machine";
			break;
		case kCntrlMsgInterruptCancelled:
			s = "Interrupt cancelled";
			break;
		case kCntrlMsgConfirmQuitStart:
			s = "\
Are you sure you want to Quit?\n\
\n\
You should shut down the emulated machine before quitting\n\
^p to prevent corrupting the mounted disk image\n\
files.";
			break;
		case kCntrlMsgQuitCancelled:
			s = "Quit cancelled";
			break;
		case kCntrlMsgEmCntrl:
			s = "Emulated 'control' key ^k";
			break;
		case kCntrlMsgBaseStart:
		default:
			s = "To leave the Control Mode, release the 'control' key.";
			break;
	}
	return s;
}

LOCALPROC InitAscii2CellMap(void)
{
	int i;
	char pc;
	char const *p = CellMap;

	for (i = 0; i < 255; ++i) {
		Ascii2CellMap[i] = kCellSpace;
	}

	i = kNumCells;
	while ((pc = *p++) != 0) {
		Ascii2CellMap[(unsigned char)pc] = i;
		i++;
	}
}

#define ControlBoxh0 0
#define ControlBoxw 62
#define ControlBoxv0 0

GLOBALPROC DrawControlMode(void)
{
	int i;
	int j;
	int k;
	char *s = ControlMessage2Str();
	int ControlBoxvsz = 5 + CountNewLines(s);
	int hLimit = ControlBoxh0 + ControlBoxw - 1;
	int vLimit = ControlBoxv0 + ControlBoxvsz - 1;

	InitAscii2CellMap();

	DrawCell(ControlBoxh0 + 0, ControlBoxv0 + 0, kCellUpperLeft);
	for (i = ControlBoxh0 + 1; i < hLimit; ++i) {
		DrawCell(i, ControlBoxv0 + 0, kCellUpperMiddle);
	}
	DrawCell(hLimit, ControlBoxv0 + 0, kCellUpperRight);

	for (j = ControlBoxv0 + 1; j < vLimit; ++j) {
		DrawCell(ControlBoxh0, j, kCellMiddleLeft);
		for (i = ControlBoxh0 + 1; i < hLimit; ++i) {
			DrawCell(i, j, kCellSpace);
		}
		DrawCell(hLimit, j, kCellMiddleRight);
	}

	DrawCell(ControlBoxh0 + 0, vLimit, kCellLowerLeft);
	for (i = ControlBoxh0 + 1; i < hLimit; ++i) {
		DrawCell(i, vLimit, kCellLowerMiddle);
	}
	DrawCell(hLimit, vLimit, kCellLowerRight);

	DrawCellsFromStr(ControlBoxh0 + 6, ControlBoxv0 + 1, ControlMode2Str());

	DrawCellsFromStr(ControlBoxh0 + 1, ControlBoxv0 + 3, s);

	k = kCellIcon00;
	for (j = ControlBoxv0 + 0; j < ControlBoxv0 + 3; ++j) {
		for (i = ControlBoxh0 + 1; i < ControlBoxh0 + 5; ++i) {
			DrawCell(i, j, k);
			k++;
		}
	}

	for (i = ControlBoxh0 + 5; i < hLimit; ++i) {
		DrawCell(i, ControlBoxv0 + 2, kCellGraySep);
	}
}
#endif
