
/*
 * MIXMOS - an oss mixer
 *
 * Copyright (C) 2003 Tomasz Mka
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "mixer.h"
#include "prefs.h"
#include "gui.h"
#include "config.h"
#include "i18n.h"

#define	max_device_name	64

gchar default_mixer_device[max_device_name];

void mixmos_usage (void)
{
	printf(_("\nMixmos - an OSS mixer\n"));
	printf("----------------------\n\n");
	printf(_("Usage:   mixmos [options]\n\n"));
	printf(_("Options: -d <device_name> : specify mixer device\n"));
	printf(_("         -s <preset> : store mixer state (preset: 1, 2, 3, 4)\n"));
	printf(_("         -r <preset> : restore mixer state (preset: 1, 2, 3, 4)\n"));
	printf(_("         -c : removes mixmos config file\n"));
	printf(_("         -i : displays current mixer state\n"));
	printf(_("         -v : displays version\n"));
	printf(_("         -h : displays this text\n"));
	printf("\n");
}


int main(int argc, char **argv)
{
int opt, i, o, s, r, m, c, preset;
char *optstring = "d:s:r:civh";

	o = s = r = m = c = FALSE;
	memset(default_mixer_device ,0, max_device_name);

	preset = 1;
	opterr = 0;

	while ((opt = getopt(argc, argv, optstring)) != -1) {

		switch (opt) {

			case 'd': 	strncpy (default_mixer_device, optarg, max_device_name);
						break;

			case 's':	s = o = TRUE;
						preset = atoi(optarg);
						if(preset < 1 || preset > 4) {
							printf(_("\nERROR: preset should be 1, 2, 3 or 4.\n\n"));
							exit(-1);
						}
						break;

			case 'r':	r = o = TRUE;
						preset = atoi(optarg);
						if(preset < 1 || preset > 4) {
							printf(_("\nERROR: preset should be 1, 2, 3 or 4.\n\n"));
							exit(-1);
						}
						break;

			case 'c':	c = o = TRUE;
						break;

			case 'i':	m = o = TRUE;
						break;

			case 'h':   mixmos_usage();
						o = TRUE;
						break;

			case 'v':   printf("%s\n", VERSION);
						o = TRUE;
						break;

			case '?':   printf(_("\nWARNING: invalid option or wrong syntax...\n"));
						mixmos_usage();
						exit(-1);
						break;

			default:	break;
		}
	}


	/* MIXER DEVICE */

	if(default_mixer_device[0]=='\0') 
		strcpy(default_mixer_device, DEFAULT_MIXER_DEVICE);
	else
		printf(_("mixer device = %s\n"), default_mixer_device);

	/* TRY TO OPEN MIXER AND READ CONFIG FILE*/

	mixer_check_error(mixer_open(default_mixer_device));
	mixer_check_error(mixer_state_read());

	read_config ();

	config.run_counter++;

	/* STORE MIXER STATE */

	if(s==TRUE) {

		for(i=0;i<mixer_get_channels();i++) {

			config.a_vol_l[preset-1][i] = mixer_get_volume_l(i);
			config.a_vol_r[preset-1][i] = mixer_get_volume_r(i);

			if (mixer_get_channel_recordable(i) && mixer_get_recordsrc(i)) 
				config.a_rec[preset-1] = i;
		}


		printf(_("mixer state stored (preset=%d).\n"), preset);
		write_config ();
	}

	/* RESTORE MIXER STATE */

	if(r==TRUE) {

		for(i=0;i<mixer_get_channels();i++) {
			if(mixer_get_channel_stereo(i)) {
				mixer_set_volume_stereo(i, config.a_vol_l[preset-1][i], 
										config.a_vol_r[preset-1][i]);
			} else {
				mixer_set_volume_mono(i, config.a_vol_l[preset-1][i]);
			}
		}

		mixer_set_recordsrc(config.a_rec[preset-1]);

		printf(_("mixer state restored (preset=%d).\n"), preset);

	}

	/* DISPLAYS CURRENT MIXER STATE */

	if(m==TRUE)
		mixer_state_print();

	/* REMOVE MIXMOS CONFIGFILE */

	if(c==TRUE) {

		if(unlink(s_getfilename_config ()) == -1)
			printf(_("mixmos configure file NOT removed.\n"));
		else
			printf(_("mixmos configure file removed.\n"));

	}

	/* START GUI */

	if(o==TRUE) exit(-1);

#if ENABLE_NLS
    gtk_set_locale();
    bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
    textdomain(PACKAGE);
#endif

	if (!gtk_init_check (&argc, &argv)) {
		mixmos_usage();
		exit(-1);
	}

	gui_create_mixer_window ();

	gtk_main ();

	write_config ();
	mixer_check_error(mixer_close());

	return 0;

}


