/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2004 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

/*                    Memory leaks at reconnection */ 
/*                    First   -  Second            */           
/*  Display             50 K       -               */           
/*  screen                -        -               */           
/*  font                  -        -               */           
/*  pixmap             900 K       -               */           
/*  gc                    -        -               */           
/*  cursor                -        -               */           
/*  color                 -        -               */           
/*  window                -        -               */           
/*                        Totals                   */           
/*                    920 K                        */           
 
#include <signal.h>

#include "X.h"
#include "Xproto.h"
#include "mi.h"
#include "fb.h"

#include "Agent.h"
#include "Drawable.h"
#include "Control.h"
#include "Reconnect.h"

#ifdef NXAGENT_RECONNECT

extern Bool nxagentReOpenDisplay(void*);
extern Bool nxagentReCreateScreen(void*);
extern Bool nxagentReCreateFonts(void*);
extern Bool nxagentReCreatePixmaps(void*);
extern Bool nxagentReCreateGCs(void*);
extern Bool nxagentReCreateCursors(void*);
extern Bool nxagentReCreateColormaps(void*);
extern Bool nxagentReCreateWindows(void*);
extern Bool nxagentReCreateGlyphSets(void*);
extern Bool nxagentReCreatePictFormats(void*);
extern Bool nxagentReCreatePictures(void*);

extern Bool nxagentDisconnectPictures(void);
extern Bool nxagentDisconnectWindows(void);
extern Bool nxagentDisconnectCursors(void);
extern Bool nxagentDisconnectPixmaps(void);
extern Bool nxagentDisconnectGCs(void);
extern Bool nxagentDisconnectFonts(void);
extern Bool nxagentDisconnectDisplay(void);

extern void nxagentBackupDisplayInfo(void);
extern void nxagentCleanBackupDisplayInfo(void);

extern Bool nxagentRenderEnable;

jmp_buf         pauseByError;
int             pauseByUser;

static enum SESSION_STATE
{
  SESSION_UP,
  SESSION_DOWN,
  SESSION_SLEEPING,
  SESSION_WAKE
} sessionState;

enum RECONNECTION_STEP
{
  DISPLAY_STEP = 0,
  SCREEN_STEP,
  FONT_STEP,
  PIXMAP_STEP,
  GC_STEP,
  CURSOR_STEP,
  COLORMAP_STEP,
  WINDOW_STEP,
  GLYPHSET_STEP,
  PICTFORMAT_STEP,
  PICTURE_STEP,
  NONE
};

char            *nxagentReconnectMSG;
void            *reconnectLossyLevel[ NONE ];
static enum RECONNECTION_STEP failedStep;

SIGVAL serverPauseByUser(int n)
{
#ifdef NXAGENT_RECONNECT_DEBUG
  fprintf(stderr, "serverPauseByUser: SIGHUP arrived.\n");
#endif

  if (pauseByUser)
  {
    pauseByUser = 0;
#ifdef NXAGENT_RECONNECT_DEBUG
    fprintf(stderr, "serverPauseByUser: Agent reconnects.\n");
#endif

  }
  else
  {
    pauseByUser = 1;
#ifdef NXAGENT_RECONNECT_DEBUG
    fprintf(stderr, "serverPauseByUser: Agent disconnects.\n");
#endif

  }
  if (sessionState == SESSION_SLEEPING)
  {
    sessionState = SESSION_WAKE;
  }
}

void nxagentInitializeRecLossyLevel()
{
  *(int*)reconnectLossyLevel[ DISPLAY_STEP ] = 0;
  *(int*)reconnectLossyLevel[ SCREEN_STEP ] = 0;
  *(int*)reconnectLossyLevel[ FONT_STEP ] = 0;
  *(int*)reconnectLossyLevel[ PIXMAP_STEP ] = 0;
  *(int*)reconnectLossyLevel[ GC_STEP ] = 0;
  *(int*)reconnectLossyLevel[ CURSOR_STEP ] = 0;
  *(int*)reconnectLossyLevel[ COLORMAP_STEP ] = 0;
  *(int*)reconnectLossyLevel[ WINDOW_STEP ] = 0;
  *(int*)reconnectLossyLevel[ GLYPHSET_STEP ] = 0;
  *(int*)reconnectLossyLevel[ PICTFORMAT_STEP ] = 0;
  *(int*)reconnectLossyLevel[ PICTURE_STEP ] = 0;
}

void nxagentInitReconnector(void)
{
  struct sigaction sa;

  sa.sa_handler = serverPauseByUser;
  sigfillset(&sa.sa_mask);
  sa.sa_flags = SA_RESTART;

  sigaction (SIGHUP, &sa, NULL);

  pauseByUser = 0;
  sessionState = SESSION_UP;

  reconnectLossyLevel[ DISPLAY_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ SCREEN_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ FONT_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ PIXMAP_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ GC_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ CURSOR_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ COLORMAP_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ WINDOW_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ GLYPHSET_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ PICTFORMAT_STEP ] = xalloc(sizeof(int));
  reconnectLossyLevel[ PICTURE_STEP ] = xalloc(sizeof(int));
}

void nxagentDisconnect(void)
{
  if (sessionState == SESSION_UP)
  {
    nxagentInitializeRecLossyLevel();

    nxagentBackupDisplayInfo();
  }

  if (nxagentRenderEnable)
  {
    nxagentDisconnectPictures();
  }

  nxagentDisconnectWindows();
  nxagentDisconnectCursors();
  nxagentDisconnectPixmaps();
  nxagentDisconnectGCs();
  nxagentDisconnectFonts();
  nxagentDisconnectDisplay(); 

  sessionState = SESSION_DOWN;

  fprintf(stderr, "Info: Disconnected from user display.\n");
}

/*
 * We could stay in this function for ages,
 * so we want to save cpu time blocking signals.
 */

#ifdef __sun

void nxagentWaitingForDisplay(void)
{
  sigset_t sleep_set, current_set;
  struct timeval timeout;

  sigemptyset(&current_set);
  sigemptyset(&sleep_set);
  
  sigaddset(&sleep_set, SIGALRM);
  
  fprintf(stderr, "Info: Going to sleep waiting for reconnection.\n");
  
  /* Blocking SIG_ALRM, and saving current set. */
  sigprocmask(SIG_BLOCK, &sleep_set, &current_set);

  sessionState = SESSION_SLEEPING;

  while (sessionState != SESSION_WAKE)
  {
    timeout.tv_sec = 2;
    timeout.tv_usec = 0;
    
    select(0, NULL, NULL, NULL, &timeout); 
  }

  /* Restoring old set. */
  if (!sigismember(&current_set, SIGALRM))
  {
    sigset_t restore_set;

    sigemptyset(&restore_set);
    sigaddset(&restore_set, SIGALRM);
    
    sigprocmask(SIG_UNBLOCK, &restore_set, NULL);
  }

  fprintf(stderr, "Info: Going to reconnect to a new display.\n");
}

#else

void nxagentWaitingForDisplay(void)
{
  sigset_t sleep_set, awake_set, current_set, restore_set;
  
  sigemptyset(&restore_set);
  sigemptyset(&current_set);
  sigemptyset(&sleep_set);
  sigemptyset(&awake_set);
  
  sigaddset(&awake_set, SIGHUP);
  sigaddset(&awake_set, SIGALRM);
  sigaddset(&sleep_set, SIGALRM);
  
  fprintf(stderr, "Info: Going to sleep waiting for reconnection.\n");
  
  /* Blocking SIG_ALRM and SIG_HUP, and saving current set. */
  sigprocmask(SIG_BLOCK, &awake_set, &current_set);

  sessionState = SESSION_SLEEPING;

  while (sessionState != SESSION_WAKE)
  {
    /* Blocking SIG_ALRM. */
    pselect(0, NULL, NULL, NULL, NULL, &sleep_set); 
  }

  /* Restoring old set. */
  if (!sigismember(&current_set, SIGHUP))
    sigaddset(&restore_set, SIGHUP);
  
  if (!sigismember(&current_set, SIGALRM))
    sigaddset(&restore_set, SIGALRM);

  sigprocmask(SIG_UNBLOCK, &restore_set, NULL);

  fprintf(stderr, "Info: Going to reconnect to a new display.\n");
}
#endif

Bool nxagentReconnect(void)
{
  if (!nxagentReOpenDisplay(reconnectLossyLevel[ DISPLAY_STEP ]))
  {
    failedStep = DISPLAY_STEP;

    goto failure;
  }

  if (!nxagentReCreateScreen(reconnectLossyLevel[ SCREEN_STEP ]))
  {
    failedStep = SCREEN_STEP;

    goto failure;
  }

  if (!nxagentReCreateFonts(reconnectLossyLevel[ FONT_STEP ]))
  {
    failedStep = FONT_STEP;

    goto failure;
  }

  if (!nxagentReCreatePixmaps(reconnectLossyLevel[ PIXMAP_STEP ]))
  {
    failedStep = PIXMAP_STEP;

    goto failure;
  }

  if (!nxagentReCreateGCs(reconnectLossyLevel[ GC_STEP ]))
  {
    failedStep = GC_STEP;

    goto failure;
  }

  if (!nxagentReCreateCursors(reconnectLossyLevel[ CURSOR_STEP ]))
  {
    failedStep = CURSOR_STEP;

    goto failure;
  }

  if (!nxagentReCreateColormaps(reconnectLossyLevel[ COLORMAP_STEP ]))
  {
    failedStep = COLORMAP_STEP;

    goto failure;
  }

  if (!nxagentReCreateWindows(reconnectLossyLevel[ WINDOW_STEP ]))
  {
    failedStep = WINDOW_STEP;

    goto failure;
  }

  if (nxagentRenderEnable)
  {
    if (!nxagentReCreateGlyphSets(reconnectLossyLevel[ GLYPHSET_STEP ]))
    {
      failedStep = GLYPHSET_STEP;

      goto failure;
    }

    if (!nxagentReCreatePictFormats(reconnectLossyLevel[ PICTFORMAT_STEP ]))
    {
      failedStep = PICTFORMAT_STEP;

      goto failure;
    }

    if (!nxagentReCreatePictures(reconnectLossyLevel[ PICTURE_STEP ]))
    {
      failedStep = PICTURE_STEP;

      goto failure;
    }
  }

  nxagentWakeupByReconnect();

  nxagentCleanBackupDisplayInfo();

  sessionState = SESSION_UP;

  /*
   * This should not be required.
   *
   * XSync(nxagentDisplay, False);
   */

  fprintf(stderr, "Info: Reconnection succeded.\n");

  return True;
  
failure:

  fprintf(stderr, "Error: Reconnection failed - %s\n", nxagentReconnectMSG);

  if (failedStep == FONT_STEP)
    *(int*)reconnectLossyLevel[ FONT_STEP ] = 1;

  return False;
}

#endif /* NXAGENT_RECONNECT */






