/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "Channel.h"

#include "Compressor.h"
#include "Decompressor.h"

#include "Statistics.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Define this to know how many messages
// are allocated and deallocated.
//

#undef  REFERENCES

//
// Set to false at the time the first
// channel is connected to X client
// or server.
//

int Channel::firstAgent_ = 1;

//
// This is used for reference count.
//

#ifdef REFERENCES

int Channel::references_ = 0;

#endif

Channel::Channel(Transport *transport, Compressor *compressor,
                     Decompressor *decompressor)

    : transport_(transport), compressor_(compressor),
          decompressor_(decompressor)
{
  fd_ = transport_ -> fd();

  finish_     = 0;
  pending_    = 0;
  congestion_ = 0;

  flush_    = 0;
  priority_ = 0;

  firstRequest_ = 1;
  firstReply_   = 1;

  clientSequenceReset_ = 0;
  serverSequenceReset_ = 0;

  lastCongestionTs_ = nullTimestamp();

  lowerClient_ = CONNECTIONS_LIMIT - 1;
  upperClient_ = 0;

  //
  // Cache status of last keep operation
  // on message stores.
  //

  lastKeep_ = 1;

  //
  // Must be set by proxy.
  //

  opcodeStore_ = NULL;
 
  clientStore_ = NULL;
  serverStore_ = NULL;

  clientCache_ = NULL;
  serverCache_ = NULL;

  #ifdef REFERENCES
  *logofs << "Channel: Created new Channel at " 
          << this << " out of " << ++references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

Channel::~Channel()
{
  #ifdef REFERENCES
  *logofs << "Channel: Deleted Channel at " 
          << this << " out of " << --references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

int Channel::handleEncode(EncodeBuffer &encodeBuffer, ChannelCache *channelCache,
                              MessageStore *store, const unsigned char opcode,
                                  const unsigned char *buffer, const unsigned int size)
{
  //
  // Check if message can be differentially
  // encoded using a similar message in the
  // message store.
  // 

  if (handleEncode(encodeBuffer, channelCache,
                       store, buffer, size) > 0)
  {
    return 1;
  }

  //
  // A similar message could not be found in
  // cache or message must be discarded. Must
  // encode the message using the best method
  // known.
  //

  if (handleEncode(encodeBuffer, channelCache, store, buffer,
                       size, bigEndian_) < 0)
  {
    return -1;
  }

  //
  // Check if message has a distinct data part.
  //

  if (store -> enableData)
  {
    //
    // If message split was requested by agent then send data
    // out-of-band, dividing it in small chunks. Until message
    // is completely transferred, store a dummy version of the
    // message with data replaced with blanks.
    //
    // While data is being transferred, agent should have put
    // client asleep. It can happen, though, that a different
    // client would reference the same image. We cannot issue
    // a cache hit for images being split (such images are put
    // in store in 'locked' state), so we need to handle this
    // case.
    //
    // Is not difficult to imagine an implementation accepting
    // multiple clients waiting for the a single message being
    // split. The real problem is encoding of identity. When a
    // client is restarted, infact, it needs to commit a messa-
    // ge that has not been modified since it's insertion in
    // cache, so we would need to put in split store both the
    // identity and data.
    //

    if (store -> enableSplit)
    {
      if (handleSplit(encodeBuffer, store, buffer, size) > 0)
      {
        //
        // Lock message to avoid it would be
        // removed from message store.
        //

        store -> lock(store -> lastAdded);

        #if defined(INFO) || defined(TEST)

        checkLocks(store);

        #endif

        return 0;
      }
    }

    //
    // Check if static compression of data part is
    // enabled. This is the case of all messages not
    // having a special differential encoding as well
    // as of all messages that we want to store in
    // cache in compressed format.
    //

    if (store -> enableCompress)
    {
      unsigned char *data   = NULL;
      unsigned int dataSize = 0;

      int compressed = handleCompress(encodeBuffer, opcode, buffer, size,
                                          store -> dataOffset, data, dataSize);
      if (compressed < 0)
      {
        return -1;
      }
      else if (compressed > 0)
      {
        //
        // Update message's size according
        // to result of data compression.
        //

        handleUpdate(store, size - store -> dataOffset, dataSize);
      }
    }
    else
    {
      handleCopy(encodeBuffer, opcode, buffer, size);
    }
  }

  return 0;
}

int Channel::handleDecode(DecodeBuffer &decodeBuffer, ChannelCache *channelCache,
                              MessageStore *store, unsigned char &opcode,
                                  unsigned char *&buffer, unsigned int &size)
{
  //
  // Check first if message is in cache.
  //

  if (handleDecode(decodeBuffer, channelCache,
                       store, buffer, size) > 0)
  {
    return 1;
  }

  //
  // Decode the full identity.
  //

  if (handleDecode(decodeBuffer, channelCache, store, buffer,
                       size, bigEndian_, &writeBuffer_) < 0)
  {
    return -1;
  }

  //
  // Check if message has a distinct
  // data part.
  //

  if (store -> enableData)
  {
    //
    // Check if message has been split.
    //

    if (store -> enableSplit)
    {
      if (handleUnsplit(decodeBuffer, store, buffer, size))
      {
        //
        // Save message in store in a dummy format. 
        //

        handleSave(store, buffer, size);

        store -> lock(store -> lastAdded);

        #if defined(INFO) || defined(TEST)

        checkLocks(store);

        #endif

        writeBuffer_.removeMessage(size - 4);
        size = 4;

        opcode = X_NoOperation;

        return 0;
      }
    }

    //
    // Decode the data part.
    //

    if (store -> enableCompress)
    {
      const unsigned char *data = NULL;
      unsigned int dataSize = 0;

      int decompressed = handleDecompress(decodeBuffer, opcode, buffer, size,
                                              store -> dataOffset, data, dataSize);
      if (decompressed < 0)
      {
        return -1;
      }
      else if (decompressed > 0)
      {
        //
        // Store message in compressed format.
        //

        handleSave(store, buffer, size, data, dataSize);

        return 0;
      }
    }
    else
    {
      handleCopy(decodeBuffer, opcode, buffer, size);
    }
  }

  //
  // Store message in plain format.
  //

  handleSave(store, buffer, size);

  return 0;
}

int Channel::handleEncode(EncodeBuffer &encodeBuffer, ChannelCache *channelCache,
                              MessageStore *store,  const unsigned char *buffer,
                                  const unsigned int size)
{
  if (control -> LocalDeltaCompression == 0 ||
          store == NULL || store -> enableCache == 0)
  {
    encodeBuffer.encodeStatusValue(is_discarded, store -> lastStatusCache);

    store -> lastStatus = is_discarded;

    return 0;
  }

  #ifdef DEBUG
  *logofs << "handleEncode: " << store -> name() 
          << ": Going to handle a new message of this class.\n" 
          << logofs_flush;
  #endif

  //
  // Update the reference timestamp.
  //

  store -> newTimestamp();

  //
  // Check if the estimated size of cache is greater
  // than the requested limit. If it is the case make
  // some room by deleting one or more messages.
  //

  int position;

  while (mustCleanStore(store) == 1 && canCleanStore(store) == 1)
  {
    #ifdef DEBUG
    *logofs << "handleEncode: " << store -> name() 
            << ": Trying to reduce size of message store.\n"
            << logofs_flush;
    #endif

    position = store -> clean(use_checksum);

    if (position == nothing)
    {
      #ifdef TEST
      *logofs << "handleEncode: " << store -> name() 
              << ": WARNING! No message found to be "
              << "actually removed.\n" << logofs_flush;
      #endif

      break;
    }

    #ifdef DEBUG
    *logofs << "handleEncode: " << store -> name() 
            << ": Message at position " << position
            << " has been removed.\n" << logofs_flush;

    *logofs << "handleEncode: " << store -> name() << ": There are " 
            << store -> getSize() << " messages in the store out of " 
            << store -> cacheSlots << " slots.\n" << logofs_flush;

    *logofs << "handleEncode: " << store -> name() 
            << ": Size of store is " << store -> getLocalStorageSize() 
            << " bytes locally and " << store -> getRemoteStorageSize() 
            << " bytes remotely.\n" << logofs_flush;

    *logofs << "handleEncode: " << store -> name() 
            << ": Size of total cache is " << store -> getLocalTotalStorageSize()
            << " bytes locally and " << store -> getRemoteTotalStorageSize()
            << " bytes remotely.\n" << logofs_flush;
    #endif

    encodeBuffer.encodeStatusValue(is_removed, store -> lastStatusCache);

    //
    // Encode position of message to discard.
    //

    store -> lastRemoved = position;

    encodeBuffer.encodePositionValue(store -> lastRemoved, store -> lastRemovedCache);
  }

  #ifdef DEBUG

  if (mustCleanStore(store) == 1 && canCleanStore(store) == 0)
  {
    *logofs << "handleEncode: " << store -> name() 
            << ": Store would need a clean but operation will be delayed.\n" 
            << logofs_flush;

    *logofs << "handleEncode: " << store -> name() << ": There are " 
            << store -> getSize() << " messages in the store out of " 
            << store -> cacheSlots << " slots.\n"  << logofs_flush;

    *logofs << "handleEncode: " << store -> name() 
            << ": Size of store is " << store -> getLocalStorageSize() 
            << " bytes locally and " << store -> getRemoteStorageSize() 
            << " bytes remotely.\n" << logofs_flush;

    *logofs << "handleEncode: " << store -> name() 
            << ": Size of total cache is " << store -> getLocalTotalStorageSize()
            << " bytes locally and " << store -> getRemoteTotalStorageSize()
            << " bytes remotely.\n" << logofs_flush;
  }

  #endif

  //
  // If 'on the wire' size of message exceeds the
  // allowed limit then avoid to store it in the
  // cache.
  //

  if (((int) size) < control -> MinimumMessageSizeThreshold ||
          ((int) size) > control -> MaximumMessageSizeThreshold)
  {
    #ifdef TEST
    *logofs << "handleEncode: " << store -> name() 
            << ": Message ignored as its size doesn't comply "
            << "with the " << control -> MinimumMessageSizeThreshold << "-" 
            << control -> MaximumMessageSizeThreshold << " range.\n" 
            << logofs_flush;
    #endif

    encodeBuffer.encodeStatusValue(is_discarded, store -> lastStatusCache);

    store -> lastStatus = is_discarded;

    return 0;
  }

  //
  // Fill message object with received data.
  //

  Message *message = store -> getTemporary();

  if (message == NULL)
  {
    #ifdef PANIC
    *logofs << "handleEncode: " << store -> name()
            << ": PANIC! Can't allocate memory for "
            << "a new message.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't allocate memory for "
         << "a new message in context [D].\n";

    HandleCleanup();
  }

  //
  // As we are at encoding side, it is enough to store checksum
  // for the object while data can be erased. Both identity and
  // data part will never be sent through the wire again as long
  // as they are stored in the cache at decoding side.
  //

  store -> parse(message, buffer, size, use_checksum,
                     discard_data, bigEndian_);

  #ifdef DUMP

  store -> dump(message);

  #endif

  //
  // Search the object in the message
  // store. If found get the position.
  //

  #ifdef DEBUG
  *logofs << "handleEncode: " << store -> name() 
          << ": Searching object of size " << size
          << " in the cache.\n" << logofs_flush;
  #endif

  int added;
  int locked;

  position = store -> findOrAdd(message, use_checksum, added, locked);

  if (position == nothing)
  {
    #ifdef WARNING
    *logofs << "handleEncode: " << store -> name() 
            << ": WARNING! Can't store object in the cache.\n"
            << logofs_flush;
    #endif

    encodeBuffer.encodeStatusValue(is_discarded, store -> lastStatusCache);

    store -> lastStatus = is_discarded;

    return 0;
  }
  else if (locked)
  {
    //
    // We can't issue a cache hit. Encoding identity
    // differences while message it's being split
    // would later result in agent to commit a wrong
    // version of message.
    //

    #ifdef WARNING
    *logofs << "handleEncode: " << store -> name() 
            << ": WARNING! Message of size " << store -> plainSize(position)
            << " at position " << position << " is locked.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Message of size " << store -> plainSize(position)
         << " at position " << position << " is locked.\n";

    encodeBuffer.encodeStatusValue(is_discarded, store -> lastStatusCache);

    store -> lastStatus = is_discarded;

    return 0;
  }
  else if (added)
  {
    store -> resetTemporary();

    #ifdef DEBUG
    *logofs << "handleEncode: " << store -> name() << ": Message of size "
            << store -> plainSize(position) << " has been stored at position "
            << position << ".\n" << logofs_flush;

    *logofs << "handleEncode: " << store -> name() << ": There are " 
            << store -> getSize() << " messages in the store out of " 
            << store -> cacheSlots << " slots.\n" << logofs_flush;

    *logofs << "handleEncode: " << store -> name() 
            << ": Size of store is " << store -> getLocalStorageSize() 
            << " bytes locally and " << store -> getRemoteStorageSize() 
            << " bytes remotely.\n" << logofs_flush;

    *logofs << "handleEncode: " << store -> name() 
            << ": Size of total cache is " << store -> getLocalTotalStorageSize()
            << " bytes locally and " << store -> getRemoteTotalStorageSize()
            << " bytes remotely.\n" << logofs_flush;
    #endif

    //
    // Inform the decoding side that message 
    // must be inserted in cache.
    //

    encodeBuffer.encodeStatusValue(is_added, store -> lastStatusCache);

    store -> lastStatus = is_added;

    //
    // Encode position where insertion took
    // place.
    //

    store -> lastAdded = position;

    encodeBuffer.encodePositionValue(store -> lastAdded, store -> lastAddedCache);

    //
    // Ensure that images used at startup
    // are retained in repository and saved
    // on persistent cache.
    //

    handleStartup(store, store -> lastAdded);

    return 0;
  }
  else
  {
    #ifdef DEBUG
    *logofs << "handleEncode: " << store -> name()
            << ": Cache hit. Found object at position " 
            << position << ".\n" << logofs_flush;
    #endif

    Message *cachedMessage = store -> get(position);

    //
    // This is quite unlikely to happen
    // unless there is a coding error.
    //

    #ifdef TEST

    if (cachedMessage == NULL)
    {
      #ifdef PANIC
      *logofs << "handleEncode: " << store -> name() 
              << ": PANIC! Object at position " << position
              << " is NULL.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Object of type " << store -> name()
           << " at position " << position << " is NULL.\n";

      HandleCleanup();
    }

    #endif

    //
    // Increase rating of the cached
    // message.
    //

    store -> touch(cachedMessage);

    #ifdef DEBUG
    *logofs << "handleEncode: " << store -> name() << ": Hits for "
            << "object at position " << position << " are now " 
            << store -> getTouches(position) << ".\n"
            << logofs_flush;
    #endif

    //
    // Send to the decoding side position
    // where object can be found in cache.
    //

    encodeBuffer.encodeStatusValue(is_cached, store -> lastStatusCache);

    store -> lastStatus = is_cached;

    store -> lastHit = position;

    encodeBuffer.encodePositionValue(store -> lastHit, store -> lastHitCache);

    //
    // Send the field by field differences in
    // respect to the original message stored
    // in cache.
    //

    store -> updateIdentity(encodeBuffer, message, cachedMessage, channelCache);

    //
    // Check if message must be kept in cache.
    //

    handleStartup(store, store -> lastHit);

    return 1;
  }
}

int Channel::handleUpdate(MessageStore *store, unsigned int dataSize,
                              unsigned int compressedDataSize)
{
  if (store -> lastStatus != is_discarded)
  {
    #ifdef DEBUG
    *logofs << "handleUpdate: " << store -> name() << ": Updating "
            << "object at position " << store -> lastAdded << " of size "
            << store -> plainSize(store -> lastAdded) << " (" << dataSize
            << "/" << compressedDataSize << ").\n" << logofs_flush;
    #endif

    store -> updateData(store -> lastAdded, dataSize, compressedDataSize);

    #ifdef DEBUG
    *logofs << "handleUpdate: " << store -> name() << ": There are " 
            << store -> getSize() << " messages in the store out of " 
            << store -> cacheSlots << " slots.\n" << logofs_flush;

    *logofs << "handleUpdate: " << store -> name() 
            << ": Size of store is " << store -> getLocalStorageSize() 
            << " bytes locally and " << store -> getRemoteStorageSize() 
            << " bytes remotely.\n" << logofs_flush;

    *logofs << "handleUpdate: " << store -> name() 
            << ": Size of total cache is " << store -> getLocalTotalStorageSize()
            << " bytes locally and " << store -> getRemoteTotalStorageSize()
            << " bytes remotely.\n" << logofs_flush;
    #endif

    return 1;
  }

  return 0;
}

int Channel::handleDecode(DecodeBuffer &decodeBuffer, ChannelCache *channelCache,
                              MessageStore *store, unsigned char *&buffer,
                                  unsigned int &size)
{
  //
  // Create a new message object and 
  // fill it with received data.
  //

  #ifdef DEBUG
  *logofs << "handleDecode: " << store -> name() 
          << ": Going to handle a new message of this class.\n" 
          << logofs_flush;
  #endif

  //
  // Update the reference timestamp.
  //

  store -> newTimestamp();

  //
  // Decode bits telling how to handle
  // this message.
  //

  unsigned char status;

  decodeBuffer.decodeStatusValue(status, store -> lastStatusCache);

  //
  // Clean operations must always come 
  // before any operation on message.
  //

  while (status == is_removed)
  {
    decodeBuffer.decodePositionValue(store -> lastRemoved, store -> lastRemovedCache);

    #ifdef DEBUG

    if (store -> get(store -> lastRemoved))
    {
      *logofs << "handleDecode: " << store -> name() << ": Cleaning up "
              << "object at position " << store -> lastRemoved
              << " of size " << store -> plainSize(store -> lastRemoved)
              << " (" << store -> plainSize(store -> lastRemoved) << "/"
              << store -> compressedSize(store -> lastRemoved) << ").\n"
              << logofs_flush;
    }

    #endif

    if (store -> remove(store -> lastRemoved, discard_checksum) == nothing)
    {
      #ifdef PANIC
      *logofs << "handleDecode: " << store -> name()
              << ": PANIC! Can't remove message at position "
              << store -> lastRemoved << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't remove message at position "
           << store -> lastRemoved << " from store "
           << store -> name() << ".\n";

      HandleCleanup();
    }

    decodeBuffer.decodeStatusValue(status, store -> lastStatusCache);
  }

  //
  // If it's a cache hit, the position
  // where object can be found follows.
  //

  if ((T_status) status == is_cached)
  {
    decodeBuffer.decodePositionValue(store -> lastHit, store -> lastHitCache);

    //
    // Get data from the cache at given position.
    //

    #ifdef DEBUG

    if (store -> get(store -> lastHit))
    {
      *logofs << "handleDecode: " << store -> name() << ": Retrieving "
              << "object at position " << store -> lastHit
              << " of size " << store -> plainSize(store -> lastHit)
              << " (" << store -> plainSize(store -> lastHit)  << "/"
              << store -> compressedSize(store -> lastHit) << ").\n"
              << logofs_flush;
    }

    #endif

    Message *message = store -> get(store -> lastHit);

    if (message == NULL)
    {
      #ifdef PANIC
      *logofs << "handleDecode: " << store -> name() 
              << ": PANIC! Can't retrieve object from store "
              << "at position " << store -> lastHit << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't retrieve object from store "
           << store -> name() << " at position "
           << store -> lastHit << ".\n";

      HandleCleanup();
    }

    //
    // Note that hits as registered at decoding side are likely
    // not to be the same as at encoding side. Encoder keeps
    // hits in a defined range  in order to give a chance to
    // newer messages  to be placed in cache but doesn't propa-
    // gate this information to the remote peer.
    //

    store -> touch(store -> lastHit);

    #ifdef DEBUG
    *logofs << "handleDecode: " << store -> name() << ": Hits for "
            << "object at position " << store -> lastHit << " are now " 
            << store -> getTouches(store -> lastHit) << ".\n"
            << logofs_flush;
    #endif

    //
    // Make room for the outgoing message.
    //

    size = store -> plainSize(store -> lastHit);

    buffer = writeBuffer_.addMessage(size);

    #ifdef DEBUG
    *logofs << "handleDecode: " << store -> name() 
            << ": Prepared an outgoing buffer of " 
            << size << " bytes.\n" << logofs_flush;
    #endif

    //
    // Decode the variant part. Pass client
    // or server cache to the message store.
    //

    store -> updateIdentity(decodeBuffer, message, channelCache);

    //
    // Write each field in the outgoing buffer.
    //

    store -> unparse(message, buffer, size, bigEndian_);

    #ifdef DUMP

    store -> dump(message);

    #endif

    store -> lastStatus = is_cached;

    return 1;
  }
  else if ((T_status) status == is_added)
  {
    decodeBuffer.decodePositionValue(store -> lastAdded, store -> lastAddedCache);

    #ifdef DEBUG
    *logofs << "handleDecode: " << store -> name() 
            << ": Message will later be stored at position "
             << store -> lastAdded << ".\n" << logofs_flush;
    #endif

    store -> lastStatus = is_added;

    return 0;
  }
  else
  {
    #ifdef DEBUG
    *logofs << "handleDecode: " << store -> name() 
            << ": Message will later be discarded.\n"
            << logofs_flush;
    #endif

    store -> lastStatus = is_discarded;

    return 0;
  }
}

int Channel::handleSave(MessageStore *store, unsigned char *buffer, unsigned int size,
                            const unsigned char *compressedData,
                                const unsigned int compressedDataSize)
{
  if (store == NULL || store -> enableCache == 0)
  {
    return 0;
  }

  if (store -> lastStatus == is_discarded)
  {
    #ifdef TEST
    *logofs << "handleSave: " << store -> name() 
            << ": Message ignored as it doesn't comply "
            << "with current parameters.\n" << logofs_flush;
    #endif

    return 0;
  }

  Message *message = store -> getTemporary();

  if (message == NULL)
  {
    #ifdef PANIC
    *logofs << "handleSave: " << store -> name() 
            << ": PANIC! Can't access temporary storage "
            << "for message at position " << store -> lastAdded
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't access temporary storage "
         << "for message  at position " << store -> lastAdded
         << ".\n";

    HandleCleanup();
  }

  if (compressedData == NULL)
  {
    store -> parse(message, buffer, size, discard_checksum,
                       use_data, bigEndian_);
  }
  else
  {
    store -> parse(message, buffer, size, compressedData, 
                       compressedDataSize, discard_checksum,
                           use_data, bigEndian_);
  }

  if (store -> add(message, store -> lastAdded,
          discard_checksum) == nothing)
  {
    #ifdef PANIC
    *logofs << "handleSave: " << store -> name()
            << ": PANIC! Can't store message in the cache "
            << "at position " << store -> lastAdded << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't store message of type "
         << store -> name() << "in the cache at position "
         << store -> lastAdded << ".\n";
  }
  else
  {
    store -> resetTemporary();

    #ifdef DEBUG
    *logofs << "handleSave: " << store -> name() << ": Stored "
            << (compressedData == NULL ? "plain" : "compressed")
            << " object at position " << store -> lastAdded
            << " of size " << store -> plainSize(store -> lastAdded)
            << " (" << store -> plainSize(store -> lastAdded) << "/"
            << store -> compressedSize(store -> lastAdded) << ").\n"
            << logofs_flush;
    #endif
  }

  #ifdef DEBUG
  *logofs << "handleSave: " << store -> name() 
          << ": Size of store is " << store -> getLocalStorageSize() 
          << " bytes locally and " << store -> getRemoteStorageSize() 
          << " bytes remotely.\n" << logofs_flush;

  *logofs << "handleSave: " << store -> name() 
          << ": Size of total cache is " << store -> getLocalTotalStorageSize()
          << " bytes locally and " << store -> getRemoteTotalStorageSize()
          << " bytes remotely.\n" << logofs_flush;
  #endif

  return 1;
}

int Channel::handleStartup(MessageStore *store, const int position)
{
  if (lastKeep_ == 1)
  {
    if (control -> isStartup() &&
            (control -> SessionMode == SESSION_X ||
                 control -> SessionMode == SESSION_APPLICATION))
    {
      unsigned char opcode = store -> opcode();

      if (opcode == opcodeStore_ -> putPackedImage ||
              opcode == X_PutImage)
      {
        lastKeep_ = store -> keep(position);
      }
    }
    else
    {
      lastKeep_ = 0;
    }
  }

  return lastKeep_;
}

int Channel::handleFlush(T_flush type)
{
  if (finish_ == 1)
  {
    #ifdef TEST
    *logofs << "handleFlush: No data flushed for finishing channel for FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    writeBuffer_.fullReset();

    return -1;
  }

  int bufferLength = writeBuffer_.getLength();
  int scratchBufferLength = writeBuffer_.getScratchLength();

  if (scratchBufferLength > 0 ||
          (type == flush_if_any && bufferLength > 0) ||
              (type == flush_if_needed && bufferLength >=
                   control -> TransportWriteThreshold))
  {
    #ifdef TEST
    *logofs << "handleFlush: Going to flush " << bufferLength
            << " + " << scratchBufferLength << " bytes of data "
            << "to FD#" << fd_ << ".\n" << logofs_flush;
    #endif

    //
    // First write data from normal buffer followed by
    // data from scratch buffer then cleanup everything.
    // Best performances are with small buffers (between
    // 1 and 4 KBytes) with TCP_NODELAY set on socket.
    //

    int result = 0;

    if (bufferLength > 0)
    {
      result = transport_ -> write(write_immediate,
                                       writeBuffer_.getData(),
                                           bufferLength);
    }

    if (result >= 0 && scratchBufferLength > 0)
    {
      result = transport_ -> write(write_immediate,
                                       writeBuffer_.getScratchData(),
                                           scratchBufferLength);
    }

    if (type == flush_if_any)
    {
      writeBuffer_.fullReset();
    }
    else
    {
      writeBuffer_.partialReset();
    }

    //
    // If we failed to write to the X connection then
    // set the finish flag. The caller should continue
    // to handle all the remaining messages. At the
    // real end error will be propagated to the upper
    // layers which will perform any needed cleanup.
    //

    if (result < 0)
    {
      finish_ = 1;

      return -1;
    }

    //
    // If we exceeded the buffer limit and channel is
    // considered reliable (i.e. it is very likely to
    // consume its data in short time, as in the case
    // of our X server) then force write to the socket.
    // This mainly helps to reduce memory requirements
    // but also saves CPU due to buffering overhead.
    //

    if (isReliable() == 1)
    {
      int limit = getLimit();

      int length = transport_ -> length();

      if (length > limit)
      {
        if (length > control -> TransportForceBufferSize)
        {
          //
          // Try to report congestions earlier.
          //

          if (control -> CongestionTimeout > 0)
          {
            control -> CongestionTimeout /= 2;

            if (control -> CongestionTimeout < 5)
            {
              control -> CongestionTimeout = 0;
            }

            #ifdef WARNING
            *logofs << "handleFlush: WARNING! Updated congestion timeout "
                    << "to " << control -> CongestionTimeout << " Ms.\n"
                    << logofs_flush;
            #endif

            cerr << "Warning" << ": Updated congestion timeout to "
                 << control -> CongestionTimeout << " Ms.\n";
          }

          if (transport_ -> force(0) < 0)
          {
            return -1;
          }
        }
        else if (control -> AgentForceReliable > 0)
        {
          if (transport_ -> force(0) < 0)
          {
            return -1;
          }
        }
      }
    }

    return 1;
  }

  return 0;
}

int Channel::handleCompress(EncodeBuffer &encodeBuffer, const unsigned char opcode,
                                const unsigned char *buffer, const unsigned int size,
                                    unsigned int offset, unsigned char *&compressedData,
                                        unsigned int &compressedDataSize)
{
  if (size <= offset)
  {
    #ifdef DEBUG
    *logofs << "handleCompress: Not compressing data for FD#" << fd_
            << " as offset is " << offset << " with data size "
            << size << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  //
  // It is responsibility of Compressor to
  // mark data if compression can actually
  // take place.
  //

  if (compressor_ -> compressBuffer(buffer + offset, size - offset, compressedData,
                                        compressedDataSize, encodeBuffer) <= 0)
  {
    #ifdef DEBUG
    *logofs << "handleCompress: Sent " << size - offset
            << " bytes of plain data for FD#" << fd_
            << ".\n" << logofs_flush;
    #endif

    return 0;
  }
  else
  {
    #ifdef DEBUG
    *logofs << "handleCompress: Sent " << compressedDataSize
            << " bytes of compressed data for FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    return 1;
  }
}

int Channel::handleDecompress(DecodeBuffer &decodeBuffer, const unsigned char opcode,
                                  unsigned char *buffer, const unsigned int size,
                                      unsigned int offset, const unsigned char *&compressedData,
                                          unsigned int &compressedDataSize)
{
  if (size <= offset)
  {
    return 0;
  }

  int result = decompressor_ -> decompressBuffer(buffer + offset, size - offset,
                                                     compressedData, compressedDataSize,
                                                         decodeBuffer);
  if (result < 0)
  {
    #ifdef PANIC
    *logofs << "handleDecompress: PANIC! Failed to decompress "
            << size - offset << " bytes of data for FD#" << fd_
            << " with OPCODE#" << (unsigned int) opcode << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Data decompression failed for OPCODE#"
         << (unsigned int) opcode << ".\n";

    return -1;
  }
  else if (result == 0)
  {
    #ifdef DEBUG
    *logofs << "handleDecompress: Received " << size - offset
            << " bytes of plain data for FD#" << fd_
            << ".\n" << logofs_flush;
    #endif

    return 0;
  }
  else
  {
    #ifdef DEBUG
    *logofs << "handleDecompress: Received " << compressedDataSize
            << " bytes of compressed data for FD#" << fd_
            << ".\n" << logofs_flush;
    #endif

    return 1;
  }
}

int Channel::handleCongestion()
{
  //
  // Check congestion according to the type of
  // channel. If there is a change in congestion
  // state do the following:
  //
  // - If a timeout has not previously set then
  //   do that now.
  //
  // - If a timeout exists and it has elapsed
  //   then save the new congestion state and
  //   report the transition to proxy.
  //
  // If no change in congestion state is detected
  // then cancel any pending timeout.
  //

  int state = isCongested();

  if (state != congestion_)
  {
    if (state == 1)
    {
      #ifdef TEST
      *logofs << "handleCongestion: Channel for FD#" << fd_
              << " is in congestion state.\n"
              << logofs_flush;
      #endif

      //
      // If channel is considered reliable,
      // wait for a timeout before reporting
      // the condition.
      //

      if (isReliable() == 1 && control -> CongestionTimeout > 0)
      {
        T_timestamp nowTs = getTimestamp();

        int diffTs = diffTimestamp(lastCongestionTs_, nowTs);

        if (diffTs < 0)
        {
          #ifdef TEST
          *logofs << "handleCongestion: Set congestion timeout for FD#"
                  << fd_<< ".\n" << logofs_flush;
          #endif

          lastCongestionTs_ = nowTs;

          return 0;
        }
        else if (diffTs < control -> CongestionTimeout)
        {
          #ifdef TEST
          *logofs << "handleCongestion: Congestion timeout for FD#"
                  << fd_<< " still not elapsed.\n"
                  << logofs_flush;
          #endif

          return 0;
        }
      }
    }
    #ifdef TEST
    else
    {
      *logofs << "handleCongestion: Channel for FD#" << fd_
              << " is out of congestion state.\n"
              << logofs_flush;
    }
    #endif

    lastCongestionTs_ = nullTimestamp();

    //
    // Tell proxy to send a congestion
    // message to the remote end.
    //

    congestion_ = state;

    return 1;
  }

  //
  // Proxy calls this function any time it needs
  // to force channel to check its congestion state,
  // for example just after the channel's transport
  // buffer has been flushed. This could cause the
  // newly detected state to invalidate the timeout
  // set to trigger a new transition.
  //

  #if defined(INFO) || defined(TEST)
  *logofs << "handleCongestion: No change reported in "
          << "congestion state for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  lastCongestionTs_ = nullTimestamp();

  return 0;
}

int Channel::handleReset()
{
  #ifdef TEST
  *logofs << "handleReset: Resetting base channel class "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Leave the finish flag unchanged. There should
  // be no need to reset transport, compressor, de-
  // compressor and channel's write buffer. The X
  // channels will persist after the reset so keep
  // all the negotiated settings.
  //

  pending_    = 0;
  congestion_ = 0;

  flush_    = 0;
  priority_ = 0;

  //
  // Reset timestamp on congestion timeout.
  // State of congestion is canceled at the
  // time proxy is reset.
  //

  lastCongestionTs_ = nullTimestamp();

  #ifdef TEST
  *logofs << "handleReset: Completed reset of base channel "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

void Channel::handleUpdateAgentClients(int client)
{
  if (client < 0 || client >= CONNECTIONS_LIMIT)
  {
    #ifdef PANIC
    *logofs << "ClientChannel: PANIC! Client id " << client
            << " is out of range with limit set to "
            << CONNECTIONS_LIMIT << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Client id " << client
         << " is out of range with limit set to "
         << CONNECTIONS_LIMIT << ".\n";

    HandleCleanup();
  }

  if (client < lowerClient_)
  {
    lowerClient_ = client;
  }

  if (client > upperClient_)
  {
    upperClient_ = client;
  }
}

int Channel::setOpcodes(OpcodeStore *opcodeStore)
{
  opcodeStore_ = opcodeStore;

  #ifdef TEST
  *logofs << "setOpcodes: Propagated opcodes store to channel "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

int Channel::setStores(ClientStore *clientStore, ServerStore *serverStore)
{
  clientStore_ = clientStore;
  serverStore_ = serverStore;

  #ifdef TEST
  *logofs << "setStores: Propagated message stores to channel "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

int Channel::setCaches(ClientCache *clientCache, ServerCache *serverCache)
{
  clientCache_ = clientCache;
  serverCache_ = serverCache;

  #ifdef TEST
  *logofs << "setCaches: Propagated encode caches to channel "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

int Channel::checkLocks(MessageStore *store)
{
  if (store == NULL)
  {
    return 0;
  }

  int p, n, s;

  s = store -> cacheSlots;

  for (p = 0, n = 0; p < s; p++)
  {
    if (store -> getLocks(p) == 1)
    {
      n++;
    }
    else if (store -> getLocks(p) != 0)
    {
      #ifdef PANIC
      *logofs << "checkLocks: PANIC! Repository for OPCODE#"
              << (unsigned int) store -> opcode() << " has "
              << store -> getLocks(p) << " locks for message "
              << "at position " << p << ".\n" << logofs_flush;
      #endif
    }
  }

  #ifdef TEST
  *logofs << "checkLocks: Repository for OPCODE#"
          << (unsigned int) store -> opcode()
          << " has " << n << " locked messages.\n"
          << logofs_flush;
  #endif

  return 1;
}
