/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <string.h>

#include <X11/X.h>
#include <X11/Xatom.h>

#include "NXproto.h"

#include "ClientChannel.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

#include "Compressor.h"
#include "Decompressor.h"

#include "Statistics.h"

#include "Split.h"

#include "PutImage.h"
#include "PutPackedImage.h"

//
// Set the verbosity level. You also
// need to define OPCODES in Misc.cpp
// if you want literals instead of
// opcodes' numbers.
//

#define PANIC
#define WARNING
#undef  OPCODES
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// By defining this, a simple procedure is activated at
// startup which just allocate and deallocate a plenty
// of cache objects. This is used to help determine
// current memory requirements.
//

#undef  MEMORY

//
// Inspects target of common X operations.
//

#undef  TARGETS

#ifdef TARGETS

#include <set>
#include <map>

typedef set < unsigned int, less<unsigned int> > T_windows;
typedef set < unsigned int, less<unsigned int> > T_pixmaps;
typedef map < unsigned int, unsigned int, less<unsigned int> > T_gcontexts;

T_windows   windows;
T_pixmaps   pixmaps;
T_gcontexts gcontexts;

#endif

//
// Define this to log when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// Here are the static members.
//

#ifdef REFERENCES

int ClientChannel::references_ = 0;

#endif

ClientChannel::ClientChannel(Transport *transport, Compressor *compressor,
                                 Decompressor *decompressor)

  : Channel(transport, compressor, decompressor),
        readBuffer_(transport_, this)
{
  //
  // Create X channels' specific caches
  // only if protocol is older than 3.
  //

  if (control -> isProtoStep3() == 0)
  {
    clientCache_ = new ClientCache();
    serverCache_ = new ServerCache();

    if (clientCache_ == NULL || serverCache_ == NULL)
    {
      #ifdef PANIC
      *logofs << "ClientChannel: PANIC! Failed to create channel's caches.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Failed to create channel's caches.\n";

      HandleCleanup();
    }
  }

  //
  // Current sequence and opcodes known
  // by client and X server.
  //

  clientOpcode_ = 0;
  serverOpcode_ = 0;

  clientSequence_ = 0;
  serverSequence_ = 0;

  //
  // Current sequence known by NX agent.
  //

  lastSequence_ = 0;

  //
  // Clear the wakeup queues.
  //

  for (int i = 0; i < CONNECTIONS_LIMIT; i++)
  {
    syncClientTs_[i]  = nullTimestamp();
    karmaClientTs_[i] = nullTimestamp();
  }

  syncCounter_  = 0;
  karmaCounter_ = 0;

  taintCounter_ = 0;

  //
  // No client requested a split yet.
  //

  splitClient_ = nothing;

  #ifdef MEMORY

  *logofs << "ClientChannel: Created 1 ClientCache and 1 ServerCache. "
          << "You have 30 seconds to check the allocated size.\n"
          << logofs_flush;

  sleep(30);

  ClientCache *clientCacheTestArray[100];
  ServerCache *serverCacheTestArray[100];

  for (int i = 0; i < 100; i++)
  {
    clientCacheTestArray[i] = new ClientCache();
  }

  *logofs << "ClientChannel: Created further 100 ClientCache. "
          << "You have 30 seconds to check the allocated size.\n"
          << logofs_flush;

  sleep(30);

  for (int i = 0; i < 100; i++)
  {
    serverCacheTestArray[i] = new ServerCache();
  }

  *logofs << "ClientChannel: Created further 100 ServerCache. "
          << "You have 30 seconds to check the allocated size.\n"
          << logofs_flush;

  sleep(30);

  for (int i = 0; i < 100; i++)
  {
    delete clientCacheTestArray[i];
    delete serverCacheTestArray[i];
  }

  *logofs << "ClientChannel: Deleted 100 ClientCache and 100 ServerCache. "
          << "You have 30 seconds to check the allocated size.\n"
          << logofs_flush;

  sleep(30);

  #endif

  #ifdef REFERENCES
  *logofs << "ClientChannel: Created new object at " 
          << this << " for FD#" << fd_ << " out of " 
          << ++references_ << " allocated channels.\n"
          << logofs_flush;
  #endif
}

ClientChannel::~ClientChannel()
{
  if (control -> isProtoStep3() == 0)
  {
    delete clientCache_;
    delete serverCache_;
  }

  #ifdef REFERENCES
  *logofs << "ClientChannel: Deleted object at " 
          << this << " for FD#" << fd_ << " out of "
          << --references_ << " allocated channels.\n"
          << logofs_flush;
  #endif
}

//
// Beginning of handleRead().
//

int ClientChannel::handleRead(EncodeBuffer& encodeBuffer)
{
  #ifdef TEST
  *logofs << "handleRead: Called for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Pointer to located message and
  // its size in bytes.
  //

  const unsigned char *inputMessage;
  unsigned int inputLength;

  //
  // Keep handling X messages until
  // it's time to yield.
  //

  int yield = 0;
  int loops = 0;
  int reads = 0;

  //
  // Set when message is found in
  // cache.
  //

  int hit;

  while (yield == 0)
  {

  #ifdef TEST

  *logofs << "handleRead: Initial time spent handling messages is "
          << control -> getTimeInARow() << " Ms.\n"
          << logofs_flush;

  *logofs << "handleRead: Initial bytes to write on proxy link are "
          << control -> getBytesInARow() << " bytes.\n"
          << logofs_flush;

  #endif

  //
  // Before reading from socket handle any
  // data left in the read buffer.
  //

  if (pending_ == 0)
  {
    if (readBuffer_.readMessage() < 0)
    {
      return -1;
    }
  }

  reads++;

  #if defined(INFO) || defined(TEST)
  *logofs << "handleRead: Getting messages from FD#" << fd_ << " with "
          << readBuffer_.getLength() << " bytes in the read buffer.\n"
          << logofs_flush;
  #endif

  //
  // Locate any available X message
  // and encode it.
  //

  while (yield == 0 && (inputMessage = readBuffer_.getMessage(inputLength)) != 0)
  {
    loops++;

    hit = 0;

    if (firstRequest_)
    {
      for (unsigned int i = 0; i < inputLength; i++)
      {
        encodeBuffer.encodeValue((unsigned int) inputMessage[i], 8);
      }

      firstRequest_ = 0;

      #if defined(TEST) || defined(OPCODES)

      int bits = encodeBuffer.getBits();

      *logofs << "handleRead: Handled first request. " << inputLength
              << " bytes in, " << bits << " bits (" << ((float) bits) / 8
              << " bytes) out.\n" << logofs_flush;
      #endif

      priority_++;
    }
    else
    {
      //
      // First of all we get the opcode.
      //

      unsigned char inputOpcode = *inputMessage;

      //
      // Try to short-circuit some replies at this
      // side. XSync requests, for example, weight
      // for half of the total round-trips.
      //

      if (control -> AgentTaintReplies > 0 &&
              handleTaintRequest(inputOpcode) < 0)
      {
        return -1;
      }

      encodeBuffer.encodeOpcodeValue(inputOpcode, clientCache_ -> opcodeCache);

      clientOpcode_ = inputOpcode;

      //
      // Update the current sequence.
      //

      clientSequence_++;
      clientSequence_ &= 0xffff;

      #ifdef DEBUG
      *logofs << "handleRead: Last client sequence number for FD#" 
              << fd_ << " is " << clientSequence_ << ".\n"
              << logofs_flush;
      #endif

      if (clientSequenceReset_ == 1)
      {
        encodeBuffer.encodeValue(clientSequence_, 32, 16);

        #if defined(INFO) || defined(TEST)
        *logofs << "handleRead: Reset last client sequence number to "
                << clientSequence_ << " for FD#" << fd_ << ".\n"
                << logofs_flush;
        #endif

        clientSequenceReset_ = 0;
      }

      //
      // If differential compression is disabled
      // then use the most simple encoding.
      //

      if (control -> LocalDeltaCompression == 0)
      {
        int result = handleFastReadRequest(encodeBuffer, inputOpcode,
                                               inputMessage, inputLength);
        if (result < 0)
        {
          return -1;
        }
        else if (result > 0)
        {
          if (pending_ == 0 && control -> isTimeToYield(yield_in_channel) == 1)
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleRead: Giving up because of "
                    << control -> getBytesInARow() << " bytes to write after "
                    << control -> getTimeInARow() << " Ms for FD#" << fd_
                    << ".\n" << logofs_flush;
            #endif

            yield = 1;
          }

          continue;
        }
      }

      //
      // Go to the message's specific encoding.
      //

      switch (inputOpcode)
      {
      case X_AllocColor:
        {
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_), 29,
                             clientCache_ -> colormapCache);
          const unsigned char *nextSrc = inputMessage + 8;
          unsigned int colorData[3];
          for (unsigned int i = 0; i < 3; i++)
          {
            unsigned int value = GetUINT(nextSrc, bigEndian_);
            encodeBuffer.encodeCachedValue(value, 16,
                               *(clientCache_ -> allocColorRGBCache[i]), 4);
            colorData[i] = value;
            nextSrc += 2;
          }

          sequenceQueue_.push(clientSequence_, inputOpcode,
                                  colorData[0], colorData[1], colorData[2]);

          //
          // Don't set priority in case of RDP
          // sessions. NX desktop uses a specific
          // request to pipeline multiple replies.
          //

          if (control -> SessionMode != SESSION_RDP)
          {
            priority_++;
          }
        }
        break;
      case X_ReparentWindow:
        {
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8, bigEndian_),
                             clientCache_ -> windowCache);
          encodeBuffer.encodeValue(GetUINT(inputMessage + 12, bigEndian_), 16, 11);
          encodeBuffer.encodeValue(GetUINT(inputMessage + 14, bigEndian_), 16, 11);
        }
        break;
      case X_ChangeProperty:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ChangeProperty);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          unsigned char format = inputMessage[16];
          encodeBuffer.encodeCachedValue(format, 8,
                             clientCache_ -> changePropertyFormatCache);
          unsigned int dataLength = GetULONG(inputMessage + 20, bigEndian_);
          encodeBuffer.encodeValue(dataLength, 32, 6);
          encodeBuffer.encodeValue(inputMessage[1], 2);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_), 29,
                             clientCache_ -> changePropertyPropertyCache, 9);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12, bigEndian_), 29,
                             clientCache_ -> changePropertyTypeCache, 9);
          const unsigned char *nextSrc = inputMessage + 24;
          if (format == 8)
          {
            clientCache_ -> changePropertyTextCompressor.reset();
            for (unsigned int i = 0; i < dataLength; i++)
              clientCache_ -> changePropertyTextCompressor.
                                    encodeChar(*nextSrc++, encodeBuffer);
          }
          else if (format == 32)
          {
            for (unsigned int i = 0; i < dataLength; i++)
            {
              encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_), 32,
                                 clientCache_ -> changePropertyData32Cache);
              nextSrc += 4;
            }
          }
          else
          {
            for (unsigned int i = 0; i < dataLength; i++)
            {
              encodeBuffer.encodeValue(GetUINT(nextSrc, bigEndian_), 16);
              nextSrc += 2;
            }
          }
        }
        break;
      case X_SendEvent:
        {
          //
          // TODO: This can be improved. In the worst
          // cases, it appears to provide a poor 1.6:1
          // ratio.
          //

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_SendEvent);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          unsigned int window = GetULONG(inputMessage + 4, bigEndian_);

          if (window == 0 || window == 1)
          {
            encodeBuffer.encodeBool(1);
            encodeBuffer.encodeBool(window);
          }
          else
          {
            encodeBuffer.encodeBool(0);
            encodeBuffer.encodeXidValue(window, clientCache_ -> windowCache);
          }

          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_), 32,
                             clientCache_ -> sendEventMaskCache, 9);
          encodeBuffer.encodeCachedValue(*(inputMessage + 12), 8, 
                             clientCache_ -> sendEventCodeCache);
          encodeBuffer.encodeCachedValue(*(inputMessage + 13), 8, 
                             clientCache_ -> sendEventByteDataCache);

          unsigned int newSeq = GetUINT(inputMessage + 14, bigEndian_);
          unsigned int diffSeq = newSeq - clientCache_ -> sendEventLastSequence;
          clientCache_ -> sendEventLastSequence = newSeq;
          encodeBuffer.encodeValue(diffSeq, 16, 4);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 16, bigEndian_), 32,
                             clientCache_ -> sendEventIntDataCache);

          for (unsigned int i = 20; i < 44; i++)
          {
            encodeBuffer.encodeCachedValue((unsigned int) inputMessage[i], 8,
                               clientCache_ -> sendEventEventCache);
          }
        }
        break;
      case X_ChangeWindowAttributes:
        {
          encodeBuffer.encodeValue((inputLength - 12) >> 2, 4);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          unsigned int bitmask = GetULONG(inputMessage + 8, bigEndian_);
          encodeBuffer.encodeCachedValue(bitmask, 15,
                             clientCache_ -> createWindowBitmaskCache);
          const unsigned char *nextSrc = inputMessage + 12;
          unsigned int mask = 0x1;
          for (unsigned int j = 0; j < 15; j++)
          {
            if (bitmask & mask)
            {
              encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_), 32,
                                 *clientCache_ -> createWindowAttrCache[j]);
              nextSrc += 4;
            }
            mask <<= 1;
          }
        }
        break;
      case X_ClearArea:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_ClearArea target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_ClearArea target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_ClearArea target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ClearArea);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          const unsigned char *nextSrc = inputMessage + 8;
          for (unsigned int i = 0; i < 4; i++)
          {
            encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                               *clientCache_ -> clearAreaGeomCache[i], 8);
            nextSrc += 2;
          }
        }
        break;
      case X_CloseFont:
        {
          unsigned int font = GetULONG(inputMessage + 4, bigEndian_);
          encodeBuffer.encodeValue(font - clientCache_ -> lastFont, 29, 5);
          clientCache_ -> lastFont = font;
        }
        break;
      case X_ConfigureWindow:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ConfigureWindow);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          unsigned int bitmask = GetUINT(inputMessage + 8, bigEndian_);
          encodeBuffer.encodeCachedValue(bitmask, 7,
                                  clientCache_ -> configureWindowBitmaskCache);
          unsigned int mask = 0x1;
          const unsigned char *nextSrc = inputMessage + 12;
          for (unsigned int i = 0; i < 7; i++)
          {
            if (bitmask & mask)
            {
              encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_),
                                 CONFIGUREWINDOW_FIELD_WIDTH[i],
                                     *clientCache_ -> configureWindowAttrCache[i], 8);
              nextSrc += 4;
            }
            mask <<= 1;
          }
        }
        break;
      case X_ConvertSelection:
        {
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_), 29,
                             clientCache_ -> convertSelectionRequestorCache, 9);
          const unsigned char* nextSrc = inputMessage + 8;
          for (unsigned int i = 0; i < 3; i++)
          {
            encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_), 29,
                               *(clientCache_ -> convertSelectionAtomCache[i]), 9);
            nextSrc += 4;
          }
          unsigned int timestamp = GetULONG(nextSrc, bigEndian_);
          encodeBuffer.encodeValue(timestamp -
                             clientCache_ -> convertSelectionLastTimestamp, 32, 4);
          clientCache_ -> convertSelectionLastTimestamp = timestamp;
        }
        break;
      case X_CopyArea:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_CopyArea source id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_CopyArea source id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_CopyArea source id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          t_id = GetULONG(inputMessage + 8, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_CopyArea target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_CopyArea target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_CopyArea target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_CopyArea);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 12,
                             bigEndian_), clientCache_ -> gcCache);
          const unsigned char *nextSrc = inputMessage + 16;
          for (unsigned int i = 0; i < 6; i++)
          {
            encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                               *clientCache_ -> copyAreaGeomCache[i], 8);
            nextSrc += 2;
          }
        }
        break;
      case X_CopyGC:
        {
          #ifdef TARGETS

          unsigned int s_g_id = GetULONG(inputMessage + 4, bigEndian_);
          unsigned int d_g_id = GetULONG(inputMessage + 8, bigEndian_);

          *logofs << "handleRead: X_CopyGC source gcontext id is " << s_g_id  
                  << " destination gcontext id is " << d_g_id << ".\n" 
                  << logofs_flush;

          #endif

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> gcCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12,
                             bigEndian_), 23, clientCache_ -> createGCBitmaskCache);
        }
        break;
      case X_CopyPlane:
        {
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 12,
                             bigEndian_), clientCache_ -> gcCache);
          const unsigned char *nextSrc = inputMessage + 16;
          for (unsigned int i = 0; i < 6; i++)
          {
            encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                               *clientCache_ -> copyPlaneGeomCache[i], 8);
            nextSrc += 2;
          }
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 28, bigEndian_), 32,
                                         clientCache_ -> copyPlaneBitPlaneCache, 10);
        }
        break;
      case X_CreateGC:
        {
          #ifdef TARGETS

          unsigned int g_id = GetULONG(inputMessage + 4, bigEndian_);
          unsigned int t_id = GetULONG(inputMessage + 8, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_CreateGC id " << g_id
                    << " target id is pixmap " << t_id
                    << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_CreateGC id " << g_id
                    << " target id is window " << t_id
                    << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_CreateGC id " << g_id
                    << " target id is unrecognized.\n"
                    << logofs_flush;
          }

          gcontexts.insert(T_gcontexts::value_type(g_id, t_id));

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_CreateGC);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> gcCache);
          const unsigned char *nextSrc = inputMessage + 8;
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> drawableCache);
          nextSrc += 4;
          unsigned int bitmask = GetULONG(nextSrc, bigEndian_);
          nextSrc += 4;
          encodeBuffer.encodeCachedValue(bitmask, 23,
                             clientCache_ -> createGCBitmaskCache);
          unsigned int mask = 0x1;
          for (unsigned int i = 0; i < 23; i++)
          {
            if (bitmask & mask)
            {
              unsigned int value = GetULONG(nextSrc, bigEndian_);
              nextSrc += 4;
              unsigned int fieldWidth = CREATEGC_FIELD_WIDTH[i];
              if (fieldWidth <= 4)
              {
                encodeBuffer.encodeValue(value, fieldWidth);
              }
              else
              {
                encodeBuffer.encodeCachedValue(value, fieldWidth,
                                   *clientCache_ -> createGCAttrCache[i]);
              }
            }
            mask <<= 1;
          }
        }
        break;
      case X_ChangeGC:
        {
          #ifdef TARGETS

          unsigned int g_id = GetULONG(inputMessage + 4, bigEndian_);

          T_gcontexts::iterator i = gcontexts.find(g_id);

          if (i != gcontexts.end())
          {
            unsigned int t_id = i -> second;

            if (pixmaps.find(t_id) != pixmaps.end())
            {
              *logofs << "handleRead: X_ChangeGC gcontext id is " << g_id  
                      << " target id is pixmap " << t_id << ".\n" 
                      << logofs_flush;
            }
            else if (windows.find(t_id) != windows.end())
            {
              *logofs << "handleRead: X_ChangeGC gcontext id is " << g_id  
                      << " target id is window " << t_id << ".\n" 
                      << logofs_flush;
            }
            else
            {
              *logofs << "handleRead: X_ChangeGC gcontext is " << g_id  
                      << " target id is unrecognized.\n" 
                      << logofs_flush;
            }
          }
          else
          {
            *logofs << "handleRead: X_ChangeGC gcontext id " << g_id  
                    << " is unrecognized.\n" << logofs_flush;
          }

          gcontexts.erase(g_id);

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ChangeGC);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> gcCache);
          const unsigned char *nextSrc = inputMessage + 8;
          unsigned int bitmask = GetULONG(nextSrc, bigEndian_);
          nextSrc += 4;
          encodeBuffer.encodeCachedValue(bitmask, 23,
                             clientCache_ -> createGCBitmaskCache);
          unsigned int mask = 0x1;
          for (unsigned int i = 0; i < 23; i++)
          {
            if (bitmask & mask)
            {
              unsigned int value = GetULONG(nextSrc, bigEndian_);
              nextSrc += 4;
              unsigned int fieldWidth = CREATEGC_FIELD_WIDTH[i];
              if (fieldWidth <= 4)
              {
                encodeBuffer.encodeValue(value, fieldWidth);
              }
              else
              {
                encodeBuffer.encodeCachedValue(value, fieldWidth,
                                   *clientCache_ -> createGCAttrCache[i]);
              }
            }
            mask <<= 1;
          }
        }
        break;
      case X_CreatePixmap:
        {
          #ifdef TARGETS

          *logofs << "handleRead: X_CreatePixmap depth " << (unsigned) inputMessage[1]
                  << ", pixmap id " << GetULONG(inputMessage + 4, bigEndian_)
                  << ", drawable " << GetULONG(inputMessage + 8, bigEndian_)
                  << ", width " << GetUINT(inputMessage + 12, bigEndian_)
                  << ", height " << GetUINT(inputMessage + 14, bigEndian_)
                  << ", size " << GetUINT(inputMessage + 2, bigEndian_) << 2
                  << ".\n" << logofs_flush;

          unsigned int   p_id = GetULONG(inputMessage + 4, bigEndian_);
          unsigned short p_sx = GetUINT(inputMessage + 12, bigEndian_);
          unsigned short p_sy = GetUINT(inputMessage + 14, bigEndian_);

          *logofs << "handleRead: X_CreatePixmap id is " << p_id 
                  << " width is " << p_sx << " height is " << p_sy 
                  << ".\n" << logofs_flush;

          if (p_sx * p_sy <= 64 * 64)
          {
            *logofs << "handleRead: X_CreatePixmap id " << p_id << " of size "
                    << p_sx << "x" << p_sy << "=" << p_sx * p_sy 
                    << " will be painted at client side.\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_CreatePixmap id " << p_id << " of size "
                    << p_sx << "x" << p_sy << "=" << p_sx * p_sy 
                    << " will be painted at server side.\n" << logofs_flush;
          }

          pixmaps.insert(p_id);

          #endif

          encodeBuffer.encodeCachedValue(inputMessage[1], 8,
                                         clientCache_ -> depthCache);
          unsigned int pixmap = GetULONG(inputMessage + 4, bigEndian_);
          unsigned int diff = pixmap - clientCache_ -> createPixmapLastPixmap;
          if (diff == 0)
          {
            encodeBuffer.encodeBool(1);
          }
          else
          {
            encodeBuffer.encodeBool(0);
            encodeBuffer.encodeValue(diff, 29, 4);
            clientCache_ -> createPixmapLastPixmap = pixmap;
          }
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 12,
                             bigEndian_), 16, clientCache_ -> createPixmapXCache, 8);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 14,
                             bigEndian_), 16, clientCache_ -> createPixmapYCache, 8);
        }
        break;
      case X_CreateWindow:
        {
          #ifdef TARGETS

          unsigned int w_id = GetULONG(inputMessage + 4, bigEndian_);

          *logofs << "handleRead: X_CreateWindow id is " << w_id
                  << ".\n" << logofs_flush;

          windows.insert(w_id);

          #endif

          unsigned bitmask = GetULONG(inputMessage + 28, bigEndian_);
          encodeBuffer.encodeCachedValue((unsigned int) inputMessage[1], 8,
                                         clientCache_ -> depthCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8, bigEndian_),
                             clientCache_ -> windowCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          const unsigned char *nextSrc = inputMessage + 12;
          for (unsigned int i = 0; i < 6; i++)
          {
            encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                               *clientCache_ -> createWindowGeomCache[i], 8);
            nextSrc += 2;
          }
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 24,
                                 bigEndian_), 29, clientCache_ -> visualCache);
          encodeBuffer.encodeCachedValue(bitmask, 15,
                                     clientCache_ -> createWindowBitmaskCache);
          nextSrc = inputMessage + 32;
          unsigned int mask = 0x1;
          for (unsigned int j = 0; j < 15; j++)
          {
            if (bitmask & mask)
            {
              encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_), 32,
                                 *clientCache_ -> createWindowAttrCache[j]);
              nextSrc += 4;
            }
            mask <<= 1;
          }
        }
        break;
      case X_DeleteProperty:
        {
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          encodeBuffer.encodeValue(GetULONG(inputMessage + 8, bigEndian_), 29, 9);
        }
        break;
      case X_FillPoly:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_FillPoly target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_FillPoly target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_FillPoly target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_FillPoly);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          //
          // TODO: This still obtains poor results.
          // It can be a matter of cleaning padding
          // bytes.
          //

          unsigned int numPoints = ((inputLength - 16) >> 2);
          encodeBuffer.encodeCachedValue(numPoints, 14,
                             clientCache_ -> fillPolyNumPointsCache, 4);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8, bigEndian_),
                             clientCache_ -> gcCache);
          encodeBuffer.encodeValue((unsigned int) inputMessage[12], 2);
          encodeBuffer.encodeBool((unsigned int) inputMessage[13]);
          int relativeCoordMode = (inputMessage[13] != 0);
          const unsigned char *nextSrc = inputMessage + 16;
          unsigned int pointIndex = 0;

          for (unsigned int i = 0; i < numPoints; i++)
          {
            //
            // TODO: Should encode each coordinate
            // as cached difference in respect to
            // the previous one.
            //

            if (relativeCoordMode)
            {
              encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                                 *clientCache_ -> fillPolyXRelCache[pointIndex], 8);
              nextSrc += 2;
              encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                                 *clientCache_ -> fillPolyYRelCache[pointIndex], 8);
              nextSrc += 2;
            }
            else
            {
              unsigned int x = GetUINT(nextSrc, bigEndian_);
              nextSrc += 2;
              unsigned int y = GetUINT(nextSrc, bigEndian_);
              nextSrc += 2;
              unsigned int j;
              for (j = 0; j < 8; j++)
                if ((x == clientCache_ -> fillPolyRecentX[j]) &&
                    (y == clientCache_ -> fillPolyRecentY[j]))
                  break;
              if (j < 8)
              {
                encodeBuffer.encodeBool(1);
                encodeBuffer.encodeValue(j, 3);
              }
              else
              {
                encodeBuffer.encodeBool(0);
                encodeBuffer.encodeCachedValue(x, 16,
                             *clientCache_ -> fillPolyXAbsCache[pointIndex], 8);
                encodeBuffer.encodeCachedValue(y, 16,
                             *clientCache_ -> fillPolyYAbsCache[pointIndex], 8);
                clientCache_ -> fillPolyRecentX[clientCache_ -> fillPolyIndex] = x;
                clientCache_ -> fillPolyRecentY[clientCache_ -> fillPolyIndex] = y;
                clientCache_ -> fillPolyIndex++;
                if (clientCache_ -> fillPolyIndex == 8)
                  clientCache_ -> fillPolyIndex = 0;
              }
            }

            if (control -> isProtoStep3() == 1)
            {
              if (++pointIndex == 10) pointIndex = 0;
            }
            else
            {
              if (pointIndex + 1 < 10) pointIndex++;
            }
          }
        }
        break;
      case X_FreeColors:
        {
          unsigned int numPixels = GetUINT(inputMessage + 2, bigEndian_) - 3;
          encodeBuffer.encodeValue(numPixels, 16, 4);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_), 29,
                                         clientCache_ -> colormapCache);
          encodeBuffer.encodeValue(GetULONG(inputMessage + 8, bigEndian_), 32, 4);
          const unsigned char *nextSrc = inputMessage + 12;
          while (numPixels)
          {
            encodeBuffer.encodeValue(GetULONG(nextSrc, bigEndian_), 32, 8);
            nextSrc += 4;
            numPixels--;
          }
        }
        break;
      case X_FreeCursor:
        {
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                                         29, clientCache_ -> cursorCache, 9);
        }
        break;
      case X_FreeGC:
        {
          #ifdef TARGETS

          unsigned int g_id = GetULONG(inputMessage + 4, bigEndian_);

          T_gcontexts::iterator i = gcontexts.find(g_id);

          if (i != gcontexts.end())
          {
            unsigned int t_id = i -> second;

            if (pixmaps.find(t_id) != pixmaps.end())
            {
              *logofs << "handleRead: X_FreeGC gcontext id is " << g_id  
                      << " target id is pixmap " << t_id << ".\n" 
                      << logofs_flush;
            }
            else if (windows.find(t_id) != windows.end())
            {
              *logofs << "handleRead: X_FreeGC gcontext id is " << g_id  
                      << " target id is window " << t_id << ".\n" 
                      << logofs_flush;
            }
            else
            {
              *logofs << "handleRead: X_FreeGC gcontext id is " << g_id  
                      << " target id is unrecognized.\n" 
                      << logofs_flush;
            }
          }
          else
          {
            *logofs << "handleRead: X_FreeGC gcontext id " << g_id  
                    << " is unrecognized.\n" << logofs_flush;
          }

          gcontexts.erase(g_id);

          #endif

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> gcCache);
        }
        break;
      case X_FreePixmap:
        {
          #ifdef TARGETS

          unsigned int p_id = GetULONG(inputMessage + 4, bigEndian_);

          *logofs << "handleRead: X_FreePixmap id is " << p_id << ".\n" << logofs_flush;

          pixmaps.erase(p_id);

          #endif

          unsigned int pixmap = GetULONG(inputMessage + 4, bigEndian_);
          unsigned int diff = pixmap - clientCache_ -> createPixmapLastPixmap;
          if (diff == 0)
          {
            encodeBuffer.encodeBool(1);
          }
          else
          {
            encodeBuffer.encodeBool(0);
            clientCache_ -> createPixmapLastPixmap = pixmap;
            encodeBuffer.encodeValue(diff, 29, 4);
          }
        }
        break;
      case X_GetAtomName:
        {
          encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 29, 9);

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GetGeometry:
        {
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> drawableCache);

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GetInputFocus:
        {
          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GetModifierMapping:
        {
          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GetKeyboardMapping:
        {
          encodeBuffer.encodeValue((unsigned int) inputMessage[4], 8);
          encodeBuffer.encodeValue((unsigned int) inputMessage[5], 8);

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GetProperty:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_GetProperty);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            //
            // Save a reference to identify the reply.
            //

            unsigned int property = GetULONG(inputMessage + 8, bigEndian_);

            sequenceQueue_.push(clientSequence_, inputOpcode, property);

            priority_++;

            hit = 1;

            break;
          }

          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          unsigned int property = GetULONG(inputMessage + 8, bigEndian_);
          encodeBuffer.encodeValue(property, 29, 9);
          encodeBuffer.encodeValue(GetULONG(inputMessage + 12, bigEndian_), 29, 9);
          encodeBuffer.encodeValue(GetULONG(inputMessage + 16, bigEndian_), 32, 2);
          encodeBuffer.encodeValue(GetULONG(inputMessage + 20, bigEndian_), 32, 8);

          sequenceQueue_.push(clientSequence_, inputOpcode, property);

          priority_++;
        }
        break;
      case X_GetSelectionOwner:
        {
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_), 29,
                             clientCache_ -> getSelectionOwnerSelectionCache, 9);

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GrabButton:
        {
          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 8, bigEndian_), 16,
                             clientCache_ -> grabButtonEventMaskCache);
          encodeBuffer.encodeBool((unsigned int) inputMessage[10]);
          encodeBuffer.encodeBool((unsigned int) inputMessage[11]);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12, bigEndian_), 29,
                             clientCache_ -> grabButtonConfineCache, 9);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 16, bigEndian_), 29,
                             clientCache_ -> cursorCache, 9);
          encodeBuffer.encodeCachedValue(inputMessage[20], 8,
                             clientCache_ -> grabButtonButtonCache);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 22, bigEndian_), 16,
                             clientCache_ -> grabButtonModifierCache);
        }
        break;
      case X_GrabPointer:
        {
          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 8, bigEndian_), 16,
                             clientCache_ -> grabButtonEventMaskCache);
          encodeBuffer.encodeBool((unsigned int) inputMessage[10]);
          encodeBuffer.encodeBool((unsigned int) inputMessage[11]);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12,
                           bigEndian_), 29,
                               clientCache_ -> grabButtonConfineCache, 9);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 16,
                           bigEndian_), 29, clientCache_ -> cursorCache, 9);

          unsigned int timestamp = GetULONG(inputMessage + 20, bigEndian_);
          encodeBuffer.encodeValue(timestamp -
                           clientCache_ -> grabKeyboardLastTimestamp, 32, 4);
          clientCache_ -> grabKeyboardLastTimestamp = timestamp;

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GrabKeyboard:
        {
          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          unsigned int timestamp = GetULONG(inputMessage + 8, bigEndian_);
          encodeBuffer.encodeValue(timestamp -
                             clientCache_ -> grabKeyboardLastTimestamp, 32, 4);
          clientCache_ -> grabKeyboardLastTimestamp = timestamp;
          encodeBuffer.encodeBool((unsigned int) inputMessage[12]);
          encodeBuffer.encodeBool((unsigned int) inputMessage[13]);

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GrabServer:
      case X_UngrabServer:
      case X_NoOperation:
        {
        }
        break;
      case X_PolyText8:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_PolyText8 target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_PolyText8 target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_PolyText8 target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyText8);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          int block = 8;

          if (control -> isProtoStep2() == 1)
          {
            block = 0;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);
          unsigned int x = GetUINT(inputMessage + 12, bigEndian_);
          int xDiff = x - clientCache_ -> polyTextLastX;
          clientCache_ -> polyTextLastX = x;
          encodeBuffer.encodeCachedValue(xDiff, 16,
                             clientCache_ -> polyTextCacheX, block);
          unsigned int y = GetUINT(inputMessage + 14, bigEndian_);
          int yDiff = y - clientCache_ -> polyTextLastY;
          clientCache_ -> polyTextLastY = y;
          encodeBuffer.encodeCachedValue(yDiff, 16,
                             clientCache_ -> polyTextCacheY, block);
          const unsigned char *end = inputMessage + inputLength - 1;
          const unsigned char *nextSrc = inputMessage + 16;
          while (nextSrc < end)
          {
            unsigned int textLength = (unsigned int) *nextSrc++;
            encodeBuffer.encodeBool(1);
            encodeBuffer.encodeValue(textLength, 8);
            if (textLength == 255)
            {
              encodeBuffer.encodeCachedValue(GetULONG(nextSrc, 1), 29,
                                 clientCache_ -> polyTextFontCache);
              nextSrc += 4;
            }
            else
            {
              encodeBuffer.encodeCachedValue(*nextSrc++, 8,
                                 clientCache_ -> polyTextDeltaCache);
              clientCache_ -> polyTextTextCompressor.reset();
              for (unsigned int i = 0; i < textLength; i++)
                clientCache_ -> polyTextTextCompressor.encodeChar(*nextSrc++, encodeBuffer);
            }
          }
          encodeBuffer.encodeBool(0);
        }
        break;
      case X_PolyText16:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_PolyText16 target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_PolyText16 target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_PolyText16 target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyText16);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          int block = 8;

          if (control -> isProtoStep2() == 1)
          {
            block = 0;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);
          unsigned int x = GetUINT(inputMessage + 12, bigEndian_);
          int xDiff = x - clientCache_ -> polyTextLastX;
          clientCache_ -> polyTextLastX = x;
          encodeBuffer.encodeCachedValue(xDiff, 16,
                             clientCache_ -> polyTextCacheX, block);
          unsigned int y = GetUINT(inputMessage + 14, bigEndian_);
          int yDiff = y - clientCache_ -> polyTextLastY;
          clientCache_ -> polyTextLastY = y;
          encodeBuffer.encodeCachedValue(yDiff, 16,
                             clientCache_ -> polyTextCacheY, block);
          const unsigned char *end = inputMessage + inputLength - 1;
          const unsigned char *nextSrc = inputMessage + 16;
          while (nextSrc < end)
          {
            unsigned int textLength = (unsigned int) *nextSrc++;
            encodeBuffer.encodeBool(1);
            encodeBuffer.encodeValue(textLength, 8);
            if (textLength == 255)
            {
              encodeBuffer.encodeCachedValue(GetULONG(nextSrc, 1), 29,
                                 clientCache_ -> polyTextFontCache);
              nextSrc += 4;
            }
            else
            {
              encodeBuffer.encodeCachedValue(*nextSrc++, 8,
                                 clientCache_ -> polyTextDeltaCache);
              clientCache_ -> polyTextTextCompressor.reset();
              for (unsigned int i = 0; i < textLength * 2; i++)
                clientCache_ -> polyTextTextCompressor.encodeChar(*nextSrc++, encodeBuffer);
            }
          }
          encodeBuffer.encodeBool(0);
        }
        break;
      case X_ImageText8:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_ImageText8 target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_ImageText8 target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_ImageText8 target id "
                    << t_id << " is unrecognized.\n"
                    << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ImageText8);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          int block = 8;

          if (control -> isProtoStep2() == 1)
          {
            block = 0;
          }

          unsigned int textLength = (unsigned int) inputMessage[1];
          encodeBuffer.encodeCachedValue(textLength, 8,
                             clientCache_ -> imageTextLengthCache, 4);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);
          unsigned int x = GetUINT(inputMessage + 12, bigEndian_);
          int xDiff = x - clientCache_ -> imageTextLastX;
          clientCache_ -> imageTextLastX = x;
          encodeBuffer.encodeCachedValue(xDiff, 16,
                             clientCache_ -> imageTextCacheX, block);
          unsigned int y = GetUINT(inputMessage + 14, bigEndian_);
          int yDiff = y - clientCache_ -> imageTextLastY;
          clientCache_ -> imageTextLastY = y;
          encodeBuffer.encodeCachedValue(yDiff, 16,
                             clientCache_ -> imageTextCacheY, block);
          const unsigned char *nextSrc = inputMessage + 16;
          clientCache_ -> imageTextTextCompressor.reset();
          for (unsigned int j = 0; j < textLength; j++)
            clientCache_ -> imageTextTextCompressor.encodeChar(*nextSrc++, encodeBuffer);
        }
        break;
      case X_ImageText16:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_ImageText16 target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_ImageText16 target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_ImageText16 target id "
                    << t_id << " is unrecognized.\n"
                    << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ImageText16);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          int block = 8;

          if (control -> isProtoStep2() == 1)
          {
            block = 0;
          }

          unsigned int textLength = (unsigned int) inputMessage[1];
          encodeBuffer.encodeCachedValue(textLength, 8,
                             clientCache_ -> imageTextLengthCache, 4);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);
          unsigned int x = GetUINT(inputMessage + 12, bigEndian_);
          int xDiff = x - clientCache_ -> imageTextLastX;
          clientCache_ -> imageTextLastX = x;
          encodeBuffer.encodeCachedValue(xDiff, 16,
                             clientCache_ -> imageTextCacheX, block);
          unsigned int y = GetUINT(inputMessage + 14, bigEndian_);
          int yDiff = y - clientCache_ -> imageTextLastY;
          clientCache_ -> imageTextLastY = y;
          encodeBuffer.encodeCachedValue(yDiff, 16,
                             clientCache_ -> imageTextCacheY, block);
          const unsigned char *nextSrc = inputMessage + 16;
          clientCache_ -> imageTextTextCompressor.reset();
          for (unsigned int j = 0; j < textLength * 2; j++)
            clientCache_ -> imageTextTextCompressor.encodeChar(*nextSrc++, encodeBuffer);
        }
        break;
      case X_InternAtom:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_InternAtom);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            //
            // Save a reference to identify the reply.
            //

            sequenceQueue_.push(clientSequence_, inputOpcode);

            //
            // We set priority. This could penalize those
            // well written clients using XInternAtoms()
            // to pipeline multiple replies.
            //

            priority_++;

            hit = 1;

            break;
          }

          unsigned int nameLength = GetUINT(inputMessage + 4, bigEndian_);
          encodeBuffer.encodeValue(nameLength, 16, 6);
          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          const unsigned char *nextSrc = inputMessage + 8;
          clientCache_ -> internAtomTextCompressor.reset();
          for (unsigned int i = 0; i < nameLength; i++)
          {
            clientCache_ -> internAtomTextCompressor.encodeChar(*nextSrc++, encodeBuffer);
          }

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_ListExtensions:
        {
          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_ListFonts:
        {
          unsigned int textLength = GetUINT(inputMessage + 6, bigEndian_);
          encodeBuffer.encodeValue(textLength, 16, 6);
          encodeBuffer.encodeValue(GetUINT(inputMessage + 4, bigEndian_), 16, 6);
          const unsigned char* nextSrc = inputMessage + 8;
          clientCache_ -> polyTextTextCompressor.reset();
          for (unsigned int i = 0; i < textLength; i++)
          {
            clientCache_ -> polyTextTextCompressor.encodeChar(*nextSrc++, encodeBuffer);
          }

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_LookupColor:
      case X_AllocNamedColor:
        {
          unsigned int textLength = GetUINT(inputMessage + 8, bigEndian_);
          encodeBuffer.encodeValue(textLength, 16, 6);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                                         29, clientCache_ -> colormapCache);
          const unsigned char *nextSrc = inputMessage + 12;
          clientCache_ -> polyTextTextCompressor.reset();
          for (unsigned int i = 0; i < textLength; i++)
          {
            clientCache_ -> polyTextTextCompressor.encodeChar(*nextSrc++, encodeBuffer);
          }

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_MapWindow:
      case X_UnmapWindow:
      case X_MapSubwindows:
      case X_GetWindowAttributes:
      case X_DestroyWindow:
      case X_DestroySubwindows:
      case X_QueryPointer:
      case X_QueryTree:
        {
          #ifdef TARGETS

          if (inputOpcode == X_DestroyWindow)
          {
            unsigned int w_id = GetULONG(inputMessage + 4, bigEndian_);

            *logofs << "handleRead: X_DestroyWindow id is "
                    << w_id << ".\n" << logofs_flush;

            windows.erase(w_id);
          }

          #endif

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> windowCache);
          if ((inputOpcode == X_QueryPointer) ||
              (inputOpcode == X_GetWindowAttributes) ||
              (inputOpcode == X_QueryTree))
          {
            sequenceQueue_.push(clientSequence_, inputOpcode);

            priority_++;
          }
        }
        break;
      case X_OpenFont:
        {
          unsigned int nameLength = GetUINT(inputMessage + 8, bigEndian_);
          encodeBuffer.encodeValue(nameLength, 16, 7);
          unsigned int font = GetULONG(inputMessage + 4, bigEndian_);
          encodeBuffer.encodeValue(font - clientCache_ -> lastFont, 29, 5);
          clientCache_ -> lastFont = font;
          const unsigned char *nextSrc = inputMessage + 12;
          clientCache_ -> openFontTextCompressor.reset();
          for (; nameLength; nameLength--)
          {
            clientCache_ -> openFontTextCompressor.
                  encodeChar(*nextSrc++, encodeBuffer);
          }
        }
        break;
      case X_PolyFillRectangle:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_PolyFillRectangle target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_PolyFillRectangle target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_PolyFillRectangle target id "
                    << t_id << " is unrecognized.\n"
                    << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyFillRectangle);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0;
          unsigned int lastWidth = 0, lastHeight = 0;

          //
          // TODO: Could send size at the beginning
          // instead of a bool at each iteration.
          //

          for (unsigned int i = 12; i < inputLength;)
          {
            unsigned int x = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newX = x;
            x -= lastX;
            lastX = newX;
            encodeBuffer.encodeCachedValue(x, 16,
                         *clientCache_ -> polyFillRectangleCacheX[index], 8);
            i += 2;
            unsigned int y = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newY = y;
            y -= lastY;
            lastY = newY;
            encodeBuffer.encodeCachedValue(y, 16,
                         *clientCache_ -> polyFillRectangleCacheY[index], 8);
            i += 2;
            unsigned int width = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newWidth = width;
            width -= lastWidth;
            lastWidth = newWidth;
            encodeBuffer.encodeCachedValue(width, 16,
                         *clientCache_ -> polyFillRectangleCacheWidth[index], 8);
            i += 2;
            unsigned int height = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newHeight = height;
            height -= lastHeight;
            lastHeight = newHeight;
            encodeBuffer.encodeCachedValue(height, 16,
                         *clientCache_ -> polyFillRectangleCacheHeight[index], 8);
            i += 2;

            if (++index == 4) index = 0;

            encodeBuffer.encodeBool((i < inputLength) ? 1 : 0);
          }
        }
        break;
      case X_PolyFillArc:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead:  X_PolyFillArc target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead:  X_PolyFillArc target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead:  X_PolyFillArc target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyFillArc);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0;
          unsigned int lastWidth = 0, lastHeight = 0;
          unsigned int lastAngle1 = 0, lastAngle2 = 0;

          //
          // TODO: Could send size at the beginning
          // instead of a bool at each iteration.
          //

          for (unsigned int i = 12; i < inputLength;)
          {
            unsigned int x = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newX = x;
            x -= lastX;
            lastX = newX;
            encodeBuffer.encodeCachedValue(x, 16,
                         *clientCache_ -> polyFillArcCacheX[index], 8);
            i += 2;
            unsigned int y = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newY = y;
            y -= lastY;
            lastY = newY;
            encodeBuffer.encodeCachedValue(y, 16,
                         *clientCache_ -> polyFillArcCacheY[index], 8);
            i += 2;
            unsigned int width = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newWidth = width;
            width -= lastWidth;
            lastWidth = newWidth;
            encodeBuffer.encodeCachedValue(width, 16,
                         *clientCache_ -> polyFillArcCacheWidth[index], 8);
            i += 2;
            unsigned int height = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newHeight = height;
            height -= lastHeight;
            lastHeight = newHeight;
            encodeBuffer.encodeCachedValue(height, 16,
                         *clientCache_ -> polyFillArcCacheHeight[index], 8);
            i += 2;
            unsigned int angle1 = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newAngle1 = angle1;
            angle1 -= lastAngle1;
            lastAngle1 = newAngle1;
            encodeBuffer.encodeCachedValue(angle1, 16,
                         *clientCache_ -> polyFillArcCacheAngle1[index], 8);
            i += 2;
            unsigned int angle2 = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newAngle2 = angle2;
            angle2 -= lastAngle2;
            lastAngle2 = newAngle2;
            encodeBuffer.encodeCachedValue(angle2, 16,
                         *clientCache_ -> polyFillArcCacheAngle2[index], 8);
            i += 2;

            if (++index == 2) index = 0;

            encodeBuffer.encodeBool((i < inputLength) ? 1 : 0);
          }
        }
        break;
      case X_PolyArc:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_PolyArc target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_PolyArc target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_PolyArc target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyArc);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0;
          unsigned int lastWidth = 0, lastHeight = 0;
          unsigned int lastAngle1 = 0, lastAngle2 = 0;

          //
          // TODO: Could send size at the beginning
          // instead of a bool at each iteration.
          //

          for (unsigned int i = 12; i < inputLength;)
          {
            unsigned int x = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newX = x;
            x -= lastX;
            lastX = newX;
            encodeBuffer.encodeCachedValue(x, 16,
                         *clientCache_ -> polyArcCacheX[index], 8);
            i += 2;
            unsigned int y = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newY = y;
            y -= lastY;
            lastY = newY;
            encodeBuffer.encodeCachedValue(y, 16,
                         *clientCache_ -> polyArcCacheY[index], 8);
            i += 2;
            unsigned int width = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newWidth = width;
            width -= lastWidth;
            lastWidth = newWidth;
            encodeBuffer.encodeCachedValue(width, 16,
                         *clientCache_ -> polyArcCacheWidth[index], 8);
            i += 2;
            unsigned int height = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newHeight = height;
            height -= lastHeight;
            lastHeight = newHeight;
            encodeBuffer.encodeCachedValue(height, 16,
                         *clientCache_ -> polyArcCacheHeight[index], 8);
            i += 2;
            unsigned int angle1 = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newAngle1 = angle1;
            angle1 -= lastAngle1;
            lastAngle1 = newAngle1;
            encodeBuffer.encodeCachedValue(angle1, 16,
                         *clientCache_ -> polyArcCacheAngle1[index], 8);
            i += 2;
            unsigned int angle2 = GetUINT(inputMessage + i, bigEndian_);
            unsigned int newAngle2 = angle2;
            angle2 -= lastAngle2;
            lastAngle2 = newAngle2;
            encodeBuffer.encodeCachedValue(angle2, 16,
                         *clientCache_ -> polyArcCacheAngle2[index], 8);
            i += 2;

            if (++index == 2) index = 0;

            encodeBuffer.encodeBool((i < inputLength) ? 1 : 0);
          }
        }
        break;
      case X_PolyPoint:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_PolyPoint target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_PolyPoint target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_PolyPoint target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyPoint);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeValue(GetUINT(inputMessage + 2, bigEndian_) - 3, 16, 4);
          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8, bigEndian_),
                             clientCache_ -> gcCache);
          const unsigned char *nextSrc = inputMessage + 12;

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0;

          for (unsigned int i = 12; i < inputLength; i += 4)
          {
            unsigned int x = GetUINT(nextSrc, bigEndian_);
            nextSrc += 2;
            unsigned int tmp = x;
            x -= lastX;
            lastX = tmp;
            encodeBuffer.encodeCachedValue(x, 16,
                               *clientCache_ -> polyPointCacheX[index], 8);
            unsigned int y = GetUINT(nextSrc, bigEndian_);
            nextSrc += 2;
            tmp = y;
            y -= lastY;
            lastY = tmp;
            encodeBuffer.encodeCachedValue(y, 16,
                               *clientCache_ -> polyPointCacheY[index], 8);

            if (++index == 2) index = 0;
          }
        }
        break;
      case X_PolyLine:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_PolyLine target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_PolyLine target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_PolyLine target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyLine);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeValue(GetUINT(inputMessage + 2, bigEndian_) - 3, 16, 4);
          encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);
          const unsigned char *nextSrc = inputMessage + 12;

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0;

          for (unsigned int i = 12; i < inputLength; i += 4)
          {
            unsigned int x = GetUINT(nextSrc, bigEndian_);
            nextSrc += 2;
            unsigned int tmp = x;
            x -= lastX;
            lastX = tmp;
            encodeBuffer.encodeCachedValue(x, 16,
                               *clientCache_ -> polyLineCacheX[index], 8);
            unsigned int y = GetUINT(nextSrc, bigEndian_);
            nextSrc += 2;
            tmp = y;
            y -= lastY;
            lastY = tmp;
            encodeBuffer.encodeCachedValue(y, 16,
                               *clientCache_ -> polyLineCacheY[index], 8);

            if (++index == 2) index = 0;
          }
        }
        break;
      case X_PolyRectangle:
        {
          encodeBuffer.encodeValue((GetUINT(inputMessage + 2,
                                            bigEndian_) - 3) >> 1, 16, 3);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);
          const unsigned char *end = inputMessage + inputLength;
          const unsigned char *nextSrc = inputMessage + 12;
          while (nextSrc < end)
          {
            for (unsigned int i = 0; i < 4; i++)
            {
              encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                                 *clientCache_ -> polyRectangleGeomCache[i], 8);
              nextSrc += 2;
            }
          }
        }
        break;
      case X_PolySegment:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_PolySegment target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_PolySegment target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_PolySegment target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolySegment);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          encodeBuffer.encodeValue((GetUINT(inputMessage + 2,
                                            bigEndian_) - 3) >> 1, 16, 4);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 8,
                             bigEndian_), clientCache_ -> gcCache);
          const unsigned char *end = inputMessage + inputLength;
          const unsigned char *nextSrc = inputMessage + 12;
          // unsigned int index = 0;
          // unsigned int lastX1, lastY1, lastX2, lastY2;
          while (nextSrc < end)
          {
            unsigned int x = GetUINT(nextSrc, bigEndian_);
            nextSrc += 2;
            unsigned int xDiff0 =
            x - clientCache_ -> polySegmentLastX[0];
            unsigned int xDiff1 =
            x - clientCache_ -> polySegmentLastX[1];
            int xDiff0Abs = (int) xDiff0;
            if (xDiff0Abs < 0)
              xDiff0Abs = -xDiff0Abs;
            int xDiff1Abs = (int) xDiff1;
            if (xDiff1Abs < 0)
              xDiff1Abs = -xDiff1Abs;

            unsigned int y = GetUINT(nextSrc, bigEndian_);
            nextSrc += 2;
            unsigned int yDiff0 =
            y - clientCache_ -> polySegmentLastY[0];
            unsigned int yDiff1 =
            y - clientCache_ -> polySegmentLastY[1];
            int yDiff0Abs = (int) yDiff0;
            if (yDiff0Abs < 0)
              yDiff0Abs = -yDiff0Abs;
            int yDiff1Abs = (int) yDiff1;
            if (yDiff1Abs < 0)
              yDiff1Abs = -yDiff1Abs;

            int diff0 = xDiff0Abs + yDiff0Abs;
            int diff1 = xDiff1Abs + yDiff1Abs;
            if (diff0 < diff1)
            {
              encodeBuffer.encodeBool(0);
              encodeBuffer.encodeCachedValue(xDiff0, 16,
                                         clientCache_ -> polySegmentCacheX, 6);
              encodeBuffer.encodeCachedValue(yDiff0, 16,
                                         clientCache_ -> polySegmentCacheY, 6);
            }
            else
            {
              encodeBuffer.encodeBool(1);
              encodeBuffer.encodeCachedValue(xDiff1, 16,
                                         clientCache_ -> polySegmentCacheX, 6);
              encodeBuffer.encodeCachedValue(yDiff1, 16,
                                         clientCache_ -> polySegmentCacheY, 6);
            }

            clientCache_ -> polySegmentLastX[clientCache_ -> polySegmentCacheIndex] = x;
            clientCache_ -> polySegmentLastY[clientCache_ -> polySegmentCacheIndex] = y;

            clientCache_ -> polySegmentCacheIndex =
                  clientCache_ -> polySegmentCacheIndex == 1 ? 0 : 1;
          }
        }
        break;
      case X_PutImage:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_PutImage target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_PutImage target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_PutImage target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PutImage);

          hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                 inputOpcode, inputMessage, inputLength);

          //
          // TODO: We should do a better job determine
          // how much data was actually produced for
          // the X server. Anyway, what we are really
          // interested in, is the load introduced by
          // decoding messages at the remote side.
          //

          if (wasSplitMessage(messageStore) == 1)
          {
            control -> removeOutputInARow(inputLength);
          }
        }
        break;
      case X_QueryBestSize:
        {
          encodeBuffer.encodeValue((unsigned int)inputMessage[1], 2);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          encodeBuffer.encodeValue(GetUINT(inputMessage + 8, bigEndian_), 16, 8);
          encodeBuffer.encodeValue(GetUINT(inputMessage + 10, bigEndian_), 16, 8);

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_QueryColors:
        {
          //
          // Use ZLIB encoding at startup (when dif-
          // ferential encoding is quite ineffective)
          // and differential encoding later.
          //

          if (control -> isStartup() == 0)
          {
            // Differential encoding.
            encodeBuffer.encodeBool(1);

            unsigned int numColors = ((inputLength - 8) >> 2);
            encodeBuffer.encodeValue(numColors, 16, 5);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_), 29,
                                           clientCache_ -> colormapCache);
            const unsigned char *nextSrc = inputMessage + 8;
            unsigned int predictedPixel = clientCache_ -> queryColorsLastPixel;
            for (unsigned int i = 0; i < numColors; i++)
            {
              unsigned int pixel = GetULONG(nextSrc, bigEndian_);
              nextSrc += 4;
              if (pixel == predictedPixel)
                encodeBuffer.encodeBool(1);
              else
              {
                encodeBuffer.encodeBool(0);
                encodeBuffer.encodeValue(pixel, 32, 9);
              }
              if (i == 0)
                clientCache_ -> queryColorsLastPixel = pixel;
              predictedPixel = pixel + 1;
            }
          }
          else
          {
            // Plain data compression.
            encodeBuffer.encodeBool(0);

            // Request length.
            encodeBuffer.encodeValue(GetUINT(inputMessage + 2, bigEndian_), 16, 10);

            unsigned int compressedDataSize = 0;
            unsigned char *compressedData   = NULL;

            int compressed = handleCompress(encodeBuffer, inputOpcode, inputMessage,
                                                inputLength, 4, compressedData,
                                                    compressedDataSize);
            if (compressed < 0)
            {
              return -1;
            }
          }

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_QueryExtension:
        {
          unsigned int nameLength = GetUINT(inputMessage + 4, bigEndian_);
          encodeBuffer.encodeValue(nameLength, 16, 6);
          const unsigned char *nextSrc = inputMessage + 8;
          for (; nameLength; nameLength--)
          {
            encodeBuffer.encodeValue((unsigned int) *nextSrc++, 8);
          }

          unsigned int extension = 0;

          if (strncmp((char *) inputMessage + 8, "SHAPE", 5) == 0)
          {
            extension = X_NXInternalShapeExtension;
          }
          else if (strncmp((char *) inputMessage + 8, "RENDER", 6) == 0)
          {
            extension = X_NXInternalRenderExtension;
          }

          sequenceQueue_.push(clientSequence_, inputOpcode, extension);

          priority_++;
        }
        break;
      case X_QueryFont:
        {
          unsigned int font = GetULONG(inputMessage + 4, bigEndian_);
          encodeBuffer.encodeValue(font - clientCache_ -> lastFont, 29, 5);
          clientCache_ -> lastFont = font;

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_SetClipRectangles:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_SetClipRectangles);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            hit = 1;

            break;
          }

          unsigned int numRectangles = ((inputLength - 12) >> 3);
          encodeBuffer.encodeValue(numRectangles, 13, 4);
          encodeBuffer.encodeValue((unsigned int) inputMessage[1], 2);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> gcCache);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 8, bigEndian_), 16,
                             clientCache_ -> setClipRectanglesXCache, 8);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 10, bigEndian_), 16,
                             clientCache_ -> setClipRectanglesYCache, 8);
          const unsigned char *nextSrc = inputMessage + 12;
          for (unsigned int i = 0; i < numRectangles; i++)
          {
            for (unsigned int j = 0; j < 4; j++)
            {
              encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                                 *clientCache_ -> setClipRectanglesGeomCache[j], 8);
              nextSrc += 2;
            }
          }
        }
        break;
      case X_SetDashes:
        {
          unsigned int numDashes = GetUINT(inputMessage + 10, bigEndian_);
          encodeBuffer.encodeCachedValue(numDashes, 16,
                             clientCache_ -> setDashesLengthCache, 5);
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4, bigEndian_),
                             clientCache_ -> gcCache);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 8, bigEndian_), 16,
                             clientCache_ -> setDashesOffsetCache, 5);
          const unsigned char *nextSrc = inputMessage + 12;
          for (unsigned int i = 0; i < numDashes; i++)
            encodeBuffer.encodeCachedValue(*nextSrc++, 8,
                                clientCache_ -> setDashesDashCache_[i & 1], 5);
        }
        break;
      case X_SetSelectionOwner:
        {
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_), 29,
                                    clientCache_ -> setSelectionOwnerCache, 9);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_), 29,
                           clientCache_ -> getSelectionOwnerSelectionCache, 9);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12, bigEndian_), 32,
                           clientCache_ -> setSelectionOwnerTimestampCache, 9);
        }
        break;
      case X_TranslateCoords:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_TranslateCoords source id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_TranslateCoords source id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_TranslateCoords source id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          t_id = GetULONG(inputMessage + 8, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_TranslateCoords target id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_TranslateCoords target id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_TranslateCoords target id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_TranslateCoords);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            //
            // Save a reference to identify the reply.
            //

            sequenceQueue_.push(clientSequence_, inputOpcode);

            priority_++;

            hit = 1;

            break;
          }

          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_), 29,
                                   clientCache_ -> translateCoordsSrcCache, 9);
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_), 29,
                                   clientCache_ -> translateCoordsDestCache, 9);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 12, bigEndian_), 16,
                                   clientCache_ -> translateCoordsXCache, 8);
          encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 14, bigEndian_), 16,
                                   clientCache_ -> translateCoordsYCache, 8);

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GetImage:
        {
          #ifdef TARGETS

          unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

          if (pixmaps.find(t_id) != pixmaps.end())
          {
            *logofs << "handleRead: X_GetImage source id is pixmap "
                    << t_id << ".\n" << logofs_flush;
          }
          else if (windows.find(t_id) != windows.end())
          {
            *logofs << "handleRead: X_GetImage source id is window "
                    << t_id << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "handleRead: X_GetImage source id " << t_id
                    << " is unrecognized.\n" << logofs_flush;
          }

          #endif

          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_GetImage);

          if (handleEncode(encodeBuffer, clientCache_, messageStore,
                               inputMessage, inputLength))
          {
            //
            // Save a reference to identify the reply.
            //

            sequenceQueue_.push(clientSequence_, inputOpcode);

            priority_++;

            hit = 1;

            break;
          }

          // Format.
          encodeBuffer.encodeValue((unsigned int) inputMessage[1], 2);
          // Drawable.
          encodeBuffer.encodeXidValue(GetULONG(inputMessage + 4,
                             bigEndian_), clientCache_ -> drawableCache);
          // X.
          unsigned int x = GetUINT(inputMessage + 8, bigEndian_);
          int xDiff = x - clientCache_ -> putImageLastX;
          clientCache_ -> putImageLastX = x;
          encodeBuffer.encodeCachedValue(xDiff, 16,
                             clientCache_ -> putImageXCache, 8);
          // Y.
          unsigned int y = GetUINT(inputMessage + 10, bigEndian_);
          int yDiff = y - clientCache_ -> putImageLastY;
          clientCache_ -> putImageLastY = y;
          encodeBuffer.encodeCachedValue(yDiff, 16,
                             clientCache_ -> putImageYCache, 8);
          // Width.
          unsigned int width = GetUINT(inputMessage + 12, bigEndian_);
          encodeBuffer.encodeCachedValue(width, 16,
                             clientCache_ -> putImageWidthCache, 8);
          // Height.
          unsigned int height = GetUINT(inputMessage + 14, bigEndian_);
          encodeBuffer.encodeCachedValue(height, 16,
                             clientCache_ -> putImageHeightCache, 8);
          // Plane mask.
          encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 16, bigEndian_), 32,
                             clientCache_ -> getImagePlaneMaskCache, 5);

          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GetPointerMapping:
        {
          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      case X_GetKeyboardControl:
        {
          sequenceQueue_.push(clientSequence_, inputOpcode);

          priority_++;
        }
        break;
      default:
        {
          if (inputOpcode == opcodeStore_ -> sync ||
                  inputOpcode == opcodeStore_ -> karma)
          {
            //
            // The agent's client which sent the request.
            //

            int client = (int) GetULONG(inputMessage + 4, bigEndian_);

            handleUpdateAgentClients(client);

            int notify = 0;

            if (inputOpcode == opcodeStore_ -> sync)
            {
              int mode = *(inputMessage + 1);

              #if defined(INFO) || defined(TEST)
              *logofs << "handleRead: Received agent sync request OPCODE#"
                      << (unsigned int) opcodeStore_ -> sync << " with client id "
                      << client << " and mode " << mode << ".\n"
                      << logofs_flush;
              #endif

              if (mode != NXSyncFlush)
              {
                notify = opcodeStore_ -> syncNotify;
              }

              //
              // This was a XSync(). We assume that some client
              // has finished drawing the screen. According to
              // current link configuration, server channel may
              // produce a X_GetInputFocus to force the X server
              // to flush any outstandig event. It will not send
              // the server's reply to us, so we don't need to
              // push the sequence number.
              //

              flush_++;
            }
            else
            {
              #if defined(INFO) || defined(TEST)
              *logofs << "handleRead: Received agent karma request OPCODE#"
                      << (unsigned int) opcodeStore_ -> karma << " with client id "
                      << client << ".\n" << logofs_flush;
              #endif

              notify = opcodeStore_ -> karmaNotify;
            }

            //
            // Put the notify event in the queue.
            //

            if (notify != 0)
            {
              #if defined(INFO) || defined(TEST)
              *logofs << "handleRead: Sending immediate agent notify "
                      << "event TYPE#" << notify << " to client id "
                      << client << ".\n" << logofs_flush;
              #endif

              if (handleWakeup(notify, immediate, client,
                                   nothing, nothing, nothing) < 0)
              {
                return -1;
              }
            }
          }
          else if (inputOpcode == opcodeStore_ -> shapeExtension)
          {
            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXInternalShapeExtension);

            hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                   inputOpcode, inputMessage, inputLength);
          }
          else if (inputOpcode == opcodeStore_ -> renderExtension &&
                       control -> isProtoStep3() == 1)
          {
            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXInternalRenderExtension);

            hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                   inputOpcode, inputMessage, inputLength);
          }
          else if (inputOpcode == opcodeStore_ -> putPackedImage)
          {
            #ifdef TARGETS

            unsigned int t_id = GetULONG(inputMessage + 4, bigEndian_);

            if (pixmaps.find(t_id) != pixmaps.end())
            {
              *logofs << "handleRead: X_NXPutPackedImage target id is pixmap "
                      << t_id << ".\n" << logofs_flush;
            }
            else if (windows.find(t_id) != windows.end())
            {
              *logofs << "handleRead: X_NXPutPackedImage target id is window "
                      << t_id << ".\n" << logofs_flush;
            }
            else
            {
              *logofs << "handleRead: X_NXPutPackedImage target id " << t_id
                      << " is unrecognized.\n" << logofs_flush;
            }

            #endif

            #ifdef DEBUG
            *logofs << "handleRead: Encoding packed image request for FD#"
                    << fd_ << ".\n" << logofs_flush;
            #endif

            //
            // Some packed images are unpacked to a series
            // of X_ChangeGC and X_PolyFillRectangle so we
            // need to update our sequence counter.
            //

            unsigned char method = *(inputMessage + 12);

            if (method == PACK_RDP_TEXT)
            {
              unsigned int width = GetUINT(inputMessage + 28, bigEndian_);

              unsigned int requests = (width * 2);

              clientSequence_ += requests;
              clientSequence_ &= 0xffff;

              #ifdef DEBUG
              *logofs << "handleRead: Updated last client sequence number for FD#" 
                      << fd_ << " to " << clientSequence_ << ".\n"
                      << logofs_flush;
              #endif
            }

            unsigned int outputLength = GetULONG(inputMessage + 20, bigEndian_);

            //
            // TODO: Field carries the destination data length.
            // We add, if applicable, the header's size of final
            // X_PutImage. Actually, we should move this to the
            // agent's library.
            //

            if (method != PACK_RDP_TEXT)
            {
              outputLength += 24;
            }

            if (control -> CollectStatistics)
            {
              statistics -> addPackedBytesIn(inputLength);

              statistics -> addPackedBytesOut(outputLength);
            }

            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXPutPackedImage);

            hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                   inputOpcode, inputMessage, inputLength);

            if (wasSplitMessage(messageStore) == 1)
            {
              control -> removeOutputInARow(inputLength);
            }
            else
            {
              control -> addOutputInARow(outputLength - inputLength);
            }
          }
          else if (inputOpcode == opcodeStore_ -> setUnpackColormap)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Encoding set unpack colormap request "
                    << "for FD#" << fd_ << " with size " << inputLength
                    << ".\n" << logofs_flush;
            #endif

            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXSetUnpackColormap);

            hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                   inputOpcode, inputMessage, inputLength);
          }
          else if (inputOpcode == opcodeStore_ -> setUnpackAlpha &&
                       control -> isProtoStep4() == 1)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Encoding set unpack alpha request "
                    << "for FD#" << fd_ << " with size " << inputLength
                    << ".\n" << logofs_flush;
            #endif

            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXSetUnpackAlpha);

            hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                   inputOpcode, inputMessage, inputLength);
          }
          else if (inputOpcode == opcodeStore_ -> setUnpackGeometry)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Encoding set unpack geometry request "
                    << "for FD#" << fd_ << " with size " << inputLength
                    << ".\n" << logofs_flush;
            #endif

            if (control -> isProtoStep4() == 1)
            {
              MessageStore *messageStore = clientStore_ ->
                                   getRequestStore(X_NXSetUnpackGeometry);

              hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                     inputOpcode, inputMessage, inputLength);
            }
            else
            {
              const unsigned char *nextChar = inputMessage + 4;

              for (int i = 0; i < 6; i++)
              {
                encodeBuffer.encodeCachedValue(*nextChar++, 8,
                                   clientCache_ -> depthCache);
              }

              encodeBuffer.encodeValue(GetULONG(inputMessage + 12, bigEndian_), 32);
              encodeBuffer.encodeValue(GetULONG(inputMessage + 16, bigEndian_), 32);
              encodeBuffer.encodeValue(GetULONG(inputMessage + 20, bigEndian_), 32);
            }
          }
          else if (inputOpcode == opcodeStore_ -> startSplit)
          {
            //
            // The agent's client which sent request.
            //

            int client = (int) GetULONG(inputMessage + 4, bigEndian_);

            handleUpdateAgentClients(client);

            #if defined(INFO) || defined(TEST)
            if (splitClient_ != nothing)
            {
              *logofs << "handleRead: PANIC! Client id " << client
                      << " requested splits while handling client id "
                      << splitClient_ << ".\n" << logofs_flush;
            }
            #endif

            splitClient_ = client;

            #if defined(INFO) || defined(TEST)
            *logofs << "handleRead: Registered client id " << client
                    << " as waiting for notify event TYPE#"
                    << (unsigned int) opcodeStore_ -> splitNotify << ".\n"
                    << logofs_flush;
            #endif
          }
          else if (inputOpcode == opcodeStore_ -> endSplit)
          {
            //
            // The agent's client which sent the request.
            //

            int client = (int) GetULONG(inputMessage + 4, bigEndian_);

            handleUpdateAgentClients(client);

            if (splitClient_ == nothing)
            {
              #ifdef WARNING
              *logofs << "handleRead: WARNING! Received end of split while "
                      << "no client id was currently selected.\n"
                      << logofs_flush;
              #endif
            }
            else if (splitClient_ != client)
            {
              #ifdef WARNING
              *logofs << "handleRead: WARNING! Received end of split for "
                      << "client id " << client << " while waiting for "
                      << "client id " << splitClient_ << ".\n"
                      << logofs_flush;
              #endif
            }

            //
            // At the time we receive an end of split, we mark the
            // last split added as the one ending the row for this
            // client. If we can't mark the split it means we did
            // not add any split to the repository and we can thus
            // restart the client immediately.
            //

            if (clientStore_ -> getSplitStore() -> markCompletion() == 0)
            {
              #if defined(INFO) || defined(TEST)
              *logofs << "handleRead: Sending immediate agent split "
                      << "event TYPE#" << (unsigned int) opcodeStore_ ->
                         splitNotify << " to wakeup " << "client id "
                      << client << ".\n" << logofs_flush;
              #endif

              if (handleWakeup(opcodeStore_ -> splitNotify, immediate,
                                   nothing, nothing, nothing, client) < 0)
              {
                return -1;
              }
            }

            #if defined(INFO) || defined(TEST)
            *logofs << "handleRead: Reset id " << client << " as client id "
                    << "selected for splits.\n" << logofs_flush;
            #endif

            splitClient_ = nothing;
          }
          else if (inputOpcode == opcodeStore_ -> commitSplit)
          {
            if (handleCommit(encodeBuffer, inputMessage, inputLength) < 0)
            {
              return -1;
            }
          }
          else if (inputOpcode == opcodeStore_ -> freeUnpack &&
                       control -> isProtoStep4() == 1)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Encoding free unpack request "
                    << "for FD#" << fd_ << " with size " << inputLength
                    << ".\n" << logofs_flush;
            #endif

            encodeBuffer.encodeCachedValue(*(inputMessage + 1), 8,
                         clientCache_ -> resourceCache);
          }
          else if (inputOpcode == opcodeStore_ -> getControlParameters)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Encoding get control parameters "
                    << "request for FD#" << fd_ << " with size "
                    << inputLength << ".\n" << logofs_flush;
            #endif

            //
            // Add the reply to the write buffer. If found
            // to contain a message, it it will be flushed
            // to the X client before leaving the loop.
            //

            unsigned char *reply = writeBuffer_.addMessage(32);

            *(reply + 0) = X_Reply;

            PutUINT(clientSequence_, reply + 2, bigEndian_);

            PutULONG(0, reply + 4, bigEndian_);

            //
            // Save the sequence number we used
            // to auto-generate this reply.
            //

            lastSequence_ = clientSequence_;

            #if defined(INFO) || defined(TEST)
            *logofs << "handleRead: Registered " << lastSequence_
                    << " as last auto-generated sequence number.\n"
                    << logofs_flush;
            #endif

            *(reply + 8) = control -> LinkMode;

            *(reply + 9) = control -> LocalVersionMajor;
            *(reply + 10) = control -> LocalVersionMinor;
            *(reply + 11) = control -> LocalVersionPatch;

            PutULONG(control -> AgentKarmaSize, reply + 12, bigEndian_);
            PutULONG(control -> AgentSplitSize, reply + 16, bigEndian_);

            *(reply + 20) = control -> AgentPackMethod;
            *(reply + 21) = control -> AgentPackQuality;

            *(reply + 22) = control -> LocalDataCompressionLevel;
            *(reply + 23) = control -> LocalStreamCompressionLevel;
            *(reply + 24) = control -> LocalDeltaCompression;

            *(reply + 25) = (control -> LocalDeltaCompression == 1 &&
                                 control -> PersistentCacheEnableLoad == 1);
            *(reply + 26) = (control -> LocalDeltaCompression == 1 &&
                                 control -> PersistentCacheEnableSave == 1);
            *(reply + 27) = (control -> LocalDeltaCompression == 1 &&
                                 control -> PersistentCacheEnableLoad == 1 &&
                                     control -> PersistentCacheName != NULL);

            if (handleFlush(flush_if_any) < 0)
            {
              return -1;
            }
          }
          else if (inputOpcode == opcodeStore_ -> getCleanupParameters)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Encoding get cleanup parameters "
                    << "request for FD#" << fd_ << " with size "
                    << inputLength << ".\n" << logofs_flush;
            #endif

            //
            // Add a new message to the write buffer.
            //

            unsigned char *reply = writeBuffer_.addMessage(32);

            *(reply + 0) = X_Reply;

            PutUINT(clientSequence_, reply + 2, bigEndian_);

            PutULONG(0, reply + 4, bigEndian_);

            //
            // Save the sequence number we used
            // to auto-generate this reply.
            //

            lastSequence_ = clientSequence_;

            #if defined(INFO) || defined(TEST)
            *logofs << "handleRead: Registered " << lastSequence_
                    << " as last auto-generated sequence number.\n"
                    << logofs_flush;
            #endif

            *(reply + 8)  = control -> AgentCleanGet;
            *(reply + 9)  = control -> AgentCleanAlloc;
            *(reply + 10) = control -> AgentCleanFlush;
            *(reply + 11) = control -> AgentCleanSend;
            *(reply + 12) = control -> AgentCleanImages;

            if (handleFlush(flush_if_any) < 0)
            {
              return -1;
            }
          }
          else if (inputOpcode == opcodeStore_ -> getImageParameters)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Encoding get image parameters "
                    << "request for FD#" << fd_ << " with size "
                    << inputLength << ".\n" << logofs_flush;
            #endif

            unsigned char *reply = writeBuffer_.addMessage(32);

            *(reply + 0) = X_Reply;

            PutUINT(clientSequence_, reply + 2, bigEndian_);

            PutULONG(0, reply + 4, bigEndian_);

            lastSequence_ = clientSequence_;

            #if defined(INFO) || defined(TEST)
            *logofs << "handleRead: Registered " << lastSequence_
                    << " as last auto-generated sequence number.\n"
                    << logofs_flush;
            #endif

            *(reply + 8)  = control -> AgentImageSplit;
            *(reply + 9)  = control -> AgentImageMask;
            *(reply + 10) = control -> AgentImageFrame;
            *(reply + 11) = control -> AgentImageMaskMethod;
            *(reply + 12) = control -> AgentImageSplitMethod;

            if (handleFlush(flush_if_any) < 0)
            {
              return -1;
            }
          }
          else if (inputOpcode == opcodeStore_ -> getUnpackParameters)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Encoding get unpack parameters "
                    << "request for FD#" << fd_ << " with size "
                    << inputLength << ".\n" << logofs_flush;
            #endif

            sequenceQueue_.push(clientSequence_, inputOpcode);
          }
          else if (inputOpcode == opcodeStore_ -> getShmemParameters)
          {
            //
            // Let the operation fail with a X error
            // if the remote proxy only supports an
            // older protocol versions.
            //

            if (control -> isProtoStep4() == 1)
            {
              unsigned int stage;

              if (handleShmemInit(encodeBuffer, inputOpcode, stage,
                                      inputMessage, inputLength) < 0)
              {
                return -1;
              }

              if (stage != 1)
              {
                sequenceQueue_.push(clientSequence_, inputOpcode);

                priority_++;
              }
            }
            else
            {
              #ifdef WARNING
              *logofs << "handleRead: WARNING! Encoding unsupported shmem "
                      << "request OPCODE#" << (unsigned int) inputOpcode
                      << " for FD#" << fd_ << " as generic request.\n"
                      << logofs_flush;
              #endif

              MessageStore *messageStore = clientStore_ ->
                                   getRequestStore(X_NXInternalGenericRequest);

              hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                     inputOpcode, inputMessage, inputLength);
            }
          }
          else if (inputOpcode == opcodeStore_ -> setExposeEvents)
          {
            //
            // Enable or disable expose events
            // coming from the real server.
            //

            encodeBuffer.encodeBool(*(inputMessage + 4));
            encodeBuffer.encodeBool(*(inputMessage + 5));
            encodeBuffer.encodeBool(*(inputMessage + 6));
          }
          else
          {
            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXInternalGenericRequest);

            hit = handleEncode(encodeBuffer, clientCache_, messageStore,
                                   inputOpcode, inputMessage, inputLength);

            //
            // Don't flush if opcode is unrecognized
            // but flush if it is an extension.
            //

            if (control -> isProtoStep3() == 1 &&
                    inputOpcode > 127)
            {
              priority_++;
            }
          }
        }
      } // End of switch on opcode.

      int bits = encodeBuffer.getBits();

      #if defined(TEST) || defined(OPCODES)

      const char *cacheString = (hit ? "cached " : "");

      *logofs << "handleRead: Handled " << cacheString << "request OPCODE#" 
              << (unsigned int) inputOpcode << " (" << DumpOpcode(inputOpcode)
              << ")" << " for FD#" << fd_ << " sequence " << clientSequence_
              << ". " << inputLength  << " bytes in, " << bits << " bits ("
              << ((float) bits) / 8 << " bytes) out.\n" << logofs_flush;

      #endif

      if (control -> CollectStatistics)
      {
        if (hit)
        {
          statistics -> addCachedRequest(inputOpcode);
        }

        statistics -> addRequestBits(inputOpcode, inputLength << 3, bits);
      }

      control -> addOutputInARow(inputLength);

      control -> addBitsInARow(bits);

    }  // End if (firstRequest_)... else ...

    //
    // If we consumed too many ticks or produced
    // enough data, then leave remaining requests
    // in the read buffer.
    //

    if (pending_ == 0 && control -> isTimeToYield(yield_in_channel) == 1)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up because of "
              << control -> getBytesInARow() << " bytes to write after "
              << control -> getTimeInARow() << " Ms for FD#" << fd_
              << ".\n" << logofs_flush;
      #endif

      yield = 1;
    }

  } // End of while (yield == 0 && (inputMessage = readBuffer_.getMessage(inputLength)) != 0) ...

  //
  // Set pending flag if we quit the loop before
  // consuming all messages and bytes left in the
  // read buffer are considered enough to make a
  // complete message.
  //

  if (yield == 1)
  {
    pending_ = readBuffer_.checkMessage();
  }
  else
  {
    //
    // Give up if we just needed to handle
    // messages left in the read buffer
    //

    if (pending_ > 0)
    {
      pending_ = 0;

      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up because of no more "
              << "pending messages for FD#" << fd_
              << ".\n" << logofs_flush;
      #endif

      yield = 1;
    }

    //
    // Don't perform more reads if there
    // is prioritized data.
    //

    else if (priority_ > 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up with " << priority_
              << " prioritized messages for FD#" << fd_
              << ".\n" << logofs_flush;
      #endif

      yield = 1;
    }

    //
    // Check if we read enough data.
    //

    else if (reads >= control -> ClientReadsInARowLimit ||
                loops >= control -> ClientMessagesInARowLimit)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up because of limit of "
              << control -> ClientReadsInARowLimit << " reads and "
              << control -> ClientMessagesInARowLimit
              << " messages for FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      yield = 1;
    }

    //
    // If it's not time to yield, check if
    // we can read more bytes from socket.
    //

    else if (transport_ -> readable() <= 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up with no bytes "
              << "available on FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      yield = 1;
    }

    #if defined(INFO) || defined(TEST)

    if (yield == 1)
    {
      *logofs << "handleRead: Performed " << reads << " reads "
              << "and handled " << loops << " messages producing "
              << control -> getBytesInARow() << " bytes in "
              << control -> getTimeInARow() << " Ms for FD#"
              << fd_ << ".\n" << logofs_flush;
    }
    else
    {
      *logofs << "handleRead: Still reading messages with "
              << transport_ -> readable() << " bytes available "
              << "for FD#" << fd_ << ".\n" << logofs_flush;
    }

    #endif

  } // End of if (yield == 1) ... else ...

  } // End of while (yield == 0) ...

  //
  // Reset the read buffer.
  //

  readBuffer_.partialReset();

  return 1;
}

//
// End of handleRead().
//

//
// Beginning of handleWrite().
//

int ClientChannel::handleWrite(const unsigned char *message, unsigned int length)
{
  #ifdef TEST
  *logofs << "handleWrite: Called for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Create the buffer from which to 
  // decode messages.
  //

  DecodeBuffer decodeBuffer(message, length);

  #if defined(INFO) || defined(TEST)
  *logofs << "handleWrite: Cumulative length of decode "
          << "buffer is " << length << " bytes.\n"
          << logofs_flush;
  #endif

  if (firstReply_)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "handleWrite: First reply detected.\n" << logofs_flush;
    #endif

    //
    // NX client needs this line to consider
    // the initialization phase successfully
    // completed.
    //

    if (firstAgent_)
    {
      cerr << "Info" << ": Established X client connection.\n" ;

      firstAgent_ = 0;
    }

    unsigned int outputOpcode;

    decodeBuffer.decodeValue(outputOpcode, 8);
    unsigned int secondByte;
    decodeBuffer.decodeValue(secondByte, 8);
    unsigned int major;
    decodeBuffer.decodeValue(major, 16);
    unsigned int minor;
    decodeBuffer.decodeValue(minor, 16);
    unsigned int extraLength;
    decodeBuffer.decodeValue(extraLength, 16);
    unsigned int outputLength = 8 + (extraLength << 2);

    unsigned char *outputMessage = writeBuffer_.addMessage(outputLength);
    *outputMessage = (unsigned char) outputOpcode;
    outputMessage[1] = (unsigned char) secondByte;
    PutUINT(major, outputMessage + 2, bigEndian_);
    PutUINT(minor, outputMessage + 4, bigEndian_);
    PutUINT(extraLength, outputMessage + 6, bigEndian_);
    unsigned char *nextDest = outputMessage + 8;
    unsigned int cached;
    decodeBuffer.decodeBool(cached);

    if (cached)
    {
      memcpy(nextDest, ServerCache::lastInitReply.getData(), outputLength - 8);
    }
    else
    {
      for (unsigned i = 8; i < outputLength; i++)
      {
        unsigned int nextByte;
        decodeBuffer.decodeValue(nextByte, 8);
        *nextDest++ = (unsigned char) nextByte;
      }

      ServerCache::lastInitReply.set(outputLength - 8, outputMessage + 8);
    }

    imageByteOrder_ = outputMessage[30];
    bitmapBitOrder_ = outputMessage[31];
    scanlineUnit_   = outputMessage[32];
    scanlinePad_    = outputMessage[33];

    firstReply_ = 0;

  } // End of if (firstReply_)
  else
  {
    #ifdef DEBUG
    *logofs << "handleWrite: Starting loop on opcodes.\n"
            << logofs_flush;
    #endif

    unsigned char outputOpcode;

    //
    // Set if we should flush immediately
    // any data accumulated in the write
    // buffer.
    //

    unsigned int outputFlush = 0;
  
    while (decodeBuffer.decodeOpcodeValue(outputOpcode, serverCache_ -> opcodeCache, 1))
    {
      serverOpcode_ = outputOpcode;

      #ifdef DEBUG
      *logofs << "handleWrite: Decoded a new OPCODE#"
              << (unsigned int) outputOpcode << ".\n"
              << logofs_flush;
      #endif

      unsigned char *outputMessage = NULL;
      unsigned int outputLength    = 0;

      //
      // General-purpose temp variables 
      // for decoding ints and chars.
      //

      unsigned int value   = 0;
      unsigned char cValue = 0;

      //
      // Check first if we need to abort any split,
      // then if this is a reply, finally if it is
      // en event or error.
      //

      if (outputOpcode == opcodeStore_ -> abortSplit)
      {
        //
        // It's an abort split, not a normal
        // burst of proxy data.
        //

        handleAbortSplit(decodeBuffer);

        continue;
      }
      else if (outputOpcode == X_Reply)
      {
        #ifdef DEBUG
        *logofs << "handleWrite: Decoding sequence number of reply.\n"
                << logofs_flush;
        #endif

        unsigned int sequenceNum;
        unsigned int sequenceDiff;

        if (serverSequenceReset_ == 1)
        {
          decodeBuffer.decodeValue(sequenceNum, 32, 16);

          #if defined(INFO) || defined(TEST)
          *logofs << "handleWrite: Reset last server sequence number to "
                  << sequenceNum << " for FD#" << fd_ << ".\n"
                  << logofs_flush;
          #endif

          serverSequenceReset_ = 0;
        }
        else
        {
          decodeBuffer.decodeCachedValue(sequenceDiff, 16,
                             serverCache_ -> replySequenceCache, 7);

          sequenceNum = (serverSequence_ + sequenceDiff) & 0xffff;
        }

        serverSequence_ = sequenceNum;

        #ifdef DEBUG
        *logofs << "handleWrite: Last server sequence number for FD#" 
                << fd_ << " is " << serverSequence_ << " with "
                << "difference " << sequenceDiff << ".\n"
                << logofs_flush;
        #endif

        //
        // In case of reply we can follow the X server and
        // override any event's sequence number generated
        // by this side.
        //

        #ifdef TEST
        *logofs << "handleWrite: Updating last event's sequence "
                << lastSequence_ << " to reply's sequence number "
                << serverSequence_ << " for FD#" << fd_ << ".\n"
                << logofs_flush;
        #endif

        lastSequence_ = serverSequence_;

        unsigned short int requestSequenceNum;
        unsigned char requestOpcode;

        #ifdef DEBUG
        *logofs << "handleWrite: Peek of sequence number returned "
                << sequenceQueue_.peek(requestSequenceNum, requestOpcode)
                << " with sequence " << requestSequenceNum << " and opcode "
                << (unsigned int) requestOpcode << ".\n" << logofs_flush;
        #endif

        if (sequenceQueue_.peek(requestSequenceNum, requestOpcode) &&
                (requestSequenceNum == sequenceNum))
        {
          unsigned int requestData[3];

          sequenceQueue_.pop(requestSequenceNum, requestOpcode,
                                    requestData[0], requestData[1], requestData[2]);

          #ifdef DEBUG
          *logofs << "handleWrite: Identified reply to OPCODE#"
                  << (unsigned int) requestOpcode << ".\n"
                  << logofs_flush;
          #endif

          //
          // Is differential encoding disabled?
          //

          if (control -> RemoteDeltaCompression == 0)
          {
            int result = handleFastWriteReply(decodeBuffer, requestOpcode,
                                                  outputMessage, outputLength);
            if (result < 0)
            {
              return -1;
            }
            else if (result > 0)
            {
              continue;
            }
          }

          switch (requestOpcode)
          {
          case X_AllocColor:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              unsigned char *nextDest = outputMessage + 8;
              for (unsigned int i = 0; i < 3; i++)
              {
                decodeBuffer.decodeBool(value);
                if (value)
                {
                  PutUINT(requestData[i], nextDest, bigEndian_);
                }
                else
                {
                  decodeBuffer.decodeValue(value, 16, 6);
                  PutUINT(requestData[i] + value, nextDest, bigEndian_);
                }
                nextDest += 2;
              }
              decodeBuffer.decodeValue(value, 32, 9);
              PutULONG(value, outputMessage + 16, bigEndian_);
            }
            break;
          case X_GetAtomName:
            {
              unsigned int nameLength;
              decodeBuffer.decodeValue(nameLength, 16, 6);
              outputLength = RoundUp4(nameLength) + 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              PutUINT(nameLength, outputMessage + 8, bigEndian_);
              unsigned char* nextDest = outputMessage + 32;

              if (control -> isProtoStep3() == 1)
              {
                serverCache_ -> getAtomNameTextCompressor.reset();
                for (unsigned int i = 0; i < nameLength; i++)
                {
                  *nextDest++ = serverCache_ -> getAtomNameTextCompressor.
                                      decodeChar(decodeBuffer);
                }
              }
              else
              {
                //
                // Should not decode data using the
                // local side cache. It didn't cause
                // problems because cache was used
                // per-channel.
                //

                clientCache_ -> internAtomTextCompressor.reset();
                for (unsigned int i = 0; i < nameLength; i++)
                {
                  *nextDest++ = clientCache_ -> internAtomTextCompressor.
                                      decodeChar(decodeBuffer);
                }
              }
            }
            break;
          case X_GetGeometry:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 serverCache_ -> depthCache);
              outputMessage[1] = cValue;
              decodeBuffer.decodeCachedValue(value, 29,
                                 serverCache_ -> getGeometryRootCache, 9);
              PutULONG(value, outputMessage + 8, bigEndian_);
              unsigned char *nextDest = outputMessage + 12;
              for (unsigned int i = 0; i < 5; i++)
              {
                decodeBuffer.decodeCachedValue(value, 16,
                                   *serverCache_ -> getGeometryGeomCache[i], 8);
                PutUINT(value, nextDest, bigEndian_);
                nextDest += 2;
              }
            }
            break;
          case X_GetInputFocus:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeValue(value, 2);
              outputMessage[1] = (unsigned char) value;
              decodeBuffer.decodeCachedValue(value, 29,
                                 serverCache_ -> getInputFocusWindowCache);
              PutULONG(value, outputMessage + 8, bigEndian_);
            }
            break;
          case X_GetKeyboardMapping:
            {
              decodeBuffer.decodeBool(value);
              if (value)
              {
                unsigned int dataLength =
                ServerCache::getKeyboardMappingLastMap.getLength();
                outputLength = 32 + dataLength;
                outputMessage = writeBuffer_.addMessage(outputLength);
                outputMessage[1] =
                ServerCache::getKeyboardMappingLastKeysymsPerKeycode;
                memcpy(outputMessage + 32,
                           ServerCache::getKeyboardMappingLastMap.getData(),
                               dataLength);
                break;
              }
              unsigned int numKeycodes;
              decodeBuffer.decodeValue(numKeycodes, 8);
              unsigned int keysymsPerKeycode;
              decodeBuffer.decodeValue(keysymsPerKeycode, 8, 4);
              ServerCache::getKeyboardMappingLastKeysymsPerKeycode =
                                  keysymsPerKeycode;
              outputLength = 32 + numKeycodes * keysymsPerKeycode * 4;
              outputMessage = writeBuffer_.addMessage(outputLength);
              outputMessage[1] = (unsigned char) keysymsPerKeycode;
              unsigned char *nextDest = outputMessage + 32;
              unsigned char previous = 0;
              for (unsigned int count = numKeycodes * keysymsPerKeycode;
                       count; --count)
              {
                decodeBuffer.decodeBool(value);
                if (value)
                  PutULONG((unsigned int) NoSymbol, nextDest, bigEndian_);
                else
                {
                  unsigned int keysym;
                  decodeBuffer.decodeCachedValue(keysym, 24,
                             serverCache_ -> getKeyboardMappingKeysymCache, 9);
                  decodeBuffer.decodeCachedValue(cValue, 8,
                           serverCache_ -> getKeyboardMappingLastByteCache, 5);
                  previous += cValue;
                  PutULONG((keysym << 8) | previous, nextDest, bigEndian_);
                }
                nextDest += 4;
              }
              ServerCache::getKeyboardMappingLastMap.set(outputLength - 32,
                                                             outputMessage + 32);
            }
            break;
          case X_GetModifierMapping:
            {
              unsigned int keycodesPerModifier;
              decodeBuffer.decodeValue(keycodesPerModifier, 8);
              outputLength = 32 + (keycodesPerModifier << 3);
              outputMessage = writeBuffer_.addMessage(outputLength);
              outputMessage[1] = (unsigned char) keycodesPerModifier;
              unsigned char *nextDest = outputMessage + 32;
              decodeBuffer.decodeBool(value);
              if (value)
              {
                memcpy(outputMessage + 32,
                           ServerCache::getModifierMappingLastMap.getData(),
                               ServerCache::getModifierMappingLastMap.getLength());
                break;
              }
              for (unsigned int count = outputLength - 32; count; count--)
              {
                decodeBuffer.decodeBool(value);
                if (value)
                  *nextDest++ = 0;
                else
                {
                  decodeBuffer.decodeValue(value, 8);
                  *nextDest++ = value;
                }
              }
              ServerCache::getModifierMappingLastMap.set(outputLength - 32,
                                                             outputMessage + 32);
            }
            break;
          case X_GetProperty:
            {
              if (control -> isProtoStep3() == 1)
              {
                MessageStore *messageStore = serverStore_ ->
                                     getReplyStore(X_GetProperty);

                handleDecode(decodeBuffer, serverCache_, messageStore,
                                 requestOpcode, outputMessage, outputLength);

                break;
              }

              unsigned char format;
              decodeBuffer.decodeCachedValue(format, 8,
                                       serverCache_ -> getPropertyFormatCache);
              unsigned int length;
              decodeBuffer.decodeValue(length, 32, 9);
              unsigned int numBytes = length;
              if (format == 16)
                numBytes <<= 1;
              else if (format == 32)
                numBytes <<= 2;
              outputLength = 32 + RoundUp4(numBytes);
              outputMessage = writeBuffer_.addMessage(outputLength);
              outputMessage[1] = format;
              PutULONG(length, outputMessage + 16, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 29,
                                      serverCache_ -> getPropertyTypeCache, 9);
              PutULONG(value, outputMessage + 8, bigEndian_);
              decodeBuffer.decodeValue(value, 32, 9);
              PutULONG(value, outputMessage + 12, bigEndian_);
              unsigned char *nextDest = outputMessage + 32;
              if (format == 8 && numBytes < SERVER_TEXT_CACHE_SIZE)
              {
                if (requestData[0] == XA_RESOURCE_MANAGER)
                {
                  decodeBuffer.decodeBool(value);
                  if (value)
                  {
                    memcpy(nextDest, ServerCache::xResources.getData(),
                           ServerCache::xResources.getLength());
                    break;
                  }
                }
                serverCache_ -> getPropertyTextCompressor.reset();
                for (unsigned int i = 0; i < numBytes; i++)
                {
                  unsigned char nextChar;
                  nextChar = *nextDest++ =
                    serverCache_ -> getPropertyTextCompressor.decodeChar(
                                                              decodeBuffer);
                  if (nextChar == 10)
                  {
                    serverCache_ -> getPropertyTextCompressor.reset(nextChar);
                  }
                }
                if (requestData[0] == XA_RESOURCE_MANAGER)
                  ServerCache::xResources.set(numBytes, outputMessage + 32);
              }
              else
              {
                const unsigned char *compressedData = NULL;
                unsigned int compressedDataSize = 0;

                int decompressed = handleDecompress(decodeBuffer, requestOpcode, outputMessage,
                                                        outputLength, 32, compressedData,
                                                            compressedDataSize);
                if (decompressed < 0)
                {
                  return -1;
                }
              }
            }
            break;
          case X_GetSelectionOwner:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeCachedValue(value, 29,
                                    serverCache_ -> getSelectionOwnerCache, 9);
              PutULONG(value, outputMessage + 8, bigEndian_);
            }
            break;
          case X_GetWindowAttributes:
            {
              outputLength = 44;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeValue(value, 2);
              outputMessage[1] = (unsigned char) value;
              decodeBuffer.decodeCachedValue(value, 29,
                                             serverCache_ -> visualCache);
              PutULONG(value, outputMessage + 8, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                             serverCache_ -> getWindowAttributesClassCache, 3);
              PutUINT(value, outputMessage + 12, bigEndian_);
              decodeBuffer.decodeCachedValue(cValue, 8,
                           serverCache_ -> getWindowAttributesBitGravityCache);
              outputMessage[14] = cValue;
              decodeBuffer.decodeCachedValue(cValue, 8,
                           serverCache_ -> getWindowAttributesWinGravityCache);
              outputMessage[15] = cValue;
              decodeBuffer.decodeCachedValue(value, 32,
                            serverCache_ -> getWindowAttributesPlanesCache, 9);
              PutULONG(value, outputMessage + 16, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 32,
                             serverCache_ -> getWindowAttributesPixelCache, 9);
              PutULONG(value, outputMessage + 20, bigEndian_);
              decodeBuffer.decodeBool(value);
              outputMessage[24] = (unsigned char) value;
              decodeBuffer.decodeBool(value);
              outputMessage[25] = (unsigned char) value;
              decodeBuffer.decodeValue(value, 2);
              outputMessage[26] = (unsigned char) value;
              decodeBuffer.decodeBool(value);
              outputMessage[27] = (unsigned char) value;
              decodeBuffer.decodeCachedValue(value, 29,
                                             serverCache_ -> colormapCache, 9);
              PutULONG(value, outputMessage + 28, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 32,
                            serverCache_ -> getWindowAttributesAllEventsCache);
              PutULONG(value, outputMessage + 32, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 32,
                           serverCache_ -> getWindowAttributesYourEventsCache);
              PutULONG(value, outputMessage + 36, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                        serverCache_ -> getWindowAttributesDontPropagateCache);
              PutUINT(value, outputMessage + 40, bigEndian_);
            }
            break;
          case X_GrabKeyboard:
          case X_GrabPointer:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeValue(value, 3);
              outputMessage[1] = (unsigned char) value;
            }
            break;
          case X_InternAtom:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeValue(value, 29, 9);
              PutULONG(value, outputMessage + 8, bigEndian_);
            }
            break;
          case X_ListExtensions:
            {
              decodeBuffer.decodeValue(value, 32, 8);
              outputLength = 32 + (value << 2);
              outputMessage = writeBuffer_.addMessage(outputLength);
              unsigned int numExtensions;
              decodeBuffer.decodeValue(numExtensions, 8);
              outputMessage[1] = (unsigned char) numExtensions;
              unsigned char *nextDest = outputMessage + 32;
              for (; numExtensions; numExtensions--)
              {
                unsigned int length;
                decodeBuffer.decodeValue(length, 8);
                *nextDest++ = (unsigned char) length;
                for (; length; length--)
                {
                  decodeBuffer.decodeValue(value, 8);
                  *nextDest++ = value;
                }
              }
            }
            break;
          case X_ListFonts:
            {
              //
              // Differential compression can achieve a 12:1 to 14:1
              // ratio, while the best ZLIB compression can achieve
              // a mere 4:1 to 5:1. In the first case, though, the
              // huge amount of data constituting the message would
              // be stored uncompressed at the remote side. We need
              // to find a compromise. The solution is to use diffe-
              // rential compression at startup and ZLIB compression
              // later on.
              //

              MessageStore *messageStore = serverStore_ ->
                                   getReplyStore(X_ListFonts);

              if (handleDecode(decodeBuffer, serverCache_, messageStore,
                                   outputMessage, outputLength))
              {
                break;
              }

              decodeBuffer.decodeValue(value, 32, 8);
              outputLength = 32 + (value << 2);
              outputMessage = writeBuffer_.addMessage(outputLength);
              unsigned int numFonts;
              decodeBuffer.decodeValue(numFonts, 16, 6);
              PutUINT(numFonts, outputMessage + 8, bigEndian_);

              // Differential or plain data compression?
              decodeBuffer.decodeBool(value);

              if (value)
              {
                unsigned char* nextDest = outputMessage + 32;
                for (; numFonts; numFonts--)
                {
                  unsigned int length;
                  decodeBuffer.decodeValue(length, 8);
                  *nextDest++ = (unsigned char)length;
                  serverCache_ -> getPropertyTextCompressor.reset();
                  for (; length; length--)
                  {
                    *nextDest++ = serverCache_ -> getPropertyTextCompressor.
                                                     decodeChar(decodeBuffer);
                  }
                }

                handleSave(messageStore, outputMessage, outputLength);
              }
              else
              {
                const unsigned char *compressedData = NULL;
                unsigned int compressedDataSize = 0;

                int decompressed = handleDecompress(decodeBuffer, requestOpcode, outputMessage,
                                                        outputLength, messageStore -> dataOffset,
                                                            compressedData, compressedDataSize);
                if (decompressed < 0)
                {
                  return -1;
                }
                else if (decompressed > 0)
                {
                  handleSave(messageStore, outputMessage, outputLength,
                                 compressedData, compressedDataSize);
                }
                else
                {
                  handleSave(messageStore, outputMessage, outputLength);
                }
              }
            }
            break;
          case X_LookupColor:
          case X_AllocNamedColor:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              unsigned char *nextDest = outputMessage + 8;
              if (requestOpcode == X_AllocNamedColor)
              {
                decodeBuffer.decodeValue(value, 32, 9);
                PutULONG(value, nextDest, bigEndian_);
                nextDest += 4;
              }
              unsigned int count = 3;
              do
              {
                decodeBuffer.decodeValue(value, 16, 9);
                PutUINT(value, nextDest, bigEndian_);
                unsigned int visualColor;
                decodeBuffer.decodeValue(visualColor, 16, 5);
                visualColor += value;
                visualColor &= 0xffff;
                PutUINT(visualColor, nextDest + 6, bigEndian_);
                nextDest += 2;
              }
              while (--count);
            }
            break;
          case X_QueryBestSize:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeValue(value, 16, 8);
              PutUINT(value, outputMessage + 8, bigEndian_);
              decodeBuffer.decodeValue(value, 16, 8);
              PutUINT(value, outputMessage + 10, bigEndian_);
            }
            break;
          case X_QueryColors:
            {
              // Differential or plain data compression?
              decodeBuffer.decodeBool(value);

              if (value)
              {
                decodeBuffer.decodeBool(value);
                if (value)
                {
                  unsigned int numColors =
                  serverCache_ -> queryColorsLastReply.getLength() / 6;
                  outputLength = 32 + (numColors << 3);
                  outputMessage = writeBuffer_.addMessage(outputLength);
                  PutUINT(numColors, outputMessage + 8, bigEndian_);
                  const unsigned char *nextSrc =
                  serverCache_ -> queryColorsLastReply.getData();
                  unsigned char *nextDest = outputMessage + 32;
                  for (; numColors; numColors--)
                  {
                    for (unsigned int i = 0; i < 6; i++)
                      *nextDest++ = *nextSrc++;
                    nextDest += 2;
                  }
                }
                else
                {
                  unsigned int numColors;
                  decodeBuffer.decodeValue(numColors, 16, 5);
                  outputLength = 32 + (numColors << 3);
                  outputMessage = writeBuffer_.addMessage(outputLength);
                  PutUINT(numColors, outputMessage + 8, bigEndian_);
                  unsigned char *nextDest = outputMessage + 32;
                  for (unsigned int c = 0; c < numColors; c++)
                  {
                    for (unsigned int i = 0; i < 3; i++)
                    {
                      decodeBuffer.decodeValue(value, 16);
                      PutUINT(value, nextDest, bigEndian_);
                      nextDest += 2;
                    }
                  }
                  serverCache_ -> queryColorsLastReply.set(numColors * 6,
                                       outputMessage + 32);
                  const unsigned char *nextSrc = nextDest - 1;
                  nextDest = outputMessage + 32 + ((numColors - 1) << 3) + 5;
                  for (; numColors > 1; numColors--)
                  {
                    for (unsigned int i = 0; i < 6; i++)
                      *nextDest-- = *nextSrc--;
                    nextDest -= 2;
                  }
                }
              }
              else
              {
                // Reply length.
                unsigned int numColors;
                decodeBuffer.decodeValue(numColors, 16, 5);
                outputLength = 32 + (numColors << 3);
                outputMessage = writeBuffer_.addMessage(outputLength);
                PutUINT(numColors, outputMessage + 8, bigEndian_);

                const unsigned char *compressedData = NULL;
                unsigned int compressedDataSize = 0;

                int decompressed = handleDecompress(decodeBuffer, requestOpcode, outputMessage,
                                                        outputLength, 32, compressedData,
                                                            compressedDataSize);
                if (decompressed < 0)
                {
                  return -1;
                }
              }
            }
            break;
          case X_QueryExtension:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeBool(value);
              outputMessage[8] = (unsigned char) value;
              decodeBuffer.decodeValue(value, 8);
              outputMessage[9] = (unsigned char) value;
              decodeBuffer.decodeValue(value, 8);
              outputMessage[10] = (unsigned char) value;
              decodeBuffer.decodeValue(value, 8);
              outputMessage[11] = (unsigned char) value;

              //
              // We use a predefined opcode to address
              // extensions' message stores, while real
              // opcodes are used for communication with
              // X server and clients.
              //

              if (requestData[0] == X_NXInternalShapeExtension)
              {
                opcodeStore_ -> shapeExtension = outputMessage[9];

                #if defined(INFO) || defined(TEST)
                *logofs << "handleWrite: Shape extension opcode for FD#" << fd_
                        << " is " << (unsigned int) opcodeStore_ -> shapeExtension
                        << ".\n" << logofs_flush;
                #endif
              }
              else if (requestData[0] == X_NXInternalRenderExtension)
              {
                opcodeStore_ -> renderExtension = outputMessage[9];

                #if defined(INFO) || defined(TEST)
                *logofs << "handleWrite: Render extension opcode for FD#" << fd_
                        << " is " << (unsigned int) opcodeStore_ -> renderExtension
                        << ".\n" << logofs_flush;
                #endif
              }
            }
            break;
          case X_QueryFont:
            {
              //
              // Use differential compression at startup and plain
              // data compression later. Check X_ListFonts message
              // for an explaination.
              //

              MessageStore *messageStore = serverStore_ ->
                                   getReplyStore(X_QueryFont);

              if (handleDecode(decodeBuffer, serverCache_, messageStore,
                                   outputMessage, outputLength))
              {
                break;
              }

              // Differential or plain data compression?
              decodeBuffer.decodeBool(value);

              if (value)
              {
                unsigned int numProperties;
                unsigned int numCharInfos;
                decodeBuffer.decodeValue(numProperties, 16, 8);
                decodeBuffer.decodeValue(numCharInfos, 32, 10);
                outputLength = 60 + numProperties * 8 + numCharInfos * 12;
                outputMessage = writeBuffer_.addMessage(outputLength);
                PutUINT(numProperties, outputMessage + 46, bigEndian_);
                PutULONG(numCharInfos, outputMessage + 56, bigEndian_);
                handleDecodeCharInfo(decodeBuffer, outputMessage + 8);
                handleDecodeCharInfo(decodeBuffer, outputMessage + 24);
                decodeBuffer.decodeValue(value, 16, 9);
                PutUINT(value, outputMessage + 40, bigEndian_);
                decodeBuffer.decodeValue(value, 16, 9);
                PutUINT(value, outputMessage + 42, bigEndian_);
                decodeBuffer.decodeValue(value, 16, 9);
                PutUINT(value, outputMessage + 44, bigEndian_);
                decodeBuffer.decodeBool(value);
                outputMessage[48] = (unsigned char) value;
                decodeBuffer.decodeValue(value, 8);
                outputMessage[49] = (unsigned char) value;
                decodeBuffer.decodeValue(value, 8);
                outputMessage[50] = (unsigned char) value;
                decodeBuffer.decodeBool(value);
                outputMessage[51] = (unsigned char) value;
                decodeBuffer.decodeValue(value, 16, 9);
                PutUINT(value, outputMessage + 52, bigEndian_);
                decodeBuffer.decodeValue(value, 16, 9);
                PutUINT(value, outputMessage + 54, bigEndian_);
                unsigned char *nextDest = outputMessage + 60;
                decodeBuffer.decodeBool(value);

                int end = 0;

                if (value == 1)
                {
                  unsigned int index;
                  decodeBuffer.decodeValue(index, 4);
                  unsigned int length;
                  const unsigned char *data;
                  ServerCache::queryFontFontCache.get(index, length, data);
                  memcpy(nextDest, data, length);

                  end = 1;
                }

                if (end == 0)
                {
                  unsigned char *saveDest = nextDest;
                  unsigned int length = numProperties * 8 + numCharInfos * 12;
                  for (; numProperties; numProperties--)
                  {
                    decodeBuffer.decodeValue(value, 32, 9);
                    PutULONG(value, nextDest, bigEndian_);
                    decodeBuffer.decodeValue(value, 32, 9);
                    PutULONG(value, nextDest + 4, bigEndian_);
                    nextDest += 8;
                  }
                  for (; numCharInfos; numCharInfos--)
                  {
                    handleDecodeCharInfo(decodeBuffer, nextDest);

                    nextDest += 12;
                  }
                  ServerCache::queryFontFontCache.set(length, saveDest);
                }

                handleSave(messageStore, outputMessage, outputLength);
              }
              else
              {
                // Reply length.
                unsigned int replyLength;
                decodeBuffer.decodeValue(replyLength, 32, 16);
                outputLength = 32 + (replyLength << 2);
                outputMessage = writeBuffer_.addMessage(outputLength);

                const unsigned char *compressedData = NULL;
                unsigned int compressedDataSize = 0;

                int decompressed = handleDecompress(decodeBuffer, requestOpcode, outputMessage,
                                                        outputLength, messageStore -> dataOffset,
                                                            compressedData, compressedDataSize);
                if (decompressed < 0)
                {
                  return -1;
                }
                else if (decompressed > 0)
                {
                  handleSave(messageStore, outputMessage, outputLength,
                                 compressedData, compressedDataSize);
                }
                else
                {
                  handleSave(messageStore, outputMessage, outputLength);
                }
              }
            }
            break;
          case X_QueryPointer:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeBool(value);
              outputMessage[1] = (unsigned char) value;
              decodeBuffer.decodeCachedValue(value, 29,
                                     serverCache_ -> queryPointerRootCache, 9);
              PutULONG(value, outputMessage + 8, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 29,
                                    serverCache_ -> queryPointerChildCache, 9);
              PutULONG(value, outputMessage + 12, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                    serverCache_ -> motionNotifyRootXCache, 8);
              serverCache_ -> motionNotifyLastRootX += value;
              PutUINT(serverCache_ -> motionNotifyLastRootX, outputMessage + 16,
                      bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                    serverCache_ -> motionNotifyRootYCache, 8);
              serverCache_ -> motionNotifyLastRootY += value;
              PutUINT(serverCache_ -> motionNotifyLastRootY, outputMessage + 18,
                      bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                   serverCache_ -> motionNotifyEventXCache, 8);
              PutUINT(serverCache_ -> motionNotifyLastRootX + value,
                      outputMessage + 20, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                   serverCache_ -> motionNotifyEventYCache, 8);
              PutUINT(serverCache_ -> motionNotifyLastRootY + value,
                      outputMessage + 22, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                       serverCache_ -> motionNotifyStateCache);
              PutUINT(value, outputMessage + 24, bigEndian_);
            }
            break;
          case X_QueryTree:
            {
              if (control -> isProtoStep3() == 1)
              {
                unsigned int children;
                decodeBuffer.decodeValue(children, 16, 8);

                outputLength = 32 + (children << 2);
                outputMessage = writeBuffer_.addMessage(outputLength);

                PutULONG(outputLength, outputMessage + 4, bigEndian_);

                decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> queryTreeWindowCache);

                PutULONG(value, outputMessage + 8, bigEndian_);

                decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> queryTreeWindowCache);

                PutULONG(value, outputMessage + 12, bigEndian_);

                unsigned char *next = outputMessage + 32;

                PutUINT(children, outputMessage + 16, bigEndian_);

                for (unsigned int i = 0; i < children; i++)
                {
                  decodeBuffer.decodeCachedValue(value, 29,
                                     serverCache_ -> queryTreeWindowCache);

                  PutULONG(value, next + (i * 4), bigEndian_);
                }
              }
              else
              {
                unsigned int secondByte;
                decodeBuffer.decodeValue(secondByte, 8);
                unsigned int replyLength;
                decodeBuffer.decodeValue(replyLength, 32);
                outputLength = 32 + (replyLength << 2);
                outputMessage = writeBuffer_.addMessage(outputLength);
                outputMessage[1] = (unsigned char) secondByte;
                unsigned char *nextDest = outputMessage + 8;
                for (unsigned int i = 8; i < outputLength; i++)
                {
                  unsigned int nextByte;
                  decodeBuffer.decodeValue(nextByte, 8);
                  *nextDest++ = (unsigned char) nextByte;
                }
              }
            }
            break;
          case X_TranslateCoords:
            {
              outputLength = 32;
              outputMessage = writeBuffer_.addMessage(outputLength);
              decodeBuffer.decodeBool(value);
              outputMessage[1] = (unsigned char) value;
              decodeBuffer.decodeCachedValue(value, 29,
                                 serverCache_ -> translateCoordsChildCache, 9);
              PutULONG(value, outputMessage + 8, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                     serverCache_ -> translateCoordsXCache, 8);
              PutUINT(value, outputMessage + 12, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                     serverCache_ -> translateCoordsYCache, 8);
              PutUINT(value, outputMessage + 14, bigEndian_);
            }
            break;
          case X_GetImage:
            {
              MessageStore *messageStore = serverStore_ ->
                                   getReplyStore(X_GetImage);

              if (handleDecode(decodeBuffer, serverCache_, messageStore,
                                   outputMessage, outputLength))
              {
                break;
              }

              // Depth.
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 serverCache_ -> depthCache);
              // Reply length.
              unsigned int replyLength;
              decodeBuffer.decodeValue(replyLength, 32, 9);
              outputLength = 32 + (replyLength << 2);
              outputMessage = writeBuffer_.addMessage(outputLength);
              outputMessage[1] = (unsigned char) cValue;
              // Visual.
              unsigned int visual;
              decodeBuffer.decodeCachedValue(visual, 29, 
                                 serverCache_ -> visualCache);
              PutULONG(visual, outputMessage + 8, bigEndian_);

              const unsigned char *compressedData = NULL;
              unsigned int compressedDataSize = 0;

              int decompressed = handleDecompress(decodeBuffer, requestOpcode, outputMessage,
                                                      outputLength, messageStore -> dataOffset,
                                                          compressedData, compressedDataSize);
              if (decompressed < 0)
              {
                return -1;
              }
              else if (decompressed > 0)
              {
                handleSave(messageStore, outputMessage, outputLength,
                               compressedData, compressedDataSize);
              }
              else
              {
                handleSave(messageStore, outputMessage, outputLength);
              }
            }
            break;
          case X_GetPointerMapping:
            {
              unsigned int nextByte;
              decodeBuffer.decodeValue(nextByte, 8, 4);
              unsigned int replyLength;
              decodeBuffer.decodeValue(replyLength, 32, 4);
              outputLength = 32 + (replyLength << 2);
              outputMessage = writeBuffer_.addMessage(outputLength);
              outputMessage[1] = (unsigned char) nextByte;
              unsigned char *nextDest = outputMessage + 32;
              for (unsigned int i = 32; i < outputLength; i++)
              {
                decodeBuffer.decodeValue(nextByte, 8, 4);
                *nextDest++ = (unsigned char) nextByte;
              }
            }
            break;
          case X_GetKeyboardControl:
            {
              unsigned int nextByte;
              decodeBuffer.decodeValue(nextByte, 8, 2);
              unsigned int replyLength;
              decodeBuffer.decodeValue(replyLength, 32, 8);
              outputLength = 32 + (replyLength << 2);
              outputMessage = writeBuffer_.addMessage(outputLength);
              outputMessage[1] = (unsigned char) nextByte;
              unsigned char *nextDest = outputMessage + 8;
              for (unsigned int i = 8; i < outputLength; i++)
              {
                decodeBuffer.decodeValue(nextByte, 8, 4);
                *nextDest++ = (unsigned char) nextByte;
              }
            }
            break;
          default:
            {
              if (requestOpcode == opcodeStore_ -> getUnpackParameters)
              {
                #if defined(INFO) || defined(TEST)
                *logofs << "handleWrite: Received get unpack parameters reply "
                        << "OPCODE#" << (unsigned int) opcodeStore_ -> getUnpackParameters
                        << ".\n" << logofs_flush;
                #endif

                outputLength = 32 + PACK_METHOD_LIMIT;

                outputMessage = writeBuffer_.addMessage(outputLength);

                unsigned int method;

                //
                // Let agent use only the unpack methods
                // implemented at both sides.
                //

                for (int i = 0; i < PACK_METHOD_LIMIT; i++)
                {
                  decodeBuffer.decodeBool(method);

                  control -> RemoteUnpackMethods[i] = method;

                  *(outputMessage + 32 + i) =
                      (control -> LocalUnpackMethods[i] == 1 &&
                           method == 1);
                }
              }
              else if (requestOpcode == opcodeStore_ -> getShmemParameters)
              {
                if (handleShmemInit(decodeBuffer, requestOpcode,
                                        outputMessage, outputLength) < 0)
                {
                  return -1;
                }
              }
              else
              {
                #ifdef PANIC
                *logofs << "handleWrite: PANIC! No matching request for "
                        << "reply with sequence number " << sequenceNum
                        << ".\n" << logofs_flush;
                #endif

                cerr << "Error" << ": No matching request for "
                     << "reply with sequence number " << sequenceNum
                     << ".\n";

                HandleCleanup();
              }
            }
          }

          #if defined(TEST) || defined(OPCODES)
          *logofs << "handleWrite: Handled reply to OPCODE#"
                  << (unsigned int) requestOpcode << " (" << DumpOpcode(requestOpcode)
                  << ")" << " for FD#" << fd_ << " with sequence " << serverSequence_
                  << ". Output size is " << outputLength << ".\n" << logofs_flush;
          #endif

          if (control -> CollectStatistics)
          {
            statistics -> addRepliedRequest(requestOpcode);
          }
        }
        else // End of if (sequenceQueue_.peek() && ...)
        {
          //
          // Reply didn't match any request opcode.
          // Check again if differential encoding
          // is disabled.
          //

          #ifdef DEBUG
          *logofs << "handleWrite: Identified generic reply.\n"
                  << logofs_flush;
          #endif

          requestOpcode = X_Reply;

          if (control -> RemoteDeltaCompression == 0)
          {
            int result = handleFastWriteReply(decodeBuffer, requestOpcode,
                                                  outputMessage, outputLength);
            if (result < 0)
            {
              return -1;
            }
            else if (result > 0)
            {
              continue;
            }
          }

          //
          // All replies whose opcode is not pushed in
          // sequence number queue are cached together.
          // Among such replies are those to extension
          // requests.
          //

          MessageStore *messageStore = serverStore_ ->
                               getReplyStore(X_NXInternalGenericReply);

          handleDecode(decodeBuffer, serverCache_, messageStore,
                           requestOpcode, outputMessage, outputLength);

          #if defined(TEST) || defined(OPCODES)
          *logofs << "handleWrite: Handled generic reply for FD#" << fd_
                  << " with sequence " << serverSequence_ << ". Output size is "
                  << outputLength << ".\n" << logofs_flush;
          #endif

          if (control -> CollectStatistics)
          {
            statistics -> addRepliedRequest(requestOpcode);
          }

        } // End of if (sequenceQueue_.peek() && ...) else ...

        //
        // If any output was produced then write opcode,
        // sequence number and size to the buffer.
        //

        if (outputLength > 0)
        {
          *outputMessage = outputOpcode;

          PutUINT(serverSequence_, outputMessage + 2, bigEndian_);

          PutULONG((outputLength - 32) >> 2, outputMessage + 4, bigEndian_);

          outputFlush++;
        }

      } // End of if (outputOpcode == 1)...
      else
      {
        //
        // It's an event or error.
        //

        unsigned int sequenceNum;
        unsigned int sequenceDiff;

        if (serverSequenceReset_ == 1)
        {
          decodeBuffer.decodeValue(sequenceNum, 32, 16);

          #if defined(INFO) || defined(TEST)
          *logofs << "handleWrite: Reset last server sequence number to "
                  << sequenceNum << " for FD#" << fd_ << ".\n"
                  << logofs_flush;
          #endif

          serverSequenceReset_ = 0;
        }
        else
        {
          decodeBuffer.decodeCachedValue(sequenceDiff, 16,
                             serverCache_ -> eventSequenceCache, 7);

          sequenceNum = (serverSequence_ + sequenceDiff) & 0xffff;
        }

        serverSequence_ = sequenceNum;

        #ifdef DEBUG
        *logofs << "handleWrite: Last server sequence number for FD#" 
                << fd_ << " is " << serverSequence_ << " with "
                << "difference " << sequenceDiff << ".\n"
                << logofs_flush;
        #endif

        //
        // Check if this is an error that matches
        // a sequence number for which we were
        // expecting a reply.
        //

        if (outputOpcode == X_Error)
        {
          unsigned short int dummySequenceNum;
          unsigned char dummyOpcode;

          if (sequenceQueue_.peek(dummySequenceNum, dummyOpcode) &&
                  ((unsigned int) dummySequenceNum == serverSequence_))
          {
            //
            // Remove the enqueued reference to reply.
            //

            #if defined(INFO) || defined(TEST)
            *logofs << "handleWrite: WARNING! Removing reply to OPCODE#"
                    << (unsigned int) dummyOpcode << " sequence "
                    << dummySequenceNum << " for FD#" << fd_
                    << " due to error.\n" << logofs_flush;
            #endif

            sequenceQueue_.pop(dummySequenceNum, dummyOpcode);

            //
            // TODO: This should be done for any error, also
            // for errors which did not match a reply. Problem
            // is that you get sequence lost from Xlib in case
            // errors come after the auto-generated reply and
            // some clients can get confused.
            //

            if (control -> SessionMode == SESSION_APPLICATION)
            {
              #ifdef TEST
              *logofs << "handleWrite: Updating last event's sequence "
                      << lastSequence_ << " to X server's error sequence "
                      << "number " << serverSequence_ << " for FD#"
                      << fd_ << ".\n" << logofs_flush;
              #endif

              lastSequence_ = serverSequence_;
            }
          }

          //
          // In case of errors always send to client the
          // original X server's sequence associated to
          // the failing request.
          //

          if (control -> SessionMode != SESSION_APPLICATION)
          {
            #ifdef TEST
            *logofs << "handleWrite: Updating last event's sequence "
                    << lastSequence_ << " to X server's error sequence "
                    << "number " << serverSequence_ << " for FD#"
                    << fd_ << ".\n" << logofs_flush;
            #endif

            lastSequence_ = serverSequence_;
          }
        }

        //
        // Check if by producing events at client side we
        // have modified the events' sequence numbering.
        // In this case taint the original sequence to
        // comply with the last one known by client.
        //

        if (serverSequence_ > lastSequence_)
        {
          #ifdef TEST
          *logofs << "handleWrite: Updating last event's sequence "
                  << lastSequence_ << " to X server's sequence number "
                  << serverSequence_ << " for FD#" << fd_
                  << ".\n" << logofs_flush;
          #endif

          lastSequence_ = serverSequence_;
        }
        #ifdef TEST
        else if (serverSequence_ < lastSequence_)
        {
          //
          // Use our last auto-generated sequence.
          //

          *logofs << "handleWrite: Tainting sequence number "
                  << serverSequence_ << " to last event's sequence "
                  << lastSequence_ << " for FD#" << fd_ << ".\n"
                  << logofs_flush;
        }
        #endif

        //
        // Check if remote side used fast encoding.
        //

        if (control -> RemoteDeltaCompression == 0)
        {
          int result = handleFastWriteEvent(decodeBuffer, outputOpcode,
                                                outputMessage, outputLength);
          if (result < 0)
          {
            return -1;
          }
          else if (result > 0)
          {
            continue;
          }
        }

        //
        // Make space for message in the outgoing buffer
        // and write opcode and sequence number.
        //

        outputLength = 32;
        outputMessage = writeBuffer_.addMessage(outputLength);

        *outputMessage = outputOpcode;

        PutUINT(lastSequence_, outputMessage + 2, bigEndian_);

        #ifdef DEBUG
        *logofs << "handleWrite: Going to handle event or error OPCODE#"
                << (unsigned int) outputOpcode << " for FD#" << fd_
                << " sequence " << lastSequence_ << " (real was "
                << serverSequence_ << ").\n" << logofs_flush;
        #endif

        switch (outputOpcode)
        {
        case X_Error:
          {
            unsigned char code;
            decodeBuffer.decodeCachedValue(code, 8,
                               serverCache_ -> errorCodeCache);
            outputMessage[1] = code;

            #if defined(INFO) || defined(TEST) || defined(OPCODES)
            *logofs << "handleWrite: Handled error ERR_CODE#"
                    << (unsigned int) code << " for FD#" << fd_;
            #endif

            if ((code != 11) && (code != 8) &&
                    (code != 15) && (code != 1))
            {
              decodeBuffer.decodeValue(value, 32, 16);
              PutULONG(value, outputMessage + 4, bigEndian_);

              #if defined(INFO) || defined(TEST) || defined(OPCODES)
              *logofs << " RES_ID#" << value;
              #endif
            }

            if (code >= 18)
            {
              decodeBuffer.decodeCachedValue(value, 16,
                                 serverCache_ -> errorMinorCache);
              PutUINT(value, outputMessage + 8, bigEndian_);

              #if defined(INFO) || defined(TEST) || defined(OPCODES)
              *logofs << " MIN_OP#" << value;
              #endif
            }

            decodeBuffer.decodeCachedValue(cValue, 8,
                               serverCache_ -> errorMajorCache);
            outputMessage[10] = cValue;

            #if defined(INFO) || defined(TEST) || defined(OPCODES)
            *logofs << " MAJ_OP#" << (unsigned int) cValue;
            #endif

            if (code >= 18)
            {
              unsigned char *nextDest = outputMessage + 11;
              for (unsigned int i = 11; i < 32; i++)
              {
                decodeBuffer.decodeValue(value, 8);
                *nextDest++ = (unsigned char) cValue;
              }
            }

            #if defined(INFO) || defined(TEST) || defined(OPCODES)
            *logofs << " sequence " << lastSequence_ << " (real was "
                    << serverSequence_ << ") . Size is "
                    << (unsigned int) outputLength << ".\n"
                    << logofs_flush;
            #endif
          }
          break;
        case ButtonPress:
        case ButtonRelease:
        case KeyPress:
        case KeyRelease:
        case MotionNotify:
        case EnterNotify:
        case LeaveNotify:
          {
            if (outputOpcode == MotionNotify)
            {
              decodeBuffer.decodeBool(value);
            }
            else if (outputOpcode == EnterNotify || outputOpcode == LeaveNotify)
            {
              decodeBuffer.decodeValue(value, 3);
            }
            else if (outputOpcode == KeyRelease)
            {
              decodeBuffer.decodeBool(value);
              if (value)
              {
                value = serverCache_ -> keyPressLastKey;
              }
              else
              {
                decodeBuffer.decodeValue(value, 8);
              }
            }
            else if (outputOpcode == ButtonPress || outputOpcode == ButtonRelease)
            {
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 serverCache_ -> buttonCache);
              value = (unsigned int) cValue;
            }
            else
            {
              decodeBuffer.decodeValue(value, 8);
            }

            outputMessage[1] = (unsigned char) value;
            decodeBuffer.decodeCachedValue(value, 32,
                               serverCache_ -> motionNotifyTimestampCache, 9);
            serverCache_ -> lastTimestamp += value;
            PutULONG(serverCache_ -> lastTimestamp, outputMessage + 4,
                     bigEndian_);
            unsigned char *nextDest = outputMessage + 8;
            int skipRest = 0;
            if (outputOpcode == KeyRelease)
            {
              decodeBuffer.decodeBool(value);
              if (value)
              {
                for (unsigned int i = 0; i < 23; i++)
                {
                  *nextDest++ = serverCache_ -> keyPressCache[i];
                }
                skipRest = 1;
              }
            }

            if (!skipRest)
            {
              for (unsigned int i = 0; i < 3; i++)
              {
                decodeBuffer.decodeCachedValue(value, 29,
                                   *serverCache_ -> motionNotifyWindowCache[i], 6);
                PutULONG(value, nextDest, bigEndian_);
                nextDest += 4;
              }
              decodeBuffer.decodeCachedValue(value, 16,
                                    serverCache_ -> motionNotifyRootXCache, 6);
              serverCache_ -> motionNotifyLastRootX += value;
              PutUINT(serverCache_ -> motionNotifyLastRootX, outputMessage + 20,
                      bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                    serverCache_ -> motionNotifyRootYCache, 6);
              serverCache_ -> motionNotifyLastRootY += value;
              PutUINT(serverCache_ -> motionNotifyLastRootY, outputMessage + 22,
                      bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                   serverCache_ -> motionNotifyEventXCache, 6);
              PutUINT(serverCache_ -> motionNotifyLastRootX + value,
                      outputMessage + 24, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                   serverCache_ -> motionNotifyEventYCache, 6);
              PutUINT(serverCache_ -> motionNotifyLastRootY + value,
                      outputMessage + 26, bigEndian_);
              decodeBuffer.decodeCachedValue(value, 16,
                                       serverCache_ -> motionNotifyStateCache);
              PutUINT(value, outputMessage + 28, bigEndian_);
              if (outputOpcode == EnterNotify || outputOpcode == LeaveNotify)
              {
                decodeBuffer.decodeValue(value, 2);
              }
              else
              {
                decodeBuffer.decodeBool(value);
              }
              outputMessage[30] = (unsigned char) value;
              if (outputOpcode == EnterNotify || outputOpcode == LeaveNotify)
              {
                decodeBuffer.decodeValue(value, 2);
                outputMessage[31] = (unsigned char) value;
              }
              else if (outputOpcode == KeyPress)
              {
                serverCache_ -> keyPressLastKey = outputMessage[1];
                for (unsigned int i = 8; i < 31; i++)
                {
                  serverCache_ -> keyPressCache[i - 8] = outputMessage[i];
                }
              }
            }
          }
          break;
        case ColormapNotify:
          {
            decodeBuffer.decodeCachedValue(value, 29,
                                 serverCache_ -> colormapNotifyWindowCache, 8);
            PutULONG(value, outputMessage + 4, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                               serverCache_ -> colormapNotifyColormapCache, 8);
            PutULONG(value, outputMessage + 8, bigEndian_);
            decodeBuffer.decodeBool(value);
            outputMessage[12] = (unsigned char) value;
            decodeBuffer.decodeBool(value);
            outputMessage[13] = (unsigned char) value;
          }
          break;
        case ConfigureNotify:
          {
            unsigned char *nextDest = outputMessage + 4;
            for (unsigned int i = 0; i < 3; i++)
            {
              decodeBuffer.decodeCachedValue(value, 29,
                                 *serverCache_ -> configureNotifyWindowCache[i], 9);
              PutULONG(value, nextDest, bigEndian_);
              nextDest += 4;
            }
            for (unsigned int j = 0; j < 5; j++)
            {
              decodeBuffer.decodeCachedValue(value, 16,
                                 *serverCache_ -> configureNotifyGeomCache[j], 8);
              PutUINT(value, nextDest, bigEndian_);
              nextDest += 2;
            }
            decodeBuffer.decodeBool(value);
            *nextDest = value;
          }
          break;
        case CreateNotify:
          {
            decodeBuffer.decodeCachedValue(value, 29,
                                 serverCache_ -> createNotifyWindowCache, 9);
            PutULONG(value, outputMessage + 4, bigEndian_);
            decodeBuffer.decodeValue(value, 29, 5);
            serverCache_ -> createNotifyLastWindow += value;
            serverCache_ -> createNotifyLastWindow &= 0x1fffffff;
            PutULONG(serverCache_ -> createNotifyLastWindow, outputMessage + 8,
                     bigEndian_);
            unsigned char* nextDest = outputMessage + 12;
            for (unsigned int i = 0; i < 5; i++)
            {
              decodeBuffer.decodeValue(value, 16, 9);
              PutUINT(value, nextDest, bigEndian_);
              nextDest += 2;
            }
            decodeBuffer.decodeBool(value);
            *nextDest = (unsigned char) value;
          }
          break;
        case Expose:
          {
            decodeBuffer.decodeCachedValue(value, 29,
                               serverCache_ -> exposeWindowCache, 9);
            PutULONG(value, outputMessage + 4, bigEndian_);
            unsigned char *nextDest = outputMessage + 8;
            for (unsigned int i = 0; i < 5; i++)
            {
              decodeBuffer.decodeCachedValue(value, 16,
                                 *serverCache_ -> exposeGeomCache[i], 6);
              PutUINT(value, nextDest, bigEndian_);
              nextDest += 2;
            }
          }
          break;
        case FocusIn:
        case FocusOut:
          {
            decodeBuffer.decodeValue(value, 3);
            outputMessage[1] = (unsigned char) value;
            decodeBuffer.decodeCachedValue(value, 29,
                                        serverCache_ -> focusInWindowCache, 9);
            PutULONG(value, outputMessage + 4, bigEndian_);
            decodeBuffer.decodeValue(value, 2);
            outputMessage[8] = (unsigned char) value;
          }
          break;
        case KeymapNotify:
          {
            decodeBuffer.decodeBool(value);
            if (value)
            memcpy(outputMessage + 1, ServerCache::lastKeymap.getData(), 31);
            else
            {
              unsigned char *nextDest = outputMessage + 1;
              for (unsigned int i = 1; i < 32; i++)
              {
                decodeBuffer.decodeValue(value, 8);
                *nextDest++ = (unsigned char) value;
              }
            ServerCache::lastKeymap.set(31, outputMessage + 1);
            }
          }
          break;
        case MapNotify:
        case UnmapNotify:
        case DestroyNotify:
          {
            decodeBuffer.decodeCachedValue(value, 29,
                                       serverCache_ -> mapNotifyEventCache, 9);
            PutULONG(value, outputMessage + 4, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                      serverCache_ -> mapNotifyWindowCache, 9);
            PutULONG(value, outputMessage + 8, bigEndian_);
            if (outputOpcode == MapNotify || outputOpcode == UnmapNotify)
            {
              decodeBuffer.decodeBool(value);
              outputMessage[12] = (unsigned char) value;
            }
          }
          break;
        case NoExpose:
          {
            decodeBuffer.decodeCachedValue(value, 29,
                                     serverCache_ -> noExposeDrawableCache, 9);
            PutULONG(value, outputMessage + 4, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 16,
                                           serverCache_ -> noExposeMinorCache);
            PutUINT(value, outputMessage + 8, bigEndian_);
            decodeBuffer.decodeCachedValue(cValue, 8,
                                           serverCache_ -> noExposeMajorCache);
            outputMessage[10] = cValue;
          }
          break;
        case PropertyNotify:
          {
            decodeBuffer.decodeCachedValue(value, 29,
                                 serverCache_ -> propertyNotifyWindowCache, 9);
            PutULONG(value, outputMessage + 4, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> propertyNotifyAtomCache, 9);
            PutULONG(value, outputMessage + 8, bigEndian_);
            decodeBuffer.decodeValue(value, 32, 9);
            serverCache_ -> lastTimestamp += value;
            PutULONG(serverCache_ -> lastTimestamp, outputMessage + 12,
                     bigEndian_);
            decodeBuffer.decodeBool(value);
            outputMessage[16] = (unsigned char) value;
          }
          break;
        case ReparentNotify:
          {
            unsigned char* nextDest = outputMessage + 4;
            for (unsigned int i = 0; i < 3; i++)
            {
              decodeBuffer.decodeCachedValue(value, 29,
                             serverCache_ -> reparentNotifyWindowCache, 9);
              PutULONG(value, nextDest, bigEndian_);
              nextDest += 4;
            }
            decodeBuffer.decodeValue(value, 16, 6);
            PutUINT(value, nextDest, bigEndian_);
            decodeBuffer.decodeValue(value, 16, 6);
            PutUINT(value, nextDest + 2, bigEndian_);
            decodeBuffer.decodeBool(value);
            outputMessage[20] = (unsigned char)value;
          }
          break;
        case SelectionClear:
          {
            decodeBuffer.decodeValue(value, 32, 9);
            serverCache_ -> lastTimestamp += value;
            PutULONG(serverCache_ -> lastTimestamp, outputMessage + 4,
                     bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> selectionClearWindowCache, 9);
            PutULONG(value, outputMessage + 8, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> selectionClearAtomCache, 9);
            PutULONG(value, outputMessage + 12, bigEndian_);
          }
          break;
        case SelectionRequest:
          {
            decodeBuffer.decodeValue(value, 32, 9);
            serverCache_ -> lastTimestamp += value;
            PutULONG(serverCache_ -> lastTimestamp, outputMessage + 4,
                     bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> selectionClearWindowCache, 9);
            PutULONG(value, outputMessage + 8, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> selectionClearWindowCache, 9);
            PutULONG(value, outputMessage + 12, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> selectionClearAtomCache, 9);
            PutULONG(value, outputMessage + 16, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> selectionClearAtomCache, 9);
            PutULONG(value, outputMessage + 20, bigEndian_);
            decodeBuffer.decodeCachedValue(value, 29,
                                   serverCache_ -> selectionClearAtomCache, 9);
            PutULONG(value, outputMessage + 24, bigEndian_);
          }
          break;
        case VisibilityNotify:
          {
            decodeBuffer.decodeCachedValue(value, 29,
                               serverCache_ -> visibilityNotifyWindowCache, 9);
            PutULONG(value, outputMessage + 4, bigEndian_);
            decodeBuffer.decodeValue(value, 2);
            outputMessage[8] = (unsigned char) value;
          }
          break;
        default:
          {
            #ifdef TEST
            *logofs << "handleWrite: Using generic event compression "
                    << "for OPCODE#" << (unsigned int) outputOpcode
                    << ".\n" << logofs_flush;
            #endif

            if (control -> isProtoStep3() == 1)
            {
              decodeBuffer.decodeCachedValue(*(outputMessage + 1), 8,
                           serverCache_ -> genericEventCharCache);

              for (unsigned int i = 0; i < 14; i++)
              {
                decodeBuffer.decodeCachedValue(value, 16,
                             *serverCache_ -> genericEventIntCache[i]);

                PutUINT(value, outputMessage + i * 2 + 4, bigEndian_);
              }
            }
            else
            {
              unsigned int secondByte;
              decodeBuffer.decodeValue(secondByte, 8);
              outputMessage[1] = secondByte;
              unsigned char *nextDest = outputMessage + 4;
              for (unsigned int i = 4; i < outputLength; i++)
              {
                unsigned int nextByte;
                decodeBuffer.decodeValue(nextByte, 8);
                *nextDest++ = (unsigned char) nextByte;
              } 
            }
          }
        } // End of switch (outputOpcode)...

        #if defined(TEST) || defined(OPCODES)
        if (outputOpcode != X_Error)
        {
          *logofs << "handleWrite: Handled event OPCODE#"
                  << (unsigned int) outputOpcode << " for FD#"
                  << fd_ << " sequence " << lastSequence_ << " (real was "
                  << serverSequence_ << "). Size is " << outputLength
                  << ".\n" << logofs_flush;
        }
        #endif

        //
        // Check if we need to suppress the error.
        //

        if (control -> AgentTaintReplies > 0 && outputOpcode == X_Error &&
                    handleTaintError(*(outputMessage + 10)) == 1)
        {
          writeBuffer_.removeMessage(32);
        }

      } // End of if (outputOpcode == 1)... else ...

      //
      // Check if we produced enough data. We need to
      // decode all provided messages. Just update the
      // finish flag in case of failure.
      //

      if (outputFlush > 0)
      {
        handleFlush(flush_if_any);

        outputFlush = 0;
      }
      else
      {
        handleFlush(flush_if_needed);
      }

    } // End of while (decodeBuffer.decodeOpcodeValue(outputOpcode, 8, ...

  } // End of if (firstReply_)... else ...

  //
  // Write any remaining data to the X connection.
  //

  if (handleFlush(flush_if_any) < 0)
  {
    return -1;
  }

  return 1;
}

//
// End of handleWrite().
//

//
// Other members.
//

int ClientChannel::handleSplit(EncodeBuffer& encodeBuffer, MessageStore *store, 
                                   const unsigned char *buffer, const unsigned int size)
{
  //
  // If no client has been registered by
  // agent or the same message is already
  // being split, then don't split the
  // new message.
  //

  if (store -> lastStatus != is_added ||
          splitClient_ == nothing)
  {
    encodeBuffer.encodeBool(0);

    return 0;
  }

  //
  // Split was requested by agent. Let's
  // see if we can actually satisfy it.
  //

  if (canSplitMessage(store, size))
  {
    //
    // Signal the server side that message 
    // will be split.
    //

    #if defined(INFO) || defined(TEST)
    *logofs << "handleSplit: Message OPCODE#"
            << (unsigned int) store -> opcode() << " of size "
            << size << " will be split.\n" << logofs_flush;
    #endif

    encodeBuffer.encodeBool(1);

    //
    // Send MD5 of the message being split and
    // let the remote side check if message can
    // be loaded from disk.
    //

    if (control -> isProtoStep3() == 1 &&
            (control -> ImageCacheEnableLoad > 0 ||
                 control -> ImageCacheEnableSave > 0))
    {
      md5_byte_t *checksum = store -> getChecksum(store -> lastAdded);

      for (unsigned int i = 0; i < MD5_LENGTH; i++)
      {
        encodeBuffer.encodeValue((unsigned int) checksum[i], 8);
      }
    }

    //
    // Pass to split store the full message's size.
    //

    clientStore_ -> getSplitStore() -> add(store, splitClient_,
                                               store -> lastAdded, buffer, size);

    #if defined(INFO) || defined(TEST)
    *logofs << "handleSplit: Added message to the split store.\n" 
            << logofs_flush;
    #endif

    #if defined(INFO) || defined(TEST)
    *logofs << "handleSplit: There are " << clientStore_ ->
               getSplitStore() -> getSize() << " messages "
            << "in the split store.\n" << logofs_flush;
    #endif

    return 1;
  }
  else
  {
    //
    // Signal server side that there was no split.
    // Remember to wakeup agent's client as soon
    // as end of split is received.
    //

    encodeBuffer.encodeBool(0);

    #if defined(INFO) || defined(TEST)
    if (((int) size) > control -> SplitDataThreshold)
    {
      *logofs << "handleSplit: WARNING! Can't split message of size "
              << size << " for FD#" << fd_ << ".\n" << logofs_flush;

      *logofs << "handleSplit: WARNING! Split store has "
              << clientStore_ -> getSplitStore() -> getSize()
              << " messages and " << clientStore_ -> getSplitStore() ->
                 getTotalStorageSize() << " allocated bytes.\n"
              << logofs_flush;
    }
    #endif

    return 0;
  }
}

int ClientChannel::handleSplit(EncodeBuffer& encodeBuffer, int packetLimit)
{
  if (firstRequest_ == 1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "handleSplit: Delaying sending of splits for FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleSplit: Handling splits for FD#" << fd_
          << " with " << clientStore_ -> getSplitStore() -> getSize()
          << " elements and " << packetLimit
          << " bytes to write.\n" << logofs_flush;
  #endif

  //
  // We want to write all bytes that fit in the
  // current row, calculated at the time this
  // function is called. Get the amount of bytes
  // to write and than subtract bytes written at
  // each iteration.
  //

  int packetSize = packetLimit;

  int totalSplits = 0;

  while (packetSize > 0 && clientStore_ -> getSplitStore() -> getSize() > 0)
  {
    //
    // Use an appropriate opcode to signal
    // the other side this is a split.
    //

    encodeBuffer.encodeOpcodeValue(opcodeStore_ -> split, clientCache_ -> opcodeCache);

    clientOpcode_ = opcodeStore_ -> split;

    if (clientStore_ -> getSplitStore() -> send(encodeBuffer, packetSize) < 0)
    {
      #ifdef PANIC
      *logofs << "handleSplit: PANIC! Error sending splits for FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Error sending splits for FD#"
           << fd_ << ".\n";

      return -1;
    }

    //
    // There can be three cases:
    //
    // - We are still in the middle of a split.
    // - We completely recomposed a full message at the remote side.
    // - We recomposed all the messages added for a given client.
    //

    int client   = clientStore_ -> getSplitStore() -> getLastClient();
    int request  = clientStore_ -> getSplitStore() -> getLastRequest();
    int position = clientStore_ -> getSplitStore() -> getLastPosition();

    if (client != nothing && request != nothing && position != nothing)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleSplit: Sending deferred agent split event TYPE#"
              << (unsigned int) opcodeStore_ -> splitNotify << " to commit "
              << "request " << request << " at position " << position
              << ".\n" << logofs_flush;
      #endif

      if (handleWakeup(opcodeStore_ -> splitNotify, deferred,
                           client, request, position, nothing) < 0)
      {
        return -1;
      }

      int completion = clientStore_ -> getSplitStore() -> getLastCompletion();

      if (completion == 1)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "handleSplit: Sending deferred agent split event TYPE#"
                << (unsigned int) opcodeStore_ -> splitNotify << " to wakeup "
                << "client id " << client << ".\n" << logofs_flush;
        #endif

        if (handleWakeup(opcodeStore_ -> splitNotify, deferred,
                             nothing, nothing, nothing, client) < 0)
        {
          return -1;
        }
      }
    }

    //
    // Update statistics for this special opcode.
    //

    int bits = encodeBuffer.getBits();

    #if defined(TEST) || defined(OPCODES)
    *logofs << "handleSplit: Handled request OPCODE#"
            << (unsigned int) opcodeStore_ -> split << " ("
            << DumpOpcode(opcodeStore_ -> split) << ")" << " for FD#"
            << fd_ << " sequence none. 0 bytes in, " << bits << " bits ("
            << ((float) bits) / 8 << " bytes) out.\n" << logofs_flush;
    #endif

    if (control -> CollectStatistics)
    {
      statistics -> addRequestBits(opcodeStore_ -> split, 0, bits);
    }

    control -> addBitsInARow(bits);

    packetSize -= bits >> 3;

    totalSplits++;

    #if defined(INFO) || defined(TEST)
    *logofs << "handleSplit: There are still "
            << clientStore_ -> getSplitStore() -> getSize()
            << " splits to be transferred for FD#" << fd_ 
            << ".\n" << logofs_flush;
    #endif
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleSplit: Sent " << totalSplits
          << " splits in a row for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  if (totalSplits > 0)
  {
    //
    // Don't force sending of split in order
    // to maximize effectiveness of stream
    // compression.
    //

    return 1;
  }

  return 0;
}

int ClientChannel::handleUnsplit(DecodeBuffer &decodeBuffer, MessageStore *store,
                                     unsigned char *buffer, const unsigned int size)
{
  #ifdef PANIC
  *logofs << "ClientChannel: PANIC! Function handleUnsplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleUnsplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int ClientChannel::handleUnsplit(DecodeBuffer &decodeBuffer)
{
  #ifdef PANIC
  *logofs << "ClientChannel: PANIC! Function handleUnsplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleUnsplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int ClientChannel::handleAbortSplit(EncodeBuffer &encodeBuffer)
{
  #ifdef PANIC
  *logofs << "ClientChannel: PANIC! Function handleAbortSplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleAbortSplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int ClientChannel::handleAbortSplit(DecodeBuffer &decodeBuffer)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "handleUnsplit: Going to handle abort split messages "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Get MD5 of the message being split.
  //

  unsigned int value;

  md5_byte_t *checksum = NULL;

  checksum = new md5_byte_t[MD5_LENGTH];

  for (unsigned int i = 0; i < MD5_LENGTH; i++)
  {
    decodeBuffer.decodeValue(value, 8);

    if (checksum != NULL)
    {
      checksum[i] = (unsigned char) value;
    }
  }

  if (checksum != NULL)
  {
    #ifdef TEST
    *logofs << "handleUnsplit: Checking abort split for checksum ["
            << DumpChecksum(checksum) << "].\n" << logofs_flush;
    #endif

    T_splits *splits = clientStore_ -> getSplitStore() -> getSplits();

    for (T_splits::iterator i = splits -> begin(); i != splits -> end(); i++)
    {
      Split *split = *i;

      MessageStore *store = split -> getStore();

      Message *message = split -> getMessage();

      #ifdef TEST
      *logofs << "handleUnsplit: Comparing checksum with message ["
              << DumpChecksum(store -> getChecksum(message)) << "].\n"
              << logofs_flush;
      #endif

      if (memcmp(checksum, store -> getChecksum(message), MD5_LENGTH) == 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "handleUnsplit: Found aborted split with checksum ["
                << DumpChecksum(checksum) << "].\n" << logofs_flush;
        #endif

        split -> setAbort();

        break;
      }
    }
  }

  return 1;
}

int ClientChannel::handleTaintRequest(unsigned char &opcode)
{
  //
  // Should short-circuit other common replies
  // whose values could be queried only once.
  // Examples are X_InterAtom, X_ListExtension
  // and X_QueryExtension.
  //

  if (opcode != X_GetInputFocus)
  {
    return 0;
  }
  else if (taintCounter_ >= control -> AgentTaintThreshold)
  {
    #ifdef DEBUG
    *logofs << "handleTaintRequest: Reset taint counter after "
            << taintCounter_ << " replies managed.\n"
            << logofs_flush;
    #endif

    taintCounter_ = 0;

    return 0;
  }

  //
  // Check if we are rolling the counter.
  // The client sequence number has not
  // been incremented yet in the loop.
  //

  unsigned int sequence = (clientSequence_ + 1) & 0xffff;

  #ifdef DEBUG
  *logofs << "handleTaintRequest: Opcode is " << (unsigned) opcode
          << " expected client sequence is " << sequence
          << ".\n" << logofs_flush;
  #endif

  if (sequence == 0xffff)
  {
    return 0;
  }

  unsigned short t1;
  unsigned char  t2;

  //
  // Check if there is a previous reply
  // pending.
  //

  if (sequenceQueue_.peek(t1, t2) != 0)
  {
    #ifdef DEBUG
    *logofs << "handleTaintRequest: Skipping taint of reply due to "
            << "pending request OPCODE#" << t1 << " with sequence "
            << (unsigned int) t2 << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  #ifdef DEBUG
  *logofs << "handleTaintRequest: Suppressing get input focus "
          << "request for FD#" << fd_ << " with sequence "
          << sequence << ".\n" << logofs_flush;
  #endif

  unsigned char *reply = writeBuffer_.addMessage(32);

  *(reply + 0) = X_Reply;

  PutUINT(sequence, reply + 2, bigEndian_);

  PutULONG(0, reply + 4, bigEndian_);

  //
  // Set revert-to to none.
  //

  *(reply + 1) = 0;

  //
  // Set focus to none.
  //

  PutULONG(0, reply + 8, bigEndian_);

  //
  // Save the sequence number, not incremented
  // yet, we used to auto-generate this reply.
  //

  lastSequence_ = clientSequence_ + 1;

  #if defined(INFO) || defined(TEST)
  *logofs << "handleTaintRequest: Registered " << lastSequence_
          << " as last auto-generated sequence number.\n"
          << logofs_flush;
  #endif

  //
  // Taint request in a X_NoOperation message.
  //

  opcode = X_NoOperation;

  //
  // TODO: We assume client has finished drawing
  // even if this seems to perceively affect the
  // performances.
  //

  priority_++;

  if (handleFlush(flush_if_any) < 0)
  {
    return -1;
  }

  taintCounter_++;

  return 1;
}

int ClientChannel::handleTaintError(unsigned char request)
{
  //
  // By enabling short-circuiting of replies
  // some window managers can get confused
  // by some otherwise innocuous X errors.
  //

  if (request == X_GrabKey || request == X_ReparentWindow ||
          request == X_ConfigureWindow)
  {
    #if defined(TEST) || defined(OPCODES)
    *logofs << "handleTaintError: WARNING! Suppressed error "
            << "on OPCODE#" << (unsigned int) request << " for FD#"
            << fd_ << " sequence " << lastSequence_ << " (real was "
            << serverSequence_ << ").\n" << logofs_flush;
    #endif

    return 1;
  }

  return 0;
}

int ClientChannel::handleWakeup(unsigned char type, enum T_notification how,
                                    int client, int request, int position,
                                        int completion)
{
  if (finish_ == 1)
  {
    #ifdef TEST
    *logofs << "handleWakeup: Discarding wakeup on "
            << "channel for FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    return 0;
  }

  T_timestamp *timestamp = NULL;

  if (type == opcodeStore_ -> syncNotify)
  {
    timestamp = &syncClientTs_[client];
  }
  else if (type == opcodeStore_ -> karmaNotify)
  {
    timestamp = &karmaClientTs_[client];
  }

  if (timestamp != NULL)
  {
    if (how == deferred || client == nothing)
    {
      //
      // Send event immediately and reset
      // timestamp for this event/client
      // pair.
      //

      *timestamp = nullTimestamp();
    }
    else
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleWakeup: Saving timestamp for "
              << "agent notify event TYPE#" << (unsigned int) type
              << " with client " << client << " request " << request
              << " and position " << position << ".\n" << logofs_flush;
      #endif

      *timestamp = getTimestamp();

      //
      // There is at least one event to send
      // for this channel.
      //

      if (type == opcodeStore_ -> syncNotify)
      {
        syncCounter_++;
      }
      else
      {
        karmaCounter_++;
      }

      return 1;
    }
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleWakeup: Going to add "
          << (how == immediate ? "immediate " : "deferred ")
          << "agent notify event TYPE#" << (unsigned int) type
          << " with client " << client << " request " << request
          << " position " << position << " and completion "
          << completion << ".\n" << logofs_flush;
  #endif

  //
  // Add a new message to the write buffer.
  //

  unsigned char *event = writeBuffer_.addMessage(32);

  //
  // Event is ClientMessage, atom and
  // window are 0, format is 32.
  //

  *(event + 0) = ClientMessage;

  PutULONG(0, event + 4, bigEndian_);
  PutULONG(0, event + 8, bigEndian_);

  *(event + 1) = 32;

  //
  // If the event follows immediately the request (that is 'how'
  // is immediate) then sequence number is the one of the last
  // request, else it should be the sequence number encoded by
  // peer proxy but, as we are inserting events in the stream,
  // we must ensure that any sequence we send is not inferior
  // to the last sequence we auto-generated.
  //

  if (how == immediate)
  {
    //
    // Save the sequence number we used
    // to auto-generate this event.
    //

    lastSequence_ = clientSequence_;

    #if defined(INFO) || defined(TEST)
    *logofs << "handleWakeup: Registered " << lastSequence_
            << " as last auto-generated sequence number.\n"
            << logofs_flush;
    #endif
  }
  else
  {
    if (serverSequence_ > lastSequence_)
    {
      #ifdef TEST
      *logofs << "handleWakeup: Updating last event's sequence "
              << lastSequence_ << " to X server's sequence number "
              << serverSequence_ << " for FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      lastSequence_ = serverSequence_;
    }
    #ifdef TEST
    else if (serverSequence_ < lastSequence_)
    {
      //
      // Use our last auto-generated sequence.
      //

      *logofs << "handleWakeup: Tainting sequence number "
              << serverSequence_ << " to last event's sequence "
              << lastSequence_ << " for FD#" << fd_ << ".\n"
              << logofs_flush;
    }
    #endif
  }

  PutUINT(lastSequence_, event + 2, bigEndian_);

  //
  // Notification type.
  //

  PutULONG(type, event + 12, bigEndian_);

  //
  // TODO: To preserve compatibility with older
  // agents, we add the client id at the first
  // unused offset. As in the older protocol
  // version, the firts word is set only if
  // agent's client must be restarted.
  //

  //
  // Client to wakeup.
  //

  if (type == opcodeStore_ -> splitNotify)
  {
    PutULONG(completion, event + 16, bigEndian_);
  }
  else
  {
    PutULONG(client, event + 16, bigEndian_);
  }

  //
  // Request extended attribute.
  //

  PutULONG(request, event + 20, bigEndian_);

  //
  // Position extended attribute.
  //

  PutULONG(position, event + 24, bigEndian_);

  //
  // Newer protocol versions report at
  // any commit the id of client which
  // originated the split.
  //

  if (type == opcodeStore_ -> splitNotify)
  {
    PutULONG(client, event + 28, bigEndian_);
  }
  else
  {
    PutULONG((unsigned) nothing, event + 28, bigEndian_);
  }

  //
  // Send the notification event to client.
  //

  if (handleFlush(flush_if_any) < 0)
  {
    return -1;
  }

  return 1;
}

int ClientChannel::handleCommit(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                    const unsigned int size)
{
  //
  // Get the store involved.
  //

  unsigned char request = *(buffer + 8); 

  MessageStore *store = clientStore_ -> getRequestStore(request);

  if (store == NULL)
  {
    #ifdef PANIC
    *logofs << "handleCommit: PANIC! Can't commit split for "
            << "request OPCODE#" << (unsigned int) request
            << ". No message store found.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't commit split for request "
         << "OPCODE#" << (unsigned int) request
         << ". No message store found.\n";

    return -1;
  }

  //
  // The position in cache of image to commit.
  // Encode it as difference in respect to the
  // last committed request.
  //

  unsigned int position = GetULONG(buffer + 4, bigEndian_);

  if (position >= (unsigned int) store -> cacheSlots)
  {
    #ifdef PANIC
    *logofs << "handleCommit: PANIC! Position " << position
            << " exceeds the number of available slots"
            << " for request " << request << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Position " << position
         << " exceeds the number of available slots"
         << " for request " << request << ".\n";

    return -1;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleCommit: Encoding request type "
          << (unsigned) request << " and position "
          << position << " as image to commit.\n"
          << logofs_flush;
  #endif

  //
  // Commit can come after store has been reset.
  // As long as the other side checks this, we
  // can encode the message in the usual way.
  //

  encodeBuffer.encodeOpcodeValue(request, clientCache_ -> opcodeCache);

  clientOpcode_ = request;

  int diffCommit = position - clientStore_ -> getSplitStore() ->
                       getLastCommit();

  clientStore_ -> getSplitStore() -> setLastCommit(position);

  encodeBuffer.encodeValue(diffCommit, 32, 5);

  //
  // Unlock message so that it can be removed 
  // when we'll need to make room in the store.
  // Function is safe in the case message does
  // not exist.
  //

  store -> unlock(position);

  if (request == X_PutImage)
  {
    PutImageStore *putImageStore = (PutImageStore *) store;

    control -> addOutputInARow(putImageStore -> getSize(position));
  }
  else if (request == X_NXPutPackedImage)
  {
    PutPackedImageStore *putPackedImageStore = (PutPackedImageStore *) store;

    control -> addOutputInARow(putPackedImageStore -> getUnpackedSize(position));
  }

  #if defined(INFO) || defined(TEST)

  checkLocks(clientStore_ -> getRequestStore(X_PutImage));
  checkLocks(clientStore_ -> getRequestStore(X_NXPutPackedImage));
  checkLocks(clientStore_ -> getRequestStore(X_NXSetUnpackColormap));
  checkLocks(clientStore_ -> getRequestStore(X_NXSetUnpackAlpha));

  #endif

  return 1;
}

int ClientChannel::handleWakeup(EncodeBuffer &encodeBuffer)
{
  if ((syncCounter_ + karmaCounter_) == 0)
  {
    #ifdef TEST
    *logofs << "handleWakeup: No wakeup events to send for FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleWakeup: Checking agent wakeup timestamps for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  T_timestamp tsNow = getTimestamp();

  int diffTs = 0;

  #ifdef DEBUG
  *logofs << "handleWakeup: Looping for wakeup events "
          << "with clients " << lowerClient_ << " to " << upperClient_
          << " and " << (syncCounter_ + karmaCounter_) << " sleeping.\n"
          << logofs_flush;
  #endif

  //
  // Wakeup clients immediately on sync.
  //

  if (syncCounter_ > 0)
  {
    for (int client = lowerClient_;
             client <= upperClient_; client++)
    {
      diffTs = diffTimestamp(syncClientTs_[client], tsNow);

      if (diffTs >= 0)
      {
        #ifdef DEBUG
        *logofs << "handleWakeup: Checking sync wakeup after "
                << diffTs << " Ms with client " << client
                << ".\n" << logofs_flush;
        #endif

        #if defined(INFO) || defined(TEST)
        *logofs << "handleWakeup: Sending deferred agent sync notify "
                << "with client " << client << ".\n" << logofs_flush;
        #endif

        if (handleWakeup(opcodeStore_ -> syncNotify, deferred,
                             client, nothing, nothing, nothing) < 0)
        {
          return -1;
        }

        syncCounter_--;
      }
    }
  }

  //
  // Wakeup clients on karma only after a timeout.
  //

  if (karmaCounter_ > 0)
  {
    for (int client = lowerClient_;
             client <= upperClient_; client++)
    {
      diffTs = diffTimestamp(karmaClientTs_[client], tsNow);

      if (diffTs >= 0)
      {
        #ifdef DEBUG
        *logofs << "handleWakeup: Checking karma wakeup after "
                << diffTs << " Ms with client " << client
                << ".\n" << logofs_flush;
        #endif

        if (diffTs >= (control -> WakeupTimeout -
                           control -> LatencyTimeout))
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "handleWakeup: Sending deferred agent karma notify "
                  << "with client " << client << ".\n" << logofs_flush;
          #endif

          if (handleWakeup(opcodeStore_ -> karmaNotify, deferred,
                               client, nothing, nothing, nothing) < 0)
          {
            return -1;
          }

          karmaCounter_--;
        }
      }
    }
  }

  if (syncCounter_ < 0)
  {
    #ifdef WARNING
    *logofs << "handleWakeup: WARNING! Reached negative value "
            << "for sync counter.\n" << logofs_flush;
    #endif

    syncCounter_ = 0;
  }

  if (karmaCounter_ < 0)
  {
    #ifdef WARNING
    *logofs << "handleWakeup: WARNING! Reached negative value "
            << "for karma counter.\n" << logofs_flush;
    #endif

    karmaCounter_ = 0;
  }

  return 1;
}

int ClientChannel::handleConfiguration()
{
  #ifdef TEST
  *logofs << "ClientChannel: Setting new buffer parameters.\n"
          << logofs_flush;
  #endif

  readBuffer_.setSize(control -> ClientInitialReadSize,
                          control -> ClientMaximumReadSize);

  writeBuffer_.setSize(control -> TransportXBufferSize,
                           control -> TransportXBufferThreshold,
                               control -> TransportMaximumBufferSize);

  transport_ -> setSize(control -> TransportXBufferSize,
                            control -> TransportXBufferThreshold,
                                control -> TransportMaximumBufferSize);

  return 1;
}

int ClientChannel::handleFinish()
{
  #ifdef TEST
  *logofs << "ClientChannel: Finishing channel for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  pending_ = 0;
  finish_  = 1;

  for (int i = 0; i < CONNECTIONS_LIMIT; i++)
  {
    syncClientTs_[i]  = nullTimestamp();
    karmaClientTs_[i] = nullTimestamp();
  }

  syncCounter_  = 0;
  karmaCounter_ = 0;

  taintCounter_ = 0;

  transport_ -> fullReset();

  return 1;
}

int ClientChannel::handleNotify(T_notification_type type)
{
  if (firstReply_ == 1 || finish_ == 1)
  {
    #ifdef TEST
    *logofs << "handleNotify: Discarding notify on "
            << "channel for FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    return 0;
  }

  int notify;
  int state;

  switch (type)
  {
    case notify_begin_congestion:
    {
      notify = NXCongestionNotify;
      state  = 1;

      break;
    }
    case notify_end_congestion:
    {
      notify = NXCongestionNotify;
      state  = 0;

      break;
    }
    case notify_begin_reset:
    {
      notify = NXResetNotify;
      state  = 1;

      break;
    }
    case notify_end_reset:
    {
      notify = NXResetNotify;
      state  = 0;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "handleNotify: PANIC! Unrecognized notify type "
              << "with code " << type << ".\n" << logofs_flush;
      #endif

      return -1;
    }
  }

  #ifdef TEST
  *logofs << "handleNotify: Going to add agent notify event "
          << "TYPE#" << (unsigned int) notify << " with state "
          << state << ".\n" << logofs_flush;
  #endif

  //
  // Add a new message to the write buffer.
  //

  unsigned char *event = writeBuffer_.addMessage(32);

  //
  // Event is ClientMessage, atom and
  // window are 0, format is 32.
  //

  *(event + 0) = ClientMessage;

  PutULONG(0, event + 4, bigEndian_);
  PutULONG(0, event + 8, bigEndian_);

  *(event + 1) = 32;

  //
  // Set event's sequence number.
  //

  if (serverSequence_ > lastSequence_)
  {
    #ifdef TEST
    *logofs << "handleNotify: Updating last event's sequence "
            << lastSequence_ << " to X server's sequence number "
            << serverSequence_ << " for FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    lastSequence_ = serverSequence_;
  }
  #ifdef TEST
  else if (serverSequence_ < lastSequence_)
  {
    *logofs << "handleNotify: Tainting sequence number "
            << serverSequence_ << " to last event's sequence "
            << lastSequence_ << " for FD#" << fd_ << ".\n"
            << logofs_flush;
  }
  #endif

  PutUINT(lastSequence_, event + 2, bigEndian_);

  #if defined(INFO) || defined(TEST)
  *logofs << "handleNotify: Registered " << lastSequence_
          << " as last auto-generated sequence number.\n"
          << logofs_flush;
  #endif

  //
  // Notification type.
  //

  PutULONG(notify, event + 12, bigEndian_);

  //
  // Is condition raised or ceased?
  //

  PutULONG(state, event + 16, bigEndian_);

  //
  // Send the notification to client.
  //

  if (handleFlush(flush_if_any) < 0)
  {
    return -1;
  }

  return 1;
}

int ClientChannel::handleReset()
{
  #ifdef TEST
  *logofs << "handleReset: Resetting client channel for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // As both message store and split store are going to
  // be recreated, we don't need to commit any request that
  // was being streamed to preserve store's integrity. This,
  // nevertheless, means we will loose those messages. It
  // would be nice to enter in a loop that would complete
  // any split and commit any pending request, but we cannot
  // rely anymore on data coming on socket. We can only make
  // sure that any stopped client is actually restarted.
  //

  int split[upperClient_ + 1];

  for (int client = lowerClient_;
           client <= upperClient_; client++)
  {
    split[client] = 0;
  }

  while (clientStore_ -> getSplitStore() -> pop())
  {
    int client     = clientStore_ -> getSplitStore() -> getLastClient();
    int completion = clientStore_ -> getSplitStore() -> getLastCompletion();

    if (client != nothing && completion == 1)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleReset: Saving client id " << client
              << " as waiting for deferred agent split event "
              << "TYPE#" << (unsigned int) opcodeStore_ -> splitNotify
              << ".\n" << logofs_flush;
      #endif

      split[client] = 1;
    }
    #if defined(INFO) || defined(TEST)
    else
    {
      int request  = clientStore_ -> getSplitStore() -> getLastRequest();
      int position = clientStore_ -> getSplitStore() -> getLastPosition();

      if (request != nothing && position != nothing)
      {
        *logofs << "handleReset: Discarding agent split event TYPE#"
                << (unsigned int) opcodeStore_ -> splitNotify << " for client "
                << client << " to commit " << "request " << request
                << " at position " << position << ".\n" << logofs_flush;
      }
    }
    #endif
  }

  //
  // Let base channel class to complete its own reset.
  //

  Channel::handleReset();

  //
  // Reallocate new caches. In protocol version 3
  // this is done in proxy that later propagates
  // pointers to channels
  //

  if (control -> isProtoStep3() == 0)
  {
    if (clientCache_ != NULL)
    {
      delete clientCache_;

      clientCache_ = new ClientCache();
    }

    if (serverCache_ != NULL)
    {
      delete serverCache_;

      serverCache_ = new ServerCache();
    }

    #ifdef TEST
    *logofs << "handleReset: Replaced encode caches in channel "
            << "for FD#" << fd_ << ".\n" << logofs_flush;
    #endif
  }

  //
  // Force channel to synchronize sequence
  // counters with the remote end.
  //

  clientSequenceReset_ = 1;
  serverSequenceReset_ = 1;

  //
  // Extract from queue any request for which we were
  // waiting for a reply and send an X error instead.
  //

  unsigned short int requestSequenceNum;
  unsigned char      requestOpcode;
  unsigned int       requestData[3];

  while (sequenceQueue_.pop(requestSequenceNum, requestOpcode,
                                requestData[0], requestData[1], requestData[2]))
  {
    //
    // Send an error instead of reply.
    //

    #if defined(INFO) || defined(TEST)
    *logofs << "handleReset: Sending error for request "
            << (unsigned int) requestOpcode << " with sequence "
            << requestSequenceNum << ". Request data is "
            << requestData[0] << " " << requestData[1] << " "
            << requestData[2] << ".\n" << logofs_flush;
    #endif

    unsigned char *error = writeBuffer_.addMessage(32);

    *(error + 0) = X_Error;
    *(error + 1) = BadAccess;

    PutUINT(requestSequenceNum, error + 2, bigEndian_);

    PutULONG(0, error + 4, bigEndian_);
    PutUINT(0, error + 8, bigEndian_);

    *(error + 10) = requestOpcode;

    if (handleFlush(flush_if_any) < 0)
    {
      return -1;
    }
  }

  //
  // Wakeup any agent's client that could
  // have been put to sleep.
  //

  for (int client = lowerClient_;
           client <= upperClient_; client++)
  {
    if (split[client] != 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleReset: Sending deferred agent split "
              << "event TYPE#" << (unsigned int) opcodeStore_ ->
                 splitNotify << " to wakeup " << "client id "
              << client << ".\n" << logofs_flush;
      #endif

      if (handleWakeup(opcodeStore_ -> splitNotify, deferred,
                           nothing, nothing, nothing, client) < 0)
      {
        return -1;
      }
    }

    if (isTimestamp(syncClientTs_[client]) == 1)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleReset: Sending deferred agent sync "
              << "event TYPE#" << (unsigned int) opcodeStore_ ->
                 syncNotify << " to client id " << client
              << ".\n" << logofs_flush;
      #endif

      if (handleWakeup(opcodeStore_ -> syncNotify, deferred,
                           client, nothing, nothing, nothing) < 0)
      {
        return -1;
      }
    }

    if (isTimestamp(karmaClientTs_[client]) == 1)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleReset: Sending deferred agent karma "
              << "event TYPE#" << (unsigned int) opcodeStore_ ->
                 karmaNotify << " to client id " << client
              << ".\n" << logofs_flush;
      #endif

      if (handleWakeup(opcodeStore_ -> karmaNotify, deferred,
                           client, nothing, nothing, nothing) < 0)
      {
        return -1;
      }
    }
  }

  //
  // Last request could need a reply, so we
  // have to force an error or client would
  // hang forever.
  //

  if (clientOpcode_ < X_NXFirstOpcode ||
          clientOpcode_ > X_NXLastOpcode)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "handleReset: Sending error for last received "
            << "request OPCODE#" << (unsigned int) clientOpcode_
            << " with sequence " << clientSequence_ << ".\n"
            << logofs_flush;
    #endif

    unsigned char *error = writeBuffer_.addMessage(32);

    *(error + 0) = X_Error;
    *(error + 1) = BadAccess;

    PutUINT(clientSequence_, error + 2, bigEndian_);

    PutULONG(0, error + 4, bigEndian_);
    PutUINT(0, error + 8, bigEndian_);

    *(error + 10) = clientOpcode_;

    if (handleFlush(flush_if_any) < 0)
    {
      return -1;
    }
  }

  clientOpcode_ = 0;
  serverOpcode_ = 0;

  lastSequence_ = (clientSequence_ > serverSequence_ ?
                           clientSequence_ : serverSequence_);

  #ifdef TEST
  *logofs << "handleReset: Updated last agent notify sequence to "
          << lastSequence_ << " for FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  splitClient_ = nothing;

  #ifdef TEST
  *logofs << "handleReset: Completed reset of client channel for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

//
// If differential compression is disabled then use the
// most simple encoding. Compress X_ListExtensions and
// X_QueryExtension messages (needed to get opcode of
// shape or hide extensions) and NX spefic messages in
// the usual way.
//

int ClientChannel::handleFastReadRequest(EncodeBuffer &encodeBuffer, const unsigned char &opcode,
                                             const unsigned char *&buffer, const unsigned int &size)
{
  //
  // We have 3 new NX requests in
  // the latest protocol version.
  //

  if ((opcode >= X_NXFirstOpcode &&
           opcode <= (X_NXLastOpcode - 3)) ||
               opcode == X_ListExtensions ||
                   opcode == X_QueryExtension)
  {
    return 0;
  }
  else if (control -> isProtoStep4() == 1 &&
               opcode > (X_NXLastOpcode - 3) &&
                   opcode <= X_NXLastOpcode)
  {
    return 0;
  }

  #ifdef DEBUG
  *logofs << "handleFastReadRequest: Encoding raw request OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_
          << " with size " << size << ".\n"
          << logofs_flush;
  #endif

  encodeBuffer.encodeMemory(buffer, size);

  //
  // We can't short-circuit the replies at client
  // side if there is any pending request waiting
  // a reply. Normal X clients, not designed to
  // deal with our benevolent protocol violations,
  // will sometimes get confused.
  //

  switch (opcode)
  {
    case X_AllocColor:
    case X_GetAtomName:
    case X_GetGeometry:
    case X_GetInputFocus:
    case X_GetModifierMapping:
    case X_GetKeyboardMapping:
    case X_GetProperty:
    case X_GetSelectionOwner:
    case X_GrabPointer:
    case X_GrabKeyboard:
    case X_InternAtom:
    case X_ListExtensions:
    case X_ListFonts:
    case X_LookupColor:
    case X_AllocNamedColor:
    case X_QueryPointer:
    case X_GetWindowAttributes:
    case X_QueryTree:
    case X_QueryBestSize:
    case X_QueryColors:
    case X_QueryFont:
    case X_TranslateCoords:
    case X_GetImage:
    case X_GetPointerMapping:
    case X_GetKeyboardControl:
    {
      //
      // Also put request on the fast
      // track if it needs a reply.
      //

      sequenceQueue_.push(clientSequence_, opcode);

      priority_++;

      break;
    }
    default:
    {
      //
      // Don't set priority in the case of RDP
      // sessions. The RDP agent uses a special
      // request to batch all the alloc colors
      // in a single write.
      //

      if (opcode == X_AllocColor)
      {
        sequenceQueue_.push(clientSequence_, opcode);

        if (control -> SessionMode != SESSION_RDP)
        {
          priority_++;
        }
      }
      else if (control -> isProtoStep3() == 1 &&
                   opcode != opcodeStore_ -> shapeExtension &&
                       opcode != opcodeStore_ -> renderExtension &&
                           opcode > 127)
      {
        priority_++;
      }

      break;
    }
  }

  int bits = encodeBuffer.getBits();

  #if defined(TEST) || defined(OPCODES)

  *logofs << "handleFastReadRequest: Handled raw request OPCODE#" 
          << (unsigned int) opcode << " (" << DumpOpcode(opcode) << ")"
          << " for FD#" << fd_ << " sequence " << clientSequence_
          << ". " << size << " bytes in, " << bits << " bits ("
          << ((float) bits) / 8 << " bytes) out.\n" << logofs_flush;

  #endif

  if (control -> CollectStatistics)
  {
    statistics -> addRequestBits(opcode, size << 3, bits);
  }

  control -> addOutputInARow(size);

  control -> addBitsInARow(bits);

  return 1;
}

int ClientChannel::handleFastWriteReply(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                                            unsigned char *&buffer, unsigned int &size)
{
  if ((opcode >= X_NXFirstOpcode &&
           opcode <= X_NXLastOpcode) ||
               opcode == X_ListExtensions ||
                   opcode == X_QueryExtension)
  {
    return 0;
  }

  #ifdef DEBUG
  *logofs << "handleFastWriteReply: Decoding raw reply OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  buffer = writeBuffer_.addMessage(8);

  memcpy(buffer, decodeBuffer.decodeMemory(8), 8);

  size = 32 + (GetULONG(buffer + 4, bigEndian_) << 2);

  writeBuffer_.registerPointer(&buffer);

  if (writeBuffer_.getAvailable() < size - 8 ||
          (int) size >= control -> TransportWriteThreshold)
  {
    #ifdef DEBUG
    *logofs << "handleFastWriteReply: Using scratch buffer for OPCODE#"
            << (unsigned int) opcode << " with size " << size << " and "
            << writeBuffer_.getLength() << " bytes in buffer.\n"
            << logofs_flush;
    #endif

    writeBuffer_.removeMessage(8);

    buffer = writeBuffer_.addScratchMessage(((unsigned char *)
                             decodeBuffer.decodeMemory(size - 8)) - 8, size);
  }
  else
  {
    writeBuffer_.addMessage(size - 8);

    memcpy(buffer + 8, decodeBuffer.decodeMemory(size - 8), size - 8);
  }

  writeBuffer_.unregisterPointer();

  //
  // We don't need to write our local sequence
  // number. Replies are always sent with the
  // original X server's sequence number.
  //

  #if defined(TEST) || defined(OPCODES)
  *logofs << "handleFastWriteReply: Handled raw reply OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_ << " with sequence "
          << serverSequence_ << ". Output size is " << size << ".\n"
          << logofs_flush;
  #endif

  #ifdef DEBUG
  *logofs << "handleFastWriteReply: Length of sequence queue is "
          << sequenceQueue_.length() << ".\n" << logofs_flush;
  #endif

  if (control -> CollectStatistics)
  {
    statistics -> addRepliedRequest(opcode);
  }

  //
  // Flush immediately on replies.
  //

  handleFlush(flush_if_any);

  return 1;
}

int ClientChannel::handleFastWriteEvent(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                                            unsigned char *&buffer, unsigned int &size)
{
  #ifdef DEBUG
  *logofs << "handleFastWriteEvent: Decoding raw "
          << (opcode == X_Error ? "error" : "event") << " OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  size = 32;

  buffer = writeBuffer_.addMessage(size);

  memcpy(buffer, decodeBuffer.decodeMemory(size), size);

  //
  // Use our local sequence number.
  //

  PutUINT(lastSequence_, buffer + 2, bigEndian_);

  #if defined(TEST) || defined(OPCODES)
  *logofs << "handleFastWriteEvent: Handled raw "
          << (opcode == X_Error ? "error" : "event") << " OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_ << " with sequence "
          << lastSequence_ << ". Output size is " << size << ".\n"
          << logofs_flush;
  #endif

  //
  // Check if we need to suppress the error.
  //

  if (control -> AgentTaintReplies > 0 && opcode == X_Error &&
          handleTaintError(*(buffer + 10)) == 1)
  {
    writeBuffer_.removeMessage(32);
  }

  handleFlush(flush_if_needed);

  return 1;
}

int ClientChannel::handleShmemInit(EncodeBuffer &encodeBuffer, const unsigned char opcode,
                                       unsigned int &stage, const unsigned char *buffer,
                                           const unsigned int size)
{
  //
  // Will push sequence and set
  // priority according to stage.
  //

  stage = *(buffer + 1);

  #ifdef TEST
  *logofs << "handleShmemInit: Encoding shmem request "
          << "OPCODE#" << (unsigned int) opcode << " for FD#"
          << fd_ << " with size " << size << " at stage "
          << stage << ".\n" << logofs_flush;
  #endif

  encodeBuffer.encodeValue(stage, 2);

  if (stage == 0)
  {
    unsigned int enableClient = 0;
    unsigned int enableServer = 0;

    if (control -> AgentShmemClient == 1)
    {
      enableClient = *(buffer + 4);
    }

    if (control -> AgentShmemServer == 1)
    {
      enableServer = *(buffer + 5);
    }

    encodeBuffer.encodeBool(enableClient);
    encodeBuffer.encodeBool(enableServer);

    unsigned int clientSegment = GetULONG(buffer + 8,  bigEndian_);
    unsigned int serverSegment = GetULONG(buffer + 12, bigEndian_);

    encodeBuffer.encodeValue(clientSegment, 29, 9);
    encodeBuffer.encodeValue(serverSegment, 29, 9);

    #ifdef TEST
    *logofs << "handleShmemInit: Enable client is "
            << enableClient << " enable server is " << enableServer
            << " client segment is " << (void *) clientSegment
            << " server segment is " << (void *) serverSegment
            << ".\n" << logofs_flush;
    #endif
  }

  return 1;
}

int ClientChannel::handleShmemInit(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                                       unsigned char *&buffer, unsigned int &size)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "handleShmemInit: Received shmem parameters "
          << "reply OPCODE#" << (unsigned int) opcode
          << ".\n" << logofs_flush;
  #endif

  size   = 32;
  buffer = writeBuffer_.addMessage(size);

  unsigned int stage;

  decodeBuffer.decodeValue(stage, 2);

  *(buffer + 1) = stage;

  if (stage == 2)
  {
    unsigned int clientEnabled;
    unsigned int serverEnabled;

    decodeBuffer.decodeBool(clientEnabled);
    decodeBuffer.decodeBool(serverEnabled);

    //
    // Client support is not yet implemented.
    //

    clientEnabled = 0;

    *(buffer + 8) = clientEnabled;
    *(buffer + 9) = serverEnabled;

    #if defined(INFO) || defined(TEST)

    if (serverEnabled == 1)
    {
      *logofs << "handleShmemInit: Enabled shared memory "
              << "support in X server.\n" << logofs_flush;
    }

    #endif
  }
  else
  {
    *(buffer + 8) = 0;
    *(buffer + 9) = 0;
  }

  return 1;
}

void ClientChannel::handleDecodeCharInfo(DecodeBuffer& decodeBuffer, unsigned char *nextDest)
{
  unsigned int value;

  decodeBuffer.decodeCachedValue(value, 32,
                     *serverCache_ -> queryFontCharInfoCache[0], 6);

  PutUINT(value & 0xffff, nextDest, bigEndian_);
  PutUINT(value >> 16, nextDest + 10, bigEndian_);

  nextDest += 2;

  for (unsigned int i = 1; i < 5; i++)
  {
    unsigned int value;

    decodeBuffer.decodeCachedValue(value, 16,
                       *serverCache_ -> queryFontCharInfoCache[i], 6);

    PutUINT(value, nextDest, bigEndian_);

    nextDest += 2;
  }
}

void ClientChannel::setBigEndian(int flag)
{
  bigEndian_ = flag;
}
