/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <fstream.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include "Proxy.h"

#include "Socket.h"
#include "Channel.h"
#include "Statistics.h"
#include "Alerts.h"

//
// We need to adjust some values related
// to these messages at the time cache
// is reconfigured.
//

#include "PutImage.h"
#include "ChangeGC.h"
#include "PolyFillRectangle.h"
#include "PutPackedImage.h"

//
// Set the verbosity level. You also
// need to define DUMP in Misc.cpp
// if DUMP is defined here.
//

#define WARNING
#define PANIC
#undef  OPCODES
#undef  TEST
#undef  DEBUG
#undef  DUMP

Proxy::Proxy(int fd)

  : transport_(new ProxyTransport(fd)), fd_(fd), readBuffer_(transport_)
{
  for (int channelId = 0;
           channelId < CONNECTIONS_LIMIT;
               channelId++)
  {
    channels_[channelId]    = NULL;
    transports_[channelId]  = NULL;
    congestions_[channelId] = 0;
  }

  inputChannel_   = -1;
  outputChannel_  = -1;

  lowerChannel_ = 0;
  upperChannel_ = 0;
  firstChannel_ = 0;

  activeChannels_ = 0;

  controlLength_ = 0;

  operation_ = OPERATION_IN_MESSAGES;

  pending_    = 0;
  priority_   = 0;
  shutdown_   = 0;
  reset_      = 0;
  congestion_ = 0;

  lastBytesInTs_  = getTimestamp();
  lastBytesOutTs_ = getTimestamp();

  lastRetryTs_    = nullTimestamp();
  lastRetryIndex_ = 0;

  lastRetryTable_[0] = 4;
  lastRetryTable_[1] = 3;
  lastRetryTable_[2] = 2;
  lastRetryTable_[3] = 1;
  lastRetryTable_[4] = 1.1;

  lastPingTs_  = getTimestamp();
  lastAlertTs_ = nullTimestamp();

  lastLimitTs_ = nullTimestamp();

  lastStatisticsStream_ = NULL;

  //
  // Create compressor and decompressor
  // for image and data payload.
  //

  compressor_ = new Compressor(control -> LocalDataCompressionLevel,
                                   control -> LocalDataCompressionThreshold);

  decompressor_ = new Decompressor();

  //
  // Create object storing NX specific
  // opcodes.
  //

  opcodeStore_ = new OpcodeStore();

  //
  // Create the message stores.
  //

  clientStore_ = new ClientStore(compressor_, decompressor_);
  serverStore_ = new ServerStore(compressor_, decompressor_);

  lastLoadTs_ = nullTimestamp();

  if (control -> isProtoStep3() == 1)
  {
    clientCache_ = new ClientCache();
    serverCache_ = new ServerCache();

    if (clientCache_ == NULL || serverCache_ == NULL)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Failed to create encode caches.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Failed to create encode caches.\n";

      HandleCleanup();
    }
  }
  else
  {
    clientCache_ = NULL;
    serverCache_ = NULL;
  }

  //
  // Cache of the internal proxy state.
  //

  stateReadProxy_    = -1;
  stateReadChannels_ = -1;

  stateUnblockProxy_    = -1;
  stateUnblockChannels_ = -1;

  stateLimit_      = -1;
  stateRetry_      = -1;
  statePending_    = -1;
  statePriority_   = -1;
  stateMotion_     = -1;
  stateWakeup_     = -1;
  stateCongestion_ = -1;
  stateSplit_      = -1;
  stateFlush_      = -1;

  stateBlocked_   = -1;
  stateQueued_    = -1;
  stateLength_    = -1;
  stateFlushable_ = -1;

  stateNotify_ = -1;

  #ifdef DEBUG
  *logofs << "Proxy: Created new object at " << this
          << ".\n" << logofs_flush;
  #endif
}

Proxy::~Proxy()
{
  for (int channelId = 0;
           channelId < CONNECTIONS_LIMIT;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      delete transports_[channelId];
      transports_[channelId] = NULL;

      delete channels_[channelId];
      channels_[channelId] = NULL;
    }
  }

  delete transport_;

  delete compressor_;
  delete decompressor_;

  //
  // Delete storage shared among channels.
  //

  delete opcodeStore_;

  delete clientStore_;
  delete serverStore_;

  delete clientCache_;
  delete serverCache_;

  #ifdef DEBUG
  *logofs << "Proxy: Deleted proxy object at " << this
          << ".\n" << logofs_flush;
  #endif
}

//
// This function is executed at the beginning
// of eache loop. Set for how long and for how
// many bytes this loop is going to last.
//

void Proxy::setSchedule()
{
  //
  // Only collect enough bytes to make a full
  // scheduled write. This includes data from
  // previous blocking writes and ZLIB data
  // to flush.
  //

  control -> resetBytesInARow();

  control -> resetOutputInARow();

  control -> resetTimeInARow();

  //
  // Impose an initial queuing for any new
  // proxy data that is going to be sent.
  //

  if (stateFlush_ == -1)
  {
    control -> setLastFlushInfo();
  }

  //
  // If we were in retry state, check if we left
  // behind the split timeout. This check is very
  // fast and thus we avoid to run another loop
  // with a null timeout.
  //

  if (stateRetry_ == fd_ && control -> isTimeToSplit() == 1)
  {
    stateSplit_ = needSplit();
  }

  #ifdef TEST
  *logofs << "Proxy: Loop started at " << strMsTimestamp()
          << " with " << control -> getTimeToNextFlush()
          << " Ms to next flush.\n" << logofs_flush;
  #endif

  #ifdef TEST
  *logofs << "Proxy: Set scheduled write at "
          << control -> getBytesToNextFlush() << " bytes from now "
          << "with length " << transport_ -> length()
          << ".\n" << logofs_flush;
  #endif

  #ifdef TEST
  *logofs << "Proxy: Output in a row is "
          << control -> getOutputInARow() << " bytes with time in a row "
          << control -> getTimeInARow() << " Ms.\n"
          << logofs_flush;
  #endif
}

//
// The following functions are executed at the end
// of the main loop to setup the upcoming select.
// Add the file descriptors of all connections to
// attend.
//

void Proxy::setReadFDs(fd_set *fdSet, int &fdMax)
{
  stateLimit_ = -1;

  stateReadProxy_    = 0;
  stateReadChannels_ = 0;

  //
  // Update cache of internal state.
  //

  stateBlocked_   = transport_ -> blocked();
  stateQueued_    = transport_ -> queued();
  stateLength_    = transport_ -> length();
  stateFlushable_ = transport_ -> flushable();

  //
  // When ZLIB compression is enabled, length reported
  // by transport includes both data from previous
  // blocking writes and data still to be compressed.
  // We need to distinguish between the two values.
  //

  stateLength_ -= stateFlushable_;

  int fd;

  int fdFinish;
  int fdPending;
  int fdCongestion;

  int fdRequisite;

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    //
    // Sanity check on enqued data on the proxy link.
    // If proxy is blocked or data exceeds the limit
    // then stop reading from any X connections.
    // Check even if there are proxy messages pending
    // as there could be some congestions to attend.
    //

    fdRequisite = (reset_ == 0 && congestion_ == 0 &&
                       pending_ == 0 && stateBlocked_ == 0);

    if (fdRequisite == 1)
    {
      fdRequisite = (control -> FlushTimeout == 0 ||
                         (stateLength_ + stateQueued_) <
                              control -> LocalBytesInARowLimit);
    }
  }
  else
  {
    //
    // Be less picky at server side. Consider advisable
    // to maximize the throughput of path 'X server to
    // clients', given it is mostly represented by X
    // protocol replies.
    //

    fdRequisite = (reset_ == 0 && congestion_ == 0 &&
                       stateBlocked_ == 0);

    if (fdRequisite == 1)
    {
      fdRequisite = (control -> FlushTimeout == 0 ||
                         stateLength_ < control -> LocalBytesInARowLimit);
    }
  }

  if (fdRequisite == 1)
  {
    for (int channelId = lowerChannel_;
             channelId <= upperChannel_;
                 channelId++)
    {
      if (channels_[channelId] != NULL)
      {
        fd = getFd(channelId);

        if (channels_[channelId] -> needLimit())
        {
          #ifdef TEST
          *logofs << "Proxy: WARNING! Channel for descriptor FD#"
                  << fd << " is beyond its buffer limit.\n"
                  << logofs_flush;
          #endif

          stateLimit_ = fd;
        }

        if (fdRequisite == 1)
        {
          fdFinish  = channels_[channelId] -> getFinish();
          fdPending = channels_[channelId] -> getPending();

          fdCongestion = congestions_[channelId];

          //
          // Select X connections to read from. Read buffer is
          // checked. If there is still data to encode since
          // last read then don't get more messages.
          //

          if (fdFinish == 0 && fdPending == 0 && fdCongestion == 0)
          {
            stateReadChannels_ = 1;

            FD_SET(fd, fdSet);

            #ifdef TEST
            *logofs << "Proxy: Descriptor FD#" << fd
                    << " selected for read with buffer length "
                    << transports_[channelId] -> length()
                    << ".\n" << logofs_flush;
            #endif
          }
          #ifdef TEST
          else
          {
            *logofs << "Proxy: Descriptor FD#" << fd
                    << " not selected for read with finish " << fdFinish
                    << " pending " << fdPending << " congestion " << fdCongestion
                    << " length " << transports_[channelId] -> length()
                    << ".\n" << logofs_flush;
          }
          #endif
        }

        #ifdef DEBUG
        *logofs << "Proxy: Including descriptor FD#" << fd
                << " in count of total descriptors.\n"
                << logofs_flush;
        #endif

        if (fd >= fdMax)
        {
          fdMax = fd + 1;
        }
      }
    }
  }
  #if defined(INFO) || defined(TEST)
  else
  {
    *logofs << "Proxy: WARNING! Disabled reading from X connections.\n"
            << logofs_flush;

    *logofs << "Proxy: WARNING! Reset is " << reset_ << " congestion "
            << congestion_ << " pending " << pending_ << " blocked "
            << stateBlocked_ << " length " << stateLength_
            << " queued " << stateQueued_ << ".\n"
            << logofs_flush;
  }
  #endif

  //
  // Ensure we return to attend to proxy
  // if limit timeout is elapsed.
  //

  if (stateLimit_ != -1)
  {
    if (control -> LimitTimeout > 0)
    {
      int diffTs = diffTimestamp(lastLimitTs_, getTimestamp());

      if (diffTs < 0)
      {
        #ifdef DEBUG
        *logofs << "Proxy: Setting new limit timeout "
                << "for proxy FD#" << fd_ << ".\n"
                << logofs_flush;
        #endif

        lastLimitTs_ = getTimestamp();
      }
      else if (diffTs >= (control -> LimitTimeout -
                              (control -> LatencyTimeout * 10)))
      {
        #ifdef TEST
        *logofs << "Proxy: WARNING! Attending proxy FD#"
                << fd_ << " with " << diffTs << " Ms "
                << "since last read.\n" << logofs_flush;
        #endif

        stateLimit_ = -1;
      }
    }
    else
    {
      #ifdef DEBUG
      *logofs << "Proxy: Overriding limit timeout "
              << "for proxy FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      stateLimit_ = -1;
    }
  }

  //
  // Sanity check on enqued data to X connections.
  // If it exceeds the buffer limit or there are
  // pending messages in the read buffer then stop
  // reading from proxy.
  //

  int proxyRequisite = (pending_ == 0 && stateLimit_ == -1);

  if (proxyRequisite == 1)
  {
    stateReadProxy_ = 1;

    FD_SET(fd_, fdSet);

    #ifdef TEST

    int fdMaxLength = 0;

    for (int channelId = lowerChannel_;
             channelId <= upperChannel_;
                 channelId++)
    {
      if (channels_[channelId] != NULL &&
              transports_[channelId] -> length() > fdMaxLength)
      {
        fdMaxLength = transports_[channelId] -> length();
      }
    }

    *logofs << "Proxy: Proxy descriptor FD#" << fd_ << " selected "
            << "for read. Maximum size of buffers is " << fdMaxLength
            << ".\n" << logofs_flush;

    #endif
  }
  #if defined(INFO) || defined(TEST)
  else
  {
    if (pending_ != 0)
    {
      *logofs << "Proxy: WARNING! Disabled reading from proxy link "
              << "FD#" << fd_ << " with pending " << pending_
              << ".\n" << logofs_flush;
    }
    else
    {
      *logofs << "Proxy: WARNING! Disabled reading from proxy link "
              << "FD#" << fd_ << " with offending channel "
              << stateLimit_ << ".\n" << logofs_flush;
    }
  }
  #endif

  #ifdef DEBUG
  *logofs << "Proxy: Including proxy descriptor FD#" << fd_
          << " in count of total descriptors.\n"
          << logofs_flush;
  #endif

  if (fd_ >= fdMax)
  {
    fdMax = fd_ + 1;
  }
}

//
// Add file descriptors of all X connections to write to.
//

void Proxy::setWriteFDs(fd_set *fdSet, int &fdMax)
{
  stateUnblockProxy_    = 0;
  stateUnblockChannels_ = 0;

  int fd;

  int fdLength;
  int fdBlocked;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      fd = getFd(channelId);

      fdLength  = transports_[channelId] -> length();
      fdBlocked = transports_[channelId] -> blocked();

      if (fdLength > 0)
      {
        stateUnblockChannels_ = 1;

        FD_SET(fd, fdSet);

        #ifdef TEST
        *logofs << "Proxy: Descriptor FD#" << fd 
                << " selected for write. Blocked is " << fdBlocked 
                << " length is " << fdLength << ".\n" << logofs_flush;
        #endif

        if (fd >= fdMax)
        {
          fdMax = fd + 1;
        }
      }
      #ifdef TEST
      else
      {
        *logofs << "Proxy: Descriptor FD#" << fd 
                << " not selected for write. Blocked is " << fdBlocked 
                << " length is " << fdLength << ".\n" << logofs_flush;
      }
      #endif

      #if defined(INFO) || defined(TEST)

      if (fdLength > 0 && fdBlocked != 1)
      {
        *logofs << "Proxy: WARNING! Descriptor FD#" << fd
                << " has data to write but blocked flag is "
                << fdBlocked << ".\n" << logofs_flush;

        cerr << "Warning" << ": Descriptor FD#" << fd
             << " has data to write but blocked flag is "
             << fdBlocked << ".\n";
      }

      #endif
    }
  }

  //
  // Check if the proxy link has data to be written.
  //

  if (stateLength_ > 0 && stateBlocked_ == 1)
  {
    stateUnblockProxy_ = 1;

    FD_SET(fd_, fdSet);

    #ifdef TEST
    *logofs << "Proxy: Proxy descriptor FD#"
            << fd_ << " selected for write. Blocked is "
            << stateBlocked_ << " length is " << stateLength_
            << ".\n" << logofs_flush;
    #endif

    if (fd_ >= fdMax)
    {
      fdMax = fd_ + 1;
    }
  }
  #ifdef TEST
  else
  {
    *logofs << "Proxy: Proxy descriptor FD#"
            << fd_ << " not selected for write. Blocked is "
            << stateBlocked_ << " length is " << stateLength_
            << ".\n" << logofs_flush;
  }
  #endif

  #if defined(INFO) || defined(TEST)

  if (stateLength_ > 0 && stateBlocked_ == 0 &&
          control -> FlushTimeout == 0)
  {
    *logofs << "Proxy: WARNING! Proxy descriptor FD#" << fd_
            << " has data to write but blocked flag is "
            << stateBlocked_ << ".\n" << logofs_flush;

    cerr << "Warning" << ": Proxy descriptor FD#" << fd_
         << " has data to write but blocked flag is "
         << stateBlocked_ << ".\n";
  }

  #endif
}

void Proxy::setTimeout(T_timestamp &tsMax)
{
  //
  // Query the internal state of proxy and channels.
  //

  statePending_    = needPending();
  stateMotion_     = needMotion();
  stateWakeup_     = needWakeup();
  stateCongestion_ = needCongestion();
  stateSplit_      = needSplit();
  stateFlush_      = needFlush();

  //
  // Initial timeout used to ping the remote proxy.
  //

  setTimestamp(tsMax, control -> PingTimeout);

  //
  // Can we read from remote proxy?
  //

  if (stateLimit_ == -1)
  {
    //
    // Yes, so reset any previous timeout.
    //

    lastLimitTs_ = nullTimestamp();

    //
    // Always try to restart the proxy if there is data in
    // its own read buffer. We cannot postpone handling of
    // control messages coming from remote end.
    //

    if (pending_ == 1)
    {
      #ifdef TEST
      *logofs << "Proxy: Requesting timeout of "
              << control -> PendingTimeout << " Ms as proxy FD#"
              << fd_ << " has pending messages.\n"
              << logofs_flush;
      #endif

      if (tsMax.tv_sec > 0 ||
              tsMax.tv_usec > control -> PendingTimeout * 1000)
      {
        tsMax.tv_sec  = 0;
        tsMax.tv_usec = control -> PendingTimeout * 1000;
      }
    }
  }
  else
  {
    //
    // Some channels reached the buffer limit. By having
    // stopped reading from remote proxy we can count on
    // the fact it received the congestion event and is
    // not attending to the offending clients. As soon
    // as timeout is elapsed we can start reading again
    // from our peer.
    //

    #ifdef TEST
    *logofs << "Proxy: Requesting timeout of "
            << control -> LimitTimeout << " Ms with FD#"
            << stateLimit_ << " having reached the "
            << "buffer limit.\n" << logofs_flush;
    #endif

    if (tsMax.tv_sec > 0 ||
            tsMax.tv_usec > control -> LimitTimeout * 1000)
    {
      tsMax.tv_sec  = 0;
      tsMax.tv_usec = control -> LimitTimeout * 1000;
    }
  }

  //
  // Check if there is a transition in congestion
  // state to handle.
  //

  if (stateCongestion_ != -1)
  {
    #ifdef TEST
    *logofs << "Proxy: Requesting timeout of "
            << control -> CongestionTimeout << " Ms as FD#"
            << stateCongestion_ << " has congestion events "
            << "to send.\n" << logofs_flush;
    #endif

    if (tsMax.tv_sec > 0 ||
            tsMax.tv_usec > control -> CongestionTimeout * 1000)
    {
      tsMax.tv_sec  = 0;
      tsMax.tv_usec = control -> CongestionTimeout * 1000;
    }
  }

  //
  // In case of network congestion, don't wakeup agent's
  // clients, delay sending of further splits and wait
  // until the next timeout to send outstanding data to
  // the remote proxy.
  //
  // We need to keep TCP buffers big enough, or we would
  // otherwise heavily affect the overall throughput. At
  // the same time, cannot rely on blocked flag as this
  // would cause too much latency when running over slow
  // links. So we must monitor the socket's TCP buffer to
  // ensure that all data enqueued from previous writes
  // has been actually written to the network.
  //

  int proxyRequisite;
  int proxySafeRequisite;

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    proxyRequisite = (reset_ == 0 && congestion_ == 0 &&
                          pending_ == 0 && stateBlocked_ == 0);

    proxySafeRequisite = (control -> FlushTimeout == 0 ||
                              (stateLength_ + stateQueued_) <
                                   control -> LocalBytesInARowLimit);
  }
  else
  {
    proxyRequisite = (reset_ == 0 && congestion_ == 0 &&
                          stateBlocked_ == 0);

    proxySafeRequisite = (control -> FlushTimeout == 0 ||
                              stateLength_ < control -> LocalBytesInARowLimit);
  }

  if (proxyRequisite == 0 || proxySafeRequisite == 0)
  {
    //
    // We are in retry state.
    //

    stateRetry_ = fd_;

    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Delaying timeouts on proxy FD#" << fd_
            << " with congestion " << congestion_ << " pending "
            << pending_ << " and blocked " << stateBlocked_
            << ".\n" << logofs_flush;

    *logofs << "Proxy: State of link is length " << stateLength_
            << " flushable " << stateFlushable_ << " and queued "
            << stateQueued_ << ".\n" << logofs_flush;
    #endif

    //
    // We must wait for proxy link to become writable.
    // Reset split timeout (as we want to give priority
    // to normal X traffic) and flush timeout (as we
    // have to wait for at least that amount of time).
    // Finally, slow down the most demanding clients by
    // moving forward the 'put to sleep at' timestamp.
    //

    control -> setLastSplitInfo();

    control -> setLastFlushInfo();

    handleUpdateWakeup();

    stateMotion_ = -1;
    stateWakeup_ = -1;
    stateSplit_  = -1;
    stateFlush_  = -1;

    if (proxyRequisite == 1 && proxySafeRequisite == 0)
    {
      //
      // Be sure to give a chance to proxy to check how
      // many bytes are still to be written to the link
      // from the underlying TCP layer. A timeout isn't
      // obviously not the best solution. We should be
      // notified as soon as size of the TCP buffer has
      // changed or have a tiny thread checking for this.
      //

      if (stateUnblockProxy_ == 0)
      {
        if (tsMax.tv_sec > 0 ||
                tsMax.tv_usec > lastRetryTs_.tv_usec)
        {
          tsMax.tv_sec  = 0;
          tsMax.tv_usec = lastRetryTs_.tv_usec;
        }

        handleIncreaseRetry();

        #ifdef DEBUG
        *logofs << "Proxy: Next retry timeout will be "
                << lastRetryTs_.tv_usec / 1000 << " Ms.\n"
                << logofs_flush;
        #endif
      }
    }
  }
  else
  {
    //
    // We are out of proxy link retry state.
    //

    stateRetry_ = -1;

    //
    // Retry timeout is increased at any failed
    // restart. We are out of proxy congestion
    // so we can reset it to its initial value.
    //

    handleResetRetry();

    #ifdef DEBUG
    *logofs << "Proxy: Next retry timeout reset to "
            << lastRetryTs_.tv_usec / 1000 << " Ms.\n"
            << logofs_flush;
    #endif

    //
    // Check if we left data in any read buffer.
    //

    if (statePending_ != -1)
    {
      #ifdef TEST
      *logofs << "Proxy: Requesting timeout of "
              << control -> PendingTimeout << " Ms as FD#"
              << statePending_ << " has pending messages.\n"
              << logofs_flush;
      #endif

      if (tsMax.tv_sec > 0 ||
              tsMax.tv_usec > control -> PendingTimeout * 1000)
      {
        tsMax.tv_sec  = 0;
        tsMax.tv_usec = control -> PendingTimeout * 1000;
      }
    }

    //
    // Restart the proxy if there are motion
    // events to flush.
    //

    if (stateMotion_ != -1)
    {
      #ifdef TEST
      *logofs << "Proxy: Requesting timeout of "
              << control -> MotionTimeout << " Ms as FD#"
              << stateMotion_ << " has motion events "
              << "to flush.\n" << logofs_flush;
      #endif

      if (tsMax.tv_sec > 0 ||
              tsMax.tv_usec > control -> MotionTimeout * 1000)
      {
        tsMax.tv_sec  = 0;
        tsMax.tv_usec = control -> MotionTimeout * 1000;
      }
    }

    //
    // Check if we need to restart agent's clients.
    //

    if (stateWakeup_ != -1)
    {
      #ifdef TEST
      *logofs << "Proxy: Requesting timeout of "
              << control -> WakeupTimeout << " Ms as FD#"
              << stateWakeup_ << " has wakeup events "
              << "to send.\n" << logofs_flush;
      #endif

      if (tsMax.tv_sec > 0 ||
              tsMax.tv_usec > control -> WakeupTimeout * 1000)
      {
        tsMax.tv_sec  = 0;
        tsMax.tv_usec = control -> WakeupTimeout * 1000;
      }
    }

    if (stateSplit_ != -1)
    {
      int timeToNextSplit = control -> getTimeToNextSplit();

      #ifdef TEST
      *logofs << "Proxy: Requesting timeout of "
              << timeToNextSplit << " Ms as FD#"
              << stateSplit_ << " has splits to send.\n"
              << logofs_flush;
      #endif

      if (tsMax.tv_sec > 0 ||
              tsMax.tv_usec > timeToNextSplit * 1000)
      {
        tsMax.tv_sec  = 0;
        tsMax.tv_usec = timeToNextSplit * 1000;
      }
    }

    if (stateFlush_ != -1)
    {
      int timeToNextFlush = control -> getTimeToNextFlush();

      #ifdef TEST
      *logofs << "Proxy: Requesting timeout of "
              << timeToNextFlush << " Ms as FD#"
              << stateFlush_ << " has data to flush.\n"
              << logofs_flush;
      #endif

      if (tsMax.tv_sec > 0 ||
              tsMax.tv_usec > timeToNextFlush * 1000)
      {
        tsMax.tv_sec  = 0;
        tsMax.tv_usec = timeToNextFlush * 1000;
      }
    }
  }

  #ifdef TEST
  *logofs << "Proxy: Loop completed at " << strMsTimestamp()
          << " with timeout of " << tsMax.tv_sec << " S and "
          << (double) tsMax.tv_usec / 1000 << " Ms.\n\n"
          << logofs_flush;
  #endif
}

int Proxy::needPending() const
{
  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            congestions_[channelId] == 0 &&
                channels_[channelId] -> getPending() > 0)
    {
      return getFd(channelId);
    }
  }

  return -1;
}

int Proxy::needPriority() const
{
  if (priority_ > 0)
  {
    #ifdef TEST
    *logofs << "Proxy: Forcing flush with priority on "
            << "proxy FD#" << fd_ << ".\n" << logofs_flush;
    #endif
    
    return fd_;
  }

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      if (channels_[channelId] -> getPriority() > 0)
      {
        return getFd(channelId);
      }

      if (control -> AgentFlushPriority != 0 &&
              channels_[channelId] -> getFlush() > 0)
      {
        #ifdef TEST
        *logofs << "Proxy: Flush for FD#" << getFd(channelId)
                << " is " << channels_[channelId] -> getFlush()
                << ".\n" << logofs_flush;
        #endif

        if ((stateLength_ + stateFlushable_) >=
                control -> LocalBytesInARowLimit /
                    control -> AgentFlushPriority)
        {
          #ifdef TEST
          *logofs << "Proxy: Forcing flush with length "
                  << stateLength_ << " and flushable " << stateFlushable_
                  << " because of FD#" << getFd(channelId) << " having priority "
                  << channels_[channelId] -> getPriority() << " and flush "
                  << channels_[channelId] -> getFlush() << ".\n"
                  << logofs_flush;
          #endif

          return getFd(channelId);
        }
      }
    }
  }

  return -1;
}

int Proxy::needLimit() const
{
  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> needLimit() > 0)
    {
      return getFd(channelId);
    }
  }

  return -1;
}

int Proxy::needCongestion() const
{
  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> needCongestion() > 0)
    {
      return getFd(channelId);
    }
  }

  return -1;
}

int Proxy::getChannels(T_channel_type type)
{
  int channels = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            (type == CHANNEL_NONE ||
                 type == channels_[channelId] ->
                     getType()))
    {
      channels++;
    }
  }

  return channels;
}

T_channel_type Proxy::getType(int fd)
{
  int channelId = getChannel(fd);

  if (channelId < 0 || channels_[channelId] == NULL)
  {
    return CHANNEL_NONE;
  }

  return channels_[channelId] -> getType();
}

//
// Handle data from channels selected for read
// or having pending messages in their buffer.
//

int Proxy::handleRead(fd_set &fdSet)
{
  #ifdef DEBUG
  *logofs << "Proxy: Looping through descriptors selected for read.\n"
          << logofs_flush;
  #endif

  for (int j = (firstChannel_ + 1 > upperChannel_ ?
                    lowerChannel_ : firstChannel_ + 1);
                        ; (j + 1 > upperChannel_ ?
                               j = lowerChannel_ : j++))
  {
    #ifdef DEBUG
    *logofs << "Proxy: Looping with first " << firstChannel_
            << " and current " << j << ".\n" << logofs_flush;
    #endif

    int fd = getFd(j);

    if (fd >= 0 && (FD_ISSET(fd, &fdSet) || getPending(fd) > 0))
    {
      #ifdef DEBUG
      *logofs << "Proxy: Going to read messages from FD#"
              << fd << ".\n" << logofs_flush;
      #endif

      if (handleRead(fd) < 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Loop: Failure in proxy reading messages "
                << "from FD#" << fd << ".\n" << logofs_flush;
        #endif

        return -1;
      }
    }

    if (j == firstChannel_)
    {
      firstChannel_ = (firstChannel_ + 1 > upperChannel_ ?
                           lowerChannel_ : firstChannel_ + 1);
      break;
    }
  }

  return 1;
}

int Proxy::handleRead()
{
  if ((pending_ == 0 && stateReadProxy_ == 0) ||
           stateLimit_ != -1)
  {
    #ifdef TEST
    if (stateLimit_ != -1)
    {
      *logofs << "Proxy: WARNING! Proxy could have data to read "
              << "but channels reached the buffer limit.\n"
              << logofs_flush;
    }
    #endif

    return 0;
  }

  #ifdef TEST
  *logofs << "Proxy: Handling data from the peer proxy.\n"
          << logofs_flush;
  #endif

  //
  // If there are pending messages then get
  // only the available data from buffer,
  // else read from the network.
  //

  int result = 0;
  int yield  = 0;

  if (pending_ == 0)
  {
    //
    // Read result reflects number of bytes
    // enqueued in the read buffer.
    //

    result = readBuffer_.readMessage();

    if (result < 0)
    {
      if (shutdown_ == 0)
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Lost connection to peer proxy on FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Lost connection to peer proxy on FD#"
             << fd_ << ".\n";
      }
      #if defined(INFO) || defined(TEST)
      else
      {
        *logofs << "Proxy: Close of proxy link detected after clean shutdown.\n"
                << logofs_flush;
      }
      #endif

      return -1;
    }
    else if (result == 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Proxy: WARNING! Not enough data read from "
              << "proxy FD#" << fd_ << "\n" << logofs_flush;
      #endif

      return 0;
    }
    else
    {
      //
      // Set time of last read from remote
      // proxy.
      //

      lastBytesInTs_ = getTimestamp();

      //
      // Jump out of proxy congestion state.
      //

      congestion_ = 0;

      //
      // Show the 'no data received' dialog
      // at next timeout.
      //

      lastAlertTs_ = nullTimestamp();
    }
  }

  #if defined(TEST) || defined(OPCODES)
  *logofs << "Proxy: Getting messages from proxy FD#" << fd_
          << " with " << readBuffer_.getLength() << " bytes "
          << "in the read buffer.\n" << logofs_flush;
  #endif

  unsigned int controlLength;
  unsigned int dataLength;

  const unsigned char *message;

  while (yield == 0 && (message = readBuffer_.getMessage(controlLength, dataLength)) != NULL)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addFrameIn();
    }

    if ((controlLength == 3) && (message[0] == 0) &&
            (message[1] < CODE_LAST_TAG))
    {
      if (message[1] == CODE_SWITCH_CONNECTION)
      {
        int channelId = message[2];

        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_SWITCH_CONNECTION message "
                << "with FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
        #endif

        //
        // If channel is invalid further messages will
        // be ignored. The acknowledged shutdown of
        // channels should prevent this.
        //

        inputChannel_ = channelId;
      }
      else if (message[1] == CODE_BEGIN_CONGESTION)
      {
        //
        // Set congestion state for channel
        // reported by remote proxy.
        //

        int channelId = message[2];

        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_BEGIN_CONGESTION message "
                << "for FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
        #endif

        if (channels_[channelId] != NULL)
        {
          congestions_[channelId] = 1;
        }
        #ifdef WARNING
        else
        {
          *logofs << "Proxy: WARNING! Received a begin congestion message "
                  << "for invalid channel id ID#" << channelId
                  << " with FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
        }
        #endif
      }
      else if (message[1] == CODE_END_CONGESTION)
      {
        //
        // Attend again to channel.
        //

        int channelId = message[2];

        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_END_CONGESTION message "
                << "for FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
        #endif

        if (channels_[channelId] != NULL)
        {
          congestions_[channelId] = 0;

          //
          // We can now attend the following events.
          //

          statePending_ = needPending();
          stateMotion_  = needMotion();
          stateWakeup_  = needWakeup();
        }
        #ifdef WARNING
        else
        {
          *logofs << "Proxy: WARNING! Received an end congestion message "
                  << "for invalid channel id ID#" << channelId
                  << " with FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
        }
        #endif
      }
      else if (message[1] == CODE_PING_REQUEST)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_PING_REQUEST message.\n"
                << logofs_flush;
        #endif

        if (handlePingFromProxy() < 0)
        {
          return -1;
        }
      }
      else if (message[1] == CODE_PING_REPLY)
      {
        //
        // Nothing to do except that this will
        // update the appropriate timestamp.
        //

        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_PING_REPLY message.\n"
                << logofs_flush;
        #endif
      }
      else if (message[1] == CODE_NEW_X_CONNECTION ||
                   message[1] == CODE_NEW_SYNC_CONNECTION ||
                       message[1] == CODE_NEW_KEYBD_CONNECTION ||
                           message[1] == CODE_NEW_SAMBA_CONNECTION ||
                               message[1] == CODE_NEW_MEDIA_CONNECTION)
      {
        int result;

        int channelId = message[2];

        if (message[1] == CODE_NEW_X_CONNECTION)
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Received CODE_NEW_X_CONNECTION message.\n"
                  << logofs_flush;
          #endif

          result = handleNewXConnectionFromProxy(channelId);
        }
        else if (message[1] == CODE_NEW_SYNC_CONNECTION)
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Received CODE_NEW_SYNC_CONNECTION message.\n"
                  << logofs_flush;
          #endif

          result = handleNewSyncConnectionFromProxy(channelId);
        }
        else if (message[1] == CODE_NEW_KEYBD_CONNECTION)
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Received CODE_NEW_KEYBD_CONNECTION message.\n"
                  << logofs_flush;
          #endif

          result = handleNewKeybdConnectionFromProxy(channelId);
        }
        else if (message[1] == CODE_NEW_SAMBA_CONNECTION)
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Received CODE_NEW_SAMBA_CONNECTION message.\n"
                  << logofs_flush;
          #endif

          result = handleNewSambaConnectionFromProxy(channelId);
        }
        else
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Received CODE_NEW_MEDIA_CONNECTION message.\n"
                  << logofs_flush;
          #endif

          result = handleNewMediaConnectionFromProxy(channelId);
        }

        if (result < 0)
        {
          //
          // Realization of new channel failed.
          // Send channel shutdown message to
          // the peer proxy.
          //

          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Sending CODE_DROP_CONNECTION message.\n"
                  << logofs_flush;
          #endif

          if (handleControl(CODE_DROP_CONNECTION, channelId) < 0)
          {
            return -1;
          }
        }
      }
      else if (message[1] == CODE_DROP_CONNECTION)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_DROP_CONNECTION message.\n"
                << logofs_flush;
        #endif

        int channelId = message[2];

        if (channelId >= 0 && channelId < CONNECTIONS_LIMIT &&
                channels_[channelId] != NULL)
        {
          handleFinishFromProxy(channelId);

          //
          // Check if it's time to save content
          // of message stores. Don't abort the
          // connection if can't write to disk.
          //

          handleCheckSave();
        }
        #ifdef WARNING
        else
        {
          *logofs << "Proxy: WARNING! Received a drop message for "
                  << "invalid channel id ID#" << channelId
                  << " with FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
        }
        #endif
      }
      else if (message[1] == CODE_FINISH_CONNECTION)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_FINISH_CONNECTION message.\n"
                << logofs_flush;
        #endif

        //
        // Prepare the channel for closing connection and
        // send channel shutdown message to the peer proxy.
        //

        int channelId = message[2];

        int channelValid = 0;

        if (channelId >= 0 && channelId < CONNECTIONS_LIMIT &&
                channels_[channelId] != NULL)
        {
          //
          // Force finish state on channel.
          //

          channels_[channelId] -> handleFinish();

          handleFinishFromProxy(channelId);

          channelValid = 1;
        }
        #ifdef WARNING
        else
        {
          *logofs << "Proxy: WARNING! Received a finish message for "
                  << "invalid channel id ID#" << channelId
                  << " with FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
        }
        #endif

        //
        // Send anyway channel shutdown
        // message to the peer proxy.
        //

        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Sending CODE_DROP_CONNECTION message.\n"
                << logofs_flush;
        #endif

        if (handleControl(CODE_DROP_CONNECTION, channelId) < 0)
        {
          return -1;
        }

        //
        // If we actually dropped the channel
        // check if it's time to save content
        // of message stores to disk.
        //

        if (channelValid == 1 && handleCheckSave() < 0)
        {
          return -1;
        }
      }
      else if (message[1] == CODE_RESET_REQUEST)
      {
        //
        // Remote proxy completed its reset.
        //

        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_RESET_REQUEST message.\n"
                << logofs_flush;
        #endif

        reset_ = 0;
      }
      else if (message[1] == CODE_SHUTDOWN_REQUEST)
      {
        //
        // Time to rest in peace.
        //

        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_SHUTDOWN_REQUEST message.\n"
                << logofs_flush;
        #endif

        shutdown_ = 1;
      }
      else if (message[1] == CODE_LOAD_REQUEST)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_LOAD_REQUEST message.\n"
                << logofs_flush;
        #endif

        if (handleLoadFromProxy() < 0)
        {
          return -1;
        }
      }
      else if (message[1] == CODE_SAVE_REQUEST)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_SAVE_REQUEST message.\n"
                << logofs_flush;
        #endif

        //
        // Don't abort the connection
        // if can't write to disk.
        //

        handleSaveFromProxy();
      }
      else if (message[1] == CODE_STATISTICS_REQUEST)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_STATISTICS_REQUEST message.\n"
                << logofs_flush;
        #endif

        int type = message[2];

        if (handleStatisticsFromProxy(type) < 0)
        {
          return -1;
        }
      }
      else if (message[1] == CODE_STATISTICS_REPLY)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_STATISTICS_REPLY message.\n"
                << logofs_flush;
        #endif

        operation_ = OPERATION_IN_STATISTICS;
      }
      else if (message[1] == CODE_ALERT_REQUEST)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Received CODE_ALERT_REQUEST message.\n"
                << logofs_flush;
        #endif

        HandleAlert(message[2]);
      }
      else
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Received bad control message number "
                << (unsigned int) message[1] << " with attribute "
                << (unsigned int) message[2] << ".\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Received bad control message number "
             << (unsigned int) message[1] << " with attribute "
             << (unsigned int) message[2] << ".\n";

        HandleCleanup();
      }
    }
    else if (operation_ == OPERATION_IN_MESSAGES)
    {
      int channelId = inputChannel_;

      #if defined(TEST) || defined(OPCODES)
      *logofs << "Proxy: Identified message of " << dataLength
              << " bytes for FD#" << getFd(channelId) << " channel ID#"
               << channelId << ".\n" << logofs_flush;
      #endif

      if (channelId >= 0 && channelId < CONNECTIONS_LIMIT &&
              channels_[channelId] != NULL)
      {
        int fd = getFd(channelId);

        int finish = channels_[channelId] -> getFinish();

        #ifdef WARNING
        if (finish == 1)
        {
          *logofs << "Proxy: WARNING! Handling data for finishing FD#"
                  << fd << " channel ID#" << channelId << ".\n"
                  << logofs_flush;
        }
        #endif

        int result = channels_[channelId] -> handleWrite(message, dataLength);

        //
        // Check if we have abort split events
        // to send to the remote peer.
        //

        if (result >= 0)
        {
          result = handleAbortSplit(fd);

          if (result < 0)
          {
            return -1;
          }
        }

        //
        // Check if this is the first time that
        // failure on descriptor was detected.
        //

        if (result < 0 && finish == 0)
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Failed to write proxy data to FD#"
                  << fd << " channel ID#" << channelId << ".\n"
                  << logofs_flush;
          #endif

          if (handleFinish(channelId) < 0)
          {
            return -1;
          }
        }

        //
        // This could have triggered the need for
        // a congestion event...
        //

        stateCongestion_ = fd;

        //
        // ...Infact, if we had to force write to
        // channel's socket, quit handling of further
        // messages and send the congestion event as
        // soon as possible...
        //

        if (transports_[channelId] -> forced() == 1)
        {
          #ifdef TEST
          *logofs << "Proxy: Giving up proxy because of forced "
                  << "transport for FD#" << fd << ".\n"
                  << logofs_flush;
          #endif

          yield = 1;
        }

        //
        // ...Oherwise check if channel has reached
        // its buffer limit.
        //

        else if (channels_[channelId] -> needLimit() > 0)
        {
          #ifdef TEST
          *logofs << "Proxy: Giving up proxy because of buffer "
                  << "for FD#" << fd << " beyond the limit.\n"
                  << logofs_flush;
          #endif

          stateLimit_ = fd;

          yield = 1;
        }

      }
      #ifdef WARNING
      else
      {
        *logofs << "Proxy: WARNING! Ignoring data received for "
                << "invalid channel ID#" << channelId << ".\n"
                << logofs_flush;
      }
      #endif
    }
    else if (operation_ == OPERATION_IN_STATISTICS)
    {
      #ifdef TEST
      *logofs << "Proxy: Received statistics data from remote proxy.\n"
              << logofs_flush;
      #endif

      if (handleStatisticsFromProxy(message, dataLength) < 0)
      {
        return -1;
      }

      operation_ = OPERATION_IN_MESSAGES;
    }

    //
    // if ((dataLength == 3) ...
    // else if (operation_ == OPERATION_IN_STATISTICS) ...
    // else if (operation_ == OPERATION_IN_MESSAGES) ...
    // else ...
    //

    else
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Unrecognized message received on proxy FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Unrecognized message received on proxy FD#"
           << fd_ << ".\n";

      return -1;
    }

    //
    // If handling of proxy messages took more time
    // than advisable then leave remaining data in
    // the read buffer.
    //

    if (yield == 0 && control -> isTimeToYield(yield_in_proxy))
    {
      #ifdef TEST
      *logofs << "Proxy: Giving up proxy because of "
              << control -> getTimeInARow() << " Ms "
              << "spent handling messages.\n"
              << logofs_flush;
      #endif

      yield = 1;
    }

  } // while (yield == 0 && (message = readBuffer_.getMessage(dataLength)) != NULL) ...

  //
  // Check if we left data in the read buffer.
  //

  if (yield == 1)
  {
    pending_ = readBuffer_.checkMessage();
  }
  else
  {
    pending_ = 0;
  }

  //
  // Reset the read buffer.
  //

  readBuffer_.partialReset();

  //
  // Write any produced data to the proxy link.
  //

  return handleWrite();
}

int Proxy::handleRead(int fd)
{
  if (stateReadChannels_ == 0 && statePending_ == -1)
  {
    return 0;
  }

  #ifdef TEST
  *logofs << "Proxy: Handling data from an X connection.\n"
          << logofs_flush;
  #endif

  int channelId = getChannel(fd);

  if (channelId < 0 || channelId >= CONNECTIONS_LIMIT ||
          channels_[channelId] == NULL)
  {
    #ifdef TEST
    *logofs << "Proxy: No channel for descriptor FD#" << fd
            << " channel ID#" << channelId << ".\n"
            << logofs_flush;
    #endif

    return 0;
  }
  else if (channels_[channelId] -> getFinish())
  {
    #ifdef TEST
    *logofs << "Proxy: Skipping read on finishing "
            << "descriptor FD#" << fd << " channel ID#"
            << channelId << ".\n" << logofs_flush;
    #endif

    return 0;
  }
  else if (congestions_[channelId] == 1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping read on congested "
            << "descriptor FD#" << fd << " channel ID#"
            << channelId << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  //
  // Check if reading from any other channel we
  // have already reached the amount of data set
  // for this scheduled write.
  //

  if (control -> isTimeToYield(yield_in_channel))
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping read from FD#" << fd << " with "
            << control -> getTimeInARow() << " Ms elapsed and "
            << control -> getBytesInARow() << " bytes to flush.\n"
            << logofs_flush;
    #endif

    return 0;
  }

  //
  // Let's see if we can safely write more data
  // to the proxy link.
  //

  stateBlocked_ = transport_ -> blocked();

  int fdRequisite = 0;

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    fdRequisite = (reset_ == 0 && congestion_ == 0 &&
                       pending_ == 0 && stateBlocked_ == 0);

  }
  else
  {
    fdRequisite = (reset_ == 0 && congestion_ == 0 &&
                       stateBlocked_ == 0);
  }

  if (fdRequisite == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping read from FD#" << fd << " with reset "
            << reset_ << " congestion " << congestion_ << " pending "
            << pending_ << " blocked " << stateBlocked_ << ".\n"
            << logofs_flush;
    #endif

    return 0;
  }

  stateLength_    = transport_ -> length();
  stateQueued_    = transport_ -> queued();
  stateFlushable_ = transport_ -> flushable();

  stateLength_ -= stateFlushable_;

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    fdRequisite = (control -> FlushTimeout == 0 ||
                       (stateLength_ + stateQueued_) <
                            control -> LocalBytesInARowLimit);
  }
  else
  {
    fdRequisite = (control -> FlushTimeout == 0 || stateLength_ <
                       control -> LocalBytesInARowLimit);
  }

  if (fdRequisite == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping read from FD#" << fd_
            << " with length " << stateLength_ << " and queued "
            << stateQueued_ << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  //
  // Let the channel object read all the new data from
  // its file descriptor, isolate messages, compress
  // those messages, and append the compressed form to
  // the encode buffer.
  //

  #if defined(TEST) || defined(OPCODES)
  *logofs << "Proxy: Reading messages from FD#" << fd
          << " channel ID#" << channelId << ".\n"
          << logofs_flush;
  #endif

  int result = channels_[channelId] -> handleRead(encodeBuffer_);

  //
  // Even in case of failure write produced data to the
  // proxy connection. To keep stores synchronized remote
  // side needs to decode any encoded message, also if X
  // socket was closed in the meanwhile. If this is the
  // case, the decompressed output will not be passed to
  // transport and will be silently discarded.
  //

  if (result < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Failed to read data from X connection FD#"
            << fd << " channel ID#" << channelId << ".\n"
            << logofs_flush;
    #endif

    encodeBuffer_.fullReset();

    if (handleFinish(channelId) < 0)
    {
      return -1;
    }
  }
  else if (encodeBuffer_.getLength() + controlLength_ > 0)
  {
    if (handleSwitch(channelId) < 0)
    {
      return -1;
    }
    else if (handleWrite() < 0)
    {
      return -1;
    }
  }

  //
  // Reading from channel we could have ceased the
  // congestion state by having received a reply
  // to a sync message.
  //

  channels_[channelId] -> updateCongestion();

  //
  // We could have also triggered the need for one
  // of the following events.
  //

  stateMotion_     = needMotion();
  stateWakeup_     = needWakeup();
  stateSplit_      = needSplit();
  stateCongestion_ = needCongestion();

  return 1;
}

int Proxy::handleWakeup()
{
  if (stateWakeup_ == -1)
  {
    #ifdef TEST
    if (stateRetry_ == -1 && needWakeup() != -1)
    {
      *logofs << "Proxy: WARNING! Channel for FD#" << needWakeup()
              << " has clients to wakeup but state is " << stateWakeup_
              << ".\n" << logofs_flush;
    }
    #endif

    return 0;
  }

  stateBlocked_ = transport_ -> blocked();

  int wakeupRequisite = (reset_ == 0 && congestion_ == 0 &&
                             pending_ == 0 && stateBlocked_ == 0);

  if (wakeupRequisite == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping wakeup on proxy FD#"
            << fd_ << " with reset " << reset_ << " congestion "
            << congestion_ << " pending " << pending_ << " blocked "
            << stateBlocked_ << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  stateLength_    = transport_ -> length();
  stateQueued_    = transport_ -> queued();
  stateFlushable_ = transport_ -> flushable();

  stateLength_ -= stateFlushable_;

  wakeupRequisite = (control -> FlushTimeout == 0 ||
                         (stateLength_ + stateQueued_) <
                              control -> LocalBytesInARowLimit);

  if (wakeupRequisite == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping wakeup on proxy FD#"
            << fd_ << " with length " << stateLength_
            << " and queued " << stateQueued_ << ".\n"
            << logofs_flush;
    #endif

    return 0;
  }

  #ifdef TEST
  *logofs << "Proxy: Handling wakeup of channels "
          << "for proxy FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> needWakeup() == 1)
    {
      if (channels_[channelId] -> getFinish())
      {
        #ifdef TEST
        *logofs << "Proxy: Skipping wakeup on finishing "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
        #endif

        continue;
      }
      else if (congestions_[channelId] == 1)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Skipping wakeup on congested "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
        #endif

        continue;
      }

      #ifdef TEST
      *logofs << "Proxy: Going to send wakeup events for FD#"
              << getFd(channelId) << ".\n" << logofs_flush;
      #endif

      int result = channels_[channelId] -> handleWakeup(encodeBuffer_);

      if (result < 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Failed to handle wakeup events for FD#"
                << getFd(channelId) << " channel ID#" << channelId
                << ".\n" << logofs_flush;
        #endif

        encodeBuffer_.fullReset();

        if (handleFinish(channelId) < 0)
        {
          return -1;
        }
      }
      else if (encodeBuffer_.getLength() + controlLength_ > 0)
      {
        if (handleSwitch(channelId) < 0)
        {
          return -1;
        }
        else if (handleWrite() < 0)
        {
          return -1;
        }
      }
    }
  }

  return 1;
}

int Proxy::handleSplit()
{
  if (stateSplit_ == -1)
  {
    return 0;
  }

  if (control -> isTimeToSplit() == 0)
  {
    #ifdef TEST
    *logofs << "Proxy: Still waiting "
            << control -> getTimeToNextSplit()
            << " Ms before sending the next split.\n"
            << logofs_flush;
    #endif

    return 0;
  }

  //
  // Sanity check on enqued data on the proxy link.
  // If it exceeds the thresholds then don't send
  // new splits.
  //

  stateBlocked_ = transport_ -> blocked();

  int splitRequisite = (reset_ == 0 && congestion_ == 0 &&
                            stateBlocked_ == 0);

  if (splitRequisite == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping split on proxy FD#"
            << fd_ << " with reset " << reset_ << " congestion "
            << congestion_ << " blocked " << stateBlocked_
            << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  stateLength_    = transport_ -> length();
  stateQueued_    = transport_ -> queued();
  stateFlushable_ = transport_ -> flushable();

  stateLength_ -= stateFlushable_;

  splitRequisite = (control -> FlushTimeout == 0 ||
                        (stateLength_ + stateQueued_) <
                             control -> LocalBytesInARowLimit);

  if (splitRequisite == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping split on proxy FD#" << fd_
            << " with length " << stateLength_ << " and queued "
            << stateQueued_ << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> needSplit() == 1)
    {
      if (channels_[channelId] -> getFinish())
      {
        #ifdef TEST
        *logofs << "Proxy: Skipping split on finishing "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
        #endif

        continue;
      }
      else if (congestions_[channelId] == 1)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Skipping split on congested "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
        #endif

        continue;
      }

      #ifdef TEST
      *logofs << "Proxy: Going to encode splits for FD#" << getFd(channelId)
              << " with " << clientStore_ -> getSplitStore() -> getSize()
              << " elements and " << control -> getBytesToSplitInARow()
              << " bytes to write.\n" << logofs_flush;
      #endif

      //
      // Let the channel encode a new handful of splits.
      //

      int packetLimit = control -> getBytesToSplitInARow();

      int result = channels_[channelId] -> handleSplit(encodeBuffer_, packetLimit);

      //
      // If any channel fails, try with the next.
      //

      if (result < 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Failed to handle splits for FD#"
                << getFd(channelId) << " channel ID#" << channelId
                << ".\n" << logofs_flush;
        #endif

        encodeBuffer_.fullReset();

        if (handleFinish(channelId) < 0)
        {
          return -1;
        }
      }
      else
      {
        //
        // Set time to next split.
        //

        control -> setLastSplitInfo();

        if (encodeBuffer_.getLength() + controlLength_ > 0)
        {
          if (handleSwitch(channelId) < 0)
          {
            return -1;
          }
          else if (handleWrite() < 0)
          {
            return -1;
          }
        }

        //
        // All channels currently share the same split
        // store so we send splits only once.
        //

        return 1;
      }
    }
  }

  return 1;
}

int Proxy::handleAbortSplit(int fd)
{
  if (clientStore_ -> getSplitStore() -> getAbort() == 0)
  {
    return 0;
  }

  int channelId = getChannel(fd);

  int result = channels_[channelId] -> handleAbortSplit(encodeBuffer_);

  if (result < 0)
  {
    //
    // Finish on channel must be called in
    // the parent procedure.
    //

    #if defined(INFO) || defined(TEST)
    *logofs << "ServerProxy: Failed to handle abort splits "
            << "for FD#" << getFd(channelId) << " channel ID#"
            << channelId << ".\n" << logofs_flush;
    #endif

    encodeBuffer_.fullReset();

    return -1;
  }

  if (encodeBuffer_.getLength() + controlLength_ > 0)
  {
    if (handleSwitch(channelId) < 0)
    {
      return -1;
    }
    else if (handleWrite() < 0)
    {
      return -1;
    }
  }

  return 1;
}

int Proxy::handleMotion()
{
  if (stateMotion_ == -1)
  {
    return 0;
  }

  //
  // Check if we can actually write
  // to the proxy link.
  //

  stateBlocked_ = transport_ -> blocked();

  int motionRequisite = (reset_ == 0 && congestion_ == 0 &&
                             stateBlocked_ == 0);

  if (motionRequisite == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Skipping motion on proxy FD#"
            << fd_ << " with reset " << reset_ << " congestion "
            << congestion_ << " blocked " << stateBlocked_
            << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  #ifdef TEST
  *logofs << "Proxy: Handling motion on channels "
          << "for proxy FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> needMotion() == 1)
    {
      if (channels_[channelId] -> getFinish())
      {
        #ifdef TEST
        *logofs << "Proxy: Skipping motion on finishing "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
        #endif

        continue;
      }
      else if (congestions_[channelId] == 1)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Skipping motion on congested "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
        #endif

        continue;
      }

      #ifdef TEST
      *logofs << "Proxy: Going to send motion events for FD#"
              << getFd(channelId) << ".\n" << logofs_flush;
      #endif

      int result = channels_[channelId] -> handleMotion(encodeBuffer_);

      if (result < 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "Proxy: Failed to handle motion events for FD#"
                << getFd(channelId) << " channel ID#" << channelId
                << ".\n" << logofs_flush;
        #endif

        encodeBuffer_.fullReset();

        if (handleFinish(channelId) < 0)
        {
          return -1;
        }
      }
      else if (encodeBuffer_.getLength() + controlLength_ > 0)
      {
        if (handleSwitch(channelId) < 0)
        {
          return -1;
        }
        else if (handleWrite() < 0)
        {
          return -1;
        }
      }
    }
  }

  return 1;
}

//
// There are two types of congestion events:
//
// 1. A X congestion event is sent to X channels if client
// proxy can't write its data to the remote peer. X channels
// should stop accepting more messages from their clients
// until a new event, ceasing the previous condition, is
// received.
//
// 2. Proxy sends to the remote peer a begin congestion
// control code in case the local end of the channel is not
// consuming the data accumulated in its buffer. This allows
// the remote proxy to stop accepting more data for the
// channel. Proxy usually waits for a timeout before trig-
// gering the event. Channel can thus be in one of the fol-
// lowing states:
//
// - In congestion.
//
//   Channel has the congestion_ flag set and no
//   congestion timeout.
//
// - Out of congestion state.
//
//   The congestion flag is not set and there is
//   no congestion timeout.
//
// - Waiting for a transition in congestion state.
//
//   In this case the timestamp of the last condition
//   triggering a change in congestion state is saved
//   in channel. The code handling congestion transi-
//   tion must be invoked by proxy when the timeout
//   is elapsed.
//   
// The stateCongestion_ flag is set by either iterating
// through channels in the needCongestion() function or
// by forcing the value, if a transition is expected due
// to a previous action.
//
// TODE: The needCongestion() function only returns a
// file descriptor if the channel has set a congestion
// timeout. To force a new check, proxy must explicitly
// set this timeout to avoid that a further call to
// needCongestion() would reset the state flag. This is
// clumsy and could potentially delay the transmission
// of congestion events. Code must be rewritten to ensure
// that proxy can force a new check without setting the
// timeout.
//

int Proxy::handleCongestion()
{
  if (control -> ProxyMode == PROXY_CLIENT &&
          stateNotify_ != stateRetry_)
  {
    #ifdef TEST
    *logofs << "Proxy: Handling congestion notify "
            << "on channels for proxy FD#" << fd_
            << ".\n" << logofs_flush;
    #endif

    if (stateRetry_ != -1)
    {
      handleNotify(notify_begin_congestion);
    }
    else
    {
      handleNotify(notify_end_congestion);
    }

    //
    // Save the retry state for the next iteration.
    //

    stateNotify_ = stateRetry_;
  }

  if (stateCongestion_ == -1)
  {
    return 0;
  }

  //
  // Handle congestion state in channels even if we
  // cannot presently write to the proxy link. First
  // of all having the message in the transport buf-
  // fer will save a loop, secondly the congestion
  // timeout can be null and, in this case, we would
  // end up taking all the CPU.
  //

  #ifdef TEST
  *logofs << "Proxy: Handling congestion on channels "
          << "for proxy FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      if (channels_[channelId] -> getFinish())
      {
        #ifdef TEST
        *logofs << "Proxy: Skipping congestion on finishing "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
        #endif

        continue;
      }

      //
      // Channel will report if a transition in congestion
      // state has occurred. Proxy shall get the new value
      // and send an appropriate control message.
      //

      if (channels_[channelId] -> handleCongestion())
      {
        int state = channels_[channelId] -> getCongestion();

        T_proxy_code controlCode;

        if (state == 1)
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Sending CODE_BEGIN_CONGESTION message "
                  << "for FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
          #endif

          controlCode = CODE_BEGIN_CONGESTION;
        }
        else
        {
          #if defined(INFO) || defined(TEST)
          *logofs << "Proxy: Sending CODE_END_CONGESTION message "
                  << "for FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
          #endif

          controlCode = CODE_END_CONGESTION;
        }

        priority_ = 1;

        if (handleControl(controlCode, channelId) < 0)
        {
          return -1;
        }
        else if (handleWrite() < 0)
        {
          return -1;
        }
      }
    }
  }

  return 1;
}

int Proxy::handleWrite()
{
  //
  // Write any outstanding control message, followed by any
  // outstanding compressed X message, to proxy transport.
  // This code assumes that encode buffer data is at a loca-
  // tion offset several bytes from start of the buffer, so
  // that the length header and any necessary control bytes
  // can be inserted in front of data already in the buffer.
  // This is the easiest way to encapsulate header and data
  // together in a single frame.
  //

  unsigned int dataLength = encodeBuffer_.getLength();

  if (dataLength + controlLength_ == 0)
  {
    return 0;
  }
  else if (controlLength_ > CONTROL_CODES_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Overflow in control messages length. "
            << "Size is " << controlLength_ << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Overflow in control messages length. "
         << "Size is " << controlLength_ << ".\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Proxy: Going to write a new message for remote proxy.\n"
          << logofs_flush;
  #endif

  unsigned char temp[5];

  unsigned int lengthLength = 0;
  unsigned int length = dataLength;

  while (length)
  {
    temp[lengthLength++] = (unsigned char) (length & 0x7f);

    length >>= 7;
  }

  unsigned char *data = encodeBuffer_.getData();

  unsigned char *outputMessage = data - (controlLength_ + lengthLength);

  int outputLength = dataLength + controlLength_ + lengthLength;

  unsigned char *nextDest = outputMessage;

  for (int i = 0; i < controlLength_; i++)
  {
    *nextDest++ = controlCodes_[i];
  }

  for (int j = lengthLength - 1; j > 0; j--)
  {
    *nextDest++ = (temp[j] | 0x80);
  }

  if (lengthLength)
  {
    *nextDest++ = temp[0];
  }

  #if defined(TEST) || defined(OPCODES)
  *logofs << "Proxy: Produced plain output for " << dataLength << "+"
          << controlLength_ << "+" << lengthLength << " out of "
          << outputLength << " bytes.\n" << logofs_flush;
  #endif

  //
  // Write immediately if we collected enough data.
  //

  stateLength_ = transport_ -> length();

  T_write type = write_delayed;

  if (control -> FlushTimeout == 0 ||
          (stateLength_ + outputLength) >=
               control -> LocalBytesInARowLimit ||
                   control -> isTimeToYield(yield_in_channel))
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Going to flush proxy FD#" << fd_
            << " with length " << stateLength_ + outputLength
            << ".\n" << logofs_flush;
    #endif

    type = write_immediate;
  }

  int result = transport_ -> write(type, outputMessage, outputLength);

  #ifdef DUMP
  *logofs << "Proxy: Sent " << outputLength << " bytes of data "
          << "with checksum ";

  DumpChecksum(outputMessage, outputLength);

  *logofs << " on proxy FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  #ifdef DUMP
  *logofs << "Proxy: Partial checksums are:\n";

  DumpBlockChecksums(outputMessage, outputLength, 256);

  *logofs << logofs_flush;
  #endif

  //
  // Clean up the encode buffer and
  // revert to the initial size.
  //

  encodeBuffer_.fullReset();

  if (result < 0)
  {
    #ifdef TEST
    *logofs << "Proxy: Failed write on proxy FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    return -1;
  }

  if (control -> CollectStatistics)
  {
    statistics -> addFrameOut();

    statistics -> addFramingBits((controlLength_ + lengthLength) << 3);
  }

  control -> addBitsInARow(lengthLength << 3);

  controlLength_ = 0;

  //
  // Update timer of next ping.
  //

  lastBytesOutTs_ = getTimestamp();

  //
  // If we actually flushed the proxy link
  // reset buffers, counters and priority
  // flag on proxy and channels.
  //

  //
  // Maybe need to flush the proxy link.
  //

  stateFlush_ = needFlush();

  if (stateFlush_ == -1)
  {
    handleResetFlush();
  }

  return result;
}

//
// Perform flush on descriptors selected for write.
//

int Proxy::handleFlush(fd_set &fdSet)
{
  #ifdef DEBUG
  *logofs << "Proxy: Looping through descriptors selected for write.\n"
          << logofs_flush;
  #endif

  for (int j = (firstChannel_ + 1 > upperChannel_ ?
                    lowerChannel_ : firstChannel_ + 1);
                        ; (j + 1 > upperChannel_ ?
                               j = lowerChannel_ : j++))
  {
    #ifdef DEBUG
    *logofs << "Proxy: Looping with first " << firstChannel_
            << " and current " << j << ".\n" << logofs_flush;
    #endif

    int fd = getFd(j);

    if (fd >= 0 && FD_ISSET(fd, &fdSet))
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Proxy: X descriptor FD#" << fd
              << " reported to be writable.\n"
              << logofs_flush;
      #endif

      //
      // It can happen that, in handling reads, we have
      // destroyed buffer associated to a closed socket
      // so we don't complain for errors.
      //

      #ifdef DEBUG
      *logofs << "Proxy: Going to flush data to FD#"
              << fd << ".\n" << logofs_flush;
      #endif

      handleFlush(flush_if_any, fd);
    }

    if (j == firstChannel_)
    {
      firstChannel_ = (firstChannel_ + 1 > upperChannel_ ?
                           lowerChannel_ : firstChannel_ + 1);
      break;
    }
  }

  return 1;
}

//
// Check if it's time to write data to the proxy link.
//

int Proxy::handleFlush(T_flush type)
{
  //
  // Force update of current bitrate.
  //

  control -> setLastWriteInfo(0);

  if (stateUnblockProxy_ == 0 && stateFlush_ == -1)
  {
    return 0;
  }

  //
  // Cache the values in state variables.
  //

  stateLength_  = transport_ -> length();
  stateBlocked_ = transport_ -> blocked();

  //
  // Never try to write when link is blocked but
  // do it when forced, as happens when socket
  // is reported to be writable.
  //

  if (stateLength_ == 0 ||
          (stateBlocked_ == 1 &&
               type != flush_if_any))
  {
    return 0;
  }

  stateQueued_    = transport_ -> queued();
  stateFlushable_ = transport_ -> flushable();

  stateLength_ -= stateFlushable_;

  //
  // Verify priority in proxy and channels. Note that
  // function needs to check the state variables we
  // just updated. We assume that we have already read
  // data from channels and thus their priority state
  // can't change across multiple calls. Priority on
  // proxy, instead, is further verified later.
  //

  int flushRequisite = (control -> isTimeToYield(yield_in_channel) == 1);

  if (flushRequisite == 0)
  {
    statePriority_ = needPriority();

    flushRequisite = (statePriority_ != -1);
  }

  if (flushRequisite == 0 &&
          type == flush_if_priority)
  {
    return 0;
  }

  //
  // In general is advisable to delay the flush as we
  // reduce packet fragmentation and obtain a better
  // stream compression, but there are conditions to
  // satisfy. I tried to optimize this but it become
  // hard to read and maintain. I would really need
  // gotos here, but I don't want to be injured as a
  // bad programmer.
  //

  if (flushRequisite == 0)
  {
    flushRequisite = (priority_ > 0 || type == flush_if_any ||
                          control -> AgentFlushImmediate == 1 ||
                              control -> FlushTimeout == 0);
  }

  //
  // We have to flush only if needed and scheduled
  // writes are enabled. Check if we accumulated
  // enough data to make a full scheduled write.
  //

  if (flushRequisite == 0)
  {
    if ((stateLength_ + stateFlushable_) >=
             control -> LocalBytesInARowLimit)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Proxy: Forcing flush with length "
              << stateLength_ << " flushable " << stateFlushable_
              << " and queued " << stateQueued_ << ".\n"
              << logofs_flush;
      #endif  

      flushRequisite = 1;
    }
  }

  //
  // Check flush timeout.
  //

  if (flushRequisite == 0)
  {
    if (control -> isTimeToFlush() == 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Proxy: Delaying flush on proxy FD#" << fd_
              << " with flushable " << stateFlushable_ << " length "
              << stateLength_ << " queued " << stateQueued_ << " and "
              << control -> getTimeToNextFlush() << " Ms to run.\n"
              << logofs_flush;
      #endif

      return 0;
    }
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Going to flush proxy FD#" << fd_
          << " with blocked " << stateBlocked_ << " flushable "
          << stateFlushable_ << " length " << stateLength_
          << " and queued " << stateQueued_ << ".\n"
          << logofs_flush;
  #endif

  int result = transport_ -> flush();

  if (result < 0)
  {
    return -1;
  }

  //
  // Get ready for the next flush.
  //

  handleResetFlush();

  //
  // If proxy became available after it was blocked,
  // we could have triggered the need for some out-
  // standing events. By enabling them here we avoid
  // to go through another timeout.
  //

  if (stateUnblockProxy_ == 1)
  {
    statePending_ = needPending();
    stateMotion_  = needMotion();
    stateWakeup_  = needWakeup();
    stateSplit_   = needSplit();
    stateFlush_   = needFlush();
  }

  return result;
}

//
// Type is ignored for channels' flush.
//

int Proxy::handleFlush(T_flush type, int fd)
{
  if (stateUnblockChannels_ == 0)
  {
    return 0;
  }

  int channelId = getChannel(fd);

  if (channelId < 0 || channels_[channelId] == NULL)
  {
    #ifdef TEST
    *logofs << "Proxy: WARNING! Skipping flush on invalid "
            << "descriptor FD#" << fd << " channel ID#"
            << channelId << ".\n" << logofs_flush;
    #endif

    return 0;
  }
  else if (channels_[channelId] -> getFinish() == 1)
  {
    #ifdef TEST
    *logofs << "Proxy: Skipping flush on finishing "
            << "descriptor FD#" << fd << " channel ID#"
            << channelId << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Going to flush FD#" << fd
          << " with blocked " << transports_[channelId] -> blocked()
          << " length " << transports_[channelId] -> length()
          << " and queued " << transports_[channelId] -> queued()
          << ".\n" << logofs_flush;
  #endif

  int result = transports_[channelId] -> flush();

  if (result < 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Proxy: Failed to flush data to FD#"
            << getFd(channelId) << " channel ID#" << channelId
            << ".\n" << logofs_flush;
    #endif

    handleFinish(channelId);

    return -1;
  }

  //
  // Let channel check its own congestion state.
  //

  channels_[channelId] -> updateCongestion();

  stateCongestion_ = fd;

  //
  // Update state of buffer limits if this can
  // save a loop.
  //

  if ((pending_ == 1 || stateReadProxy_ == 1) &&
           stateLimit_ != -1)
  {
    #ifdef TEST
    *logofs << "Proxy: WARNING! Checking if we can newly "
            << "attend proxy FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    stateLimit_ = needLimit();
  }

  //
  // Reset channel's transport buffers.
  //

  transports_[channelId] -> partialReset();

  return result;
}

int Proxy::handleStatistics(int type, ostream *stream)
{
  if (control -> CollectStatistics == 0 ||
          statistics == NULL || stream == NULL)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Cannot produce statistics "
            << " for proxy FD#" << fd_ << ". Invalid settings "
            << "for statistics or stream.\n" << logofs_flush;
    #endif

    return 0;
  }
  else if (lastStatisticsStream_ != NULL)
  {
    //
    // Need to update the stream pointer as the
    // previous one could have been destroyed.
    //
 
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Replacing stream while producing "
            << "statistics in stream at " << lastStatisticsStream_
            << " for proxy FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif
  }

  lastStatisticsStream_ = stream;

  //
  // Get statistics of remote peer.
  //

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Sending CODE_STATISTICS_REQUEST message.\n"
          << logofs_flush;
  #endif

  if (handleControl(CODE_STATISTICS_REQUEST, type) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleStatisticsFromProxy(int type)
{
  if (control -> CollectStatistics != 0 &&
          statistics != NULL)
  {
    //
    // Allocate buffer for output.
    //

    char *buffer = new char[STATISTICS_LENGTH];
 
    *buffer = '\0';

    if (control -> ProxyMode == PROXY_CLIENT)
    {
      #ifdef TEST
      *logofs << "Proxy: Producing "
              << (type == TOTAL_STATS ? "total" : "partial")
              << " client statistics for proxy FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      statistics -> getClientProtocolStats(type, buffer);

      statistics -> getClientOverallStats(type, buffer);
    }
    else
    {
      #ifdef TEST
      *logofs << "Proxy: Producing "
              << (type == TOTAL_STATS ? "total" : "partial")
              << " server statistics for proxy FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      statistics -> getServerProtocolStats(type, buffer);
    }

    if (type == PARTIAL_STATS)
    {
      statistics -> resetPartialStats();
    }

    unsigned int length = strlen((char *) buffer) + 1;

    encodeBuffer_.encodeValue(type, 8);

    encodeBuffer_.encodeValue(length, 32);

    #ifdef TEST
    *logofs << "Proxy: Encoding " << length
            << " bytes of statistics data for proxy FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    encodeBuffer_.encodeMemory((unsigned char *) buffer, length);

    control -> addBitsInARow(length << 3);

    //
    // Account statistics data as framing bits.
    //

    if (control -> CollectStatistics)
    {
      statistics -> addFramingBits(length << 3);
    }

    delete [] buffer;
  }
  else
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Got statistics request "
            << "but local statistics are disabled.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Got statistics request "
         << "but local statistics are disabled.\n";

    type = NO_STATS;

    encodeBuffer_.encodeValue(type, 8);

    #ifdef TEST
    *logofs << "Proxy: Sending error code to remote proxy on FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif
  }

  //
  // This will write both statistics
  // data and control message.
  //

  if (handleControl(CODE_STATISTICS_REPLY, type) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleStatisticsFromProxy(const unsigned char *message, unsigned int length)
{
  if (lastStatisticsStream_ == NULL)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Unexpected statistics data received "
            << "from remote proxy on FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Unexpected statistics data received "
         << "from remote proxy.\n";

    return 0;
  }

  //
  // Allocate the decode buffer and tet at
  // least the 'type' field to see if there
  // was an error.
  //

  DecodeBuffer decodeBuffer(message, length);

  unsigned int type;

  decodeBuffer.decodeValue(type, 8);

  if (type == NO_STATS)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Couldn't get statistics from remote "
            << "proxy on FD#" << fd_ << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Couldn't get statistics from remote proxy.\n";
  }
  else if (type != TOTAL_STATS && type != PARTIAL_STATS)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Cannot produce statistics "
            << "with qualifier '" << type << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot produce statistics "
         << "with qualifier '" << type << "'.\n";

    return -1;
  }
  else
  {
    unsigned int size;

    decodeBuffer.decodeValue(size, 32);

    char *buffer = new char[STATISTICS_LENGTH];

    *buffer = '\0';

    if (control -> CollectStatistics != 0 &&
            statistics != NULL)
    {
      if (control -> ProxyMode == PROXY_CLIENT)
      {
        #ifdef TEST
        *logofs << "Proxy: Finalizing "
                << (type == TOTAL_STATS ? "total" : "partial")
                << " client statistics for proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        statistics -> getClientCacheStats(type, buffer);

        #ifdef TEST
        *logofs << "Proxy: Decoding " << size
                << " bytes of statistics data for proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        strncat(buffer, (char *) decodeBuffer.decodeMemory(size), size);

        statistics -> getClientProtocolStats(type, buffer);

        statistics -> getClientOverallStats(type, buffer);
      }
      else
      {
        #ifdef TEST
        *logofs << "Proxy: Finalizing "
                << (type == TOTAL_STATS ? "total" : "partial")
                << " server statistics for proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        statistics -> getServerCacheStats(type, buffer);

        statistics -> getServerProtocolStats(type, buffer);

        #ifdef TEST
        *logofs << "Proxy: Decoding " << size
                << " bytes of statistics data for proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        strncat(buffer, (char *) decodeBuffer.decodeMemory(size), size);
      }

      if (type == PARTIAL_STATS)
      {
        statistics -> resetPartialStats();
      }

      *lastStatisticsStream_ << buffer;

      //
      // Mark the end of text to help external parsing.
      //

      *lastStatisticsStream_ << '\4';

      *lastStatisticsStream_ << flush;
    }
    else
    {
      //
      // It can be that statistics were enabled at the time
      // we issued the request (otherwise we could not have
      // set the stream), but now they have been disabled
      // by user. We must decode statistics data if we want
      // to keep the connection.
      //

      #ifdef TEST
      *logofs << "Proxy: Discarding " << size
              << " bytes of statistics data for proxy FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      strncat(buffer, (char *) decodeBuffer.decodeMemory(size), size);
    }

    delete [] buffer;
  }

  lastStatisticsStream_ = NULL;

  return 1;
}

int Proxy::handleAlert(int alert)
{
  #ifdef WARNING
  *logofs << "Proxy: WARNING! Sending request for alert "
          << alert << " to remote proxy on FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Sending CODE_ALERT_REQUEST message.\n"
          << logofs_flush;
  #endif

  if (handleControl(CODE_ALERT_REQUEST, alert) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleFinish(int channelId)
{
  //
  // Do any finalizations needed on channel
  // and send channel shutdown message to
  // the peer proxy.
  //

  channels_[channelId] -> handleFinish();

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Sending CODE_FINISH_CONNECTION message.\n"
          << logofs_flush;
  #endif

  if (handleControl(CODE_FINISH_CONNECTION, channelId) < 0)
  {
    return -1;
  }

  return 1;
}

//
// Close the channel and deallocate resources.
//

int Proxy::handleFinishFromProxy(int channelId)
{
  if (channelId < 0 || channelId >= CONNECTIONS_LIMIT ||
          channels_[channelId] == NULL)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Trying to destroy an invalid "
            << "channel id ID#" << channelId << " with FD#"
            << getFd(channelId) << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Trying to destroy an invalid "
            << "channel id ID#" << channelId << ".\n";

    return -1;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Handling finish from proxy for FD#"
          << getFd(channelId) << " channel ID#" << channelId
          << ".\n" << logofs_flush;
  #endif

  //
  // Get rid of channel.
  //

  close(getFd(channelId));

  char *channelType;

  switch (channels_[channelId] -> getType())
  {
    case CHANNEL_SYNC:
    {
      channelType = "sync";

      break;
    }
    case CHANNEL_SAMBA:
    {
      channelType = "samba";

      break;
    }
    case CHANNEL_MEDIA:
    {
      channelType = "media";

      break;
    }
    default:
    {
      channelType = NULL;

      break;
    }
  }

  if (channelType != NULL)
  {
    #if defined(INFO) || defined (TEST)
    *logofs << "Proxy: Closed connection to "
            << channelType << " server.\n"
            << logofs_flush;
    #endif

    cerr << "Info" << ": Closed connection to "
         << channelType << " server.\n";
  }

  delete channels_[channelId];
  channels_[channelId] = NULL;

  cleanupChannelMap(channelId);

  //
  // Get rid of socket.
  //

  delete transports_[channelId];
  transports_[channelId] = NULL;

  congestions_[channelId] = 0;

  decreaseActiveChannels(channelId);

  return 1;
}

//
// Send an empty message to the remote peer
// just to verify if link is alive and help
// the remote proxy to detect congestion.
//

int Proxy::handlePing()
{
  //
  // Be sure we take into account any clock drift. This
  // can be caused by the user changing the system timer
  // or by small adjustments introduced by the operating
  // system making the clock go backward.
  //

  T_timestamp nowTs = getTimestamp();

  if (checkTimestamp(lastBytesInTs_, nowTs) == 0)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Detected drift in system timer. "
            << " Setting difference to 0 Ms.\n" << logofs_flush;
    #endif
  }

  //
  // Check timestamp of last read from remote proxy. It can
  // happen that we stayed in the main loop long enough to
  // have idle timeout expired, for example if proxy was
  // stopped and restarted or because of an extremely high
  // load of the system. If this is the case do not complain
  // if there is something to read from proxy socket.
  //

  int diffIn = diffTimestamp(lastBytesInTs_, nowTs);

  if (diffIn >= ((control -> PingTimeout * 2) -
          control -> LatencyTimeout) &&
              transport_ -> readable() == 0)
  {
    //
    // Switch to proxy congestion state. At X server
    // side we must return to read data from channels
    // after a while, because we need to read the key
    // sequence CTRL+ALT+SHIFT+ESC.
    //

    if (congestion_ == 0)
    {
      congestion_ = 1;
    }
    else if (control -> ProxyMode == PROXY_SERVER)
    {
      congestion_ = 0;
    }

    if (control -> ProxyTimeout > 0 &&
            diffIn >= (control -> ProxyTimeout -
                control -> LatencyTimeout))
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! No data received from "
              << "remote proxy on FD#" << fd_ << " within "
              << (diffIn + control -> LatencyTimeout) / 1000
              << " seconds.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": No data received from remote proxy within "
           << (diffIn + control -> LatencyTimeout) / 1000
           << " seconds.\n";

      HandleAbort();
    }
    else
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Proxy: WARNING! No data received from "
              << "remote proxy on FD#" << fd_ << " since "
              << diffIn << " Ms.\n" << logofs_flush;
      #endif

      if (control -> ProxyTimeout > 0 &&
              isTimestamp(lastAlertTs_) == 0 &&
                  diffIn >= (control -> ProxyTimeout -
                      control -> LatencyTimeout) / 2)
      {
        cerr << "Warning" << ": No data received from remote proxy within "
             << (diffIn + control -> LatencyTimeout) / 1000
             << " seconds.\n";

        if (control -> ProxyMode == PROXY_CLIENT)
        {
          HandleAlert(CLOSE_DEAD_PROXY_CONNECTION_CLIENT_ALERT);
        }
        else
        {
          HandleAlert(CLOSE_DEAD_PROXY_CONNECTION_SERVER_ALERT);
        }

        lastAlertTs_ = nowTs;
      }
    }
  }

  //
  // Send a new ping only we didn't receive anything
  // from remote for longer than ping timeout. Client
  // side send a ping, server side must respond with
  // a reply.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    int diffPing = diffTimestamp(lastPingTs_, nowTs);

    if (diffIn >= (control -> PingTimeout -
            control -> LatencyTimeout * 5) &&
                diffPing >= (control -> PingTimeout -
                    control -> LatencyTimeout * 5))
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "Proxy: Sending CODE_PING_REQUEST message.\n"
              << logofs_flush;
      #endif

      //
      // Next flush will carry all data enqueued on
      // proxy so far, even data not sent because
      // of congestion.
      //

      priority_ = 1;

      if (handleControl(CODE_PING_REQUEST) < 0)
      {
        return -1;
      }

      lastPingTs_ = nowTs;
    }
  }

  return 1;
}

int Proxy::handlePingFromProxy()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Sending CODE_PING_REPLY message.\n"
          << logofs_flush;
  #endif

  //
  // Next flush will carry all data enqueued on
  // proxy so far, even data not sent because
  // of congestion.
  //

  priority_ = 1;

  if (handleControl(CODE_PING_REPLY) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleReset()
{
  #ifdef TEST
  *logofs << "Proxy: Resetting proxy for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Need to reset:
  //
  // 1. Channels, including sending an X error
  //    to clients for any pending reply and
  //    restarting agent's clients stopped
  //    because of sync, karma or splits.
  //
  // 2. Internal state, resetting congestion
  //    state for all channels.
  //
  // 3. Message stores, including their own
  //    encode caches.
  //
  // 4. Channel caches, only in protocol 3
  //    or greater.
  //
  // 5. Split stores, including encode caches.
  //
  // 6. Channels' encode caches.
  //
  // 7. Proxy ZLIB streams in any of compressor,
  //    decompressor and transport.
  //
  // 8. Any unreliable data accumulated in
  //    proxy read buffer.
  //
  // 9. If client proxy, tell finally to remote
  //    proxy to load message stores from last
  //    negotiated persistent cache.
  //
  // Stores are recreated as soon as reset message
  // is received from remote proxy.
  //

  //
  // Reset channels.
  //

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      if (channels_[channelId] -> handleReset() < 0)
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Failed to reset channel for FD#"
                << getFd(channelId) << ".\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Failed to reset channel for FD#"
             << getFd(channelId) << ".\n";
      }
    }
  }

  //
  // Reset ZLIB structures in compressor
  // and transport and reset proxy read
  // buffer.
  //

  compressor_ -> fullReset();
  decompressor_ -> fullReset();

  transport_ -> fullReset();

  readBuffer_.fullReset();

  encodeBuffer_.fullReset();

  //
  // Reset internal state.
  //

  controlLength_ = 0;

  operation_ = OPERATION_IN_MESSAGES;

  pending_    = 0;
  priority_   = 0;
  shutdown_   = 0;
  reset_      = 1;
  congestion_ = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    congestions_[channelId] = 0;
  }

  lastBytesInTs_  = getTimestamp();
  lastBytesOutTs_ = getTimestamp();

  lastRetryTs_    = nullTimestamp();
  lastRetryIndex_ = 0;

  lastPingTs_  = getTimestamp();
  lastAlertTs_ = nullTimestamp();

  lastLimitTs_ = nullTimestamp();

  lastStatisticsStream_ = NULL;

  //
  // Reset message stores and tell remote
  // proxy to reload content from disk.
  //

  if (handleResetOpcodes() < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Failed to reset opcodes store.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  if (handleResetStores() < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Failed to reset message stores.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  if (handleResetCaches() < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Failed to reset channel caches.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  //
  // Force load of last negotiated persistent cache
  // if one or more channels already exist. It will
  // actually produce a load request if we are at
  // client side.
  //

  if (handleCheckLoad(LOAD_IF_ANY) < 0)
  {
    return -1;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Sending CODE_RESET_REQUEST message.\n"
          << logofs_flush;
  #endif

  priority_ = 1;

  if (handleControl(CODE_RESET_REQUEST) < 0)
  {
    return -1;
  }

  stateReadProxy_    = -1;
  stateReadChannels_ = -1;

  stateUnblockProxy_    = -1;
  stateUnblockChannels_ = -1;

  stateLimit_      = -1;
  stateRetry_      = -1;
  statePending_    = -1;
  statePriority_   = -1;
  stateMotion_     = -1;
  stateWakeup_     = -1;
  stateCongestion_ = -1;
  stateSplit_      = -1;
  stateFlush_      = -1;

  stateBlocked_   = -1;
  stateQueued_    = -1;
  stateLength_    = -1;
  stateFlushable_ = -1;

  //
  // Force an end congestion notification
  // when the proxy link becomes writable.
  //

  stateNotify_ = fd_;

  #ifdef TEST
  *logofs << "Proxy: Completed reset of proxy for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Write any data produced for the remote
  // proxy, including any load request and
  // the reset message.
  //

  if (handleWrite() < 0)
  {
    return -1;
  }

  return 1;
}


int Proxy::handleResetPersistentCache()
{
  char *fullName = new char[strlen(control -> PersistentCachePath) +
                                strlen(control -> PersistentCacheName) + 2];

  strcpy(fullName, control -> PersistentCachePath);
  strcat(fullName, "/");
  strcat(fullName, control -> PersistentCacheName);

  #ifdef TEST
  *logofs << "Proxy: Going to remove persistent cache file '"
          << fullName << "'\n" << logofs_flush;
  #endif

  remove(fullName);

  delete [] fullName;

  delete [] control -> PersistentCacheName;

  control -> PersistentCacheName = NULL;

  return 1;
}

int Proxy::handleResetOpcodes()
{
  //
  // Nothing to be done. Opcodes are
  // not changing given that we remain
  // connected to the same X server.
  //

  return 1;
}

int Proxy::handleResetStores()
{
  //
  // Recreate the message stores.
  //

  delete clientStore_;
  delete serverStore_;

  clientStore_ = new ClientStore(compressor_, decompressor_);
  serverStore_ = new ServerStore(compressor_, decompressor_);

  lastLoadTs_ = nullTimestamp();

  //
  // Replace message stores in channels.
  //

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      if (channels_[channelId] -> setStores(clientStore_, serverStore_) < 0)
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Failed to replace message stores in "
                << "channel for FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Failed to replace message stores in "
             << "channel for FD#" << getFd(channelId) << ".\n";

        return -1;
      }
      #ifdef TEST
      else
      {
        *logofs << "Proxy: Replaced message stores in channel "
                << "for FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
      }
      #endif
    }
  }

  return 1;
}

int Proxy::handleResetCaches()
{
  //
  // If protocol is 3 or newer then recreate the
  // channel caches. In older protocol versions
  // caches are not shared and they are reset in
  // the channel specific method.
  //

  if (control -> isProtoStep3() == 1)
  {
    delete clientCache_;
    delete serverCache_;

    clientCache_ = new ClientCache();
    serverCache_ = new ServerCache();

    if (clientCache_ == NULL || serverCache_ == NULL)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Failed to create channel's caches.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Failed to create channel's caches.\n";

      HandleCleanup();
    }

    //
    // Replace channel caches in channels.
    //

    for (int channelId = lowerChannel_;
             channelId <= upperChannel_;
                 channelId++)
    {
      if (channels_[channelId] != NULL)
      {
        if (channels_[channelId] -> setCaches(clientCache_, serverCache_) < 0)
        {
          #ifdef PANIC
          *logofs << "Proxy: PANIC! Failed to replace channel caches in "
                  << "channel for FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Failed to replace channel caches in "
               << "channel for FD#" << getFd(channelId) << ".\n";

          return -1;
        }
        #ifdef TEST
        else
        {
          *logofs << "Proxy: Replaced encode caches in channel "
                  << "for FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
        }
        #endif
      }
    }
  }

  return 1;
}

void Proxy::handleResetFlush()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Going to reset counters "
          << "for proxy FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  //
  // Restore buffers to their initial
  // size.
  //

  transport_ -> partialReset();

  //
  // Reset counters used to determine
  // the best time to flush the link.
  //

  control -> resetTimeInARow();

  control -> resetBytesInARow();

  control -> resetOutputInARow();

  //
  // Reset timer to the next flush.
  //

  control -> setLastFlushInfo();

  //
  // Reset priority flag on proxy
  // and channels.
  //

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      channels_[channelId] -> clearPriority();
      channels_[channelId] -> clearFlush();
    }
  }

  priority_ = 0;
}

int Proxy::handleShutdown()
{
  //
  // Send shutdown message to remote proxy.
  //

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Sending CODE_SHUTDOWN_REQUEST message.\n"
          << logofs_flush;
  #endif

  priority_ = 1;
  shutdown_ = 1;

  handleControl(CODE_SHUTDOWN_REQUEST);

  //
  // Ensure data is flushed.
  //

  for (int i = 0; i < 5; i++)
  {
    handleFlush(flush_if_any);

    usleep(200000);

    if (transport_ -> length() == 0 &&
            transport_ -> queued() <= 0)
    {
      break;
    }
  }

  //
  // Give time to the remote end
  // to read the shutdown message.
  //

  for (int i = 0; i < 5; i++)
  {
    if (transport_ -> readable() < 0)
    {
      break;
    }

    usleep(200000);
  }

  return 1;
}

int Proxy::handleFinish()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Closing down all the X channels.\n"
          << logofs_flush;
  #endif

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getType() == CHANNEL_X &&
                channels_[channelId] -> getFinish() == 0)
    {
      shutdown(getFd(channelId), SHUT_RD);

      if (handleFinish(channelId) < 0)
      {
        return -1;
      }
    }
  }

  return 1;
}

int Proxy::handleSocketConfiguration()
{
  //
  // Set linger mode on proxy to correctly
  // get shutdown notification.
  //

  SetLingerTimeout(fd_, 30);

  //
  // Set keep-alive on socket so that if remote link
  // terminates abnormally (as killed hard or because
  // of a power-off) process will get a SIGPIPE. In
  // practice this is useless as proxies already ping
  // each other every few seconds.
  //

  if (control -> OptionProxyKeepAlive == 1)
  {
    SetKeepAlive(fd_);
  }

  //
  // Set 'priority' flag at TCP layer for path
  // proxy-to-proxy. Look at IPTOS_LOWDELAY in
  // man 7 ip.
  //

  if (control -> OptionProxyLowDelay == 1)
  {
    SetLowDelay(fd_);
  }

  //
  // Update size of TCP send and receive buffers.
  //

  if (control -> OptionProxySendBuffer != -1)
  {
    SetSendBuffer(fd_, control -> OptionProxySendBuffer);
  }

  if (control -> OptionProxyReceiveBuffer != -1)
  {
    SetReceiveBuffer(fd_, control -> OptionProxyReceiveBuffer);
  }

  //
  // Update TCP_NODELAY settings. Note that turning off Nagle
  // algorithm when proxy is run through a PPP link seems to
  // not work. PPP, kernel or whoever is responsible for this,
  // stop delivering us data if a serious network congestion
  // is encountered. This is most likely to happen at server
  // side.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    if (control -> OptionProxyClientNoDelay != -1)
    {
      SetNoDelay(fd_, control -> OptionProxyClientNoDelay);
    }
  }
  else
  {
    if (control -> OptionProxyServerNoDelay != -1)
    {
      SetNoDelay(fd_, control -> OptionProxyServerNoDelay);
    }
  }

  return 1;
}

int Proxy::handleLinkConfiguration()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Configuring proxy for FD#"
          << fd_ << " according to control parameters.\n"
          << logofs_flush;
  #endif

  //
  // Translate parameters in 'local' and 'remote'.
  //

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    control -> LocalTimeInARowLimit   = control -> ClientTimeInARowLimit;
    control -> LocalBytesInARowLimit  = control -> ClientBytesInARowLimit;
    control -> LocalOutputInARowLimit = control -> ClientOutputInARowLimit;
  }
  else
  {
    control -> LocalTimeInARowLimit   = control -> ServerTimeInARowLimit;
    control -> LocalBytesInARowLimit  = control -> ServerBytesInARowLimit;
    control -> LocalOutputInARowLimit = control -> ServerOutputInARowLimit;
  }

  if (control -> LocalTimeInARowLimit <= 0 ||
          control -> LocalBytesInARowLimit <= 0 ||
              control -> LocalOutputInARowLimit <= 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Bad control parameters '"
            << control -> LocalTimeInARowLimit << "/"
            << control -> LocalBytesInARowLimit << "/"
            << control -> LocalOutputInARowLimit << "' "
            << "for local limits.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Bad control parameters '"
         << control -> LocalTimeInARowLimit << "/"
         << control -> LocalBytesInARowLimit << "/"
         << control -> LocalOutputInARowLimit << "' "
         << "for local limits.\n";

    return -1;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Propagating parameters to channels' read buffers.\n"
          << logofs_flush;
  #endif

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      channels_[channelId] -> handleConfiguration();
    }
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Propagating parameters to proxy buffers.\n"
          << logofs_flush;
  #endif

  readBuffer_.setSize(control -> ProxyInitialReadSize,
                          control -> ProxyMaximumReadSize);

  encodeBuffer_.setSize(control -> TransportProxyBufferSize,
                            control -> TransportProxyBufferThreshold,
                                control -> TransportMaximumBufferSize);

  transport_ -> setSize(control -> TransportProxyBufferSize,
                            control -> TransportProxyBufferThreshold,
                                control -> TransportMaximumBufferSize);

  //
  // Retry timeout depends on flush frequency
  // selected by user. Set now according to
  // the newly negotiated link parameters.
  //

  handleResetRetry();

  //
  // Adjust multiply factors in retry table
  // according to the newly selected link.
  //

  switch (control -> LinkMode)
  {
    //
    // Any restart of proxy to check if data can be written
    // to the proxy link happens after a timeout determined
    // by multiplying the current flush timeout for the value
    // in this table at position corresponding to the number
    // of restarts already performed in a row. After the 5th
    // restart, the factor at position 4 is taken. The choice
    // of factors in table are critical for performances. By
    // reducing the interval between retries it is possible
    // to benefit the responsiveness, increasing, at the same
    // time the overall system load.
    //
    // The default values were tested with flush timeout of
    // 20 and 10 for modem and isdn settings. When reducing
    // the flush timeout to 10 or 5 is suggested that you
    // adjust the table as follows.
    //
    // case LINK_TYPE_MODEM:
    // {
    //   lastRetryTable_[0] = 8;
    //   lastRetryTable_[1] = 6;
    //   lastRetryTable_[2] = 4;
    //   lastRetryTable_[3] = 2;
    //   lastRetryTable_[4] = 1.2;
    //
    //   break;
    // }
    // case LINK_TYPE_ISDN:
    // {
    //   lastRetryTable_[0] = 6;
    //   lastRetryTable_[1] = 4;
    //   lastRetryTable_[2] = 2;
    //   lastRetryTable_[3] = 1;
    //   lastRetryTable_[4] = 1.1;
    //
    //   break;
    // }
    //

    default:
    {
      lastRetryTable_[0] = 4;
      lastRetryTable_[1] = 3;
      lastRetryTable_[2] = 2;
      lastRetryTable_[3] = 1;
      lastRetryTable_[4] = 1.1;

      break;
    }
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Set factors in retry table to "
          << lastRetryTable_[0] << "/" << lastRetryTable_[1] << "/"
          << lastRetryTable_[2] << "/" << lastRetryTable_[3] << "/"
          << lastRetryTable_[4] << " with flush timeout "
          << control -> FlushTimeout << ".\n"
          << logofs_flush;
  #endif

  return 1;
}

int Proxy::handleCacheConfiguration()
{
  #if defined(INFO) || defined(TEST)
  *logofs << "Proxy: Configuring cache according to pack parameters.\n"
          << logofs_flush;
  #endif

  //
  // Further adjust cache parameters. If packing
  // of images is enabled, reduce size available
  // for plain images. At the moment this doesn't
  // apply do normal X applications as this has
  // not been built yet inside Xlib.
  //

  if (control -> SessionMode != SESSION_APPLICATION)
  {
    if (control -> AgentPackMethod != NO_PACK)
    {
      clientStore_ -> getRequestStore(X_PutImage) ->
          cacheThreshold = PUTIMAGE_CACHE_THRESHOLD_IF_PACKED;

      clientStore_ -> getRequestStore(X_PutImage) ->
          cacheLowerThreshold = PUTIMAGE_CACHE_LOWER_THRESHOLD_IF_PACKED;
    }
  }

  //
  // If this is a windows session (i.e. using any of
  // RDP pack methods), increase size of X_ChangeGC
  // and X_PolyFillRectangle cache.
  //

  if (control -> SessionMode == SESSION_RDP)
  {
    clientStore_ -> getRequestStore(X_ChangeGC) ->
        cacheThreshold = CHANGEGC_CACHE_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_ChangeGC) ->
        cacheLowerThreshold = CHANGEGC_CACHE_LOWER_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_PolyFillRectangle) ->
        cacheThreshold = POLYFILLRECTANGLE_CACHE_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_PolyFillRectangle) ->
        cacheLowerThreshold = POLYFILLRECTANGLE_CACHE_LOWER_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_NXPutPackedImage) ->
        cacheThreshold = PUTPACKEDIMAGE_CACHE_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_NXPutPackedImage) ->
        cacheLowerThreshold = PUTPACKEDIMAGE_CACHE_LOWER_THRESHOLD_IF_PACKED_RDP;
  }

  return 1;
}

int Proxy::handleSave()
{
  //
  // Save content of stores on disk.
  //

  char *cacheToAdopt = NULL;

  if (control -> PersistentCacheEnableSave)
  {
    #ifdef TEST
    *logofs << "Proxy: Going to save content of client store.\n"
            << logofs_flush;
    #endif

    cacheToAdopt = handleSaveStores(control -> PersistentCachePath);
  }
  #if defined(INFO) || defined(TEST)
  else
  {
    if (control -> ProxyMode == PROXY_CLIENT)
    {
      *logofs << "Proxy: Saving persistent cache to disk disabled.\n"
              << logofs_flush;
    }
    else
    {
      *logofs << "Proxy: PANIC! Protocol violation in command save.\n"
              << logofs_flush;

      cerr << "Error" << ": Protocol violation in command save.\n";

      HandleCleanup();
    }
  }
  #endif

  if (cacheToAdopt != NULL)
  {
    //
    // Do we have a cache already?
    //

    if (control -> PersistentCacheName != NULL)
    {
      //
      // Check if old and new cache are the same.
      // In this case don't remove the old cache.
      //

      if (strcasecmp(control -> PersistentCacheName, cacheToAdopt) != 0)
      {
        handleResetPersistentCache();
      }

      delete [] control -> PersistentCacheName;
    }

    #ifdef TEST
    *logofs << "Proxy: Setting current persistent cache file to '"
            << cacheToAdopt << "'\n" << logofs_flush;
    #endif

    control -> PersistentCacheName = cacheToAdopt;

    return 1;
  }
  #ifdef TEST
  else
  {
    *logofs << "Proxy: No cache file produced from message stores.\n"
            << logofs_flush;
  }
  #endif

  //
  // It can be that we didn't generate a new cache
  // because store was too small or persistent cache
  // was disabled. This is not an error.
  //

  return 0;
}

int Proxy::handleLoad()
{
  //
  // Restore content of client store from disk
  // if any valid cache was negotiated between
  // proxies.
  //

  if (control -> PersistentCacheEnableLoad == 1 &&
          control -> PersistentCachePath != NULL &&
              control -> PersistentCacheName != NULL)
  {
    #ifdef TEST
    *logofs << "Proxy: Going to load content of client store.\n"
            << logofs_flush;
    #endif

    //
    // Returns the same string passed as name of
    // the cache, or NULL if it was not possible
    // to load the cache from disk.
    //

    if (handleLoadStores(control -> PersistentCachePath,
                             control -> PersistentCacheName) == NULL)
    {
      //
      // Corrupted cache should have been removed
      // from disk. Get rid of reference to it so
      // we don't try to load it again in case of
      // proxy reset.
      //

      if (control -> PersistentCacheName != NULL)
      {
        delete [] control -> PersistentCacheName;
      }

      control -> PersistentCacheName = NULL;
 
      return -1;
    }

    //
    // Set timestamp of last time cache
    // was loaded from data on disk.
    //

    lastLoadTs_ = getTimestamp();

    return 1;
  }
  #ifdef TEST
  else
  {
    if (control -> ProxyMode == PROXY_CLIENT)
    {
      *logofs << "Proxy: Loading of cache disabled or no cache file selected.\n"
              << logofs_flush;
    }
    else
    {
      *logofs << "Proxy: PANIC! Protocol violation in command load.\n"
              << logofs_flush;

      cerr << "Error" << ": Protocol violation in command load.\n";

      HandleCleanup();
    }
  }
  #endif

  return 0;
}

int Proxy::handleControl(T_proxy_code controlCode, int controlData)
{
  //
  // Send the given control messages
  // to the remote proxy.
  //

  controlCodes_[controlLength_++] = 0;
  controlCodes_[controlLength_++] = (unsigned char) controlCode;
  controlCodes_[controlLength_++] = (unsigned char) controlData;

  control -> addBitsInARow(24);

  if (handleWrite() < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleSwitch(int channelId)
{
  //
  // If data is for a different channel than last
  // selected for output, prepend to data the new
  // channel id.
  //

  if (channelId != outputChannel_)
  {
    #ifdef TEST
    *logofs << "Proxy: Sending CODE_SWITCH_CONNECTION message "
            << "with FD#" << getFd(channelId) << ".\n"
            << logofs_flush;
    #endif

    outputChannel_ = channelId;

    controlCodes_[controlLength_++] = 0;
    controlCodes_[controlLength_++] = (unsigned char) CODE_SWITCH_CONNECTION;
    controlCodes_[controlLength_++] = (unsigned char) channelId;

    control -> addBitsInARow(24);

    if (controlLength_ == CONTROL_CODES_LENGTH)
    {
      if (handleWrite() < 0)
      {
        return -1;
      }
    }
  }

  return 1;
}

void Proxy::handleFailOnSave(const char *fullName, const char *failContext) const
{
  #ifdef PANIC
  *logofs << "Proxy: PANIC! Error saving stores to cache file "
          << "in context [" << failContext << "].\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Error saving stores to cache file "
       << "in context [" << failContext << "].\n";

  #ifdef PANIC
  *logofs << "Proxy: PANIC! Removing corrupted cache '"
          << fullName << "'.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Removing corrupted cache '"
       << fullName << "'.\n";

  remove(fullName);
}

void Proxy::handleFailOnLoad(const char *fullName, const char *failContext) const
{
  #ifdef PANIC
  *logofs << "Proxy: PANIC! Error loading stores from cache file "
          << "in context [" << failContext << "].\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Error loading stores from cache file "
       << "in context [" << failContext << "].\n";

  #ifdef PANIC
  *logofs << "Proxy: PANIC! Removing corrupted cache '"
          << fullName << "'.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Removing corrupted cache '"
       << fullName << "'.\n";

  remove(fullName);
}

int Proxy::handleSaveVersion(unsigned char *buffer, int &major,
                                 int &minor, int &patch) const
{
  if (control -> isProtoStep4() == 1)
  {
    major = control -> LocalVersionMajor;
    minor = control -> LocalVersionMinor;
    patch = control -> LocalVersionPatch;

    *(buffer + 0) = major;
    *(buffer + 1) = minor;

    PutUINT(patch, buffer + 2, storeBigEndian());
  }
  else if (control -> isProtoStep3() == 1)
  {
    //
    // We'll save cache in a compatible
    // format.
    //

    major = 1;
    minor = 2;
    patch = 2;

    *(buffer + 0) = major;
    *(buffer + 1) = minor;

    PutUINT(patch, buffer + 2, storeBigEndian());
  }
  else
  {
    major = control -> LocalVersionMajor;

    minor = 0;
    patch = 0;

    *((int *) buffer) = major;
  }

  return 1;
}

int Proxy::handleLoadVersion(const unsigned char *buffer, int &major,
                                 int &minor, int &patch) const
{
  if (control -> isProtoStep3() == 1)
  {
    major = *(buffer + 0);
    minor = *(buffer + 1);

    patch = GetUINT(buffer + 2, storeBigEndian());
  }
  else
  {
    major = *((int *) buffer);

    minor = 0;
    patch = 0;
  }

  if (major != control -> LocalVersionMajor)
  {
    return -1;
  }

  if (control -> isProtoStep3() == 1)
  {
    //
    // Refuse to load caches with a minor version
    // greater than local, while different patch
    // versions are assumed to be compatible.
    //

    if (minor > control -> LocalVersionMinor)
    {
      return -1;
    }

    //
    // Caches from 1.2.1 are not compatible while
    // caches from version 1.2.2 are managed in
    // a compatible way...
    //

    if (minor == 2 && patch <= 1)
    {
      return -1;
    }

    //
    // ...But not if both sides use protocol 4.
    //

    if (control -> isProtoStep4() == 1)
    {
      if (minor == 2 && patch == 2)
      {
        return -1;
      }
    }
  }

  return 1;
}

char *Proxy::handleSaveStores(const char *savePath) const
{
  int cumulativeSize = MessageStore::getCumulativeTotalStorageSize();

  if (cumulativeSize < control -> PersistentCacheThreshold)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: Cache not saved as size is "
            << cumulativeSize << " with threshold set to "
            << control -> PersistentCacheThreshold
            << ".\n" << logofs_flush;
    #endif

    return NULL;
  }
  else if (savePath == NULL)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! No name provided for save path.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": No name provided for save path.\n";

    return NULL;
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Going to save content of message stores.\n"
          << logofs_flush;
  #endif

  //
  // Our parent process is likely going to terminate.
  // Until we finish saving cache we must ignore its
  // SIGIPE.
  //

  DisableSignals();

  ofstream *cachefs = NULL;

  md5_state_t *md5StateStream = NULL;
  md5_byte_t  *md5DigestStream = NULL;

  md5_state_t *md5StateClient = NULL;
  md5_byte_t  *md5DigestClient = NULL;

  char *tempName = NULL;

  char md5_string[MD5_LENGTH * 2 + 2];

  char fullName[strlen(savePath) + MD5_LENGTH * 2 + 4];

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    tempName = tempnam(savePath, "Z-C-");
  }
  else
  {
    tempName = tempnam(savePath, "Z-S-");
  }

  cachefs = new ofstream(tempName, ios::out | ios::binary);

  if (tempName == NULL || cachefs == NULL)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Can't create temporary file in '"
            << savePath << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't create temporary file in '"
         << savePath << "'.\n";

    EnableSignals();

    if (tempName != NULL)
    {
      free(tempName);
    }

    if (cachefs != NULL)
    {
      delete cachefs;
    }

    return NULL;
  }

  md5StateStream  = new md5_state_t();
  md5DigestStream = new md5_byte_t[MD5_LENGTH];

  md5_init(md5StateStream);

  //
  // First write nxproxy version. In protocol
  // level 3 we write all version information
  // not just the major number.
  //

  unsigned char version[4];

  int major;
  int minor;
  int patch;

  handleSaveVersion(version, major, minor, patch);

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Saving cache using version '"
          << major << "." << minor << "." << patch
          << "'.\n" << logofs_flush;
  #endif

  if (PutData(cachefs, version, 4) < 0)
  {
    handleFailOnSave(tempName, "A");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    EnableSignals();

    return NULL;
  }

  //
  // Don't include version in checksum. This
  // will allow proxy to load caches produced
  // by a different version.
  //

  if (control -> isProtoStep3() == 0)
  {
    md5_append(md5StateStream, version, 4);
  }

  //
  // Make space for the calculated MD5 so we
  // can later rewind the file and write it
  // at this position.
  //

  if (PutData(cachefs, md5DigestStream, MD5_LENGTH) < 0)
  {
    handleFailOnSave(tempName, "B");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    EnableSignals();

    return NULL;
  }

  md5StateClient  = new md5_state_t();
  md5DigestClient = new md5_byte_t[MD5_LENGTH];

  md5_init(md5StateClient);

  #ifdef DUMP

  ofstream *cache_dump = NULL;

  ofstream *tempfs = (ofstream*) logofs;

  if (control -> ProxyMode == PROXY_CLIENT)
  {
    cache_dump = new ofstream("/tmp/nxproxy-client-cache.dump", ios::out);
  }
  else
  {
    cache_dump = new ofstream("/tmp/nxproxy-server-cache.dump", ios::out);
  }

  logofs = cache_dump;

  #endif

  //
  // Use the virtual method of the concrete proxy class.
  //

  int allSaved = handleSaveStores(cachefs, md5StateStream, md5StateClient);

  #ifdef DUMP

  logofs = tempfs;

  delete cache_dump;

  #endif

  if (allSaved == 0)
  {
    handleFailOnSave(tempName, "C");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    delete md5StateClient;
    delete [] md5DigestClient;

    EnableSignals();

    return NULL;
  }

  md5_finish(md5StateClient, md5DigestClient);

  for (unsigned int i = 0; i < MD5_LENGTH; i++)
  {
    sprintf(md5_string + (i * 2), "%02X", md5DigestClient[i]);
  }

  strcpy(fullName, (control -> ProxyMode == PROXY_CLIENT) ? "C-" : "S-");

  strcat(fullName, md5_string);

  md5_append(md5StateStream, (const md5_byte_t *) fullName, strlen(fullName));
  md5_finish(md5StateStream, md5DigestStream);

  //
  // Go to the beginning of file plus
  // the integer where we wrote our
  // proxy version.
  //

  cachefs -> seekp(4);

  if (PutData(cachefs, md5DigestStream, MD5_LENGTH) < 0)
  {
    handleFailOnSave(tempName, "D");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    delete md5StateClient;
    delete [] md5DigestClient;

    EnableSignals();

    return NULL;
  }

  delete cachefs;

  //
  // Save finally the resulting cache name
  // without the path.
  //

  char *cacheName = new char[MD5_LENGTH * 2 + 4];

  strcpy(cacheName, fullName);

  strcpy(fullName, savePath);
  strcat(fullName, (control -> ProxyMode == PROXY_CLIENT) ? "/C-" : "/S-");
  strcat(fullName, md5_string);

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Renaming cache file to "
          << fullName << "'.\n" << logofs_flush;
  #endif

  rename(tempName, fullName);

  delete md5StateStream;
  delete [] md5DigestStream;

  delete md5StateClient;
  delete [] md5DigestClient;

  free(tempName);

  //
  // Restore the original handlers.
  //

  EnableSignals();

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Successfully saved cache file '"
          << cacheName << "'.\n" << logofs_flush;
  #endif

  return cacheName;
}

const char *Proxy::handleLoadStores(const char *loadPath, const char *loadName) const
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Going to load content of message stores.\n"
          << logofs_flush;
  #endif

  //
  // Until we finish loading cache we
  // must at least ignore any SIGIPE.
  //

  DisableSignals();

  if (loadPath == NULL || loadName == NULL)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! No path or no file name provided for cache to restore.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": No path or no file name provided for cache to restore.\n";

    EnableSignals();

    return NULL;
  }
  else if (strlen(loadName) != MD5_LENGTH * 2 + 2)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Bad file name provided for cache to restore.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Bad file name provided for cache to restore.\n";

    EnableSignals();

    return NULL;
  }

  istream *cachefs = NULL;
  char md5_string[(MD5_LENGTH * 2) + 2];
  md5_byte_t md5_from_file[MD5_LENGTH];

  char *cacheName = new char[strlen(loadPath) + strlen(loadName) + 3];

  strcpy(cacheName, loadPath);
  strcat(cacheName, "/");
  strcat(cacheName, loadName);

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Name of cache file is '"
          << cacheName << "'.\n" << logofs_flush;
  #endif

  cachefs = new ifstream(cacheName, ios::in | ios::binary);

  unsigned char version[4];

  if (cachefs == NULL || GetData(cachefs, version, 4) < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Can't read cache file '"
            << cacheName << "'.\n" << logofs_flush;;
    #endif

    handleFailOnLoad(cacheName, "A");

    delete cachefs;

    delete [] cacheName;

    EnableSignals();

    return NULL;
  }

  int major;
  int minor;
  int patch;

  if (handleLoadVersion(version, major, minor, patch) < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Incompatible version '"
            << major << "." << minor << "." << patch
            << "' in cache file '" << cacheName
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Incompatible version '"
         << major << "." << minor << "." << patch
         << "' in cache file '" << cacheName
         << "'.\n" << logofs_flush;

    handleFailOnLoad(cacheName, "B");

    delete cachefs;

    delete [] cacheName;

    EnableSignals();

    return NULL;
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Reading from cache file version '"
          << major << "." << minor << "." << patch
          << "'.\n" << logofs_flush;
  #endif

  if (GetData(cachefs, md5_from_file, MD5_LENGTH) < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! No checksum in cache file '"
            << loadName << "'.\n" << logofs_flush;
    #endif

    handleFailOnLoad(cacheName, "C");

    delete cachefs;

    delete [] cacheName;

    EnableSignals();

    return NULL;
  }

  md5_state_t *md5StateStream = NULL;
  md5_byte_t  *md5DigestStream = NULL;

  md5StateStream  = new md5_state_t();
  md5DigestStream = new md5_byte_t[MD5_LENGTH];

  md5_init(md5StateStream);

  //
  // Don't include version in checksum.
  //

  if (control -> isProtoStep3() == 0)
  {
    md5_append(md5StateStream, version, 4);
  }

  //
  // Use the virtual method of the concrete proxy class.
  //

  if (handleLoadStores(cachefs, md5StateStream) < 0)
  {
    handleFailOnLoad(cacheName, "D");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    delete [] cacheName;

    EnableSignals();

    return NULL;
  }

  md5_append(md5StateStream, (const md5_byte_t *) loadName, strlen(loadName));
  md5_finish(md5StateStream, md5DigestStream);

  for (int i = 0; i < MD5_LENGTH; i++)
  {
    if (md5DigestStream[i] != md5_from_file[i])
    {
      #ifdef PANIC

      *logofs << "Proxy: PANIC! Bad checksum for cache file '"
              << cacheName << "'.\n" <<  logofs_flush;

      for (unsigned int i = 0; i < MD5_LENGTH; i++)
      {
        sprintf(md5_string + (i * 2), "%02X", md5_from_file[i]);
      }

      *logofs << "Proxy: PANIC! Saved checksum is '"
              << md5_string << "'.\n" <<  logofs_flush;

      for (unsigned int i = 0; i < MD5_LENGTH; i++)
      {
        sprintf(md5_string + (i * 2),"%02X", md5DigestStream[i]);
      }

      *logofs << "Proxy: PANIC! Calculated checksum is '"
              << md5_string << "'.\n" <<  logofs_flush;

      #endif

      handleFailOnLoad(cacheName, "E");

      delete cachefs;

      delete md5StateStream;
      delete [] md5DigestStream;

      delete [] cacheName;

      EnableSignals();

      return NULL;
    }
  }

  delete cachefs;

  delete md5StateStream;
  delete [] md5DigestStream;

  delete [] cacheName;

  //
  // Restore the original handlers.
  //

  EnableSignals();

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Successfully loaded cache file '"
          << loadName << "'.\n" << logofs_flush;
  #endif

  //
  // Return the string provided by caller.
  //

  return loadName;
}

void Proxy::increaseActiveChannels(int channelId)
{
  activeChannels_++;

  if (channelId > upperChannel_)
  {
    upperChannel_ = channelId;

    while (channels_[lowerChannel_] == NULL &&
               lowerChannel_ < upperChannel_)
    {
      lowerChannel_++;
    }
  }

  if (channelId < lowerChannel_)
  {
    lowerChannel_ = channelId;

    while (channels_[upperChannel_] == NULL &&
               upperChannel_ > lowerChannel_)
    {
      upperChannel_--;
    }
  }

  if (firstChannel_ > upperChannel_)
  {
    firstChannel_ = upperChannel_;
  }
  else if (firstChannel_ < lowerChannel_)
  {
    firstChannel_ = lowerChannel_;
  }

  #ifdef TEST
  *logofs << "Proxy: Active channels are "
          << activeChannels_ << " lower channel is "
          << lowerChannel_ << " upper channel is "
          << upperChannel_ << " first channel is "
          << firstChannel_ << ".\n" << logofs_flush;
  #endif
}

void Proxy::decreaseActiveChannels(int channelId)
{
  activeChannels_--;

  while (channels_[upperChannel_] == NULL &&
             upperChannel_ > lowerChannel_)
  {
    upperChannel_--;
  }

  while (channels_[lowerChannel_] == NULL &&
             lowerChannel_ < upperChannel_)
  {
    lowerChannel_++;
  }

  if (firstChannel_ > upperChannel_)
  {
    firstChannel_ = upperChannel_;
  }
  else if (firstChannel_ < lowerChannel_)
  {
    firstChannel_ = lowerChannel_;
  }

  #ifdef TEST
  *logofs << "Proxy: Active channels are "
          << activeChannels_ << " lower channel is "
          << lowerChannel_ << " upper channel is "
          << upperChannel_ << " first channel is "
          << firstChannel_ << ".\n" << logofs_flush;
  #endif
}
