/*
 * Copyright (C) 2003 Apple Computer, Inc.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
#ifndef WebCoreResourceLoader_h
#define WebCoreResourceLoader_h

#include <glib.h>

class WebCoreResponse;

/** 
 * Interface for resource loading notifications
 * These functions to pass data to WebCore
 * 
 * an instance of WebCoreResourceLoader must not
 * go out of scope before call to either

 * \WebCoreResourceLoader::finish()
 * \WebCoreResourceLoader::reportError()
 * \WebCoreResourceLoader::cancel()

 */ 
class WebCoreResourceLoader
{
public:

    virtual ~WebCoreResourceLoader() {}

    virtual void receivedResponse(WebCoreResponse* response) =0;
    virtual void redirectedToURL(const gchar* URL) =0;
    virtual void addData(const void *data, unsigned int length) =0;

    /* A call to  finish, reportError, or cancel MUST be received 
	 * before the loader is deleted 
	 */

	/** called on finished loading*/
    virtual void finish(void) =0;

	/** called on error while loading */
    virtual void reportError(void) =0;

	/** called on loading cancelled (@see \WebCoreResourceHandle) */
    virtual void cancel(void) =0;
};

/**
 * WebCoreResourceHandle per resource request
 *
 * WebCore calls cancel() to stop the transfer when it wants
 * to give up the loading.
 * 
 * Must not be referenced after call to cancel(), or after
 * corresponding WebCoreResourceLoader has received one of the
 * following:
 * \WebCoreResourceLoader::finish()
 * \WebCoreResourceLoader::reportError()
 * \WebCoreResourceLoader::cancel()
 *
 *
 */
class WebCoreResourceHandle 
{
public:
    virtual ~WebCoreResourceHandle() {}

    /** stop the transfer*/  
    virtual void cancel()=0;
    
};

#endif
