/*
 * MD5 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * MD5 algorithm by Ron Rivest, see RFC1321
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2202
 *
 * License:
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact astmail@yahoo.com
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the MD5 routines.
 *
 * Data sizes:
 *
 * data block for MD5 transformation	WORD08[64]
 * resulting MD5 or HMAC hash		WORD08[16]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * MD5_FULL defined:
 *
 * 1. Call md5init
 * 2. For all data to be hashed call md5next
 * 3. To retrieve the hash call md5end
 *
 * MD5_FAST defined:
 *
 * Call md5 to get the hash for the specified data
 *
 * MD5_HMAC_FULL defined:
 *
 * 1. Call md5hmkey once to preprocess the selected key
 * 2. Call md5hminit
 * 3. For all data to be hashed call md5hmnext
 * 4. To retrieve the hash call md5hmend
 * 5. When the preprocessed key is no longer required
 *    reset the MD5HMDATA structure
 *
 * MD5_HMAC_FAST defined:
 *
 * 1. Call md5hmkey once to preprocess the selected key
 * 2. Call md5hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the MD5HMDATA structure
 */

#include "common.h"
#include "md5.h"

/*
 * procedure md5block
 *
 * input: block - pointer to the 64 byte message block
 *
 * inout: md5 - pointer to the md5 working buffer
 *
 * This procedure processes the 64 byte message block
 * and updates the current md5 result.
 */

static void CRYPTOCALL md5block(WORD32 *md5,WORD08 *block)
{
	register WORD32 a;	/* md5 work buffer	*/
	register WORD32 b;	/* md5 work buffer	*/
	register WORD32 c;	/* md5 work buffer	*/
	register WORD32 d;	/* md5 work buffer	*/
	WORD32 bfr[16];		/* md5 work buffer	*/


	/* get 64 byte message block into 32 bit array */

	a=block[3];
	a<<=8;
	a+=block[2];
	a<<=8;
	a+=block[1];
	a<<=8;
	bfr[0]=a+block[0];

	a=block[7];
	a<<=8;
	a+=block[6];
	a<<=8;
	a+=block[5];
	a<<=8;
	bfr[1]=a+block[4];

	a=block[11];
	a<<=8;
	a+=block[10];
	a<<=8;
	a+=block[9];
	a<<=8;
	bfr[2]=a+block[8];

	a=block[15];
	a<<=8;
	a+=block[14];
	a<<=8;
	a+=block[13];
	a<<=8;
	bfr[3]=a+block[12];

	a=block[19];
	a<<=8;
	a+=block[18];
	a<<=8;
	a+=block[17];
	a<<=8;
	bfr[4]=a+block[16];

	a=block[23];
	a<<=8;
	a+=block[22];
	a<<=8;
	a+=block[21];
	a<<=8;
	bfr[5]=a+block[20];

	a=block[27];
	a<<=8;
	a+=block[26];
	a<<=8;
	a+=block[25];
	a<<=8;
	bfr[6]=a+block[24];

	a=block[31];
	a<<=8;
	a+=block[30];
	a<<=8;
	a+=block[29];
	a<<=8;
	bfr[7]=a+block[28];

	a=block[35];
	a<<=8;
	a+=block[34];
	a<<=8;
	a+=block[33];
	a<<=8;
	bfr[8]=a+block[32];

	a=block[39];
	a<<=8;
	a+=block[38];
	a<<=8;
	a+=block[37];
	a<<=8;
	bfr[9]=a+block[36];

	a=block[43];
	a<<=8;
	a+=block[42];
	a<<=8;
	a+=block[41];
	a<<=8;
	bfr[10]=a+block[40];

	a=block[47];
	a<<=8;
	a+=block[46];
	a<<=8;
	a+=block[45];
	a<<=8;
	bfr[11]=a+block[44];

	a=block[51];
	a<<=8;
	a+=block[50];
	a<<=8;
	a+=block[49];
	a<<=8;
	bfr[12]=a+block[48];

	a=block[55];
	a<<=8;
	a+=block[54];
	a<<=8;
	a+=block[53];
	a<<=8;
	bfr[13]=a+block[52];

	a=block[59];
	a<<=8;
	a+=block[58];
	a<<=8;
	a+=block[57];
	a<<=8;
	bfr[14]=a+block[56];

	a=block[63];
	a<<=8;
	a+=block[62];
	a<<=8;
	a+=block[61];
	a<<=8;
	bfr[15]=a+block[60];

	/* get previous md5 block result into working buffer */

	a=md5[0];
	b=md5[1];
	c=md5[2];
	d=md5[3];

	/* do round one */

	a+=bfr[0];
	a+=0xd76aa478;
	a+=d^(b&(c^d));
	a=(a<<7)|(BITS32(a)>>25);
	a+=b;
	d+=bfr[1];
	d+=0xe8c7b756;
	d+=c^(a&(b^c));
	d=(d<<12)|(BITS32(d)>>20);
	d+=a;
	c+=bfr[2];
	c+=0x242070db;
	c+=b^(d&(a^b));
	c=(c<<17)|(BITS32(c)>>15);
	c+=d;
	b+=bfr[3];
	b+=0xc1bdceee;
	b+=a^(c&(d^a));
	b=(b<<22)|(BITS32(b)>>10);
	b+=c;
	a+=bfr[4];
	a+=0xf57c0faf;
	a+=d^(b&(c^d));
	a=(a<<7)|(BITS32(a)>>25);
	a+=b;
	d+=bfr[5];
	d+=0x4787c62a;
	d+=c^(a&(b^c));
	d=(d<<12)|(BITS32(d)>>20);
	d+=a;
	c+=bfr[6];
	c+=0xa8304613;
	c+=b^(d&(a^b));
	c=(c<<17)|(BITS32(c)>>15);
	c+=d;
	b+=bfr[7];
	b+=0xfd469501;
	b+=a^(c&(d^a));
	b=(b<<22)|(BITS32(b)>>10);
	b+=c;
	a+=bfr[8];
	a+=0x698098d8;
	a+=d^(b&(c^d));
	a=(a<<7)|(BITS32(a)>>25);
	a+=b;
	d+=bfr[9];
	d+=0x8b44f7af;
	d+=c^(a&(b^c));
	d=(d<<12)|(BITS32(d)>>20);
	d+=a;
	c+=bfr[10];
	c+=0xffff5bb1;
	c+=b^(d&(a^b));
	c=(c<<17)|(BITS32(c)>>15);
	c+=d;
	b+=bfr[11];
	b+=0x895cd7be;
	b+=a^(c&(d^a));
	b=(b<<22)|(BITS32(b)>>10);
	b+=c;
	a+=bfr[12];
	a+=0x6b901122;
	a+=d^(b&(c^d));
	a=(a<<7)|(BITS32(a)>>25);
	a+=b;
	d+=bfr[13];
	d+=0xfd987193;
	d+=c^(a&(b^c));
	d=(d<<12)|(BITS32(d)>>20);
	d+=a;
	c+=bfr[14];
	c+=0xa679438e;
	c+=b^(d&(a^b));
	c=(c<<17)|(BITS32(c)>>15);
	c+=d;
	b+=bfr[15];
	b+=0x49b40821;
	b+=a^(c&(d^a));
	b=(b<<22)|(BITS32(b)>>10);
	b+=c;

	/* do round two */

	a+=bfr[1];
	a+=0xf61e2562;
	a+=c^(d&(b^c));
	a=(a<<5)|(BITS32(a)>>27);
	a+=b;
	d+=bfr[6];
	d+=0xc040b340;
	d+=b^(c&(a^b));
	d=(d<<9)|(BITS32(d)>>23);
	d+=a;
	c+=bfr[11];
	c+=0x265e5a51;
	c+=a^(b&(d^a));
	c=(c<<14)|(BITS32(c)>>18);
	c+=d;
	b+=bfr[0];
	b+=0xe9b6c7aa;
	b+=d^(a&(c^d));
	b=(b<<20)|(BITS32(b)>>12);
	b+=c;
	a+=bfr[5];
	a+=0xd62f105d;
	a+=c^(d&(b^c));
	a=(a<<5)|(BITS32(a)>>27);
	a+=b;
	d+=bfr[10];
	d+=0x02441453;
	d+=b^(c&(a^b));
	d=(d<<9)|(BITS32(d)>>23);
	d+=a;
	c+=bfr[15];
	c+=0xd8a1e681;
	c+=a^(b&(d^a));
	c=(c<<14)|(BITS32(c)>>18);
	c+=d;
	b+=bfr[4];
	b+=0xe7d3fbc8;
	b+=d^(a&(c^d));
	b=(b<<20)|(BITS32(b)>>12);
	b+=c;
	a+=bfr[9];
	a+=0x21e1cde6;
	a+=c^(d&(b^c));
	a=(a<<5)|(BITS32(a)>>27);
	a+=b;
	d+=bfr[14];
	d+=0xc33707d6;
	d+=b^(c&(a^b));
	d=(d<<9)|(BITS32(d)>>23);
	d+=a;
	c+=bfr[3];
	c+=0xf4d50d87;
	c+=a^(b&(d^a));
	c=(c<<14)|(BITS32(c)>>18);
	c+=d;
	b+=bfr[8];
	b+=0x455a14ed;
	b+=d^(a&(c^d));
	b=(b<<20)|(BITS32(b)>>12);
	b+=c;
	a+=bfr[13];
	a+=0xa9e3e905;
	a+=c^(d&(b^c));
	a=(a<<5)|(BITS32(a)>>27);
	a+=b;
	d+=bfr[2];
	d+=0xfcefa3f8;
	d+=b^(c&(a^b));
	d=(d<<9)|(BITS32(d)>>23);
	d+=a;
	c+=bfr[7];
	c+=0x676f02d9;
	c+=a^(b&(d^a));
	c=(c<<14)|(BITS32(c)>>18);
	c+=d;
	b+=bfr[12];
	b+=0x8d2a4c8a;
	b+=d^(a&(c^d));
	b=(b<<20)|(BITS32(b)>>12);
	b+=c;

	/* do round three */

	a+=bfr[5];
	a+=0xfffa3942;
	a+=b^c^d;
	a=(a<<4)|(BITS32(a)>>28);
	a+=b;
	d+=bfr[8];
	d+=0x8771f681;
	d+=a^b^c;
	d=(d<<11)|(BITS32(d)>>21);
	d+=a;
	c+=bfr[11];
	c+=0x6d9d6122;
	c+=d^a^b;
	c=(c<<16)|(BITS32(c)>>16);
	c+=d;
	b+=bfr[14];
	b+=0xfde5380c;
	b+=c^d^a;
	b=(b<<23)|(BITS32(b)>>9);
	b+=c;
	a+=bfr[1];
	a+=0xa4beea44;
	a+=b^c^d;
	a=(a<<4)|(BITS32(a)>>28);
	a+=b;
	d+=bfr[4];
	d+=0x4bdecfa9;
	d+=a^b^c;
	d=(d<<11)|(BITS32(d)>>21);
	d+=a;
	c+=bfr[7];
	c+=0xf6bb4b60;
	c+=d^a^b;
	c=(c<<16)|(BITS32(c)>>16);
	c+=d;
	b+=bfr[10];
	b+=0xbebfbc70;
	b+=c^d^a;
	b=(b<<23)|(BITS32(b)>>9);
	b+=c;
	a+=bfr[13];
	a+=0x289b7ec6;
	a+=b^c^d;
	a=(a<<4)|(BITS32(a)>>28);
	a+=b;
	d+=bfr[0];
	d+=0xeaa127fa;
	d+=a^b^c;
	d=(d<<11)|(BITS32(d)>>21);
	d+=a;
	c+=bfr[3];
	c+=0xd4ef3085;
	c+=d^a^b;
	c=(c<<16)|(BITS32(c)>>16);
	c+=d;
	b+=bfr[6];
	b+=0x04881d05;
	b+=c^d^a;
	b=(b<<23)|(BITS32(b)>>9);
	b+=c;
	a+=bfr[9];
	a+=0xd9d4d039;
	a+=b^c^d;
	a=(a<<4)|(BITS32(a)>>28);
	a+=b;
	d+=bfr[12];
	d+=0xe6db99e5;
	d+=a^b^c;
	d=(d<<11)|(BITS32(d)>>21);
	d+=a;
	c+=bfr[15];
	c+=0x1fa27cf8;
	c+=d^a^b;
	c=(c<<16)|(BITS32(c)>>16);
	c+=d;
	b+=bfr[2];
	b+=0xc4ac5665;
	b+=c^d^a;
	b=(b<<23)|(BITS32(b)>>9);
	b+=c;

	/* do round four */

	a+=bfr[0];
	a+=0xf4292244;
	a+=c^(b|~d);
	a=(a<<6)|(BITS32(a)>>26);
	a+=b;
	d+=bfr[7];
	d+=0x432aff97;
	d+=b^(a|~c);
	d=(d<<10)|(BITS32(d)>>22);
	d+=a;
	c+=bfr[14];
	c+=0xab9423a7;
	c+=a^(d|~b);
	c=(c<<15)|(BITS32(c)>>17);
	c+=d;
	b+=bfr[5];
	b+=0xfc93a039;
	b+=d^(c|~a);
	b=(b<<21)|(BITS32(b)>>11);
	b+=c;
	a+=bfr[12];
	a+=0x655b59c3;
	a+=c^(b|~d);
	a=(a<<6)|(BITS32(a)>>26);
	a+=b;
	d+=bfr[3];
	d+=0x8f0ccc92;
	d+=b^(a|~c);
	d=(d<<10)|(BITS32(d)>>22);
	d+=a;
	c+=bfr[10];
	c+=0xffeff47d;
	c+=a^(d|~b);
	c=(c<<15)|(BITS32(c)>>17);
	c+=d;
	b+=bfr[1];
	b+=0x85845dd1;
	b+=d^(c|~a);
	b=(b<<21)|(BITS32(b)>>11);
	b+=c;
	a+=bfr[8];
	a+=0x6fa87e4f;
	a+=c^(b|~d);
	a=(a<<6)|(BITS32(a)>>26);
	a+=b;
	d+=bfr[15];
	d+=0xfe2ce6e0;
	d+=b^(a|~c);
	d=(d<<10)|(BITS32(d)>>22);
	d+=a;
	c+=bfr[6];
	c+=0xa3014314;
	c+=a^(d|~b);
	c=(c<<15)|(BITS32(c)>>17);
	c+=d;
	b+=bfr[13];
	b+=0x4e0811a1;
	b+=d^(c|~a);
	b=(b<<21)|(BITS32(b)>>11);
	b+=c;
	a+=bfr[4];
	a+=0xf7537e82;
	a+=c^(b|~d);
	a=(a<<6)|(BITS32(a)>>26);
	a+=b;
	d+=bfr[11];
	d+=0xbd3af235;
	d+=b^(a|~c);
	d=(d<<10)|(BITS32(d)>>22);
	d+=a;
	c+=bfr[2];
	c+=0x2ad7d2bb;
	c+=a^(d|~b);
	c=(c<<15)|(BITS32(c)>>17);
	c+=d;
	b+=bfr[9];
	b+=0xeb86d391;
	b+=d^(c|~a);
	b=(b<<21)|(BITS32(b)>>11);
	b+=c;

	/* create new md5 block result */

	md5[0]+=a;
	md5[1]+=b;
	md5[2]+=c;
	md5[3]+=d;
}

/*
 * procedure md5init
 *
 * input: ptr - pointer to MD5 internal data
 *
 * This procedure initializes the MD5 internal data structure.
 */

#if defined(MD5_FULL) || defined(MD5_HMAC)

void CRYPTOCALL md5init(register MD5DATA *ptr)
{
	ptr->total=ptr->size=0;
	ptr->md5[0]=0x67452301;
	ptr->md5[1]=0xefcdab89;
	ptr->md5[2]=0x98badcfe;
	ptr->md5[3]=0x10325476;
}

#endif

/*
 * procedure md5next
 *
 * input: data   - the data to be hashed
 *	  length - the length of the data to be hashed
 *	  ptr    - pointer to MD5 internal data
 *
 * This procedure hashes the given block of data. Incomplete data blocks
 * are buffered in the MD5 internal data structure.
 */

#if defined(MD5_FULL) || defined(MD5_HMAC)

void CRYPTOCALL md5next(register WORD08 *data,register WORD32 length,
	register MD5DATA *ptr)
{
	register WORD32 i;	/* buffer size/index	*/


	/* adjust total length */

	ptr->total+=length;

	/* while there is a remainder in the buffer and data are to be
	   processed and the buffer is notfull copy current byte to
	   buffer */

	for(i=ptr->size;(i&63)&&length;length--)ptr->bfr.b[i++]=*data++;

	/* if the buffer is full */

	if(i==64)
	{
		/* reset buffer size */

		i=0;

		/* hash buffer */

		md5block(ptr->md5,ptr->bfr.b);
	}

	/* do all complete blocks */

	for(;length>63;data+=64,length-=64)
		md5block(ptr->md5,data);

	/* copy remainder to buffer */

	for(;length;length--)ptr->bfr.b[i++]=*data++;

	/* save new buffer size/index */

	ptr->size=(WORD08)(i);
}

#endif

/*
 * procedure md5end
 *
 * input:  ptr - pointer to MD5 internal data
 *
 * output: result - the MD5 hash
 *
 * This procedure finalizes and returns the MD5 hash.
 */

#if defined(MD5_FULL) || defined(MD5_HMAC)

void CRYPTOCALL md5end(register WORD08 *result,register MD5DATA *ptr)
{
	register WORD32 i=ptr->size;		/* counter		*/


	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	ptr->bfr.b[i++]=0x80;
	if(i>56)
	{
		for(;i<64;i++)ptr->bfr.b[i]=0;
		i=0;
		md5block(ptr->md5,ptr->bfr.b);
	}
	for(;i<56;i++)ptr->bfr.b[i]=0;

	/* append bits 0 through 31 of total length in bits to buffer */

	ptr->bfr.b[56]=(WORD08)((ptr->total)<<3);
	ptr->bfr.b[57]=(WORD08)((ptr->total)>>5);
	ptr->bfr.b[58]=(WORD08)((ptr->total)>>13);
	ptr->bfr.b[59]=(WORD08)((ptr->total)>>21);

	/* append bits 32 through 63 of total length in bits to buffer */

	ptr->bfr.b[60]=(WORD08)(BITS32(ptr->total)>>29);
	ptr->bfr.b[61]=0;
	ptr->bfr.b[62]=0;
	ptr->bfr.b[63]=0;

	/* hash buffer */

	md5block(ptr->md5,ptr->bfr.b);

	/* return final hash result to caller */

	result[ 0]=(WORD08) (ptr->md5[0]);
	result[ 1]=(WORD08)((ptr->md5[0])>>8);
	result[ 2]=(WORD08)((ptr->md5[0])>>16);
	result[ 3]=(WORD08)((ptr->md5[0])>>24);
	result[ 4]=(WORD08) (ptr->md5[1]);
	result[ 5]=(WORD08)((ptr->md5[1])>>8);
	result[ 6]=(WORD08)((ptr->md5[1])>>16);
	result[ 7]=(WORD08)((ptr->md5[1])>>24);
	result[ 8]=(WORD08) (ptr->md5[2]);
	result[ 9]=(WORD08)((ptr->md5[2])>>8);
	result[10]=(WORD08)((ptr->md5[2])>>16);
	result[11]=(WORD08)((ptr->md5[2])>>24);
	result[12]=(WORD08) (ptr->md5[3]);
	result[13]=(WORD08)((ptr->md5[3])>>8);
	result[14]=(WORD08)((ptr->md5[3])>>16);
	result[15]=(WORD08)((ptr->md5[3])>>24);

	/* assert user data removal data from buffer */

#ifdef MD5_PARANOID
	ptr->bfr.l[0]=0;
	ptr->bfr.l[1]=0;
	ptr->bfr.l[2]=0;
	ptr->bfr.l[3]=0;
	ptr->bfr.l[4]=0;
	ptr->bfr.l[5]=0;
	ptr->bfr.l[6]=0;
	ptr->bfr.l[7]=0;
	ptr->bfr.l[8]=0;
	ptr->bfr.l[9]=0;
	ptr->bfr.l[10]=0;
	ptr->bfr.l[11]=0;
	ptr->bfr.l[12]=0;
	ptr->bfr.l[13]=0;
	ptr->bfr.l[14]=0;
	ptr->bfr.l[15]=0;
#endif
}

#endif

/*
 * procedure md5
 *
 * input:  data   - the data to be hashed
 *	   length - the length of the data to be hashed
 *
 * output: result - the MD5 hash
 *
 * This procedure hashes the given data.
 */

#ifdef MD5_FAST

void CRYPTOCALL md5(register WORD08 *data,register WORD32 length,WORD08 *result)
{
	register WORD32 i;	/* counter and index		*/
	WORD32 total;		/* data length buffer		*/
	WORD32 md5[4];		/* MD5 current result buffer	*/
	union
	{
	  WORD32 l[16];		/* paranoia memory clear	*/
	  WORD08 b[64];		/* final block(s) buffer	*/
	}bfr;


	/* initialize MD5 internal data */

	total=length;
	md5[0]=0x67452301;
	md5[1]=0xefcdab89;
	md5[2]=0x98badcfe;
	md5[3]=0x10325476;

	/* do all complete blocks */

	for(;length>63;data+=64,length-=64)md5block(md5,data);

	/* copy remainder to buffer */

	for(i=0;length;length--)bfr.b[i++]=*data++;

	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	bfr.b[i++]=0x80;
	if(i>56)
	{
		for(;i<64;i++)bfr.b[i]=0;
		i=0;
		md5block(md5,bfr.b);
	}
	for(;i<56;i++)bfr.b[i]=0;

	/* append bits 0 through 31 of total length in bits to buffer */

	bfr.b[56]=(WORD08)((total)<<3);
	bfr.b[57]=(WORD08)((total)>>5);
	bfr.b[58]=(WORD08)((total)>>13);
	bfr.b[59]=(WORD08)((total)>>21);

	/* append bits 32 through 63 of total length in bits to buffer */

	bfr.b[60]=(WORD08)(BITS32(total)>>29);
	bfr.b[61]=0;
	bfr.b[62]=0;
	bfr.b[63]=0;

	/* hash buffer */

	md5block(md5,bfr.b);

	/* return final hash result to caller */

	result[ 0]=(WORD08) (md5[0]);
	result[ 1]=(WORD08)((md5[0])>>8);
	result[ 2]=(WORD08)((md5[0])>>16);
	result[ 3]=(WORD08)((md5[0])>>24);
	result[ 4]=(WORD08) (md5[1]);
	result[ 5]=(WORD08)((md5[1])>>8);
	result[ 6]=(WORD08)((md5[1])>>16);
	result[ 7]=(WORD08)((md5[1])>>24);
	result[ 8]=(WORD08) (md5[2]);
	result[ 9]=(WORD08)((md5[2])>>8);
	result[10]=(WORD08)((md5[2])>>16);
	result[11]=(WORD08)((md5[2])>>24);
	result[12]=(WORD08) (md5[3]);
	result[13]=(WORD08)((md5[3])>>8);
	result[14]=(WORD08)((md5[3])>>16);
	result[15]=(WORD08)((md5[3])>>24);

	/* assert user data removal data from buffer */

#ifdef MD5_PARANOID
	bfr.l[0]=0;
	bfr.l[1]=0;
	bfr.l[2]=0;
	bfr.l[3]=0;
	bfr.l[4]=0;
	bfr.l[5]=0;
	bfr.l[6]=0;
	bfr.l[7]=0;
	bfr.l[8]=0;
	bfr.l[9]=0;
	bfr.l[10]=0;
	bfr.l[11]=0;
	bfr.l[12]=0;
	bfr.l[13]=0;
	bfr.l[14]=0;
	bfr.l[15]=0;
#endif

}

#endif

/*
 * procedure md5hmkey
 *
 * input:  key    - the HMAC key
 *	   length - the length of the HMAC key
 *
 * output: ptr - the preprocessed HMAC key hashes
 *
 * This procedure processes a key for HMAC use.
 */

#ifdef MD5_HMAC

void CRYPTOCALL md5hmkey(WORD08 *key,WORD32 keylength,MD5HMDATA *ptr)
{
	register WORD32 i;		/* counter and index	*/
	MD5(hash);			/* hash of key		*/
	union
	{
		WORD08 pad[64];		/* padding buffer	*/
		MD5DATA md5data;	/* MD5 internal data	*/
	}u;


	/* if the key is larger than 64 bytes hash it and use the
	   hash as the key */

	if(keylength>64)
	{
		md5init(&u.md5data);
		md5next(key,keylength,&u.md5data);
		md5end(hash,&u.md5data);
		key=hash;
		keylength=MD5_SIZE;
		u.md5data.md5[0]=u.md5data.md5[1]=u.md5data.md5[2]=
			u.md5data.md5[3]=0;
	}

	/* copy initial values to HMAC internal data structure */

	ptr->imd5[0]=ptr->omd5[0]=0x67452301;
	ptr->imd5[1]=ptr->omd5[1]=0xefcdab89;
	ptr->imd5[2]=ptr->omd5[2]=0x98badcfe;
	ptr->imd5[3]=ptr->omd5[3]=0x10325476;

	/* copy key to padding buffer, fill up buffer
	   and pre-hash inner pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x36;
	for(;i<64;i++)u.pad[i]=0x36;
	md5block(ptr->imd5,u.pad);

	/* copy key to padding buffer, fill up buffer
	   and pre-hash outer pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x5c;
	for(;i<64;i++)u.pad[i]=0x5c;
	md5block(ptr->omd5,u.pad);

	/* clean up */

	for(i=0;i<MD5_SIZE;i++)hash[i]=0;
	for(i=0;i<64;i++)u.pad[i]=0;
}

#endif

/*
 * procedure md5hminit
 *
 * input:  key - the preprocessed HMAC key
 *
 * output: ptr - the MD5 internal data
 *
 * This procedure initializes MD5 data for HMAC use.
 */

#ifdef MD5_HMAC_FULL

void CRYPTOCALL md5hminit(MD5DATA *ptr,MD5HMDATA *key)
{
	ptr->total=64;
	ptr->size=0;
	ptr->md5[0]=key->imd5[0];
	ptr->md5[1]=key->imd5[1];
	ptr->md5[2]=key->imd5[2];
	ptr->md5[3]=key->imd5[3];
}

#endif

/*
 * procedure md5hmend
 *
 * input:  key - the preprocessed HMAC key
 *	   ptr - the MD5 internal data
 *
 * output: result - the HMAC result (a MD5 hash)
 *
 * This procedure finalizes and returns the HMAC.
 */

#ifdef MD5_HMAC_FULL

void CRYPTOCALL md5hmend(WORD08 *result,MD5DATA *ptr,MD5HMDATA *key)
{
	/* complete inner MD5 */

	md5end(result,ptr);

	/* initialize outer MD5 */

	ptr->total=64;
	ptr->size=0;
	ptr->md5[0]=key->omd5[0];
	ptr->md5[1]=key->omd5[1];
	ptr->md5[2]=key->omd5[2];
	ptr->md5[3]=key->omd5[3];

	md5next(result,MD5_SIZE,ptr);
	md5end(result,ptr);
}

#endif

/*
 * procedure md5hmac
 *
 * input:  data   - the data to be processed
 *	   length - the length of the data to be processed
 *	   key    - the preprocessed HMAC key
 *
 * output: result - the HMAC result (a MD5 hash)
 *
 * This procedure creates and returns a HMAC.
 */

#ifdef MD5_HMAC_FAST

void CRYPTOCALL md5hmac(WORD08 *data,WORD32 length,WORD08 *result,
	MD5HMDATA *key)
{
	MD5DATA md5data;	/* MD5 internal data */


	/* initialize inner MD5 */

	md5data.total=64;
	md5data.size=0;
	md5data.md5[0]=key->imd5[0];
	md5data.md5[1]=key->imd5[1];
	md5data.md5[2]=key->imd5[2];
	md5data.md5[3]=key->imd5[3];

	/* hash given data */

	md5next(data,length,&md5data);

	/* complete inner MD5 */

	md5end(result,&md5data);

	/* initialize outer MD5 */

	md5data.total=64;
	md5data.size=0;
	md5data.md5[0]=key->omd5[0];
	md5data.md5[1]=key->omd5[1];
	md5data.md5[2]=key->omd5[2];
	md5data.md5[3]=key->omd5[3];

	/* complete and return outer MD5 */

	md5next(result,MD5_SIZE,&md5data);
	md5end(result,&md5data);
}

#endif

