/*****************************************************************************
 * grab-v4l2.c: v4l2 interface
 *****************************************************************************
 * $Id: grab-v4l2.c,v 1.27 2004/09/25 20:18:15 alainjj Exp $
 *****************************************************************************
 * Copyright (C) 2003 Alainjj
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * for the moment just the skeleton
 * very inspired from
 * xawtv/libng/plugins/drv0-v4l2.c
 *
*****************************************************************************/

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include "config.h"
#include "strtab.h"
#include "grab.h"
#ifndef HAVE_V4L
struct GRABBER grab_v4l2; /* for compatibility... */
#else
#include "videodev.h"
#include "videodev2.h"
#include "colorspace.h"
#include "memcpy.h"
#include "blackborder.h"
extern int x11_bigendian;

static int fmts_xaw_to_v4l2[MAX_VIDEO_FMT] = {
  [VIDEO_GRAY8]  = V4L2_PIX_FMT_GREY,
  [VIDEO_RGB08]  = V4L2_PIX_FMT_RGB332,
  [VIDEO_HI240]  = V4L2_PIX_FMT_HI240,
#ifdef WORDS_BIGENDIAN
  [VIDEO_RGB15]  = V4L2_PIX_FMT_RGB555X,
  [VIDEO_RGB16]  = V4L2_PIX_FMT_RGB565X,
  [VIDEO_RGB24]  = V4L2_PIX_FMT_RGB24,
  /* it should be [VIDEO_RGB32P] = V4L2_PIX_FMT_RGB32
     (see http://v4l2spec.bytesex.org/spec/pixfmt-rgb.html)
     but the drivers do not follow the specification 
     (normal the specification is not logical...)
  */
  [VIDEO_RGB32] = V4L2_PIX_FMT_RGB32,
  [VIDEO_RGB15X] = V4L2_PIX_FMT_RGB555,
  [VIDEO_RGB16X] = V4L2_PIX_FMT_RGB565,
  [VIDEO_RGB24X] = V4L2_PIX_FMT_BGR24,
  [VIDEO_RGB32X] = V4L2_PIX_FMT_BGR32,
#else
  [VIDEO_RGB15]  = V4L2_PIX_FMT_RGB555,
  [VIDEO_RGB16]  = V4L2_PIX_FMT_RGB565,
  [VIDEO_RGB24]  = V4L2_PIX_FMT_BGR24,
  [VIDEO_RGB32]  = V4L2_PIX_FMT_BGR32,
  [VIDEO_RGB15X] = V4L2_PIX_FMT_RGB555X,
  [VIDEO_RGB16X] = V4L2_PIX_FMT_RGB565X,
  [VIDEO_RGB24X] = V4L2_PIX_FMT_RGB24,
  /* idem it should be [VIDEO_RGB32PX]=V4L2_PIX_FMT_RGB32*/
  [VIDEO_RGB32X]= V4L2_PIX_FMT_RGB32,
#endif
  [VIDEO_YUYV]   = V4L2_PIX_FMT_YUYV,
  [VIDEO_UYVY]   = V4L2_PIX_FMT_UYVY,
  [VIDEO_Y41P]   = V4L2_PIX_FMT_Y41P,
  [VIDEO_YVU420] = V4L2_PIX_FMT_YVU420,
  [VIDEO_YUV420] = V4L2_PIX_FMT_YUV420,
  [VIDEO_YVU410] = V4L2_PIX_FMT_YVU410,
  [VIDEO_YUV410] = V4L2_PIX_FMT_YUV410,
  [VIDEO_YUV422P]= V4L2_PIX_FMT_YUV422P,
  [VIDEO_YUV411P]= V4L2_PIX_FMT_YUV411P,
  [VIDEO_NV12]   = V4L2_PIX_FMT_NV12,
};

/* prototypes */
static int grab_audio (int mute, int volume, int *mode);
static int is525(int norm);
static int issecam(int norm);
static int stop_capture(void);
static int start_capture(void);
static int changeformat(struct v4l2_format *);
static void* get_buf(int i);

/* pass 0/1 by reference */
static int one = 1, zero = 0;
static int grab = 0;
int overlay = 0;
static struct v4l2_framebuffer ov_fbuf;
static struct v4l2_clip ov_clips[32];
struct v4l2_fmtdesc fmtdesc;

static struct STRTAB device_cap[] = {
  {V4L2_CAP_VIDEO_CAPTURE, "capture"},
  {V4L2_CAP_VIDEO_OUTPUT,"output"},
  {V4L2_CAP_VIDEO_OVERLAY,"overlay"},
  {V4L2_CAP_VBI_CAPTURE,"vbi_capture"},
  {V4L2_CAP_RDS_CAPTURE,"rds"},
  {V4L2_CAP_TUNER,"tuner"},
  {V4L2_CAP_AUDIO,"audio"},
  {V4L2_CAP_READWRITE,"readwrite"},
  {V4L2_CAP_ASYNCIO,"asynci/o"},
  {V4L2_CAP_STREAMING,"streaming"},
  {-1,NULL}
};

#define MAX_INPUT   16
#define MAX_NORM    16
#define MAX_FORMAT  32

static struct {
  int  ninputs,nstds,nfmts;
  struct v4l2_capability	cap;
  struct v4l2_streamparm	streamparm;
  struct v4l2_input		inp[MAX_INPUT];
  struct v4l2_standard      	std[MAX_NORM];
  struct v4l2_fmtdesc		fmt[MAX_FORMAT];
  struct v4l2_queryctrl	brightness,contrast,hue,saturation;
  struct v4l2_queryctrl	mute,volume;
} drv;

extern struct GRABBER grab_v4l2;

extern int x11_bigendian;
extern struct GRABBER grab_dummy;
static int bttvfd = -1;
/* not static for xdtv_scantv...*/
extern int vbifd;
extern void video_overlay
(int (*set_overlay)(int,int,int,int,int,struct OVERLAY_CLIP *,int));

/* screen grab */
struct v4l2_buffer *gb2;
static int gb_grab=0,gb_sync=0;
static int imgq=0,imgw=0;
static struct v4l2_requestbuffers reqbufs;

//struct video_mbuf gb_buffers = { 2 * 0x151000, 0, {0, 0x151000} };
static char **maps = NULL;

extern int nbufs;
static struct v4l2_format gbformat;
static double dur_frame = 0.04;
extern int have_xv;

extern int owidth,oheight;
static int grab_open(struct device_t *device) 
{
  int i, ov_error=0;
  if (-1 != bttvfd)
    goto err;

  if (debug)
    fprintf (stderr, "v4l2: open\n");
  
  if (-1 == (bttvfd = open (device->video ? device->video : "/dev/video", O_RDWR)))
    {
      fprintf (stderr, "open %s: %s\n",
               device->video ? device->video : "/dev/video", strerror (errno));
      goto err;
    }
  
  if (-1 == ioctl(bttvfd,VIDIOC_QUERYCAP,&drv.cap)) 
    {
      fprintf (stderr, "driver is not v4l2\n");
      goto err;
    }

  if ((drv.cap.capabilities & V4L2_CAP_VBI_CAPTURE) && (device->vbi))
    if (-1 == (vbifd = open (device->vbi, O_RDONLY)))
      {
	fprintf (stderr, "v4l2: cant't open %s for reading, %s\n",
		 device->vbi, strerror (errno));
	//goto err;
      }

  if (debug)
    {
      fprintf (stderr, "v4l2: driver=%.16s card=%.32s\n", 
	       drv.cap.driver,drv.cap.card);
      fprintf(stderr,"v4l2: capabilities =");
      for (i = 0; device_cap[i].str != NULL; i++)
        if (drv.cap.capabilities & device_cap[i].nr)
          fprintf (stderr, " %s", device_cap[i].str);
      fprintf (stderr, "\n");
    }
  
  
  for (drv.ninputs = 0; drv.ninputs < MAX_INPUT; drv.ninputs++) {
    drv.inp[drv.ninputs].index = drv.ninputs;
    if (-1 == ioctl(bttvfd, VIDIOC_ENUMINPUT, &drv.inp[drv.ninputs]))
      break;
  }
  for (drv.nstds = 0; drv.nstds < MAX_NORM; drv.nstds++) {
    drv.std[drv.nstds].index = drv.nstds;
    if (-1 == ioctl(bttvfd, VIDIOC_ENUMSTD, &drv.std[drv.nstds]))
      break;
  }
  for (drv.nfmts = 0; drv.nfmts < MAX_FORMAT; drv.nfmts++) {
    drv.fmt[drv.nfmts].index = drv.nfmts;
    drv.fmt[drv.nfmts].type  = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    if (-1 == ioctl(bttvfd, VIDIOC_ENUM_FMT, &drv.fmt[drv.nfmts]))
      break;
  }
  
  drv.streamparm.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  ioctl(bttvfd,VIDIOC_G_PARM,&drv.streamparm);
  
  drv.brightness.id = V4L2_CID_BRIGHTNESS;
  if(-1 == ioctl(bttvfd,VIDIOC_QUERYCTRL,&drv.brightness))
    drv.brightness.id = 0;
  drv.contrast.id = V4L2_CID_CONTRAST;
  if(-1 == ioctl(bttvfd,VIDIOC_QUERYCTRL,&drv.contrast))
    drv.contrast.id = 0;
  drv.hue.id = V4L2_CID_HUE;
  if(-1 == ioctl(bttvfd,VIDIOC_QUERYCTRL,&drv.hue))
    drv.hue.id = 0;
  drv.saturation.id = V4L2_CID_SATURATION;
  if(-1 == ioctl(bttvfd,VIDIOC_QUERYCTRL,&drv.saturation))
    drv.saturation.id = 0;
  drv.mute.id = V4L2_CID_AUDIO_MUTE;
  if(-1 == ioctl(bttvfd,VIDIOC_QUERYCTRL,&drv.mute))
    drv.mute.id = 0;
  drv.volume.id = V4L2_CID_AUDIO_VOLUME;
  if(-1 == ioctl(bttvfd,VIDIOC_QUERYCTRL,&drv.volume) ||
     drv.volume.minimum==drv.volume.maximum)
    drv.volume.id = 0;
  //exit(0);

  grab_v4l2.norms=malloc(sizeof(struct STRTAB) * (drv.nstds+1));
  for (i = 0; i < drv.nstds; i++) {
    grab_v4l2.norms[i].nr  = i;
    grab_v4l2.norms[i].str = drv.std[i].name;
  }
  grab_v4l2.norms[i].nr  = -1;
  grab_v4l2.norms[i].str = NULL;
  if(debug) {
    printf("norms=");
    for(i=0;i<drv.nstds;i++) printf(" %s",grab_v4l2.norms[i].str);
    printf("\n");
  }

  grab_v4l2.inputs=malloc(sizeof(struct STRTAB) * (drv.ninputs+1));
  for (i = 0; i < drv.ninputs; i++) {
    grab_v4l2.inputs[i].nr  = i;
    grab_v4l2.inputs[i].str = drv.inp[i].name;
  }
  grab_v4l2.inputs[i].nr  = -1;
  grab_v4l2.inputs[i].str = NULL;
  if(debug) {
    printf("inputs=");
    for(i=0;i<drv.ninputs;i++) printf(" %s",grab_v4l2.inputs[i].str);
    printf("\n");
  }

  /* frame buffer */
  if (-1 == ioctl (bttvfd, VIDIOC_G_FBUF, &ov_fbuf)) {
    perror ("ioctl VIDIOC_G_FBUF");
    fprintf(stderr, "classical overlay is disabled");
    ov_error=1;
  } else {
    if (debug)
      fprintf (stderr, "  fbuffer : size=%dx%d bpl=%d\n",
	       ov_fbuf.fmt.width, ov_fbuf.fmt.height,
	       ov_fbuf.fmt.bytesperline);
    if(ov_fbuf.base==NULL) { 
      fprintf(stderr,
	      "WARNING: video memory base unknown, may be caused by a problem\n"
	      "  with xdtv_v4l-conf or a non-availability of DGA\n"
	      "  and frame buffer devices: CLASSICAL OVERLAY IS DISABLED !\n");
      ov_error=1;
    }
  }
  if(ov_error)
    grab_v4l2.grab_overlay = NULL; //to force capture mode...

  gbformat.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  gbformat.fmt.pix.field = V4L2_FIELD_ANY;
  //gbformat.fmt.pix.field = V4L2_FIELD_INTERLACED;

  /* with V4L2_FIELD_ANY I must do a big S_FMT at first...*/
  if(grab_format)
    gbformat.fmt.pix.pixelformat = fmts_xaw_to_v4l2[grab_format];
  else
    gbformat.fmt.pix.pixelformat = drv.fmt[0].pixelformat;
  gbformat.fmt.pix.width = 768;
  gbformat.fmt.pix.height = 576;
  if(-1 == ioctl(bttvfd,VIDIOC_S_FMT, &gbformat))
    perror("ioctl VIDIOC_S_FMT");
  gbformat.fmt.pix.pixelformat = -1; // to force changeformat...

  return 0;
  
 err:
  if (bttvfd != -1) {
      close (bttvfd);
      bttvfd = -1;
  }
  if (vbifd != -1)  {
    close (vbifd);
    vbifd = -1;
  }
  if(gb2!=NULL) free(gb2);
  if(maps!=NULL) free(maps);
  return -1;
}


static int
grab_close ()
{
  if (-1 == bttvfd)
    return 0;
  grab_audio (1, -1, NULL);
  if(grab) stop_capture();
  close(bttvfd);
  bttvfd = -1;
  close(vbifd);
  vbifd = -1;
  free(grab_v4l2.norms); grab_v4l2.norms = NULL;
  free(grab_v4l2.inputs); grab_v4l2.inputs = NULL;
  return 0;
}


static int
grab_overlay (int x, int y, int width, int height, int format,
              struct OVERLAY_CLIP *oc, int count)
{
  struct v4l2_format win;
  int i;

  if(bttvfd == -1) return -1;

  if(grab) stop_capture();
  if (width == 0 || height == 0)
    {
      if (debug)
        fprintf (stderr, "v4l2: overlay off\n");
      if( -1 == ioctl (bttvfd, VIDIOC_OVERLAY, &zero))
	perror("ioctl VIDIOC_OVERLAY");
      overlay = 0;
      changeformat(&gbformat);
      return 0;
    }
  memset(&win,0,sizeof(win));
  win.type = V4L2_BUF_TYPE_VIDEO_OVERLAY;
  win.fmt.win.w.left    = x;
  win.fmt.win.w.top     = y;
  win.fmt.win.w.width   = width;
  win.fmt.win.w.height  = height;
  
  if (drv.cap.capabilities & V4L2_FBUF_CAP_LIST_CLIPPING) {
    win.fmt.win.clips      = ov_clips;
    win.fmt.win.clipcount  = count;
    
    for (i = 0; i < count; i++) {
      ov_clips[i].next = (i+1 == count) ? NULL : &ov_clips[i+1];
      ov_clips[i].c.left = oc[i].x1;
      ov_clips[i].c.top = oc[i].y1;
      ov_clips[i].c.width = oc[i].x2 - oc[i].x1 /* XXX */ ;
      ov_clips[i].c.height = oc[i].y2 - oc[i].y1;
      if (debug)
	fprintf (stderr, "v4l2: clip=%dx%d+%d+%d\n",
		 ov_clips[i].c.width, ov_clips[i].c.height,
		 ov_clips[i].c.left, ov_clips[i].c.top);
	}
  }
  
  if(-1 == ioctl(bttvfd,VIDIOC_S_FMT, &win)) {
    perror("ioctl VIDIOC_S_FMT");
    return -1;
  }
  
  if(!overlay) {
    if( -1 == ioctl (bttvfd, VIDIOC_OVERLAY, &one)) {
      perror("ioctl VIDIOC_OVERLAY");
      return -1;
    }
    overlay = 1;
  }

  return 0;
}


static int
grab_queue (struct v4l2_buffer *gb)
{
  if (debug > 1)
    fprintf (stderr, "g%d", gb->index);

  if(overlay) {fprintf(stderr,"BIG PB\n");exit(1);}
  if (-1 == ioctl (bttvfd, VIDIOC_QBUF, gb))
    {
      if (errno == EAGAIN)
        fprintf (stderr, "grabber chip can't sync (no station tuned in?)\n");
      else
        fprintf (stderr, "ioctl VIDIOC_QBUF(%d): %s\n",
                 gb->index, strerror (errno));
      return -1;
    }

  gb_grab++;
  if (debug > 1)
    fprintf (stderr, "* ");

  return 0;
}

static int
grab_wait (struct v4l2_buffer  *gb)
{
  fd_set  rdset;
  struct timeval tv;
  int r;

  if (debug)
    fprintf (stderr, "s%d", gb->index);

  do {
    tv.tv_sec  = 5;
    tv.tv_usec = 0;
    FD_ZERO(&rdset);
    FD_SET(bttvfd, &rdset);
    r = select(bttvfd + 1, &rdset, NULL, NULL, &tv);
  } while(r==-1 && errno == EINTR);
  if(r==-1) {
    perror("grab_wait: select");
    return -1;
  }
  if(r==0) {
    fprintf(stderr,"grab_wait: select timeout\n");
    return -1;
  }

  if (-1 == ioctl (bttvfd, VIDIOC_DQBUF, gb))
    {
      perror ("ioctl VIDIOC_DQBUF");
      return -1;
    }
  
  gb_sync++;
  if (debug)
    fprintf (stderr, "* ");
  
  return 0;
}

static inline void reset(void) {
  fprintf(stderr,"total reset\n");
  for (imgw=0;imgw<nbufs;imgw++) grab_wait(&gb2[imgw]);
  gb_grab=gb_sync=imgq=imgw=0;
}


/* with v4l2 the buffers are not associated to a format....*/
char * get_img2(void) {
  if(imgq!=gb_grab%nbufs || imgw!=gb_sync%nbufs)
    fprintf(stderr,"PB imgq=%d gb_grab=%d imgw=%d gb_sync=%d nbufs=%d\n",
	    imgq,gb_grab,imgw,gb_sync,nbufs);
  
  while ((gb_grab-gb_sync)<nbufs) {
    if (-1 == grab_queue(&gb2[imgq]))  {
      reset();
      return NULL;
    }
    imgq++;
    if (imgq>=nbufs) imgq=0;
  }
  
  if (-1 == grab_wait(&gb2[imgw])) {
    reset();
    return NULL;
  }
  
  img=imgw;
  
  imgw++;
  if (imgw==nbufs) imgw=0;
  videostampmin = gb2[img].timestamp.tv_usec/1000000.0+
    gb2[img].timestamp.tv_sec + dur_frame;
  videostampmax = videostampmin + dur_frame;
  if(debug>=3) {
    double now;
    struct timeval tv;
    gettimeofday (&tv, NULL);
    now = tv.tv_usec/1000000.0+tv.tv_sec;
    printf("img=%d min=%.4lf max=%.4lf now=%.4lf\n",img,
	   videostampmin-(int)videostampmin,
	   videostampmax-(int)videostampmax,now-(int)now);
  }
  return maps[img];
}


static int stop_capture(void) {
  int i,type;
  while (gb_grab!=gb_sync)
    if(grab_wait(&gb2[imgw])<0) {reset(); break;}
  type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  if(-1 == ioctl(bttvfd,VIDIOC_STREAMOFF,&type)) {
    perror("ioctl VIDIOC_STREAMOFF");
    return -1;
  }
  for(i = 0; i < nbufs; i++)
    if(maps[i]!=NULL) munmap(maps[i],gb2[i].length);
  free(maps);free(gb2);
  maps=NULL; gb2=NULL;
  grab = 0;
  return 0;
}

static int start_capture(void) {
  int i,type;
  if(overlay) video_overlay (NULL);;
  gb_grab=gb_sync=imgq=imgw=0;
  reqbufs.count  = nbufs_default;
  reqbufs.type   = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  reqbufs.memory = V4L2_MEMORY_MMAP;
  if (-1 == ioctl(bttvfd, VIDIOC_REQBUFS, &reqbufs)) {
    perror("ioctl VIDIOC_REQBUFS");
    return -1;
  }
  nbufs = reqbufs.count;
  if(debug) 
    fprintf(stderr,"allocated %d kernel buffers\n",nbufs);
  gb2=(struct v4l2_buffer *) malloc(nbufs*sizeof(struct v4l2_buffer));
  maps=(char **) malloc(nbufs*sizeof(char*));
  memset(maps, 0, nbufs * sizeof(char *));
  for (i = 0; i < nbufs; i++) {
    gb2[i].index  = i;
    gb2[i].type   = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    gb2[i].memory = V4L2_MEMORY_MMAP;
    if (-1 == ioctl(bttvfd, VIDIOC_QUERYBUF, &gb2[i], 0)) {
      perror("ioctl VIDIOC_QUERYBUF");
      return -1;
    }
    maps[i] = mmap(NULL, gb2[i].length, PROT_READ | PROT_WRITE, MAP_SHARED,
		   bttvfd, gb2[i].m.offset);
    if(maps[i] == MAP_FAILED) {
      perror("mmap");
      return -1;
    }
  }
  if(-1 == ioctl(bttvfd,VIDIOC_STREAMON,&type)) {
    perror("ioctl VIDIOC_STREAMON");
    return -1;
  }
  grab = 1;
  return 0;
}

static int fmt_available(video_fmt f) {
  int i,format = fmts_xaw_to_v4l2[f];
  if(!format) return 0;
  for (i = 0; i < drv.nfmts; i++)
    if(drv.fmt[i].pixelformat==format) return 1;
  return 0;
}


static int changeformat(struct v4l2_format * fmt) {
  if(grab)
    if(-1 == stop_capture()) return -1;

  if(fmt->fmt.pix.pixelformat==-1) return -1;

  if(-1 == ioctl(bttvfd,VIDIOC_S_FMT, fmt)) {
    perror("ioctl VIDIOC_S_FMT");
    return -1;
  }
  if(!overlay)
    if(-1 == start_capture()) return -1;
    
  return 0;
}


static void *get_img (video_fmt format,int width, int height)
{
  int f=fmts_xaw_to_v4l2[format];
  int overlay2 = overlay;
  void *img;
  if(overlay2) video_overlay (NULL);
  if(gbformat.fmt.pix.width!=width || gbformat.fmt.pix.height!=height ||
     gbformat.fmt.pix.pixelformat!=f) {
    gbformat.fmt.pix.width=width;
    gbformat.fmt.pix.height=height;
    gbformat.fmt.pix.pixelformat=f;
    if(changeformat(&gbformat) == -1) return 0;
    if(gbformat.fmt.pix.width!=width ||
       gbformat.fmt.pix.height!=height) {
      fprintf(stderr, "asked for %dx%d, and I have %dx%d\n",
	      width,height,gbformat.fmt.pix.width,
	      gbformat.fmt.pix.height);
      fprintf(stderr, "try to run with -capt_width %d -capt_height %d\n",
	      gbformat.fmt.pix.width,gbformat.fmt.pix.height);
      fprintf(stderr, "      or -force_capt_width %d -force_capt_height %d\n",
	      gbformat.fmt.pix.width,gbformat.fmt.pix.height);
      exit(1);
    }
  }
  img=get_img2();
  if(overlay2) video_overlay (grab_overlay);
  return img;
}

/* ---------------------------------------------------------------------- */

static int
grab_tune (unsigned long freq)
{
  struct v4l2_frequency f;
  memset(&f,0,sizeof(f));
  f.type = V4L2_TUNER_ANALOG_TV;
  f.frequency = freq;
  if(ioctl(bttvfd, VIDIOC_S_FREQUENCY, &f) == -1)
    perror("VIDIOC_S_FREQUENCY");
  return 0;
}

static int
grab_tuned ()
{
  struct v4l2_tuner tuner;
  
  memset(&tuner,0,sizeof(tuner));
  if (-1 == ioctl(bttvfd,VIDIOC_G_TUNER,&tuner))
    return 0;
  return tuner.signal ? 1 : 0;
}

static int
grab_input (int input, int norm)
{
  static int prev_norm = -1;
  if (-1 != input) {
    if (debug)
      fprintf (stderr, "v4l2: input: %d\n", input);
    if( -1 == ioctl (bttvfd, VIDIOC_S_INPUT, &drv.inp[input].index))
      perror("ioctl VIDIOC_S_INPUT");
  }
  if (-1 != norm)
    {
      if (debug)
        fprintf (stderr, "v4l2: norm : %d\n", norm);
      if(norm != prev_norm) {
	int overlay2 = overlay;
	if(overlay2) video_overlay (NULL);
	if(grab) stop_capture();
	if( -1 == ioctl(bttvfd, VIDIOC_S_STD, &drv.std[norm].id)) {
	  perror("ioctl VIDIOC_S_STD");
	  return -1;
	}
	if(overlay2)
	  video_overlay (grab_overlay);
	else
	  start_capture();
	prev_norm = norm;
	if(is525(norm)) 
	  dur_frame = 10001.0/30000;
	else
	  dur_frame = 1.0/25;
      }
    }
  return 0;
}

static inline int control(int x,struct v4l2_queryctrl *ctrl) {
  struct v4l2_control c;
  if(x == -1) return 0;
  if(ctrl->id == 0) return -1;
  c.id = ctrl->id;
  c.value = (float)x/65535*(ctrl->maximum-ctrl->minimum)+ctrl->minimum+0.5;
  if(-1 == ioctl(bttvfd, VIDIOC_S_CTRL, &c)) {
    perror("ioctl VIDIOC_S_CTRL");
    return -1;
  }
  return 0;
}
		   
static int
grab_picture (int color, int bright, int hue, int contrast)
{
  control(color,&drv.saturation);
  control(bright,&drv.brightness);
  control(hue,&drv.hue);
  control(contrast,&drv.contrast);
  return 0;
}

static int
grab_audio (int mute, int volume, int *mode)
{
  struct v4l2_control c;
  struct v4l2_tuner tuner;
  if(mute !=-1 && drv.mute.id) {
    c.id = V4L2_CID_AUDIO_MUTE;
    c.value = mute;
    if(-1 == ioctl(bttvfd, VIDIOC_S_CTRL,&c))
      perror("VIDIOC_S_CTRL, V4L2_CID_AUDIO_MUTE");
  }
  if(volume !=-1 && drv.volume.id) {
    c.id = V4L2_CID_AUDIO_VOLUME;
    c.value = volume;
    if(-1 == ioctl(bttvfd, VIDIOC_S_CTRL,&c))
      perror("VIDIOC_S_CTRL, V4L2_CID_AUDIO_VOLUME");
  }
  if(mode) {
    memset(&tuner,0,sizeof(tuner));
    if(-1 == ioctl(bttvfd,VIDIOC_G_TUNER,&tuner))
      perror("VIDIOC_G_TUNER");
    if(*mode) {
      switch(*mode) {
      case 1: tuner.audmode = V4L2_TUNER_MODE_MONO; break;
      case 2: tuner.audmode = V4L2_TUNER_MODE_STEREO; break;
      }
      if(-1 == ioctl(bttvfd,VIDIOC_S_TUNER,&tuner))
	perror("VIDIOC_S_TUNER");
      if(-1 == ioctl(bttvfd,VIDIOC_G_TUNER,&tuner))
	perror("VIDIOC_G_TUNER");
    }
    switch(tuner.audmode) {
    case V4L2_TUNER_MODE_MONO: *mode=1; break;
    case V4L2_TUNER_MODE_STEREO: *mode=2; break;
    default: *mode=0;
    }
  }
  return 0;
}

static int is525(int norm) {
  return drv.std[norm].id & V4L2_STD_525_60;
}

static int issecam(int norm) {
  return drv.std[norm].id & V4L2_STD_SECAM;
}

static void* get_buf(int i){
  if(maps==NULL) return NULL;
  return maps[i];
}

struct GRABBER grab_v4l2 = {
  "v4l2",
  NULL, NULL,
  grab_open,
  grab_close,
  grab_overlay,
  fmt_available,
  get_img,
  grab_tune,
  grab_tuned,
  grab_input,
  grab_picture,
  grab_audio,
  is525,
  issecam,
  get_buf
};

#endif
