#! /bin/bash
#
# xdtv_record.sh
#  
# Made by (Cyriac BENOIT)
# Login   <cyriac@free-unices.org>
#
# Started on  Mon Sep 22 01:03:19 2003 Cyriac BENOIT
# Last update Wed Jan 16 11:08:32 2004 Cyriac BENOIT
#

# The only parameters to be configured...
config_file="${HOME}/.xdtv/xdtvrc"
config_dir="${HOME}/.xdtv/records"

# You shouldn't edit no further
VERSION="0.0.23"
channel=""
start_time=""
end_time=""
avi_title="default"
channel_number=""
divx_path=$(grep divx_path ${HOME}/.xdtv/xdtvrc | cut -d ' ' -f 3)
avi_dir="HOME"
others=""
expert="OFF"
quick="OFF"
shutdown="OFF"
cron="OFF"
edit=${EDITOR}

if [ ! -d ${HOME}/.xdtv ]; then
  echo "You must run and configure xdtv first!"
  exit 1
fi

if [ ! -d $config_dir ]; then
  mkdir $config_dir
fi

# Checking for needed binaries
binaries="grep crontab ls wc awk cut basename cat sed amixer killall xdtv sleep mv sudo /sbin/shutdown at chmod"
for bin in $binaries; do
  $(which $bin 1>/dev/null 2>&1)
  if [ "$?" -eq 1 ]; then
    echo "$bin not found in \$PATH. This script may not work..."
  fi
done

if [ $(bash --version | grep version | sed -e 's/.*version \(.\).*/\1/') -lt 2 ]; then
  echo "You must have bash 2 in order to run this script."
  exit 1
fi

if [ -z "$edit" ]; then
  edit=$(which vi)
  if [ -z "$edit" ]; then
    edit=":"
    echo "No \$EDITOR found and vi not present."
    echo "Press Enter."
    read
  fi
fi

# Helper function
usage () {
  echo "Syntax: $(basename $0) [-c channel] [-d date ] [-s start time]"
  echo "        [-e end time] [-D avi directory] [-t avi title] [-x | --expert]"
  echo "        [-q | --quick] [-v | --version] [-S | --shutdown] [-h | --help]"
  echo ""
  echo "        Dates are in DD.MM.YY format and times are in HH:MM format."
  echo ""
  echo "        -x will enable \"expert mode\", asking more advanced parameters."
  echo "        In expert mode, you can schedule periodical records."
  echo ""
  echo "        -q will enable \"quick mode\", asking only date, times and channel"
  echo "           (if not given on command line)."
  echo ""
  exit 1
}

# Time functions
get_time () {

  # We modify IFS in order to split user input on ':'
  oldifs="$IFS"
  IFS=$':\n'
  
  read -p "Enter $1 time (HH:MM): " tmp_hour tmp_minute
  
  # Error checking
  check_hour=$(echo -n "$tmp_hour" | wc -c)
  check_minute=$(echo -n "$tmp_minute" | wc -c)

  if [ $check_hour -ne 2 -o $check_minute -ne 2 ]; then
    echo "Wrong format. Must be HH:MM"
    IFS="$oldifs"
    get_time $1 $2
  fi

  IFS="$oldifs"

  # Setting variable
  eval "$2=$tmp_hour:$tmp_minute"
  
}

get_recording_time () {
  start_hour=${1::2}
  start_minute=${1:3}
  stop_hour=${2::2}
  stop_minute=${2:3}

  if [ $start_hour -lt 10 ]; then
    start_hour=$(echo $start_hour | tr -d '0')
  fi
  if [ $stop_hour -lt 10 ]; then
    stop_hour=$(echo $stop_hour | tr -d '0')
  fi

  start=$((start_hour * 60 + start_minute))
  stop=$((stop_hour * 60 + stop_minute))

  # Adding a whole day for special cases (23:30 - 01:15)
  if [ $stop -lt $start ]; then
    stop=$((stop + 24 * 60))
  fi

  record_time=$((stop-start))
  if [ $record_time -lt 0 ]; then
    echo "Recording time would be $record_time minutes!"
    echo "There must be an error here. Exiting..."
    exit 1
  fi

  echo "Recording time will be $record_time minutes"
  record_time=$((record_time * 60))
}

# Channel functions
get_channel () {

  # Parsing config file, looking for channel names ([name]) and stripping
  # square brackets
  channames=$(grep "^\[" $config_file | tr -d "\[\]")

  # Setting PS3 prompt (for select) and IFS as newline
  PS3="Select channel: "
  oldifs="$IFS"
  IFS=$'\n'
  select channel in $channames; do
    echo $channel
    break
  done
  channel_number=$((REPLY - 1))

  # Restoring IFS
  IFS="$oldifs"
}

get_channel_name () {
  channel=$1
  grep "^\[" $config_file | awk 'NR == '$channel' { print }' | tr -d "\[\]"
}

# Menu functions

# Delete
delete_job () {
  read -p "Do you want to delete recording job $1 ? [N/y] " answer
  if [ "$answer" = "y" -o "$answer" = "Y" ]; then
    echo $job_to_delete | grep "-" >/dev/null 2>&1
    check=$?
    if [ $check -eq 1 ]; then
      # At job
      # Removing script file
      script=$(grep -H "###JOB### $job_to_delete" $config_dir/2* | cut -d':' -f1)
      rm -f $script

      # Call to atrm
      atrm $1
    else
      script="$config_dir/cronjob_$job_to_delete"
      rm -f $script

      # Canceling job
      (crontab -l | grep -v "uniq_record: $job_to_delete" | crontab -)
    fi
    
    echo "Job deleted. Press Enter..."
    read
  fi
}

# List
list_records () {
  clear
  if [ $(ls $config_dir/2* 2>/dev/null | wc -l) -ne 0 ]; then

    # We get all informations from script files
    for script in $config_dir/2*; do
      infos=$(grep "###INFOS###" $script)
      date=$(echo $infos | awk '{ print $2 }')
      year=$(echo $date | cut -c 1-4)
      month=$(echo $date | cut -c 5-6)
      day=$(echo $date | cut -c 7-8)
      date="${day}.${month}.${year}"
      channel=$(echo $infos | awk '{ print $3 }')
      channel=$(get_channel_name $((channel + 1)))
      time=$(echo $infos | awk '{ print $4 }')
      job=$(grep "###JOB###" $script | awk '{ print $2 }')
      echo "$job) $date on $channel at $time"
    done
  else

    # Or not
    echo "No records scheduled."
  fi
  echo ""
  read -p "$@" job_to_delete
  channel=""
  return $job_to_delete
}

list_cron_records () {
  clear
  if [ $(ls $config_dir/cronjob_* 2>/dev/null | wc -l) -ne 0 ]; then
    i=1
    
    # We get all informations from script files
    for script in $config_dir/cronjob_*; do
      infos=$(grep "###CRONJOB###" $script | cut -d' ' -f2-)
      jobid=$(grep "###JOB###" $script | cut -d' ' -f2)
      echo "$i) $infos - Id: $jobid"
      i=$((i + 1))
    done
  else

    # Or not
    echo "No records scheduled."
  fi
  echo ""
  read -p "$@" job
  jobs=($(ls $config_dir/cronjob_* 2>/dev/null))
  job_to_delete=$(echo ${jobs[$job - 1]} | cut -d'_' -f2)
}

# Create xdtv parameters line
make_command_line () {

  # Forcing grabdisplay in order to record something...
  parameters="-m -capture grabdisplay"
  
  if [ -n "$codec" ]; then
    parameters="${parameters} -rec_vcodec $codec"
  fi

  if [ -n "$size" ]; then
    parameters="${parameters} -rec_width $width -rec_height $height"
  fi

  if [ -n "$sub" ]; then
    parameters="${parameters} -rec_sub 1 -novbi"
  fi

  if [ -n "$quality" ]; then
    parameters="${parameters} -rec_mp3vbr $quality"
  fi

  if [ -n "$bitrate" ]; then
    parameters="${parameters} -rec_mp3cbr $bitrate"
  fi

  if [ -n "$nbufs" ]; then
    parameters="${parameters} -nbufs $nbufs"
  fi

  if [ -n "$others" ]; then
    parameters="${parameters} $others"
  fi
}

# Options parsing
while getopts ":c:s:e:d:D:t:qhvxS-:" option; do
  if [ "$option" = "-" ]; then
    case $OPTARG in
      help) option=h ;;
      quick) option=q ;;
      expert) option=x ;;
      version) option=v ;;
      shutdown) option=S ;;
    esac
  fi
  case $option in
    c) channel=$OPTARG ;;
    s) start_time=$OPTARG ;;
    e) end_time=$OPTARG ;;
    d) rec_date=$OPTARG ;;
    D) avi_dir=$OPTARG ;;
    t) avi_title=$OPTARG ;;
    x) expert="ON" ;;
    q) quick="ON" ;;
    S) shutdown="ON" ;;
    h) usage ;;
    v) echo "$(basename $0) version $VERSION"
       exit 0
       ;;
    :) echo "Missing argument for option -$OPTARG"
       usage ;;
    ?) echo "Unknown option -$OPTARG"
       usage ;;
  esac
done

# Main menu
while [ 1 -a "$quick" = "OFF" ]; do
  clear
  echo "1) Schedule a record"
  echo "2) Schedule a record in expert mode"
  echo "3) List scheduled records"
  echo "4) Delete a recording job"
  echo "5) Edit/view job"
  echo "6) Quit"
  echo ""

  read -p "Enter your choice: [1] " choice

  case "$choice" in
    2) expert="ON"
       clear
       break ;;
    3) clear
       echo "1) List regular jobs"
       echo "2) List periodical records (cron)"
       echo ""
       read -p "Enter your choice: [1] " choice

       if [ $choice -eq 2 ]; then
        list_cron_records "Press Enter..."
       else
         list_records "Press Enter..."
       fi ;;
    4) clear
       echo "1) Delete regular jobs"
       echo "2) Delete periodical records (cron)"
       echo ""
       read -p "Enter your choice: [1] " choice

       if [ $choice -eq 2 ]; then
         list_cron_records "Enter job to delete: "
       else
        list_records "Enter job to delete: "
       fi
       if [ -n "$job_to_delete" ]; then
         delete_job $job_to_delete
       fi
       ;;
    5) clear
       echo "1) Edit regular job"
       echo "2) Edit periodical records (cron)"
       echo ""
       read -p "Enter your choice: [1] " choice

       if [ $choice -eq 2 ]; then
         list_cron_records "Enter job to edit: "
       else
         list_records "Enter job to edit: "
       fi
       if [ -n "$job_to_delete" ]; then
         if [ $choice -eq 2 ]; then
           $edit $config_dir/cronjob_$job_to_delete
         else
           script=$(grep -H "###JOB### $job_to_delete" $config_dir/2* | cut -d':' -f1)
           $edit $script
         fi
       fi
       ;;
    6) exit 0 ;;
    *) clear; break ;;
  esac
done

# In expert mode we can make a cron job
if [ "$expert" = "ON" ]; then
  echo "1) Schedule a record (one time - at)"
  echo "2) Schedule a periodical record (cron)"
  echo ""

  read -p "Enter your choice: [1] " choice

  if [ $choice -eq 2 ]; then
    cron="ON"
  fi
fi

clear

if [ "$cron" = "OFF" ]; then
  # Getting day
  if [ -z "$rec_date" ]; then
    read -p "Enter recording date (DD.MM.YY): [$(date +"%d.%m.%y")] " rec_date
    if [ -z "$rec_date" ]; then
      rec_date=$(date +"%d.%m.%y")
    fi
  fi
else
  # Getting cron time specification
  cat <<_HELP_
We use the usual cron terminology.
So, an '*' mean 'every', as in 'every minute, every hour, every month, every
day and every day of the week' (* * * * *).
You can also specify ranges (2-8 = from the second to the eigth day of the
month) and specific days as 1,3,5,28.

Finally, you can mix everything:
  */2 0-9,12,16,18,20-23 * * * /usr/local/script

  script will run every two minutes, from midnight to nine, at 12, 16, 18 and from 
  twenty to eleven, every day, every month and every day of the week.

You should really look at crontab(5) man page if you don't know what we're
talking about.

_HELP_

  read -p "Enter day of month (1-31): [*] " cron_month_day
  if [ -z "$cron_month_day" ]; then 
    cron_month_day="*"
  fi
  if [ "$cron_month_day" = "*" ]; then
    tmd="Every day "
  else
    tmd="On $cron_month_day "
  fi

  read -p "Enter month (1-12): [*] " cron_month
  if [ -z "$cron_month" ]; then
    cron_month="*"
  fi
  if [ "$cron_month" = "*" ]; then
    tm="every month "
  else
    tm="$cron_month "
  fi

  read -p "Enter day of week (0-7, Sunday is either 0 or 7): [*] " cron_week_day
  if [ -z "$cron_week_day" ]; then
    cron_week_day="*"
  fi
  if [ "$cron_week_day" = "*" ]; then
    twd="every day of the week"
  else
    twd="the ${cron_week_day}th day of the week"
  fi
  date="$tmd$tm$twd"
fi

# Getting missing hours
if [ -z "$start_time" ]; then
  get_time start start_time
fi

if [ -z "$end_time" ]; then
  get_time end end_time
fi

# Display total recording time
get_recording_time $start_time $end_time
echo ""

# Getting channel name and number
if [ -z "$channel" ]; then
  get_channel
else 
  channel_number=$((channel - 1))
  channel=$(get_channel_name $channel)
fi

echo ""

# Getting AVI directory
if [ "$avi_dir" = "HOME" -a "$quick" = "OFF" ]; then
  read -p "Enter saving directory: [${divx_path:-$HOME}] " avi_dir
fi

if [ ! -d "$avi_dir" -o -z "$avi_dir" -o "$avi_dir" = "HOME" ]; then
  avi_dir=${divx_path:-$HOME}
fi

if [ ! -d "$avi_dir" ]; then
  avi_dir=${HOME}
fi

# Getting AVI title
if [ "$avi_title" = "default" -a "$quick" = "OFF" ]; then
  read -p "Enter movie title: [${channel}-${start_time}-${end_time}] " avi_title
  avi_title="${avi_title}"
fi

if [ "$avi_title" = "default" -o -z "$avi_title" ]; then
  avi_title="${channel}-${start_time}-${end_time}"
fi

echo ""

# If we are in expert mode, we ask more questions
if [ "$expert" = "ON" ]; then

  # Getting codec - We do not verify that all codecs are supported !!
  codecs=(xvid divx4linux ffmpeg_mpeg1 ffmpeg_mpeg4)
  i=1
  for name in ${codecs[*]}; do
    echo "$i) $name"
    i=$((i + 1))
  done
  read -p "Enter codec number: [1) xvid] " codec_number

  if [ -z "$codec_number" ]; then
    codec="xvid"
  else
    codec=${codecs[$codec_number - 1]}
  fi

  echo ""

  # Getting resolution
  sizes=(384x288 480x360 576x432 768x576)
  i=1
  for size in ${sizes[*]}; do
    echo "$i) $size"
    i=$((i + 1))
  done
  echo "$i) Other"
  read -p "Enter resolution: [1) 384x288] " size

  if [ -z "$size" ]; then
    size="384x288"
  elif [ $size -eq $i ]; then
    read -p "Enter your choice: " size
  else
    size=${sizes[$size - 1]}
  fi
  
  # Get width and height from size
  width=${size/x*}
  height=${size/#*x}

  echo ""
  read -p "Do you wanna record subtitles ? [N/y] " sub
  
  echo ""
  read -p "Modify sound parameters ? [N/y] " answer
  if [ "$answer" = "y" -o "$answer" = "Y" ]; then
    read -p "Constant or variable bitrate ? [C/v] " answer
    if [ "$answer" = "v" -o "$answer" = "V" ]; then
      read -p "Enter quality (0=best, 9=worst): [8] " quality
      if [ -z "$quality" ]; then
        quality=8
      fi
    else
      read -p "Enter bitrate (16-512): [128] " bitrate
      if [ -z "$bitrate" ]; then
        bitrate=128
      fi
    fi
  fi

  echo ""
  read -p "Enter number of kernel-buffered frames: [3] " nbufs
  if [ -z "$nbufs" ]; then
    nbufs=3
  fi

  echo ""
  read -p "Enter other options (for plugins for example): " others

  echo ""
  read -p "Do you wanna mute sound (for night recording) ? [Y/n] " sound

# End of expert mode
fi

if [ "$shutdown" = "OFF" ]; then
  read -p "Do you wanna shut your computer down after recording ? [N/y] " shutdown
  if [ -z "$shutdown" ]; then
    shutdown="OFF"
  fi
fi

make_command_line

if [ "$cron" = "OFF" ]; then
  date=$(echo $rec_date | sed -e 's/\(..\).\(..\).\(..\)/20\3\2\1/')
fi

if [ "$cron" = "OFF" ]; then
  file="$config_dir/${date}_${channel_number}_${start_time/:/}"
  echo "###INFOS### $date $channel_number $start_time" > $file
else
  # Hopefully unique...
  uniq="$$-$RANDOM$RANDOM$RANDOM"
  file="$config_dir/cronjob_$uniq"
  echo -e "#! /bin/bash\n" > $file
  echo "###CRONJOB### $date on ${channel} ($start_time - $end_time)" >> $file
  echo -e "\nexport PATH=/bin:/usr/bin:/usr/local/bin" >> $file
  date=""
fi

# We need XdTV filename to rename it
#default_avi="movie-${channel// /\\ }-${date}-${start_time/:/}??-1.avi"

# Creating script
cat >> $file <<_END_

if [ -z "\$DISPLAY" ]; then
  export DISPLAY=":0.0"
fi

cd $avi_dir

default_avi="movie-${channel// /\\ }-\$(date +%Y%m%d)-${start_time/:/}??-?.avi"

_END_

if [ -z "$sound" -o "$sound" = "y" -o "$sound" = "Y" ]; then
  echo "amixer set Master mute >/dev/null 2>&1" >> $file
fi

cat >> $file <<_END_
echo "Will kill all running xdtv !"
killall xdtv 2>/dev/null
xdtv $parameters -e "setstation $channel_number; record" &
sleep $record_time
xdtv_cmd "record; quit"
sleep 1
amixer set Line mute >/dev/null 2>&1

# Just to be sure...
killall xdtv 2>/dev/null

i=1
for file in \$(ls \$default_avi)
  do
    mv ${divx_path:-$avi_dir}/\$file "${avi_title}-\$i.avi"
    i=\$((i + 1))
  done
_END_

if [ -z "$sound" -o "$sound" = "y" -o "$sound" = "Y" ]; then
  echo "amixer set Master unmute >/dev/null 2>&1" >> $file
fi

if [ "$cron" = "OFF" ]; then
  echo "rm -f $file" >> $file
fi

if [ "$shutdown" != "OFF" ]; then
  echo "sudo shutdown -h now" >> $file
fi

# Scheduling
if [ "$cron" = "OFF" ]; then
  job=$(at -m -f $file $start_time $rec_date 2>&1 | grep job | cut -d' ' -f2)
  echo -e "\n###JOB### $job\n" >> $file
else
  (crontab -l; echo "$start_minute $start_hour $cron_month_day $cron_month $cron_week_day $file #uniq_record: $uniq") | crontab -
  echo -e "\n###JOB### $uniq\n" >> $file
  chmod +x $file
fi

# Display informations
echo -n "Will record $avi_title on channel $channel_number ($channel) from $start_time to $end_time in $avi_dir"
if [ "$cron" = "OFF" ]; then
  echo ", the $rec_date"
else
  echo ""
fi
echo "Script in $file"
echo ""

