/*
 * xmms-mad - mp3 plugin for xmms
 * Copyright (C) 2001-2002 Sam Clegg - See COPYING
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $superduper: input.c,v 1.8 2003/09/24 18:01:50 sam Exp $
 */
#include "config.h"

#ifdef HAVE_ASSERT_H
#include <assert.h>
#endif /* HAVE_ASSERT_H */

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif /* HAVE_SYS_SOCKET_H */

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif /* HAVE_NETINET_IN_H */

#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif /* HAVE_ARPA_INET_H */

#ifdef HAVE_NETDB_H 
#include <netdb.h>
#endif /* HAVE_NETDB_H */

#ifdef HAVE_SYS_STAT_H 
#include <sys/stat.h>
#endif /* HAVE_SYS_STAT_H */

#ifdef HAVE_SYS_TIME_H 
#include <sys/time.h>
#endif /* HAVE_SYS_TIME_H */

#include <fcntl.h>
#include <errno.h>

#include "input.h"

#define DIR_SEPARATOR '/'
#define HEADER_SIZE 256
#define LINE_LENGTH 256

/*#define DEBUG 1*/

extern gboolean scan_file (struct mad_info_t *info, gboolean fast);

struct streamdata_t
{
  char *host;
  int port;

  struct hostent *hostinfo;
  struct sockaddr_in address;
  int sock;
  int udpsock;
  int udpport;

  char *f_buffer;
  char *b_buffer;
  int f_buffer_start;
  int f_buffer_end;
  int b_buffer_end;
  char *name;
  char *notice1;
  char *notice2;
  char *genre;
  char *url;
  int metaint;
  int bitrate;
};

struct streamdata_t *
streamdata_new ()
{
  struct streamdata_t * data = (struct streamdata_t *) g_malloc (sizeof (struct streamdata_t));
  memset (data, 0, sizeof (struct streamdata_t));
  data->port = 80;
  /* malloc stream buffer */
  data->b_buffer = (char *) g_malloc (HTTP_BUFFER_SIZE);
  data->f_buffer = (char *) g_malloc (HTTP_BUFFER_SIZE);
  return data;
}

void
streamdata_free (struct streamdata_t *sdata)
{
  shutdown(sdata->sock, SHUT_RDWR);
  g_free (sdata->b_buffer);
  g_free (sdata->f_buffer);
  g_free (sdata);
}

static void
input_parse_url (struct mad_info_t *info)
{
  char *file;
  char *port;
  char *host = info->url;

  info->filename = 0;
  info->remote = 0;
  
  if (!strncasecmp ("http://", host, 7))
    {
      info->remote = 1;
      /* skip leading "http://" */
      host += 7;
    }
  else
    {
      info->filename = g_strdup(info->url);
      return;
    }

  /* parse host:port/filename */
  if ((port = strchr (host, ':')))
    {
      *port = '\0';
      port++;
      if ((file = strchr (port, '/')))
        {
          info->filename = g_strdup (file);
          *file = '\0';
          file++;
        }
      info->sdata->port = atoi (port);
    }
  else
    {
      if ((file = strchr (host, '/')))
        {
          *file = '\0';
          file++;
          info->filename = g_strdup (file);
        }
    }
  info->sdata->host = g_strdup (host);
  /* 
   * If we didn't find a filename just use '/'
   * This allows us to access a streaming servers root file.
   */
  if (!info->filename)
    {
      info->filename = g_strdup ("/");
    }
}

/**
 * Read a line from the network socket.
 */
int
input_http_readline (struct mad_info_t *madinfo, char *buffer,
                     int buffer_size)
{
  int i;
  assert (madinfo && madinfo->sdata);
  for (i = 0; i < buffer_size; i++)
    {
      read (madinfo->sdata->sock, buffer + i, 1);
      if (buffer[i] == '\n')
        break;
      if (buffer[i] == '\r')
        i--;
    }
  buffer[i] = '\0';
  return i;
}

/**
 * Open the UDP title streaming socket.
 */
gboolean
input_udp_init (struct mad_info_t *madinfo)
{
  struct sockaddr_in sin;
  int sinlen = sizeof (struct sockaddr_in);

  struct streamdata_t * sdata = madinfo->sdata;
  sdata->udpsock = socket (AF_INET, SOCK_DGRAM, 0);

  if (sdata->udpsock == -1)
    {
      xmmsmad_error ("failed to init UDP socket");
      return FALSE;
    }

  memset (&sin, 0, sinlen);
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = htonl (INADDR_ANY);

  if (bind (sdata->udpsock, (struct sockaddr *) &sin, sinlen))
    {
      xmmsmad_error ("failed to bind to a UDP port");
      return FALSE;
    }
  fcntl (sdata->udpsock, F_SETFL, O_NONBLOCK);
  memset (&sin, 0, sinlen);
  if (getsockname (sdata->udpsock, (struct sockaddr *) &sin, &sinlen) < 0)
    {
      xmmsmad_error ("failed to retrieve socket info: %s", strerror (errno));
      close (sdata->udpsock);
      return FALSE;
    }

  sdata->udpport = ntohs (sin.sin_port);
  /* printf ("Listening on local %s:%d\n", inet_ntoa (sin.sin_addr), sdata->udpport); */

  return TRUE;
}

/**
 * Update title from UDP socket
 */
int
input_udp_read (struct mad_info_t *info)
{
  char buffer[LINE_LENGTH];
  int len;
  assert (info && info->sdata);
  len = recv (info->sdata->udpsock, buffer, LINE_LENGTH, 0);
  /* read (madinfo->connection->udpsock, buffer, LINE_LENGTH); */
  if (len > 0)
    {
      printf ("Got some UDP:\n");
      buffer[len] = '\0';
      printf ("%s\n", buffer);
    }
  return 0;
}

/**
 * Open the http connection and the UDP title stream connection
 */
static int
input_http_init (struct mad_info_t *info)
{
  char header[HEADER_SIZE];
  char buffer[LINE_LENGTH];

  info->sdata->sock = socket (AF_INET, SOCK_STREAM, 0);
  info->sdata->address.sin_family = AF_INET;

  if (!(info->sdata->hostinfo = gethostbyname (info->sdata->host)))
    {
      xmmsmad_error ("failed to look up host: %s", info->sdata->host);
      return -1;
    }

  memcpy (&info->sdata->address.sin_addr.s_addr,
          *(info->sdata->hostinfo->h_addr_list),
          sizeof (info->sdata->address.sin_addr.s_addr));
  info->sdata->address.sin_port = htons (info->sdata->port);

  if (connect (info->sdata->sock, (struct sockaddr *)
               &info->sdata->address, sizeof (struct sockaddr_in)) == -1)
    {
      if (errno != EINPROGRESS)
        {
          xmmsmad_error ("couldn't connect to host %s", info->sdata->host);
          return -1;
        }
    }
  snprintf (header, HEADER_SIZE - 1, "GET %s HTTP/1.0\r\n"
            "Host: %s\r\n"
            "User-Agent: %s/%s\r\n"
            "x-audiocast-udpport: %d\r\n"
            "\r\n", info->filename, info->sdata->host, PACKAGE, "0.1", info->sdata->udpport);

  /* "Icy-MetaData:1\r\n" */
  write (info->sdata->sock, header, strlen (header));

  input_http_readline (info, buffer, LINE_LENGTH);
  if (strncmp (buffer, "ICY 200", 7))
    {
      xmmsmad_error ("server said: %s", buffer);
      return -1;
    }

  while (input_http_readline (info, buffer, LINE_LENGTH) > 0)
    {
      if (!strncmp (buffer, "icy-name:", 9))
        info->sdata->name = g_strdup (buffer + 9);
      else if (!strncmp (buffer, "x-audiocast-name:", 17))
        info->sdata->name = g_strdup (buffer + 17);
      else if (!strncmp (buffer, "icy-metaint:", 12))
        info->sdata->metaint = atoi (buffer + 12);
      else if (!strncmp (buffer, "icy-url:", 8))
        info->sdata->url = g_strdup (buffer + 8);
      else if (!strncmp (buffer, "icy-genre:", 10))
        info->sdata->genre = g_strdup (buffer + 10);
      else if (!strncmp (buffer, "icy-notice1:", 12))
        info->sdata->notice1 = g_strdup (buffer + 12);
      else if (!strncmp (buffer, "icy-notice2:", 12))
        info->sdata->notice2 = g_strdup (buffer + 12);
      else if (!strncmp (buffer, "icy-br:", 7))
        info->sdata->bitrate = atoi (buffer + 7);
      else if (!strncmp (buffer, "icy-pub:", 8))
        {
        }                       /* nothing */
      else
        printf ("unknown header: %s\n", buffer);
    }

  info->title = g_strdup (info->sdata->name);
  info->bitrate = info->sdata->bitrate;

  fcntl (info->sdata->sock, F_SETFL, O_NONBLOCK);
  return 0;
}

/**
 * init the mad_info_t struct.
 */
gboolean
input_init (struct mad_info_t *info, const char *url)
{
#ifdef DEBUG
  printf ("f: input_init\n");
#endif
  memset (info, 0, sizeof (struct mad_info_t));

  info->fmt           = FMT_S16_LE;
  info->channels      = -1;
  info->mpeg_layer    = -1;
  info->size          = -1;
  info->freq          = -1;
  info->seek          = -1;
  info->duration      = mad_timer_zero;
  info->pos           = mad_timer_zero;
  info->url           = g_strdup (url);
  info->current_frame = 0;
  info->frames        = 0;
  info->bitrate       = 0;
  info->vbr           = 0;
  info->mode          = 0;
  info->title         = 0;
  info->sdata         = streamdata_new ();

  input_parse_url (info);

  if (!info->remote)
    {
      struct stat buf;
      info->infile = open (info->filename, O_RDONLY);
      if (info->infile == -1)
        {
          /* can't show error here.. why? */
          /* xmmsmad_error ("failed to open: %s", info->filename); */
          return FALSE;
        }
      if (fstat (info->infile, &buf) == -1)
        {
          /* can't show error here.. why? */
          /* xmmsmad_error ("failed to stat: %s", info->filename); */
          return FALSE;
        }
      info->size = buf.st_size;
    }
#ifdef DEBUG
  printf ("e: input_init\n");
#endif
  return TRUE;
}


gchar* 
input_id3_get_string (struct id3_tag *tag, char *frame_name)
{
  gchar* rtn;
  const id3_ucs4_t *string;
  struct id3_frame *frame;
  union id3_field *field;

  frame = id3_tag_findframe (tag, frame_name, 0);
  if (!frame) return NULL;

  if (frame_name == ID3_FRAME_COMMENT)
      field = id3_frame_field (frame, 3);
  else
      field = id3_frame_field (frame, 1);

  if (!field) return NULL;

  if (frame_name == ID3_FRAME_COMMENT)
      string = id3_field_getfullstring (field); 
  else
      string = id3_field_getstrings (field, 0); 

  if (!string) return NULL;

  if (frame_name == ID3_FRAME_GENRE) 
     string = id3_genre_name (string);

  rtn = id3_ucs4_latin1duplicate (string);
  return rtn;
}

/**
 * read the ID3 tag 
 */
static void 
input_read_tag (struct mad_info_t *info)
{
  gchar *string;
  TitleInput *title_input;
  XMMS_NEW_TITLEINPUT (title_input);

  info->id3file = id3_file_open (info->filename, ID3_FILE_MODE_READONLY);
  if (!info->id3file) return;

  info->tag = id3_file_tag (info->id3file);
  if (!info->tag) return;

  title_input->performer = input_id3_get_string (info->tag, ID3_FRAME_ARTIST);
  title_input->track_name = input_id3_get_string (info->tag, ID3_FRAME_TITLE);
  title_input->album_name = input_id3_get_string (info->tag, ID3_FRAME_ALBUM);
  title_input->genre = input_id3_get_string (info->tag, ID3_FRAME_GENRE);
  title_input->comment = input_id3_get_string (info->tag, ID3_FRAME_COMMENT);
  string = input_id3_get_string (info->tag, ID3_FRAME_TRACK);
  if (string) 
    {
      title_input->track_number = atoi (string);
      g_free (string);
    }

  info->title = xmms_get_titlestring (xmms_get_gentitle_format (), title_input);
  if (title_input->track_name)   g_free (title_input->track_name);
  if (title_input->performer)    g_free (title_input->performer);
  if (title_input->album_name)   g_free (title_input->album_name);
  if (title_input->genre)        g_free (title_input->genre);
  if (title_input->comment)      g_free (title_input->comment);
  g_free (title_input);
}

/**
 * Retrieve meta-information about URL.
 * For local files this means ID3 tag etc.
 */
gboolean
input_get_info (struct mad_info_t *info, gboolean fast_scan)
{
#ifdef DEBUG
  g_message ("f: input_get_info: %s", info->title);
#endif /* DEBUG */
  if (info->remote)
    { /* remote stream */
      if (input_http_init (info))
        return FALSE;
      if (input_udp_init (info))
        return FALSE;
    }
  else
    { /* local mp3 file */
      input_read_tag (info);
      /* scan mp3 file, decoding headers unless fast_scan is set */
      if (scan_file (info, fast_scan) == FALSE)
	{ 
	  return FALSE;
	}
      /* reset the input file to the start */
      lseek (info->infile, 0, SEEK_SET);
      /* use the filename for the title as a last resort */
      if (!info->title) 
        {
          char *pos = strrchr (info->filename, DIR_SEPARATOR);
          if (pos)
            info->title = g_strdup (pos + 1);
          else
            info->title = g_strdup (info->filename);
        }
    }
#ifdef DEBUG
  g_message ("e: input_get_info");
#endif /* DEBUG */
  return TRUE;
}

int
read_from_socket (struct mad_info_t *madinfo)
{
  static int bytes_read = 0;
  int free_space;
  int len;
  assert (madinfo && madinfo->sdata);
  free_space = HTTP_BUFFER_SIZE - madinfo->sdata->b_buffer_end;
  input_udp_read (madinfo);
  len = read (madinfo->sdata->sock,
              madinfo->sdata->b_buffer + madinfo->sdata->b_buffer_end,
              free_space);
  if (len > 0)
    madinfo->sdata->b_buffer_end += len;
  bytes_read += len;
/*
  if (bytes_read > sdata.metaint)
    {
      char * metadata = sdata.b_buffer+sdata.b_buffer_end - ( bytes_read + len - sdata.metaint );
      printf ("numblocks = %d\n", metadata[0]);
      bytes_read = bytes_read + len - sdata.metaint;
    }
*/
  return len;
}

int
input_rebuffer (struct mad_info_t *madinfo)
{
  int total = 0;
  int data_read = 0;
  int len = 0;
  int rtn;
  fd_set rfds;
  struct timeval tv;
  char message[64];

  assert (madinfo && madinfo->sdata);
  total = HTTP_BUFFER_SIZE - madinfo->sdata->b_buffer_end;
  /* Wait up up to thirty seconds. */
  tv.tv_sec = 30;
  tv.tv_usec = 0;

  mad_plugin.set_info ("buffering: 0k", -1, 0, 0, madinfo->channels);
  FD_ZERO (&rfds);
  FD_SET (madinfo->sdata->sock, &rfds);
  while (data_read < total && !madinfo->stop)
    {
      snprintf (message, 63, "buffering %d/%dk", madinfo->sdata->b_buffer_end/1024, HTTP_BUFFER_SIZE/1024);
      mad_plugin.set_info (message, -1, 0, 0, madinfo->channels);
      rtn = select (madinfo->sdata->sock + 1, &rfds, 0, 0, &tv);
      if (rtn == 0)
        {
          xmmsmad_error ("timeout while waiting for data from server");
          return -1;
        }
      if (rtn < 0)
        {
          xmmsmad_error ("error reading from socket");
          return -1;
        }
      len = read (madinfo->sdata->sock,
                  madinfo->sdata->b_buffer + madinfo->sdata->b_buffer_end,
                  total - data_read);
      if (len == -1)
        return -1;
      madinfo->sdata->b_buffer_end += len;
      data_read += len;
    }

  if (madinfo->sdata->b_buffer_end != HTTP_BUFFER_SIZE) 
    return -1;
  else 
    return 0;
}

/**
 * Read data from the source given my madinfo into the buffer
 * provided.  Return the number of bytes read.
 * @return 0 on EOF
 * @return -1 on error
 */
int
input_get_data (struct mad_info_t *madinfo, char *buffer, int buffer_size)
{
  int len = 0;
  int remainder = 0;
#ifdef DEBUG
  g_message ("f: input_get_data: %d", buffer_size);
#endif
  if (madinfo->remote)
    {
      int available;
      struct streamdata_t *sdata = madinfo->sdata;
      read_from_socket (madinfo);
      available = +sdata->b_buffer_end
        + sdata->f_buffer_end - sdata->f_buffer_start;
      /* make sure we are not asking for too much at one time */
      assert (buffer_size < HTTP_BUFFER_SIZE);
      if (available < buffer_size)
        {
          /* we ran out completely rebuffer */
          if (input_rebuffer (madinfo) == -1)
            {
              xmmsmad_error ("error filling http buffer");
	      return 0;
            }
        }

      if (buffer_size > sdata->f_buffer_end - sdata->f_buffer_start)
        {
          char *tmp;

          /* copy out the remainder of the front buffer */
          len = sdata->f_buffer_end - sdata->f_buffer_start;
          memcpy (buffer, sdata->f_buffer + sdata->f_buffer_start, len);

          /* swap the buffers */
          /* printf ("swapping buffers\n"); */
          tmp = sdata->f_buffer;
          sdata->f_buffer = sdata->b_buffer;
          sdata->f_buffer_start = 0;
          sdata->f_buffer_end = sdata->b_buffer_end;

          sdata->b_buffer = tmp;
          sdata->b_buffer_end = 0;
        }

      remainder = buffer_size - len;
      if (remainder)
        {
          assert (remainder < sdata->f_buffer_end - sdata->f_buffer_start);
          memcpy (buffer + len, sdata->f_buffer + sdata->f_buffer_start,
                  remainder);
          sdata->f_buffer_start += remainder;
          len += remainder;
        }
    }
  else
    {
      /* simply read to data from the file */
      assert (madinfo->infile >= 0);
      len = read (madinfo->infile, buffer, buffer_size);
    }
#ifdef DEBUG
  g_message ("e: input_get_data: %d", len);
#endif
  return len;
}

/**
 * Free up all mad_info_t related resourses.
 */
gboolean
input_term (struct mad_info_t *info)
{
#ifdef DEBUG
  g_message ("f: input_term");
#endif

  if (info->title   ) g_free (info->title);
  if (info->url     ) g_free (info->url);
  if (info->filename) g_free (info->filename);
  if (info->infile  ) close (info->infile);
  if (info->sdata   ) streamdata_free (info->sdata);
  if (info->id3file ) id3_file_close (info->id3file);

  /* set everything to zero in case it gets used again. */
  memset(info, 0, sizeof(struct mad_info_t));
#ifdef DEBUG
  g_message ("e: input_term");
#endif
  return TRUE;
}
