# MkdFile is a type of file, which contains a MultiMarkdown-formatted text on
# the file system layer.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::DocumentMode::MkdFile;

our @ISA = qw(CMSsh::FileSystem::TypedFile);

use strict;
use CMSsh::FileSystem::TypedFile;
use Text::MultiMarkdown 'markdown';
use CMSsh::DocumentMode::Config;
use CMSsh::DocumentMode::Messages;

my $Msg = CMSsh::DocumentMode::Messages->new;
my $Cfg = CMSsh::DocumentMode::Config->new;

sub new ($$) {
    my ($class_name, $abs_path) = @_;

    my $self = CMSsh::FileSystem::TypedFile->new($abs_path, get_ext());
    unless ($self) {
        warn $Msg->get('CANT_INST_FSO', $abs_path);
        return undef;
    }

    bless $self, $class_name;
    return $self;
}

sub get_ext (;$) {
    return $Cfg->get('EXT_DOC');
}

sub _read_file ($){
    my $self = shift;

    return 1 if ($self->{_content});

    $self->{_content} = $self->get_content
    or do {
        warn $Msg->get('FILE_READING_ERR', $self->{_path});
        return 0;
    };

    if ($self->{_content} =~ /^\s*$/) {
        warn $Msg->get('FILE_CONT_ERR', $self->{_path});
        return 0;
    }

    return 1;
}

sub get_metadata ($) {
    my $self = shift;

    unless ($self->{_content}) {
        $self->_read_file or return undef;
    }

    my ($metadata, $key);
    while ($self->{_content} =~ /\G(?:(\w[\w ]+):)?[\t ]+(.+)\n/g) {
        if (defined $1) {
            $key = $1;
            $metadata->{$key} = $2;
        } else {
            $metadata->{$key} .= " $2";
        }
    }

    $metadata->{Title} = $self->{_name} unless ($metadata->{Title});

    return $metadata;
}

sub get_markup ($) {
    my $self = shift;

    unless ($self->{_content}) {
        $self->_read_file or return undef;
    }

    my $html = markdown($self->{_content}, $Cfg->get('MMKD_PARAMS'));
# Empty markup means nothing
    $html = '' if ($html =~ /^\s*$/);

    return $html;
}

1
